# -*- coding: utf-8 -*-

# ==============================================================================
# COPYRIGHT (C) 1991 - 2003  EDF R&D                  WWW.CODE-ASTER.ORG
# THIS PROGRAM IS FREE SOFTWARE; YOU CAN REDISTRIBUTE IT AND/OR MODIFY
# IT UNDER THE TERMS OF THE GNU GENERAL PUBLIC LICENSE AS PUBLISHED BY
# THE FREE SOFTWARE FOUNDATION; EITHER VERSION 2 OF THE LICENSE, OR
# (AT YOUR OPTION) ANY LATER VERSION.
#
# THIS PROGRAM IS DISTRIBUTED IN THE HOPE THAT IT WILL BE USEFUL, BUT
# WITHOUT ANY WARRANTY; WITHOUT EVEN THE IMPLIED WARRANTY OF
# MERCHANTABILITY OR FITNESS FOR A PARTICULAR PURPOSE. SEE THE GNU
# GENERAL PUBLIC LICENSE FOR MORE DETAILS.
#
# YOU SHOULD HAVE RECEIVED A COPY OF THE GNU GENERAL PUBLIC LICENSE
# ALONG WITH THIS PROGRAM; IF NOT, WRITE TO EDF R&D CODE_ASTER,
#    1 AVENUE DU GENERAL DE GAULLE, 92141 CLAMART CEDEX, FRANCE.
# ==============================================================================

"""
Definition of ASTER_SYSTEM class.
"""


import os
import sys
import re
import time
import glob
import getpass
import resource
import tempfile

from asrun.i18n      import _
from asrun.mystring  import print3, convert, to_unicode
from asrun.utils     import get_encoding


# colors and tty...
label = { 'OK' : '  OK  ', 'FAILED' : 'FAILED', 'SKIP' : ' SKIP ' }
if sys.stdout.isatty():
   label['OK']     = '\033[1;32m%s\033[m\017' % label['OK']
   label['FAILED'] = '\033[1;31m%s\033[m\017' % label['FAILED']
#   label['SKIP']   = '\033[1;34m%s\033[m\017' % label['SKIP']
for k in label:
   label[k] = '['+label[k]+']'


# store local user (will be the default for remote machines)
local_user      = getpass.getuser()
local_full_host = os.uname()[1]
local_host      = local_full_host.split('.')[0]  # without domain name


def _exitcode(status, default=99):
   """
   Extract the exit code from status. Return 'default' if the process
   has not been terminated by exit.
   """
   if os.WIFEXITED(status):
      iret = os.WEXITSTATUS(status)
   elif os.WIFSIGNALED(status):
      iret = os.WTERMSIG(status)
   elif os.WIFSTOPPED(status):
      iret = os.WSTOPSIG(status)
   else:
      iret = default
   return iret



class ASTER_SYSTEM:
   """
   Class to encapsultate "system" commands on local or remote machine.
   Available methods :
      'Copy', 'Shell', 'IsRemote', 'MkDir', 'Delete', 'Exists', 'IsDir',
      'Gzip', 'Gunzip', 'AddToEnv', 'GetHostName', 'PathOnly', 'Rename',
      'VerbStart, 'VerbEnd', 'VerbIgnore', 'SetLimit', 'Which',
      'GetCpuInfo', 'FindPattern'
   Supported protocols :
   - for copy :
      LOCAL, RCP, SCP, RSYNC, HTTP
   - for shell execution :
      LOCAL, RSH, SSH
   """
   # this values must be set during installation step.
   MaxCmdLen = 32768
   MultiThreading = sys.version_info >= (2, 5)
   # line length -9
   _LineLen = 80-9
   # required parameters for initialisation
   RequiredParam = ['remote_copy_protocol', 'remote_shell_protocol']

   def __init__(self, run, **kargs):
      """
      run : ASTER_RUN object
      OR just a dictionnary to define 'RequiredParam' + 'verbose', 'debug'
      """
      self.verbose = run['verbose']
      self.debug = run['debug']
      # ----- fill param
      self.param = {}
      for key in self.RequiredParam:
         self.param[key] = run[key]
      # ----- reference to ASTER_RUN object which manages the execution
      self.run = run
      # ----- check if protocols are supported
      if not self.param['remote_shell_protocol'] in ('LOCAL', 'RSH', 'SSH'):
         self._mess(_(u'unknown protocol : %s') % self.param['remote_shell_protocol'],
               '<A>_ALARM')
      if not self.param['remote_copy_protocol'] in ('LOCAL', 'RCP', 'SCP', 'RSYNC', 'HTTP'):
         self._mess(_(u'unknown protocol : %s') % self.param['remote_copy_protocol'],
               '<A>_ALARM')
      self._dbg('ASTER_SYSTEM param :', self.param)
      # ----- cache cpuinfo, meminfo
      self._cpuinfo_cache = {}
      self._meminfo_cache = {}
      # ----- check for shared folders (never use remote copy for them)
      self.shared_folders = []
      if self.run.get('shared_folders'):
         self.shared_folders = [rep.strip() \
                  for rep in self.run['shared_folders'].split(',')]
      # ----- add shortcuts - example: run.Copy instead of run.system.Copy
      if type(run) != dict:
         for meth in ASTER_SYSTEM.__dict__.keys():
            if meth[0] == meth[0].upper() and not hasattr(run, meth) \
                  and hasattr(self, meth) and callable(getattr(self, meth)):
               setattr(run, meth, getattr(self, meth))


   def getuser_host(self):
      return local_user, local_host


   def _mess(self, msg, cod='', store=False):
      """
      Just print a message
      """
      if hasattr(self.run, 'Mess'):
         self.run.Mess(msg, cod, store)
      else:
         print3('%-18s %s' % (cod, msg))


   def _dbg(self, *args, **kargs):
      """
      Print debug informations
      """
      if hasattr(self.run, 'DBG'):
         kargs['stack_id'] = kargs.get('stack_id', 1)
         self.run.DBG(*args, **kargs)
      elif self.debug:
         print3('<DBG> %s // %s' % (str(args), str(kargs)))


   def VerbStart(self, cmd, verbose=None):
      """
      Start message in verbose mode
      """
      Lm = self._LineLen
      if verbose == None:
         verbose = self.verbose
      if verbose:
         pcmd = cmd
         if len(cmd) > Lm-2 or cmd.count(os.linesep) > 0:
            pcmd = pcmd+os.linesep+' '*Lm
         print3(('%-'+str(Lm)+'s') % (pcmd, ), end=' ')
         sys.stdout.flush()


   def VerbEnd(self, iret, output='', verbose=None):
      """
      End message in verbose mode
      """
      if verbose == None:
         verbose = self.verbose
      if verbose:
         if iret == 0:
            print3(label['OK'])
         else:
            print3(label['FAILED'])
            print3(_(u'Exit code : %d') % iret)
         if (iret != 0 or self.debug) and output:
            print3(output)


   def VerbIgnore(self, verbose=None):
      """
      End message in verbose mode
      """
      if verbose == None:
         verbose = self.verbose
      if verbose:
         print3(label['SKIP'])


   def local_shell(self, cmd, bg=False, verbose=False, follow_output=False,
                   alt_comment=None, interact=False, separated_stderr=False):
      """
      Execute a command shell
         cmd           : command
         bg            : put command in background if True
         verbose       : print status messages during execution if True
         follow_output : follow interactively output of command
         alt_comment   : print this "alternative comment" instead of "cmd"
      Return :
         iret     : exit code if bg = False,
                    0 if bg = True
         output   : output lines (as string)
      """
      if not alt_comment:
         alt_comment = cmd
      if len(cmd) > self.MaxCmdLen:
         self._mess((_(u'length of command shell greater '\
               'than %d characters.') % self.MaxCmdLen), '<A>_ALARM')
      self._dbg('cmd :', cmd, 'background : %s' % bg, 'follow_output : %s' % follow_output, all=True, stack_id=3)
      self.VerbStart(alt_comment, verbose=verbose)
      if follow_output and verbose:
         print3(os.linesep+_(u'Command output :'))

      fout = tempfile.NamedTemporaryFile()
      ferr = tempfile.NamedTemporaryFile()
      if bg:
         # new_cmd = cmd + ' &' => NO : stdout/stderr must be closed not to block
         new_cmd = '( %s ) > %s 2>&1 &' % (cmd, fout.name)
      elif follow_output:
         if not separated_stderr:
            new_cmd = '( %s ) 2>&1 | tee %s' % (cmd, fout.name)
         else:
            new_cmd = '( %s ) 2> %s | tee %s' % (cmd, ferr.name, fout.name)
      else:
         if not separated_stderr:
            new_cmd = '( %s ) > %s 2>&1' % (cmd, fout.name)
         else:
            new_cmd = '( %s ) > %s 2> %s' % (cmd, fout.name, ferr.name)
      #self._dbg('command line :', new_cmd)
      # execution
      iret = os.system(new_cmd)
      fout.seek(0)
      output = to_unicode(fout.read())
      ferr.seek(0)
      error = to_unicode(ferr.read())
      fout.close()
      ferr.close()
      
      if follow_output:
         # repeat header message
         self.VerbStart(alt_comment, verbose=verbose)
      mat = re.search('EXIT_CODE=([0-9]+)', output)
      if mat:
         iret = int(mat.group(1))
      self.VerbEnd(iret, output, verbose=verbose)
      if iret != 0:
         self._dbg('ERROR : iret = %s' % iret, output, all=True)
      if bg:
         iret = 0
      if not separated_stderr:
         result = iret, output
      else:
         result = iret, output, error
      return result


   def filename2dict(self, s):
      """
      Convert file name 's' given in format [[user[:passwd]@]machine:]name
      into a dictionnary with keys = user, passwd, mach, name
      """
      d = {
         'user'   : '',
         'passwd' : '',
         'mach'   : '',
         'name'   : ''
      }
      if type(s) not in (str, unicode):
         self._mess(_(u'unexpected type'), '<F>_PROGRAM_ERROR')
      mat = re.search('(.*):(.*)@(.*):(.*)', s)
      n = 4
      if mat == None:
         mat = re.search('(.*)@(.*):(.*)', s)
         n = 3
         if mat == None:
            mat = re.search('(.*):(.*)', s)
            n = 2
            if mat == None:
               d['name'] = s
               n = 1
      if n >= 2:
         g = list(mat.groups())
         d['name'] = g.pop()
         d['mach'] = g.pop()
      if n >= 3:
         d['user'] = g.pop()
      if n >= 4:
         d['passwd'] = d['user']
         d['user'] = g.pop()
      if d['mach'] != '' and d['user'] == '':
         d['user'] = local_user
      for rep in self.shared_folders:
         if d['name'].startswith(rep):
            d['user'] = d['passwd'] = d['mach'] = ''
            break
      return d


   def PathOnly(self, filename):
      """
      Return only the path of file given as [[user[:passwd]@]machine:]name.
      Very useful when `filename` contains 'user@mach:' string but refers to a
      local filename.
      """
      return self.filename2dict(filename)['name']


   def Copy(self, dest, *src, **opts):
      """
      Copy files/directories (from 'src' list) to 'dest'
      Remote names format :
         [[user[:passwd]@]machine:]name
      where name is an absolute or a relative path.
      optionnal arguments : verbose, follow_output passed to local_shell,
         niverr : default is set to <F>_COPY_ERROR, but may be '<A>' or less !
         protocol : to use a different protocol just for this copy.
      Note : if src[i] is a directory and dest doesn't exist
             dest is created and the content of src[i] is copied into dest.
             This is not a default behavior of rsync which would create
             `basename src[i]` in dest even if dest doesn't exist.
      """
      # cp command
      if sys.platform in ('linux2', 'x86_64', 'cygwin'):
         cpcmd = 'cp -L -r'
      else:
         cpcmd = 'cp -r'
      
      iret = 0
      kargs = {}
      # take values from opts if exist
      kargs['verbose'] = opts.get('verbose', self.verbose)
      kargs['follow_output'] = opts.get('follow_output', False)
      if opts.get('niverr'):
         niverr = opts['niverr']
      else:
         niverr = '<F>_COPY_ERROR'
      ddest = self.filename2dict(dest)
      if self.IsRemote(dest) \
            or (ddest['user'] != '' and ddest['user'] != local_user):
         fdest = ddest['user']+'@'+ddest['mach']+':'+ddest['name']
      else:
         fdest = ddest['name']
      self._dbg('source list : %s' % (src,), 'destination : %s' % fdest)

      if len(src) < 1:
         self._mess(_(u'no source file to copy'), '<A>_ALARM')

      for f in src:
         # here because we can change 'proto' if necessary
         proto = opts.get('protocol', self.param['remote_copy_protocol'])
         jret = 0
         df = self.filename2dict(f)
         if self.IsRemote(f):
            fsrc = df['user']+'@'+df['mach']+':'+df['name']
         else:
            fsrc = df['name']
            df['user'] = df['mach'] = ''
         cmd = ''
         tail = '.../'+'/'.join(f.split('/')[-2:])
         if not opts.has_key('alt_comment'):
            kargs['alt_comment'] = 'copying %s' % tail
         else:
            kargs['alt_comment'] = opts['alt_comment']
         if df['mach'] == '' and ddest['mach'] == '':
            cmd = cpcmd+' '+fsrc+' '+fdest
         else:
            if proto == 'RSYNC' and df['mach'] != '' and ddest['mach'] != '':
               proto = 'RCP'
               self._mess(_(u"copying a remote file to another remote server " \
                     "isn't allowed through RSYNC, trying with RCP."))
            if proto == 'RCP':
               cmd = 'rcp -r '+fsrc+' '+fdest
            elif proto == 'SCP':
               cmd = 'scp -rBCq '+fsrc+' '+fdest
            elif proto == 'RSYNC':
               if self.IsDir(f) and not self.Exists(dest):
                  self.MkDir(dest)
                  cmd = 'rsync -rz '+os.path.join(fsrc, '*')+' '+fdest
               else:
                  cmd = 'rsync -rz '+fsrc+' '+fdest
            elif proto == 'HTTP':
               str_user = ''
               if not df['user'] in ('', 'anonymous'):
                  str_user = df['user']+'@'
               # dest must be local
               if ddest['mach'] == '':
                  cmd = 'wget http://'+str_user+df['mach']+df['name']+' -O '+fdest
               else:
                  cmd = ''
                  self._mess(_(u'remote destination not allowed through %s' \
                        ' : %s') % (proto, fdest), niverr)
         if cmd != '':
            jret, out = self.local_shell(cmd, **kargs)
            if jret != 0 and niverr != 'SILENT':
               self._mess(_(u'error during copying %s to %s') % (f, fdest) \
                  + os.linesep + _(u'message : %s') % out, niverr)
         else:
            self._mess(_(u'unexpected error or unknown protocol : %s') \
                   % proto, niverr)
         iret = max(jret, iret)
      return iret


   def Rename(self, src, dest, **opts):
      """
      Rename 'src' to 'dest'.
      Try using os.rename then 'mv -f' if failure.
      """
      iret = 0
      try:
         os.rename(src, dest)
         assert os.path.exists(dest)      # could avoid NFS failure
      except (AssertionError, OSError):
         cmd = 'mv -f %s %s' % (src, dest)
         iret, out = self.local_shell(cmd, **opts)
      return iret


   def Shell(self, cmd, mach='', user='', **opts):
      """
      Execute a command shell on local or remote machine.
      Options : see local_shell
      """
      iret = 1
      # valeurs par défaut
      kargs = {
         'verbose'         : self.verbose,
         'bg'              : False,
         'interact'        : False,
         'follow_output'   : False,
         'alt_comment'     : cmd,
      }
      # surcharge par opts
      kargs.update(opts)

      proto = self.param['remote_shell_protocol']
      
      # distant ?
      mach_loc   = self.GetHostName().split('.')[0]
      user_loc   = local_user
      mach_short = mach.split('.')[0] or mach_loc
      user = user or user_loc
      distant = mach_loc != mach_short or user != user_loc
      self._dbg('remote command (%s <> %s and %s <> %s) ? %s' % (mach, mach_loc, user, user_loc, distant))
      
      if not distant:
         iret, out = self.local_shell(cmd, **kargs)
      else:
         action = ''
         # pour recuperer correctement l'output, il faut des " et non des ' autour des
         # commandes sous "sh -c" et sous "xterm -e"
         if cmd.find('"') != -1:
            cmd = cmd.replace('"', '\\"')
         cmd = '"%s"' % cmd
         if proto == 'RSH':
            action = 'rsh -n -l '+user+' '+mach+' '+cmd
         elif proto == 'SSH':
            action = 'ssh -n -l '+user+' '+mach+' '+cmd
         if action != '':
            iret, out = self.local_shell(action, **kargs)
         else:
            self._mess(_(u'unknown protocol : %s') % proto, '<A>_ALARM')
      return iret, out


   def Gzip(self, src, **opts):
      """
      Compress file or content of directory 'src'.
      optionnal arguments : verbose, niverr, cmd (to override gzip command).
      Only local files/directories supported !
      """
      return self._gzip_gunzip('gzip', src, **opts)


   def Gunzip(self, src, **opts):
      """
      Decompress file or content of directory 'src'.
      optionnal arguments : verbose, niverr, cmd (to override gzip -d command).
      Only local files/directories supported !
      """
      return self._gzip_gunzip('gunzip', src, **opts)


   def _gzip_gunzip(self, mode, src, **opts):
      """
      Called by Gzip (mode=gzip) and Gunzip (mode=gunzip) methods.
      Return status and filename of src after de/compression.
      Only local files/directories supported !
      """
      para = {
         'gzip' : {
            'cmd'     : 'gzip ',
            'niverr'  : '<F>_COMPRES_ERROR',
            'msg0'    : _(u'no file/directory to compress'),
            'comment' : _(u'compressing %s'),
            'msgerr'  : _(u'error during compressing %s'),
         },
         'gunzip' : {
            # sometimes gunzip doesn't exist, gzip seems safer
            'cmd'     : 'gzip -d ',
            'niverr'  : '<F>_DECOMPRESSION',
            'msg0'    : _(u'no file/directory to decompress'),
            'comment' : _(u'decompressing %s'),
            'msgerr'  : _(u'error during decompressing %s'),
         },
      }
      if not mode in para.keys():
         self._mess(_(u'unknown mode : %s') % mode, '<F>_PROGRAM_ERROR')
      iret = 0
      if opts.has_key('cmd'):
         cmd = opts['cmd']
      else:
         cmd = para[mode]['cmd']
      if opts.has_key('verbose'):
         verbose = opts['verbose']
      else:
         verbose = self.verbose
      if opts.has_key('niverr'):
         niverr = opts['niverr']
      else:
         niverr = para[mode]['niverr']

      if len(src) < 1:
         self._mess(para[mode]['msg0'], '<A>_ALARM')
         return iret

      if not os.path.isdir(src):
         if mode == 'gzip':
            name = src+'.gz'
         else:
            name = re.sub('\.gz$', '', src)
         lf = [src]
      else:
         name = src
         lf = glob.glob(os.path.join(src, '*'))
      for f in lf:
         jret = 0
         comment = para[mode]['comment'] % f
         if os.path.isdir(f):
            self.VerbStart(comment, verbose=verbose)
            self.VerbIgnore(verbose=verbose)
         else:
            jret, out = self.local_shell(cmd+f, verbose=verbose, alt_comment=comment)
         if jret != 0:
            self._mess(para[mode]['msgerr'] % f, niverr)
         iret = max(iret, jret)
      return iret, name


   def IsRemote(self, path):
      """
      Return True if 'path' seems to be on a remote host.
      NB : we suppose that host and host.domain are identical.
      """
      dico  = self.filename2dict(path)
      mach = dico['mach'].split('.')[0]
      diff_host = ( mach != '' and mach != local_host)
      return diff_host


   def Exists(self, path):
      """
      Return True if 'path' exists (file or directory).
      """
      iret = 0
      exists = True
      dico = self.filename2dict(path)
      if self.IsRemote(path):
         cmd = "sh -c 'if test -f "+dico['name']+' ; then echo FILE_EXISTS; ' \
                         "elif test -d "+dico['name']+' ; then echo DIR_EXISTS ; ' \
                                                         "else echo FALSE ; fi'"
         iret, out = self.Shell(cmd, dico['mach'], dico['user'], verbose=self.verbose)
         if out.find('FALSE') > -1:
            exists = False
      else:
         exists = os.path.exists(dico['name'])
      return exists


   def IsDir(self, path):
      """
      Return True if 'path' is a directory.
      """
      iret = 0
      isdir = False
      dico = self.filename2dict(path)
      if self.IsRemote(path):
         cmd = "sh -c 'if test -d "+dico['name']+' ; then echo IS_DIRECTORY; ' \
                                             "else echo FALSE ; fi'"
         iret, out = self.Shell(cmd, dico['mach'], dico['user'], verbose=self.verbose)
         if out.find('IS_DIRECTORY') > -1:
            isdir = True
      else:
         isdir = os.path.isdir(dico['name'])
      return isdir


   def IsText(self, path):
      """
      Return True if `path` is a text file.
      Warning : return False if `path` is remote
      """
      iret = 0
      istext = False
      dico = self.filename2dict(path)
      if not self.IsRemote(path):
         cmd = "file %s" % path
         iret, out = self.Shell(cmd, dico['mach'], dico['user'], verbose=self.verbose)
         if out.lower().find('text') > -1:
            istext = True
      return istext


   def FileCat(self, src=None, dest=None, text=None):
      """
      Append content of 'src' to 'dest' (filename or file object).
      Warning : Only local filenames !
      """
      assert dest != None and (src != None or text != None)
      if type(dest) != file:
         if not os.path.isdir(dest) or os.access(dest, os.W_OK):
            f2 = open(dest, 'a')
         else:
            self._mess(_(u'No write access to %s') % dest, '<F>_ERROR')
            return
      else:
         f2 = dest
      if text is None:
         if os.path.exists(src):
            if os.path.isfile(src):
               text = open(src, 'r').read()
            else:
               self._mess(_(u'file not found : %s') % src, '<F>_FILE_NOT_FOUND')
               return
      if text:
         f2.write(text)
      
      if type(dest) != file:
         f2.close()


   def MkDir(self, rep, niverr=None, verbose=None):
      """
      Create the directory 'rep' (mkdir -p ...)
      """
      if niverr == None:
         niverr = '<F>_MKDIR_ERROR'
      if verbose == None:
         verbose = self.verbose
      iret = 0
      dico = self.filename2dict(rep)
      if self.IsRemote(rep):
         cmd = 'mkdir -p '+dico['name']
         iret, out = self.Shell(cmd, dico['mach'], dico['user'], verbose=verbose)
      else:
         self.VerbStart(_(u'creating directory %s') % dico['name'], verbose)
         # ----- it's not tested in remote
         if self.IsDir(rep):
            iret = 0
            self.VerbIgnore(verbose)
         else:
            s = ''
            try:
               if os.path.isdir(dico['name']):
                  self._dbg(_(u'mkdir skipped, directory already exists %s') % dico['name'])
               else:
                  os.makedirs(dico['name'])
            except OSError, s:
               iret = 4
            self.VerbEnd(iret, s, verbose)
      if iret != 0:
         self._mess(_(u'can not create directory %s') % rep, niverr)
      return iret


   def Delete(self, rep, remove_dirs=True, verbose=None):
      """
      Delete a file or a directory tree (rm -rf ...).
      Set 'remove_dirs' to False to be sure to delete 'rep' only if it's a file.
      """
      if verbose == None:
         verbose = self.verbose
      iret = 0
      dico = self.filename2dict(rep)
      # preventing to delete first level directories (as /, /home, /usr...)
      if dico['name'][0] == '/' and len(dico['name'][:-1].split(os.sep)) <= 2:
         self._mess(_(u'deleting this directory seems too dangerous. ' \
               '%s has not been removed') % dico['name'], '<A>_ALARM')
         return
      
      if remove_dirs:
         cmd = r'\rm -rf '+dico['name']
      else:
         cmd = r'\rm -f '+dico['name']
      
      comment = _(u'deleting %s') % rep
      if self.IsRemote(rep):
         iret, out = self.Shell(cmd, dico['mach'], dico['user'],
                                alt_comment=comment, verbose=verbose)
      else:
         iret, out = self.Shell(cmd, alt_comment=comment, verbose=verbose)
      return


   def AddToEnv(self, profile):
      """
      Read 'profile' file (with sh/bash/ksh syntax) and add updated
      variables to os.environ.
      """
      def env2dict(s):
         """Convert output to a dictionnary."""
         l = s.split(os.linesep)
         d = {}
         for line in l:
            mat = re.search('^([-a-zA-Z_0-9@\+]+)=(.*$)', line)
            if mat != None:
               d[mat.group(1)] = mat.group(2)
         return d
      if not os.path.isfile(profile):
         self._mess(_(u'file not found : %s') % profile, '<A>_FILE_NOT_FOUND')
         return
      # read initial environment
      iret, out = self.Shell('sh -c env')
      env_init = env2dict(out)
      if iret != 0:
         self._mess(_(u'error getting environment'), '<E>_ABNORMAL_ABORT')
         return
      # read profile and dump modified environment
      iret, out = self.Shell('sh -c ". %s ; env"' % profile)
      env_prof = env2dict(out)
      if iret != 0:
         self._mess(_(u'error reading profile : %s') % profile,
               '<E>_ABNORMAL_ABORT')
         return
      # "diff"
      for k, v in env_prof.items():
         if env_init.get(k, None) != v:
            if self.debug:
               self._dbg('AddToEnv adding : %s=%s' % (k, v))
            os.environ[k] = v
      for k in [k for k in env_init.keys() if env_prof.get(k) is None]:
         self._dbg('unset %s ' % k, DBG=True)
         try:
            del os.environ[k]
         except:
            pass


   def GetHostName(self, host=None):
      """
      Return hostname of the machine 'host' or current machine if None.
      """
      from socket import gethostname, gethostbyaddr
      if host == None:
         host = gethostname()
      try:
         fqn, alias, ip = gethostbyaddr(host)
      except:
         fqn, alias, ip = host, [], None
      if fqn.find('localhost') > -1:
         alias = [a for a in alias if a.find('localhost')<0]
         if len(alias)>0:
            fqn = alias[0]
         for a in alias:
            if a.find('.')>-1:
               fqn = a
               break
      return fqn


   def GetCpuInfo(self, what, mach='', user=''):
      """
      Return CPU information.
      what='numcpu'    : number of processors
      what='numthread' : number of threads (deoends on MultiThreading attribute)
      """
      if self._cpuinfo_cache.get(what+mach+user) is not None:
         return self._cpuinfo_cache[what+mach+user]
      if what in ('numcpu', 'numthread'):
         num = 1
         if self.MultiThreading or what == 'numcpu':
            if sys.platform == 'linux2':
               iret, out = self.Shell('cat /proc/cpuinfo', mach, user)
               exp = re.compile('^processor\s+:\s+([0-9]+)', re.MULTILINE)
               l_ids = exp.findall(out)
               if len(l_ids) > 1:      # else: it should not !
                  num = max([int(i) for i in l_ids]) + 1
         self._cpuinfo_cache[what+mach+user] = num
         self._dbg("GetCpuInfo '%s' returns : %s" % (what, num))
         return num
      else:
         return None


   def GetMemInfo(self, what, mach='', user=''):
      """
      Return memory information.
      """
      if self._meminfo_cache.get(what+mach+user) is not None:
         return self._meminfo_cache[what+mach+user]
      if what in ('memtotal',):
         num = None
         if sys.platform == 'linux2':
            iret, out = self.Shell('cat /proc/meminfo', mach, user)
            mat = re.search('^%s *: *([0-9]+) *kb' % what, out, re.MULTILINE | re.IGNORECASE)
            if mat != None:         # else: it should not !
               num = int(mat.group(1))/1024
         self._meminfo_cache[what+mach+user] = num
         self._dbg("GetMemInfo '%s' returns : %s" % (what, num))
         return num
      else:
         return None


   def SendMail(self, dest, author=None, subject='no subject', text=''):
      """
      Send a message by mail.
      Use ", " in `dest` to separate multiple recipients.
      """
      sign = 'sent by as_run/SendMail from %s' % self.GetHostName()
      sign = os.linesep*2 + '-'*len(sign) + os.linesep \
           + sign \
           + os.linesep   + '-'*len(sign) + os.linesep
      try:
         import smtplib
         from email.MIMEText import MIMEText
      except ImportError:
         self._mess(_(u'Can not send mail from this machine'),
            '<A>_NO_MAILER')
         return
      dest = [s.strip() for s in dest.split(',')]
      mail_encoding = get_encoding()
      msg = MIMEText(convert(text+sign, mail_encoding), _charset=mail_encoding)
      msg['Subject'] = convert(subject, mail_encoding)
      if author == None:
         author = '%s@%s' % (getpass.getuser(), self.GetHostName())
      msg['From'] = author
      msg['To'] = ', '.join(dest)
      s = smtplib.SMTP()
      s.connect()
      s.sendmail(msg['From'], dest, msg.as_string())
      s.close()


   def SetLimit(self, what, *l_limit):
      """
      Set a system limit.
      `what` is one of CORE, CPU... (see help of resource module).
      If provided `l_limit` contains (soft limit, hard limit).
      """
      nomp = 'RLIMIT_%s' % what.upper()
      param = getattr(resource, nomp)
      if param != None:
         if len(l_limit) == 1:
            l_limit = (l_limit[0], l_limit[0])
         elif len(l_limit) != 2:
            l_limit = (-1, -1)
         l_limit = list(l_limit)
         for i, lim in enumerate(l_limit):
            if type(lim) not in (int, long):
               l_limit[i] = -1
         try:
            self._dbg([what, nomp, l_limit])
            resource.setrlimit(param, l_limit)
         except Exception, msg:
            self._mess(_(u'unable to set %s limit to %s') % (nomp, l_limit))


   def Which(self, cmd):
      """
      Same as `which cmd`. Returns the path found or None.
      """
      ldirs = os.environ.get('PATH').split(':')
      for d in ldirs:
         path = os.path.join(d, cmd)
         if os.path.isfile(path) and os.access(path, os.X_OK):
            return path
      return None


   def FindPattern(self, root, pattern, maxdepth=5):
      """
      Return a list of the files matching 'pattern'.
      The same as glob if maxdepth=0.
      """
      if self.IsRemote(root):
         self._mess(_(u'Find files matching a pattern only works on local files/directories'), \
            '<F>_PROGRAM_ERROR')
      root = self.PathOnly(root)
      if os.path.isfile(root):
         return [root, ]
      root = os.path.realpath(root)
      dirs = [root,]
      if maxdepth > 0:
         level=len(root.split(os.path.sep))
         for base, l_dirs, l_nondirs in os.walk(root):
            lev=len(base.split(os.path.sep))
            if lev <= (level + maxdepth):
               if not base in dirs:
                  dirs.append(base)
            else:
               del l_dirs[:] # empty dirs list so we don't walk needlessly
      res = []
      for d in dirs:
         res.extend(glob.glob(os.path.join(d, pattern)))
      self._dbg('FindPattern : rootdir=%s  pattern=%s' % (root, repr(pattern)), res)
      return res



class ASTER_SYSTEM_MINIMAL(ASTER_SYSTEM):
   """
   Fake ASTER_RUN to use easily ASTER_SYSTEM outside of asrun.
   """
   def __init__(self, **kargs):
      """
      Initialization
      """
      opts = { 
         'debug'   : False,
         'verbose' : False,
         'remote_copy_protocol' : 'SCP',
         'remote_shell_protocol' : 'SSH',
      }
      opts.update(kargs)
      ASTER_SYSTEM.__init__(self, run=opts)


if __name__ == '__main__':
   system = ASTER_SYSTEM_MINIMAL()
   

