#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <dirent.h>

#ifdef MACOSX
#include <sys/stat.h>
#endif

#include "player.h"
#include "files.h"
#include "main.h"


// This function reads the lines of text from
// a file. It returns a pointer to a random line
// of text. On error, a NULL is returned.
char *Random_Line(char *filename)
{
  FILE *my_file;
  char line[MAX_INSULT_LINE];
  char *return_value;
  char *result;
  int line_count = 0;
  int grab_line;

  if (! filename) return NULL;
  my_file = fopen(filename, "r");
  if (! my_file) return NULL;

  // count the lines inthe file
  result = fgets(line, MAX_INSULT_LINE, my_file);
  while (result)
    {
      line_count++;
      result = fgets(line, MAX_INSULT_LINE, my_file);
    }

  if (! line_count) return NULL;

  // pick a random value
  fseek(my_file, 0, SEEK_SET);
  grab_line = rand() % line_count;

  // find the selected line
  line_count = 0;
  while (line_count < grab_line)
    {
      result = fgets(line, MAX_INSULT_LINE, my_file);
      line_count++;
    }
  fclose(my_file);

  // remove CR and LF
  if ( strchr(line, '\n') )
    strchr(line, '\n')[0] = '\0';
  if ( strchr(line, '\r') )
    strchr(line, '\r')[0] = '\0';
  // return the line we read
  return_value = (char *) calloc( strlen(line) + 1, sizeof(char) );
  if (! return_value) return NULL;
  strcpy(return_value, line);
  return return_value;
}



/*
This function saves the game in progress.
All data is saved in a text file for flexiblity.
Returns TRUE on success and FALSE on failure.

Changed to write binary data. The original is below.
-- Sven
*/
bool Save_Game(GLOBALDATA *global, ENVIRONMENT *env)
{
  ofstream ofsFile;
  char *file_path = NULL;
  int player_count = 0;
  PLAYER *my_player = NULL;
  int count;
  int global_player_number;
  bool bResult = true;

  // figure out file name
  file_path = (char *) calloc( strlen(global->configDir) +
                               strlen(global->game_name) + 24,
                               sizeof(char) );
  if (! file_path)
    bResult = false;
  else
    {
      sprintf(file_path, "%s/%s.sav", global->configDir, global->game_name);

      ofsFile.open(file_path);
      free(file_path);
    }
  if (!ofsFile.is_open())
    bResult = false;

  // write global data
  if (bResult) bResult = pushData("MaxRounds  ", global->rounds,        ofsFile);
  if (bResult) bResult = pushData("CurrRound  ", global->currentround,  ofsFile);
  if (bResult) bResult = pushData("Campaign   ", global->campaign_mode, ofsFile);
  if (bResult) bResult = pushData("PlayerCount", global->numPlayers,    ofsFile);

  // write player data
  while ( bResult && (player_count < global->numPlayers) )
    {
      my_player = global->players[player_count];
      global_player_number = 0;
      while ( strcmp(global->allPlayers[global_player_number]->getName(), my_player->getName() ) )
        global_player_number++;
      if (bResult) bResult = pushData("PlayerNumber", global_player_number, ofsFile);
      if (bResult) bResult = pushData("Score       ", my_player->score, ofsFile);
      if (bResult) bResult = pushData("Money       ", my_player->money, ofsFile);
      if (bResult) bResult = pushData("PlayerType  ", my_player->type, ofsFile);
      if (bResult) bResult = pushData("TypeSaved   ", my_player->type_saved, ofsFile);
      // Save revengees:
      if (bResult && my_player->revenge)
        {
          int iRevengeNr = 0;
          while ( strcmp(global->allPlayers[iRevengeNr]->getName(), my_player->revenge->getName() ) )
            iRevengeNr++;
          bResult = pushData("Revengee Nr ", iRevengeNr, ofsFile);
        }
      for (count = 0; count < WEAPONS; count++)
        if (bResult) bResult = pushData(weapon[count].name, my_player->nm[count], ofsFile);
      for (count = 0; count < ITEMS; count++)
        if (bResult) bResult = pushData(item[count].name,   my_player->ni[count], ofsFile);
      if (bResult) bResult = pushData(FILEPART_ENDPLYR, ofsFile);
      player_count++;
    }
  if (bResult) bResult = pushData(FILEPART_ENDSECT,     ofsFile);
  if (bResult) bResult = pushData(FILEPART_ENDFILE,     ofsFile);

  if (ofsFile.is_open())
    ofsFile.close();

  // save the current config
  if (bResult)
    {
      file_path = (char *) calloc( strlen(global->configDir) +
                                   strlen(global->game_name) + 24,
                                   sizeof(char) );
      if (! file_path)
        bResult = false;
      else
        {
          sprintf(file_path, "%s/%s.dat", global->configDir, global->game_name);
          if ( !(bResult = Save_Game_Settings(global, env, file_path, true)) )
            perror("WARNING: Game settings could not be saved in files::Save_Game()!");
          free(file_path);
        }
    }
  return(bResult);
}

//int Save_Game(GLOBALDATA *global, ENVIRONMENT *env)
//{
//  FILE *game_file = NULL;
//  char *file_path = NULL;
//  int player_count = 0;
//  PLAYER *my_player = NULL;
//  int count;
//  int global_player_number;
//
//  // figure out file name
//  file_path = (char *) calloc( strlen(global->configDir) +
//                               strlen(global->game_name) + 24,
//                               sizeof(char) );
//  if (! file_path)
//    return FALSE;
//
//  sprintf(file_path, "%s/%s.sav", global->configDir, global->game_name);
//
//  game_file = fopen(file_path, "w");
//  free(file_path);
//  if (!game_file)
//    return FALSE;
//
//  // write global data
//  fprintf(game_file, "GLOBAL\n");
//  fprintf(game_file, "ROUNDS=%f\n", global->rounds);
//  fprintf(game_file, "CURRENTROUND=%d\n", global->currentround);
//  fprintf(game_file, "CAMPAIGNMODE=%f\n", global->campaign_mode);
//  fprintf(game_file, "***\n");
//
//  // write envrionment data
//  fprintf(game_file, "ENVIRONMENT\n");
//  fprintf(game_file, "***\n");
//
//  // write player data
//  fprintf(game_file, "PLAYERS\n");
//  while ( player_count < global->numPlayers )
//    {
//      my_player = global->players[player_count];
//      global_player_number = 0;
//      while ( strcmp(global->allPlayers[global_player_number]->getName(), my_player->getName() ) )
//        global_player_number++;
//      fprintf(game_file, "PLAYERNUMBER=%d\n", global_player_number);
//      fprintf(game_file, "SCORE=%d\n", my_player->score);
//      fprintf(game_file, "MONEY=%d\n", my_player->money);
//      fprintf(game_file, "TYPE=%f\n", my_player->type);
//      fprintf(game_file, "TYPESAVED=%f\n", my_player->type_saved);
//      for (count = 0; count < WEAPONS; count++)
//        fprintf(game_file, "WEAPON=%d %d\n", count, my_player->nm[count]);
//      for (count = 0; count < ITEMS; count++)
//        fprintf(game_file, "ITEM=%d %d\n", count, my_player->ni[count]);
//      fprintf(game_file, "***\n");
//      player_count++;
//    }
//
//  fclose(game_file);
//
//  // save the current config
//  file_path = (char *) calloc( strlen(global->configDir) +
//                               strlen(global->game_name) + 24,
//                               sizeof(char) );
//  if (! file_path)
//    return FALSE;
//  sprintf(file_path, "%s/%s.txt", global->configDir, global->game_name);
//  Save_Game_Settings(global, env, file_path);
//  free(file_path);
//
//  return TRUE;
//}



/*
This function attempts to load a saved
game.
The function returns TRUE on success and
FALSE is an error occures.
-- Jesse

Changed to work in binary mode, the original method is below.
-- Sven
*/
bool Load_Game(GLOBALDATA *global, ENVIRONMENT *env)
{
  ifstream ifsFile;
  char *file_path;
  char chArea[NAME_LENGTH] = "";
  double dData = 0.0;
  bool bResult = true;
  bool bIsFinished = false;
  bool bIsData = false;
  int player_number = 0;
  int player_count = 0;
  int max_players = 10; // default
  int count = 0;

  int revengees[MAXPLAYERS];
  for (int i = 0; i < MAXPLAYERS; i++) revengees[i] = -1;

  // load config settings
  file_path = (char *) calloc( strlen(global->configDir) +
                               strlen(global->game_name) + 24,
                               sizeof(char) );
  if (! file_path)
    bResult = false;
  else
    {
      sprintf(file_path, "%s/%s.dat", global->configDir, global->game_name);
      bResult = Load_Game_Settings(global, env, file_path);
      free(file_path);
    }

  if (bResult)
    {
      // Now the .sav can be opened
      file_path = (char *) calloc( strlen(global->configDir) +
                                   strlen(global->game_name) + 24,
                                   sizeof(char) );
      if (! file_path)
        bResult = false;
      else
        {
          sprintf(file_path, "%s/%s.sav", global->configDir, global->game_name);
          ifsFile.open(file_path);
          free(file_path);
          if (!ifsFile.is_open())
            bResult = false;
        }
    }

  if (bResult)
    {
      // Before we can add players, we need to remove old players
      PLAYER * tempPlayer;
      while (global->numPlayers)
        {
          tempPlayer = global->players[global->numPlayers - 1];
          if (tempPlayer)
            global->removePlayer(tempPlayer);
          else
            global->numPlayers--;
        }
      tempPlayer = NULL;
    }

  // if everything worked, the .sav can be loaded
  while (!bIsFinished && (player_count < max_players) && (bResult = popData(chArea, bIsData, ifsFile)))
    {
      if (!bIsData && !strcmp(chArea, FILEPART_ENDFILE))   bIsFinished = true;
      else if (!bIsData && !strcmp(chArea, FILEPART_ENDPLYR))
        {
          // Next Player will come!
          player_number = 0;
          player_count++;
        }
      else if (!strcmp(chArea, "MaxRounds  ")) bResult = popData(global->rounds,        ifsFile);
      else if (!strcmp(chArea, "CurrRound  ")) bResult = popData(global->currentround,  ifsFile);
      else if (!strcmp(chArea, "Campaign   ")) bResult = popData(global->campaign_mode, ifsFile);
      else if (!strcmp(chArea, "PlayerCount")) bResult = popData(max_players,    ifsFile);
      else if (!strcmp(chArea, "PlayerNumber") && (bResult = popData(player_number, ifsFile)) )
        {
          global->addPlayer( global->allPlayers[player_number] );
          global->players[player_count]->initialise();
        }
      else if (!strcmp(chArea, "Score       ")) bResult = popData(global->players[player_count]->score,      ifsFile);
      else if (!strcmp(chArea, "Money       ")) bResult = popData(global->players[player_count]->money,      ifsFile);
      else if (!strcmp(chArea, "PlayerType  ")) bResult = popData(global->players[player_count]->type,       ifsFile);
      else if (!strcmp(chArea, "TypeSaved   ")) bResult = popData(global->players[player_count]->type_saved, ifsFile);
      else if (!strcmp(chArea, "Revengee Nr ")) bResult = popData(revengees[player_count], ifsFile);
      else if (bIsData)
        {
          // weapon or item data
          count = 0;
          while (!bIsFinished && ((count < WEAPONS) || (count < ITEMS)))
            {
              if ((count < WEAPONS) && !strcmp(chArea, weapon[count].name))
                {
                  bResult = popData(global->players[player_count]->nm[count], ifsFile);
                  bIsFinished = true;
                }
              if ((count < ITEMS) && !strcmp(chArea, item[count].name))
                {
                  bResult = popData(global->players[player_count]->ni[count], ifsFile);
                  bIsFinished = true;
                }
              count++;
            }
          // This was only used locally and needs to be resetted.
          bIsFinished = false;
        }
      else if (!bIsFinished && bIsData)
        bResult = popData(dData, ifsFile); // Something we do not know (any more/yet), so dump it!
    }

  if (ifsFile.is_open())
    ifsFile.close();

  // Now apply revengees:
  if (bResult)
    {
      for (int i = 0; i < max_players; i++)
        {
          if ( (revengees[i] > -1) && (revengees[i] != i) )
            {
              int iRevengeNr = 0;
              while ( strcmp(global->allPlayers[revengees[i]]->getName(), global->players[iRevengeNr]->getName()) )
                iRevengeNr++;
              global->players[i]->revenge = global->players[iRevengeNr];
            }
          else
            global->players[i]->revenge = NULL;
        }
    }

  return(bResult);
}

//int Load_Game(GLOBALDATA *global)
//{
//  char line[512];
//  char *field, *value;
//  int index, amount, player_count = 0;
//  FILE *my_file;
//  char *file_path;
//  int stage = NO_STAGE;
//  char *got_line;
//  int global_player_number;
//
//  // load config settings
//  /*
//  file_path = (char *) calloc( strlen(homedir) +
//                               strlen(global->game_name) + 24,
//                               sizeof(char) );
//  if (! file_path)
//    return FALSE;
//  sprintf(file_path, "%s/.atanks/%s.txt", homedir, global->game_name);
//  my_file = fopen(file_path, "r");
//  free(file_path);
//  if (! my_file)
//    return FALSE;
//  global->loadFromFile_Text(my_file);
//  env->loadFromFile_Text(my_file);
//  fclose(my_file);
//  */
//
//  file_path = (char *) calloc( strlen(global->configDir) +
//                               strlen(global->game_name) + 24,
//                               sizeof(char) );
//  if (! file_path)
//    return FALSE;
//
//  sprintf(file_path, "%s/%s.sav", global->configDir, global->game_name);
//
//  my_file = fopen(file_path, "r");
//  free(file_path);
//  if (! my_file)
//    return FALSE;
//
//  // read a line from the file
//  got_line = fgets(line, 512, my_file);
//  // keep reading until we hit EOF or error
//  while ( (got_line) && (player_count < MAXPLAYERS) )
//    {
//      // clear end of line
//      if ( strchr(line, '\n') )
//        strchr(line, '\n')[0] = '\0';
//      if ( strchr(line, '\r') )
//        strchr(line, '\r')[0] = '\0';
//
//      // check to see if we found a new stage
//      if (! strcasecmp(line, "global") )
//        stage = GLOBAL_STAGE;
//      else if (! strcasecmp(line, "environment") )
//        stage = ENVIRONMENT_STAGE;
//      else if (! strcasecmp(line, "players") )
//        stage = PLAYER_STAGE;
//
//      // not a new stage, seperate field and value
//      else
//        {
//          value = strchr(line, '=');
//          if (value)        // valid line
//            {
//              field = line;
//              value[0] = '\0';   // seperate field and value;
//              value++;           // go to first place after =
//              // interpret data
//              switch (stage)
//                {
//                case GLOBAL_STAGE:
//                  if (! strcasecmp(field, "rounds") )
//                    sscanf(value, "%lf", &global->rounds);
//                  else if (! strcasecmp(field, "currentround") )
//                    sscanf(value, "%d", &global->currentround);
//                  else if (! strcasecmp(field, "campaignmode") )
//                    sscanf(value, "%lf", &global->campaign_mode);
//
//                  break;
//                case ENVIRONMENT_STAGE:
//
//                  break;
//                case PLAYER_STAGE:
//                  if (! strcasecmp(field, "playernumber") )
//                    {
//                      sscanf(value, "%d", &global_player_number);
//                      global->addPlayer( global->allPlayers[global_player_number] );
//                      /*  the loading of the preferences is (unfortunately) disabled and I do not know why.
//                          As long as they *are* disabled, PerPlay-Config bots have to get a new config here,
//                          or they use the global config, rendering PerPlay-Config useless. */
//                      if	( (global->players[player_count]->preftype	==	PERPLAY_PREF)
//                           && (global->players[player_count]->type			!=	HUMAN_PLAYER))
//                        global->players[player_count]->generatePreferences();
//                      global->players[player_count]->initialise();
//                    }
//                  else if (! strcasecmp(field, "score") )
//                    sscanf(value, "%d", &(global->players[player_count]->score) );
//                  else if (! strcasecmp(field, "money") )
//                    sscanf(value, "%d", &(global->players[player_count]->money) );
//                  else if (! strcasecmp(field, "type") )
//                    sscanf(value, "%lf", &(global->players[player_count]->type) );
//                  else if (! strcasecmp(field, "typesaved") )
//                    sscanf(value, "%lf", &(global->players[player_count]->type_saved) );
//                  else if (! strcasecmp(field, "weapon") )
//                    {
//                      sscanf(value, "%d %d", &index, &amount);
//                      global->players[player_count]->nm[index] = amount;
//                    }
//                  else if (! strcasecmp(field, "item") )
//                    {
//                      sscanf(value, "%d %d", &index, &amount);
//                      global->players[player_count]->ni[index] = amount;
//                    }
//                  break;
//                }    // end of stage
//
//            }       //     end of valid line
//          else if ( (! strcmp(line, "***") ) && (stage == PLAYER_STAGE) )
//            player_count++;
//
//        }     // end of field value area
//
//      // read next line of file
//      got_line = fgets(line, 512, my_file);
//
//    }    // end of reading file
//  fclose(my_file);
//  return TRUE;
//}



/*
Check to see if a saved game exists with the given name.

As the files are binary, this method is changed to iostream.
The original method is below.
-- Sven
*/
bool Check_For_Saved_Game(GLOBALDATA *global)
{
  bool bResult = true;
  ifstream ifsFile;
  char *path_name;

  path_name = (char *) calloc( strlen(global->configDir) + strlen(global->game_name) + 24,
                               sizeof(char) );
  if (! path_name)
    bResult = false;
  else
    {
      sprintf(path_name, "%s/%s.sav", global->configDir, global->game_name);
      ifsFile.open(path_name);
      free(path_name);
    }
  if (ifsFile.is_open())
    ifsFile.close();
  else
    bResult = false;
  return(bResult);
}

//int Check_For_Saved_Game(GLOBALDATA *global)
//{
//  FILE *my_file;
//  char *path_name;
//
//  path_name = (char *) calloc( strlen(global->configDir) + strlen(global->game_name) + 24,
//                               sizeof(char) );
//  if (! path_name)
//    return FALSE;
//
//  sprintf(path_name, "%s/%s.sav", global->configDir, global->game_name);
//
//  my_file = fopen(path_name, "r");
//  free(path_name);
//  if (my_file)
//    {
//      fclose(my_file);
//      return TRUE;
//    }
//  else
//    return FALSE;
//
//}



/*
This function copies the atanks config file
from the HOME_DIR folder to HOME_DIR/.atanks
If the .atanks folder does not exist, this
function will create it.
Returns TRUE on success and FALSE on error.

Changed to use binary mode, the original is below.
-- Sven
*/
bool Copy_Config_File(GLOBALDATA *global)
{
  ifstream ifsSource;
  ofstream ofsDest;
  char chLine[256] = "";

  bool bResult = true;

  char source_path[1024], dest_path[1024];
  char *my_home_folder;

  int status = 0;

  // figure out where home is
  my_home_folder = getenv(HOME_DIR);
  if (! my_home_folder)
    my_home_folder = (char *)".";

  // check to see if the config file has already been copied
  snprintf(source_path, 1024, "%s/atanks-config.dat", global->configDir);
  ifsSource.open(source_path); // We use the ifstream, so that the file won't be generated right now if missing

  if (ifsSource.is_open())     // config file is in the right place
    ifsSource.close();
  else
    {
      // It isn't where it is supposed to be, so we have some work to be done:

      // First: make sure the directory exists:
      DIR * pDestDir = opendir(global->configDir);
      if (!pDestDir)
        {
#ifdef WIN32
          status = mkdir(global->configDir);
#else
          status = mkdir(global->configDir, 0700);
#endif
          if (status == -1)
            {
              // No, something is badly wrong!
              printf( (char *)"Error occured. Unable to create sub directory:\n\"%s\"", global->configDir);
              bResult = false;
            }
        }
      else
        {
          closedir(pDestDir);
          pDestDir = NULL;
        }

      // check to make sure we have a source file
      if (bResult)
        {
          snprintf(source_path, 1024, "%s/.atanks-config.dat", my_home_folder);
          ifsSource.open(source_path, ios::binary);
          if (!ifsSource.is_open())
            bResult = false;
        }

      if (bResult)
        {
          // we already have an open source file, create destination file
          snprintf(dest_path, 1024, "%s/atanks-config.dat", global->configDir);
          ofsDest.open(dest_path, ios::binary);
          if (!ofsDest.is_open())
            {
              // We have the directory now, but still no file...
              printf("Unable to create destination file:\n\"%s\"", dest_path);
              ifsSource.close();
              bResult = false;
            }
        }

      while (bResult && (!ifsSource.eof()))
        {
          // we have open files, let's copy
          ifsSource.getline(chLine, 255);
          if (ifsSource.good())
            ofsDest << chLine << endl;
          if (!ifsSource.good() || !ofsDest.good())
            bResult = false;
        }
    }

  if (ifsSource.is_open())  ifsSource.close();
  if (ofsDest.is_open())    ofsDest.close();

  return(bResult);
}

//int Copy_Config_File(GLOBALDATA *global)
//{
//  FILE *source_file, *dest_file;
//  char source_path[1024], dest_path[1024];
//  char buffer[1024];
//  char *my_home_folder;
//  int status;
//
//  // figure out where home is
//  my_home_folder = getenv(HOME_DIR);
//  if (! my_home_folder)
//    my_home_folder = (char *)".";
//
//  // check to see if the config file has already been copied
////  snprintf(source_path, 1024, "%s/atanks-config.txt", global->configDir);
//  snprintf(source_path, 1024, "%s/atanks-config.dat", global->configDir);
//  source_file = fopen(source_path, "r");
//  if (source_file)     // config file is n the right place
//    {
//      fclose(source_file);
//      return TRUE;
//    }
//
//  /*
//  // check to make sure we have a source file
//  snprintf(source_path, 1024, "%s/.atanks-config.txt", my_home_folder);
//  source_file = fopen(source_path, "r");
//  if (! source_file)
//     return TRUE;
//  */
//
//  // file not copied yet, create the required directory
//  snprintf(buffer, 1024, "%s/.atanks", my_home_folder);
//#ifdef WIN32
//  status = mkdir(buffer);
//#else
//  status = mkdir(buffer, 0700);
//#endif
//  if (status == -1)
//    {
//      printf( (char *)"Error occured. Unable to create sub directory.\n");
//      return FALSE;
//    }
//
//  // check to make sure we have a source file
////  snprintf(source_path, 1024, "%s/.atanks-config.txt", my_home_folder);
//  snprintf(source_path, 1024, "%s/.atanks-config.dat", my_home_folder);
//  source_file = fopen(source_path, "r");
//  if (! source_file)
//    return TRUE;
//
//  // we already have an open source file, create destination file
////  snprintf(dest_path, 1024, "%s/atanks-config.txt", global->configDir);
//  snprintf(dest_path, 1024, "%s/atanks-config.dat", global->configDir);
//  dest_file = fopen(dest_path, "wb");
//  if (! dest_file)
//    {
//      printf( (char *)"Unable to create destination file.\n");
//      fclose(source_file);
//      return FALSE;
//    }
//
//  // we have open files, let's copy
//  status = fread(buffer, 1, 1024, source_file);
//  while (status)
//    {
//      status = fwrite(buffer, 1, 1024, dest_file);
//      status = fread(buffer, 1, 1024, source_file);
//    }
//
//  fclose(source_file);
//  fclose(dest_file);
//  return TRUE;
//}



/*
Displays lines of text.
*/
void renderTextLines (GLOBALDATA *global, ENVIRONMENT *env,
                      const LINESEQ& lines, int scrollOffset,
                      const FONT* fnt, const int spacing )
{
  const int textheight = text_height (fnt) ;
  const int textheight_s = textheight * spacing ;
  int yOffset = scrollOffset;

  for (int count = 0;
       (count != lines.size()) && (yOffset < global->screenHeight);
       count++)
    {
      const char *text = lines[count] + 1;
      const char header = lines[count][0];
      int tLen = text_length (fnt, text);

      switch (header)
        {
        case 'T':
        case 't':
          textout_centre_ex (env->db, fnt, text,
                             global->halfWidth + 2, global->halfHeight + yOffset + 2, BLACK, -1);
          textout_centre_ex (env->db, fnt, text,
                             global->halfWidth, global->halfHeight + yOffset, WHITE, -1);
          if (header == 'T')
            {
              yOffset += textheight + 1;
              hline (env->db, global->halfWidth - tLen / 2,
                     global->halfHeight + yOffset, global->halfWidth + tLen / 2,
                     WHITE);
              hline (env->db, global->halfWidth - tLen / 2 + 2,
                     global->halfHeight + yOffset + 2,
                     global->halfWidth + tLen / 2 + 2, BLACK);
              yOffset -= textheight / 2;
            }
          break;
        case 'L':
          textout_centre_ex (env->db, fnt, text, global->halfWidth,
                             global->halfHeight + yOffset, BLACK, -1);
          break;
        }
      yOffset += textheight_s;
    }
}




/*
Scroll text in a box
*/
void scrollTextList (GLOBALDATA *global, ENVIRONMENT *env,
                     const LINESEQ& lines)
{
  /* Justin says: this function, along with renderTextLines, aren't
  exactly efficient.  I think they could use rewrite. */
  static const int numItemsSrc[] = { 100, 30 } ;

  DATAFILE *dffont ;
  const FONT* fnt = font;
  int spacing = 2;
  int tOffset = rand ();
  int itemType = rand () / (RAND_MAX/2 + 1) ;
  int numItems = numItemsSrc[itemType];

  /* Attempt to load the (char *)"big" font */
  if (( dffont = load_datafile_object( (char *)"bigfont.dat", "bigfont") ))
    {
      fnt = (FONT*)dffont->dat ;
      spacing = 1 ;
    }

  draw_circlesBG (global, env->db, 0, 0, global->screenWidth, global->screenHeight);
  drawMenuBackground (global, env, BACKGROUND_BLANK, abs (tOffset), numItems);
  quickChange (global, env->db);
  set_clip_rect (env->db, global->halfWidth - 300 + 1, 100 + 1,
                 global->halfWidth + 300 - 1, global->screenHeight - 100 - 1);
  int scrollOffset = 0;
  flush_inputs ();
  if (! global->os_mouse) show_mouse (NULL);

  do
    {
      drawMenuBackground (global, env, BACKGROUND_BLANK, abs (tOffset), numItems);
      renderTextLines (global, env, lines, scrollOffset,
                       fnt, spacing );
      blit (env->db, screen, global->halfWidth - 300, 100, global->halfWidth - 300, 100,
            601, global->screenHeight - 199);
      LINUX_REST;
      tOffset++;
      scrollOffset--;
      if (scrollOffset < -(global->halfHeight - 100 + lines.size() * 20))
        scrollOffset = global->halfHeight - 100;
      if (global->close_button_pressed)
        break;
    }
  while (!keypressed () && !mouse_b);

  if (! global->os_mouse) show_mouse (screen);
  set_clip_rect (env->db, 0, 0, (global->screenWidth-1), (global->screenHeight-1));
  flush_inputs ();
  if ( dffont )
    unload_datafile_object( dffont );
}



/* Flush key buffer and waits for button releases */
void flush_inputs()
{
  do { }
  while (mouse_b);
  clear_keybuf();
}



// This file loads weapons, naturals and items
// from a text file
// Returns TRUE on success and FALSE on failure
int Load_Weapons_Text(GLOBALDATA *global)
{
  FILE *wfile;
  char *path_to_file;
  char *status;
  char line[256];
  int file_stage = 0;     // weapons, natruals, items
  int data_stage = 0;         // name, descrption, data
  int item_count = 0, weapon_count = 0, natural_count = 0;

  // get path name
  path_to_file = (char *) calloc( strlen(global->dataDir) + 32, sizeof(char) );
  if (! path_to_file)
    {
      printf( (char *)"Memory error occured while loading weapons.\n");
      return FALSE;
    }

  if (global->language == LANGUAGE_ENGLISH)
    sprintf(path_to_file, "%s/weapons.txt", global->dataDir);
  if (global->language == LANGUAGE_PORTUGUESE)
    sprintf(path_to_file, "%s/weapons.pt_BR.txt", global->dataDir);
  if (global->language == LANGUAGE_FRENCH)
    sprintf(path_to_file, "%s/weapons_fr.txt", global->dataDir);
  if (global->language == LANGUAGE_GERMAN)
    sprintf(path_to_file, "%s/weapons_de.txt", global->dataDir);
  if (global->language == LANGUAGE_SLOVAK)
    sprintf(path_to_file, "%s/weapons_sk.txt", global->dataDir);

  // open file
  wfile = fopen(path_to_file, "r");
  // free memory
  free(path_to_file);
  if (! wfile)
    {
      printf( (char *)"Unable to open weapons file. (%s)\n", path_to_file);
      return FALSE;
    }

  // read line
  status = fgets(line, 256, wfile);
  while (status)
    {
      // clear end of line
      if ( strchr(line, '\n') )
        strchr(line, '\n')[0] = '\0';
      if ( strchr(line, '\r') )
        strchr(line, '\r')[0] = '\0';

      // skip # and empty lines
      if ( (! (line[0] == '#') ) && ( strlen(line) > 2 ) )
        {
          // check for header
          if (! strcasecmp(line, "*WEAPONS*") )
            {
              file_stage = 0;
              data_stage = 0;
            }
          else if (! strcasecmp(line, "*NATURALS*") )
            {
              file_stage = 1;
              data_stage = 0;
            }
          else if (! strcasecmp(line, "*ITEMS*") )
            {
              file_stage = 2;
              data_stage = 0;
            }

          // not a special line, let's read some data
          else
            {
              // weapon
              if ( (file_stage == 0) && (weapon_count < WEAPONS) )
                {
                  if (data_stage == 0)   // name
                    strcpy(weapon[weapon_count].name, line);
                  else if (data_stage == 1)
                    strcpy(weapon[weapon_count].description, line);
                  else if (data_stage == 2)
                    {
                      sscanf(line, "%d %d %lf %lf %d %d %d %d %d %d %d %d %d %d %d %d %d %lf %d %lf %lf %lf %d %lf",
                             &(weapon[weapon_count].cost),
                             &(weapon[weapon_count].amt),
                             &(weapon[weapon_count].mass),
                             &(weapon[weapon_count].drag),
                             &(weapon[weapon_count].radius),
                             &(weapon[weapon_count].sound),
                             &(weapon[weapon_count].etime),
                             &(weapon[weapon_count].damage),
                             &(weapon[weapon_count].eframes),
                             &(weapon[weapon_count].picpoint),
                             &(weapon[weapon_count].spread),
                             &(weapon[weapon_count].delay),
                             &(weapon[weapon_count].noimpact),
                             &(weapon[weapon_count].techLevel),
                             &(weapon[weapon_count].warhead),
                             &(weapon[weapon_count].numSubmunitions),
                             &(weapon[weapon_count].submunition),
                             &(weapon[weapon_count].impartVelocity),
                             &(weapon[weapon_count].divergence),
                             &(weapon[weapon_count].spreadVariation),
                             &(weapon[weapon_count].launchSpeed),
                             &(weapon[weapon_count].speedVariation),
                             &(weapon[weapon_count].countdown),
                             &(weapon[weapon_count].countVariation) );
                    }
                  data_stage++;
                  if (data_stage > 2)
                    {
                      data_stage = 0;
                      weapon_count++;
                    }
                }       // end of weapon section

              // naturals
              else if ( (file_stage == 1) && (natural_count < NATURALS) )
                {
                  if (data_stage == 0)   // name
                    strcpy(naturals[natural_count].name, line);
                  else if (data_stage == 1)
                    strcpy(naturals[natural_count].description, line);
                  else if (data_stage == 2)
                    {
                      sscanf(line, "%d %d %lf %lf %d %d %d %d %d %d %d %d %d %d %d %d %d %lf %d %lf %lf %lf %d %lf",
                             &(naturals[natural_count].cost),
                             &(naturals[natural_count].amt),
                             &(naturals[natural_count].mass),
                             &(naturals[natural_count].drag),
                             &(naturals[natural_count].radius),
                             &(naturals[natural_count].sound),
                             &(naturals[natural_count].etime),
                             &(naturals[natural_count].damage),
                             &(naturals[natural_count].eframes),
                             &(naturals[natural_count].picpoint),
                             &(naturals[natural_count].spread),
                             &(naturals[natural_count].delay),
                             &(naturals[natural_count].noimpact),
                             &(naturals[natural_count].techLevel),
                             &(naturals[natural_count].warhead),
                             &(naturals[natural_count].numSubmunitions),
                             &(naturals[natural_count].submunition),
                             &(naturals[natural_count].impartVelocity),
                             &(naturals[natural_count].divergence),
                             &(naturals[natural_count].spreadVariation),
                             &(naturals[natural_count].launchSpeed),
                             &(naturals[natural_count].speedVariation),
                             &(naturals[natural_count].countdown),
                             &(naturals[natural_count].countVariation) );
                    }

                  data_stage++;
                  if (data_stage > 2)
                    {
                      data_stage = 0;
                      natural_count++;
                    }

                }       // end of naturals

              // item section
              else if ( (file_stage == 2) && (item_count < ITEMS) )
                {
                  if (data_stage == 0)   // name
                    strcpy(item[item_count].name, line);
                  else if (data_stage == 1)
                    strcpy(item[item_count].description, line);
                  else if (data_stage == 2)
                    {
                      sscanf(line, "%d %d %d %d %d %lf %lf %lf %lf %lf %lf",
                             &(item[item_count].cost),
                             &(item[item_count].amt),
                             &(item[item_count].selectable),
                             &(item[item_count].techLevel),
                             &(item[item_count].sound),
                             &(item[item_count].vals[0]),
                             &(item[item_count].vals[1]),
                             &(item[item_count].vals[2]),
                             &(item[item_count].vals[3]),
                             &(item[item_count].vals[4]),
                             &(item[item_count].vals[5]) );
                    }

                  data_stage++;
                  if (data_stage > 2)
                    {
                      data_stage = 0;
                      item_count++;
                    }
                }         // end of items

            }       // end of reading data from a valid line

        }     // end of valid line

      // read in data
      status = fgets(line, 256, wfile);
    }

  // close file
  fclose(wfile);
  // go home
  return TRUE;
}




/*
Filter out files that do not have .sav in the name.
*/
#ifndef MACOSX
int Filter_File( const struct dirent *my_file )
#else
int Filter_File( struct dirent *my_file )
#endif
{
  if ( strstr(my_file->d_name, ".sav") )
    return TRUE;
  else
    return FALSE;
}

/*
This function finds a list of saved games on your profile.
On error, NULL is returned. If all goes well, a list of file names
are returned.
After use, the return value should be freed.
*/
#ifndef WIN32
struct dirent ** Find_Saved_Games(GLOBALDATA *global, int *num_files_found)
  {
    struct dirent **my_list;
    int status;

    status = scandir(global->configDir, &my_list, Filter_File, alphasort);
    if (status < 0)
      {
        printf( (char *)"Error trying to find saved games.\n");
        return NULL;
      }

    *num_files_found = status;
    return my_list;
  }
#endif



/*
This function hunts for saved games. If games are found, the
function returns an array of filenames. If an error occures
or no files are found, NULL is returned.
*/
#ifdef WIN32
struct dirent ** Find_Saved_Games(GLOBALDATA *global, int *num_files_found)
  {
    struct dirent **my_list;
    struct dirent *one_file;
    int file_count = 0;
    DIR *game_dir;

    my_list = (struct dirent **) calloc(256, sizeof(struct dirent *));
    if (! my_list)
      return NULL;

    game_dir = opendir(global->configDir);
    if (! game_dir)
      return NULL;

    one_file = readdir(game_dir);
    while ( (one_file) && (file_count < 256) )
      {
        // check to see if this is a save game file
        if ( strstr( one_file->d_name, ".sav" ) )
          {
            my_list[file_count] = (struct dirent *) calloc(1, sizeof(struct dirent) );
            if ( my_list[file_count] )
              {
                memcpy(my_list[file_count], one_file, sizeof(struct dirent) );
                file_count++;
              }
          }

        one_file = readdir(game_dir);
      }

    closedir(game_dir);
    *num_files_found = file_count;
    return my_list;
  }
#endif

