/* disk.c: low-level access functions to the disk
 *
 * Copyright (C) 1995-97 Roman Hodek <Roman.Hodek@informatik.uni-erlangen.de>
 *               1996-97 Michael Schlueter <schlue00@marvin.informatik.uni-dortmund.de>
 *
 * This program is free software.  You can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation: either version 2 or
 * (at your option) any later version.
 *
 */

/* $Id: disk.c,v 1.10 1997/08/22 12:35:38 rnhodek Exp $
 *
 * $Log: disk.c,v $
 * Revision 1.10  1997/08/22 12:35:38  rnhodek
 * One more libc6 fix
 *
 * Revision 1.9  1997/08/11 13:14:59  rnhodek
 * Include <asm/unistd.h> to get definition of __NR__llseek if it exists
 *
 * Revision 1.8  1997/07/08 18:59:00  rnhodek
 * Ouch, too many changes without commits in between...
 * Implemented moving of partitions
 * Implemented checking of "extended start condition" (great changes in
 *   new_partition())
 * Some more tests in verify()
 *
 * Revision 1.7  1997/06/22 10:36:53  rnhodek
 * Added msread, mswrite for moving partitions
 *
 * Revision 1.6  1997/06/22 10:30:54  rnhodek
 * Add __attribute__((unused)) to cvid
 *
 * Revision 1.5  1997/06/21 20:47:42  rnhodek
 * Added RCS keywords
 *
 * Revision 1.4  1997/06/13 12:57:03  rnhodek
 * Little fixes
 * 
 * Revision 1.3  1997/06/13 12:48:22  rnhodek
 * Nicer dump_sector
 * 
 * Revision 1.2  1997/06/13 10:18:45  rnhodek
 * Implemented settings bootflags on XGM part.
 * allow write command if opened read-only and ALPHA_VERSION; actual writing
 *   skipped in swrite
 * nicer output of list_table
 * 
 * Revision 1.1  1997/06/11 14:36:35  rnhodek
 * Initial revision
 * 
 * Revision 1.1.1.1  1997/06/11 14:36:35  rnhodek
 * Started using CVS for atafdisk
 *
 */

#ifndef lint
static char vcid[] __attribute__ ((unused)) =
"$Id: disk.c,v 1.10 1997/08/22 12:35:38 rnhodek Exp $";
#endif /* lint */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>
#ifdef __linux__
#include <asm/unistd.h>
#endif
#include <fcntl.h>
#include <errno.h>

#include "fdisk.h"
#include "version.h"
#include "disk.h"
#include "input.h"
#include "util.h"
#include "readpart.h"


/*
 * sseek: seek to specified sector - return 0 on failure
 *
 * For >4GB disks lseek needs a > 32bit arg, and we have to use llseek.
 * On the other hand, a 32 bit sector number is OK until 2TB.
 * The routines _llseek and sseek below are the only ones that
 * know about the loff_t type.
 */

#ifdef __NR__llseek

static _syscall5( int, _llseek, uint, fd, ulong, hi, ulong, lo,
		  loff_t *, res, uint, wh );

/* seek to a sector */
int sseek( unsigned int fd, unsigned long s )
{
    loff_t in, out;

    in = (loff_t)s * SECTOR_SIZE;
    out = 1;

    if (_llseek (fd, in>>32, in & 0xffffffff, &out, SEEK_SET) != 0) {
	perror( "llseek" );
	fprintf( stderr, "seek error - cannot seek to sector %lu\n", s );
	return 0;
    }

    if (in != out) {
	fprintf( stderr, "seek error: wanted 0x%08x%08x, got 0x%08x%08x\n",
		 (uint)(in>>32), (uint)(in & 0xffffffff),
		 (uint)(out>>32), (uint)(out & 0xffffffff) );
	return 0;
    }
    return 1;
}

#else

/* Sorry, kernel doesn't know _llseek call :-((( */

int sseek( unsigned int fd, unsigned long s )
{
    off_t in, out;

    in = s * SECTOR_SIZE;
    out = lseek( fd, in, SEEK_SET );

    if (in != out) {
	if (out == -1)
	    perror( "lseek" );
	else
	    fprintf( stderr, "seek error: wanted 0x%08lx, got 0x%08lx\n",
		     in, out );
	return 0;
    }
    return 1;
}

#endif

/* read one sector */
void sread( char *buffer, unsigned long secno )
{
    if (!sseek( fd, secno ))
	fatal( unable_to_seek );
    if (read( fd, buffer, SECTOR_SIZE ) != SECTOR_SIZE) {
	perror( "read" );
	fatal( unable_to_read );
    }
}

/* read multiple sectors (for moving) */
int msread( char *buffer, unsigned long secno, unsigned n )
{
    if (!sseek( fd, secno )) {
	perror( "seek" );
	return( -1 );
    }
    if (read( fd, buffer, n*SECTOR_SIZE ) != n*SECTOR_SIZE) {
	perror( "read" );
	return( -1 );
    }
    return( 0 );
}

#ifdef ALPHA_VERSION
void dump_sector( unsigned long sec, unsigned char *buf )
{
    int i, j;
    struct rootsector *rs = (struct rootsector *)buf;
    
    printf( "About to write to sector %lu:\n", sec );
    for( i = 0; i < SECTOR_SIZE; i += 16 ) {
	printf( "%03x: ", i );
	for( j = 0; j < 16; ++j )
	    printf( "%02x ", buf[i+j] );
	for( j = 0; j < 16; ++j )
	    printf( "%c", buf[i+j] >= ' ' ? buf[i+j] : '.' );
	printf( "\n" );
    }

    swab_rs( rs );
    for( i = 0; i < 4; ++i ) {
	printf( "Slot #%d: flag=%02x ID=%c%c%c start=%9lu size=%9lu\n",
		i, rs->part[i].flag,
		rs->part[i].id[0], rs->part[i].id[1], rs->part[i].id[2],
		rs->part[i].start, rs->part[i].size );
    }
    printf( "hd_size=%lu bsl_st=%lu bsl_cnt=%lu cksum=%x bootable=%d\n",
	    rs->hd_size, rs->bsl_st, rs->bsl_cnt, rs->checksum,
	    check_rootsec_checksum( buf ) );
    swab_rs( rs );

    printf( "\n" );
}
#endif /* ALPHA_VERSION */

/* write a sector to disk; dump it first and ask if this is a ALPHA version */
void swrite( char *buf, unsigned long secno )
{
    if (!dangerous_write( fd, buf, secno ))
	fatal( unable_to_write );
}

/* write a sector to arbitrary fd (for restoring savetab file) */
int dangerous_write( int fd, char *buf, unsigned long secno )
{
    if (!sseek( fd, secno ))
	fatal( unable_to_seek );

#ifdef ALPHA_VERSION
    dump_sector( secno, buf );
    if (type_open == O_RDONLY ||
	!read_yesno( "Do you really want to write that data to disk", -1 ))
	/* just fake that operation was ok */
	return( 1 );
#endif /* ALPHA_VERSION */

    return( write( fd, buf, SECTOR_SIZE ) == SECTOR_SIZE );
}

/* write multiple sectors (for moving) */
int mswrite( char *buf, unsigned long secno, unsigned n )
{
    if (!sseek( fd, secno )) {
	perror( "seek" );
	return( -1 );
    }
#ifdef ALPHA_VERSION
    if (type_open == O_RDONLY) {
	printf( "ALPHA and ro, not actually writing %u sectors to %lu\n",
		n, secno );
	return( 0 );
    }
#endif
    if (write( fd, buf, n*SECTOR_SIZE ) != n*SECTOR_SIZE) {
	perror( "write" );
	return( -1 );
    }
    return( 0 );
}


/* Local Variables: */
/* tab-width: 8     */
/* End:             */
