/***************************************************************************

athcool is a program that controls the power-saving mode on AMD 
Athlon/Duron CPUs.

Since enabling power-saving mode, you can save power consumption, lower 
CPU temprature when CPU is idle.

athcool only sets/unsets the "Disconnect enable when STPGNT detected"
bit in the northbridge of the chipsets.
To save power, someone has to send the STPGNT signal when the CPU is
idle.
This is done by the ACPI subsystem when the C2 state is entered.
Hence, you need ACPI support in your kernel to achieve power savings.

!!!WARNING!!!
Depending on your motherboard and/or hardware components, 
enabling Athlon powersaving mode may cause:

 * noisy or distorted sound playback
 * a slowdown in harddisk performance
 * system locks or instability
 * massive filesystem corruption (rare, but observed at least once)

Before use athcool, you must recognize these potential DANGERS.
Please use athcool AT YOUR OWN RISK.

athcool is supplied "as is". The author disclaims all warranties,
expressed or implied. The author and any other persons
assume no liability for damages, direct or consequential,
which may result from the use of athcool.

   Osamu Kayasono <jacobi@jcom.home.ne.jp>
   http://members.jcom.home.ne.jp/jacobi/linux/softwares.html

  (Related links)
 * CoolOn Project [ written in Japanese ]
	http://homepage2.nifty.com/coolon/
 * pciutils
	http://atrey.karlin.mff.cuni.cz/~mj/pciutils.shtml
 * Athlon Powersaving HOWTO
	http://cip.uni-trier.de/nofftz/linux/Athlon-Powersaving-HOWTO.html
 * LVCool
	http://vcool.occludo.net/

 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdlib.h>
#include <unistd.h>
#include <stdio.h>
#include <stdint.h>
#include <sys/io.h>
#include <string.h>
#include <pci/pci.h>
#include  "athcool.h"
#include  "chips.h"

struct pci_filter filter;
struct pci_access *pacc;
struct device *scan_devices( void );
word get_conf_word(struct device *d, unsigned int pos);

static void print_title( void )
{
  printf( PACKAGE " version " VERSION );
  printf( " - control power-saving mode on AMD Athlon/Duron CPUs\n\n");
}

static void print_warnings( void )
{
  printf( "!!!WARNING!!!\n");
  printf( "Depending on your motherboard and/or hardware components, \n" );
  printf( "enabling Athlon powersaving mode may cause:\n" );
  printf( " * noisy or distorted sound playback\n" );
  printf( " * a slowdown in harddisk performance\n" );
  printf( " * system locks or instability\n" );
  printf( " * massive filesystem corruption (rare, but observed at least once)\n\n");
  printf( "Before use athcool, you must recognize these potential DANGERS.\n");
  printf( "Please use athcool AT YOUR OWN RISK.\n\n");
  printf( "athcool is supplied \"as is\". The author disclaims all warranties,\n");
  printf( "expressed or implied. The author and any other persons assume\n");
  printf( "no liability for damages, direct or consequential, which may \n");
  printf( "result from the use of athcool.\n\n");
  
}

static void list_chipsets( void )
{
  AthlonChips_s	*p_athlon;

  printf( "  Supported Chipsets:\n" );
  printf( "\t[Reg ID]  chipsets name\n" );
  for ( p_athlon = AthlonChips; p_athlon->name != NULL; p_athlon++ ) {
    printf( "\t[ 0x%02x ]  %s\n", p_athlon->chipreg, p_athlon->name );
  }
}

static void usage( const char *package )
{
  printf( " Usage: %s option\n", package );
  printf( "  supported options\n" );
  printf( "\ton\t Enable power-saving mode\n" );
  printf( "\toff\t Disable power-saving mode\n" );
  printf( "\tstat\t Query power-saving mode\n" );
  printf( "\tlist\t List supported chipsets\n" );
  printf( "\tfixup\t fixup some problems for specific hardwares (EXPERIMENTAL)\n" );
  printf( "\t\t (it may cause system locks or instability)\n\n" );

#ifdef ENABLE_FORCEID
  printf( " %s on|off|stat force RegID\n", package );
  printf( "\tmanual specification of chipsets\n" );
  printf( "\tDANGEROUS. Must not use this option, unless it is sure\n" );
  printf( "\tthat it has same config as your chipsets.\n\n" );
#endif
}

int setAthlonPowersave( int m, int force_id, struct device *first_dev )
{
  struct device	*d = NULL;
  const char	*msg;
  AthlonChips_s	*p_athlon = NULL;
  ChipReg_s	*p_reg;
  EachReg_s	*p_eachreg;
  unsigned int	regval_cur, regval_new;
  int fixup = FALSE;
  uint8_t addr;
  uint8_t rev;
#ifndef DISABLE_WRITE_REG
  unsigned int res;
#endif

  if (m & ATHLONPM_FIXUP) {
    fixup = TRUE;
    m = m & ~ATHLONPM_FIXUP;
  }

#ifdef ENABLE_FORCEID
  word class_id;

  if (force_id >= 0) {
    /*    d = first_dev; */
    /* search Host Bridge device */
    for ( d = first_dev; d != NULL; d= d->next ){
      class_id = get_conf_word(d, PCI_CLASS_DEVICE);
      if (class_id == PCI_CLASS_BRIDGE_HOST ) {
	printf( "Host Bridge found : %02x:%02x.%x , %04x %04x\n", 
		d->dev->bus, d->dev->dev, d->dev->func,
		d->dev->vendor_id, d->dev->device_id);
	break;
      }	/* else {
		printf( "skipping : %02x:%02x.%x [%04x], %04x %04x\n", 
		d->dev->bus, d->dev->dev, d->dev->func, class_id, 
		d->dev->vendor_id, d->dev->device_id);
	} */
    }
    if ( d == NULL) {
      printf( " Host Bridge is not found. exit. \n");
      exit (1);
    }
    /* search forced Reg ID */
    for ( p_reg = ChipReg; p_reg->regs->desc != NULL; p_reg++ ) {
      if ( p_reg->regid == force_id ) {	break; }
    }
    printf( " *** disabled auto detection on user request.\n" );
  } else {
#endif
    /* search supported chipset */
    for ( d = first_dev; d != NULL; d = d->next ) {
      for ( p_athlon = AthlonChips; p_athlon->name != NULL ; p_athlon++ ) {
	if ( (d->dev->vendor_id == p_athlon->vendorid) &&
	     (d->dev->device_id == p_athlon->deviceid) ) {
	  printf( "%s (%04x %04x) found\n", p_athlon->name,
		  d->dev->vendor_id, d->dev->device_id );
	  break;
	}
      }
      if ( p_athlon->name != NULL ) { break; }
    }
    
    if ( p_athlon == NULL || p_athlon->name == NULL ) {
      printf( "no supported Chipset found. nothing changed.\n" );
      return 1;
    }

    /* search chipset config */
    for ( p_reg = ChipReg; p_reg->regs->desc != NULL; p_reg++ ) {
      if ( p_reg->regid == p_athlon->chipreg ) { break; }
    }
#ifdef ENABLE_FORCEID
  }
#endif
  if ( p_reg->regs->desc == NULL ) {
    printf( "internal error."
	    " supported Chipset found,"
	    " but Chipset Configuration not found.\n" );
    return 1;
  }

  /* read current value of register */
  for ( p_eachreg = p_reg->regs; p_eachreg->desc != NULL; p_eachreg++ ) {
    switch (p_reg->tp) {
    case BYTE:
      regval_cur = pci_read_byte( d->dev, p_eachreg->addr );
      break;
    case SHORT:
      regval_cur = pci_read_word( d->dev, p_eachreg->addr );
      break;
    case LONG:
    default:
      regval_cur = pci_read_long( d->dev, p_eachreg->addr );
      break;
    }

    switch (m) {
    case ATHLONPM_STAT:
      /* begin of checking status */
      printf( "'%s' bit is %s.\n", p_eachreg->desc,
	      ( (regval_cur & p_eachreg->bit) == p_eachreg->bit ) ?
	      "enabled" : "disabled" );
      break;
      /* end of checking status */
    case ATHLONPM_ON:
    case ATHLONPM_OFF:
      /* begin of enabling/disabling bits */
      regval_new = regval_cur | p_eachreg->bit;
      msg = ( m == ATHLONPM_ON ) ? "enabl" : "disabl";
      printf( "%sing '%s' bit ... ", msg, p_eachreg->desc );
      
      if ( ((regval_cur == regval_new) && (m == ATHLONPM_ON)) ||
	   ((regval_cur != regval_new) && (m == ATHLONPM_OFF)) ) {
	printf( "already %sed.\n", msg );
	break;
      }

    //if ( m == ATHLONPM_OFF ) { regval_new = regval_cur ^ p_eachreg->bit; }
      if ( m == ATHLONPM_OFF ) { regval_new = regval_cur & ~p_eachreg->bit; }

      /* write new value into register */
      switch (p_reg->tp) {
      case BYTE:
#ifndef DISABLE_WRITE_REG
	res = pci_write_byte( d->dev, p_eachreg->addr, (uint8_t) regval_new );
#endif
	printf( " done\n" );
	printf( "\tAddress 0x%02X : 0x%02X -> 0x%02X\n", 
	      p_eachreg->addr, regval_cur, regval_new );
	break;
      case SHORT:
#ifndef DISABLE_WRITE_REG
	res = pci_write_word( d->dev, p_eachreg->addr, (uint16_t) regval_new );
#endif
	printf( " done\n" );
	/* x86 cpus are little-endian, so values are inverted in registers */
	printf( "\tAddress 0x%02X - 0x%02X : 0x%04X -> 0x%04X\n", 
		p_eachreg->addr, p_eachreg->addr+1, regval_cur, regval_new );
	break;
      case LONG:
      default:
#ifndef DISABLE_WRITE_REG
	res = pci_write_long( d->dev, p_eachreg->addr, (uint32_t) regval_new );
#endif
	printf( " done\n" );
	/* x86 cpus are little-endian, so values are inverted in registers */
	printf( "\tAddress 0x%02X - 0x%02X : 0x%08X -> 0x%08X\n", 
		p_eachreg->addr, p_eachreg->addr+3, regval_cur, regval_new );
	break;
      }
    
      break;
      /* end of enabling/disabling bits */
    }
  }

  /* fix something */
  if (fixup) {
    switch (p_reg->regid) {

      /* fix audio problem for VIA chipsets (0x70 bit 3) */
      /* from information provided by Wijatmoko U. Prayitno*/
      /* 0.3.5: enable bit3(Enhance CPU to PCI Write?),
	               bit2(PCI Master Read Buffering),
		       bit1(Delay Transaction) (=0x0E) on all VIA platform */
    case REGID_VIA_KT133:
    case REGID_VIA_KT266:
    case REGID_VIA_KT400:
      addr = 0x70;
      regval_cur = pci_read_byte( d->dev, addr );
      regval_new = regval_cur | 0x0E;
      printf( "Fixup for VIA audio problem ... ");
      if (regval_cur != regval_new) {
#ifndef DISABLE_WRITE_REG
	res = pci_write_byte( d->dev, addr, (uint8_t) regval_new );
#endif
	printf( "done.\n");
	printf( "\tAddress 0x%02X : 0x%02X -> 0x%02X\n", 
	      addr, regval_cur, regval_new );
      } else {
	printf( "already fixed.\n");
      }
      break;

      /* fixup for C1 HALT disconnect problem on nForce2 system */
      /* A hang is caused when the CPU generates a very fast CONNECT/HALT
	 cycle sequence.  Workaround is to set the SYSTEM_IDLE_TIMEOUT to
	 80 ns.
         'C18D' means nforce2 rev.C1 or greater */
      /* from information provided by Allen Martin. Thanks Herrmann. */
    case REGID_NVIDIA_NFORCE2:
      if (m == ATHLONPM_ON) {
	addr = 0x6c;
	rev = pci_read_byte(d->dev, PCI_REVISION_ID);
	/* Chip  Old Value  New Value
	   C17   0x1F0FFF01 0x1F01FF01
	   C18D  0x9F0FFF01 0x9F01FF01
	*/
	//	regval_new = rev < 0xC1 ? 0x1F0FFF01 : 0x9F0FFF01;
	regval_new = rev < 0xC1 ? 0x1F01FF01 : 0x9F01FF01;
	regval_cur = pci_read_long(d->dev, addr);
	printf( "Fixup for C1 HALT disconnect problem ... ");
	if (regval_cur != regval_new) {
#ifndef DISABLE_WRITE_REG
	  res = pci_write_long(d->dev, addr, regval_new);
#endif
	  printf( "done.\n");
	  printf( "\tAddress 0x%02X - 0x%02X : 0x%08X -> 0x%08X\n", 
		  addr, addr+3, regval_cur, regval_new );
	} else {
	  printf( "already fixed.\n");
	}
      } else {
	printf( "nForce2 fixup is needed with 'on' option.\n");
      }
      break;
    default:
      printf( "not prepared any fixup.\n");
      break;
    }
  }

  return 0;
}

#ifdef ENABLE_FORCEID
int hex2int(char c)
{ 
  int result;

  if     ( '0' <= c && c <= '9'){ result = c - '0';      } // 0〜9
  else if( 'a' <= c && c <= 'f'){ result = c - 'a' + 10; } // a〜f
  else if( 'A' <= c && c <= 'F'){ result = c - 'A' + 10; } // A〜F
  else                          { result = -1; }           // Others

  return result;
}

int getforceid ( char *idstr )
{
  int result = -1;
  int i = 0;

  if ((idstr[0] == '0') && (idstr[1] == 'x')) i = 2;

  if (strlen(idstr+i) < 2) {
    result = hex2int(idstr[i]);
  } else {
    result = hex2int(idstr[i]) * 16 + hex2int(idstr[i+1]);
  }

  return result;
}
#endif

int main ( int argc, char *argv[] )
{
  int	mode = ATHLONPM_USAGE;
  int	force_id = -1;
  int   fixup = FALSE;
  int	result, c;

  /* read optarg */
  for (c = 1; c < argc; c++ ) {
    if ( strcmp( argv[c], "on") == 0 ) {
      mode = ATHLONPM_ON;
    } else if ( strcmp( argv[c], "off" ) == 0 ) {
      mode = ATHLONPM_OFF;
    } else if ( strcmp( argv[c], "fixup" ) == 0 ) {
      fixup = TRUE;
    } else if ( strcmp( argv[c], "stat" ) == 0 ) {
      mode = ATHLONPM_STAT;
    } else if ( strcmp( argv[c], "list" ) == 0 ) {
      mode = ATHLONPM_LIST;
#ifdef ENABLE_FORCEID
    } else if ( strcmp( argv[c], "force" ) == 0 ) {
      c++;
      if (c < argc) force_id = getforceid( argv[c] );
#endif
    }
  }

  print_title();

  switch (mode) {
  case ATHLONPM_USAGE:
    usage( argv[0] );
    exit (1);
  case ATHLONPM_LIST:
    list_chipsets();
    exit (1);
  }

  if (fixup) { mode |= ATHLONPM_FIXUP; }

  print_warnings();

  if ( iopl( 3 ) < 0 ) {
    printf( "%s : must run as root. exit\n", argv[0] );
    exit (1);
  }

  /* init pciutil, and scan devices */
  pacc = pci_alloc();
  pci_filter_init( pacc, &filter );
  pci_init( pacc );

  result = setAthlonPowersave( mode, force_id, scan_devices() );

  /* clean up, exit */
  pci_cleanup( pacc );

  return result;
}
/* End Of File	**************************************************************/
