/***************************************************************************************
	Copyright 2000-2001 ATMEL Corporation.
	
	This file is part of atmel wireless lan drivers.

    Atmel wireless lan drivers is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    Atmel wireless lan drivers is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with Atmel wireless lan drivers; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

**************************************************************************************/


#include "vnet.h"
#include "interrupt.h"

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
// 
//    CardReset														      
//
//    This function resets the card
//						  
//
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

BOOLEAN
CardReset(PVNet_ADAPTER Adapter)
{
	USHORT					gcr;
	
	CardReadUshort(Adapter, GCR, &gcr);

	if(Adapter->CardType != CARD_TYPE_SPI_FLASH)
	{
#ifdef PCI
		CardWriteUshort(Adapter, GCR, (USHORT)(gcr|0x0040));
		CardReadUshort(Adapter, GCR, &gcr);
		CardWriteUshort(Adapter, GCR, (USHORT)(gcr&0xFFBF));
#else
		CardWriteUshort(Adapter, GCR, (USHORT)(gcr|0x0060));
		CardReadUshort(Adapter, GCR, &gcr);
		CardWriteUshort(Adapter, GCR, (USHORT)(gcr&0xFF9F));
#endif
	}
	else
	{
		CardWriteUshort(Adapter, GCR, (USHORT)(gcr|0x0040));
		CardReadUshort(Adapter, GCR, &gcr);
		CardWriteUshort(Adapter, GCR, (USHORT)(gcr|0x0400));
		CardReadUshort(Adapter, GCR, &gcr);	
		CardWriteUshort(Adapter, GCR, (USHORT)(gcr&0xFFBF));
	}

	return TRUE;
}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
//
//    CardGetHostInfo												       
//
//	This function selects SRAM through BSR, polls MR1/3 to detect 
//	MAC Boot Completion, gets the address of the IFACE structure from 
//	register MR2 and creates a copy of IFACE within Adapter 							  
//                                                                        
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

BOOLEAN 
CardGetHostInfo(PVNet_ADAPTER Adapter)
{
	ULONG					bytes = sizeof(IFACE);
	USHORT					MR1data, MR3data;
	ULONG					retry;

	CardWriteUshort(Adapter, BSR, BSS_SRAM);

	if(Adapter->CardType == CARD_TYPE_SPI_FLASH)
		VnetSleep(100000);

	retry = 0;
	do{
		CardReadUshort(Adapter, MR1, &MR1data);
		CardReadUshort(Adapter, MR3, &MR3data);
		retry++;
#ifdef PCI
	}while( ((!(MR3data & MAC_BOOT_COMPLETE))) && (retry < LOOP_RETRY_LIMIT) );
#else
	}while( ((!(MR1data & MAC_BOOT_COMPLETE)) && (!(MR3data & MAC_BOOT_COMPLETE))) && (retry < LOOP_RETRY_LIMIT) );
#endif

	if(retry == LOOP_RETRY_LIMIT)
	{
		IF_DEBUG_ERRORS(DbgPrint("MAC failed to BOOT (MR1 = 0x%X - MR3 = 0x%X)\n", MR1data, MR3data);)
		return FALSE;
	}

	CardReadUshort(Adapter, MR2, &Adapter->HostInfoBase);
	
	IF_LOUD(DbgPrint("IFACE base = 0x%X\n",Adapter->HostInfoBase);)

	if(Adapter->HostInfoBase == 0xffff)
	{
		IF_DEBUG_ERRORS(DbgPrint("--> Possible Card Removal <--\n");)
		Adapter->StationState = STATION_STATE_NO_CARD;

		return FALSE;
	}
	
	CardMemToHostMem16(Adapter, (PUCHAR)&Adapter->HostInfo, Adapter->HostInfoBase, bytes);

	return TRUE;
}


/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
//
//    CardStartUp													      
//
//    This function enables the completion of MAC initialization through  
//	  the FunCtrl field of the IFACE, polls MR1 to detect completion of	  
//	  MAC initialization, checks completion status, sets interrupt mask,  
//	  enables interrupts and calls Tx and Rx initialization functions     
//                                                                        
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

BOOLEAN 
CardStartUp(PVNet_ADAPTER Adapter)
{
	USHORT					MR1data, MR3data;
	ULONG					retry;
	UCHAR					tmpUC;

	tmpUC = FUNC_CTRL_INIT_COMPLETE;
	SetFuncCtrl( Adapter, &tmpUC);

	retry = 0;
	do{
		CardReadUshort(Adapter, MR1, &MR1data);
		CardReadUshort(Adapter, MR3, &MR3data);
		retry++;
#ifdef PCI
	}while( ((!(MR3data & MAC_INIT_COMPLETE))) && (retry < LOOP_RETRY_LIMIT) );
#else
	}while( ((!(MR1data & MAC_INIT_COMPLETE)) && (!(MR3data & MAC_INIT_COMPLETE))) && (retry < LOOP_RETRY_LIMIT) );
#endif

	if(retry == LOOP_RETRY_LIMIT)
	{
		IF_DEBUG_ERRORS(DbgPrint("MAC failed to complete initialization (MR1 = 0x%X - MR3 = 0x%X)\n", MR1data, MR3data);)
		return FALSE;
	}
//
//(4.0.2.29): Check for MAC_INIT_OK only on the register that the MAC_INIT_OK was set
//
	if(MR3data & MAC_INIT_COMPLETE)
	{
		CardReadUshort(Adapter, MR3, &MR3data);
		if(!(MR3data & MAC_INIT_OK))
		{
			IF_DEBUG_ERRORS(DbgPrint("MAC failed to be properly initialized (MR3 = 0x%X)...\n", MR3data);)
			return FALSE;
		}
	}
	else
	if(MR1data & MAC_INIT_COMPLETE)
	{
		CardReadUshort(Adapter, MR1, &MR1data);
		if(!(MR1data & MAC_INIT_OK))
		{
			IF_DEBUG_ERRORS(DbgPrint("MAC failed to be properly initialized (MR1 = 0x%X)...\n", MR1data);)
			return FALSE;
		}
	}

	SetInterruptMask(Adapter, Adapter->InterruptMask);
		
	TxInit(Adapter);			// Init Tx Operation and enable Tx

	RxInit(Adapter);			// Init Rx Operation and enable Rx

	EnableInterrupts(Adapter);

	return TRUE;

}


/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
//
//    CardStop															   
//																           
//    This function resets AMBA BUS and ARM core and then keeps ARM in	  
//    Reset state														  
//
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

VOID
CardStop(PVNet_ADAPTER Adapter)
{

#ifndef PCI
	//
	// Reset
	//
	CardWriteUshort(Adapter, GCR, 0x0060);
#endif //PCI

	//
	// Keep ARM in RESET state
	//
	CardWriteUshort(Adapter, GCR, 0x0040);
}


/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
//
//    CardMemToHostMem8															   
//																           
//    This function copies a number of bytes from card memory to host memory	  
//    with 8 bit access														  
//
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

VOID CardMemToHostMem8(PVNet_ADAPTER Adapter, PUCHAR pDestBuf, ULONG SramAd, ULONG Bytes)
{
	ULONG k;

	CardWriteUshort(Adapter, AR, (USHORT)SramAd);
	for(k=0; k<Bytes; k++)
		CardReadUchar(Adapter, DR, pDestBuf+k);
}


/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
//
//    HostMemToCardMem8															   
//																           
//    This function copies a number of bytes from host memory to card memory	  
//    with 8 bit access														  
//
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

VOID HostMemToCardMem8(PVNet_ADAPTER Adapter, ULONG SramAd, PUCHAR pSrcBuf, ULONG Bytes)
{
	ULONG k;

	CardWriteUshort(Adapter, AR, (USHORT)SramAd);
	for(k=0; k<Bytes; k++)
		CardWriteUchar(Adapter, DR, *(pSrcBuf+k));
}


/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
//
//    CardMemToHostMem16															   
//																           
//    This function copies a number of bytes from card memory to host memory	  
//    with 16 bit access														  
//
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

VOID CardMemToHostMem16(PVNet_ADAPTER Adapter, PUCHAR pDestBuf, ULONG SramAd, ULONG Bytes)
{
	ULONG	k;

	if(Bytes == 0)
		return;

	if(SramAd%2)
	{
		CardWriteUshort(Adapter, AR, (USHORT)SramAd);
		CardReadUchar(Adapter, DR, pDestBuf);
		Bytes -= 1;
		if(Bytes % 2){
			for(k=0; k<Bytes-1; k+=2){
				CardReadUshort(Adapter, DR, (PUSHORT)((PUCHAR)pDestBuf+1+k));
			}
			CardReadUchar(Adapter, DR, pDestBuf+1+k);
		}
		else{
			for(k=0; k<Bytes; k+=2){
				CardReadUshort(Adapter, DR, (PUSHORT)((PUCHAR)pDestBuf+1+k));
			}
		}
	}
	else
	{
		CardWriteUshort(Adapter, AR, (USHORT)SramAd);

		if(Bytes % 2)
		{
			for(k=0; k<Bytes-1; k+=2)
			{
				CardReadUshort(Adapter, DR, (PUSHORT)((PUCHAR)pDestBuf+k));
			}
			CardReadUchar(Adapter, DR, pDestBuf+k);
		}
		else
		{
			for(k=0; k<Bytes; k+=2)
			{
				CardReadUshort(Adapter, DR, (PUSHORT)((PUCHAR)pDestBuf+k));
			}
		}
	}
	

}

/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////
//
//    HostMemToCardMem16															   
//																           
//    This function copies a number of bytes from host memory to card memory	  
//    with 16 bit access														  
//
/////////////////////////////////////////////////////////////////////////////
/////////////////////////////////////////////////////////////////////////////

VOID HostMemToCardMem16(PVNet_ADAPTER Adapter, ULONG SramAd, PUCHAR pSrcBuf, ULONG Bytes)
{
	ULONG	k;

	if(Bytes == 0)
		return;

	if(SramAd%2)
	{
		CardWriteUshort(Adapter, AR, (USHORT)SramAd);
		CardWriteUchar(Adapter, DR, *pSrcBuf);
		Bytes -= 1;
		if(Bytes % 2)
		{
			for(k=0; k<Bytes-1; k+=2)
			{
				CardWriteUshort(Adapter, DR, *(PUSHORT)((PUCHAR)pSrcBuf+1+k));
			}
			CardWriteUchar(Adapter, DR, *(pSrcBuf+1+k));
		}
		else
		{
			for(k=0; k<Bytes; k+=2)
			{
				CardWriteUshort(Adapter, DR, *(PUSHORT)((PUCHAR)pSrcBuf+1+k));
			}
		}
	}
	else
	{
		CardWriteUshort(Adapter, AR, (USHORT)SramAd);

		if(Bytes % 2)
		{
			for(k=0; k<Bytes-1; k+=2)
			{
				CardWriteUshort(Adapter, DR, *(PUSHORT)((PUCHAR)pSrcBuf+k));
			}
			CardWriteUchar(Adapter, DR, *(pSrcBuf+k));
		}
		else
		{
			for(k=0; k<Bytes; k+=2)
			{
				CardWriteUshort(Adapter, DR, *(PUSHORT)((PUCHAR)pSrcBuf+k));
			}
		}
	}
}

////////////////////////////////////////////////////////////////
//
// Linux I/O functions
//
////////////////////////////////////////////////////////////////

VOID CardReadUshort(PVNet_ADAPTER Adapter, UCHAR Offset, PUSHORT pData)
{
	*(pData) = inw(Adapter->IoBase + (Offset));
}


VOID CardWriteUshort(PVNet_ADAPTER Adapter, UCHAR Offset, USHORT Data)
{
	outw(Data, (Adapter->IoBase + Offset));
	if (Offset == AR) {
		USHORT i;
		USHORT temp;
		for(i=0;i<10;i++) {
			CardReadUshort(Adapter, Offset,&temp);
			if (temp == (USHORT) Data )
				break;
			outw(Data, (Adapter->IoBase + Offset));
		}
	}
}


VOID CardWriteUchar(PVNet_ADAPTER Adapter, UCHAR Offset, UCHAR Data)
{
	outb(Data, (Adapter->IoBase + Offset));
}

VOID CardReadUchar(PVNet_ADAPTER Adapter, UCHAR Offset, PUCHAR pData)
{
	*(pData) = inb(Adapter->IoBase + Offset);
}

////////////////////////////////////////////////////////////////
//
// Functions for accessing IFACE fields
//
////////////////////////////////////////////////////////////////

VOID SetFuncCtrl(PVNet_ADAPTER Adapter, PUCHAR FuncCtrl)
{
	HostMemToCardMem8(Adapter, Adapter->HostInfoBase + IFACE_FUNC_CTRL_OFFSET, FuncCtrl, 1);
}


VOID SetLockoutMac(PVNet_ADAPTER Adapter, PUCHAR pData)
{
	HostMemToCardMem8(Adapter, Adapter->HostInfoBase + IFACE_LOCKOUT_MAC_OFFSET, (PUCHAR)pData, 1);
}


VOID SetIntStatus(PVNet_ADAPTER Adapter, PUCHAR IntStatus)
{
	HostMemToCardMem8(Adapter, Adapter->HostInfoBase + IFACE_INT_STATUS_OFFSET, (PUCHAR)IntStatus, 1);
}


VOID SetIntMask(PVNet_ADAPTER Adapter, PUCHAR IntMask)
{
	HostMemToCardMem8(Adapter, Adapter->HostInfoBase + IFACE_INT_MASK_OFFSET, (PUCHAR)IntMask, 1);
}


VOID GetLockoutHost(PVNet_ADAPTER Adapter, PUCHAR pData)
{
	CardMemToHostMem8(Adapter, (PUCHAR)pData, Adapter->HostInfoBase + IFACE_LOCKOUT_HOST_OFFSET, 1);
}


VOID GetLockoutMac(PVNet_ADAPTER Adapter, PUCHAR pData)
{
	CardMemToHostMem8(Adapter, (PUCHAR)pData, Adapter->HostInfoBase + IFACE_LOCKOUT_MAC_OFFSET, 1);
}

VOID GetIntStatus(PVNet_ADAPTER Adapter, PUCHAR IntStatus)
{
	CardMemToHostMem8(Adapter, (PUCHAR)IntStatus, Adapter->HostInfoBase + IFACE_INT_STATUS_OFFSET, 1);
}


VOID GetFuncCtrl(PVNet_ADAPTER Adapter, PUCHAR FuncCtrl)
{
	CardMemToHostMem8(Adapter, (PUCHAR)FuncCtrl, Adapter->HostInfoBase + IFACE_FUNC_CTRL_OFFSET, 1);
}

VOID GetGenericIrqType(PVNet_ADAPTER Adapter, PUSHORT GenericIntType)
{
	CardMemToHostMem8(Adapter, (PUCHAR)GenericIntType, Adapter->HostInfoBase + IFACE_GENERIC_INT_TYPE_OFFSET, 2);
}
