/****************************************************************
**
** Attal : Lords of Doom
**
** itemSelection.cpp
** Allow selection of specific item panels
**
** Version : $Id: itemSelection.cpp,v 1.17 2004/02/01 22:03:06 audoux Exp $
**
** Author(s) : Pascal Audoux
**
** Date : 22/12/2001
**
** Licence :
**	This program is free software; you can redistribute it and/or modify
**   	it under the terms of the GNU General Public License as published by
**     	the Free Software Foundation; either version 2, or (at your option)
**      any later version.
**
**	This program is distributed in the hope that it will be useful,
** 	but WITHOUT ANY WARRANTY; without even the implied warranty of
**	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**	GNU General Public License for more details.
**
****************************************************************/

#include "itemSelection.h"

// generic include files
// include files for QT
#include <qpushbutton.h>
#include <qsignalmapper.h>
// application specific include files
#include "libClient/bonus.h"
#include "libClient/building.h"
#include "libClient/cell.h"
#include "libClient/chest.h"
#include "libClient/event.h"
#include "libClient/graphicalArtefact.h"
#include "libClient/gui.h"
#include "libClient/imageTheme.h"
#include "libClient/lord.h"
#include "libClient/map.h"
#include "libClient/mapCreature.h"

#include "libCommon/dataTheme.h"

#include "libServer/gameData.h"

#include "scenarioEditor/screen.h"

extern DataTheme DataTheme;
extern ImageTheme ImageTheme;
extern Map * theMap;

//
// ----- ItemSelection -----
//

ItemSelection::ItemSelection( QWidget * parent, const char * name )
	:QWidget( parent, name )
{
	_bg = new QButtonGroup( this );
	_bg->hide();
	_bg->setExclusive( true );
	setMinimumWidth( 220 );
	_currentX = 5;
	_currentY = 5;
	_currentHeight = 5;
}

void ItemSelection::addButton( QPixmap * pixmap, int pos, bool breakLine )
{
	QPushButton * pb = new QPushButton( this );
	pb->setPixmap( * pixmap );
	pb->setToggleButton( true );
	_bg->insert( pb, pos );
	pb->setFixedSize( QSize(40,40).expandedTo(pb->sizeHint()));
	addNewWidget( pb, breakLine );
}

void ItemSelection::addButton( const QString & text, int pos, bool breakLine )
{
	QPushButton * pb = new QPushButton( text, this );
	pb->setToggleButton( true );
	_bg->insert( pb, pos );
	pb->setFixedSize( QSize(40,40).expandedTo(pb->sizeHint()));
	addNewWidget( pb, breakLine );
}

void ItemSelection::addNewWidget( QWidget * widget, bool breakLine )
{
	if( ( _currentX + widget->width() ) > 210 ) {
		breakLine = true;
	}
	if( _currentX == 5 ) {
		breakLine = false;
	}
	if( breakLine ) {
		_currentY += _currentHeight + 5;
		_currentX = 5;
		_currentHeight = 0;
	}
	_currentHeight = QMAX( _currentHeight, widget->height() );
	widget->move( _currentX, _currentY );
	_currentX += ( 5 + widget->width() );
	setMinimumHeight( _currentY + _currentHeight + 5 );
}

void ItemSelection::addNewLine()
{
	_currentX = 5;
	_currentY += _currentHeight + 5;
}

int ItemSelection::button()
{
// XXX: replace selectedId with Id (selected)
	return _bg->id ( _bg->selected() );
}


//
// ----- TransitionSelection -----
//

TransitionSelection::TransitionSelection( QWidget * parent, const char * name )
	:ItemSelection( parent, name )
{
	addButton("Auto",0);
	for( uint i = 0; i < DataTheme.tiles.count() - 1; i++ ) {
		for( uint j = 0; j < NB_TRANSITION; j++ ) {
			addButton( ImageTheme.transition[i]->image( j ), ( i * NB_TRANSITION ) + j + 1, j == 0 );
		}
	}
}

void TransitionSelection::handleLeftClick( Cell * cell )
{
	if( button() == 0 ) {
		cell->setTransition( computeTransition( (GenericMap*)theMap, (GenericCell *)cell ) );
		cell->setTransitionCellType( computeTransitionCellType( (GenericMap*)theMap, (GenericCell *)cell ) );
	} else if( button() > -1 ) {
		cell->setTransition( ( ( button() - 1 ) % NB_TRANSITION ) + 1 );
		cell->setTransitionCellType( ( ( button() - 1 ) / NB_TRANSITION ) + 1 );
	}
}


//
// ----- DecorationSelection -----
//

DecorationSelection::DecorationSelection( QWidget * parent, const char * name )
	:ItemSelection( parent, name )
{
	for( uint i = 0; i < ImageTheme.decoration->count(); i++ ) {
		addButton( ImageTheme.decoration->image( i ), i );
	}
}

void DecorationSelection::handleLeftClick( Cell * cell )
{
	if( button() != -1 ) {
		cell->setDecoration( button() + 1 );
	}
}

//
// ----- BuildingSelection -----
//

BuildingSelection::BuildingSelection( QWidget * parent, const char * name )
	:ItemSelection( parent, name )
{
	uint nb = DataTheme.buildings.count();
	for ( uint i = 0; i < nb; i++ ) {
		addButton( ImageTheme.buildings[i]->image( 0 ), i );
	}
}

void BuildingSelection::handleLeftClick( Cell * cell )
{
	if( button() != -1 ) {
		if( ( ! cell->getBuilding() )  && ( ! cell->getBase() ) ) {
			Building * building = (Building *)_data->getNewBuilding();//new Building( theMap );
			building->setType( button() );
			building->setPosition( (GenericCell *)cell );
		}
	}
}

//
// ----- BaseSelection -----
//

BaseSelection::BaseSelection( QWidget * parent, const char * name )
	:ItemSelection( parent, name )
{
	uint nb = ImageTheme.bases->count();
	for( uint i = 0; i < nb; i++ ) {
		addButton( ImageTheme.bases->image(i), i );
	}
}

void BaseSelection::handleLeftClick( Cell * cell )
{
	if( button() != -1 ) {
		if( ( ! cell->getBuilding() ) && ( ! cell->getBase() ) ) {
			Base * base = (Base *)_data->getNewBase();
			base->setRace( button() );
			base->setPosition( (GenericCell *)cell );
		}
	}
}
//
// ----- LordSelection -----
//

LordSelection::LordSelection( QWidget * parent, const char * name )
	:ItemSelection( parent, name )
{
	_num = 0;
	addButton( ImageTheme.lords->image( 0 ), 0 );
}

void LordSelection::handleLeftClick( Cell * cell )
{
	if( button() != -1 ) {
		if( ! cell->getLord() ) {
			_num++;
			Lord * lord = (Lord*)_data->getLord( _num );
			lord->setVisible( true );
			lord->setPosition( (GenericCell *)cell );
		}
	}
}

//
// ----- ArtefactSelection -----
//

ArtefactSelection::ArtefactSelection( QWidget * parent, const char * name )
	:ItemSelection( parent, name )
{
	uint nb = ImageTheme.artefacts->count();
	for ( uint i = 0; i < nb; i++ ) {
		addButton( ImageTheme.artefacts->image(i), i );
	}
}

void ArtefactSelection::handleLeftClick( Cell * cell )
{
	if( button() != -1 ) {
		Event * event = (Event*)_data->getNewArtefact();
		Artefact * artefact = (Artefact*) event->getArtefact();
		artefact->setType( button() );
		if( cell->getEvent() ) {
			Event * temp = (Event*)( cell->getEvent() );
			delete temp;
		}
		cell->setEvent( (GenericEvent*)event );
		event->setCell( (GenericCell*)cell );
	}
}

//
// ----- BonusSelection -----
//

BonusSelection::BonusSelection( QWidget * parent, const char * name )
	:ItemSelection( parent, name )
{
	int i;
	addButton( ImageTheme.chest->image( 0 ), 0, true );
	int nbRes = DataTheme.resources.count();
	for( i = 0; i < nbRes; i++ ) {
		addButton( ImageTheme.bonus[0]->image( i ), i+1, i == (nbRes -1) );
	}
	for( i = 0; i < 3; ++i ) {
		addButton( ImageTheme.bonus[1+i]->image( 0 ), nbRes+i+1 );
	}
}

void BonusSelection::handleLeftClick( Cell * cell )
{
	if( button() != -1 ) {
		Event * event;
		if( button() == 0 ) {
			event = (Event*) _data->getNewChest();
			///Chest * chest = (Chest *) event->getChest();
		} else {
			uint nbRes = DataTheme.resources.count();
			event = (Event*) _data->getNewBonus();
			Bonus * bonus = (Bonus *) event->getBonus();

			if( button() <= (int)nbRes ) {
				bonus->setType( GenericBonus::BonusResource );
				bonus->addParam( button() - 1 );
				bonus->setupBonus();
			} else {
				bonus->setType( (GenericBonus::BonusType) (button()-nbRes) );
				bonus->setupBonus();
			}
		}

		if( cell->getEvent() ) {
			Event * temp = (Event*)( cell->getEvent() );
			delete temp;
		}
		cell->setEvent( (GenericEvent*)event );
		event->setCell( (GenericCell*)cell );
	}
}

//
// ----- CreatureSelection -----
//

CreatureSelection::CreatureSelection( QWidget * parent, const char * name )
	:ItemSelection( parent, name )
{
	int cpt = 0;
	for( uint i = 0; i < DataTheme.countRace(); i++ ) {
		for( uint j = 0; j < DataTheme.creatures.getRace( i )->count(); j++ ) {
			addButton(ImageTheme.getMapCreature( i, j ), cpt );
			cpt++;
		}
		addNewLine();
	}
}

void CreatureSelection::handleLeftClick( Cell * cell )
{
	if( button() != -1 ) {
		int race = 0, level = 0;
		int cpt = 0;

		for( uint i = 0; i < DataTheme.countRace(); i++ ) {
			for( uint j = 0; j < DataTheme.creatures.getRace( i )->count(); j++ ) {
				if( cpt == button() ) {
					race = i;
					level = j;
				}
				cpt++;
			}
		}

		MapCreature * creature = (MapCreature*) _data->getNewMapCreature(); //new MapCreature( theMap );
		creature->setCreature( race, level );
		creature->setStack( 0, 1 ); /// XXX: to change
		creature->setCell( cell );
		cell->setCreature( creature );
	}
}

//
// ----- ToolSelection -----
//

ToolSelection::ToolSelection( Screen * screen, QWidget * parent, const char * name )
	:ItemSelection( parent, name )
{
	_screen = screen;
	addButton("Eraser",0);
	addButton("Auto-transition",1);
}

void ToolSelection::handleLeftClick( Cell * cell )
{
	switch( button() ) {
	case 0:
		_screen->handleDelete();
		break;
	case 1:
		cell->setTransition( computeTransition( (GenericMap*)theMap, (GenericCell *)cell ) );
		cell->setTransitionCellType( computeTransitionCellType( (GenericMap*)theMap, (GenericCell *)cell 		) );
		break;
	}
}
