/**
 * Copyright (c) 2014-2023 Governikus GmbH & Co. KG, Germany
 */

#pragma once

#include <QByteArray>
#include <QSharedPointer>
#include <openssl/ec.h>

class test_EcdhGenericMapping;

namespace governikus
{

/*!
 * Algorithm according to TR 03110 Part 3 chapter A.3.4.1.Generic Mapping
 */
class EcdhGenericMapping
{
	friend class ::test_EcdhGenericMapping;

	private:
		const QSharedPointer<EC_GROUP> mCurve;
#if OPENSSL_VERSION_NUMBER >= 0x30000000L
		QSharedPointer<EVP_PKEY> mTerminalKey;
#else
		QSharedPointer<EC_KEY> mTerminalKey;
#endif

		QSharedPointer<EC_POINT> createNewGenerator(const QSharedPointer<const EC_POINT>& pCardPubKey, const QSharedPointer<const BIGNUM>& pS);

		bool setGenerator(const QSharedPointer<const EC_POINT>& pNewGenerator);

	public:
		explicit EcdhGenericMapping(const QSharedPointer<EC_GROUP>& pCurve);

		[[nodiscard]] const QSharedPointer<EC_GROUP>& getCurve() const;

		/*!
		 * \brief Generates the terminal's mapping data, that will be sent to the card.
		 * \return the terminal's mapping data
		 */
		QByteArray generateTerminalMappingData();

		/*!
		 * \brief Generates the ephemeral domain parameters.
		 * \param pCardMappingData the card's mapping data
		 * \param pNonce the nonce that was generated by the card
		 * \return the terminal's mapping data
		 */
		bool generateEphemeralDomainParameters(const QByteArray& pCardMappingData, const QByteArray& pNonce);
};

} // namespace governikus
