/*--------------------------------------------------------------------------*
 * AUTOPROFILE                                                              *
 *                                                                          *
 * A Gaim away message and profile manager that supports dynamic text       *
 *                                                                          *
 * AutoProfile is the legal property of its developers.  Please refer to    *
 * the COPYRIGHT file distributed with this source distribution.            *
 *                                                                          *
 * This program is free software; you can redistribute it and/or modify     *
 * it under the terms of the GNU General Public License as published by     *
 * the Free Software Foundation; either version 2 of the License, or        *
 * (at your option) any later version.                                      *
 *                                                                          *
 * This program is distributed in the hope that it will be useful,          *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of           *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the            *
 * GNU General Public License for more details.                             *
 *                                                                          *
 * You should have received a copy of the GNU General Public License        *
 * along with this program; if not, write to the Free Software              *
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA *
 *--------------------------------------------------------------------------*/

#include "autoprofile.h"

#include "gtkdialogs.h"
#include "gtkimhtml.h"
#include "sound.h"

static time_t response_timeout;

/* VARIABLE DEFINITIONS */

/* List of previous away messages */
static GtkListStore *message_list;
static GtkWidget *message_list_view;

/* The general window + text of message */
static GtkWidget *dialog = NULL;
static GtkWidget *msg_window = NULL;
static GtkWidget *awaytext = NULL;

/* Progress bar */
static GtkWidget *progress_box = NULL;
static GtkWidget *progress = NULL;
static int progress_minutes = 0;
static int progress_seconds = 0;
static gpointer fraction_timeout = NULL;
static gpointer second_timeout = NULL;

/* FUNCTION DEFINITIONS */

/* Changes fraction progress of progress bar */
static gboolean update_fraction (gpointer data)
{
  double fraction_increment;
  double cur_fraction;
  double result;

  fraction_increment = 250 /
    (60 * 1000 * (double) gaim_prefs_get_int (
     "/plugins/gtk/autoprofile/delay_away"));
  cur_fraction = gtk_progress_bar_get_fraction (GTK_PROGRESS_BAR(progress));

  result = cur_fraction + fraction_increment;
  if (result >= 1)
    result = 1;

  gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR(progress), result);

  return TRUE;
}

/* Changes text of progress bar */
static gboolean update_seconds (gpointer data)
{
  char *text = (char *)malloc(1024);

  if (progress_seconds == 0) {
    progress_minutes--;
    progress_seconds = 59;
  } else {
    progress_seconds--;
  }

  if (progress_minutes < 0) {
    progress_minutes = 0;
    progress_seconds = 0;
  }
 
  sprintf (text, "next update in %d:%02d", progress_minutes, 
           progress_seconds);
  gtk_progress_bar_set_text (GTK_PROGRESS_BAR(progress), text);

  return TRUE;
}

/* Replace displayed message with a new one */
static void gtk_replace_displayed_message (char *msg)
{
  /* Update the main window w/ the text */
  if (awaytext)
    gtk_widget_destroy (awaytext);

  awaytext = gtk_imhtml_new (NULL, NULL);
  gtk_container_add (GTK_CONTAINER(msg_window), awaytext);
  gaim_setup_imhtml (awaytext);
  gtk_widget_show (awaytext);
  gtk_imhtml_append_text (GTK_IMHTML(awaytext), msg, GTK_IMHTML_NO_TITLE |
    GTK_IMHTML_NO_COMMENTS | GTK_IMHTML_NO_SCROLL);
  gtk_imhtml_append_text (GTK_IMHTML(awaytext), "<BR>",
    GTK_IMHTML_NO_TITLE | GTK_IMHTML_NO_COMMENTS |
    GTK_IMHTML_NO_SCROLL);
}

/* Changes the text displayed as the current away message */
static void gtk_away_msgs_update (char *msg)
{
  GtkTreeIter iter;
  struct tm *cur_time;
  char *time_string;
  time_t *general_time;
  GtkTreeSelection *selection;

  /* reset the progress bar */
  if (progress) {
    gtk_container_remove (GTK_CONTAINER(progress_box), progress);
    g_source_remove (GPOINTER_TO_INT (second_timeout));
    g_source_remove (GPOINTER_TO_INT (fraction_timeout));
  }
 
  /* create new progress bar */
  progress = gtk_progress_bar_new ();
  gtk_box_pack_start (GTK_BOX(progress_box), progress, TRUE, TRUE, 0);
  gtk_widget_show (progress);

  progress_seconds = 1;
  progress_minutes = gaim_prefs_get_int (
    "/plugins/gtk/autoprofile/delay_away");
  gtk_progress_bar_set_fraction (GTK_PROGRESS_BAR(progress), 0);
  update_seconds (NULL);

  fraction_timeout = GINT_TO_POINTER (g_timeout_add (
    250, update_fraction, NULL));
  second_timeout = GINT_TO_POINTER (g_timeout_add (
    1000, update_seconds, NULL));

  /* Update the list view */
  general_time = (time_t *)malloc(sizeof(time_t));
  time (general_time);
  cur_time = localtime (general_time);
  free (general_time);

  time_string = (char *)malloc (1024);
  *time_string = '\0';

  strftime (time_string, 1023, "%I:%M %p", cur_time);
  
  gtk_list_store_prepend (message_list, &iter);
  gtk_list_store_set (message_list, &iter,
                      0, time_string,
                      1, msg,
                      -1);
  selection = gtk_tree_view_get_selection (
    GTK_TREE_VIEW (message_list_view));
  gtk_tree_selection_select_iter (selection, &iter);

  /* Cap at 10 messages listed */
  if (gtk_tree_model_iter_nth_child 
        (GTK_TREE_MODEL(message_list), &iter, NULL, 10)) {
    gtk_list_store_remove (message_list, &iter);
  }

  /* Update the displayed message */
  gtk_replace_displayed_message (msg);
  response_timeout = time (NULL);
}

/* Disable everything, come back */
static void come_back (GtkWidget *w, GtkWidget *x)
{
  GtkWidget *tmp;
  
  if (dialog) {
    tmp = dialog;
    gtk_widget_destroy (tmp);

    dialog = NULL;
    msg_window = NULL;
    awaytext = NULL;
   
    if (w != tmp) {
      return;
    }
  }

  message_list = NULL;
  message_list_view = NULL;

  progress = NULL;
  progress_box = NULL;
  g_source_remove (GPOINTER_TO_INT (second_timeout));
  g_source_remove (GPOINTER_TO_INT (fraction_timeout));

  /* Status_back may call come_back, so check */
  if (is_away) {
    status_back ();
  }
}            

static gboolean display_diff_msg (GtkTreeSelection *select, gpointer data)
{
  GtkTreeModel *model;
  GtkTreeIter iter;
  gchar *string;
 
  /* Figure out which component is selected */
  if (gtk_tree_selection_get_selected (select, &model, &iter)) {
    gtk_tree_model_get (model, &iter, 1, &string, -1);
    gtk_replace_displayed_message (string);
  }

  return FALSE;
}

/* Global procedures */
void gtk_away_msgs_create (char *msg)
{
  GtkWidget *button;
  GtkWidget *hbox, *vbox;
  GtkCellRenderer *renderer;
  GtkTreeViewColumn *column;
  GtkTreeSelection *selection;
  GtkWidget *sw;
 
  /* Make sure dialog doesn't exist already */
  if (dialog) {
    gtk_away_msgs_update (msg);
    return;
  }

  /* Create main display window */
  GAIM_DIALOG(dialog);
  gtk_window_set_title (GTK_WINDOW(dialog), _("AutoProfile Away!"));
  gtk_widget_realize (dialog);

  vbox = gtk_vbox_new (FALSE, 5);
  gtk_container_add (GTK_CONTAINER (dialog), vbox);
  gtk_container_set_border_width (GTK_CONTAINER (vbox), 5);
  gtk_widget_show (vbox);

  /* Set up progress bar container */
  progress_box = gtk_hbox_new (TRUE, 0);
  gtk_box_pack_start (GTK_BOX(vbox), progress_box, FALSE, FALSE, 0);
  gtk_widget_show (progress_box);

  /* Set up box that contains info on the away messages */
  hbox = gtk_hbox_new (FALSE, 5);
  gtk_box_pack_start (GTK_BOX(vbox), hbox, TRUE, TRUE, 0);
  gtk_widget_show (hbox);

  /* Set up list of past away messages */
  message_list = gtk_list_store_new (2, G_TYPE_STRING, G_TYPE_STRING);
  message_list_view = gtk_tree_view_new_with_model (
    GTK_TREE_MODEL (message_list));
  renderer = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes (
    NULL, renderer, "text", 0, NULL);
  gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (message_list_view), FALSE);
  gtk_tree_view_append_column (GTK_TREE_VIEW (message_list_view), column);
  sw = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(sw), 
    GTK_POLICY_NEVER, GTK_POLICY_NEVER);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(sw), GTK_SHADOW_IN);
  gtk_container_add (GTK_CONTAINER (sw), message_list_view);
  gtk_box_pack_start (GTK_BOX(hbox), GTK_WIDGET(sw), FALSE, FALSE, 0);
  gtk_widget_show (message_list_view);
  gtk_widget_show (sw);

  selection = gtk_tree_view_get_selection (
    GTK_TREE_VIEW (message_list_view));
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (display_diff_msg), NULL);
  gtk_widget_set_size_request (message_list_view, 65, 205);

  /* Set up the window to display away message in */
  msg_window = gtk_scrolled_window_new (NULL, NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(msg_window), 
    GTK_POLICY_NEVER, GTK_POLICY_ALWAYS);
  gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(msg_window), 
    GTK_SHADOW_IN);
  gtk_box_pack_start (GTK_BOX(hbox), msg_window, TRUE, TRUE, 0);
  gtk_widget_show (msg_window);

  /* Display the text */
  gtk_away_msgs_update (msg);

  /* Queue messages */
  gaim_gtk_prefs_checkbox (
    _("Queue new messages while away"),
    "/plugins/gtk/autoprofile/queue_messages_when_away", 
    vbox);


  /* Mute sound */
  gaim_gtk_prefs_checkbox (
    _("Play sounds while away"),
    "/core/sound/while_away",
    vbox);

  /* I'm back button */
  button = gaim_pixbuf_button_from_stock(_("I'm Back!"), GTK_STOCK_JUMP_TO, 
      GAIM_BUTTON_HORIZONTAL);
  gtk_box_pack_start (GTK_BOX(vbox), button, FALSE, FALSE, 0);
  g_signal_connect (G_OBJECT(button), "clicked", 
    G_CALLBACK(come_back), dialog);
  gtk_window_set_focus (GTK_WINDOW(dialog), button);
  gtk_widget_show (button);

  /* Finish up */
  g_signal_connect (G_OBJECT(dialog), "destroy",
    G_CALLBACK(come_back), dialog);
  gtk_widget_show (dialog);
}

/* Wrapper function */
void gtk_away_msgs_destroy ()
{
  come_back (dialog, dialog);
}



