/*
 * autotrust.c - implements automated updates of DNSSEC Trust Anchors (RFC5011).
 * Copyright (c) 2008, NLnet Labs. All rights reserved.
 * This software is open source.
 * For license see doc/LICENSE.
 */

#include <config.h>

#include "lib/autotrust.h"

/* Our environment in which we can perform automated updates of trust anchors */
autotrust_t* env = NULL;

/* Close autotrust environment and exit program:
 * @param status: return status
 */
static void
free_environment(int status)
{
	if (status != STATUS_OK)
		fprintf(stderr, "[%s] exit status: %s\n",
					AUTOTRUST_NAME, status_by_id(status));
	if (env)
	{
		free_trustpoints(env->trustpoints);
		free_options(env->options);
		free(env);
	}
	close_logfile();
	exit(status);
}

/* Initialize autotrust environment:
 * @return: environment on success, NULL if allocation of environment failed.
 */
static autotrust_t*
initialize_environment(void)
{
	autotrust_t* new_env = (autotrust_t*) malloc(sizeof(autotrust_t));
	if (!new_env)
		return NULL;
	/* options */
	new_env->configfile = DEFAULT_CONFIGFILE;
	new_env->options = initialize_options();
	if (!new_env->options)
		return NULL;
	/* trust points */
	new_env->trustpoints = create_trustpoint_set();
	if (new_env->trustpoints == NULL)
		return NULL;
	/* nothing changed (yet) */
	new_env->changed = 0;
	/* default log to stderr, verbosity 1 */
	open_logfile(NULL, 1);
	/* done */
	return new_env;
}

/** Print usage */
static void
usage(void)
{
	const char* version = "0.2";
	const char* bugreport = "matthijs@nlnetlabs.nl";

	fprintf(stderr, "Usage: %s [OPTIONS]\n", AUTOTRUST_NAME);
	fprintf(stderr, "Automated updates of trust anchors (RFC 5011).\n\n");
	fprintf(stderr,
		"Supported options:\n"
		"  -c configfile        Read specified <configfile> instead of "
		"the default.\n"
		"  -d                   Run %s as daemon.\n", AUTOTRUST_NAME);
	fprintf(stderr,
		"  -h                   Print this help information.\n"
		"  -v verbosity-level   Set the verbosity level to "
		"<verbosity-level>.\n"
	);

	fprintf(stderr, "Version %s, Report bugs to <%s>.\n",
		version, bugreport);
}

/*
 * Main routine:
 * Set configuration settings
 * Change to the working directory
 * - working directory is current directory by default
 * - configuration file is relative to the current directory
 * - configuration settings are relative to the working directory
 * Open logfile
 * - by default, log messages are outputted to stderr
 * - log messages before the logfile is openend are outputted to stderr
 * Daemonize?
 * Load trust anchors from files
 *
 * For each zone:
 *	Query for new DNSKEYs
 *	Execute update algorithm
 * Update trust anchor files
 * Signal the resolver in case of any changes in the configured trust anchors
 *
 * @param argc: number of commandline arguments.
 * @param argv: array of commandline arguments.
 * @return: exit status of the program.
 */
int
main(int argc, char* argv[])
{
	/* read commandline arguments */
	int c, r;
	/* everything is ok until so far */
	int return_status = STATUS_OK;
	/* walk the trust point set */
	rbnode_t* node = NULL;

	/* initialize the autotrust environment */
	env = initialize_environment();
	if (!env)
		free_environment(STATUS_ERR_MALLOC);
	debug(("autotrust environment initialized"));

	/* read command line options */
#if 0
	while ((c = getopt(argc, argv, "c:dhv:")) != -1)
#else
	while ((c = getopt(argc, argv, "c:hv:")) != -1)
#endif
	{
		switch (c)
		{
			case 'c': /* config file */
				env->configfile = optarg;
				break;
#if 0
			case 'd': /* daemonize */
				r = set_cfg_option_int(env->options,
								"daemonize", 1);
				break;
#endif
			case 'v': /* verbosity */
				r = set_cfg_option_int(env->options,
						"verbosity", atoi(optarg));
				break;
			case 'h': /* help, print usage */
			case '?':
			default:
				usage();
				free_environment(STATUS_USAGE);
 				break;
		}
	}
	argc -= optind;
	argv += optind;
	if (argc != 0)
	{
		usage();
		free_environment(STATUS_ERR_GETOPT);
	}
	debug(("commandline options parsed"));

	/* read options */
	return_status = load_cfg_settings(env->options, env->configfile);
	if (return_status != STATUS_OK)
		free_environment(return_status);
	debug(("configuration settings loaded"));
#ifdef TDEBUG
	debug_print_options(env->options);
#endif /* TDEBUG */

	/* setting: working-dir: change to the configured working directory */
	if (env->options && env->options->workingdir &&
		env->options->workingdir[0])
	{
		if (chdir(env->options->workingdir) != 0)
		{
			verbos(3, "cannot chdir to %s: %s",
				env->options->workingdir, strerror(errno));
			free_environment(STATUS_ERR_CHDIR);
		}
		debug(("working directory changed to %s",
			env->options->workingdir));
	}

	/* setting: log-file: log messages to file */
	/* setting: verbosity: verbosity level */
	if (env->options)
	{
		open_logfile(env->options->logfile, env->options->verbosity);
		debug(("logfile %s opened",
			env->options->logfile?env->options->logfile:"stderr"));
	}

	/* [TODO] daemonize */

	/* setting: trust-anchor-file: load trust anchor files */
	return_status = load_trustanchor_files(env->trustpoints,
		env->options->trustanchorfiles);
	if (return_status != STATUS_OK)
		error("could not load trust anchor files: %s",
			status_by_id(return_status));
	/* setting: trusted-keys-file: load trusted keys files */
	return_status = load_trustedkeys_files(env->trustpoints,
		env->options->trustedkeysfiles);
	if (return_status != STATUS_OK)
		error("could not load trusted keys: %s",
			status_by_id(return_status));
	/* setting: trust-anchor: load additional trust anchors */
#if 0
	return_status = load_trustanchor_options(env->trustpoints,
		env->options->trustanchors);
	if (return_status != STATUS_OK)
		error("could not load trust anchor options: %s",
			status_by_id(return_status));
#endif
	/* setting: state-file: load state from this file */
	return_status = load_state(env->trustpoints, env->options->statefile);
	if (return_status != STATUS_OK)
		error("could not load state: %s", status_by_id(return_status));
	debug(("trust anchors loaded"));

#ifdef TDEBUG
	debug_print_trustpoints(env->trustpoints);
#endif /* TDEBUG */

	/* resolve and update */
	debug(("start updating"));
	if (env->trustpoints)
		node = rbt_first(env->trustpoints->sep);
	while (node != NULL)
	{
		/* libunbound context */
		struct ub_ctx* ctx;
		/* trust point */
		tp_t* tp = (tp_t*) node->key;
		if (!tp)
			continue;

		/* query for new DNSKEYs */
		verbos(2, "active refresh for zone %s", tp->name);
		ctx = ub_ctx_create();
		if (!ctx)
		{
			error("failed to create unbound context, aborting "
			      "active refresh for %s", tp->name);
			continue;
		}
		/* configuration settings */
		return_status = set_resolver_options(env->options, tp, ctx);
		if (return_status != STATUS_OK)
			warning("could not set libunbound resolver options: "
				"%s, continuing with default settings",
						status_by_id(return_status));

		/* resolve and auto update */
		return_status = query_dnskeys(ctx, tp);
		if (return_status != STATUS_OK)
			error("active refresh for zone %s failed: %s", tp->name,
						status_by_id(return_status));

		/* perform RFC 5011 statetable */
		return_status = do_statetable(tp, env->options);
		if (return_status == STATUS_CHANGED)
		{
			env->changed = 1;
			return_status = STATUS_OK;
		}
		if (return_status != STATUS_OK)
			error("failed to update state for %s: %s", tp->name,
						status_by_id(return_status));
		/* delete context */
		ub_ctx_delete(ctx);
		/* next zone */
		node = rbt_successor(node);
	}

	/* update trunst anchor files */
	debug(("update files..."));
	return_status = update_state(env->trustpoints, env->options);
	if (return_status != STATUS_OK)
		error("could not update state file: %s",
						status_by_id(return_status));

	if (return_status == STATUS_OK && env->changed)
	{
		return_status = update_trustanchor_files(env->trustpoints,
			env->options);
		if (return_status != STATUS_OK)
			error("could not update (all) trust anchor files: %s",
						status_by_id(return_status));
		else {
			int return_status2 =
				update_trustedkeys_files(env->trustpoints,
				env->options->trustedkeysfiles);
			if (return_status2 != STATUS_OK) {
				error("could not update (all) trusted keys "
					"files");
				return_status = return_status2;
			}
		}

		/* setting: resolver-pidfile: signal resolver if trust
		 * anchors were updated.
		 */
		if (return_status == STATUS_OK && env->options)
		{
			strlist_t* list = env->options->resolverpidfiles;
			while (list)
			{
				signal_resolver(list->str);
	                        list = list->next;
			}
        	}
	}

	/* done */
	debug(("done"));
	free_environment(return_status);

	/* not reached */
	return 1;
}
