/*
 * options.c - configuration settings
 * Copyright (c) 2008, NLnet Labs. All rights reserved.
 * This software is open source.
 * For license see doc/LICENSE.
 */

#include <config.h>

#include "options.h"
#include "log.h"
#include "trust_anchors.h"
#include "util.h"

/** Initialize options */
options_t*
initialize_options(void)
{
	options_t* ini = (options_t*) malloc(sizeof(options_t));
	if (!ini)
		return NULL;
	/* defaults */
	ini->statefile = (char* ) DEFAULT_STATEFILE;
	ini->workingdir = NULL;	/* current directory */
	ini->logfile = NULL; /* stderr */
	ini->roothints = NULL; /* built in root hints */
	ini->resolvconf = NULL; /* independent resolving */
	ini->unboundconf = NULL; /* no resolving configuration */
	ini->resolverpidfiles = NULL; /* run without resolver */

	ini->add_holddown = DEFAULT_ADD_HOLDDOWN;
	ini->del_holddown = DEFAULT_DEL_HOLDDOWN;
	ini->keep_missing = 0; /* forever */
	ini->alert_missing = 1; /* on by default */

	/* no default trust anchors */
	ini->trustanchors = NULL;
	ini->trustanchorfiles = NULL;
	ini->trustedkeysfiles = NULL;

	ini->verbosity = 1; /* standard verbosity */
	ini->daemonize = 0; /* one active refresh */
	ini->tp_deletion = 0; /* keep subordinate trust points */
	ini->keep_start = 0;
	ini->keep_removed = 0;
	ini->pending_count = 2;
	ini->address_family = 0; /* IP4 & IP6 */
	return ini;
}

/************************** Loading configuration settings ********************/

/** Convert the string based value to a interger */
static int
val2int(const char* val)
{
	if (my_strcmp("yes", val) == 0)
		return 1;
	else if (my_strcmp("no", val) == 0)
		return 0;
	return atoi(val);
}

/** Add a string to a list */
static strlist_t*
add_str_to_list(strlist_t* strlist, const char* val)
{
	strlist_t* list = (strlist_t*) malloc(sizeof(strlist_t));
	if (!list)
		return NULL;
	list->str = strdup(val);
	if (!list->str)
	{
		free(list);
		return NULL;
	}
	list->next = strlist;
	return list;
}

/** Set a string-based option */
int
set_cfg_option_str(options_t* options, const char* opt, const char* val)
{
	if (my_strcmp("log-file", opt) == 0)
	{
		if (options->logfile != NULL)
			free(options->logfile);
		options->logfile = strdup(val);
		if (options->logfile == NULL)
			return STATUS_ERR_MALLOC;
	}
	else if (my_strcmp("resolv-conf", opt) == 0)
	{
		if (options->resolvconf != NULL)
			free(options->resolvconf);
		options->resolvconf = strdup(val);
		if (options->resolvconf == NULL)
			return STATUS_ERR_MALLOC;
	}
	else if (my_strcmp("resolver-pidfile", opt) == 0)
	{
		options->resolverpidfiles =
			add_str_to_list(options->resolverpidfiles, val);
		if (options->resolverpidfiles == NULL)
			return STATUS_ERR_MALLOC;
	}
	else if (my_strcmp("root-hints", opt) == 0)
	{
		if (options->roothints != NULL)
			free(options->roothints);
		options->roothints = strdup(val);
		if (options->roothints == NULL)
			return STATUS_ERR_MALLOC;
	}
	else if (my_strcmp("state-file", opt) == 0)
	{
		if (my_strcmp(options->statefile, DEFAULT_STATEFILE)
			!= 0 && options->statefile != NULL)
			free(options->statefile);
		if (my_strcmp(val, DEFAULT_STATEFILE))
		{
			options->statefile = strdup(val);
			if (options->statefile == NULL)
				return STATUS_ERR_MALLOC;
		}
	}
/*
	else if (my_strcmp("trust-anchor", opt) == 0)
	{
		options->trustanchors =
				add_str_to_list(options->trustanchors, val);
		if (options->trustanchors == NULL)
			return STATUS_ERR_MALLOC;
	}
*/
	else if (my_strcmp("trust-anchor-file", opt) == 0)
	{
		options->trustanchorfiles =
				add_str_to_list(options->trustanchorfiles, val);
		if (options->trustanchorfiles == NULL)
			return STATUS_ERR_MALLOC;
	}
	else if (my_strcmp("trusted-keys-file", opt) == 0)
	{
		options->trustedkeysfiles =
				add_str_to_list(options->trustedkeysfiles, val);
		if (options->trustedkeysfiles == NULL)
			return STATUS_ERR_MALLOC;
	}
	else if (my_strcmp("unbound-conf", opt) == 0)
	{
		if (options->unboundconf != NULL)
			free(options->unboundconf);
		options->unboundconf = strdup(val);
		if (options->unboundconf == NULL)
			return STATUS_ERR_MALLOC;
	}
	else if (my_strcmp("working-dir", opt) == 0)
	{
		if (options->workingdir != NULL)
			free(options->workingdir);
		options->workingdir = strdup(val);
		if (options->workingdir == NULL)
			return STATUS_ERR_MALLOC;
	}
	else
		return STATUS_ERR_CFG_OPTION;
	return STATUS_OK;
}

/** Set an integer-based option */
int
set_cfg_option_int(options_t* options, const char* opt, int val)
{
	if (my_strcmp("add-holddown", opt) == 0)
		options->add_holddown = val;
	else if (my_strcmp("alert-missing", opt) == 0)
		options->alert_missing = val;
	else if (my_strcmp("del-holddown", opt) == 0)
		options->del_holddown = val;
	else if (my_strcmp("daemonize", opt) == 0)
		options->daemonize = val;
	else if (my_strcmp("keep-start", opt) == 0)
		options->keep_start = val;
	else if (my_strcmp("keep-missing", opt) == 0)
		options->keep_missing = val;
	else if (my_strcmp("keep-removed", opt) == 0)
		options->keep_removed = val;
	else if (my_strcmp("pending-count", opt) == 0)
		options->pending_count = val;
	else if (my_strcmp("superior-trust", opt) == 0)
		options->tp_deletion = val;
	else if (my_strcmp("verbosity", opt) == 0)
		options->verbosity = val;
	else
		return STATUS_ERR_CFG_OPTION;
	return STATUS_OK;
}

/** Set ip option */
static int
set_cfg_option_ip(options_t* options, const char* opt, const char* val)
{
	if (my_strcmp("ipv4-only", opt) == 0)
	{
		if (my_strcmp("yes", val) == 0)
			options->address_family = 1;
	}
	else if (my_strcmp("ipv6-only", opt) == 0)
	{
		if (my_strcmp("yes", val) == 0)
			options->address_family = 2;
	}
	else
		return STATUS_ERR_CFG_OPTION;
	return STATUS_OK;
}

/** include a configuration file */
static int
include_cfg(options_t* options, const char* val)
{
	return load_cfg_settings(options, val);
}

/** Set one configuration option */
static int
set_cfg_option(options_t* options, const char* opt, const char* val)
{
	if (!opt)
		return STATUS_ERR_PARSE_ERROR;

	if (my_strcmp("config", opt) == 0)
		return STATUS_OK;
	else if (my_strcmp("include", opt) == 0)
	{
		int status = include_cfg(options, val);
		if (status != STATUS_OK) {
			error("failed to include file %s");
			return STATUS_ERR_INCLUDE;
		}
	}
	else if (set_cfg_option_str(options, opt, val) != STATUS_OK)
		return set_cfg_option_int(options, opt, val2int(val));
	/* string based option was successful */
	return STATUS_OK;
}

/** Parse one single line of the config file */
static int
parse_cfg_line(char* str, char* arg, char* val, int max)
{
	int len, expect=0;
	/* remove comments and whitespace */
	remove_comments(str, '#');
	trim_string(str);
	/* line should not be empty or too long */
	len = strlen(str);
	if (len == 0)
		return STATUS_NULL;
	if (len >= max)
		return STATUS_ERR_BUFFERSIZE;

	/* read argument */
	while (*str != '\0' && *str != '\n' && *str != ':' && *str != ' ' &&
								*str != '\t')
		*arg++ = *str++;
	*arg = '\0';
	/* read whitespace */
	while (*str == ' ' || *str == '\t')
		str++;
	if (*str != ':')
		return STATUS_ERR_PARSE_ERROR;

	str++;
	/* read whitespace */
	while (*str == ' ' || *str == '\t')
		str++;
	if (*str == '\n')
	{
                *val = '\0';
                return STATUS_OK;
        }
        /* read value */
        if (*str == '\"')
	{
		expect = 1;
		str++;
	}
	while (*str != '\0' && *str != '\n' && *str != '\"' && ((*str != ' '
						&& *str != '\t') || expect))
		*val++ = *str++;
	*val = '\0';
	if (expect && *str != '\"')  /* no endquote */
		return STATUS_ERR_PARSE_ERROR;
	else if (expect && *str == '\"')
		str++;
	/* read whitespace */
	while (*str == ' ' || *str == '\t')
		str++;
	/* last character must be newline */
	if (*str == '\n')
		return STATUS_OK;
	return STATUS_ERR_PARSE_ERROR;
}

/** Load settings from file */
int
load_cfg_settings(options_t* options, const char* file)
{
	/* the file descriptor */
	FILE* fd = NULL;
	/* single line */
	char line[MAXLEN];
	/* line number, status */
	int line_nr = 0, status = STATUS_OK;

	if (file)
		fd = fopen(file, "r");
	if (!fd)
	{
		error("unable to open %s for reading: %s", file,
							strerror(errno));
		return STATUS_ERR_CONFIG;
        }

	/* fetch line by line */
	while (status == STATUS_OK && fgets(line, MAXLEN, fd) != NULL)
	{
		/* argument: value */
		char value[MAXLEN], argument[MAXLEN];

		line_nr++;
		status = parse_cfg_line(line, argument, value, MAXLEN);

		if (status == STATUS_OK) {
			if (my_strcmp("ipv4-only", argument) == 0 ||
			    my_strcmp("ipv6-only", argument) == 0)
				status = set_cfg_option_ip(options,
					argument, value);
			else
				status = set_cfg_option(options, argument,
					value);
			if (status != STATUS_OK)
				error("failed to parse configuration option "
				      "'%s' in %s at line %i: %s", argument,
					file, line_nr, status_by_id(status));
		}
		else if (status == STATUS_NULL) /* empty lines are allowed */
			status = STATUS_OK;
	}

	fclose(fd);
	return status;
}

/** Free strlist */
static void
free_strlist(strlist_t* strlist)
{
	strlist_t* next = strlist;

	if(next)
	{
		free(next->str);
		free_strlist(next->next);
		free(next);
	}
}

/** Free options */
void
free_options(options_t* options)
{
	free(options->logfile);
	free(options->resolvconf);
	free(options->roothints);
	if (my_strcmp(options->statefile, DEFAULT_STATEFILE) != 0 &&
	    options->statefile != NULL)
		free(options->statefile);
	free(options->unboundconf);
	free(options->workingdir);
	free_strlist(options->trustanchors);
	free_strlist(options->resolverpidfiles);
	free_strlist(options->trustanchorfiles);
	free_strlist(options->trustedkeysfiles);
	free(options);
}

#ifdef TDEBUG

/** File list to str */
static void
print_strlist(const char* opt, strlist_t* list)
{
	strlist_t* next = list;
	while (next)
	{
		debug(("%s: %s", opt, next->str));
		next = next->next;
	}
}

/** Print options */
void
debug_print_options(options_t* options)
{
	if (options->address_family == 1)
		debug(("address family: %s", "IPv4"));
	else if (options->address_family == 1)
		debug(("address family: %s", "IPv6"));
	else
		debug(("address family: %s", "IPv4 + IPv6"));
	debug(("logfile: %s", options->logfile?options->logfile:"stderr"));
	debug(("working dir: %s",
		options->workingdir?options->workingdir:"current directory"));
	debug(("statefile: %s", options->statefile));
	print_strlist("trust anchor file", options->trustanchorfiles);
	print_strlist("trusted keys file", options->trustedkeysfiles);
/*
	print_strlist("trust anchor", options->trustanchors);
*/
	debug(("root hints: %s",
		options->roothints?options->roothints:"built in root hints"));
	debug(("resolv.conf: %s",
		options->resolvconf?options->resolvconf:"independent "
							"resolving"));
	debug(("unbound.conf: %s",
		options->unboundconf?options->unboundconf:"none"));

	print_strlist("resolver pidfile", options->resolverpidfiles);

	debug(("add holddown timer: %i seconds", options->add_holddown));
	debug(("del holddown timer: %i seconds", options->del_holddown));
	debug(("keep START keys: %s", options->keep_start?"yes":"no"));
	debug(("keep REMOVED keys: %s", options->keep_removed?"yes":"no"));
	debug(("alert MISSING keys: %s", options->alert_missing?"yes":"no"));
	debug(("keep MISSING keys for %i seconds", options->keep_missing));
	debug(("pending count: %i", options->pending_count));
	debug(("superior trust: %s", options->tp_deletion?"yes":"no"));
	debug(("run as daemon: %s", options->daemonize?"yes":"no"));
	debug(("verbosity level: %i", options->verbosity));
}

#endif /* TDEBUG */

/************************** Loading trust anchor files ************************/

/** Parse comments */
static int
parse_comments(char* str, ta_t* ta)
{
	char* comment = (char*) malloc(sizeof(char)*MAXLEN);
	char* comments = comment;
	int len = strlen(str), pos = 0, timestamp = 0;
	if (len >= MAXLEN)
		return STATUS_ERR_BUFFERSIZE;
	while (*str != '\0' && *str != ';')
		str++;
	/* comments */
	if (*str == ';')
		str++;
	while (*str != '\0')
	{
		*comments = *str;
		comments++;
		str++;
	}
	*comments = '\0';

	comments = comment;

	if (!ta || !comments)
		return STATUS_ERR_PARSE_ERROR;
	/* read state */
	pos = position_in_string(comments, "state=");
	if (pos >= (int) strlen(comments))
	{
		free(comment);
		return STATUS_ERR_PARSE_ERROR;
	}
	if (pos <= 0)
		ta->s = STATE_START;
	else
	{
		int s = (int) comments[pos] - '0';
		/* this is only necessary when state is unknown */
		char* str = ldns_rdf2str(ldns_rr_owner(ta->rr));

		switch(s)
		{
			case STATE_START:
			case STATE_ADDPEND:
			case STATE_VALID:
			case STATE_MISSING:
			case STATE_REVOKED:
			case STATE_REMOVED:
				ta->s = s;
				break;
			default:
				warning("trust anchor [%s, DNSKEY, id=%i] has "
					"undefined state, considered NewKey",
						str, ldns_calc_keytag(ta->rr));
				ta->s = STATE_START;
				break;
		}

		free(str);
	}
	/* read pending count */
	pos = position_in_string(comments, "count=");
	if (pos >= (int) strlen(comments))
	{
		free(comment);
		return STATUS_ERR_PARSE_ERROR;
	}
	if (pos <= 0)
		ta->pending_count = 0;
	else
	{
		comments += pos;
		ta->pending_count = atoi(comments);
	}

	/* read last change */
	pos = position_in_string(comments, "lastchange=");
	if (pos >= (int) strlen(comments))
	{
		free(comment);
		return STATUS_ERR_PARSE_ERROR;
	}
	if (pos >= 0)
	{
		comments += pos;
		timestamp = atoi(comments);
	}
	if (pos < 0 || !timestamp)
	{
		ldns_rdf* owner = ldns_rr_owner(ta->rr);
		char* str = ldns_rdf2str(owner);
		warning("trust anchor [%s, DNSKEY, id=%i] has no timestamp, "
			"considered NOW", str, ldns_calc_keytag(ta->rr));
		free(str);
		ta->last_change = time(NULL);
	}
	else
		ta->last_change = timestamp;

	free(comment);
	return STATUS_OK;
}

/** Load single anchor */
static int
load_trustanchor(tp_set_t* trustpoints, char* file, char* str, int state)
{
	int status = STATUS_OK;
	ta_t* ta = NULL;
	tp_t* tp = NULL;

	if (!str_contains_data(str, ';'))
		return status; /* empty lines allowed */
	ta = add_trustanchor_frm_str(trustpoints, str, &tp, state);
	if (ta)
	{
		if (state) /* statefile, parse comments */
		{
			status = parse_comments(str, ta);
			if (rr_is_dnskey_sep(ta->rr))
			{
				if (ta->s == STATE_VALID)
					tp->valid ++;
				else if (ta->s == STATE_MISSING)
					tp->missing ++;
			}
		}
		else /* configured trust anchor, consider valid */
		{
			ta->s = STATE_VALID;
			tp->file = (const char*) file;
			debug(("set trust anchor file for %s to %s", tp->name,
				tp->file));
		}
	}
	else
		return STATUS_ERR_PARSE_ERROR;
	return status;
}

/** Load anchor file */
static int
load_trustanchor_file(tp_set_t* trustpoints, char* file, int state)
{
	/* the file descriptor */
	FILE* fd;
	/* evrything is still ok */
	int status = STATUS_OK;
	/* keep track of line numbers */
	int line_nr = 0;
	/* single line */
	char line[MAXLEN];

	if (!file)
		return STATUS_ERR_AUTO_UPDATE;

	if (!(fd = fopen(file, "r")))
	{
		verbos(2, "unable to open %s for reading: %s", file,
			strerror(errno));
		return status;
	}
	verbos(2, "reading trust anchor file %s", file);
	while (status == STATUS_OK && fgets(line, MAXLEN, fd) != NULL)
	{
		line_nr++;
		status = load_trustanchor(trustpoints, file, line, state);
		if (status != STATUS_OK) {
			error("failed to load trust anchor from %s at line %i, "
			      "skipping", file, line_nr);
			/* try to do the rest */
			status = STATUS_OK;
		}
	}

	fclose(fd);
	return status;
}

/** Loading state */
int
load_state(tp_set_t* trustpoints, char* file)
{
	return load_trustanchor_file(trustpoints, file, 1);
}

/** Read multiple anchor files */
int
load_trustanchor_files(tp_set_t* trustpoints, strlist_t* list)
{
	int status = STATUS_OK;
	while (list)
	{
		status = load_trustanchor_file(trustpoints, list->str, 0);
		if (status != STATUS_OK)
			list = NULL;
		else
			list = list->next;
	}
	return status;
}

/** Read anchor options */
int
load_trustanchor_options(tp_set_t* trustpoints, strlist_t* list)
{
	int status = STATUS_OK;
	while (list)
	{
		status = load_trustanchor(trustpoints, NULL, list->str, 0);
		if (status != STATUS_OK)
			list = NULL;
		else
			list = list->next;
	}
	return status;
}

/** Load single trusted keys file */
static int
load_trustedkey_file(tp_set_t* tp_set, char* file, int writing)
{
	/* file descriptor */
	FILE* fd;
	/* keep track of line nr, length of bind keyword */
	int line_nr = 0, rdlen = 0, at = 0;
	/* store records */
	ldns_buffer* lbuf = ldns_buffer_new(65535),
		*bbuf = ldns_buffer_new(65535);

	if (!file)
	{
		ldns_buffer_free(lbuf);
		ldns_buffer_free(bbuf);
		return STATUS_OK;
	}
	if (!(fd = fopen(file, "r")))
	{
		verbos(2, "unable to open %s for reading: %s",
							file, strerror(errno));
		ldns_buffer_free(lbuf);
		ldns_buffer_free(bbuf);
		return STATUS_OK;
	}
	if (!writing)
		verbos(2, "reading trusted keys file %s", file);

	ldns_buffer_clear(lbuf);
	if (writing)
		ldns_buffer_clear(bbuf);

	while((rdlen=bind_read_keyword(fd, lbuf, &line_nr, 1, 0)) != 0)
	{
		if(rdlen != 12 || strncmp((char*)ldns_buffer_begin(lbuf),
			"trusted-keys", 12) != 0)
		{
			if (writing)
				ldns_buffer_write_string_at(bbuf, (size_t)at,
						(char*)ldns_buffer_begin(lbuf));
			ldns_buffer_clear(lbuf);
			at += rdlen;
			continue;
		}
		if (writing)
			ldns_buffer_write_at(bbuf, (size_t)at, (const void*)
				"trusted-keys", 12);
		at += 12;
		if(!bind_skip_to_special(fd, lbuf, &line_nr, '{'))
		{
			error("parse error in trusted keys file (%s, line %d)",
								file, line_nr);
			fclose(fd);
			ldns_buffer_free(lbuf);
			ldns_buffer_free(bbuf);
			return STATUS_ERR_PARSE_ERROR;
		}

                if (writing)
			ldns_buffer_write_at(bbuf, (size_t)at, (const void*)
								" {", 2);
		at += 2;

		if (!bind_process_contents(tp_set, file, lbuf, &line_nr, fd))
		{
			error("parse error in trusted keys clause (%s, line %d)",
								file, line_nr);
			fclose(fd);
			ldns_buffer_free(lbuf);
			ldns_buffer_free(bbuf);
			return STATUS_ERR_PARSE_ERROR;
		}

		if (writing)
		{
			int status = bind_write_trusted_keys(tp_set, file, bbuf,
									&at);
			if (status != STATUS_OK)
			{
				fclose(fd);
				ldns_buffer_free(lbuf);
				ldns_buffer_free(bbuf);
				return status;
			}
		}

		if(!bind_skip_to_special(fd, lbuf, &line_nr, ';'))
		{
			error("parse error in trusted keys file (%s, line %d)",
								file, line_nr);
			fclose(fd);
			ldns_buffer_free(lbuf);
			ldns_buffer_free(bbuf);
			return STATUS_ERR_PARSE_ERROR;
		}

		if (writing)
			ldns_buffer_write_string_at(bbuf, (size_t)at, ";");
		at += 1;
		ldns_buffer_clear(lbuf);
	}

	if (writing)
	{
		ldns_buffer_write_string_at(bbuf, at, "\n");
		ldns_buffer_write_u8_at(bbuf, (at+1), (uint8_t) '\0');
		at += 2;
	}

	fclose(fd);

	if (writing)
	{
		if (!(fd = fopen(file, "w")))
		{
			error("unable to open %s for writing: %s",
							file, strerror(errno));
			ldns_buffer_free(lbuf);
			ldns_buffer_free(bbuf);
			return STATUS_ERR_AUTO_UPDATE;
		}
		verbos(2, "updating trusted keys file %s", file);

		fprintf(fd, (const char*)ldns_buffer_begin(bbuf));
		fclose(fd);
	}
	ldns_buffer_free(lbuf);
	ldns_buffer_free(bbuf);
	return STATUS_OK;
}

/** Wrapper */
int
load_trustedkeys_wrapper(tp_set_t* trustpoints, strlist_t* list, int writing)
{
	int status = STATUS_OK;
	while (list)
	{
		status = load_trustedkey_file(trustpoints, list->str,
								writing);
		if (status != STATUS_OK)
			list = NULL;
		else
			list = list->next;
	}
	return status;
}

/** Read multiple trusted keys files */
int
load_trustedkeys_files(tp_set_t* trustpoints, strlist_t* list)
{
	return load_trustedkeys_wrapper(trustpoints, list, 0);
}

/************************** Writing trust anchor files ************************/

/** Update a single trust anchor to file */
static void
update_trustanchor(ta_t* ta, FILE* fd, char* str, const char* tp_name,
	int state, int ksk, options_t* options)
{
	if (state)
	{
		if (rr_is_dnskey_sep(ta->rr))
		{
			/* ldns_rr2str places a newline at the end of the
			 * string, but we need to add additional information.
			 */
			int nl_pos = strlen(str)-1;
			str[nl_pos] = '\0';

			/* you only need to keep track of pending count in the
			 * ADDPEND state.
			 */
			if (ta->s != STATE_ADDPEND)
				ta->pending_count = 0;

			/* fprintf */
			fprintf(fd, "%s ;;state=%i ;;count=%i ;;lastchange=%u "
				";;%s", str, ta->s, ta->pending_count,
				 (unsigned int) ta->last_change,
				ctime(&(ta->last_change)));
			/* end fprintf */
		}
	}
	else if (ta->s == STATE_VALID || ta->s == STATE_MISSING ||
		 rr_is_ds(ta->rr) || (rr_is_zsk(ta->rr) && !ksk))
	{
		/* setting: alert-missing: alert of MISSING DNSKEYs. */
		if (ta->s == STATE_MISSING && options->alert_missing)
		{
			warning("trust anchor [%s, DNSKEY, id=%i] is missing",
				tp_name, ldns_calc_keytag(ta->rr));
		}

		fprintf(fd, "%s", str);
	}
}


/** Update a single trust point to file */
static void
update_trustanchor_node(tp_t* tp, FILE* fd, char* file, int state,
	options_t* options)
{
	size_t i;
	uint8_t ksk = (tp->valid + tp->missing);

	if (!state && my_strcmp((const char*) file, tp->file))
		return;

	fprintf(stderr, "updating node %s\n", tp->name);
	for (i=0; i < tp->count; i++)
	{
		char* str = ldns_rr2str(tp->anchorlist[i]->rr);

		fprintf(stderr, "updating ta %s\n", str);
		/* setting: keep-start: keep START DNSKEYs */
		if (tp->anchorlist[i]->s == STATE_START &&
						!options->keep_start)
		{
			free(str);
			continue;
		}

		/* setting: keep-removed: keep REMOVED DNSKEYs */
		else if (tp->anchorlist[i]->s == STATE_REMOVED &&
						!options->keep_removed)
		{
			free(str);
			continue;
		}

		update_trustanchor(tp->anchorlist[i], fd, str, tp->name, state,
			ksk, options);

		free(str);
	}
}

/** Update a trust anchor file */
static int
update_trustanchor_file(tp_set_t* tp_set, char* file, options_t* options,
	int state)
{
	rbnode_t* node = NULL;
	FILE* fd;

	/* open file for reading */
	if (!file || !(fd = fopen(file, "w")))
	{
		error("unable to open %s for writing: %s",
							file, strerror(errno));
		return STATUS_ERR_AUTO_UPDATE;
	}
	verbos(2, "updating trust anchor file %s", file);

	/* walk through the trust point set */
	if (tp_set)
		node = rbt_first(tp_set->sep);
	while (node)
	{
		update_trustanchor_node((tp_t*) node->key, fd, file, state,
			options);
		node = rbt_successor(node);
	}
	fclose(fd);
	return STATUS_OK;
}

/** Update state */
int
update_state(tp_set_t* trustpoints, options_t* options)
{
	return update_trustanchor_file(trustpoints, options->statefile,
								options, 1);
}

/** Read multiple anchor files */
int
update_trustanchor_files(tp_set_t* trustpoints, options_t* options)
{
	int status = STATUS_OK;
	strlist_t* list = options->trustanchorfiles;
	while (list)
	{
		status = update_trustanchor_file(trustpoints, list->str,
								options, 0);
		if (status != STATUS_OK)
			list = NULL;
		else
			list = list->next;
	}
	return status;
}

/** Write multiple trusted keys files */
int
update_trustedkeys_files(tp_set_t* trustpoints, strlist_t* list)
{
	return load_trustedkeys_wrapper(trustpoints, list, 1);
}
