/*
 * util.h - utility function library
 * Copyright (c) 2008, NLnet Labs. All rights reserved.
 * This software is open source.
 * For license see doc/LICENSE.
 */

#ifndef UTIL_H
#define UTIL_H

#include <config.h>

#include "trust_anchors.h"

/* Remove leading and trailing whitespace
 * @param str: the string to remove from
 */
void trim_string(char* str);

/* Remove comments
 * @param str: the string to remove from
 * @param c: the comment character
 */
void remove_comments(char* str, char c);

/* Calculate position of substring in string
 * @param str: the string
 * @param sub: the substring
 * @return: the position
 */
int position_in_string(char* str, const char* sub);

/* Compare two strings on lenght and contents
 * @param a: a string
 * @param b: another string
 * @return: 0 if equal, 1 otherwise
 */
int my_strcmp(const char* a, const char* b);

/* Read pidfile
 * @param pidfile: file that stores the process id
 * @param len: store length
 * @return: the process id
 */
pid_t readpid(const char* pidfile, size_t* len);

/* Bind stuff */

/**
 * Read next keyword from a bind configuration file:
 * @param fd: the bind config file.
 * @param buf: buffer to store result in.
 * @param line: keep track of line number.
 * @param comments: parse comments or skip them.
 * @param newlines: keep newlines.
 * @return: 1 on success, 0 on error.
 */
int bind_read_keyword(FILE* fd, ldns_buffer* buf, int* line, int comments,
	int newlines);

/**
 * Skip to a special character in the file:
 * @param fd: the file.
 * @param buf: buffer to store result in.
 * @param line: keep track of line number.
 * @param c: the character to skip to.
 * @return: 1 on success, 0 on error.
 */
int bind_skip_to_special(FILE* fd, ldns_buffer* buf, int* line, char c);

/**
 * Read the contents of the trusted-keys clause and store them as trust anchors:
 * @param anchors: storage for the trusted keys.
 * @param file: file that stores the contents.
 * @param buf: buffer to store result in.
 * @param line: keep track of line number.
 * @param fd: the file.
 * @return: 1 on success, 0 on error.
 */
int bind_process_contents(tp_set_t* anchors, char* file, ldns_buffer* buf,
	int* line, FILE* fd);

/**
 * Write the updated contents of the trusted-keys clause:
 * @param anchors: storage for the trusted keys.
 * @param file: file to store in.
 * @param buf: buffer to store result in.
 * @param at: character count.
 * @return: 1 on success, 0 on error.
 */
int bind_write_trusted_keys(tp_set_t* anchors, char* file, ldns_buffer* buf,
	int* at);

#endif /* UTIL_H */
