#include <stdlib.h>
#include <string.h>
#include <time.h>
#include <sys/stat.h>
#include <gtk/gtk.h>
#ifdef HAVE_EDV2
# include <endeavour2.h>
#endif

#include "../include/disk.h"

#include "guiutils.h"
#include "cdialog.h"
#include "pdialog.h"
#include "fb.h"
#include "avscanop.h"
#include "obj.h"
#include "win.h"    
#include "wincb.h"
#include "winlist.h"
#include "winresultsfio.h"
#include "windb.h"
#include "winopcb.h"
#include "core.h"
#include "help.h"
#include "cfglist.h"
#include "config.h"

#include "images/icon_trash_32x32.xpm"
#include "images/icon_avscan_32x32.xpm"


static const gchar *GET_NAME_FROM_PATH(const gchar *path);
static gboolean WinQueryOverwrite(
	core_struct *core, const gchar *path, GtkWidget *toplevel
);
static void WinCListSelectAll(GtkCList *clist);
static void WinCListUnselectAll(GtkCList *clist);
static void WinCListInvertSelection(GtkCList *clist);

void WinCloseCB(GtkWidget *widget, gpointer data);
void WinExitCB(GtkWidget *widget, gpointer data);

void WinAddCB(GtkWidget *widget, gpointer data);
static gchar *WinPDialogBrowseLocationCB(
	gpointer p, gpointer data, gint prompt_num
);
void WinEditCB(GtkWidget *widget, gpointer data);
void WinRemoveCB(GtkWidget *widget, gpointer data);
void WinCutCB(GtkWidget *widget, gpointer data);
void WinCopyCB(GtkWidget *widget, gpointer data);
void WinPasteCB(GtkWidget *widget, gpointer data);
void WinSelectAllCB(GtkWidget *widget, gpointer data);
void WinUnselectAllCB(GtkWidget *widget, gpointer data);
void WinInvertSelectionCB(GtkWidget *widget, gpointer data);

void WinStartCB(GtkWidget *widget, gpointer data);
void WinStopCB(GtkWidget *widget, gpointer data);
void WinLocationCB(GtkWidget *widget, gpointer data);
void WinRecursiveCB(GtkWidget *widget, gpointer data);
void WinExecutablesOnlyCB(GtkWidget *widget, gpointer data);
void WinIgnoreLinksCB(GtkWidget *widget, gpointer data);

static gint WinEDVRecycleProgressCB(   
	gpointer data, gulong pos, gulong total
);
void WinResultsRecycleCB(GtkWidget *widget, gpointer data);
void WinResultsMoveCB(GtkWidget *widget, gpointer data);
void WinResultsSelectAllCB(GtkWidget *widget, gpointer data);
void WinResultsUnselectAllCB(GtkWidget *widget, gpointer data);
void WinResultsInvertSelectionCB(GtkWidget *widget, gpointer data);
void WinResultsOpenCB(GtkWidget *widget, gpointer data);
void WinResultsSaveAsCB(GtkWidget *widget, gpointer data);
void WinResultsClearCB(GtkWidget *widget, gpointer data);

void WinDBRefreshCB(GtkWidget *widget, gpointer data);
void WinDBLocationCB(GtkWidget *widget, gpointer data);
void WinDBUpdateNetCB(GtkWidget *widget, gpointer data);

void WinRefreshCB(GtkWidget *widget, gpointer data);
void WinViewScanPageCB(GtkWidget *widget, gpointer data);
void WinViewResultsPageCB(GtkWidget *widget, gpointer data);
void WinViewDBPageCB(GtkWidget *widget, gpointer data);
void WinViewToolBarCB(GtkWidget *widget, gpointer data);
void WinViewStatusBarCB(GtkWidget *widget, gpointer data);

void WinHelpContentsCB(GtkWidget *widget, gpointer data);
void WinHelpScanningCB(GtkWidget *widget, gpointer data);
void WinHelpUpdatingCB(GtkWidget *widget, gpointer data);
void WinHelpAboutCB(GtkWidget *widget, gpointer data);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Returns the name portion of the path (if any).
 */
static const gchar *GET_NAME_FROM_PATH(const gchar *path)
{
	const gchar *s;

	if(STRISEMPTY(path))
	    return(path);

	s = strrchr(path, DIR_DELIMINATOR);
	return((s != NULL) ? (s + 1) : path);
}


/*
 *	Checks if path exists and if it does then it queries
 *	the user for overwrite.
 *
 *	Returns TRUE to indicate continue or FALSE to indicate do not
 * 	continue.
 */
static gboolean WinQueryOverwrite(
	core_struct *core, const gchar *path, GtkWidget *toplevel  
)
{
	gint response;
	gchar *buf;
	struct stat stat_buf;

	/* Path not specified or object does not exist? */
	if(!STRISEMPTY(path) ? stat(path, &stat_buf) : TRUE)
	    return(TRUE);

	/* Confirm overwrite */
	buf = g_strdup_printf(
	    "Overwrite existing file:\n\n    %s\n",
	    path
	);
#ifdef HAVE_EDV2
	EDVPlaySoundWarning(core->edv_ctx);
#endif
	CDialogSetTransientFor(toplevel);
	response = CDialogGetResponse(
	    "Confirm Overwrite",
	    buf,
	    NULL,
	    CDIALOG_ICON_WARNING,
	    CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
	    CDIALOG_BTNFLAG_NO
	);
	CDialogSetTransientFor(NULL);
	g_free(buf);
	switch(response)
	{
	  case CDIALOG_RESPONSE_YES:
	  case CDIALOG_RESPONSE_YES_TO_ALL:
	  case CDIALOG_RESPONSE_OK:
	    return(TRUE);
	    break;
	  default:
	    return(FALSE);
	    break;
	}
}

/*
 *	Selects all rows on the specified GtkCList.
 */
static void WinCListSelectAll(GtkCList *clist)
{
	if(clist == NULL)
	    return;

	gtk_clist_freeze(clist);
	gtk_clist_select_all(clist);
	gtk_clist_thaw(clist);
}

/*
 *	Unselects all rows on the specified GtkCList.
 */
static void WinCListUnselectAll(GtkCList *clist)
{
	if(clist == NULL)
	    return;

	gtk_clist_freeze(clist);
	gtk_clist_unselect_all(clist);
	gtk_clist_thaw(clist);
}

/*
 *	Inverts selection on the specified GtkCList.
 */
static void WinCListInvertSelection(GtkCList *clist)
{
	gint i;
	GList *glist;

	if(clist == NULL)
	    return;

	glist = (clist->selection != NULL) ?
	    g_list_copy(clist->selection) : NULL;

	gtk_clist_freeze(clist);
	for(i = 0; i < clist->rows; i++)
	{
	    if(g_list_find(glist, (gpointer)i) != NULL)
		gtk_clist_unselect_row(clist, i, 0);
	    else
		gtk_clist_select_row(clist, i, 0);
	}
	gtk_clist_thaw(clist);

	g_list_free(glist);
}


/*
 *	Close.
 */
void WinCloseCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);

	WinSavePositions(win);
	WinUnmap(win);

	WinSetBusy(win, FALSE);
}

/*
 *	Exit.
 */
void WinExitCB(GtkWidget *widget, gpointer data)
{
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	core = win->core;
	if(core == NULL)
	    return;

	WinSetBusy(win, TRUE);

	/* Mark the core's close_all_windows to close all windows during
	 * the core's main management check
	 */
	core->close_all_windows = TRUE;

	WinSavePositions(win);
	WinUpdate(win);

	WinSetBusy(win, FALSE);
}


/*
 *	Add.
 *
 *	Adds a new scan item.
 */
void WinAddCB(GtkWidget *widget, gpointer data)
{
	gint row;
	GList *glist;
	GtkCList *clist;
	obj_struct *obj;
	win_struct *win = WIN(data);
	if((win == NULL) || PDialogIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	clist = GTK_CLIST(win->scan_clist);

	/* Get last selected row (if any) */
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;

	WinSetBusy(win, TRUE);

	/* Create a new object */
	obj = ObjNew();
	if(obj != NULL)
	{
	    gint new_row;

	    /* Set default object values */
	    obj->name = STRDUP("New Scan");
	    obj->path = STRDUP(getenv("HOME"));
	    obj->options = 0;
	    obj->last_runned = 0l;

	    /* Insert or append a new row on the scan list and
	     * transfer the new object to it
	     */
	    if(row > -1)
		new_row = WinScanListInsert(win, row, obj);
	    else
		new_row = WinScanListAppend(win, obj);

	    if(new_row > -1)
	    {
		/* Select the new row */
		gtk_clist_freeze(clist);
		gtk_clist_unselect_all(clist);
		gtk_clist_select_row(clist, new_row, 0);
		gtk_clist_thaw(clist);

		/* Edit the new object */
		WinEditCB(NULL, win);
	    }
	}

	WinSetBusy(win, FALSE);
}


/*
 *	Prompt Dialog browse location callback.
 */
static gchar *WinPDialogBrowseLocationCB(
	gpointer p, gpointer data, gint prompt_num
)
{
	gboolean status;
	GtkWidget *toplevel;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint total_ftypes = 0;
	gchar **path_rtn = NULL;
	gint total_path_rtns = 0;
	static gchar path[PATH_MAX];
	win_struct *win = WIN(data);
	if((win == NULL) || FileBrowserIsQuery())
	    return(NULL);

	*path = '\0';

	toplevel = PDialogGetToplevel();

	/* Create file types list */
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    "*.*", "All files"
	);                    

	/* Query user for path */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(
	    "Select Location",
	    "Select", "Cancel",
	    PDialogGetPromptValue(prompt_num),
	    ftype, total_ftypes,
	    &path_rtn, &total_path_rtns,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(status)
	{
	    const gchar *s = (total_path_rtns > 0) ?
		path_rtn[0] : NULL;
	    if(!STRISEMPTY(s))
	    {
		strncpy(path, s, sizeof(path));
		path[sizeof(path) - 1] = '\0';
	    }
	}

	/* Delete file types list */
	FileBrowserDeleteTypeList(ftype, total_ftypes);

	return(STRISEMPTY(path) ? NULL : path);
}

/*
 *	Edit.
 *
 *	Edits the selected scan item.
 */
void WinEditCB(GtkWidget *widget, gpointer data)
{
	gchar **strv;
	gint strc, row;
	GList *glist;
	GtkWidget *toplevel;
	GtkCList *clist;
	obj_struct *obj;
	core_struct *core;
	win_struct *win = WIN(data);
	if((win == NULL) || PDialogIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	toplevel = win->toplevel;
	clist = GTK_CLIST(win->scan_clist);
	core = CORE(win->core);

	/* Get last selected row */
	glist = clist->selection_end;
	row = (glist != NULL) ? (gint)glist->data : -1;
	if((row < 0) || (row >= clist->rows))
	    return;

	/* Get object */
	obj = OBJ(gtk_clist_get_row_data(clist, row));
	if(obj == NULL)
	    return;

	WinSetBusy(win, TRUE);

	/* Set up Prompt Dialog */
	PDialogDeleteAllPrompts();
	PDialogSetSize(350, -1);
	PDialogAddPrompt(NULL, "Name:", obj->name);
	PDialogAddPromptWithBrowse(
	    NULL, "Location:", obj->path,
	    win, WinPDialogBrowseLocationCB
	);
	PDialogSetPromptCompletePath(-1);
	PDialogAddPromptToggle(
	    NULL, "Recursive", obj->options & AVSCAN_OPT_RECURSIVE
	);
	PDialogAddPromptToggle(
	    NULL, "Executables Only", obj->options & AVSCAN_OPT_EXECUTABLES_ONLY
	);
	PDialogAddPromptToggle(
	    NULL, "Ignore Links", obj->options & AVSCAN_OPT_IGNORE_LINKS
	);
	if(TRUE)
	{
#ifdef HAVE_EDV2
	    const gulong t = obj->last_runned;
	    const gchar *s = (t > 0l) ?
		EDVDateString(core->edv_ctx, t) : "Unknown";
#else
	    time_t t = (time_t)((obj != NULL) ? obj->last_runned : 0l);
	    const gchar *s = (t > 0l) ? ctime(&t) : "Unknown";
#endif
	    gchar *buf = g_strdup_printf(
		"Last Scanned: %s", s
	    );
	    PDialogAddPromptLabel(buf);
	    g_free(buf);
	}
	/* Query user */
	gtk_window_set_transient_for(
	    GTK_WINDOW(PDialogGetToplevel()), GTK_WINDOW(toplevel)
	);
/*	PDialogSetTransientFor(toplevel); */
	strv = PDialogGetResponse(
	    "Scan Item Properties",
	    NULL, NULL,
	    PDIALOG_ICON_FILE_PROPERTIES,
	    "Set", "Cancel",
	    PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
	    PDIALOG_BTNFLAG_SUBMIT,
	    &strc
	);
	gtk_window_set_transient_for(
	    GTK_WINDOW(PDialogGetToplevel()), NULL
	);
/*	PDialogSetTransientFor(NULL); */
	if((strv != NULL) && (strc > 4))
	{
	    g_free(obj->name);
	    obj->name = STRDUP(strv[0]);

	    g_free(obj->path);
	    obj->path = STRDUP(strv[1]);

	    if(ATOI(strv[2]))
		obj->options |= AVSCAN_OPT_RECURSIVE;
	    else
		obj->options &= ~AVSCAN_OPT_RECURSIVE;

	    if(ATOI(strv[3]))
		obj->options |= AVSCAN_OPT_EXECUTABLES_ONLY;
	    else
		obj->options &= ~AVSCAN_OPT_EXECUTABLES_ONLY;

	    if(ATOI(strv[4]))
		obj->options |= AVSCAN_OPT_IGNORE_LINKS;
	    else
		obj->options &= ~AVSCAN_OPT_IGNORE_LINKS;

	    WinScanListUpdateRow(win, row, obj);
	    WinScanSettingsUpdate(win, obj);
	    WinUpdate(win);
	}

	PDialogDeleteAllPrompts();

	WinSetBusy(win, FALSE);
}

/*
 *	Remove.
 *
 *	Removes the selected scan item(s).
 */
void WinRemoveCB(GtkWidget *widget, gpointer data)
{
	gint row;
	GList *glist, *obj_list;
	GtkWidget *toplevel;
	GtkCList *clist;
	obj_struct *obj;
	core_struct *core;
	win_struct *win = WIN(data);
	if((win == NULL) || CDialogIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;
	clist = GTK_CLIST(win->scan_clist);
	core = CORE(win->core);

	/* Iterate through each selected row and generate objects list */
	obj_list = NULL;                                      
	for(glist = clist->selection; glist != NULL; glist = g_list_next(glist))
	{
	    row = (gint)glist->data;
	    obj = gtk_clist_get_row_data(clist, row);
	    if(obj == NULL)
		continue;

	    obj_list = g_list_append(obj_list, obj);
	}     

	/* Confirm delete */
	if(obj_list != NULL)
	{
	    gint response;
	    const gint nobjs = g_list_length(obj_list);
	    obj_struct *obj = (nobjs == 1) ? OBJ(obj_list->data) : NULL;
	    gchar *buf;

	    /* Format confirm delete message */
	    if((obj != NULL) ? !STRISEMPTY(obj->name) : FALSE)
		buf = g_strdup_printf(
		    "Delete scan item \"%s\"?",
		    obj->name
		);
	    else
		buf = g_strdup_printf(
		    "Delete %i selected scan items?",
		    nobjs
		);

	    /* Query user to confirm delete */
#ifdef HAVE_EDV2
	    EDVPlaySoundQuestion(core->edv_ctx);
#endif
	    CDialogSetTransientFor(toplevel);
	    response = CDialogGetResponse(
		"Confirm Delete",
		buf,
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
		CDIALOG_BTNFLAG_NO
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);
	    if(response != CDIALOG_RESPONSE_YES)
	    {
		g_list_free(obj_list);
		WinSetBusy(win, FALSE);
		return;
	    }
	}

	/* Delete the rows who's data is specified in the obj_list */
	gtk_clist_freeze(clist);
	for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
	{
	    row = gtk_clist_find_row_from_data(clist, glist->data);
	    if(row > -1)
		gtk_clist_remove(clist, row);
	}
	gtk_clist_thaw(clist);

	/* Delete objects list */
	g_list_free(obj_list);

	WinUpdate(win);

	WinSetBusy(win, FALSE);
}

/*
 *	Cut.
 *
 *	Cuts the selected scan item(s) to the clipboard.
 */
void WinCutCB(GtkWidget *widget, gpointer data)
{
	guint8 *buf = NULL;
	gint row, buf_len = 0;
	GList *glist, *obj_list;
	GtkCList *clist;
	obj_struct *obj;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);

	clist = GTK_CLIST(win->scan_clist);

	/* Begin copying */
					   
	/* Iterate through each selected row and generate the buffer
	 * and objects list
	 */
	obj_list = NULL;
	for(glist = clist->selection; glist != NULL; glist = g_list_next(glist))
	{
	    row = (gint)glist->data;
	    obj = OBJ(gtk_clist_get_row_data(clist, row));
	    if(obj == NULL)
		continue;

	    obj_list = g_list_append(obj_list, obj);
	    buf = ObjDDEBufferAppend(buf, &buf_len, obj);
	}

	/* Got buffer? */
	if(buf != NULL)
	{
	    /* Set buffer to the clipboard */
	    GUIDDESetBinary(
		GTK_WIDGET(clist),
		GDK_SELECTION_PRIMARY,
		GDK_CURRENT_TIME,
		buf, buf_len
	    );

	    /* Delete buffer, it is no longer needed */
	    g_free(buf);
	}


	/* Begin deleting */

	/* Delete the rows who's data is specified in the obj_list */
	gtk_clist_freeze(clist);
	for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
	{
	    row = gtk_clist_find_row_from_data(clist, glist->data);
	    if(row > -1)
		gtk_clist_remove(clist, row);
	}
	gtk_clist_thaw(clist);

	/* Delete objects list */
	g_list_free(obj_list);

	WinUpdate(win);
	 
	WinSetBusy(win, FALSE);
}

/*
 *	Copy.
 *
 *	Coppies the selected scan item(s) to the clipboard.
 */
void WinCopyCB(GtkWidget *widget, gpointer data)
{
	guint8 *buf = NULL;
	gint row, buf_len = 0;
	GList *glist;
	GtkCList *clist;
	const obj_struct *obj;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);

	clist = GTK_CLIST(win->scan_clist);

	/* Iterate through each selected row and generate the buffer */
	for(glist = clist->selection; glist != NULL; glist = g_list_next(glist))
	{
	    row = (gint)glist->data;
	    obj = OBJ(gtk_clist_get_row_data(clist, row));
	    if(obj == NULL)
		continue;

	    buf = ObjDDEBufferAppend(buf, &buf_len, obj);
	}

	/* Got buffer? */
	if(buf != NULL)
	{
	    /* Set buffer to the clipboard */
	    GUIDDESetBinary(
		GTK_WIDGET(clist),
		GDK_SELECTION_PRIMARY,
		GDK_CURRENT_TIME,
		buf, buf_len
	    );

	    /* Delete buffer, it is no longer needed */
	    g_free(buf);
	}

	WinSetBusy(win, FALSE);
}

/*
 *	Paste.
 *
 *	Paste the scan items from the clipboard.
 */
void WinPasteCB(GtkWidget *widget, gpointer data)
{
	guint8 *buf;
	gint buf_len;
	GtkCList *clist;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);

	clist = GTK_CLIST(win->scan_clist);

	/* Get buffer from clipboard */
	buf = GUIDDEGetBinary(
	    GTK_WIDGET(clist),
	    GDK_SELECTION_PRIMARY,
	    GDK_CURRENT_TIME,
	    &buf_len
	);
	if(buf != NULL)
	{
	    /* Parse buffer into a list of objects */
	    gint row;
	    GList	*glist,
			*obj_list = ObjDDEBufferParse(buf, buf_len);
	    const obj_struct *obj;

	    /* Delete buffer, it is no longer needed */
	    g_free(buf);

	    /* Get last selected row */
	    glist = clist->selection_end;
	    row = (glist != NULL) ? (gint)glist->data : -1;

	    gtk_clist_freeze(clist);

	    /* Insert or append? */
	    if(row > -1)
	    {
		gint ins_row = row;

		for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
 		{
		    obj = OBJ(glist->data);
		    ins_row = WinScanListInsert(
			win, ins_row, ObjCopy(obj)
		    );
		    if(ins_row < 0)
			break;
		    ins_row++;
		}
 	    }
	    else
	    {
		for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
		{
		    obj = OBJ(glist->data);
		    if(WinScanListAppend(win, ObjCopy(obj)) < 0)
			break;
		}
	    }

	    gtk_clist_thaw(clist);

	    /* Delete object list */
	    g_list_foreach(obj_list, (GFunc)ObjDelete, NULL);
	    g_list_free(obj_list);
	}

	WinSetBusy(win, FALSE);
}

/*
 *	Scan List Select All.
 */
void WinSelectAllCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinCListSelectAll(GTK_CLIST(win->scan_clist));
}

/*
 *	Scan List Unselect All.
 */
void WinUnselectAllCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinCListUnselectAll(GTK_CLIST(win->scan_clist));
}

/*
 *	Scan List Invert Selection.
 */
void WinInvertSelectionCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinCListInvertSelection(GTK_CLIST(win->scan_clist));
}


/*
 *	Start.
 *
 *	Start scanning.
 */
void WinStartCB(GtkWidget *widget, gpointer data)
{
	gint i;
	GtkWidget *toplevel;
	GtkCList *clist;
	gchar *location;
	avscan_options options;
	obj_struct *obj;
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;
		   
	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;
	clist = GTK_CLIST(win->scan_clist);
	core = CORE(win->core);

	/* Get scan values */
	location = STRDUP(
	    gtk_entry_get_text(GTK_ENTRY(win->location_entry))
	);
	options = 0;
	if(GTK_TOGGLE_BUTTON_GET_ACTIVE(win->recursive_check))
	    options |= AVSCAN_OPT_RECURSIVE;
	if(GTK_TOGGLE_BUTTON_GET_ACTIVE(win->executables_only_check))
	    options |= AVSCAN_OPT_EXECUTABLES_ONLY;
	if(GTK_TOGGLE_BUTTON_GET_ACTIVE(win->ignore_links_check))
	    options |= AVSCAN_OPT_IGNORE_LINKS;

	/* Scan location not specified? */
	if(STRISEMPTY(location))
	{
#ifdef HAVE_EDV2  
	    EDVPlaySoundWarning(core->edv_ctx);
#endif
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Scan Failed",
"The scan location has not been specified",
		NULL,
		CDIALOG_ICON_WARNING,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    ); 
	    CDialogSetTransientFor(NULL);

	    WinSetBusy(win, FALSE);
	    free(location);
	    return;
	}


	/* Get selected row and see if the scan item object's location
	 * matches the specified location
	 */
	for(i = 0; i < clist->rows; i++)
	{
	    obj = OBJ(gtk_clist_get_row_data(clist, i));
	    if((obj != NULL) ? STRISEMPTY(obj->path) : TRUE)
		continue;

	    /* Object's location matches the specified location? */
	    if(!strcmp(obj->path, location))
	    {
		/* Update this object's last runned time */
		obj->last_runned = (gulong)time(NULL);
		WinScanListUpdateRow(win, i, obj);
	    }
	}

	/* Start scan */
	WinScanProcessStart(
	    win,
	    NULL,		/* Database Path */
	    location,		/* Path to object to scan */
	    options
	);

	/* Switch to results page */
	gtk_notebook_set_page(
	    GTK_NOTEBOOK(win->notebook), (gint)WIN_PAGE_NUM_RESULTS
	);

	g_free(location);

	WinSetBusy(win, FALSE);
}

/*
 *	Stop.
 *
 *	Stop scanning by incrementing the Win's stop_count.
 */
void WinStopCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->stop_count++;
}

/*
 *	Location.
 *
 *	Sets the scan location.
 */
void WinLocationCB(GtkWidget *widget, gpointer data)
{
	gboolean status;
	GtkWidget *toplevel;
	GtkEntry *entry;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint total_ftypes = 0;
	gchar **path_rtn = NULL;
	gint total_path_rtns = 0;
	win_struct *win = WIN(data);
	if((win == NULL) || FileBrowserIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	toplevel = win->toplevel;
	entry = GTK_ENTRY(win->location_entry);

	WinSetBusy(win, TRUE);

	/* Create file types list */
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    "*.*", "All files"
	);

	/* Query user for path */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(
	    "Select Location",
	    "Select", "Cancel",
	    gtk_entry_get_text(entry),
	    ftype, total_ftypes,
	    &path_rtn, &total_path_rtns,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);
					 
	/* Got user response? */
	if(status)
	{         
	    const gchar *path = (total_path_rtns > 0) ?   
		path_rtn[0] : NULL;
	    if(!STRISEMPTY(path))
		gtk_entry_set_text(entry, path);
	}

	/* Delete file types list */
	FileBrowserDeleteTypeList(ftype, total_ftypes);

	WinSetBusy(win, FALSE);
}

/*
 *	Recursive.
 *
 *	Toggles recursive.
 */
void WinRecursiveCB(GtkWidget *widget, gpointer data)
{
	gboolean v;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;
		   
	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	v = GTK_TOGGLE_BUTTON_GET_ACTIVE(win->recursive_check) ? FALSE : TRUE;
	GUIMenuItemSetCheck(
	    win->recursive_micheck, v, TRUE
	);
	gtk_toggle_button_set_active(
	    GTK_TOGGLE_BUTTON(win->recursive_check), v
	);  

	WinUpdate(win);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	Executables Only.
 *
 *	Toggles executables only.
 */
void WinExecutablesOnlyCB(GtkWidget *widget, gpointer data)
{
	gboolean v;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	v = GTK_TOGGLE_BUTTON_GET_ACTIVE(win->executables_only_check) ? FALSE : TRUE;
	GUIMenuItemSetCheck(
	    win->executables_only_micheck, v, TRUE
	);
	gtk_toggle_button_set_active(
	    GTK_TOGGLE_BUTTON(win->executables_only_check), v
	);  

	WinUpdate(win);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	Ignore Links.
 *
 *	Toggles ignore links.
 */
void WinIgnoreLinksCB(GtkWidget *widget, gpointer data)
{
	gboolean v;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	v = GTK_TOGGLE_BUTTON_GET_ACTIVE(win->ignore_links_check) ? FALSE : TRUE;
	GUIMenuItemSetCheck(
	    win->ignore_links_micheck, v, TRUE
	);
	gtk_toggle_button_set_active(
	    GTK_TOGGLE_BUTTON(win->ignore_links_check), v
	);

	WinUpdate(win);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}


/*
 *	Endeavour Recycle progress callback.
 */
static gint WinEDVRecycleProgressCB(
	gpointer data, gulong pos, gulong total
)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return(FALSE);

	WinStatusProgress(
	    win,
	    (total > 0) ? (gfloat)pos / (gfloat)total : 0.0f,
	    TRUE
	);

	return(FALSE);
}

/*
 *	Results Recycle.
 *
 *	Recycles the selected objects in the Results List.
 */
void WinResultsRecycleCB(GtkWidget *widget, gpointer data)
{
	gint row;
	gchar *name = NULL;
	GList *glist, *obj_list;
	GtkWidget *toplevel;
	GtkCList *clist;
	obj_struct *obj;
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)             
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;
	clist = GTK_CLIST(win->results_clist);
	core = CORE(win->core);

	/* Iterate through each selected row and generate objects list */
	obj_list = NULL;
	for(glist = clist->selection; glist != NULL; glist = g_list_next(glist))
	{
	    row = (gint)glist->data;
	    obj = gtk_clist_get_row_data(clist, row);
	    if(obj == NULL)
		continue;

	    obj_list = g_list_append(obj_list, obj);
	}

	/* Confirm recycle */
	if(obj_list != NULL)
	{
	    gint response;
	    gchar *buf;
	    const gint nobjs = g_list_length(obj_list);

	    /* Format confirm recycle message */
	    obj = (nobjs == 1) ? OBJ(obj_list->data) : NULL;
	    if((obj != NULL) ? !STRISEMPTY(obj->path) : FALSE)
	    {
		name = STRDUP(GET_NAME_FROM_PATH(obj->path));
		buf = g_strdup_printf(
		    "Recycle object \"%s\"?",
		    name
		);
	    }
	    else
		buf = g_strdup_printf(
		    "Recycle %i selected objects?",
		    nobjs
		);

	    /* Query user to confirm recycle */
#ifdef HAVE_EDV2
	    EDVPlaySoundQuestion(core->edv_ctx);
#endif
	    CDialogSetTransientFor(toplevel);
	    response = CDialogGetResponseIconData(
		"Confirm Recycle",
		buf,
		NULL,
		(guint8 **)icon_trash_32x32_xpm,
		CDIALOG_BTNFLAG_YES | CDIALOG_BTNFLAG_NO,
		CDIALOG_BTNFLAG_NO
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);
	    if(response != CDIALOG_RESPONSE_YES)
	    {
		g_list_free(obj_list);
		g_free(name);
		WinSetBusy(win, FALSE);
		return;
	    }
	}    

	/* Delete the objects and rows who's row data is specified in
	 * the obj_list
	 */
	win->freeze_count++;
	gtk_clist_freeze(clist);
	for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
	{
	    row = gtk_clist_find_row_from_data(clist, glist->data);
	    if(row > -1)
	    {
		gchar *buf, *path;

		obj = OBJ(glist->data);

		/* Make copy of the object's path since the object's
		 * path may become invalid
		 */
		path = STRDUP(obj->path);

		buf = g_strdup_printf(
		    "Recycling object \"%s\"",
		    GET_NAME_FROM_PATH(path)
		);
		WinStatusMessage(win, buf, TRUE);
		g_free(buf);

#ifdef HAVE_EDV2
		if(path != NULL)
		{
		    /* Recycle this object */
		    guint index = EDVRecycle(
			core->edv_ctx, path,
			TRUE,
			WinEDVRecycleProgressCB, win
		    );
		    if(index != 0)
		    {
			EDVContextSync(core->edv_ctx);
			gtk_clist_remove(clist, row);
		    }
		}
#else
/* TODO */
#endif
		WinStatusProgress(win, 0.0f, TRUE);
		g_free(path);
	    }
	}
	gtk_clist_thaw(clist);
	win->freeze_count--;

	/* Print final status */
	if(name != NULL)
	{
	    gchar *buf = g_strdup_printf(
		"Recycled object \"%s\"",
		name
	    );
	    WinStatusMessage(win, buf, FALSE);
	    g_free(buf);
	}
	else
	{
	    const gint nobjs = g_list_length(obj_list);
	    gchar *buf = g_strdup_printf(
		"Recycled %i object%s",
		nobjs,
		(nobjs == 1) ? "" : "s"
	    );
	    WinStatusMessage(win, buf, FALSE);
	    g_free(buf);
	}

	/* Delete objects list */
	g_list_free(obj_list);
	g_free(name);

	WinUpdate(win);

	WinSetBusy(win, FALSE);
}

/*
 *	Results Move.
 *
 *	Moves the selected objects in the Results List.
 */
void WinResultsMoveCB(GtkWidget *widget, gpointer data)
{
	gint row;
	gchar *name = NULL, *dest_path = NULL;
	GList *glist, *obj_list;
	GtkWidget *toplevel;
	GtkCList *clist;
	obj_struct *obj;
	core_struct *core;
	struct stat stat_buf;
	win_struct *win = WIN(data);
	if((win == NULL) || PDialogIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;
	clist = GTK_CLIST(win->results_clist);
	core = CORE(win->core);

	/* Iterate through each selected row and generate objects list */
	obj_list = NULL;
	for(glist = clist->selection; glist != NULL; glist = g_list_next(glist))
	{
	    row = (gint)glist->data;
	    obj = gtk_clist_get_row_data(clist, row);
	    if(obj == NULL)
		continue;

	    obj_list = g_list_append(obj_list, obj);
	}

	/* Query Move */
	if(obj_list != NULL)
	{
	    gint strc;
	    gchar *mesg, **strv;
	    const gint nobjs = g_list_length(obj_list);

	    obj = (nobjs == 1) ? OBJ(obj_list->data) : NULL;
	    if((obj != NULL) ? !STRISEMPTY(obj->path) : FALSE)
	    {
		name = STRDUP(GET_NAME_FROM_PATH(obj->path));
		mesg = g_strdup_printf(
		    "Move object \"%s\"",
		    name
		);
	    }
	    else
		mesg = g_strdup_printf(
		    "Move %i selected objects", 
		    nobjs
		);

	    /* Set up prompt dialog */
	    PDialogDeleteAllPrompts();
	    PDialogAddPromptWithBrowse(
		NULL, "To:", NULL,
		win, WinPDialogBrowseLocationCB
	    );
	    PDialogSetSize(350, -1);
	    gtk_window_set_transient_for(
		GTK_WINDOW(PDialogGetToplevel()), GTK_WINDOW(toplevel)
	    );
	    strv = PDialogGetResponse(
		"Move", mesg, NULL,
		PDIALOG_ICON_FILE_MOVE,
		"Move", "Cancel",
		PDIALOG_BTNFLAG_SUBMIT | PDIALOG_BTNFLAG_CANCEL,
		PDIALOG_BTNFLAG_SUBMIT,
		&strc
	    );
	    g_free(mesg);
	    gtk_window_set_transient_for(
		GTK_WINDOW(PDialogGetToplevel()), NULL
	    );
	    if((strv != NULL) && (strc > 0))
	    {
		dest_path = STRDUP((strc > 0) ? strv[0] : NULL);
	    }

	    PDialogDeleteAllPrompts();
	}

	/* Did not get destination path or user aborted? */
	if(dest_path == NULL)
	{
	    g_list_free(obj_list);
	    g_free(name);
	    WinSetBusy(win, FALSE);
	    return;
	}
	/* Destination does not exist? */
	if(stat(dest_path, &stat_buf))
	{
	    gchar *buf = g_strdup_printf(
"Destination directory does not exist:\n\
\n\
    %s\n",
		dest_path
	    );
#ifdef HAVE_EDV2
	    EDVPlaySoundError(core->edv_ctx);
#endif
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Move Failed", buf, NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_free(buf);

	    g_list_free(obj_list);
	    g_free(name);
	    g_free(dest_path);
	    WinSetBusy(win, FALSE);
	    return;
	}
	/* Destination is not a directory? */
#ifdef S_ISDIR
	else if(!S_ISDIR(stat_buf.st_mode))
#else
	else if(TRUE)
#endif
	{
#ifdef HAVE_EDV2
	    EDVPlaySoundError(core->edv_ctx);
#endif
	    CDialogSetTransientFor(toplevel);
	    CDialogGetResponse(
		"Move Failed",
		"Destination is not a directory",
		NULL,
		CDIALOG_ICON_ERROR,
		CDIALOG_BTNFLAG_OK,
		CDIALOG_BTNFLAG_OK
	    );
	    CDialogSetTransientFor(NULL);
	    g_list_free(obj_list);
	    g_free(name);
	    g_free(dest_path);
	    WinSetBusy(win, FALSE);
	    return;
	}

	/* Move the objects who's row data is specified in the
	 * obj_list
	 */
	win->freeze_count++;
	gtk_clist_freeze(clist);
	for(glist = obj_list; glist != NULL; glist = g_list_next(glist))
	{
	    row = gtk_clist_find_row_from_data(clist, glist->data);
	    if(row > -1)
	    {
		gchar *old_path, *new_path;

		obj = OBJ(glist->data);

		/* Get old path */
		old_path = STRDUP(obj->path);

		/* Get new path */
		new_path = STRDUP(PrefixPaths(
		    dest_path,
		    GET_NAME_FROM_PATH(old_path)
		));

		/* Got old & new paths? */
		if(!STRISEMPTY(old_path) && !STRISEMPTY(new_path))
		{
		    /* Move */
		    if(!rename(old_path, new_path))
		    {
			EDVNotifyQueueObjectAdded(
			    core->edv_ctx, new_path
			);
			EDVNotifyQueueObjectRemoved(
			    core->edv_ctx, old_path
			);
			EDVContextSync(core->edv_ctx);
			gtk_clist_remove(clist, row);
		    }
		}
		g_free(old_path);
		g_free(new_path);
	    }
	}    
	gtk_clist_thaw(clist);
	win->freeze_count--;  

	/* Print final status */
	if(name != NULL)
	{
	    gchar *buf = g_strdup_printf(
		"Moved object \"%s\"",
		name
	    );
	    WinStatusMessage(win, buf, FALSE);
	    g_free(buf);
	}
	else
	{
	    const gint nobjs = g_list_length(obj_list);
	    gchar *buf = g_strdup_printf(
		"Moved %i object%s",
		nobjs,
		(nobjs == 1) ? "" : "s"
	    );
	    WinStatusMessage(win, buf, FALSE);
	    g_free(buf);
	}

	/* Delete objects list */
	g_list_free(obj_list);
	g_free(name);
	g_free(dest_path);

	WinUpdate(win);

	WinSetBusy(win, FALSE);
}

/*
 *	Results List Select All.
 */
void WinResultsSelectAllCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinCListSelectAll(GTK_CLIST(win->results_clist));
}

/*
 *	Results List Unselect All.
 */
void WinResultsUnselectAllCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;
 
	if(win->freeze_count > 0)
	    return;
 
	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinCListUnselectAll(GTK_CLIST(win->results_clist));
}

/*
 *	Results List Invert Selection.
 */
void WinResultsInvertSelectionCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinCListInvertSelection(GTK_CLIST(win->results_clist));
}

/*
 *	Results Open.
 *
 *	Opens a results log file.
 */
void WinResultsOpenCB(GtkWidget *widget, gpointer data)
{
	gboolean status;
	GtkWidget *toplevel;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint total_ftypes = 0;
	gchar **path_rtn = NULL;
	gint total_path_rtns = 0;
	win_struct *win = WIN(data);
	if((win == NULL) || FileBrowserIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;

	/* Create file types list */
	FileBrowserTypeListNew(     
	    &ftype, &total_ftypes,
	    ".log", "Scan result logs"
	);
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    "*.*", "All files"
	);

	/* Query user */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(
	    "Open Scan Results Log",
	    "Open", "Cancel",
	    NULL,
	    ftype, total_ftypes,
	    &path_rtn, &total_path_rtns,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(status)
	{
	    const gchar *new_path = (total_path_rtns > 0) ?
		path_rtn[0] : NULL;
	    if(!STRISEMPTY(new_path))
	    {
		/* Open */
		WinResultsOpen(win, new_path);
		WinUpdate(win);
	    }
	}
	 
	/* Delete file types list */
	FileBrowserDeleteTypeList(ftype, total_ftypes);

	WinSetBusy(win, FALSE);
}

/*
 *	Results Save As. 
 *
 *	Save as a results log file.
 */
void WinResultsSaveAsCB(GtkWidget *widget, gpointer data)
{
	gboolean status;
	GtkWidget *toplevel;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint total_ftypes = 0;
	gchar **path_rtn = NULL;
	gint total_path_rtns = 0;
	core_struct *core;
	win_struct *win = WIN(data);
	if(win == NULL)             
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);
      
	toplevel = win->toplevel; 
	core = CORE(win->core);

	/* Create file types list */
	FileBrowserTypeListNew(     
	    &ftype, &total_ftypes,
	    ".log", "Scan result logs"
	);
	FileBrowserTypeListNew(     
	    &ftype, &total_ftypes,
	    "*.*", "All files"
	);

	/* Query user */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(
	    "Save Scan Results Log As",
	    "Save", "Cancel",
	    NULL,                       /* Initial path */
	    ftype, total_ftypes,
	    &path_rtn, &total_path_rtns,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(status)
	{
	    const gchar *new_path = (total_path_rtns > 0) ?
		path_rtn[0] : NULL;
	    if(!STRISEMPTY(new_path))
	    {
		/* Do overwrite check */
		if(WinQueryOverwrite(core, new_path, toplevel))
		{
		    /* Save */
		    WinResultsSave(win, new_path);
		    WinUpdate(win);
		}
	    }
	}

	/* Delete file types list */
	FileBrowserDeleteTypeList(ftype, total_ftypes);

	/* Reset due to possible file related change */
	FileBrowserReset();

	WinSetBusy(win, FALSE);
}

/*
 *	Results Clear.
 */
void WinResultsClearCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)             
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	WinResultsListClear(win);
	WinUpdate(win);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}


/*
 *	Virus Database Refresh.
 *
 *	Reloads the virus database to the Virus Database Tree.
 */
void WinDBRefreshCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;             

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinDBClear(win);
	WinDBQueueUpdate(win);
}

/*
 *	Virus Database Location
 *
 *	Sets the Virus Scanner's Database Location.   
 */
void WinDBLocationCB(GtkWidget *widget, gpointer data)
{
	gboolean status;
	GtkWidget *toplevel;
	fb_type_struct **ftype = NULL, *ftype_rtn = NULL;
	gint total_ftypes = 0;
	gchar **path_rtn = NULL;
	gint total_path_rtns = 0;
	cfg_item_struct *cfg_list;
	core_struct *core;
	win_struct *win = WIN(data);
	if((win == NULL) || FileBrowserIsQuery())
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	toplevel = win->toplevel;
	core = CORE(win->core);
	cfg_list = core->cfg_list;

	WinSetBusy(win, TRUE);

	/* Create file types list */
	FileBrowserTypeListNew(
	    &ftype, &total_ftypes,
	    "*.*", "All files"
	);                    

	/* Query user for path */
	FileBrowserSetTransientFor(toplevel);
	status = FileBrowserGetResponse(     
	    "Set Virus Scanner Database Location",
	    "Set", "Cancel",
	    CFGItemListGetValueS(cfg_list, CFG_PARM_DB_LOCATION),
	    ftype, total_ftypes,
	    &path_rtn, &total_path_rtns,
	    &ftype_rtn
	);
	FileBrowserSetTransientFor(NULL);

	/* Got user response? */
	if(status) 
	{
	    const gchar *path = (total_path_rtns > 0) ?
		path_rtn[0] : NULL;
	    if(!STRISEMPTY(path))
	    {
		/* Set new virus database location */
		CFGItemListSetValueS(
		    cfg_list, CFG_PARM_DB_LOCATION,
		    path, FALSE
		);
		  
		/* Update Virus Database Tree due to virus database
		 * location change
		 */
		WinDBClear(win);
		if(win->page_num == WIN_PAGE_NUM_DB)
		    WinDBQueueUpdate(win);

		WinUpdate(win);
	    }
	}

	/* Delete file types list */
	FileBrowserDeleteTypeList(ftype, total_ftypes);

	WinSetBusy(win, FALSE); 
}

/*
 *	Virus Database Update From Network.
 *
 *	Updates the virus database from the network.
 */
void WinDBUpdateNetCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)                         
	    return;             

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);

	/* Get virus database update values */

	/* Start virus database update */
	WinUpdateProcessStart(win);

	WinSetBusy(win, FALSE);
}


/*
 *	Refresh.
 */
void WinRefreshCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	WinUpdate(win);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	Go to Scan Page.
 */
void WinViewScanPageCB(GtkWidget *widget, gpointer data)
{
	const win_page_num page_num = WIN_PAGE_NUM_SCAN;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;
		   
	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	if(win->page_num != page_num)
	    gtk_notebook_set_page(
		GTK_NOTEBOOK(win->notebook), (gint)page_num
	    );

	WinUpdate(win);
}

/*
 *	Go to Results Page.
 */
void WinViewResultsPageCB(GtkWidget *widget, gpointer data)
{
	const win_page_num page_num = WIN_PAGE_NUM_RESULTS;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;
		   
	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	if(win->page_num != page_num)
	    gtk_notebook_set_page(
		GTK_NOTEBOOK(win->notebook), (gint)page_num
	    );

	WinUpdate(win);
}

/*
 *	Go to Virus Database Page.
 */
void WinViewDBPageCB(GtkWidget *widget, gpointer data)  
{
	const win_page_num page_num = WIN_PAGE_NUM_DB;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	if(win->page_num != page_num)
	    gtk_notebook_set_page(
		GTK_NOTEBOOK(win->notebook), (gint)page_num
	    );

	WinUpdate(win);
}

/*
 *	Show/Hide Tool Bar.
 */
void WinViewToolBarCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	if(WinToolBarIsShown(win))
	    WinToolBarSetShow(win, FALSE);
	else
	    WinToolBarSetShow(win, TRUE);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	Show/Hide Status Bar.
 */
void WinViewStatusBarCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	if(WinStatusBarIsShown(win))
	    WinStatusBarSetShow(win, FALSE);
	else
	    WinStatusBarSetShow(win, TRUE);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}


/*
 *	Help Contents.
 */
void WinHelpContentsCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	Help(CORE(win->core), "Contents", win->toplevel);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	Help Scanning.
 */
void WinHelpScanningCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	Help(CORE(win->core), "Scanning", win->toplevel);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	Help Updating.
 */
void WinHelpUpdatingCB(GtkWidget *widget, gpointer data)
{
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	win->freeze_count++;
	WinSetBusy(win, TRUE);

	Help(CORE(win->core), "Updating", win->toplevel);

	win->freeze_count--;
	WinSetBusy(win, FALSE);
}

/*
 *	About.
 */
void WinHelpAboutCB(GtkWidget *widget, gpointer data)
{
	gchar *s, *av_engine_name_version;
	GtkWidget *toplevel;
	win_struct *win = WIN(data);
	if(win == NULL)
	    return;

	if(win->freeze_count > 0)
	    return;

	if((widget != NULL) ? !GTK_WIDGET_SENSITIVE(widget) : FALSE)
	    return;

	WinSetBusy(win, TRUE);

	toplevel = win->toplevel;

	av_engine_name_version = AVScanGetAVEngineNameVersionString();
	s = g_strdup_printf(
PROG_NAME_FULL "\n\
Version " PROG_VERSION "\n\
\n\
" PROG_URL "\n\
\n\
" PROG_COPYRIGHT "\
\n\
AntiVirus Engine: %s",
	    (av_engine_name_version != NULL) ?
		av_engine_name_version : "None"
	);
	free(av_engine_name_version);

	CDialogSetTransientFor(toplevel);
	CDialogGetResponseIconData(
	    "About " PROG_NAME_FULL,
	    s,
	    NULL,
	    (guint8 **)icon_avscan_32x32_xpm,
	    CDIALOG_BTNFLAG_OK,
	    CDIALOG_BTNFLAG_OK
	);
	CDialogSetTransientFor(NULL);

	free(s);

	WinSetBusy(win, FALSE);
}
