#include <stdlib.h>
#include <string.h>
#include <glib.h>
#include "obj.h"


obj_struct *ObjNew(void);
obj_struct *ObjCopy(const obj_struct *obj);
void ObjDelete(obj_struct *obj);

GList *ObjDDEBufferParse(const guint8 *buf, const gint buf_len);
guint8 *ObjDDEBufferAppend(
	guint8 *buf, gint *buf_len_rtn,
	const obj_struct *obj
);
GList *ObjDDEBufferParsePath(const guint8 *buf, const gint buf_len);
guint8 *ObjDDEBufferAppendPath(
	guint8 *buf, gint *buf_len_rtn,
	const obj_struct *obj
);


#define ATOI(s)         (((s) != NULL) ? atoi(s) : 0)
#define ATOL(s)         (((s) != NULL) ? atol(s) : 0)
#define ATOF(s)         (((s) != NULL) ? atof(s) : 0.0f)
#define STRDUP(s)       (((s) != NULL) ? g_strdup(s) : NULL)

#define MAX(a,b)        (((a) > (b)) ? (a) : (b))
#define MIN(a,b)        (((a) < (b)) ? (a) : (b))
#define CLIP(a,l,h)     (MIN(MAX((a),(l)),(h)))
#define STRLEN(s)       (((s) != NULL) ? strlen(s) : 0)
#define STRISEMPTY(s)   (((s) != NULL) ? (*(s) == '\0') : TRUE)


/*
 *	Creates a new Object.
 */
obj_struct *ObjNew(void)
{
	return(
	    OBJ(g_malloc0(sizeof(obj_struct)))
	);

}

/*
 *	Coppies the Object.
 */
obj_struct *ObjCopy(const obj_struct *obj)
{
	const obj_struct *src = obj;
	obj_struct *tar = ObjNew();
	if(tar == NULL)
	    return(NULL);

	tar->name = STRDUP(src->name);
	tar->path = STRDUP(src->path);
	tar->options = src->options;
	tar->last_runned = src->last_runned;

	return(tar);
}

/*
 *	Deletes the Object.
 */
void ObjDelete(obj_struct *obj)
{
	if(obj == NULL)
	    return;

	g_free(obj->name);
	g_free(obj->path);
	g_free(obj);
}


/*
 *	Parses the DDE buffer into a list of Objects.
 */
GList *ObjDDEBufferParse(const guint8 *buf, const gint buf_len)
{
	gint len;
	gchar *s;
	const guint8 *buf_ptr, *buf_end, *buf_next;
	GList *glist = NULL;
	obj_struct *obj = NULL;

	if(buf == NULL)
	    return(glist);

	/* Iterate through the buffer */
	buf_ptr = buf;
	buf_end = buf_ptr + buf_len;
	while(buf_ptr < buf_end)
	{
	    /* Create a new Object and append it to the list */
	    obj = ObjNew();
	    if(obj == NULL)
		break;

	    glist = g_list_append(glist, obj);

	    /* Format is:
	     *
	     * string		Path
	     * string		Name
	     * guint		Options
	     * gulong		Last Runned
	     */

	    /* Path */
	    for(buf_next = buf_ptr; buf_next < buf_end; buf_next++)
	    {
		if(*(gchar *)buf_next == '\0')
		    break;
	    }
	    len = (gint)(buf_next - buf_ptr);
	    s = (gchar *)g_malloc((len + 1) * sizeof(gchar));
	    if(s != NULL)
	    {
		if(len > 0)
		    memcpy(
			s,
			buf_ptr,
			len * sizeof(gchar)
		    );
		s[len] = '\0';

		g_free(obj->path);
		obj->path = s;
	    }
	    buf_ptr = buf_next + 1;	/* Seek past the null byte */
	    if(buf_ptr >= buf_end)
		break;

	    /* Name */
	    for(buf_next = buf_ptr; buf_next < buf_end; buf_next++)
	    {
		if(*(gchar *)buf_next == '\0')
		    break;
	    }
	    len = (gint)(buf_next - buf_ptr);
	    s = (gchar *)g_malloc((len + 1) * sizeof(gchar));
	    if(s != NULL)
	    {
		if(len > 0)
		    memcpy(
			s,
			buf_ptr,
			len * sizeof(gchar)
		    );
		s[len] = '\0';
		g_free(obj->name);
		obj->name = s;
	    }
	    buf_ptr = buf_next + 1;	/* Seek past the null byte */
	    if(buf_ptr >= buf_end)
		break;

	    /* Options */
	    len = sizeof(guint);
	    buf_next = buf_ptr + len;
	    if(buf_next <= buf_end)
	    {
		obj->options = (avscan_options)*(guint *)buf_ptr;
	    }
	    buf_ptr = buf_next;
	    if(buf_ptr >= buf_end)
		break;

	    /* Last Runned */
	    len = sizeof(gulong);
	    buf_next = buf_ptr + len;
	    if(buf_next <= buf_end)
	    {
		obj->last_runned = *(gulong *)buf_ptr;
	    }
	    buf_ptr = buf_next;
	}

	if(obj != NULL)
	{
	    /* Try to complete the object's values */
	    if((obj->name == NULL) && (obj->path != NULL))
	    {
		s = (gchar *)strrchr((char *)obj->path, '/');
		if(s != NULL)
		    s++;
		else
		    s = obj->path;
		g_free(obj->name);
		obj->name = STRDUP(s);
	    }
	}

	return(glist);
}

/*
 *	Appends the Object to the DDE buffer.
 */
guint8 *ObjDDEBufferAppend(
	guint8 *buf, gint *buf_len_rtn,
	const obj_struct *obj
)
{
	gint buf_len;

	if((obj == NULL) || (buf_len_rtn == NULL))
	    return(buf);

	buf_len = *buf_len_rtn;

#define APPEND_BUF_S(_s_)	{		\
 const gint	buf_i = buf_len,		\
		s_len = STRLEN(_s_);		\
						\
 buf_len = buf_i + s_len + 1;			\
 buf = (guint8 *)g_realloc(			\
  buf,						\
  buf_len * sizeof(guint8)			\
 );						\
 if(buf != NULL) {				\
  if(s_len > 0)					\
   memcpy(					\
    buf + buf_i, (_s_),				\
    s_len * sizeof(guint8)			\
   );						\
  buf[buf_i + s_len] = '\0';			\
 } else {					\
  buf_len = 0;					\
 }						\
}

#define APPEND_BUF_D(_data_,_len_)	{	\
 const gint	buf_i = buf_len;		\
						\
 buf_len = buf_i + (_len_);			\
 buf = (guint8 *)g_realloc(			\
  buf,						\
  buf_len * sizeof(guint8)			\
 );						\
 if(buf != NULL) {				\
  if((_len_) > 0)				\
   memcpy(					\
    buf + buf_i, (_data_),			\
    (_len_) * sizeof(guint8)			\
   );						\
 } else {					\
  buf_len = 0;					\
 }						\
}

	APPEND_BUF_S(obj->path);
	APPEND_BUF_S(obj->name);
	APPEND_BUF_D((guint *)&obj->options, sizeof(guint));
	APPEND_BUF_D((gulong *)&obj->last_runned, sizeof(gulong));

	*buf_len_rtn = buf_len;

	return(buf);
#undef APPEND_BUF_S
#undef APPEND_BUF_D
}

/*
 *	Same as ObjDDEBufferParse() except that only the path is
 *	parsed from the buffer.
 */
GList *ObjDDEBufferParsePath(const guint8 *buf, const gint buf_len)
{
	gint len;
	gchar *url;
	const gchar *s;
	const guint8 *buf_end = buf + buf_len;
	GList *glist = NULL;
	obj_struct *obj;

	if(buf == NULL)
	    return(glist);

	/* Iterate through the buffer */
	while(buf < buf_end)
	{
	    /* Create a new Object and append it to the list */
	    obj = ObjNew();
	    if(obj == NULL)
		break;

	    glist = g_list_append(glist, obj);

	    /* Parse the URL */
	    s = (const gchar *)buf;
	    while(s < (const gchar *)buf_end)
	    {
		if(*s == '\0')
		    break;
		s++;
	    }
	    len = s - (const gchar *)buf;

	    url = (gchar *)g_malloc((len + 1) * sizeof(gchar));
	    if(url != NULL)
	    {
		const gchar *s2, *s3;

		memcpy(url, buf, len * sizeof(gchar));
		url[len] = '\0';

		/* Seek s2 past the "://" deliminator to the start of
		 * the path
		 */
		for(s2 = url; *s2 != '\0'; s2++)
		{
		    if(*s2 == ':')  
		    {
			if(*s2 == ':')
			    s2++;
			if(*s2 == '/')
			    s2++;
			if(*s2 == '/')
			    s2++;

			while((*s2 != '\0') && (*s2 != '/'))
			    s2++;

			break;
		    }
		    else if(*s2 == '/')
		    {
			break;
		    }
		}

		/* Get/copy the path from the URL */
		obj->path = STRDUP(s2);

		/* Get/copy the name from the path */
		s2 = obj->path;
		if(s2 != NULL)
		{
		    for(s3 = s2; *s3 != '\0'; s3++);

		    while(s3 > s2)
		    {
			if(*s3 == '/')
			{
			    s3++;
			    break;
			}
			s3--;
		    }

		    obj->name = STRDUP(s3);
		}
		 
		g_free(url);
	    }
	     
	    buf += len;

	    buf++;                      /* Skip deliminator */
	}
	 
	return(glist);
}

/*
 *	Same as ObjDDEBufferAppend() except that only the path is
 *	appended to the buffer.
 */
guint8 *ObjDDEBufferAppendPath(
	guint8 *buf, gint *buf_len_rtn,
	const obj_struct *obj
)
{
	gint i, buf_len, url_len;
	gchar *url;

	if((obj == NULL) || (buf_len_rtn == NULL))
	    return(buf); 

	/* Get current length and starting position */
	buf_len = i = *buf_len_rtn;

	/* Format the URL */
	url = g_strdup_printf("file://%s", obj->path);

	/* Calculate new buffer length needed */
	url_len = STRLEN(url);
	buf_len += url_len;
	buf_len++;				/* Null byte/deliminator */

	/* Increase the buffer allocation */
	buf = (guint8 *)g_realloc(
	    buf,
	    buf_len * sizeof(guint8)
	);
	if(buf == NULL)
	{
	    *buf_len_rtn = 0;
	    return(buf);
	}

	/* Copy the URL to the buffer */
	if(url != NULL)
	{
	    if(url_len > 0)
	    {
		memcpy(buf + i, url, url_len * sizeof(guint8));
		i += url_len;
	    }
	    g_free(url);
	}

	/* Null terminate the buffer */
	buf[i] = '\0';

	*buf_len_rtn = buf_len;

	return(buf);
}
