/*
    AWFFull - A Webalizer Fork, Full o' features
    
    $Id: awffull.c 392 2007-05-07 11:49:54Z steve $

    Copyright (C) 1997-2001  Bradford L. Barrett (brad@mrunix.net)
    Copyright 2002, 2004 by Stanislaw Yurievich Pusep
    Copyright (C) 2004, 2005, 2006, 2007 by Stephen McInerney (spm@stedee.id.au)
    Copyright (C) 2006 by Alexander Lazic (al-awffull@none.at)
    Copyright (C) 2006 by Benoit Rouits (brouits@free.fr)
    
    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version, and provided that the above
    copyright and permission notice is included with all distributed
    copies of this or derived software.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA

    This software uses the gd graphics library, which is copyright by
    Quest Protein Database Center, Cold Spring Harbor Labs.  Please
    see the documentation supplied with the library for additional
    information and license terms, or visit www.boutell.com/gd/ for the
    most recent version of the library and supporting documentation.
*/

/*********************************************/
/* STANDARD INCLUDES                         */
/*********************************************/
#include "awffull.h"                            /* main header              */

/* internal function prototypes */

void clear_month(void);                         /* clear monthly stuff */
char *unescape(char *);                         /* unescape URL's      */
char from_hex(char);                            /* convert hex to dec  */
int isurlchar(unsigned char);                   /* valid URL char fnc. */
static char *save_opt(char *);                  /* save conf option    */
void srch_string(char *);                       /* srch str analysis   */
char *get_domain(char *);                       /* return domain name  */
char *our_gzgets(gzFile, char *, int);          /* our gzgets          */
int do_agent_mangling(void);
void option_checks(void);                       /* Various early checks */

/*********************************************/
/* GLOBAL VARIABLES                          */
/*********************************************/

const char *version = PACKAGE_VERSION;          /* program version - always 4 characters!!!, else modify "17" in preserve.c, version comparison         */
const char *moddate = "07-May-2007";            /* modification date        */

int verbose = 2;                                /* 2=verbose,1=err, 0=none  */
int g_verbosity = VERBOSE0;                     /* How many extra stuff to show. If 0, none
                                                   Bigger numbers, show more. */
int time_me = 0;                                /* timing display flag      */
int local_time = 1;                             /* 1=localtime 0=GMT (UTC)  */
int ignore_hist = 0;                            /* history flag (1=skip)    */
int hourly_graph = 1;                           /* hourly graph display     */
int hourly_stats = 1;                           /* hourly stats table       */
int daily_graph = 1;                            /* daily graph display      */
int daily_stats = 1;                            /* daily stats table        */
int ctry_graph = 1;                             /* country graph display    */
int url_byhits_graph = 0;                       /* TOP URL's by Hits graph display    */
int url_byvol_graph = 0;                        /* TOP URL's by Volume graph display    */
int exit_pages_graph = 0;                       /* TOP Exit Pgaes Graph Display - 0 no, 1 hits, 2 visits */
int entry_pages_graph = 0;                      /* TOP Entry Pgaes Graph Display - 0 no, 1 hits, 2 visits */
int sites_bypages_graph = 0;                    /* TOP Sites Pages Graph - 0 no, 1 yes */
int sites_byvol_graph = 0;                      /* TOP Sites Volume Graph - 0 no, 1 yes */
int agents_graph = 0;                           /* TOP User Agents Graph - 0 no, 1 yes */
int shade_groups = 1;                           /* Group shading 0=no 1=yes */
int hlite_groups = 1;                           /* Group hlite 0=no 1=yes   */
int mangle_agent = 0;                           /* mangle user agents       */
int incremental = 0;                            /* incremental mode 1=yes   */
int use_https = 0;                              /* use 'https://' on URL's  */
int visit_timeout = 1800;                       /* visit timeout (seconds)  */
int graph_legend = 1;                           /* graph legend (1=yes)     */
int graph_lines = 2;                            /* graph lines (0=none)     */
int fold_seq_err = 0;                           /* fold seq err (0=no)      */
int log_type = LOG_AUTO;                        /* (0= auto, 1=clf, 2=ftp, 3=squid, 4=apache)  */
bool force_log_type = false;                    /* Set to true if log_type set by Command line or Config File */
int group_domains = 0;                          /* Group domains 0=none     */
int hide_sites = 0;                             /* Hide ind. sites (0=no)   */
const char *hname = NULL;                       /* hostname for reports     */
const char *state_fname = FILENAME_CURRENT;     /* run state file name      */
const char *hist_fname = FILENAME_HISTORY;      /* name of history file     */
const char *html_ext = "html";                  /* HTML file prefix         */
const char *dump_ext = "tab";                   /* Dump file prefix         */
const char *css_filename = FILENAME_CSS;        /* CSS file name            */
char *conf_fname = NULL;                        /* name of config file      */
char *log_fname = NULL;                         /* log file pointer         */
char *out_dir = NULL;                           /* output directory         */
char *blank_str = "";                           /* blank string             */
bool track_206_reqs = false;                    /* Track 206 Requests       */
bool display_match_counts = false;              /* Display the match counts vs Groups, Ignore's etc */
bool display_yearly_subtotals = false;          /* Display Yearly Subtotals on main page */
int index_months = 12;                          /* Number of Months to display (12) */
int graph_index_x = GRAPH_INDEX_X;              /* Size of Main Graph X     */
int graph_index_y = GRAPH_INDEX_Y;              /* Size of Main Graph Y     */
int graph_daily_x = GRAPH_DAILY_X;              /* Size of Monthly Graph X  */
int graph_daily_y = GRAPH_DAILY_Y;              /* Size of Monthly Graph Y  */
int graph_hourly_x = GRAPH_HOURLY_X;            /* Size of Daily Graph X    */
int graph_hourly_y = GRAPH_HOURLY_Y;            /* Size of Daily Graph Y    */
int graph_pie_x = GRAPH_PIE_X;                  /* Size of Pie Graph X      */
int graph_pie_y = GRAPH_PIE_Y;                  /* Size of Pie Graph Y      */
bool use_kanji = false;                         /* Do special offsets in graphs for Japanese Character Set */

int ntop_sites = 30;                            /* top n sites to display   */
int ntop_sitesK = 10;                           /* top n sites (by kbytes)  */
int ntop_urls = 30;                             /* top n url's to display   */
int ntop_urlsK = 10;                            /* top n url's (by kbytes)  */
int ntop_entry = 10;                            /* top n entry url's        */
int ntop_exit = 10;                             /* top n exit url's         */
int ntop_refs = 30;                             /* top n referrers ""       */
int ntop_agents = 15;                           /* top n user agents ""     */
int ntop_ctrys = 30;                            /* top n countries   ""     */
int ntop_search = 20;                           /* top n search strings     */
int ntop_users = 20;                            /* top n users to display   */
int ntop_error = 0;                             /* top n 404 errors to display   */

int all_sites = 0;                              /* List All sites (0=no)    */
int all_urls = 0;                               /* List All URL's (0=no)    */
int all_refs = 0;                               /* List All Referrers       */
int all_agents = 0;                             /* List All User Agents     */
int all_search = 0;                             /* List All Search Strings  */
int all_users = 0;                              /* List All Usernames       */
int all_errors = 0;                             /* List All ErrorPages      */

int dump_sites = 0;                             /* Dump tab delimited sites */
int dump_urls = 0;                              /* URL's                    */
int dump_refs = 0;                              /* Referrers                */
int dump_agents = 0;                            /* User Agents              */
int dump_users = 0;                             /* Usernames                */
int dump_search = 0;                            /* Search strings           */
bool dump_countries = false;                    /* Countries                */
bool dump_entry_pages = false;                  /* Entry Pages              */
bool dump_exit_pages = false;                   /* Exit Pages               */
int dump_header = 0;                            /* Dump header as first rec */
char *dump_path = NULL;                         /* Path for dump files      */

const char *flags_location = NULL;              /* Display Country Flags if not null */
bool use_geoip = false;                         /* If able AND true, use GeoIP capability */
bool have_geoip = false;                        /* If GeoIP is compiled in  */
const char *geoip_database = GEOIP_DATABASE;    /* location of the GeoIP Database file */

#if HAVE_GEOIP_H
GeoIP *gi;                                      /* GeoIP access             */
#endif

int g_cur_year = 0, g_cur_month = 0,            /* year/month/day/hour      */
    g_cur_day = 0, g_cur_hour = 0,              /* tracking variables       */
    g_cur_min = 0, g_cur_sec = 0;

u_long cur_tstamp = 0;                          /* Timestamp...             */
u_long rec_tstamp = 0;
u_long req_tstamp = 0;
u_long epoch;                                   /* used for timestamp adj.  */

int check_dup = 0;                              /* check for dup flag       */
int gz_log = 0;                                 /* gziped log? (0=no)       */

unsigned long long t_xfer = 0.0;                /* monthly total xfer value */
u_long t_hit = 0, t_file = 0, t_site = 0,       /* monthly total vars       */
    t_url = 0, t_ref = 0, t_agent = 0, t_page = 0, t_visit = 0, t_user = 0;

u_long t_bad = 0;                               /* total bad records */
u_long t_errorp = 0;                            /* total error pages */
u_long t_ignored = 0;                           /* total ignored records */

u_long t_bookmarks = 0;

unsigned long long tm_xfer[31];                 /* daily transfer totals    */

u_long tm_hit[31], tm_file[31],                 /* daily total arrays       */
 tm_site[31], tm_page[31], tm_visit[31], tm_bookm[31];

u_long dt_site;                                 /* daily 'sites' total      */

u_long ht_hit = 0, mh_hit = 0;                  /* hourly hits totals       */

u_long th_hit[24], th_file[24],                 /* hourly total arrays      */
 th_page[24], th_bookm[24];

unsigned long long th_xfer[24];

int f_day, l_day;                               /* first/last day vars      */

struct utsname system_info;                     /* system info structure    */

u_long ul_bogus = 0;                            /* Dummy counter for groups */

struct log_struct log_rec;                      /* expanded log storage     */

time_t now;                                     /* used by current_time funct   */
struct tm *tp;                                  /* to generate timestamp    */
char timestamp[64];                             /* for the reports          */
time_t temp_time_squid;                         /* For pulling in squid times */

gzFile gzlog_fp;                                /* gzip logfile pointer     */
FILE *log_fp;                                   /* regular logfile pointer  */

char buffer[BUFSIZE];                           /* log file record buffer   */
char tmp_buf[BUFSIZE];                          /* used to temp save above  */

CLISTPTR *top_ctrys = NULL;                     /* Top countries table      */

#define GZ_BUFSIZE 16384                        /* our_getfs buffer size    */
char f_buf[GZ_BUFSIZE];                         /* our_getfs buffer         */
char *f_cp = f_buf + GZ_BUFSIZE;                /* pointer into the buffer  */
int f_end;                                      /* count to end of buffer   */

char hit_color[] = DKGREEN;                     /* graph hit color          */
char file_color[] = BLUE;                       /* graph file color         */
char site_color[] = ORANGE;                     /* graph site color         */
char kbyte_color[] = RED;                       /* graph kbyte color        */
char page_color[] = CYAN;                       /* graph page color         */
char visit_color[] = YELLOW;                    /* graph visit color        */
char bookm_color[] = PURPLE;                    /* graph bookm color        */
char pie_color1[] = DKGREEN;                    /* pie additional color 1  */
char pie_color2[] = ORANGE;                     /* pie additional color 2  */
char pie_color3[] = BLUE;                       /* pie additional color 3  */
char pie_color4[] = RED;                        /* pie additional color 4  */

/*********************************************/
/* MAIN - start here                         */
/*********************************************/

int
main(int argc, char *argv[])
{
    int i;                                      /* generic counter             */
    char *cp1, *cp2, *cp3;                      /* generic char pointers       */
    LISTPTR lptr;                               /* generic list pointer        */

    extern char *optarg;                        /* used for command line       */
    extern int optind;                          /* parsing routine 'getopt'    */
    extern int opterr;

    time_t start_time, end_time;                /* program timers              */
    float temp_time;                            /* temporary time storage      */
    struct tms mytms;                           /* bogus tms structure         */

    int good_rec = 0;                           /* 1 if we had a good record   */
    u_long total_rec = 0;                       /* Total Records Processed     */
    u_long total_ignore = 0;                    /* Total Records Ignored       */
    u_long total_bad = 0;                       /* Total Bad Records           */

    bool isapage = false;                       /* Flag. Is this a page or not? */

    struct tm time_rec;                         /* Gotta convert that string'ed time into a timerec first */
    struct tm prev_time_rec;                    /* Hang onto the pevious time_rec in case of out of sequence errors */

    char str_previous_time[SIZE_DATE_TIME] = "";

    char *current_locale = NULL;                /* the locale, as returned from setlocale */
    char *message_catalog_dir = NULL;           /* Directory for all the translations */
    char *message_domain = NULL;                /* current message domain for translations */
    char *envlang, *envlanguage;

#if ENABLE_NLS
    /* Reduce Surprises. Unset most LC_* env settings. Basically, only let LANG and/or LANGUAGE change stuff.
     * Too many cross polination mess ups otherwise. */
    unsetenv("LC_CTYPE");
    unsetenv("LC_MESSAGES");
    unsetenv("LC_COLLATE");
    unsetenv("LC_MONETARY");
    unsetenv("LC_NUMERIC");
    unsetenv("LC_TIME");
    unsetenv("LC_PAPER");
    unsetenv("LC_NAME");
    unsetenv("LC_ADDRESS");
    unsetenv("LC_TELEPHONE");
    unsetenv("LC_MEASUREMENT");
    unsetenv("LC_IDENTIFICATION");

    current_locale = setlocale(LC_ALL, "");
    message_catalog_dir = bindtextdomain(PACKAGE, LOCALEDIR);
    message_domain = textdomain(PACKAGE);
    VPRINT(VERBOSE1, "Current Locale: %s\nMessage Catalog: %s\nNessage Domain: %s\n", current_locale, message_catalog_dir, message_domain);
    envlang = getenv("LANG");
    envlanguage = getenv("LANGUAGE");
    VPRINT(VERBOSE1, "Lang: %s\nLanguage: %s\n", envlang, envlanguage);
    if (envlang != NULL) {
        if (strncmp("ja_JP", envlang, 5) == 0) {
            use_kanji = true;
        }
    } else if (envlanguage != NULL) {
        if (strncmp("ja_JP", envlanguage, 5) == 0) {
            use_kanji = true;
        }
    }
#endif

    assign_messages();                          /* Load up AWFFull's general messages, country names etc */

    /* initalize epoch */
    epoch = jdate(1, 1, 1970);                  /* used for timestamp adj.     */

    /* Zero the time structures */
    memset(&time_rec, 0, sizeof(time_rec));
    memset(&prev_time_rec, 0, sizeof(prev_time_rec));

    /* add default index. alias */
    add_list_member("index.", &index_alias, USESPACE);

    snprintf(tmp_buf, sizeof(tmp_buf), "%s/%s", ETCDIR, FILENAME_CONFIG);
    /* check for default config file */
    if (!access(FILENAME_CONFIG, F_OK))
        get_config(FILENAME_CONFIG);
    else if (!access(tmp_buf, F_OK))
        get_config(tmp_buf);

    process_options(argc, argv);
    option_checks();

    if (argc - optind != 0)
        log_fname = argv[optind];
    if (log_fname && (log_fname[0] == '-'))
        log_fname = NULL;                       /* force STDIN?   */

    /* check for gzipped file - .gz */
    if (log_fname)
        if (!strcmp((log_fname + strlen(log_fname) - 3), ".gz"))
            gz_log = 1;

    /* setup our internal variables */
    init_counters();                            /* initalize main counters         */

    for (i = 0; i < MAXHASH; i++) {
        sm_htab[i] = sd_htab[i] = NULL;         /* initalize hash tables           */
        um_htab[i] = NULL;
        rm_htab[i] = NULL;
        am_htab[i] = NULL;
        sr_htab[i] = NULL;
    }

    /* open log file */
    if (gz_log) {
        gzlog_fp = gzopen(log_fname, "rb");
        if (gzlog_fp == Z_NULL) {
            /* Error: Can't open log file ... */
            ERRVPRINT(VERBOSE0, "%s %s\n", _("Error: Can't open log file"), log_fname);
            exit(1);
        }
    } else {
        if (log_fname) {
            log_fp = fopen(log_fname, "r");
            if (log_fp == NULL) {
                /* Error: Can't open log file ... */
                ERRVPRINT(VERBOSE0, "%s %s\n", _("Error: Can't open log file"), log_fname);
                exit(1);
            }
        }
    }

    /* Using logfile ... */
    VPRINT(VERBOSE1, "%s %s", _("Using logfile"), log_fname ? log_fname : "STDIN");
    if (gz_log) {
        VPRINT(VERBOSE1, " (gzip)");
    }
    VPRINT(VERBOSE1, "\n");


    /* switch directories if needed */
    if (out_dir) {
        if (chdir(out_dir) != 0) {
            /* Error: Can't change directory to ... */
            ERRVPRINT(VERBOSE0, "%s %s\n", _("Error: Can't change directory to"), out_dir);
            exit(1);
        }
    }

    /* Creating output in ... */
    VPRINT(VERBOSE1, "%s %s\n", _("Creating output in"), out_dir ? out_dir : _("current directory"));

    /* prep hostname */
    if (!hname) {
        if (uname(&system_info)) {
            hname = "localhost";
        } else {
            hname = system_info.nodename;
        }
    }

    /* Hostname for reports is ... */
    VPRINT(VERBOSE1, "%s '%s'\n", _("Hostname for reports is"), hname);
    VPRINT(VERBOSE1, "%s %s\n", _("Using CSS file:"), css_filename);

    /* get past history */
    if (ignore_hist) {
        VPRINT(VERBOSE1, "%s\n", _("Ignoring previous history..."));
    } else
        get_history();

    if (incremental) {                          /* incremental processing?         */
        if ((i = restore_state())) {            /* restore internal data structs   */
            /* Error: Unable to restore run data (error num) */
            ERRVPRINT(VERBOSE0, "%s (%d)\n", _("Error: Unable to restore run data"), i);
            exit(1);
        }
    }

    /* Allocate memory for our TOP countries array */
    if (ntop_ctrys != 0) {
        if ((top_ctrys = calloc(ntop_ctrys, sizeof(CLISTPTR))) == NULL) {
            /* Can't get memory, Top Countries disabled! */
            ERRVPRINT(VERBOSE1, "%s\n", _("Can't allocate enough memory, Top Countries disabled!"));
            ntop_ctrys = 0;
        }
    }

    /* Do any graphical output Setup */
    initialise_graphs();

    start_time = times(&mytms);

    /*********************************************
     * MAIN PROCESS LOOP - read through log file *
     *********************************************/

    while ((gz_log) ? (our_gzgets(gzlog_fp, buffer, BUFSIZE) != Z_NULL) : (fgets(buffer, BUFSIZE, log_fname ? log_fp : stdin) != NULL)) {
        total_rec++;
        if (strlen(buffer) == (BUFSIZE - 1)) {
            ERRVPRINT(VERBOSE1, "%s %lu", _("Error: Skipping oversized log record:"), total_rec);
            ERRVPRINT(VERBOSE3, "  ==> %s\n", buffer);
            total_bad++;                        /* bump bad record counter      */

            /* get the rest of the record */
            while ((gz_log) ? (our_gzgets(gzlog_fp, buffer, BUFSIZE) != Z_NULL) : (fgets(buffer, BUFSIZE, log_fname ? log_fp : stdin) != NULL)) {
                ERRVPRINT(VERBOSE3, "      %s\n", buffer);
                if (strlen(buffer) < BUFSIZE - 1) {
                    break;
                }
            }
            continue;                           /* go get next record if any    */
        }

        /* got a record... */
        VPRINT(VERBOSE3, "==> %s", buffer);
        strcpy(tmp_buf, buffer);                /* save buffer in case of error */
        if (parse_record(buffer)) {             /* parse the record             */
            /*********************************************
             * PASSED MINIMAL CHECKS, DO A LITTLE MORE   *
             *********************************************/
            /* Flag as a good one */
            good_rec = 1;

            /*********************************************
             * DO SOME PRE-PROCESS FORMATTING            *
             *********************************************/
            /* un-escape URL */
            unescape(log_rec.url);

            /* strip query portion of cgi scripts */
            cp1 = log_rec.url;
            while (*cp1 != '\0')
                if (!isurlchar(*cp1)) {
                    *cp1 = '\0';
                    break;
                } else
                    cp1++;
            if (log_rec.url[0] == '\0') {
                log_rec.url[0] = '/';
                log_rec.url[1] = '\0';
            }

            /* strip off index.html (or any aliases) */
            lptr = index_alias;
            while (lptr != NULL) {
                if ((cp1 = strstr(log_rec.url, lptr->string)) != NULL) {
                    if ((cp1 == log_rec.url) || (*(cp1 - 1) == '/')) {
                        *cp1 = '\0';
                        if (log_rec.url[0] == '\0') {
                            log_rec.url[0] = '/';
                            log_rec.url[1] = '\0';
                        }
                        break;
                    }
                }
                lptr = lptr->next;
            }

            /* unescape referrer */
            unescape(log_rec.refer);

            /* fix referrer field */
            cp1 = log_rec.refer;
            cp3 = cp2 = cp1;
            if (*cp2 != '\0') {
                while (*cp1 != '\0') {
                    if ((*cp1 < 32 && *cp1 > 0) || *cp1 == 127 || *cp1 == '<')
                        *cp1 = '\0';
                    else
                        *cp2++ = *cp1++;
                }
            }

            /* strip query portion of cgi referrals */
            cp1 = log_rec.refer;
            if (*cp1 != '\0') {
                while (*cp1 != '\0') {
                    if (!isurlchar(*cp1)) {
                        /* Save query portion in log.rec.srchstr */
                        strncpy(log_rec.srchstr, cp1, MAXSRCH);
                        *cp1++ = '\0';
                        break;
                    } else
                        cp1++;
                }
                /* handle null referrer */
                if (log_rec.refer[0] == '\0') {
                    log_rec.refer[0] = '-';
                    log_rec.refer[1] = '\0';
                }
            }

            /* if HTTP request, lowercase http://sitename/ portion */
            cp1 = log_rec.refer;
            if ((*cp1 == 'h') || (*cp1 == 'H')) {
                while ((*cp1 != '/') && (*cp1 != '\0')) {
                    if ((*cp1 >= 'A') && (*cp1 <= 'Z'))
                        *cp1 += 'a' - 'A';
                    cp1++;
                }
                /* now do hostname */
                if ((*cp1 == '/') && (*(cp1 + 1) == '/')) {
                    cp1++;
                    cp1++;
                }
                while ((*cp1 != '/') && (*cp1 != '\0')) {
                    if ((*cp1 >= 'A') && (*cp1 <= 'Z'))
                        *cp1 += 'a' - 'A';
                    cp1++;
                }
            }

            /* if necessary, shrink referrer to fit storage */
            if (strlen(log_rec.refer) >= MAXREFH) {
                ERRVPRINT(VERBOSE1, "%s [%lu]\n", _("Warning: Truncating oversized referrer field"), total_rec);
                log_rec.refer[MAXREFH - 1] = '\0';
            }

            /* if necessary, shrink URL to fit storage */
            if (strlen(log_rec.url) >= MAXURLH) {
                ERRVPRINT(VERBOSE1, "%s [%lu]\n", _("Warning: Truncating oversized request field"), total_rec);
                log_rec.url[MAXURLH - 1] = '\0';
            }

            /* Do we need to mangle? */
            if (mangle_agent) {
                do_agent_mangling();
            }

            /* fix user agent field */
            cp1 = log_rec.agent;
            cp3 = cp2 = cp1++;
            if ((*cp2 != '\0') && ((*cp2 == '"') || (*cp2 == '('))) {
                while (*cp1 |= '\0') {
                    cp3 = cp2;
                    *cp2++ = *cp1++;
                }
                *cp3 = '\0';
            }
            cp1 = log_rec.agent;                /* CHANGE !!! */
            while (*cp1 != 0) {                 /* get rid of more common _bad_ chars ;)   */
                if ((*cp1 < 32) || (*cp1 == 127) || (*cp1 == '<') || (*cp1 == '>')) {
                    *cp1 = '\0';
                    break;
                } else
                    cp1++;
            }

            /* fix username if needed */
            if (log_rec.ident[0] == 0) {
                log_rec.ident[0] = '-';
                log_rec.ident[1] = '\0';
            } else {
                cp3 = log_rec.ident;
                while (*cp3 >= 32 && *cp3 != '"')
                    cp3++;
                *cp3 = '\0';
            }
            /* unescape user name */
            unescape(log_rec.ident);

            /********************************************
             * PROCESS RECORD                           *
             ********************************************/
            if (strncmp(str_previous_time, log_rec.datetime, SIZE_DATE_TIME) != 0) {
                /* strptime is $$$ - avoid if same date/time */
                memcpy(&prev_time_rec, &time_rec, sizeof(prev_time_rec));
                memset(&time_rec, 0, sizeof(time_rec));
                if (log_type == LOG_FTP) {
                    strptime(log_rec.datetime, DATE_TIME_XFERLOG_FORMAT, &time_rec);
                } else if (log_type == LOG_SQUID) {
                    temp_time_squid = strtoul(log_rec.datetime, NULL, 10);
                    localtime_r(&temp_time_squid, &time_rec);
                } else {
                    strptime(log_rec.datetime, DATE_TIME_FORMAT, &time_rec);
                }
                time_rec.tm_isdst = -1;         /* stop mktime from resetting for daylight savings */
                rec_tstamp = mktime(&time_rec);
                strcpy(str_previous_time, log_rec.datetime);
            }

            /* Do we need to check for duplicate records? (incremental mode)   */
            if (check_dup) {
                /* check if less than/equal to last record processed            */
                if (rec_tstamp <= cur_tstamp) {
                    /* if it is, assume we have already processed and ignore it  */
                    total_ignore++;
                    continue;
                } else {
                    /* if it isn't.. disable any more checks this run            */
                    check_dup = 0;
                    /* now check if it's a new month                             */
                    if (g_cur_month != (time_rec.tm_mon + 1)) {
                        clear_month();
                        g_cur_month = time_rec.tm_mon + 1;
                        g_cur_year = time_rec.tm_year + 1900;
                        g_cur_day = time_rec.tm_mday;
                        g_cur_hour = time_rec.tm_hour;
                        g_cur_min = time_rec.tm_min;
                        g_cur_sec = time_rec.tm_sec;
                        f_day = l_day = time_rec.tm_mday;       /* reset first and last day */
                        cur_tstamp = rec_tstamp;
                    }
                }
            }

            /* Ignore records that are too far incorrect of the previous timestamp - older records as in */
            if (rec_tstamp < cur_tstamp) {
                if (!fold_seq_err && ((rec_tstamp + SLOP_VAL) < cur_tstamp)) {
                    total_ignore++;
                    continue;
                } else {
                    rec_tstamp = cur_tstamp;
                    memcpy(&time_rec, &prev_time_rec, sizeof(prev_time_rec));
                }
            }
            cur_tstamp = rec_tstamp;            /* update current timestamp */

            /* first time through? */
            if (g_cur_month == 0) {
                /* if yes, init our date vars */
                g_cur_month = time_rec.tm_mon + 1;
                g_cur_year = time_rec.tm_year + 1900;
                g_cur_day = time_rec.tm_mday;
                g_cur_hour = time_rec.tm_hour;
                g_cur_min = time_rec.tm_min;
                g_cur_sec = time_rec.tm_sec;
                f_day = time_rec.tm_mday;
            }
            /* adjust last day processed if different */
            if (time_rec.tm_mday > l_day)
                l_day = time_rec.tm_mday;

            /* update min/sec stuff */
            if (g_cur_sec != time_rec.tm_sec)
                g_cur_sec = time_rec.tm_sec;
            if (g_cur_min != time_rec.tm_min)
                g_cur_min = time_rec.tm_min;

            /* check for hour change  */
            if (g_cur_hour != time_rec.tm_hour) {
                /* if yes, init hourly stuff */
                if (ht_hit > mh_hit)
                    mh_hit = ht_hit;
                ht_hit = 0;
                g_cur_hour = time_rec.tm_hour;
            }

            /* check for day change   */
            if (g_cur_day != time_rec.tm_mday) {
                /* if yes, init daily stuff */
                tm_site[g_cur_day - 1] = dt_site;
                dt_site = 0;
                tm_visit[g_cur_day - 1] = tot_visit(sd_htab);
                del_hlist(sd_htab);
                g_cur_day = time_rec.tm_mday;
            }

            /* check for month change */
            if (g_cur_month != (time_rec.tm_mon + 1)) {
                /* if yes, do monthly stuff */
                t_visit = tot_visit(sm_htab);
                month_update_exit(req_tstamp);  /* process exit pages      */
                write_month_html();             /* generate HTML for month */
                clear_month();
                g_cur_month = time_rec.tm_mon + 1;      /* update our flags        */
                g_cur_year = time_rec.tm_year + 1900;
                f_day = l_day = time_rec.tm_mday;
            }

            /* lowercase hostname */
            cp1 = log_rec.hostname;
            while (*cp1 != '\0') {
                if ((*cp1 >= 'A') && (*cp1 <= 'Z'))
                    *cp1 += 'a' - 'A';
                if ((isalnum((int) *cp1)) || (*cp1 == '.') || (*cp1 == '-'))
                    cp1++;
                else
                    *cp1 = '\0';
            }

            /* Catch blank hostnames here */
            if (log_rec.hostname[0] == '\0')
                strncpy(log_rec.hostname, "Unknown", 8);

            /* Ignore/Include check */
            if ((isinlist(include_sites, log_rec.hostname) == NULL) &&
                (isinlist(include_urls, log_rec.url) == NULL) && (isinlist(include_refs, log_rec.refer) == NULL) && (isinlist(include_agents, log_rec.agent) == NULL)
                && (isinlist(include_users, log_rec.ident) == NULL)) {
                if (isinlist(ignored_sites, log_rec.hostname) != NULL) {
                    total_ignore++;
                    VPRINT(VERBOSE5, "IGNORES: %s", buffer);
                    continue;
                }
                if (isinlist(ignored_urls, log_rec.url) != NULL) {
                    total_ignore++;
                    VPRINT(VERBOSE5, "IGNOREU: %s", buffer);
                    continue;
                }
                if (isinlist(ignored_agents, log_rec.agent) != NULL) {
                    total_ignore++;
                    VPRINT(VERBOSE5, "IGNOREA: %s", buffer);
                    continue;
                }
                if (isinlist(ignored_refs, log_rec.refer) != NULL) {
                    total_ignore++;
                    VPRINT(VERBOSE5, "IGNORER: %s", buffer);
                    continue;
                }
                if (isinlist(ignored_users, log_rec.ident) != NULL) {
                    total_ignore++;
                    VPRINT(VERBOSE5, "IGNOREZ: %s", buffer);
                    continue;
                }
            }
            VPRINT(VERBOSE4, "LOGLINE: %s", buffer);

            /* Bump response code totals */
            switch (log_rec.resp_code) {
            case RC_CONTINUE:
                i = IDX_CONTINUE;
                break;
            case RC_SWITCHPROTO:
                i = IDX_SWITCHPROTO;
                break;
            case RC_OK:
                i = IDX_OK;
                break;
            case RC_CREATED:
                i = IDX_CREATED;
                break;
            case RC_ACCEPTED:
                i = IDX_ACCEPTED;
                break;
            case RC_NONAUTHINFO:
                i = IDX_NONAUTHINFO;
                break;
            case RC_NOCONTENT:
                i = IDX_NOCONTENT;
                break;
            case RC_RESETCONTENT:
                i = IDX_RESETCONTENT;
                break;
            case RC_PARTIALCONTENT:
                i = IDX_PARTIALCONTENT;
                break;
            case RC_MULTIPLECHOICES:
                i = IDX_MULTIPLECHOICES;
                break;
            case RC_MOVEDPERM:
                i = IDX_MOVEDPERM;
                break;
            case RC_MOVEDTEMP:
                i = IDX_MOVEDTEMP;
                break;
            case RC_SEEOTHER:
                i = IDX_SEEOTHER;
                break;
            case RC_NOMOD:
                i = IDX_NOMOD;
                break;
            case RC_USEPROXY:
                i = IDX_USEPROXY;
                break;
            case RC_MOVEDTEMPORARILY:
                i = IDX_MOVEDTEMPORARILY;
                break;
            case RC_BAD:
                i = IDX_BAD;
                break;
            case RC_UNAUTH:
                i = IDX_UNAUTH;
                break;
            case RC_PAYMENTREQ:
                i = IDX_PAYMENTREQ;
                break;
            case RC_FORBIDDEN:
                i = IDX_FORBIDDEN;
                break;
            case RC_NOTFOUND:
                i = IDX_NOTFOUND;
                break;
            case RC_METHODNOTALLOWED:
                i = IDX_METHODNOTALLOWED;
                break;
            case RC_NOTACCEPTABLE:
                i = IDX_NOTACCEPTABLE;
                break;
            case RC_PROXYAUTHREQ:
                i = IDX_PROXYAUTHREQ;
                break;
            case RC_TIMEOUT:
                i = IDX_TIMEOUT;
                break;
            case RC_CONFLICT:
                i = IDX_CONFLICT;
                break;
            case RC_GONE:
                i = IDX_GONE;
                break;
            case RC_LENGTHREQ:
                i = IDX_LENGTHREQ;
                break;
            case RC_PREFAILED:
                i = IDX_PREFAILED;
                break;
            case RC_REQENTTOOLARGE:
                i = IDX_REQENTTOOLARGE;
                break;
            case RC_REQURITOOLARGE:
                i = IDX_REQURITOOLARGE;
                break;
            case RC_UNSUPMEDIATYPE:
                i = IDX_UNSUPMEDIATYPE;
                break;
            case RC_RNGNOTSATISFIABLE:
                i = IDX_RNGNOTSATISFIABLE;
                break;
            case RC_EXPECTATIONFAILED:
                i = IDX_EXPECTATIONFAILED;
                break;
            case RC_SERVERERR:
                i = IDX_SERVERERR;
                break;
            case RC_NOTIMPLEMENTED:
                i = IDX_NOTIMPLEMENTED;
                break;
            case RC_BADGATEWAY:
                i = IDX_BADGATEWAY;
                break;
            case RC_UNAVAIL:
                i = IDX_UNAVAIL;
                break;
            case RC_GATEWAYTIMEOUT:
                i = IDX_GATEWAYTIMEOUT;
                break;
            case RC_BADHTTPVER:
                i = IDX_BADHTTPVER;
                break;
            default:
                i = IDX_UNDEFINED;
                break;
            }
            response[i].count++;

            /* now save in the various hash tables... */
            if (log_rec.resp_code == RC_OK || log_rec.resp_code == RC_PARTIALCONTENT)
                i = 1;
            else
                i = 0;

            /* Pages (pageview) calculation */
            isapage = parse_is_page(log_rec.url);
            if (isapage) {
                t_page++;
                tm_page[time_rec.tm_mday - 1]++;
                th_page[time_rec.tm_hour]++;

                /* do search string stuff if needed     */
                if (ntop_search)
                    srch_string(log_rec.srchstr);
            } else {
                if ((log_type != LOG_FTP) && (log_type != LOG_SQUID)) {
                    if (parse_is_bookmark(log_rec.url, log_rec.agent)) {
                        t_bookmarks++;
                        tm_bookm[time_rec.tm_mday - 1]++;
                        th_bookm[time_rec.tm_hour]++;
                    }
                }
            }

            /* URL/ident hash table (only if valid response code) */
            if ((log_rec.resp_code == RC_OK) || (log_rec.resp_code == RC_NOMOD) || (log_rec.resp_code == RC_PARTIALCONTENT)) {
                /* URL hash table */
                if (put_unode(log_rec.url, OBJ_REG, (u_long) 1, log_rec.xfer_size, &t_url, (u_long) 0, (u_long) 0, um_htab)) {
                    /* Error adding URL node, skipping ... */
                    ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding URL node, skipping"), log_rec.url);
                }

                /* ident (username) hash table */
                if (put_inode(log_rec.ident, OBJ_REG, 1, (u_long) i, log_rec.xfer_size, &t_user, 0, rec_tstamp, im_htab, isapage)) {
                    /* Error adding ident node, skipping .... */
                    ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding Username node, skipping"), log_rec.ident);
                }
            }

            if (ntop_error) {
                if (log_rec.resp_code == RC_NOTFOUND) {
                    if (put_enode(log_rec.url, log_rec.refer ? log_rec.refer : _("Direct Request"), OBJ_REG, (u_long) 1, &t_errorp, ep_htab)) {
                        ERRVPRINT(VERBOSE1, _("Warning: Can't add %s with referer %s to errorpagehash"), log_rec.url, log_rec.refer);
                    }
                }
            }

            /* referrer hash table */
            if (ntop_refs) {
                if (log_rec.refer[0] != '\0' && isapage)
                    if (put_rnode(log_rec.refer, OBJ_REG, (u_long) 1, &t_ref, rm_htab)) {
                        ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding Referrer node, skipping"), log_rec.refer);
                    }
            }

            /* hostname (site) hash table - daily */
            if (put_hnode(log_rec.hostname, OBJ_REG, 1, (u_long) i, log_rec.xfer_size, &dt_site, 0, isapage, rec_tstamp, "", sd_htab, isapage)) {
                /* Error adding host node (daily), skipping .... */
                ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding host node (daily), skipping"), log_rec.hostname);
            }

            /* hostname (site) hash table - monthly */
            if (put_hnode(log_rec.hostname, OBJ_REG, 1, (u_long) i, log_rec.xfer_size, &t_site, 0, isapage, rec_tstamp, "", sm_htab, isapage)) {
                /* Error adding host node (monthly), skipping .... */
                ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding host node (monthly), skipping"), log_rec.hostname);
            }

            /* user agent hash table */
            if (ntop_agents) {
                if (log_rec.agent[0] != '\0' && isapage)
                    if (put_anode(log_rec.agent, OBJ_REG, (u_long) 1, &t_agent, am_htab)) {
                        ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding User Agent node, skipping"), log_rec.agent);
                    }
            }

            /* bump monthly/daily/hourly totals        */
            t_hit++;
            ht_hit++;                           /* daily/hourly hits    */
            t_xfer += log_rec.xfer_size;        /* total xfer size      */
            tm_xfer[time_rec.tm_mday - 1] += log_rec.xfer_size; /* daily xfer total     */
            tm_hit[time_rec.tm_mday - 1]++;     /* daily hits total     */
            th_xfer[time_rec.tm_hour] += log_rec.xfer_size;     /* hourly xfer total    */
            th_hit[time_rec.tm_hour]++;         /* hourly hits total    */

            /* if RC_OK, increase file counters */
            if (log_rec.resp_code == RC_OK) {
                t_file++;
                tm_file[time_rec.tm_mday - 1]++;
                th_file[time_rec.tm_hour]++;
            }

            /*********************************************
             * RECORD PROCESSED - DO GROUPS HERE         *
             *********************************************/

            /* URL Grouping */
            if ((cp1 = isinlist(group_urls, log_rec.url)) != NULL) {
                if (put_unode(cp1, OBJ_GRP, (u_long) 1, log_rec.xfer_size, &ul_bogus, (u_long) 0, (u_long) 0, um_htab)) {
                    /* Error adding URL node, skipping ... */
                    ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding URL node, skipping"), cp1);
                }
            }

            /* Site Grouping */
            if ((cp1 = isinlist(group_sites, log_rec.hostname)) != NULL) {
                if (put_hnode(cp1, OBJ_GRP, 1, (u_long) (log_rec.resp_code == RC_OK) ? 1 : 0, log_rec.xfer_size, &ul_bogus, 0, isapage, rec_tstamp, "", sm_htab, isapage)) {
                    /* Error adding Site node, skipping ... */
                    ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding host node (monthly), skipping"), cp1);
                }
            } else {
                /* Domain Grouping */
                if (group_domains) {
                    cp1 = get_domain(log_rec.hostname);
                    if (cp1 != NULL) {
                        if (put_hnode(cp1, OBJ_GRP, 1, (u_long) (log_rec.resp_code == RC_OK) ? 1 : 0, log_rec.xfer_size, &ul_bogus, 0, isapage, rec_tstamp, "", sm_htab, isapage)) {
                            /* Error adding Site node, skipping ... */
                            ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding host node (monthly), skipping"), cp1);
                        }
                    }
                }
            }

            /* Referrer Grouping */
            if ((cp1 = isinlist(group_refs, log_rec.refer)) != NULL && isapage) {
                if (put_rnode(cp1, OBJ_GRP, (u_long) 1, &ul_bogus, rm_htab)) {
                    /* Error adding Referrer node, skipping ... */
                    ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding Referrer node, skipping"), cp1);
                }
            }

            /* User Agent Grouping */
            if ((cp1 = isinlist(group_agents, log_rec.agent)) != NULL && isapage) {
                if (put_anode(cp1, OBJ_GRP, (u_long) 1, &ul_bogus, am_htab)) {
                    /* Error adding User Agent node, skipping ... */
                    ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding User Agent node, skipping"), cp1);
                }
            }

            /* Ident (username) Grouping */
            if ((cp1 = isinlist(group_users, log_rec.ident)) != NULL) {
                if (put_inode(cp1, OBJ_GRP, 1, (u_long) (log_rec.resp_code == RC_OK) ? 1 : 0, log_rec.xfer_size, &ul_bogus, 0, rec_tstamp, im_htab, isapage)) {
                    /* Error adding Username node, skipping ... */
                    ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding Username node, skipping"), cp1);
                }
            }
        }

        /*********************************************
         * BAD RECORD                                *
         *********************************************/

        else {
            /* If first record, check if stupid Netscape header stuff      */
            if ((total_rec == 1) && (strncmp(buffer, "format=", 7) == 0)) {
                /* Skipping Netscape header record */
                VPRINT(VERBOSE1, "%s\n", _("Skipping Netscape header record"));
                /* count it as ignored... */
                total_ignore++;
            } else {
                /* really bad record... */
                total_bad++;
                ERRVPRINT(VERBOSE1, "%s (%lu)\n", _("Skipping bad record"), total_rec);
                ERRVPRINT(VERBOSE3, "  ==> %s\n", tmp_buf);
            }
        }
    }                                           /* ---- END MAIN WHILE LOOP ---- */

    /*********************************************
     * DONE READING LOG FILE - final processing  *
     *********************************************/

    /* close log file if needed */
    if (gz_log)
        gzclose(gzlog_fp);
    else if (log_fname)
        fclose(log_fp);

    if (good_rec) {                             /* were any good records?   */
        tm_site[g_cur_day - 1] = dt_site;       /* If yes, clean up a bit   */
        tm_visit[g_cur_day - 1] = tot_visit(sd_htab);
        t_visit = tot_visit(sm_htab);
        if (ht_hit > mh_hit)
            mh_hit = ht_hit;

        if (total_rec > (total_ignore + total_bad)) {   /* did we process any?   */

            t_bad += total_bad;
            t_ignored += total_ignore;

            if (incremental) {
                if (save_state()) {             /* incremental stuff        */
                    /* Error: Unable to save current run data */
                    ERRVPRINT(VERBOSE1, "%s\n", _("Error: Unable to save current run data"));
                    unlink(state_fname);
                }
            }
            month_update_exit(rec_tstamp);      /* calculate exit pages     */
            write_month_html();                 /* write monthly HTML file  */
            write_main_index();                 /* write main HTML file     */
            put_history();                      /* write history            */
        }

        if (display_match_counts) {
            VPRINT(VERBOSE0, "%s\n", _("List Match Counts:"));
            show_matched(group_sites, "GroupSite");
            show_matched(group_urls, "GroupURL");
            show_matched(group_refs, "GroupReferrer");
            show_matched(group_agents, "GroupAgent");
            show_matched(group_users, "GroupUser");

//            show_matched(hidden_sites, "HideSite");
//            show_matched(hidden_urls, "HideURL");
//            show_matched(hidden_refs, "HideReferrer");
//            show_matched(hidden_agents, "HideAgent");
//            show_matched(hidden_users, "HideUser");
            show_matched(ignored_sites, "IgnoreSite");
            show_matched(ignored_urls, "IgnoreURL");
            show_matched(ignored_refs, "IgnoreReferrer");
            show_matched(ignored_agents, "IgnoreAgent");
            show_matched(ignored_users, "IgnoreUser");

            show_matched(include_sites, "IncludeSite");
            show_matched(include_urls, "IncludeURL");
            show_matched(include_refs, "IncludeReferrer");
            show_matched(include_agents, "IncludeAgent");
            show_matched(include_users, "IncludeUser");
//            show_matched(page_type, "");
//            show_matched(not_page_type, "");
            show_matched(search_list, "SearchEngine");
            show_matched(assign_country, "AssignToCountry");
        }

        end_time = times(&mytms);               /* display timing totals?   */
        if (time_me || (verbose > 1)) {
            printf("%lu %s ", total_rec, _("records"));
            if (total_ignore) {
                printf("(%lu %s", total_ignore, _("ignored"));
                if (total_bad)
                    printf(", %lu %s) ", total_bad, _("bad"));
                else
                    printf(") ");
            } else if (total_bad)
                printf("(%lu %s) ", total_bad, _("bad"));

            /* get processing time (end-start) */
            temp_time = (float) (end_time - start_time) / sysconf(_SC_CLK_TCK);
            printf("%s %.2f %s", _("in"), temp_time, _("seconds"));

            /* calculate records per second */
            if (temp_time)
                i = ((int) ((float) total_rec / temp_time));
            else
                i = 0;

            if ((i > 0) && (i <= total_rec))
                printf(_(", %d l/sec\n"), i);
            else
                printf("\n");
        }

        /* Whew, all done! Exit with completion status (0) */
        exit(0);
    } else {
        /* No valid records found... exit with error (1) */
        VPRINT(VERBOSE1, "%s\n", _("No valid records found!"));
        exit(1);
    }
}                                               /* ---- END OF MAIN ---- */

/*********************************************
 * GET_CONFIG - get configuration file info  *
 *********************************************/

void
get_config(const char *fname)
{
    const char *kwords[] = { "Undefined",       /* 0 = undefined keyword       0  */
        "OutputDir",                            /* Output directory            1  */
        "LogFile",                              /* Log file to use for input   2  */
        "ReportTitle",                          /* Title for reports           3  */
        "HostName",                             /* Hostname to use             4  */
        "IgnoreHist",                           /* Ignore history file         5  */
        "Quiet",                                /* Run in quiet mode           6  */
        "TimeMe",                               /* Produce timing results      7  */
        "Debug",                                /* Produce debug information   8  - Deprecated */
        "HourlyGraph",                          /* Hourly stats graph          9  */
        "HourlyStats",                          /* Hourly stats table         10  */
        "TopSites",                             /* Top sites                  11  */
        "TopURLs",                              /* Top URL's                  12  */
        "TopReferrers",                         /* Top Referrers              13  */
        "TopAgents",                            /* Top User Agents            14  */
        "TopCountries",                         /* Top Countries              15  */
        "HideSite",                             /* Sites to hide              16  */
        "HideURL",                              /* URL's to hide              17  */
        "HideReferrer",                         /* Referrers to hide          18  */
        "HideAgent",                            /* User Agents to hide        19  */
        "IndexAlias",                           /* Aliases for index.html     20  */
        "HTMLHead",                             /* HTML Top1 code             21  */
        "HTMLPost",                             /* HTML Top2 code             22  */
        "HTMLTail",                             /* HTML Tail code             23  */
        "MangleAgents",                         /* Mangle User Agents         24  */
        "IgnoreSite",                           /* Sites to ignore            25  */
        "IgnoreURL",                            /* Url's to ignore            26  */
        "IgnoreReferrer",                       /* Referrers to ignore        27  */
        "IgnoreAgent",                          /* User Agents to ignore      28  */
        "ReallyQuiet",                          /* Dont display ANY messages  29  */
        "GMTTime",                              /* Local or UTC time?         30  */
        "GroupURL",                             /* Group URL's                31  */
        "GroupSite",                            /* Group Sites                32  */
        "GroupReferrer",                        /* Group Referrers            33  */
        "GroupAgent",                           /* Group Agents               34  */
        "GroupShading",                         /* Shade Grouped entries      35  */
        "GroupHighlight",                       /* BOLD Grouped entries       36  */
        "Incremental",                          /* Incremental runs           37  */
        "IncrementalName",                      /* Filename for state data    38  */
        "HistoryName",                          /* Filename for history data  39  */
        "HTMLExtension",                        /* HTML filename extension    40  */
        "HTMLPre",                              /* HTML code at beginning     41  */
        "HTMLBody",                             /* HTML body code             42  */
        "HTMLEnd",                              /* HTML code at end           43  */
        "UseHTTPS",                             /* Use https:// on URL's      44  */
        "IncludeSite",                          /* Sites to always include    45  */
        "IncludeURL",                           /* URL's to always include    46  */
        "IncludeReferrer",                      /* Referrers to include       47  */
        "IncludeAgent",                         /* User Agents to include     48  */
        "PageType",                             /* Page Type (pageview)       49  */
        "VisitTimeout",                         /* Visit timeout (seconds)    50  */
        "GraphLegend",                          /* Graph Legends (yes/no)     51  */
        "GraphLines",                           /* Graph Lines (0=none)       52  */
        "FoldSeqErr",                           /* Fold sequence errors       53  */
        "CountryGraph",                         /* Display ctry graph (0=no)  54  */
        "TopKSites",                            /* Top sites (by KBytes)      55  */
        "TopKURLs",                             /* Top URL's (by KBytes)      56  */
        "TopEntry",                             /* Top Entry Pages            57  */
        "TopExit",                              /* Top Exit Pages             58  */
        "TopSearch",                            /* Top Search Strings         59  */
        "LogType",                              /* Log Type (clf/ftp/squid)   60  */
        "SearchEngine",                         /* SearchEngine strings       61  */
        "GroupDomains",                         /* Group domains (n=level)    62  */
        "HideAllSites",                         /* Hide ind. sites (0=no)     63  */
        "AllSites",                             /* List all sites?            64  */
        "AllURLs",                              /* List all URLs?             65  */
        "AllReferrers",                         /* List all Referrers?        66  */
        "AllAgents",                            /* List all User Agents?      67  */
        "AllSearchStr",                         /* List all Search Strings?   68  */
        "AllUsers",                             /* List all Users?            69  */
        "TopUsers",                             /* Top Usernames to show      70  */
        "HideUser",                             /* Usernames to hide          71  */
        "IgnoreUser",                           /* Usernames to ignore        72  */
        "IncludeUser",                          /* Usernames to include       73  */
        "GroupUser",                            /* Usernames to group         74  */
        "DumpPath",                             /* Path for dump files        75  */
        "DumpExtension",                        /* Dump filename extension    76  */
        "DumpHeader",                           /* Dump header as first rec?  77  */
        "DumpSites",                            /* Dump sites tab file        78  */
        "DumpURLs",                             /* Dump urls tab file         79  */
        "DumpReferrers",                        /* Dump referrers tab file    80  */
        "DumpAgents",                           /* Dump user agents tab file  81  */
        "DumpUsers",                            /* Dump usernames tab file    82  */
        "DumpSearchStr",                        /* Dump search str tab file   83  */
        "DNSCache",                             /* DNS Cache file name        84  */
        "DNSChildren",                          /* DNS Children (0=no DNS)    85  */
        "DailyGraph",                           /* Daily Graph (0=no)         86  */
        "DailyStats",                           /* Daily Stats (0=no)         87  */
        "GeoIP",                                /* Use GeoIP library (0=no)   88  */
        "GeoIPDatabase",                        /* GeoIP database             89  */
        "IndexMonths",                          /* Number Months on Main Page 90  */
        "GraphIndexX",                          /* Size of Main Graph X       91  */
        "GraphIndexY",                          /* Size of Main Graph Y       92  */
        "GraphDailyX",                          /* Size of Daily Graph X      93  */
        "GraphDailyY",                          /* Size of Daily Graph Y      94  */
        "GraphHourlyX",                         /* Size of Hourly Graph X     95  */
        "GraphHourlyY",                         /* Size of Hourly Graph Y     96  */
        "GraphPieX",                            /* Size of Pie Graph X        97  */
        "GraphPieY",                            /* Size of Pie Graph Y        98  */
        "TopURLsbyHitsGraph",                   /* Display Top URL's by Hits graph (0=no)    99  */
        "TopURLsbyVolGraph",                    /* Display Top URL's by Volume graph (0=no) 100  */
        "TopExitPagesGraph",                    /* Display Top Exit Pages Pie Chart         101  */
        "TopEntryPagesGraph",                   /* Display Top Entry Pages Pie Chart        102  */
        "TopSitesbyPagesGraph",                 /* Display TOP Sites by Volume Graph        103  */
        "TopSitesbyVolGraph",                   /* Display TOP Sites by Pages Graph         104  */
        "TopAgentsGraph",                       /* Display TOP Agents Graph (by Pages)      105  */
        "ColorHit",                             /* Hit Color   (def=00805c)   106  */
        "ColorFile",                            /* File Color  (def=0000ff)   107  */
        "ColorSite",                            /* Site Color  (def=ff8000)   108  */
        "ColorKbyte",                           /* Kbyte Color (def=ff0000)   109  */
        "ColorPage",                            /* Page Color  (def=00c0ff)   110  */
        "ColorVisit",                           /* Visit Color (def=ffff00)   111  */
        "ColorBookm",                           /* Bookm Color (def=ff00ff)   112  */
        "PieColor1",                            /* Pie Color 1 (def=800080)   113  */
        "PieColor2",                            /* Pie Color 2 (def=80ffc0)   114  */
        "PieColor3",                            /* Pie Color 3 (def=ff00ff)   115  */
        "PieColor4",                            /* Pie Color 4 (def=ffc480)   116  */
        "NotPageType",                          /* Opposite of PageType - specify what is NOT a page     117 */
        "Top404Errors",                         /* Display TOP 404 Errors     118  */
        "All404Errors",                         /* Display All 404 Errors     119  */
        "AssignToCountry",                      /* Assign this address to a country code    120 */
        "GroupAndHideAgent",                    /* Group & Hide Agents        121  */
        "GroupAndHideSite",                     /* Group & Hide Sites         122  */
        "GroupAndHideReferrer",                 /* Group & Hide Referrer      123 */
        "GroupAndHideURL",                      /* Group & Hide Referrer      124 */
        "GroupAndHideUser",                     /* Group & Hide User          125 */
        "DumpCountries",                        /* Dump countries tab file    126 */
        "DumpEntryPages",                       /* Dump Entry Pages tab file  127 */
        "DumpExitPages",                        /* Dump Exit Pages tab file   128 */
        "CSSFilename",                          /* CSS File filename          129 */
        "YearlySubtotals",                      /* Display Yearly Subtotals on main page    130 */
        "TrackPartialRequests",                 /* Track 206 Requests         131 */
        "FlagsLocation"                         /* Display Country Flags if not null        132 */
    };

    FILE *fp;

    char config_buffer[BUFSIZE];
    char keyword[32];
    char value[132];
    char *cp1, *cp2;
    int i, key, count;
    int num_kwords = sizeof(kwords) / sizeof(char *);

    if ((fp = fopen(fname, "r")) == NULL) {
        ERRVPRINT(VERBOSE1, "%s %s\n", _("Error: Unable to open configuration file"), fname);
        return;
    }

    VPRINT(VERBOSE1, "%s %s\n", _("Using config file:"), fname);

    while ((fgets(config_buffer, BUFSIZE, fp)) != NULL) {
        /* skip comments and blank lines */
        if ((config_buffer[0] == '#') || isspace((int) config_buffer[0]))
            continue;

        /* Get keyword */
        cp1 = config_buffer;
        cp2 = keyword;
        count = 31;
        while ((isalnum((int) *cp1)) && (count > 0)) {
            *cp2++ = *cp1++;
            count--;
        }
        *cp2 = '\0';

        /* Get value */
        cp2 = value;
        count = 131;
        while ((*cp1 != '\n') && (*cp1 != '\0') && (isspace((int) *cp1)))
            cp1++;
        while ((*cp1 != '\n') && (*cp1 != '\0') && (count > 0)) {
            *cp2++ = *cp1++;
            count--;
        }
        *cp2-- = '\0';
        while ((isspace((int) *cp2)) && (cp2 != value))
            *cp2-- = '\0';

        /* check if blank keyword/value */
        if ((keyword[0] == '\0') || (value[0] == '\0'))
            continue;

        key = 0;
        for (i = 0; i < num_kwords; i++)
            if (!strcmp(keyword, kwords[i])) {
                key = i;
                break;
            }

        if (key == 0) {                         /* Invalid keyword       */
            ERRVPRINT(VERBOSE1, "%s '%s' (%s)\n", _("Warning: Invalid keyword"), keyword, fname);
            continue;
        }

        VPRINT(VERBOSE5, "New Key: %d, Value: %s\n", key, value);

        switch (key) {
        case 1:
            out_dir = save_opt(value);
            break;                              /* OutputDir      */
        case 2:
            log_fname = save_opt(value);
            break;                              /* LogFile        */
        case 3:
            msg_title = save_opt(value);
            break;                              /* ReportTitle    */
        case 4:
            hname = save_opt(value);
            break;                              /* HostName       */
        case 5:
            ignore_hist = (value[0] == 'n') ? 0 : 1;
            break;                              /* IgnoreHist     */
        case 6:
            verbose = (value[0] == 'n') ? 2 : 1;
            break;                              /* Quiet          */
        case 7:
            time_me = (value[0] == 'n') ? 0 : 1;
            break;                              /* TimeMe         */
        case 8:
            ERRVPRINT(VERBOSE1, "%s\n", _("Use of \"Debug\" has been deprecated."));
            break;                              /* Debug          */
        case 9:
            hourly_graph = (value[0] == 'n') ? 0 : 1;
            break;                              /* HourlyGraph    */
        case 10:
            hourly_stats = (value[0] == 'n') ? 0 : 1;
            break;                              /* HourlyStats    */
        case 11:
            ntop_sites = atoi(value);
            break;                              /* TopSites       */
        case 12:
            ntop_urls = atoi(value);
            break;                              /* TopURLs        */
        case 13:
            ntop_refs = atoi(value);
            break;                              /* TopRefs        */
        case 14:
            ntop_agents = atoi(value);
            break;                              /* TopAgents      */
        case 15:
            ntop_ctrys = atoi(value);
            break;                              /* TopCountries   */
        case 16:
            add_list_member(value, &hidden_sites, USESPACE);
            break;                              /* HideSite       */
        case 17:
            add_list_member(value, &hidden_urls, USESPACE);
            break;                              /* HideURL        */
        case 18:
            add_list_member(value, &hidden_refs, USESPACE);
            break;                              /* HideReferrer   */
        case 19:
            add_list_member(value, &hidden_agents, USESPACE);
            break;                              /* HideAgent      */
        case 20:
            add_list_member(value, &index_alias, USESPACE);
            break;                              /* IndexAlias     */
        case 21:
            add_list_member(value, &html_head, IGNORESPACE);
            break;                              /* HTMLHead       */
        case 22:
            add_list_member(value, &html_post, IGNORESPACE);
            break;                              /* HTMLPost       */
        case 23:
            add_list_member(value, &html_tail, IGNORESPACE);
            break;                              /* HTMLTail       */
        case 24:
            mangle_agent = atoi(value);
            break;                              /* MangleAgents   */
        case 25:
            add_list_member(value, &ignored_sites, USESPACE);
            break;                              /* IgnoreSite     */
        case 26:
            add_list_member(value, &ignored_urls, USESPACE);
            break;                              /* IgnoreURL      */
        case 27:
            add_list_member(value, &ignored_refs, USESPACE);
            break;                              /* IgnoreReferrer */
        case 28:
            add_list_member(value, &ignored_agents, USESPACE);
            break;                              /* IgnoreAgent    */
        case 29:
            if (value[0] == 'y')
                verbose = 0;
            break;                              /* ReallyQuiet    */
        case 30:
            local_time = (value[0] == 'y') ? 0 : 1;
            break;                              /* GMTTime        */
        case 31:
            add_list_member(value, &group_urls, USESPACE);
            break;                              /* GroupURL       */
        case 32:
            add_list_member(value, &group_sites, USESPACE);
            break;                              /* GroupSite      */
        case 33:
            add_list_member(value, &group_refs, USESPACE);
            break;                              /* GroupReferrer  */
        case 34:
            add_list_member(value, &group_agents, USESPACE);
            break;                              /* GroupAgent     */
        case 35:
            shade_groups = (value[0] == 'y') ? 1 : 0;
            break;                              /* GroupShading   */
        case 36:
            hlite_groups = (value[0] == 'y') ? 1 : 0;
            break;                              /* GroupHighlight */
        case 37:
            incremental = (value[0] == 'y') ? 1 : 0;
            break;                              /* Incremental    */
        case 38:
            state_fname = save_opt(value);
            break;                              /* State FName    */
        case 39:
            hist_fname = save_opt(value);
            break;                              /* History FName  */
        case 40:
            html_ext = save_opt(value);
            break;                              /* HTML extension */
        case 41:
            add_list_member(value, &html_pre, IGNORESPACE);
            break;                              /* HTML Pre code  */
        case 42:
            add_list_member(value, &html_body, IGNORESPACE);
            break;                              /* HTML Body code */
        case 43:
            add_list_member(value, &html_end, IGNORESPACE);
            break;                              /* HTML End code  */
        case 44:
            use_https = (value[0] == 'y') ? 1 : 0;
            break;                              /* Use https://   */
        case 45:
            add_list_member(value, &include_sites, USESPACE);
            break;                              /* IncludeSite    */
        case 46:
            add_list_member(value, &include_urls, USESPACE);
            break;                              /* IncludeURL     */
        case 47:
            add_list_member(value, &include_refs, USESPACE);
            break;                              /* IncludeReferrer */
        case 48:
            add_list_member(value, &include_agents, USESPACE);
            break;                              /* IncludeAgent   */
        case 49:
            add_list_member(value, &page_type, USESPACE);
            break;                              /* PageType       */
        case 50:
            visit_timeout = atoi(value);
            break;                              /* VisitTimeout   */
        case 51:
            graph_legend = (value[0] == 'y') ? 1 : 0;
            break;                              /* GraphLegend    */
        case 52:
            graph_lines = atoi(value);
            break;                              /* GraphLines     */
        case 53:
            fold_seq_err = (value[0] == 'y') ? 1 : 0;
            break;                              /* FoldSeqErr     */
        case 54:
            ctry_graph = (value[0] == 'y') ? 1 : 0;
            break;                              /* CountryGraph   */
        case 55:
            ntop_sitesK = atoi(value);
            break;                              /* TopKSites (KB) */
        case 56:
            ntop_urlsK = atoi(value);
            break;                              /* TopKUrls (KB)  */
        case 57:
            ntop_entry = atoi(value);
            break;                              /* Top Entry pgs  */
        case 58:
            ntop_exit = atoi(value);
            break;                              /* Top Exit pages */
        case 59:
            ntop_search = atoi(value);
            break;                              /* Top Search pgs */
        case 60:
            force_log_type = true;
            if (strncmp(value, "auto", 4) == 0) {
                log_type = LOG_AUTO;
                force_log_type = false;
            } else if (strncmp(value, "clf", 3) == 0) {
                log_type = LOG_CLF;
            } else if (strncmp(value, "ftp", 3) == 0) {
                log_type = LOG_FTP;
            } else if (strncmp(value, "squid", 5) == 0) {
                log_type = LOG_SQUID;
            } else if (strncmp(value, "combined", 8) == 0) {
                log_type = LOG_COMBINED;
            } else if (strncmp(value, "domino", 6) == 0) {
                log_type = LOG_DOMINO;
            } else {
                ERRVPRINT(VERBOSE0, "%s %s\n", _("Unknown Log Type:"), value);
                exit(1);
            }
            break;                              /* LogType        */
        case 61:
            add_list_member(value, &search_list, USESPACE);
            break;                              /* SearchEngine   */
        case 62:
            group_domains = atoi(value);
            break;                              /* GroupDomains   */
        case 63:
            hide_sites = (value[0] == 'y') ? 1 : 0;
            break;                              /* HideAllSites   */
        case 64:
            all_sites = (value[0] == 'y') ? 1 : 0;
            break;                              /* All Sites?     */
        case 65:
            all_urls = (value[0] == 'y') ? 1 : 0;
            break;                              /* All URL's?     */
        case 66:
            all_refs = (value[0] == 'y') ? 1 : 0;
            break;                              /* All Refs       */
        case 67:
            all_agents = (value[0] == 'y') ? 1 : 0;
            break;                              /* All Agents?    */
        case 68:
            all_search = (value[0] == 'y') ? 1 : 0;
            break;                              /* All Srch str   */
        case 69:
            all_users = (value[0] == 'y') ? 1 : 0;
            break;                              /* All Users?     */
        case 70:
            ntop_users = atoi(value);
            break;                              /* TopUsers       */
        case 71:
            add_list_member(value, &hidden_users, USESPACE);
            break;                              /* HideUser       */
        case 72:
            add_list_member(value, &ignored_users, USESPACE);
            break;                              /* IgnoreUser     */
        case 73:
            add_list_member(value, &include_users, USESPACE);
            break;                              /* IncludeUser    */
        case 74:
            add_list_member(value, &group_users, USESPACE);
            break;                              /* GroupUser      */
        case 75:
            dump_path = save_opt(value);
            break;                              /* DumpPath       */
        case 76:
            dump_ext = save_opt(value);
            break;                              /* Dumpfile ext   */
        case 77:
            dump_header = (value[0] == 'y') ? 1 : 0;
            break;                              /* DumpHeader?    */
        case 78:
            dump_sites = (value[0] == 'y') ? 1 : 0;
            break;                              /* DumpSites?     */
        case 79:
            dump_urls = (value[0] == 'y') ? 1 : 0;
            break;                              /* DumpURLs?      */
        case 80:
            dump_refs = (value[0] == 'y') ? 1 : 0;
            break;                              /* DumpReferrers? */
        case 81:
            dump_agents = (value[0] == 'y') ? 1 : 0;
            break;                              /* DumpAgents?    */
        case 82:
            dump_users = (value[0] == 'y') ? 1 : 0;
            break;                              /* DumpUsers?     */
        case 83:
            dump_search = (value[0] == 'y') ? 1 : 0;
            break;                              /* DumpSrchStrs?  */
        case 84:                               /* Disable DNSCache and DNSChildren */
        case 85:
            ERRVPRINT(VERBOSE1, "%s '%s' (%s)\n", _("Warning: Invalid keyword"), keyword, fname);
            break;
        case 86:
            daily_graph = (value[0] == 'n') ? 0 : 1;
            break;                              /* HourlyGraph    */
        case 87:
            daily_stats = (value[0] == 'n') ? 0 : 1;
            break;                              /* HourlyStats    */
        case 88:
            use_geoip = (value[0] == 'y') ? true : false;
            break;                              /* Use GeoIP         */
        case 89:
            geoip_database = save_opt(value);
            break;                              /* GeoIP Database File */
        case 90:
            index_months = atoi(value);
            break;                              /* Months to Display */
        case 91:
            graph_index_x = atoi(value);
            break;                              /* Size of Main Graph X */
        case 92:
            graph_index_y = atoi(value);
            break;                              /* Size of Main Graph Y */
        case 93:
            graph_daily_x = atoi(value);
            break;                              /* Size of Daily Graph X        */
        case 94:
            graph_daily_y = atoi(value);
            break;                              /* Size of Daily Graph Y        */
        case 95:
            graph_hourly_x = atoi(value);
            break;                              /* Size of Hourly Graph X       */
        case 96:
            graph_hourly_y = atoi(value);
            break;                              /* Size of Hourly Graph Y       */
        case 97:
            graph_pie_x = atoi(value);
            break;                              /* Size of Pie Graph X  */
        case 98:
            graph_pie_y = atoi(value);
            break;                              /* Size of Pie Graph Y  */
        case 99:
            url_byhits_graph = (value[0] == 'y') ? 1 : 0;
            break;                              /* URLs by HITS Graph   */
        case 100:
            url_byvol_graph = (value[0] == 'y') ? 1 : 0;
            break;                              /* URLs by Volume Graph   */
        case 101:
            if (value[0] == 'h') {
                exit_pages_graph = 1;
            }
            if (value[0] == 'v') {
                exit_pages_graph = 2;
            }
            break;                              /* Top Exit Pages Pie Chart   */
        case 102:
            if (value[0] == 'h') {
                entry_pages_graph = 1;
            }
            if (value[0] == 'v') {
                entry_pages_graph = 2;
            }
            break;                              /* Top Entry Pages Pie Chart   */
        case 103:
            sites_bypages_graph = (value[0] == 'y') ? 1 : 0;
            break;                              /* TOP Sites by Pages Graph */
        case 104:
            sites_byvol_graph = (value[0] == 'y') ? 1 : 0;
            break;                              /* TOP Sites by Volume Graph */
        case 105:
            agents_graph = (value[0] == 'y') ? 1 : 0;
            break;                              /* TOP User Agents (by pages) Pie Chart */
        case 106:
            strncpy(hit_color + 1, value, 6);
            break;
        case 107:
            strncpy(file_color + 1, value, 6);
            break;
        case 108:
            strncpy(site_color + 1, value, 6);
            break;
        case 109:
            strncpy(kbyte_color + 1, value, 6);
            break;
        case 110:
            strncpy(page_color + 1, value, 6);
            break;
        case 111:
            strncpy(visit_color + 1, value, 6);
            break;
        case 112:
            strncpy(bookm_color + 1, value, 6);
            break;
        case 113:
            strncpy(pie_color1 + 1, value, 6);
            break;
        case 114:
            strncpy(pie_color2 + 1, value, 6);
            break;
        case 115:
            strncpy(pie_color3 + 1, value, 6);
            break;
        case 116:
            strncpy(pie_color4 + 1, value, 6);
            break;
        case 117:
            add_list_member(value, &not_page_type, USESPACE);
            break;                              /* NotPageType       */
        case 118:
            ntop_error = atoi(value);
            break;                              /* Top404Error       */
        case 119:
            all_errors = (value[0] == 'y') ? 1 : 0;
            break;                              /* All 404errors?     */
        case 120:
            add_list_member(value, &assign_country, USESPACE);
            break;                              /* Assign Address to Country */
        case 121:
            add_list_member(value, &hidden_agents, USESPACE);
            add_list_member(value, &group_agents, USESPACE);
            break;                              /* GroupAndHideAgent     */
        case 122:
            add_list_member(value, &hidden_sites, USESPACE);
            add_list_member(value, &group_sites, USESPACE);
            break;                              /* GroupAndHideSite      */
        case 123:
            add_list_member(value, &hidden_refs, USESPACE);
            add_list_member(value, &group_refs, USESPACE);
            break;                              /* GroupAndHideReferrer  */
        case 124:
            add_list_member(value, &hidden_urls, USESPACE);
            add_list_member(value, &group_urls, USESPACE);
            break;                              /* GroupAndHideURL       */
        case 125:
            add_list_member(value, &hidden_users, USESPACE);
            add_list_member(value, &group_users, USESPACE);
            break;                              /* GroupAndHideUser      */
        case 126:
            dump_countries = (value[0] == 'y') ? true : false;
            break;                              /* DumpCountries?        */
        case 127:
            dump_entry_pages = (value[0] == 'y') ? true : false;
            break;                              /* DumpEntryPages?       */
        case 128:
            dump_exit_pages = (value[0] == 'y') ? true : false;
            break;                              /* DumpExitPages?        */
        case 129:
            css_filename = save_opt(value);
            break;                              /* CSS file filename     */
        case 130:
            display_yearly_subtotals = (value[0] == 'y') ? true : false;
            break;                              /* DumpExitPages?        */
        case 131:
            track_206_reqs = (value[0] == 'y') ? true : false;
            break;                              /* TrackPartialRequests? */
        case 132:
            flags_location = save_opt(value);
            break;                              /*  FlagsLocation        */
        }
    }
    if ((page_type != NULL) && (not_page_type != NULL)) {
        ERRVPRINT(VERBOSE0, "%s\n", _("FATAL! You may not specify both PageType and NotPageType in the config file.%s"));
        exit(1);
    }

    fclose(fp);
}


/********************************************
 * option_checks                            *
 *                                          *
 * Check various options for funky stuff.   *
 * Alert/Exit as necessary.                 *
 ********************************************/
void
option_checks()
{
    int max_ctry;                               /* max countries defined       */
    int i;

    /* Be polite and announce yourself... */
    uname(&system_info);
    VPRINT(VERBOSE1, "%s (%s %s) %s\n", PACKAGE_STRING, system_info.sysname, system_info.release, _("English"));

    /* GEOIP Checks */
#if HAVE_GEOIP_H
    if (use_geoip) {
        have_geoip = true;
        gi = GeoIP_open(geoip_database, GEOIP_MEMORY_CACHE);
        if (gi == NULL) {
            ERRVPRINT(VERBOSE0, "%s: %s\n", _("FATAL. Unable to open the GeoIP database"), geoip_database);
            exit(1);
        }
        VPRINT(VERBOSE1, "%s: %s\n", _("Using GeoIP for IP Address Lookups"), geoip_database);
    }
#endif
    if (use_geoip && !have_geoip) {
        VPRINT(VERBOSE1, "%s\n", _("GeoIP is not available in this binary. Ignoring request to use."));
    }

    if (page_type == NULL) {                    /* check if page types present     */
        if ((log_type == LOG_AUTO) || (log_type == LOG_CLF) || (log_type == LOG_COMBINED) || (log_type == LOG_SQUID)) {
            add_list_member("htm", &page_type, USESPACE);       /* if no page types specified, we  */
            add_list_member("html", &page_type, USESPACE);      /* if no page types specified, we  */
            add_list_member("cgi", &page_type, USESPACE);       /* use the default ones here...    */
            if (!isinlist(page_type, (char *) html_ext))
                add_list_member((char *) html_ext, &page_type, USESPACE);
        } else
            add_list_member("txt", &page_type, USESPACE);       /* FTP logs default to .txt        */
    }

    for (max_ctry = 0; ctry[max_ctry].desc; max_ctry++);
    if (ntop_ctrys > max_ctry) {
        ntop_ctrys = max_ctry;                  /* force upper limit */
    }
    if (graph_lines > 20) {
        graph_lines = 20;                       /* keep graphs sane! */
    }

    if (log_type == LOG_FTP) {
        /* disable stuff for ftp logs */
        ntop_entry = ntop_exit = 0;
        ntop_search = 0;
    } else {
        if (search_list == NULL) {
            /* If no search engines defined, define some :) */
            add_list_member("google.        q=", &search_list, USESPACE);
            add_list_member("yahoo.         p=", &search_list, USESPACE);
            add_list_member("msn.           q=", &search_list, USESPACE);
            add_list_member("search.aol.    query=", &search_list, USESPACE);
            add_list_member("altavista.com  q=", &search_list, USESPACE);
            add_list_member("netscape.com   query=", &search_list, USESPACE);
            add_list_member("ask.com        q=", &search_list, USESPACE);
            add_list_member("alltheweb.com  query=", &search_list, USESPACE);
            add_list_member("lycos.com      query=", &search_list, USESPACE);
            add_list_member("hotbot.        query=", &search_list, USESPACE);
            add_list_member("mamma.com      query=", &search_list, USESPACE);
            add_list_member("search.        q=", &search_list, USESPACE);       /* Generic Catchall... */
        }
    }

    /* ensure entry/exits don't exceed urls */
    i = (ntop_urls > ntop_urlsK) ? ntop_urls : ntop_urlsK;
    if (ntop_entry > i)
        ntop_entry = i;
    if (ntop_exit > i)
        ntop_exit = i;

}

/*********************************************/
/* SAVE_OPT - save option from config file   */
/*********************************************/

static char *
save_opt(char *str)
{
    char *cp1;

    if ((cp1 = malloc(strlen(str) + 1)) == NULL)
        return NULL;

    strcpy(cp1, str);
    return cp1;
}

/*********************************************/
/* CLEAR_MONTH - initalize monthly stuff     */
/*********************************************/

void
clear_month(void)
{
    int i;

    init_counters();                            /* reset monthly counters  */
    del_htabs();                                /* clear hash tables       */
    if (ntop_ctrys != 0)
        for (i = 0; i < ntop_ctrys; i++)
            top_ctrys[i] = NULL;
}

/*********************************************/
/* INIT_COUNTERS - prep counters for use     */
/*********************************************/

void
init_counters(void)
{
    int i;

    for (i = 0; i < TOTAL_RC; i++)
        response[i].count = 0;
    for (i = 0; i < 31; i++) {                  /* monthly totals      */
        tm_xfer[i] = 0;
        tm_hit[i] = 0;
        tm_file[i] = 0;
        tm_site[i] = 0;
        tm_page[i] = 0;
        tm_visit[i] = 0;
        tm_bookm[i] = 0;
    }
    for (i = 0; i < 24; i++) {                  /* hourly totals       */
        th_hit[i] = 0;
        th_file[i] = 0;
        th_page[i] = 0;
        th_xfer[i] = 0;
        th_bookm[i] = 0;
    }
    for (i = 0; ctry[i].desc; i++) {            /* country totals */
        ctry[i].count = 0;
        ctry[i].files = 0;
        ctry[i].xfer = 0;
        ctry[i].pages = 0;
    }
    t_hit = 0;
    t_file = 0;
    t_site = 0;
    t_url = 0;
    t_ref = 0;
    t_agent = 0;
    t_page = 0;
    t_visit = 0;
    t_user = 0;
    t_xfer = 0;
    t_bookmarks = 0;
    t_bad = 0;
    t_ignored = 0;
    mh_hit = dt_site = 0;
    f_day = l_day = 1;
}

/*********************************************/
/* CURRENT_TIME - return date/time as a string   */
/*********************************************/

char *
current_time(void)
{
    /* get system time */
    now = time(NULL);
    /* convert to timestamp string */
    if (local_time) {
        strftime(timestamp, sizeof(timestamp), "%d-%b-%Y %H:%M %Z", localtime(&now));
    } else {
        strftime(timestamp, sizeof(timestamp), "%d-%b-%Y %H:%M GMT", gmtime(&now));
    }

    return timestamp;
}

/*********************************************/
/* ISURLCHAR - checks for valid URL chars    */
/*********************************************/

int
isurlchar(unsigned char ch)
{
    if (isalnum((int) ch))
        return 1;                               /* allow letters, numbers...    */
    if (ch > 127)
        return 1;                               /* allow extended chars...      */
    return (strchr(":/\\.,' *-+_@~()[]", ch) != NULL);  /* and a few special ones */
}

/*********************************************/
/* CTRY_IDX - create unique # from domain    */
/*********************************************/

u_long
ctry_idx(char *str)
{
    int i = strlen(str), j = 0;
    u_long idx = 0;
    char *cp1 = str + i;

    for (; i > 0; i--) {
        idx += ((*--cp1 - 'a' + 1) << j);
        j += 5;
    }
    return idx;
}

/*********************************************/
/* FROM_HEX - convert hex char to decimal    */
/*********************************************/

char
from_hex(char c)
{                                               /* convert hex to dec      */
    c = (c >= '0' && c <= '9') ? c - '0' :      /* 0-9?                    */
        (c >= 'A' && c <= 'F') ? c - 'A' + 10 : /* A-F?                    */
        c - 'a' + 10;                           /* lowercase...            */
    return (c < 0 || c > 15) ? 0 : c;           /* return 0 if bad...      */
}

/*********************************************/
/* UNESCAPE - convert escape seqs to chars   */
/*********************************************/

char *
unescape(char *str)
{
    unsigned char *cp1 = (unsigned char *) str; /* force unsigned so we    */
    unsigned char *cp2 = (unsigned char *) str; /* can do > 127            */

    if (!str)
        return NULL;                            /* make sure strings valid */

    while (*cp1) {
        if (*cp1 == '%') {                      /* Found an escape?        */
            cp1++;
            if (isxdigit(*cp1)) {               /* ensure a hex digit      */
                if (*cp1)
                    *cp2 = from_hex(*cp1++) * 16;       /* convert hex to an ascii */
                if (*cp1)
                    *cp2 += from_hex(*cp1);     /* (hopefully) character   */
                if ((*cp2 < 32) || (*cp2 == 127))
                    *cp2 = '_';                 /* make '_' if its bad   */
                if (*cp1)
                    cp2++;
                cp1++;
            } else
                *cp2++ = '%';
        } else
            *cp2++ = *cp1++;                    /* if not, just continue   */
    }
    *cp2 = *cp1;                                /* don't forget terminator */
    return str;                                 /* return the string       */
}

/*********************************************/
/* SRCH_STRING - get search strings from ref */
/*********************************************/
void
srch_string(char *ptr)
{
    /* ptr should point to unescaped query string */
    char tmpbuf[BUFSIZE];
    char srch[80] = "";
    char *cp1, *cp2, *cps;
    int sp_flg = 0;

    /* Check if search engine referrer or return  */
    if ((cps = isinlist(search_list, log_rec.refer)) == NULL)
        return;

    /* Try to find query variable */
    srch[0] = '?';
    srch[sizeof(srch) - 1] = '\0';
    strcpy(&srch[1], cps);                      /* First, try "?..."      */
    if ((cp1 = strstr(ptr, srch)) == NULL) {
        srch[0] = '&';                          /* Next, try "&..."       */
        if ((cp1 = strstr(ptr, srch)) == NULL)
            return;                             /* If not found, split... */
    }
    cp2 = tmpbuf;

    while (*cp1 != '=' && *cp1 != '\0') {
        cp1++;
    }
    if (*cp1 != '\0') {
        cp1++;
    }

    while (*cp1 != '&' && *cp1 != '\0') {
        if (*cp1 == '"' || *cp1 == ',' || *cp1 == '?') {
            cp1++;
            continue;
        } /* skip bad ones..    */
        else {
            if (*cp1 == '+')
                *cp1 = ' ';                     /* change + to space  */
            if (sp_flg && *cp1 == ' ') {
                cp1++;
                continue;
            }                                   /* compress spaces    */
            if (*cp1 == ' ')
                sp_flg = 1;
            else
                sp_flg = 0;                     /* (flag spaces here) */
            *cp2++ = tolower(*cp1);             /* normal character   */
            cp1++;
        }
    }
    *cp2 = '\0';
    cp2 = tmpbuf;
    if (tmpbuf[0] == '?')
        tmpbuf[0] = ' ';                        /* format fix ?       */
    while (*cp2 != 0 && isspace(*cp2))
        cp2++;                                  /* skip leading sps.  */
    if (*cp2 == '\0')
        return;

    /* any trailing spaces? */
    cp1 = cp2 + strlen(cp2) - 1;
    while (cp1 != cp2)
        if (isspace(*cp1))
            *cp1-- = '\0';
        else
            break;

    /* strip invalid chars */
    cp1 = cp2;
    while (*cp1 != '\0') {
        if (((*cp1 > 0) && (*cp1 < 32)) || (*cp1 == 127))
            *cp1 = '_';
        cp1++;
    }

    if (put_snode(cp2, (u_long) 1, sr_htab)) {
        /* Error adding search string node, skipping .... */
        ERRVPRINT(VERBOSE1, "%s %s\n", _("Error adding Search String Node, skipping"), tmpbuf);
    }
    return;
}

/*********************************************/
/* GET_DOMAIN - Get domain portion of host   */
/*********************************************/

char *
get_domain(char *str)
{
    char *cp;
    int i = group_domains + 1;

    cp = str + strlen(str) - 1;
    if (isdigit((int) *cp))
        return NULL;                            /* ignore IP addresses */

    while (cp != str) {
        if (*cp == '.')
            if (!(--i))
                return ++cp;
        cp--;
    }
    return cp;
}

/*********************************************/
/* OUR_GZGETS - enhanced gzgets for log only */
/*********************************************/

char *
our_gzgets(gzFile fp, char *buf, int size)
{
    char *out_cp = buf;                         /* point to output */

    while (1) {
        if (f_cp > (f_buf + f_end - 1)) {       /* load? */
            f_end = gzread(fp, f_buf, GZ_BUFSIZE);
            if (f_end <= 0)
                return Z_NULL;
            f_cp = f_buf;
        }

        if (--size) {                           /* more? */
            *out_cp++ = *f_cp;
            if (*f_cp++ == '\n') {
                *out_cp = '\0';
                return buf;
            }
        } else {
            *out_cp = '\0';
            return buf;
        }
    }
}

/*****************************************************************/
/*                                                               */
/* JDATE  - Julian date calculator                               */
/*                                                               */
/* Calculates the number of days since Jan 1, 0000.              */
/*                                                               */
/* Originally written by Bradford L. Barrett (03/17/1988)        */
/* Returns an unsigned long value representing the number of     */
/* days since January 1, 0000.                                   */
/*                                                               */
/* Note: Due to the changes made by Pope Gregory XIII in the     */
/*       16th Centyry (Feb 24, 1582), dates before 1583 will     */
/*       not return a truely accurate number (will be at least   */
/*       10 days off).  Somehow, I don't think this will         */
/*       present much of a problem for most situations :)        */
/*                                                               */
/* Usage: days = jdate(day, month, year)                         */
/*                                                               */
/* The number returned is adjusted by 5 to facilitate day of     */
/* week calculations.  The mod of the returned value gives the   */
/* day of the week the date is.  (ie: dow = days % 7 ) where     */
/* dow will return 0=Sunday, 1=Monday, 2=Tuesday, etc...         */
/*                                                               */
/*****************************************************************/

u_long
jdate(int day, int month, int year)
{
    u_long days;                                /* value returned */
    int mtable[] = { 0, 31, 59, 90, 120, 151, 181, 212, 243, 273, 304, 334 };

    /* First, calculate base number including leap and Centenial year stuff */

    days = (((u_long) year * 365) + day + mtable[month - 1] + ((year + 4) / 4) - ((year / 100) - (year / 400)));

    /* now adjust for leap year before March 1st */

    if ((year % 4 == 0) && !((year % 100 == 0) && (year % 400 != 0)) && (month < 3))
        --days;

    /* done, return with calculated value */

    return (days + 5);
}

int
do_agent_mangling(void)
{
    char *cp1, *cp2;                            /* generic char pointers       */
    char *agent_start;                          /* Start and End of the Agent string - mainly bounds checking */
    char *agent_end;

    agent_start = cp2 = log_rec.agent;
    agent_end = log_rec.agent + sizeof(log_rec.agent) - 2;

    cp1 = strstr(agent_start, "ompatible");     /* check known fakers */
    if (cp1 != NULL) {
        while (*cp1 != ';' && *cp1 != '\0' && (cp1 < agent_end)) {
            cp1++;
        }
        /* kludge for Mozilla/3.01 (compatible;) */
        if (*cp1++ == ';' && strcmp(cp1, ")\"")) {      /* success! */
            while (*cp1 == ' ' && (cp1 < agent_end)) {
                cp1++;                          /* eat spaces */
            }
            while (*cp1 != '.' && *cp1 != '\0' && *cp1 != ';' && (cp1 < agent_end) && (cp2 < agent_end)) {
                *cp2++ = *cp1++;
            }
            if (mangle_agent < 5) {
                while (*cp1 != '.' && *cp1 != ';' && *cp1 != '\0' && (cp1 < agent_end) && (cp2 < agent_end)) {
                    *cp2++ = *cp1++;
                }
                if (*cp1 != ';' && *cp1 != '\0' && (cp1 < agent_end - 2) && (cp2 < agent_end - 2)) {
                    *cp2++ = *cp1++;
                    *cp2++ = *cp1++;
                }
            }
            if (mangle_agent < 4) {
                if (*cp1 >= '0' && *cp1 <= '9' && (cp1 < agent_end) && (cp2 < agent_end)) {
                    *cp2++ = *cp1++;
                }
            }
            if (mangle_agent < 3) {
                while (*cp1 != ';' && *cp1 != '\0' && *cp1 != '(' && (cp1 < agent_end) && (cp2 < agent_end)) {
                    *cp2++ = *cp1++;
                }
            }
            if (mangle_agent < 2) {
                /* Level 1 - try to get OS */
                cp1 = strstr(agent_start, ")");
                if (cp1 != NULL) {
                    *cp2++ = ' ';
                    *cp2++ = '(';
                    while (*cp1 != ';' && *cp1 != '(' && cp1 > agent_start) {
                        cp1--;
                    }
                    if (cp1 != agent_start && *cp1 != '\0' && (cp1 < agent_end)) {
                        cp1++;
                    }
                    while (*cp1 == ' ' && *cp1 != '\0' && (cp1 < agent_end)) {
                        cp1++;
                    }
                    while (*cp1 != ')' && *cp1 != '\0' && cp1 > cp2 && (cp1 < agent_end) && (cp2 < agent_end)) {
                        *cp2++ = *cp1++;
                    }
                    *cp2++ = ')';
                }
            }
            *cp2 = '\0';
        } else {                                /* nothing after "compatible", should we mangle? */
            /* not for now */
        }
    } else {
        cp1 = strstr(agent_start, "Opera");     /* Opera flavor         */
        if (cp1 != NULL) {
            while (*cp1 != '/' && *cp1 != ' ' && *cp1 != '\0' && (cp1 < agent_end) && (cp2 < agent_end)) {
                *cp2++ = *cp1++;
            }
            while (*cp1 != '.' && *cp1 != '\0' && (cp1 < agent_end) && (cp2 < agent_end)) {
                *cp2++ = *cp1++;
            }
            if (mangle_agent < 5) {
                while (*cp1 != '.' && *cp1 != '\0' && (cp1 < agent_end - 2) && (cp2 < agent_end - 2)) {
                    *cp2++ = *cp1++;
                }
                *cp2++ = *cp1++;
                *cp2++ = *cp1++;
            }
            if (mangle_agent < 4)
                if (*cp1 >= '0' && *cp1 <= '9' && (cp1 < agent_end) && (cp2 < agent_end)) {
                    *cp2++ = *cp1++;
                }
            if (mangle_agent < 3)
                while (*cp1 != ' ' && *cp1 != '\0' && *cp1 != '(' && (cp1 < agent_end) && (cp2 < agent_end)) {
                    *cp2++ = *cp1++;
                }
            if (mangle_agent < 2) {
                cp1 = strstr(agent_start, "(");
                if (cp1 != NULL) {
                    cp1++;
                    *cp2++ = ' ';
                    *cp2++ = '(';
                    while (*cp1 != ';' && *cp1 != ')' && *cp1 != '\0' && (cp1 < agent_end) && (cp2 < agent_end - 1)) {
                        *cp2++ = *cp1++;
                    }
                    *cp2++ = ')';
                }
            }
            *cp2 = '\0';
        } else {
            cp1 = strstr(agent_start, "Mozilla");       /* Netscape flavor      */
            if (cp1 != NULL) {
                while (*cp1 != '/' && *cp1 != ' ' && *cp1 != '\0' && (cp1 < agent_end) && (cp2 < agent_end)) {
                    *cp2++ = *cp1++;
                }
                if (*cp1 == ' ') {
                    *cp1 = '/';
                }
                while (*cp1 != '.' && *cp1 != '\0' && (cp1 < agent_end) && (cp2 < agent_end)) {
                    *cp2++ = *cp1++;
                }
                if (mangle_agent < 5) {
                    while (*cp1 != '.' && *cp1 != '\0' && (cp1 < agent_end - 2) && (cp2 < agent_end - 2)) {
                        *cp2++ = *cp1++;
                    }
                    *cp2++ = *cp1++;
                    *cp2++ = *cp1++;
                }
                if (mangle_agent < 4)
                    if (*cp1 >= '0' && *cp1 <= '9') {
                        *cp2++ = *cp1++;
                    }
                if (mangle_agent < 3) {
                    while (*cp1 != ' ' && *cp1 != '\0' && *cp1 != '(' && (cp1 < agent_end) && (cp2 < agent_end)) {
                        *cp2++ = *cp1++;
                    }
                }
                if (mangle_agent < 2) {
                    /* Level 1 - Try to get OS */
                    cp1 = strstr(agent_start, "(");
                    if (cp1 != NULL) {
                        cp1++;
                        *cp2++ = ' ';
                        *cp2++ = '(';
                        while (*cp1 != ';' && *cp1 != ')' && *cp1 != '\0' && (cp1 < agent_end) && (cp2 < agent_end - 1)) {
                            *cp2++ = *cp1++;
                        }
                        *cp2++ = ')';
                    }
                }
                *cp2 = '\0';
            }
        }
    }
    return (0);
}
