---------------------------- sit_shell.as -----------------------------
--
-- Shells, i.e. read-eval loops.
--
-- Copyright (c) Manuel Bronstein 1996
-- Copyright (c) INRIA 1999, Version 0.1.12
-- Logiciel Sum^it (c) INRIA 1999, dans sa version 0.1.12
-- Copyright (c) Swiss Federal Polytechnic Institute Zurich, 1996-97
-----------------------------------------------------------------------------

#include "algebra"

macro {
	B	== Boolean;
	Z	== MachineInteger;
	TX	== TextWriter;
	TAB	== SymbolTable;
	NL	== newline;
}

#if ALDOC
\thistype{Shell}
\History{Manuel Bronstein}{10/01/96}{created}
\Usage{import from \this(P, R, E)}
\Params{
P & \altype{Parser} & A parser type for the input\\
R & \altype{PartialRing} & Resulting type of the evaluation\\
E & \altype{Evaluator} R & An interpreter with target type R\\
}
\Descr{\this(P, R, E) provides a basic read-eval-loop that converts its input
to expressions of type R.}
\begin{exports}
\alexp{center}: & (TW, \altype{String}) $\to$ TW & center a line of text\\
\alexp{shell!}: &
(P, TW, TW, \altype{SymbolTable} R, \altype{Boolean}, \altype{Boolean},
\altype{String}) $\to$ Z &
Read-eval loop\\
\end{exports}
\begin{aswhere}
TW &==& \altype{TextWriter}\\
Z &==& \altype{MachineInteger}\\
\end{aswhere}
#endif

Shell(P:Parser, R:PartialRing, E:Evaluator R): with {
	center: (TX, String) -> TX;
#if ALDOC
\alpage{center}
\Usage{\name(p, s)}
\Signature{(\altype{TextWriter}, \altype{String})}{\altype{TextWriter}}
\Params{
{\em p} & \altype{TextWriter} & The port to write to\\
{\em s} & \altype{String} & A string to center\\
}
\Descr{Writes {\em s} centered in a 66-character line to the port {\em p},
followed by a newline, and returns the port after the write.}
#endif
	shell!: (P, TX, TX, TAB R, B, B, String) -> Z;
#if ALDOC
\alpage{shell!}
\Usage{\name(r, $p_1, p_2$, t, verbose?, time?, s)}
\Signature{(P, TW, TW, \altype{SymbolTable} R,
\altype{Boolean}, \altype{Boolean}, \altype{String})}{\altype{MachineInteger}}
\begin{alwhere}
TW &==& \altype{TextWriter}\\
\end{alwhere}
\Params{
{\em r} & P & A parser\\
$p_1,p_2$ & \altype{TextWriter} & The ports to write to\\
{\em t} & \altype{SymbolTable} R & The initial symbol table\\
{\em verbose?} & \altype{Boolean} & Select verbose or quiet mode\\
{\em time?} & \altype{Boolean} & Select whether to return times in msecs\\
{\em s} & \altype{String} & A string to write after processing commands\\
}
\Descr{Starts a read-eval loop which takes its input from r and writes
its output to $p_1$ or $p_2$. Mathematical output generated by user commands
is written to $p_1$, while prompts and execution times are written to $p_2$.
if verbose? is \false{} (quiet mode), nothing is printed to $p_2$ and no
prompts or execution times are communicated.
Regardless of verbose?,
if time? is \true, then the execution and garbage collection times
are written to $p_1$ in milliseconds after each command.
If s is not empty, then
it is sent to $p_1$ after each command is executed.
The table t contains the initial environment, and can be modified by the loop.
Returns an integer $n = b_1 b_0$ where $b_0$ is 1 if a syntax error occured,
0 otherwise, and $b_1$ is 1 if any other error occured, 0 otherwise.}
#endif
} == add {
	local prompt(p:TX, n:Z):TX == {
		import from Character, WriterManipulator, String;
		p << newline << n << " --> " << flush;
	}

	local err1(p:TX, msg:String, verbose?:B):() == {
		import from WriterManipulator, String;
		if verbose? then p << "*** " << msg << " ***" << endnl;
	}

	-- title is automatically centered in a 66-character field
	center(p:TX, title:String):TX == {
		import from Z, Character;
		n := max(0, 66 - #title);	-- total # of spaces to add
		n2 := n quo 2;
		left := new(n2, space);
		right := new(n - n2, space);
		p << left << title << right << NL;
	}

	-- return true if an error has occured
	shell!(pin:P,pout:TX,ptime:TX,t:TAB R,verb?:B,time?:B,done:String):Z=={
		import from Z,String, WriterManipulator, Partial ExpressionTree;
		err:Z := 0;
		count:Z := 1;
		TRACE("shell::shell!: err = ", err);
		TRACE("shell::shell!: count = ", count);
		if verb? then prompt(ptime, count);
		done? := ~empty? done;
		while ~eof? pin repeat {
			if failed?(tree := parse! pin) then {
				err := err \/ 1;
				err1(pout, "syntax error", verb? and ~eof? pin);
			}
			else {
				if process(pout, ptime, retract tree, t, verb?,
						time?) then err := err \/ 2;
				else if done? then pout << done << endnl;
			}
			count := next count;
			if verb? then prompt(ptime, count);
		}
		if verb? then ptime << endnl;
		err;
	}

	-- return true if an error has occured
	local process(p:TX, ptime:TX, tree:ExpressionTree, t:TAB R,
		verbose?:B, time?:B):B == {
			import from Z, E, R, Partial R, Timer, String;
			import from Character, WriterManipulator;
			clock := timer();
			start! clock;
			result := eval!(p, tree, t);
			msecs := stop! clock;
			if (err? := failed? result) then
				err1(p, "failed to evaluate", true);
			if time? then p << msecs << newline << gc clock <<endnl;
			if verbose? then {
				ptime := printTime(ptime, msecs);
				ptime := printTime(ptime << " (gc = ",gc clock);
				ptime << ")" << endnl;
			}
			err?;
	}

	local printTime(p:TX, time:Z):TX == {
		import from String;
		assert(time >= 0);
		(hours, time) := divide(time, 3600000);
		p << hours << ":";
		(mins, time) := divide(time, 60000);
		if mins < 10 then p << "0";
		p << mins << ":";
		(secs, time) := divide(time, 1000);
		if secs < 10 then p << "0";
		p << secs << ".";
		if time < 100 then p << "0";
		if time < 10 then p << "0";
		p << time;
	}
}
