# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::Command;
use Getopt::Long ();

use vars qw($VERSION);
$VERSION = '0.1.1';

sub new ($;$$) {
	my $class = shift;
	my $name   = shift;
	my $parent = shift;

	my $self = bless {
		name   => $name,
		parent => $parent,
	}, $class;

	if ($parent) {
		my $name0 = $name; $name0 =~ s/-/_/g;
		$self->{root}       = $parent->{root};
		$self->{prefix}     = $parent->{prefix} . " $name";
		$self->{class_file} = $parent->{class_dir} . "/$name0.pm";
	} else {
		$self->{root}       = $self;
		$self->{prefix}     = "axp";
		$self->{class_file} = __FILE__;
	}
	$self->{class_dir} = $self->{class_file};
	$self->{class_dir} =~ s/\.pm$// or die "Internal error";
	return $self;
}

sub infoline ($) {
	"Arch eXtension Platform, version $VERSION";
}

sub optusage ($) {
	my $self = shift;
	$self->has_subcommands? "[subcommand] [options]": "[options]";
}

sub helptext ($) {
	my $self = shift;
	qq{
		To see a list of available subcommands, run:

		    $self->{prefix} help
	};
}

sub options ($) {
}

sub root_options ($) {
	my $version_cb = sub { print "$VERSION\n"; exit 0; };
	return (
		version => { sh => 'v', cb => $version_cb, desc => "display the script version and exit" },
	);
}

sub mandatory_options ($) {
	my $self = shift;
	return (
		help => { sh => 'h', cb => sub { $self->show_usage }, desc => "display this help message and exit" },
	);
}

sub help_options ($) {
	return (
		recursive => { sh => 'r', desc => "process subcommands recursivelly" },
	);
}

sub all_options ($) {
	my $self = shift;
	return (
		$self->mandatory_options,
		($self == $self->{root}? $self->root_options:
			$self->has_subcommands? (): $self->options),
	);
}

sub long_option ($$) {
	my $self = shift;
	my $option = shift;
	$option = "--$option";
	$option =~ s/_/-/g;
	return $option;
}

sub has_subcommands ($) {
	my $self = shift;
	return $self->{has_subcommands} if exists $self->{has_subcommands};
	return $self->{has_subcommands} = -d $self->{class_dir};
}

sub all_subcommand_names ($) {
	my $self = shift;
	return @{$self->{all_subcommand_names}} if exists $self->{all_subcommand_names};

	my @names = ();
	goto RETURN unless $self->has_subcommands;

	my $dir = $self->{class_dir};
	opendir DIR, $dir or die "Can't opendir $dir: $!";
	my @files = readdir DIR;
	closedir DIR;
	@names = grep { /^\w+\.pm/ } @files;
	foreach (@names) { s/\.pm$//; s/_/-/g; }
	@names = ('help', sort @names);

RETURN:
	$self->{all_subcommand_names} = \@names;
	return @names;
}

sub create_subcommand ($$) {
	my $self = shift;
	my $name = shift;
	my $name0 = $name; $name0 =~ s/-/_/g;
	my $class = ref($self) . "::$name0";
	my $prefix = $self->{prefix};
	my $class_file = $self->{class_dir} . "/$name0.pm";
	if ($name eq 'help') {
		return $self->{help_subcommand} if $self->{help_subcommand};
		no strict 'refs';
		@{"${class}::ISA"} = (ref($self));
		*{"${class}::execute"} = sub { $self->show_subcommands };
		*{"${class}::options"} = sub { $self->help_options };
		*{"${class}::infoline"} = sub { "list all '$prefix' subcommands" };
		*{"${class}::optusage"} = sub { "[options]" };
		*{"${class}::helptext"} = sub { "The special subcommand 'help' exist for all composite commands\n(like '$prefix'), it briefly describes all subcommands." };
	} else {
		# promote 'tree' subcommands to the highest level for conveniece
		if (!-f $class_file && $self == $self->{root}) {
			my $class_file2 = $self->{class_dir} . "/tree/$name0.pm";
			if (-f $class_file2) {
				$class_file = $class_file2;
				$class = ref($self) . "::tree::$name0";
				$self = $self->create_subcommand('tree');
			}
		}
		die "Unknown subcommand: $name\n\tno $class_file\n\t" .
			"try '$prefix help'\n" unless -f $class_file;
#		require $class_file;
		eval "use $class;";
		die $@ if $@;
	}
	my $command = $class->new($name, $self);
	$self->{help_subcommand} = $command if $name eq 'help';
	return $command;
}

sub all_subcommands ($) {
	my $self = shift;
	return @{$self->{all_subcommands}} if exists $self->{all_subcommands};

	my @commands = $self->all_subcommand_names;
	foreach (@commands) {
		$_ = $self->create_subcommand($_);
	}
	$self->{all_subcommands} = \@commands;
	return @commands;
}

sub foreach_subcommand ($$) {
	my $self = shift;
	my $code = shift || die;
	my @result = &$code($self);
	if ($self->has_subcommands) {
		foreach my $subcommand ($self->all_subcommands) {
			push @result, $subcommand->foreach_subcommand($code);
		}
	}
	return @result;
}

sub show_subcommands ($;$) {
	my $self = shift;
	my $prefix = $self->{prefix};

	my $level = shift || 0;
	my $recursive = 1;

	if ($level == 0) {
		my $help_subcommand = $self->{help_subcommand} || {options => {}};
		my %opt = %{$help_subcommand->{options}};
		$help_subcommand->show_usage if $opt{help};

		my $infoline = $self->infoline;
		print "$prefix - $infoline\n\n";
		print "Subcommands:\n\n";
		$recursive = $opt{recursive};
	}

	my $len = 0;
	my @names = $self->all_subcommand_names;
	foreach (@names) { $len = length($_) if $len < length($_); }

	foreach my $name (@names) {
		my $delim = " : ";
		my $infoline;
		my $command = $self->create_subcommand($name);
		if ($command->has_subcommands) {
			if ($recursive) {
				$infoline = $delim = "";
			} else {
				$infoline = "composite subcommand, try '$prefix $name help'";
				$command = undef;
			}
		} else {
			$infoline = $command->infoline;
			$command = undef;
		}
		printf "%s%-${len}s%s%s\n", "    " x ($level + 1), $name, $delim, $infoline;
		$command->show_subcommands($level + 1) if $command;
	}

	if ($level == 0) {
		print "\nRun '$prefix subcommand -h' for help on 'subcommand'\n\n";
		exit 0;
	}
}

sub show_usage ($%) {
	my $self = shift;
	my %args = @_;
	my $error = $args{error} ? "[ERROR] $args{error}\n\n" : "";
	my $infoline = $args{infoline} || $self->infoline;
	my $optusage = $args{optusage} || $self->optusage;
	my $helptext = $args{helptext} || $self->helptext;
	my $prefix   = $args{prefix}   || $self->{prefix};
	my $options  = $args{options};

	my @option_entries = $options? @$options: $self->all_options;
	my %option_entries = @option_entries;
	my @option_labels = ();
	my $max_label_len = 0;
	while (@option_entries) {
		my ($option, $entry) = splice(@option_entries, 0, 2);
		my $shortcut = $entry->{sh};
		my $label = $self->long_option($option);
		$label = "-$shortcut, $label" if $shortcut;
		my $arg = $entry->{arg} || "ARG";
		my $type = $entry->{type} || "";
		$label .= " $arg" if $type =~ /=[si]$/;
		$label .= " [$arg]" if $type =~ /:[si]$/;
		$max_label_len = length($label) if $max_label_len < length($label);
		push @option_labels, [ $option, $label ];
	}
	my $option_text = "";
	foreach (@option_labels) {
		my ($option, $label) = @$_;
		$option_text .= sprintf(
			"\n\t\t  %-${max_label_len}s  %s",
			$label, $option_entries{$option}{desc} || "(no description)"
		);
	}

	$helptext =~ s/^\n*/\n/;
	$helptext =~ s/\s*$/\n/;
	my $msg = qq(
		$error$prefix - $infoline

		Usage: $prefix $optusage
		$option_text
		$helptext
	);
	$msg =~ s/\t$//;
	$msg =~ s/\n\t\t/\n/g;
	$msg =~ s/^\n//;
	my $stream = $error ? \*STDERR : \*STDOUT;
	print $stream $msg;
	exit($error ? 1 : 0);
}

sub version ($) {
	return $VERSION;
}

sub script_version ($) {
	return "axp-$VERSION";
}

sub setup_config_dir ($@) {
	my $self = shift;
	require Arch::Util;
	Arch::Util::setup_config_dir(undef, 'axp', @_);
}

sub execute ($) {
	my $self = shift;
	my $prefix = $self->{prefix};
	die "Usage: $prefix command ...\n\ttry '$prefix -h'\n";
}

sub process_options ($%) {
	my $self = shift;

	my %option_entries = @_;
	my %getopts = ();
	my %options = ();
	while (my ($option, $entry) = each %option_entries) {
		my $type = $entry->{type};
		my $key = $option;
		$key .= "|$entry->{sh}" if defined $entry->{sh};
		$key =~ s/_/-/g;
		$key .= $type if defined $type;
		my $value = $entry->{cb};
		unless ($entry->{cb}) {
			my $init_value = 0;
			$init_value = '' if $type && $type =~ /s$/;
			$init_value = $entry->{init} if exists $entry->{init};
			$options{$option} = $init_value;
			$value = \$options{$option};
		}
		$getopts{$key} = $value;
	}
	Getopt::Long::Configure('require_order', 'bundling', 'ignore_case');
	Getopt::Long::GetOptions(%getopts) or return undef;
	return \%options;
}

sub process ($) {
	my $self = shift;

	$self->{options} = $self->process_options($self->all_options);
	return unless $self->{options};

	if (@ARGV && $self->has_subcommands) {
		my $subcmd = shift @ARGV;
		unless ($subcmd =~ /^[\w-]+$/) {
			$self->show_usage(error => "Invalid subcommand: $subcmd");
		}
		my $command = $self->create_subcommand($subcmd);
		$command->process;
	} else {
		$self->execute;
	}
}

1;
