# Arch Perl library, Copyright (C) 2004 Mikhael Goikhman
#
# This program is free software; you can redistribute it and/or modify
# it under the terms of the GNU General Public License as published by
# the Free Software Foundation; either version 2 of the License, or
# (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

use 5.005;
use strict;

package AXP::Command::tree::switch_undo;
use base 'AXP::Command::tree';

use Arch::Util qw(run_tla run_cmd);
use Arch::TempFiles;

sub infoline {
	"switch between current and previous set of changes"
}

sub optusage {
	"[options] [undo-number]"
}

sub helptext {
	q{
		Switch between the current set of changes and the previously
		saved one.

		This performs 'undo', then swaps two last ,,undo-* directories
		(or the last one and some other undo dir in the middle pointed
		by undo-number), then 'redo'.
	}
}

sub execute {
	my $self = shift;
	my %opt = %{$self->{options}};

	die "The --test option is not implemented yet\n" if $opt{test};
	$ENV{DEBUG} = 1 if $opt{verbose};

	my $tree = $self->tree;
	my $dir = $tree->root;

	die "No tree changes to undo\n" unless $tree->get_changes;

	my %undo_numbers = map { $_ => 1 } grep /^\d+$/,
		map { s!^$dir/,,undo-!! } glob("$dir/,,undo-*");
	die "No ,,undo-* directories, exiting\n" unless %undo_numbers;
	my $max_undo_number = (sort keys %undo_numbers)[-1];

	my $undo_number = shift @ARGV || $max_undo_number;
	die "Unexisting undo directory number ($undo_number)\n"
		unless $undo_numbers{$undo_number};
	my $old_undo_dir = "$dir/,,undo-$undo_number";
	my $new_undo_dir = "$dir/,,undo-" . ($max_undo_number + 1);

	my $tmp = Arch::TempFiles->new;
	$tmp->root($dir);
	my $tmp_dir = $tmp->dir_name("switch-undo");

	print "* undoing current changes\n"
		unless $opt{quiet};
	run_tla("undo", "-d", $dir, "-o", $tmp_dir);
	die "Can't undo in tree $dir: exit status $?\n" if $?;

	print "* switching the latest changes and undo directory #$undo_number\n"
		unless $opt{quiet};
	run_cmd("mv", $old_undo_dir, $new_undo_dir);
	die "Error while executed mv $old_undo_dir $new_undo_dir\n" if $?;
	run_cmd("mv", $tmp_dir, $old_undo_dir) unless $?;
	die "Error while executed mv $tmp_dir $old_undo_dir\n" if $?;

	print "* redoing previous changes\n"
		unless $opt{quiet};
	run_tla("redo", "-d", $dir);
	die "Can't redo in tree $dir: exit status $?\n" if $?;
}

1;
