/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst.lucene.server;

import java.lang.ref.ReferenceQueue;
import java.lang.ref.WeakReference;
import java.util.Arrays;
import java.util.HashMap;
import java.util.Map;

/**
 * Manages a pool of unique stoplists.  Each StopList object contains
 * a distinct array of stop-words so stop-lists can be quickly and
 * easily compared. <p>
 *
 * The pool uses WeakReference objects to maintain the list so that
 * any unreferenced StopList objects will automatically be eliminated
 * from the pool.
 */

public class StopList implements IPackage, Constants
{
  /**
   * This is the pool of all referenced StopList instances.  <p>
   * Map key is PoolKey, value is WeakReference(StopList)
   */
  private static final Map pool = new HashMap();

  /**
   * The garbage collector's reference queue.  Disposed-of entries
   * will be registered here.
   */
  private static final ReferenceQueue queue = new ReferenceQueue();

  /**
   * A stop list of words in sorted order for ease of comparison with
   * other stop list instances.
   */
  private final String[] stopWords;

  private StopList(String[] stopWords)
  {
    this.stopWords = stopWords;
  }

  /**
   * Return (a copy of) the underlying stop words.
   */

  public String[] getStopWords()
  {
    return (String[])stopWords.clone();
  }

  /**
   * Create a StopList instance for an array of words, or return an
   * existing one if the word-list matches that StopList instance's
   * word list.
   */

  public static synchronized StopList create(String[] words)
  {
    String[] stopWords = (String[])words.clone();
    for (int i = 0; i < stopWords.length; i++)
      stopWords[i] = stopWords[i].intern();

    Arrays.sort(stopWords);
    PoolKey key = new PoolKey(stopWords);

    // Purge any enties associated with unreferenced StopList instances
    Entry entry;
    while (null != (entry = (Entry)queue.poll()))
      pool.remove(entry.key);

    // See if we have this StopList in our pool.
    entry = (Entry)pool.get(key);

    StopList stopList = entry == null ? null : (StopList)entry.get();
    if (stopList != null)
      return stopList;

    stopList = new StopList(stopWords);
    pool.put(key, new Entry(key, stopList, queue));
    return stopList;
  }

  /**
   * Add a PoolKey item to the WeakReference so that we can determine
   * which entry of our pool should be deleted from the WeakReferences
   * queued in the ReferenceQueue.
   */
  private static final class Entry extends WeakReference
  {
    final PoolKey key;

    Entry(PoolKey key, StopList stopList, ReferenceQueue queue)
    {
      super(stopList, queue);
      this.key = key;
    }
  }

  /**
   * A wrapper for an array of words to allow the array to be used as
   * a key to a Map.  The array of words are assumed to be sorted and
   * intern()'ed so that comparing them is a simple matter of
   * comparing consecutive references.
   */
  private static final class PoolKey
  {
    /**
     * A sorted stop list of intern()'ed Strings
     */
    private final String[] stopWords;

    PoolKey(String[] stopWords)
    {
      this.stopWords = stopWords;
    }

    /**
     * Chevk whether two PoolKey objects match.  They match if they
     * have identical word lists.
     */
    public final boolean equals(Object other)
    {
      if (other == this)
        return true;

      if (!(other instanceof PoolKey))
        return false;

      PoolKey that = (PoolKey)other;

      if (this.stopWords == that.stopWords)
        return true;

      if (this.stopWords.length != that.stopWords.length)
        return false;

      for (int i = 0; i < stopWords.length; ++i)
        //if (this.stopWords[i] != that.stopWords[i])
        if (!this.stopWords[i].equals(that.stopWords[i]))
          return false;
      return true;
    }

    public int hashCode()
    {
      int hashcode = 0;
      int start = stopWords.length > 10 ? stopWords.length - 10 : 0;

      for (int i = start; i < stopWords.length; ++i)
        hashcode ^= i ^ stopWords[i].hashCode();

      return hashcode;
    }
  }
}

