/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst.lucene.server;

import nz.net.catalyst.ELog;
import nz.net.catalyst.Log;

/**
 * This class performs a System.gc() call in order to potentially 
 * free up memory.  
 *<p>The free memory before and after garbage collection is outputted
 * to the log files.
 *<p>Usage:
 *<p><pre>CONTROL
 *Sub-Command: GC
 *END</pre>
 *<p>Please note that this will affect all applications on a single server
 *and could potentially cause a pause in the server affecting performance.
 *<p>
 *As of 18th November 2002, in testing I haven't seen a garbage collect 
 *take longer than 2 seconds [yet].
 */

class GarbageCollect implements IPackage, Constants
{
  private final Transmission input;

  GarbageCollect(Transmission transmission)
  {
    input = transmission;
  }

  Transmission execute()
  {
      
    String shutdown = input.get("Action", NO_APP);
    if (shutdown != null && shutdown.trim().equalsIgnoreCase("shutdown")) {
        System.err.println("About to shutdown lucene server");
        System.out.println("About to shutdown lucene server");
        System.exit(0);
    }    
      
    long tStart = System.currentTimeMillis();
    Transmission response = new Transmission(ECommand.CONTROL_RESPONSE);
    response.setSerial(input.getSerial());

    //if (id == null)  return error("Mandatory '" + ID + "' header is missing");
    Runtime r = Runtime.getRuntime();
    
    response.add("INFO", "Free memory before: " + r.freeMemory());
    
    Log.log(ELog.NOTICE, "About to perform system garbage collection and finalization.");
    Log.log(ELog.NOTICE, "Free memory before gc: " + r.freeMemory());
    r.gc();
    r.runFinalization();
    response.add("INFO", "Garbage Collection Suggested to JVM");
    response.add("INFO", "Running of finalizers suggested to JVM");
    response.add("INFO", "Free memory after: " + r.freeMemory());
    Log.log(ELog.NOTICE, "Free memory after gc: " + r.freeMemory());
    response.add("Timer:total time", String.valueOf(System.currentTimeMillis() - tStart));

    return response;
  }

  /**
   * Build an error response for sending back to the client.
   *
   * @param message The text of the error message
   * @return An INDEX-RESPONSE Transmission
   */

  private Transmission error(String message)
  {
    Transmission response = new Transmission(ECommand.CONTROL_RESPONSE);
    response.setSerial(input.getSerial());
    response.add(ERROR, message);
    return response;
  }
}
