package com.holub.asynch;
import com.holub.tools.Command;


/** A generic implemenation of a thread pool. Use it like this:
 *
 *	Thread_pool pool = new Thread_pool();
 *	pool.execute
 *	(	new Runnable()
 *		{	public void run()
 *			{	// execute this function on an existing
 *				// thread from the pool.
 *			}
 *		}
 *	);
 *
 * The size of the thread pool can expand automatically to accomodate
 * requests for exectuion. That is, if a thread is available in
 * the pool, it's used to execute the Runnable object, otherwise
 * a new thread can be created (and added to the pool) to execute the
 * request. A maximum count can be specified to limit the number of
 * threads in the pool, however.
 *
 * Each thread pool also forms a thread group (all threads in the
 * pool are in the group). In practice this means that the security
 * manager controls whether a thread in the pool can access threads
 * in other groups, but it also gives you an easy mechanism to make the
 * entire group a Daemon.
 *
 * <br><br>
 * <table border=1 cellspacing=0 cellpadding=5><tr><td><font size=-1><i>
 * <center>(c) 1999, Allen I. Holub.</center>
 * <p>
 * This code may not be distributed by yourself except in binary form,
 * incorporated into a java .class file. You may use this code freely
 * for personal purposes, but you may not incorporate it into any
 * commercial product without express permission of Allen I. Holub in
 * writing.
 * </td></tr></table>
 */

public class Thread_pool extends ThreadGroup
{
	private final Blocking_queue pool 	 	  = new Blocking_queue();
	private /*final*/ int maximum_size;
	private int	 pool_size;
	private boolean	 has_closed   = false;

	private static int	group_number  = 0;
	private static int 	thread_id	  = 0;

	/******************************************************************
	 * These are the objects that wait to be activiated. They are
	 * typically blocked on an empty queue. You post a Runnable
	 * object to the queue to release a thread, which will execute
	 * the run() method from that object. All Pooled-thread objects
	 * will be members of the thread group the comprises the tread
	 * pool.
	 */

	private class Pooled_thread extends Thread
	{	
		public Pooled_thread()
		{	super( Thread_pool.this, "T" + thread_id++ );
		}

		public void run() 
		{	try
			{	while( !has_closed )
				{	((Runnable)( pool.dequeue() )).run();
				}
			}
			catch(InterruptedException  e){/* ignore it, stop thread */}
			catch(Blocking_queue.Closed e){/* ignore it, stop thread */}
		}
	}

	/******************************************************************
	 *	Create a thread pool with initial_thread_count threads
	 *	in it. The pool can expand to contain additional threads
	 *  if they are needed.
	 *
	 *	@param <b>initial_thread_count</b>	The initial thread count.
	 *			If the initial count is greater than the maximum, it is
	 *			silently truncated to the maximum.
	 *  @param <b>maximum_thread_count</b> specifies the maximum number
	 *			of threads that can be in the pool. A maximum of 0
	 *			indicates that the 	pool will be permitted to grow
	 *			indefinately.
	 */

	public Thread_pool(int initial_thread_count, int maximum_thread_count)
	{	super( "Thread_pool" + group_number++ );

		maximum_size = (maximum_thread_count > 0)
							? maximum_thread_count: Integer.MAX_VALUE;

		pool_size	 = Math.min(initial_thread_count, maximum_size);

		for(int i = pool_size; --i >= 0 ;)
			new Pooled_thread().start();
	}

	/******************************************************************
	 *	Create a dynamic Thread pool as if you had used
	 *	<code>Thread_pool(0, true);</code>
	 **/

	public Thread_pool()
	{	
		super( "Thread_pool" + group_number++ );
		this.maximum_size = 0;
	}

	/******************************************************************
	 *  Execute the <code>run()</code> method of the Runnable object on a thread
	 *	in the pool. A new thread is created if the pool is
	 *	empty and the number of threads in the pool is not at the
	 *  maximum.
	 *
	 * 	@throws Thread_pool.Closed if you try to execute an action
	 *			on a pool to which a close() request has been sent.
	 */

	public synchronized void execute( Object action ) throws Closed
	{	
		// You must synchronize on the pool because the Pooled_thread's
		// run method is asychronously dequeueing elements. If we
		// didn't synchronize, it would be possilble for the is_empty()
		// query to return false, and then have a Pooled_thread sneak
		// in and put a thread on the queue (by doing a blocking dequeue).
		// In this scenario, the number of threads in the pool could
		// exceede the maximum specified in the constructor. The 
		// test for pool_size < maximum_size is outside the synchronized
		// block because I didn't want to incur the overhead of
		// synchronization unnecessarily. This means that I could
		// occasionally create an extra thread unnecessarily, but
		// the pool size will never exceed the maximum.


		if( has_closed )
			throw new Closed();

		if( pool_size < maximum_size && pool.waiting_threads() == 0 )
		{	synchronized( pool )
			{	if( pool_size < maximum_size && pool.waiting_threads()==0 )
				{	++pool_size;
					new Pooled_thread().start(); // Add thread to pool
				}
				pool.enqueue( action );			 // Attach action to it.
			}
		}
		else
			pool.enqueue( action );			 // Attach action to it.

	}

	/******************************************************************
	 *  Execute the <code>execute()</code> method of the Command object
	 *  on a thread in the pool. A new thread is created if the pool is
	 *	empty and the number of threads in the pool is not at the
	 *  maximum.
	 *
	 * 	@throws Thread_pool.Closed if you try to execute an action
	 *			on a pool to which a close() request has been sent.
	 */

	public final synchronized void execute(
    final Command action,
    final Object  argument) throws Closed
	{
    execute(	
        new Runnable() {	
            public void run() {
                action.execute( argument );
            }
        }
    );
	}

	/******************************************************************
	 * Workhorse function called by both variants on Execute.
	 */


	/******************************************************************
	 * Objects of class Thread_pool.Closed are thrown if you try to
	 * execute an action on a closed Thread_pool.
	 */

	public class Closed extends RuntimeException
	{	Closed()
		{	super("Tried to execute operation on a closed Thread_pool");
		}
	}

	/******************************************************************
	 *	Kill all the threads waiting in the thread pool, and arrange
	 *	for all threads that came out of the pool, but which are working,
	 *	to die natural deaths when they're finished with whatever they're
	 *	doing. Actions that have been passed to exectute() but which
	 *	have not been assigned to a thread for execution are discarded.
	 *	<p>
	 *  No further operations are permitted on a closed pool, though
	 *	closing a closed pool is a harmless no-op.
	 */

	public synchronized void close()
	{	has_closed = true;
		pool.close();					// release all waiting threads
	}

	/* ============================================================== */

	public static class Test
	{	
		private static Thread_pool pool = new Thread_pool( 10, 10 );

		public static void main( String[] args )
		{
			Test test_bed = new Test();
			test_bed.fire_runnable( "hello" );

			pool.execute
			(	new Command()
				{	public void execute( Object argument )
					{	System.out.println("Starting " + argument );

						try{	Thread.sleep(500); }
						catch(InterruptedException e){}

						System.out.println("Stoping " + argument );
					}
				},
				"world"
			);

			// Give the threads a chance to start before closing the pool
			try {
				Thread.sleep(1500);
			} catch (InterruptedException e) {
			}

			pool.close();
		}

		// The argument must be final in order for it to be accessed
		// from the inner class.

		private void fire_runnable( final String id )
		{	pool.execute
			(	new Runnable()
				{	public void run()
					{	System.out.println("Starting " + id );

						try{	Thread.sleep(500); }
						catch(InterruptedException e){}

						System.out.println("Stoping " + id );
					}
				}
			);
		}
	}
}
