/** *********************************************************************
 * Copyright (C) 2003 Catalyst IT                                       *
 *                                                                      *
 * This program is free software; you can redistribute it and/or modify *
 * it under the terms of the GNU General Public License as published by *
 * the Free Software Foundation; either version 2 of the License, or    *
 * (at your option) any later version.                                  *
 *                                                                      *
 * This program is distributed in the hope that it will be useful,      *
 * but WITHOUT ANY WARRANTY; without even the implied warranty of       *
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        *
 * GNU General Public License for more details.                         *
 *                                                                      *
 * You should have received a copy of the GNU General Public License    *
 * along with this program; if not, write to:                           *
 *   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    *
 *   Boston, MA  02111-1307  USA                                        *
 ************************************************************************/
package nz.net.catalyst.lucene.server;

import java.io.BufferedReader;
import java.io.IOException;
import java.io.PrintWriter;

import nz.net.catalyst.IClient;
import nz.net.catalyst.Log;
import org.dom4j.Document;
import org.dom4j.DocumentException;
import org.dom4j.DocumentHelper;

/**
 * Execute a Session with a Lucene client.
 */

public class Session implements IPackage, IClient
{
  /**
   * Header used to describe an error.
   */
  private static final String ERROR = "Error";

  /**
   * Process client requests: A new Lucene client has connected.
   * Process a single request and then close the connection.
   * <p> The first line of input will be read in.<br> If this is XML (starts
   * with &lt;) then XML processing is used and the input is swallowed until the
   * end of the first XML document (&lt; /LuceneRequest&gt;).<br> If not xml
   * then a dialogue will be created from the line.  The dialogue will read the
   * input when creating the transmission.
   * <p>THE XML NEEDS REWORKING. There are the following issues:<br>
   * - If there is no &lt;/LuceneRequest&gt; tag then the server will hang
   * waiting for it. <br>
   * - There NEEDS to be a newline character (\n, \r or both) after the closing
   * tag of the XML document.  This is due to a bufferedreader being used and
   * its readline method which only reads a complete line at a time.
   *
   * @param input The incoming request stream from the client. A BufferedReader
   * so we can take advantage of the readline() method.
   * @param output The outgoing response stream to the client. A PrintWriter so
   * we can use it like System.out
   * @param id A name for the connection, for diagnostics
   */
  public void communicate(BufferedReader input, PrintWriter output, String id) {
	Log.info("---Starting session---");
	long sessionStartTime = System.currentTimeMillis();
	
	String line   = null;  //current line of input.
	String serial = null;  //serial of current request. if any and if extracted.
	boolean isXML = false;
	
	try {
		line = input.readLine().trim();
		//this is a pretty crude way of determining whether the input is XML or not.
		if (line != null && line.startsWith("<")) {
			isXML = true;
			//long startTime = System.currentTimeMillis();

			StringBuffer sb = new StringBuffer();
			do {
				Log.debug("XML Line being processed: " + line);
				sb.append(line);
				if (XMLHelper.isEndTag(line)) break; //break out of loop if /LuceneXXXRequest
			} while (null != (line = input.readLine()));

			try {
				Log.debug("About to strip invalid(high byte) characters");
				
				//strip characters here.
				String unstripped = sb.toString();
				StringBuffer stripped = new StringBuffer(); 
				for (int i = 0; i < unstripped.length(); ++i) {
					//do stuff.
					char c = unstripped.charAt(i);
					
					int  char_value = (int) c; 
					
					//Log.debug("charvalue is: " + char_value);
					
					if (char_value < 128) {
						if (char_value > 31) {
							stripped.append(c);
						} else {
							Log.debug(" - replacing character with space:" + char_value + ": char: " + c);
							stripped.append(' ');
						}
					} else {
						Log.debug(" - stripping character with value:" + char_value + ": char: " + c);
					}
				}
				
				
				Log.debug("About to parse XML input...");
				Document doc = DocumentHelper.parseText(stripped.toString());
				Log.debug("Input parsed as XML successfully!");

				Transmission transmission = new XMLHelper().getTransmission(doc);
				serial = transmission.getSerial();
				
				Log.debug("Request body interpreted...");
								
				//use my newly added static creator with a handbuilt transmission object.
				Dialogue dialogue = Dialogue.create(transmission, input, output);

				dialogue.process();
			} catch (DocumentException e) {
				Log.warn("XML Parse Error.  Your xml is invalid, sorry.");
				Log.warn(e.getMessage());
				//TIDY THIS UP.
				Transmission response = new Transmission(ECommand.ERROR);
				response.add(ERROR, "XML Parse Error: " + e.getMessage());
				response.setXML(true);
				response.write(output);
			}
		} else {
			//Use LuceneProtocol v1.0
			Log.debug("about to read LuceneProtocol v1.0");
			
			if (line == null) {
				error(output, "Invalid input, trying to process null.");
			}
			//attempt to create a dialogue for the current line (should be the header).
			Dialogue dialogue = Dialogue.create(line, input, output);

			//if no dialogue then error
			if (dialogue == null) {
					error(output, "Expecting command, found: " + line);
			} else {
				Log.debug("About to process LuceneProtocol v1.0 Dialogue: " + dialogue);
				dialogue.process();
				dialogue = null;
			}
		}
		
	//} catch (java.net.SocketTimeoutException e) { //only in java 1.4
	} catch (java.net.SocketException e) {
		Log.error("Client Timed Out: No data sent within timeout period.");
		Log.debug("Last line: " + line);
	} catch (java.io.InterruptedIOException e) {		
		Log.error("Client Timed Out: No data sent within timeout period.");
		Log.debug("Last line: " + line);
	} catch (IOException e)	{
		Log.error("Error when reading from socket:" + e.getMessage());
		Log.debug("Can be caused by invalid encoding setting and multi-byte characters");
		throw new RuntimeException("I/O Error on \"" + id + '"', e);
	} catch (java.lang.OutOfMemoryError e) {
		Log.error("Ran out of memory while trying to complete Client Request.");
		Transmission transmission = new Transmission(ECommand.ERROR);
		if (serial != null) transmission.add("Serial", serial);
		transmission.add(ERROR, "Ran out of memory while trying to complete request.\n Suggest no sorting or limiting the number of results returned.");
		if (isXML) transmission.setXML(true);
		transmission.write(output);
		System.err.println(new java.util.Date() + " - Ran out of memory.");
	} finally {
		try {
			if (output != null) output.close();
			if (input  != null) input.close();
			output = null;
			input  = null;
		} catch (Exception e) { /* ignore */ }
		long sessionEndTime = System.currentTimeMillis();
		Log.info("===Ended session=== (Total Time: " + (sessionEndTime-sessionStartTime) + "ms)");
	}
  	line = null;
  } //comunicate
  

  static void error(PrintWriter output, String message)
  {
	Transmission transmission = new Transmission(ECommand.ERROR);
	transmission.add(ERROR, message);
	transmission.write(output);
  }
}