<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    rowmenu-defs.php                                        */
/* Author:      Paul Waite                                              */
/* Description: Definitions for a non-javascript multi-level menu       */
/*              system, in horizontal rendering with each menu level    */
/*              set one below the other in rows.                        */
/*                                                                      */
/* ******************************************************************** */
/** @package menu */

/** Tree-menu class */
include_once("treemenu-defs.php");

// ----------------------------------------------------------------------
/**
* A hierarchical menu renderer which does not use Javascript to
* implement the menuoption display. NB: this renderer is basically
* the same functionally as the treemenu but is rendered differently.
* The menu is rendered in a table of 1 column x N rows. The top row
* contains all menu items of LEVEL 0, the second row contains items
* of LEVEL 1 and so on. Rows will only be rendered if the previous
* level has an EXPANDED heading menu option. So, at the start the
* menu is rendered with only the top level (LEVEL 0) items row
* displayed. If there are sub-menus, then one or more of these top
* items will be a "heading item" and clicking on it will reveal the
* next level of sub-menu items for which it is the parent. Items
* which are expanded or current location will be given the style
* defined by the ".hlmenuitem" class in sitestyle.css. All other
* menu styles are as defined in the '.menu" class. The menu status
* is saved/recovered from ax_wwwsession.menu_status and is therefore
* persistent.
* @package menu
*/
class rowmenu extends treemenu {
  /** Separator string between menu options */
  var $separator = SEP_BAR;
  /** Array of background colours for each level */
  var $background_colours = array();
  // ....................................................................
  /**
  * Constructor
  *
  * Create a new menumaintainer.
  * @param string  $id              Unique database menu identifier
  * @param object  $webpage         Webpage object that this menu is being created for
  * @param string  $stylsheet       Name of stylesheet file to reference for menu styles
  */
  function rowmenu($id="main", $webpage=false, $stylesheet="") {
    $this->treemenu($id, $webpage, $stylesheet);
  } // rowmenu
  // ....................................................................
  /**
  * Set backgrounds colours
  * Sets the background colours for each menu level. Provide either a
  * delimited list of colours in a string in #nnnnnn format, or an
  * array of same. The first colour is for LEVEL 0, second LEVEL 1 etc.
  * @param mixed $bgcolours List or array of background colours.
  */
  function set_background_colours($bgcolours, $delim=",") {
    if (!is_array($bgcolours)) {
      $bgcolours = explode($delim, $bgcolours);
    }
    $this->background_colours = $bgcolours;
  } // set_background_colours
  // ....................................................................
  /**
  * Set the separator string.
  * Sets the string which separates menu options as listed hozizontally
  * across. This defaults to SEP_BAR ("&nbsp;| ").
  * @param string $sep The separator string to use between menu options
  */
  function set_separator($sep) {
    $this->separator = $sep;
  } // set_separator
  // ....................................................................
  /**
  * Process any menu navigation.
  * This means we take note of any $_mid value which signifies a menu
  * option of that ID is currently being focussed on and alter the menu
  * configuration accordingly. We also look at the current page and
  * compare that to the menu id 'action' to determine where we are and
  * what we should be doing.
  */
  function process_navigation() {
    global $_mid, $RESPONSE;
    if ($this->exists) {
      if (isset($_mid) && $this->menu->menuop_exists($_mid)) {
        $this->menu->current_menu_option = $_mid;
        debugbr(">> setting current menu option id=$_mid");
        $selmop = $this->menu->menuop($_mid);
        if ($selmop !== false) {
          // Expand if a submenu-heading..
          if ($selmop->is_submenuheading()) {
            $selmop->expanded = true;
          }
          foreach ($this->menu->menu_ops as $mopid => $mop) {
            if ($mopid != $_mid && $mop->level >= $selmop->level && $mop->expanded) {
              $mop->expanded = false;
              $this->menu->menu_ops[$mopid] = $mop;
            }
          }
          $this->menu->menu_ops[$_mid] = $selmop;
        }
        // Save the updated menu instance object..
        $this->save_to_session();
      }
    }
  } // process_navigation
  // ....................................................................
  /**
  * Render the menu as HTML.
  * @return string The HTML
  */
  function html() {
    debug_trace($this);
    global $LIBDIR, $RESPONSE;
    global $_mid;

    // Initialise content..
    $s = "";

    // Menu layout table..
    $Tvw = new table($this->menuid . "_menu", "menu");
    $Tvw->setwidth("");

    // MENU ITEM DETAIL..
    if ($this->menu->menuop_count() > 0) {
      $parent_id = 0;
      for ($level=0; $level <= $this->menu->level_depth; $level++) {
        $Tvw->tr();
        $menurow_items = array();
        $next_parent_id = false;
        foreach ($this->menu->menu_ops as $mopid => $mop) {
          if ($mop->parent == $parent_id /* && $mop->level == $level */ ) {
            if ($mop->expanded) {
              $next_parent_id = $mopid;
            }
            $action = $mop->action;
            $label  = $mop->label;
            if ($action == "") {
              // Link to current page for refresh..
              $action = $RESPONSE->requested;
              if ($RESPONSE->requested_query != "") {
                $action .= "?$RESPONSE->requested_query";
              }
              $heading = true;
            }
            $menuoption = !$heading;

            // Add selected menu option id to url..
            $action = href_addparm($action, "_mid", $mopid);

            // Make sure our menu gets refreshed and not stuck in a static
            // state in the web-browser's cache..
            $action = href_addparm($action, "cachecontrol", "dynamic");

            // To click or not to click, only if not current webpage..
            $req = preg_replace("/^[\\/]/", "", $RESPONSE->requested);
            $act = preg_replace("/^[\\/]/", "", $mop->action);

            if ($mop->action != "" && preg_match("/$act/i", $req)) {
              $menuop = "<span id=\"hlmenuitem\">$label</span>";
            }
            else {
              $menuop_link = new anchor($action, $label);
              if ($heading && $mop->expanded) {
                $menuop_link->setid("hlmenuitem");
              }
              $menuop = $menuop_link->render();
            }
            $menurow_items[] = $menuop;
          }
        } // foreach

        // Add the menu options row to the menu..
        $Tvw->td( implode($this->separator, $menurow_items) );
        if (isset($this->background_colours[$level])) {
          $Tvw->td_css("background-color:" . $this->background_colours[$level]);
        }

        // If we didn't find a new parent option, we're done..
        if ($next_parent_id === false) {
          break;
        }
        else {
          $parent_id = $next_parent_id;
        }
      } // for
    }
    else {
      $Tvw->tr();
      $Tvw->td( "&nbsp;" );
    }

    // Render the menu viewer table..
    $s .= $Tvw->render();
    debug_trace();
    return $s;
  } // html

} // rowmenu class

// ----------------------------------------------------------------------
?>