#!/usr/bin/php -q
<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    axyl-databases-upgrade.php                              */
/* Author:      Paul Waite                                              */
/* Description: Attempts to upgrade all Axyl databases on the local     */
/*              system.                                                 */
/*                                                                      */
/* ******************************************************************** */
/* Script setup */
$PROGNAME      = "axyl-databases-upgrade";
$PROGVER       = "1.0";
$DEBUGMODE     = false;    // Verbose logging option
$ECHO_LOGGING  = true;     // echo log messages to console too
$POSTGRES_REQD = true;     // Require Postgres DB installed
$ROOT_REQD     = false;    // Require 'root' user
$USER_INPUT    = true;     // User will be entering responses
$AXYLROOT_REQD = false;    // Script must be run in an Axyl website root

// ----------------------------------------------------------------------
// READ IN THE AXYL CONFIG - required
// ----------------------------------------------------------------------
$CONF_FILE = "/etc/axyl/axyl.conf";
$conf_content = readallfile($CONF_FILE);
if ($conf_content != "") {
  if (preg_match("/AXUSER=([\S]+)\s/", $conf_content, $matches)) {
    $AXUSER = $matches[1];
  }
  if (preg_match("/AXYL_HOME=([\S]+)\s/", $conf_content, $matches)) {
    $AXYL_HOME = $matches[1];
  }
  if (preg_match("/AXYL_DATA=([\S]+)\s/", $conf_content, $matches)) {
    $AXYL_DATA = $matches[1];
  }
  if (preg_match("/AXYL_LOGS=([\S]+)\s/", $conf_content, $matches)) {
    $AXYL_LOGS = $matches[1];
  }
}
else {
  die(
    "File $CONF_FILE was invalid. Have you installed Axyl?\n"
  . "If you have, then have you remembered to set it up properly, by\n"
  . "runing the script AXYL_HOME/install/setup-axyl.sh?\n"
  );
}
if ($AXYL_HOME != "") {
  if (!is_dir($AXYL_HOME)) {
    die("AXYL_HOME is undefined. Check your $CONF_FILE.");
  }
}
else {
  die(
    "The Axyl configuration file at $CONF_FILE does not define your\n"
  . "AXYL_HOME. This script cannot proceed without that setting.\n"
  . "Please check your Axyl configuration.\n"
  );
}

// ----------------------------------------------------------------------
// INCLUDE COMMON PRE-AMBLE & FUNCTIONS
// ----------------------------------------------------------------------
include_once("$AXYL_HOME/scripts/inc/common.inc");

// This function is required before any others are read in. It is
// used to read the Axyl configuration at the start.
function readallfile($path) {
  $content = "";
  if (file_exists($path)) {
    $fp = fopen($path, "r");
    if ($fp !== false) {
      $content = fread($fp, filesize($path));
      fclose($fp);
    }
  }
  return $content;
}

// ----------------------------------------------------------------------
// AXYL DATABASE UPGRADE
$stamp = timestamp_to_displaydate("M jS g:ia", time());
logit("\nAXYL DATABASES UPGRADE\n");
logit("$stamp\n\n");
logit("This upgrade process will attempt to upgrade all Axyl databases\n");
logit("on this machine to the latest Axyl core schema. The process will\n");
logit("be done in 'no-drops' mode, which only adds missing schema.\n\n");
$doit = true;
echo "Continue? Y or N [Y] :";
$ans = get_user_input();
if ($ans != "" && $ans != "Y" && $ans != "y") {
  $doit = false;
}

if ($doit) {
  logit("Axyl database upgrade begins..");

  // Get database names list..
  logit(" -- examining Axyl application environment..");

  // PROCESS ALL DATABASES
  // Get websites..
  $websites = get_axyl_websites();
  if (count($websites) > 0) {

    foreach ($websites as $app_name => $website_info) {
      $dbtype = $website_info[0];
      $dbname = $website_info[1];
      $dbuser = $website_info[2];
      $dbpass = $website_info[3];
      $dbhost = $website_info[4];
      $dbport = $website_info[5];
      $site_docroot = $website_info[6];

      logit(" -- ");
      logit(" -- application: $app_name");
      if ($dbtype == 1 || $dbtype == "postgres") {
        $dbid = connect_pgdb(
                  "template1",
                  $dbuser,
                  $dbpass != "none" ? $dbpass : "",
                  $dbhost != "direct" ? $dbhost : "",
                  $dbhost != "direct" ? $dbport : ""
                  );
        if ($dbid !== false) {
          $database_names = array();
          $rid = query_pgdb($dbid, "SELECT datname FROM pg_database WHERE datname='axyl_core'");
          if ($rid !== false) {
            $totrows = numrows_pgdb($rid);
            if ($totrows == 1) {
              $rid = query_pgdb($dbid, "DROP DATABASE axyl_core");
              if ($rid !== false) {
                logit(" -- dropped existing axyl_core");
              }
            }
          }
          $rid = query_pgdb($dbid, "CREATE DATABASE axyl_core");
          if ($rid !== false) {
            logit(" -- created new axyl_core");
          }
          disconnect_pgdb($dbid);
        }

        // Define latest Axyl Core schema..
        $dbid = connect_pgdb(
                  "axyl_core",
                  $dbuser,
                  $dbpass != "none" ? $dbpass : "",
                  $dbhost != "direct" ? $dbhost : "",
                  $dbhost != "direct" ? $dbport : ""
                  );
        if ($dbid !== false) {
          logit(" -- connected to axyl_core");
          $coresql = readallsql("$AXYL_HOME/db/postgres/axyl_core.sql");
          $trigsql = readallsql("$AXYL_HOME/db/postgres/axyl_trig.sql");
          $rid = query_pgdb($dbid, $coresql);
          if ($rid !== false) {
            logit(" -- axyl_core schema now loaded");
          }
          else {
            abort(" -- error loading axyl_core schema");
          }
          $rid = query_pgdb($dbid, $trigsql);
          if ($rid !== false) {
            logit(" -- axyl_core triggers and procedures loaded");
          }
          else {
            abort(" -- error loading axyl_core triggers & procs");
          }
          disconnect_pgdb($dbid);
        }

        // Connect to database to be upgraded..
        $dbid = connect_pgdb(
                  $dbname,
                  $dbuser,
                  $dbpass != "none" ? $dbpass : "",
                  $dbhost != "direct" ? $dbhost : "",
                  $dbhost != "direct" ? $dbport : ""
                  );
        if ($dbid !== false) {
          $rid = query_pgdb($dbid, "SELECT * FROM pg_class WHERE relname='ax_user'");
          if ($rid !== false) {
            $totrows = numrows_pgdb($rid);
            if ($totrows == 1) {
              // Note that we loop because sometimes the diff have problems with
              // ordering which requires them to be iterated a few times..
              $upgrading_schema = true;
              while ($upgrading_schema) {
                logit("");
                echo "upgrade Axyl database $dbname? Y or N [Y] :";
                $ans = get_user_input();
                if ($ans == "" || $ans == "Y" || $ans == "y") {

                  // Site closure - if existing site..
                  if (is_dir($site_docroot) && file_exists("$site_docroot/notclosed.php")) {
                    copy("$site_docroot/notclosed.php", "$site_docroot/closed.php");
                    $site_was_closed = file_exists("$site_docroot/closed.php");
                    logit(" -- website was closed");
                  }
                  else {
                    $site_was_closed = false;
                  }

                  // Perform upgrade now..
                  logit(" -- upgrading $dbname");
                  $outputlines = array();
                  $dbdiffcmd = "$AXYL_HOME/scripts/dbdiff.php"
                            . " --target=$dbname"
                            . " --ref=axyl_core"
                            . " --no-drops"
                            . " --user=$dbuser"
                            . ($dbpass != "none" ? " --password=$dbpass" : "")
                            . ($dbhost != "direct" ? " --targhost=$dbhost" : "")
                            . ($dbhost != "direct" ? " --targport=$dbport" : "")
                            . ($dbhost != "direct" ? " --refhost=$dbhost" : "")
                            . ($dbhost != "direct" ? " --refport=$dbport" : "");
                  exec($dbdiffcmd, $outputlines, $ret);

                  if ($ret == 0) {
                    $loglines = "";
                    $diffs = array();
                    foreach($outputlines as $line) {
                      if (trim($line) != "" && substr($line, 0, 2) != "--") {
                        if (trim($line) != "") {
                          $loglines .= trim($line) . "\n";
                        }
                        $line = preg_replace("/;$/", "^!^", $line);
                        $diffs[] = trim($line);
                      }
                    }
                    if ($loglines != "") {
                      logit(" -- dbdiff output begins +=+=+=+=+=+=+=+=+=++=+=+=+=+=+=+=+=+=+\n");
                      logit($loglines);
                      logit(" -- dbdiff output ends   +=+=+=+=+=+=+=+=+=++=+=+=+=+=+=+=+=+=+\n");
                    }
                    $diffsql = trim(implode(" ", $diffs));
                    if ($diffsql != "") {
                      echo "\napply the above SQL to upgrade $dbname (nothing was dropped)? Y or N [Y] :";
                      $ans = get_user_input();
                      if ($ans == "" || $ans == "Y" || $ans == "y") {
                        $ok = true;
                        $diffs = explode("^!^", $diffsql);
                        foreach ($diffs as $diff) {
                          if ($diff != "") {
                            $rid = query_pgdb($dbid, $diff);
                            if ($rid === false) {
                              $ok = false;
                            }
                          }
                        } // foreach
                        // Final check that diff is now reporting identical schemas..
                        if ($ok) {
                          exec($dbdiffcmd, $outputlines, $ret);
                          foreach($outputlines as $line) {
                            if (trim($line) != "" && substr($line, 0, 2) != "--") {
                              $ok = false;
                              break;
                            }
                          }
                        }

                        // Report results..
                        if ($ok) {
                          $upgrading_schema = false;
                          logit(" -- done");
                        }
                        else {
                          logit(" -- failed - see any messages above for clues");
                          logit(" -- NB: failure might just be a case of statement ordering problems so");
                          logit(" -- please go around this loop until the SQL diffs don't change.");
                        }
                      }
                      else {
                        logit(" -- upgrade SQL was aborted (user choice)");
                        logit(" -- warning: not upgrading the database compromises the upgrade.");
                        $upgrading_schema = false;
                      }
                    }
                    else {
                      logit(" -- $dbname is up to date");
                      $upgrading_schema = false;
                    }
                  } // fixups to make

                  // Open website, if it was closed..
                  if ($site_was_closed) {
                    if (file_exists("$site_docroot/closed.php")) {
                      unlink("$site_docroot/closed.php");
                      logit(" -- website was re-opened");
                    }
                  }
                }
                else {
                  logit(" -- user declined upgrade");
                  $upgrading_schema = false;
                }
              } // while
            } // is Axyl db
          }
        } // can connect to db..
      } // dbtype
    } // foreach website
  } // got websites

} // doit

?>