#!/bin/sh
# Install a POSTGRESQL database for an Axyl website.

# NOTE: This is not normally run standalone. The main DB installation
# script 'install/install-db.sh normally runs this.
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

# The database type this install script is for. This is just
# for messages and display..
DBTYPE="PostgreSQL"

# INCOMING PARAMETERS
#  $1 - INSTALLTYPE
#  $2 - DBNAME
#  $3 - DBUSER
#  $4 - DBPASSWD ('none' means a blank password)
#  $5 - DBHOST ('direct' means a local database server)
#  $6 - DBPORT
#  $7 - META ('y' or 'n')
INSTALLTYPE=
DBNAME=
DBUSER=
DBPASSWD=
DBHOST=
DBPORT=5432
META=n
[ ! -z $1 ] && INSTALLTYPE=$1
[ ! -z $2 ] && DBNAME=$2
[ ! -z $3 ] && DBUSER=$3
[ ! -z $4 ] && DBPASSWD=$4
[ ! -z $5 ] && DBHOST=$5
[ ! -z $6 ] && DBPORT=$6
[ ! -z $7 ] && META=$7
if [ "$INSTALLTYPE" = "" -o "$AXUSER" = "" -o "$AXYL_HOME" = "" -o "$AXYL_DATA" = "" ]
then
  echo "ERROR: $DBTYPE install-db.sh: parameters."
  echo "usage: install-db.sh empty|demo dbname dbuser [dbpasswd] [dbhost] [dbport]"
  exit 1
fi

echo " $DBTYPE database - $INSTALLTYPE version"
echo " database name: $DBNAME"
echo " connecting as: $DBUSER"
[ "$DBPASSWD" != "none" ] && echo " password: $DBPASSWD"
[ "$DBHOST" != "direct" ] && echo " remote host: $DBHOST"
[ "$DBHOST" != "direct" ] && echo " port: $DBPORT"
echo ""

# We require Postgres to be locally installed, at least as postgresql-client
# even if no servers are created locally. This is based on the standard
# Debian location, with a few likely fallbacks..
PG_BIN=
PG_PATHS="/usr/lib/postgresql/bin /usr/bin /usr/local/pgsql/bin"
for pg_path in ${PG_PATHS}
do
  if [ -x ${pg_path}/psql ]
  then
    PG_BIN=$pg_path
    echo "postgres binaries found in $PG_BIN"
    break
  fi
done

# Check we can find Postgres executables..
if [ -z $PG_BIN ]
then
  echo "ERROR: Postgres not installed, or not in any of the following paths:"
  echo "  $PG_PATHS"
  echo "Even if you are intending to use a remote machine for your databases,"
  echo "the 'postgresql-client' package is required to access it from this one."
  echo "Either install postgresql (or at least postgresql-client) on this"
  echo "machine, or edit this script and change the path(s) to suit your"
  echo "particular system."
  exit 1
fi

# LOCALHOST INSTALLATION
if [ "$DBHOST" = "direct" -o "$DBHOST" = "localhost" -o "$DBHOST" = "127.0.0.1" ]
then
  # Try to detect the Postgres security settings, since these will determine
  # what we can and cannot do in the way of configuration from here.
  # First, let's locate the config files..
  PG_CONF=
  PG_CONFPATHS="/etc/postgresql /var/lib/postgresql/data"
  for pg_confpath in ${PG_CONFPATHS}
  do
    if [ -e ${pg_confpath}/pg_hba.conf ]
    then
      PG_CONF=$pg_confpath
      echo "postgres config files found in $PG_CONF"
    fi
  done

  if [ -z $PG_CONF ]
  then
    echo "WARNING: Postgres config files not found, or not in any of the following paths:"
    echo "  $PG_CONFPATHS"
    echo "This is just because this install script probably doesn't grok your installation"
    echo "so don't panic. However it also means that we can't suss out your Postgres settings"
    echo "for security, so it's down to you I'm afraid. You need to go and make sure that"
    echo "your 'pg_hba.conf' permissions will allow database access for this (and future)"
    echo "Axyl databases, connecting locally. If you don't know what that means, you should"
    echo "talk to your local Postgres database guru!"
    echo -n "Press ENTER to continue.."
    read ANS
  else
    PG_HBA=${PG_CONF}/pg_hba.conf
    A=`grep -e '^local[[:space:]]*all [[:space:]]*all [[:space:]]*trust' ${PG_HBA}`
    if [ "$A" = "" ]
    then
      echo "POSTGRES SECURITY SETTINGS"
      echo "Your current Postgres security settings do not have a 'trust' set up for"
      echo "local database connections. This will probably prevent this script from"
      echo "doing its job. The script can fix this for you by putting the following"
      echo "lines into your ${PG_HBA} file for you:"
      echo "  local all  all                        trust"
      echo "  host  all  all  127.0.0.1  255.0.0.0  trust"
      echo ""
      echo "Make these changes now?"
      echo -n "Enter Y or N [Y] :"
      read ANS
      [ "$ANS" = "" ] && ANS="y"
      if [ "$ANS" = "y" -o "$ANS" = "Y" -o "$ANS" = "yes" ]
      then
        PG_HBA_NEW=${PG_HBA}.new
        (while read line ; do
          echo $line >> $PG_HBA_NEW
          A=`echo $line | grep -e '# TYPE'`
          if [ "$A" != "" ] ; then
            echo "local all  all                        trust" >> $PG_HBA_NEW
            echo "host  all  all  127.0.0.1  255.0.0.0  trust" >> $PG_HBA_NEW
          fi
        done) < $PG_HBA
        TSTAMP=`date +'%F_%H%M%S'`
        mv $PG_HBA ${PG_HBA}.${TSTAMP}
        mv $PG_HBA_NEW $PG_HBA
        echo "security changes made"
        echo "restarting Postgresql.."
        /etc/init.d/postgresql reload
      else
        echo "Ok, no changes - installation will proceed, but you are on your own!"
      fi
    else
      echo "Postgresql security settings are ok"
    fi
  fi

  # Get database version..
  pg_version="`su postgres -c \"psql template1 -qAtc 'select version()'\" | cut -f2 -d' '`"
  echo "postgres version is $pg_version"

  # User should have an account on the machine too..
  mkuser $DBUSER users /home/$DBUSER /bin/bash Axyl DB Admin

  # Postgres version - is it 'old' or 'new' with the threshold
  # being v7.3 and above..
  majv=`echo $pg_version|cut -d'.' -f1`
  minv=`echo $pg_version|cut -d'.' -f2`
  PG_OLDVER=yes
  [ $majv -gt 7 ] && PG_OLDVER=no
  [ $majv -eq 7 -a $minv -ge 3 ] && PG_OLDVER=no

  # Create the appropriate database user
  pattern=$(basename $0).XXXXXX
  tmp1=$(mktemp /tmp/$pattern)
  tmp2=$(mktemp /tmp/$pattern)
  if [ $PG_OLDVER = "yes" ]
  then
    if su -s /bin/sh postgres -c "createuser --createdb --no-adduser $DBUSER" 1> $tmp1 2> $tmp2 \
      && [ "$(head -1 $tmp1)" = 'CREATE USER' ] \
      || grep -q "user name \"$DBUSER\" already exists$" $tmp2
    then
      # Creation OK or user already existing -- no problem here
      echo "postgres '$DBUSER' user present and correct"
      rm -f $tmp1 $tmp2
    else
      echo ""
      echo "Cannot create PostgreSQL user...  This shouldn't have happened."
      echo "Maybe a problem in your PostgreSQL configuration?"
      echo "Please report a bug to the Axyl developers, with the following:"
      echo "createuser's STDOUT:"
      cat $tmp1
      echo "createuser's STDERR:"
      cat $tmp2
      rm -f $tmp1 $tmp2
      exit 1
    fi
  else
    if su -s /bin/sh postgres -c "/usr/bin/psql template1" &> /dev/null <<-EOF
CREATE USER $DBUSER WITH PASSWORD '$DBPASSWD' CREATEDB;
EOF
    then
      echo "postgres '$DBUSER' user present and correct"
      rm -f $tmp1 $tmp2
    else
      echo ""
      echo "ERROR: Cannot create PostgreSQL user '${DBUSER}'."
      echo "Maybe a problem in your PostgreSQL configuration?"
      echo "Please report a bug to the Axyl developers, with the following:"
      echo "CREATE USER's STDOUT:"
      cat $tmp1
      echo "CREATE USER's STDERR:"
      cat $tmp2
      rm -f $tmp1 $tmp2
      exit 1
    fi
  fi

  # Create the appropriate database
  tmp1=$(mktemp /tmp/$pattern)
  tmp2=$(mktemp /tmp/$pattern)
  if su -s /bin/sh postgres -c "createdb --username ${DBUSER} --encoding UTF-8 ${DBNAME}" 1> $tmp1 2> $tmp2 \
    && [ "$(head -1 $tmp1)" = 'CREATE DATABASE' ]
  then
    # Creation OK..
    echo "database '$DBNAME' created.."
    rm -f $tmp1 $tmp2
  else
    if grep -q "\"$DBNAME\" already exists" $tmp2
    then
      echo ""
      echo "ERROR: PostgreSQL database '${DBNAME}' already exists. As a"
      echo "failsafe measure we will not drop it. Please do so yourself"
      echo "and then re-run this script to install Axyl."
      rm -f $tmp1 $tmp2
      exit 1
    else
      echo ""
      echo "ERROR: Cannot create PostgreSQL database '${DBNAME}'."
      echo "Maybe a problem in your PostgreSQL configuration?"
      echo "Please report a bug to the Axyl developers, with the following:"
      echo "createdb's STDOUT:"
      cat $tmp1
      echo "createdb's STDERR:"
      cat $tmp2
      rm -f $tmp1 $tmp2
      exit 1
    fi
  fi

  pattern=$(basename $0).XXXXXX
  tmp1=$(mktemp /tmp/$pattern)
  tmp2=$(mktemp /tmp/$pattern)
  PG_LANG=${PG_BIN}/enable_lang
  if [ ! -x $PG_LANG ]
  then
    PG_LANG=${PG_BIN}/createlang
  fi
  if [ -x $PG_LANG ]
  then
    if su -s /bin/sh postgres -c "${PG_LANG} plpgsql ${DBNAME}" 1> $tmp1 2> $tmp2 \
      || grep -q "plpgsql added to $DBNAME" $tmp1 \
      || grep -q "plpgsql is already enabled in $DBNAME" $tmp1
    then
      # Creation OK or language already set up -- no problem here
      echo "plpgsql is enabled.."
      rm -f $tmp1 $tmp2
    else
      echo ""
      echo "Cannot enable the PLPGSQL language in the database."
      echo "Maybe a problem in your PostgreSQL configuration?"
      echo "Please report a bug to the Axyl developers, with the following:"
      echo "enable_lang's STDOUT:"
      cat $tmp1
      echo "enable_lang's STDERR:"
      cat $tmp2
      rm -f $tmp1 $tmp2
      exit 1
    fi
  else
    echo ""
    echo "Warning: Postgres $PG_LANG not found. Please ensure"
    echo "that the 'plpgsql' language is created/enabled in your database."
  fi
else
  PSQL=${PG_BIN}/psql
  if [ -x $PSQL ]
  then
    echo "Installing on remote database server --> ${DBHOST}:${DBPORT}"
    echo "(assuming Postgresql, user and passwords are already set up)"
    if [ "$DBPASSWD" = "none" ]
    then
      $PSQL --username $DBUSER --dbname template1 --host $DBHOST --port $DBPORT --command "CREATE DATABASE $DBNAME"
    else
      echo "Enter password, when prompted, as: $DBPASSWD"
      $PSQL --username $DBUSER --password --dbname template1 --host $DBHOST --port $DBPORT --command "CREATE DATABASE $DBNAME"
    fi
  else
    echo "ERROR: $PSQL not found."
    echo "to set up a Postgres database remotely, you must at least have the"
    echo "'psql' utility available locally. For Debian systems this requires"
    echo "installation of the 'postgresql-client' package. For other systems,"
    echo "download the applicable RPM, or tarball from the website:"
    echo "http://www.postgresql.org/"
    exit 1
  fi
fi

# Install/upgrade the database contents (tables and data)
PSQL=${PG_BIN}/psql
if [ -x $PSQL ]
then
  DB=${AXYL_HOME}/db/postgres
  DBSCHEMA=${DB}/axyl_core.sql
  DBTRIG=${DB}/axyl_trig.sql
  DBCOREDATA=${DB}/axyl_core_data.sql
  DBMETADATA=${DB}/axyl_meta_data.sql
  DBDEMO=${DB}/axyl_demo.db
  DBTEMP=/tmp/axyldb.sql
  if [ ! -f $DBMETADATA ]
  then
    META=n
  fi

  # Create database to load. We create a temporary file with the SQL in
  # it to define the database, potentially with fixed-up Lucene port..
  if [ "$LUCENE_PORT" != "" ]
  then
    CHARCNT=`echo $LUCENE_PORT | wc --chars`
    CHARCNT=`expr $CHARCNT - 1`
  fi

  case "$INSTALLTYPE" in
    empty)
      if [ "$LUCENE_PORT" != "" ]
      then
        sed -e "s/s:5:\"22222\";/s:$CHARCNT:\"$LUCENE_PORT\";/" $DBCOREDATA > $DBTEMP
      else
        sed -e "s/\"Lucene_Site_Indexing\";b:1;/\"Lucene_Site_Indexing\";b:0;/" $DBCOREDATA > $DBTEMP
      fi
      ;;
    demo)
      if [ "$LUCENE_PORT" != "" ]
      then
        sed -e "s/s:5:\"22222\";/s:$CHARCNT:\"$LUCENE_PORT\";/" $DBDEMO > $DBTEMP
      else
        sed -e "s/\"Lucene_Site_Indexing\";b:1;/\"Lucene_Site_Indexing\";b:0;/" $DBDEMO > $DBTEMP
      fi
      ;;
  esac

  # Now load the database, either direct or remotely..
  if [ "$DBHOST" = "direct" ]
  then
    case "$INSTALLTYPE" in
      empty)
        # Load core schema and then core data..
        $PSQL --username $DBUSER --dbname $DBNAME --file $DBSCHEMA
        $PSQL --username $DBUSER --dbname $DBNAME --file $DBTRIG
        $PSQL --username $DBUSER --dbname $DBNAME --file $DBTEMP
        if [ "$META" = "y" ]
        then
          $PSQL --username $DBUSER --dbname $DBNAME --file $DBMETADATA
        fi
        ;;
      demo)
        # For the demo, we load a complete database dump which
        # includes the schema and data..
        $PSQL --username $DBUSER --dbname $DBNAME --file $DBTEMP
        ;;
    esac
  else
    case "$INSTALLTYPE" in
      empty)
        # Load core schema and then core data..
        $PSQL --username $DBUSER --dbname $DBNAME --host $DBHOST --port $DBPORT --file $DBSCHEMA
        $PSQL --username $DBUSER --dbname $DBNAME --host $DBHOST --port $DBPORT --file $DBTRIG
        $PSQL --username $DBUSER --dbname $DBNAME --host $DBHOST --port $DBPORT --file $DBTEMP
        if [ "$META" = "y" ]
        then
          $PSQL --username $DBUSER --dbname $DBNAME --host $DBHOST --port $DBPORT --file $DBMETADATA
        fi
        ;;
      demo)
        # For the demo, we load a complete database dump which
        # includes the schema and data..
        $PSQL --username $DBUSER --dbname $DBNAME --host $DBHOST --port $DBPORT --file $DBTEMP
        ;;
    esac
  fi
  #rm -f $DBTEMP
  echo "finished"
else
  echo ""
  echo "ERROR: Could not import Axyl data - $PSQL not found."
  echo "to set up data, you must have the 'psql' utility available locally."
  echo "For Debian systems this requires installation of the 'postgresql-client'"
  echo "package. For other systems, download the applicable RPM, or tarball from"
  echo "the Postgres website: http://www.postgresql.org/"
  exit 1
fi

# END