<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    response-defs.php                                       */
/* Author:      Paul Waite                                              */
/* Description: Definitions for managing the RESPONSE object.           */
/*                                                                      */
/* ******************************************************************** */
/** @package core */

// -----------------------------------------------------------------------
// AUTHENTICATION OPTIONS
/** Authentication option: Redundant case, no authentication */
define("NO_AUTHENTICATION",   0 );
/** Authentication option: Browser popup form */
define("HTTP_AUTHENTICATION", 1 );
/** Authentication option: Username/password from custom form fields */
define("FORM_AUTHENTICATION", 2 );

// -----------------------------------------------------------------------
// RESPONSE COMPRESSION OPTIONS
/** Webpage compression: None. Just straight HTML */
define("NO_COMPRESSION",      0 );
/** Webpage compression: Use the builtin Php compression system. Requires Php >= v4.04 */
define("BUILTIN_COMPRESSION", 1 );
/** Webpage compression: Use custom Phplib compression. For Php < v4.04 */
define("CUSTOM_COMPRESSION",  2 );

// -----------------------------------------------------------------------
// FAILED AUTHENTICATION RESPONSE OPTIONS
// Here's what we can do when the user fails authentication.

/** Failed authentication: Die, with 'not authorised' message. */
define("AUTHFAIL_DIE_MSG",    0);
/** Failed authentication: Die silently. */
define("AUTHFAIL_DIE_SILENT", 1);
/** Failed authentication: Re-direct to specified URL. */
define("AUTHFAIL_REDIRECT",   2);
/** Failed authentication: Welcome the user as a guest instead. */
define("AUTHFAIL_GUEST",      3);

// -----------------------------------------------------------------------
// KEEP OPTIONS
/** Enable keeping variables across requests using Php session handling */
define("KEEP_ENABLED",    true );
/** Disable keeping variables across requests using Php session handling */
define("KEEP_DISABLED",   false );

// -----------------------------------------------------------------------
// METADATA OPTIONS
/** Enable metadata editing and generation enhancements */
define("METADATA_ENABLED",    true );
/** Disable metadata editing and generation enhancements */
define("METADATA_DISABLED",   false );

// -----------------------------------------------------------------------
// MULIT-LANGUAGE OPTIONS
/** Enable multi-language extensions */
define("MULTILANG_ENABLED",    true );
/** Disable multi-language extensions */
define("MULTILANG_DISABLED",   false );

// -----------------------------------------------------------------------
// BROWSER MAKES
// We recognise Internet Explorer, Mozilla (includes Netscape), and then
// class all the others under the "other" umbrella. Other browsers may
// be added in the future as requirements dictate.

/** Microsoft internet Explorer */
define("BROWSER_IE",      "msie");
/** Netscape, Mozilla */
define("BROWSER_MOZILLA", "mozilla");
/** Netscape only, this is Mozilla <5.0 */
define("BROWSER_NETSCAPE", "netscape");
/** Opera */
define("BROWSER_OPERA",    "opera");
/** Browser detection: Any WAP phone browser */
define("BROWSER_PHONE",    "phone");
/** Browser detection: Other browsers */
define("BROWSER_OTHER",   "other");
/** Browser detection: No browser; command line interface */
define("BROWSER_NONE",   "none");

// -----------------------------------------------------------------------
// DEFAULT DTD's
/** These Document Type Definition specifier strings are the defaults
* which are used in the event that (a) they are not specified in the
* application.php file, and (b) not specified in the template(s).
*/
$DEFAULT_DTD = array(
 BROWSER_TYPE_HTML => "<!DOCTYPE HTML PUBLIC \"-//W3C//DTD HTML 4.01 Transitional//EN\">",
 BROWSER_TYPE_WML  => "<!DOCTYPE wml PUBLIC \"-//WAPFORUM//DTD WML 1.1//EN\" \"http://www.wapforum.org/DTD/wml_1.1.xml\">"
);

// -----------------------------------------------------------------------
/** List of recognition patterns for web-browsers. */
$WEB_BROWSERS = "(wget)"
              . "|(mozilla)"
              . "|(opera)"
              . "|(lynx)"
              . "|(msie)"
              . "|(konqueror)"
              . "|(libwww-perl)"
              ;
/** List of recognition patterns for WAP-phones. */
$WAP_PHONES   = "(UP\.Browser)"
              . "|(Nokia)"
              . "|(Sharp)"
              . "|(Panasonic)"
              . "|(Ericsson)"
              . "|(SonyEricsson)"
              ;

// -----------------------------------------------------------------------
// SET APPLICATION DEFINES & GLOBALS
// At this point the document root should be set, but if not we fall back
// to the current working directory..
if (empty($DOCUMENT_ROOT)) {
  $DOCUMENT_ROOT = getcwd();
}
if (file_exists("$DOCUMENT_ROOT/application.xml")) {
  /** Application management classes */
  include_once("application-defs.php");
  $application = new application("$DOCUMENT_ROOT/application.xml");
  if ($application->valid) {
    foreach ($application->definitions as $defname => $defval) {
      define($defname, $defval);
    }
    foreach ($application->globals as $globname => $globval) {
      global $globname;
      $$globname = $globval;
    }
  }
}

// -----------------------------------------------------------------------
// SITE CLOSED CHECK
// We implement a nice simple method of closing a website without
// recourse to using the database which may of course be down for some
// maintenance. If this file exists, we always display it, and exit..
if (file_exists("$DOCUMENT_ROOT/closed.php")) {
  include("$DOCUMENT_ROOT/closed.php");
  exit;
}

// -----------------------------------------------------------------------
// REQUIRED SYSTEM INCLUDES
/** Various constants */
include_once("constants.php");
/** MIME types and categories */
include_once("mime-types.php");
/** Simple anti-hack filtering */
include_once("antihack.php");
/** Basic renderable classes */
include_once("renderable.php");
/** Filesystem access classes */
include_once("file-defs.php");
/** Database query classes */
include_once("query-defs.php");
/** Built-in debugger */
include_once("debugger.php");
/** User identification and management */
include_once("user-defs.php");
/** Misc. utility functions */
include_once("utils.php");
/** User session management */
include_once("session-defs.php");
/** Generic configuration classes */
include_once("configuration-defs.php");
/** Database access classes */
include_once("database-defs.php");
/** Php globals management */
include_once("globals-defs.php");
/** Axyl keep-alive classes */
include_once("keep-defs.php");
/** Core webpage classes */
include_once("webpage-defs.php");

// -----------------------------------------------------------------------
/**
* THE RESPONSE CLASS
* This object is a container for all things to do with the response which
* we will send back to the requester (the client browser). It is a
* descendant of the webpage class, and before that, the session and the
* user classes.
*
* If you use the Phplib framework then a response object is automatically
* instantiated for you. This object is assigned to the global variable
* called $RESPONSE. The framework also sets up the response parameters,
* and then calls the activate() method.
* @package core
*/
class response extends webpage {
  // Public
  /** The make of browser */
  var $browser;
  /** The version of browser requesting this response */
  var $browser_version = 0;
  /** The type of browser requesting this response */
  var $browser_type = BROWSER_TYPE_HTML;
  /** The user-agent */
  var $user_agent = "";
  /** The remote IP address */
  var $remote_address = "";
  /** The browser accept string, as submitted by user-agent */
  var $accept = "";
  /** The browser accept-encoding string */
  var $accept_encoding = "";
  /** The browser accept-language string */
  var $accept_language = "";
  /** Whether we are in multi-language mode or not, If this is true then
      we enable facilities to set language tags on webpages, and also
      deliver http content encoded in UTF-8. */
  var $multilang = false;
  /** IDs of all languages contained in this page */
  var $languages = array();
  /** Charset for this page */
  var $charset = "ISO-8859-1";
  /** Page ID - key to ax_sitepage record */
  var $page_id;
  /** Whether we have multi-byte string fns available */
  var $mbstring_avail = false;
  /** Datasources associated with this response */
  var $datasource;
  /** Metadata edit/generation mode we are in */
  var $metadata_mode = METADATA_DISABLED;
  /** The URL of this website */
  var $site_url;
  /** The host of this website ie. 'foo.thingy.co.nz' */
  var $http_host;
  /** Path to the site document root */
  var $site_docroot;
  /** Path to the requested script/page */
  var $requested;
  /** Query string (if any) supplied to requested page */
  var $requested_query;

  // Private
  /** Buffering option to use
      @access private */
  var $buffering_mode = BUFFERED;
  /** Array of hosts we wish to connect persistently to
      @access private */
  var $persistent_hosts = array();
  /** Compression type to use for response content
      @access private */
  var $compression_type;
  /** Minimum size in bytes before invoking compression
      @access private */
  var $compression_minsize;
  /** The debugger for this response
      @access private */
  var $debugger;
  /** Type of authentication in effect
      @access private */
  var $auth_type;
  /** Option to take on auth failure
      @access private */
  var $auth_fail_option;
  /** URL to redirect to on auth failure
      @access private */
  var $auth_fail_redirect;
  /** Our keep enabled flag (default enabled)
      @access private */
  var $keep_enabled = KEEP_ENABLED;
  /** Our keep. Keeps variables alive in session
      @access private */
  var $keep;
  /** Globals object. Manages the global vars in this response
      @access private */
  var $globals;
  /** Dynamic page expiry in seconds. Time to allow a dynamic page to
      'live' in user browser. Note this defaults to -1, which means
      'in the past', and which causes us to force the users browser
      to NOT cache the page at all, and revalidate every time.
      @access private */
  var $page_expiry_secs = -1;
  /** Array of DTD specifiers for this response. @see set_dtd()
      @access private */
  var $DTD;
  // ...................................................................
  /**
  * Constructor
  * Create a new response.
  * One of these objects must be created to respond to each request
  * from the user agent for a webpage. This object manages that response
  * and is a central marshalling point for all data and functions
  * associated with that process. A response object is automatically
  * created for you at the bottom of this module, and is assigned to
  * the global variable $RESPONSE.
  */
  function response() {
    // Initialise DTD array..
    global $DEFAULT_DTD;
    $this->DTD = $DEFAULT_DTD;
    $this->mbstring_avail = extension_loaded("mbstring");

    // Create the debugger..
    $this->debugger = new webdebugger();

    // Create the globaliser..
    $this->globals = new globals();

    // Globalise all server vars..
    $this->register(
        "DOCUMENT_ROOT,"        // List of Web Server vars to globalise.
      . "^HTTP_.*,"             // Note the use of Perl regex to specify
      . "^REMOTE_.*,"           // groups of server vars to globalise.
      . "^SERVER_.*,"
      . "PATH,"
      . "SCRIPT_FILENAME,"
      . "PHP_SELF,"
      . "UNIQUE_ID,"
      . "GATEWAY_INTERFACE,"
      . "SERVER_PROTOCOL,"
      . "REQUEST_METHOD,"
      . "QUERY_STRING,"
      . "REQUEST_URI,"
      . "SCRIPT_NAME,"
      . "PATH_TRANSLATED,"
      . "PHP_SELF"
      ,
      "server"
      );

    // Globalise vars used for login/logout..
    $this->register(
        "tbxUsername,user"
      . "tbxPassword,pass"
      . "tbxLogoff,"
      . "chkRememberMe,"
      . "authid,"
      . "auth_code,"
      . "admin_auth_code,"
      . "PHP_AUTH_USER,"
      . "PHP_AUTH_PW,"
      . "MAX_FILE_SIZE,"
      . "cachecontrol,"
      . "^_.*,"
      . "^recmaint.*,"
      . "theme"
      ,
      "get,post"
      );

    // Do the globalisation thing..
    $this->globalise();

    // Some globals we need to use right now..
    global $HTTP_USER_AGENT;
    global $HTTP_HOST;
    global $REMOTE_ADDR;
    global $PHP_SELF;
    global $SCRIPT_NAME;
    global $QUERY_STRING;
    global $WEB_BROWSERS, $WAP_PHONES;
    global $DOCUMENT_ROOT;

    // Initialise some vars..
    $this->initialise();

    // BROWSER TYPE IDENTIFICATION..
    $this->user_agent = $HTTP_USER_AGENT;
    $this->remote_address = $REMOTE_ADDR;
    $this->browser_type = $this->get_browser_type();

    // WEB-BROWSER IDENTIFICATION..
    switch ($this->browser_type) {
      case BROWSER_TYPE_CLI:
        $this->browser = BROWSER_NONE;
        $this->browser_version = "";
        // The site Domain and URL..
        $this->http_host = trim(`hostname --fqdn`);
        $this->site_url = "http://$this->http_host";
        $this->site_docroot = getcwd();
        // For debugging, set output mode..
        $this->debugger->debug_output(DBG_O_CLI);
        break;

      case BROWSER_TYPE_WML:
      case BROWSER_TYPE_WMLUP:
        $this->browser = BROWSER_PHONE;
        $this->browser_version = "";
        // The site Domain and URL..
        $this->http_host = $HTTP_HOST;
        $this->site_url = "http://$this->http_host";
        $this->site_docroot = $DOCUMENT_ROOT;
        break;

      case BROWSER_TYPE_HTML:
      case BROWSER_TYPE_XHTML:
        // First try to determine browser version. We assume that the people
        // who made it will be sticking to the rule that the first word
        // is the browser/version string.
        $agentbits = explode(" ", $HTTP_USER_AGENT);
        $verbits   = explode("/", $agentbits[0]);
        if (is_numeric((float) $verbits[1])) {
          $this->browser_version = $verbits[1];
        }
        // OPERA
        if (eregi("opera", $HTTP_USER_AGENT)) {
          $this->browser = BROWSER_OPERA;
          if (preg_match("/.* opera (.+) .*/i", $HTTP_USER_AGENT, $matches)) {
            if (is_numeric((float) $matches[1])) {
              $this->browser_version = $matches[1];
            }
          }
        }
        // INTERNET EXPLORER
        // Determine flavour of browser. Practicality means that we
        // resolve this down to either IE or Netscape/Mozilla
        elseif (eregi("msie", $HTTP_USER_AGENT)) {
          $this->browser = BROWSER_IE;
          // IE has its own version number..
          if (preg_match("/.* msie (.+); .*/i", $HTTP_USER_AGENT, $matches)) {
            if (is_numeric((float) $matches[1])) {
              $this->browser_version = $matches[1];
            }
          }
        }
        // MOZILLA - NETSCAPE 4.xx
        elseif (eregi("mozilla", $HTTP_USER_AGENT)) {
          $this->browser = BROWSER_MOZILLA;
          // The Mozilla engine with version nos. less than 5.0 is
          // in fact that old dog the Netscape browser..
          if ($this->browser_version > 0 && $this->browser_version < 5) {
            $this->browser = BROWSER_NETSCAPE;
          }
        }
        // MOBILE PHONES..
        elseif (eregi($WAP_PHONES, $HTTP_USER_AGENT)) {
          $this->browser = BROWSER_PHONE;
        }
        // OTHER WEB BROWSER
        // This might be an xhtml-compatible phone or a
        // PDA of some kind, who knows..
        else {
          $this->browser = BROWSER_OTHER;
        }

        // The site Domain and URL..
        $this->http_host = $HTTP_HOST;
        $this->site_url = "http://$this->http_host";
        $this->site_docroot = $DOCUMENT_ROOT;
        break;

      default:
        $this->browser = BROWSER_NONE;
        $this->browser_version = "";
        // The site Domain and URL..
        $this->http_host = $HTTP_HOST;
        $this->site_url = "http://$this->http_host";
        $this->site_docroot = $DOCUMENT_ROOT;
    } // switch

    if (dirname($PHP_SELF) != "/")
    $this->site_url .= dirname($PHP_SELF);

    // The page being requested and the query string.
    // These are just our local copies..
    $this->requested = "";
    $this->requested_query = "";
    if (isset($SCRIPT_NAME)) {
      $this->requested = $SCRIPT_NAME;
    }
    if (isset($QUERY_STRING)) {
      $this->requested_query = $QUERY_STRING;
    }

    // Set default charset according to browser type.
    // This is just the initial setting & can be overriden..
    switch ($this->browser_type) {
      case BROWSER_TYPE_WML:
      case BROWSER_TYPE_WMLUP:
        $this->charset = "UTF-8";
        break;
      default:
        $this->charset = "ISO-8859-1";
    }
  } // response
  // ...................................................................
  /**
  * Initialise values
  * Initialise our main response parameters to normal values.
  * @access private
  */
  function initialise() {
    $this->cookiename = "session_id";
    $this->compression_type = NO_COMPRESSION;
    $this->compression_minsize = 0;
    $this->persistent_hosts = array();
    $this->auth_type = FORM_AUTHENTICATION;
    $this->auth_fail_option = AUTHFAIL_GUEST;
    $this->auth_fail_redirect = "";
    $this->multilang = MULTILANG_DISABLED;
  } // initialise
  // ...................................................................
  /**
  * Determine the browser type
  * Examines the headers, if we are running as an apache module and
  * returns the browser type accordingly. If we are not running as
  * an apache module returns 'cgi' (command line).
  * @access private
  */
  function get_browser_type() {
    global $HTTP_USER_AGENT;
    global $WEB_BROWSERS, $WAP_PHONES;

    // Starting point..
    $type = BROWSER_TYPE_UNKNOWN;

    // If webserver then detect, else CLI script..
    if (isset($HTTP_USER_AGENT) && $HTTP_USER_AGENT != "") {
      // Determine accept headers..
      $headers = getallheaders();
      $this->accept = trim($headers["Accept"]);
      $this->accept_encoding = trim($headers["Accept-Encoding"]);
      $this->accept_language = trim($headers["Accept-Language"]);

      // STANDARD BROWSERS - User Agent Detection
      if (eregi($WEB_BROWSERS, $HTTP_USER_AGENT)) {
        $type = BROWSER_TYPE_HTML;
      }
      // WAP PHONES - User Agent Detection
      elseif (eregi($WAP_PHONES, $HTTP_USER_AGENT)) {
        $type = BROWSER_TYPE_WML;
      }

      // GENERIC - Accept Encodings Detection
      // This might override the more general user-agent detection
      // phase which has been undertaken above. Note that Vodafone
      // gateways do not pass through Accept headers.
      if     (stristr($this->accept, "application/vnd.wap")) $type = BROWSER_TYPE_WML;
      elseif (stristr($this->accept, "text/vnd.wap"))        $type = BROWSER_TYPE_WML;
      elseif (stristr($this->accept, "application/xhtml"))   $type = BROWSER_TYPE_XHTML;
      elseif (stristr($this->accept, "text/html"))           $type = BROWSER_TYPE_HTML;

      // A bit of post-processing for WAP phones which
      // might have Phone.com extensions..
      if ($type == BROWSER_TYPE_WML) {
        if (stristr($this->accept, "application/x-up")
         || stristr($this->accept, "application/vnd.phonecom")
        ) {
          $type = BROWSER_TYPE_WMLUP;
        }
        elseif (eregi("(up)", $HTTP_USER_AGENT)) {
          $type = BROWSER_TYPE_WMLUP;
        }
      }
    }
    else {
      // Default browser type to CLI (command line)..
      $type = BROWSER_TYPE_CLI;
    }

    // Fallback to default if unknown..
    if ($type == BROWSER_TYPE_UNKNOWN) {
      $type = BROWSER_TYPE_DEFAULT;
    }
    return $type;
  } // get_browser_type
  // .....................................................................
  /**
  * Set the DTDs array for all possible content types for this webpage.
  * The array is associative, with content type as the key, and the
  * DTD specifier as the value. Currently we only support two content
  * types: "html" and "wml".
  * @param array $DTD Array of DTD specifiers per content type
  */
  function set_dtd($DTD) {
    if (is_array($DTD)) {
      $this->DTD = $DTD;
    }
  } // set_dtd
  // ...................................................................
  /**
  * Set up the page attributes for our response. This is an important
  * call, and should be made just after including response-defs.php in
  * your source code. It sets up the page title, the template file which
  * defines the page structure, the theme and stylesheet.
  * @param string $title      Webpage title string
  * @param string $template   Template for this webpage
  * @param string $theme      Theme to apply. This is for branding purposes
  * @param string $stylesheet Name of stylesheet to use eg: 'sitestyle.css'
  * @param string $dtd        Override the DTD specifier for page
  */
  function page($title="", $template="main", $theme="", $stylesheet="", $dtd="") {
    // The ordering of these calls is important, since the
    // theme setting influences the path settings for the
    // template and the stylesheet..
    $this->set_title($title);
    $this->set_theme($theme);
    $this->set_template($template);
    $this->set_stylesheet($stylesheet);
    if ($dtd != "") {
      $this->head->set_dtd($dtd);
    }
    $this->check_group_membership();
  } // page
  // ...................................................................
  /**
  * Set up the Wap page (card) attributes for our response. This is
  * the exact equivalent to the page() method above, but for WAP
  * phones instead.
  * @param string $title      Card title string
  * @param string $template   Template for this WML card
  * @param string $theme      Theme to apply. This is for branding purposes
  * @param string $stylesheet Name of stylesheet to use eg: 'sitestyle.css'
  * @param string $dtd        Override the DTD specifier for WML page
  */
  function card($title="", $template="main", $theme="", $stylesheet="", $dtd="") {
    if ($this->browser != BROWSER_PHONE) {
      // Enable normal browsers to see Wap content..
      include_once("wml-defs.php");
      $this->body = new deck($title, $template, $theme, $stylesheet);
    }
    $this->page($title, $template, $theme, $stylesheet, $dtd);
    global $CARD;
    $CARD = new WMLcard(basename($template), $this->head->title);
    $this->check_group_membership();
    return $CARD;
  } // card
  // ...................................................................
  /**
  * Activate response
  * Activate the response object. This isn't done in the constructor
  * so that the application code can make various calls etc. to set
  * up the environment for the response, such as session parameters.
  * After this setup phase is complete, then this function is called.
  * @access private
  */
  function activate() {
    global $CONTEXT;
    global $tbxUsername, $user; // Username submitted by user form
    global $tbxPassword, $pass; // Password submitted
    global $tbxLogoff;          // Logoff sequence submission
    global $PHP_AUTH_USER;      // HTTP Authentication username
    global $PHP_AUTH_PW;        // HTTP Authentication password
    global $cachecontrol;       // Cache/page expiry control override

    debug_trace($this);

    // Set site-wide language, if multi-language mode, and given..
    if ($this->multilang) {
      debugbr("multilang: multi-language mode is enabled", DBG_DEBUG);
      if ($this->mbstring_avail) {
        mb_internal_encoding("UTF-8");
        mb_http_output("UTF-8");
      }
      // Use the session-specific language if present. How this session
      // var gets set is entirely application-specific. If it is defined
      // then it becomes the 'default' language by virtue of being first..
      if (isset($this->session_record["lang_id"]) && $this->session_record["lang_id"] != "") {
        $this->add_language($this->session_record["lang_id"]);
      }
      // Otherwise, use the language defaults. These are again in an
      // order, with the first one the designated default..
      else {
        $q  = "SELECT lang_id FROM ax_language";
        $q .= " WHERE enabled=TRUE";
        $q .= "   AND is_default=TRUE";
        $q .= " ORDER BY display_order";
        $langs = dbrecordset($q);
        if ($langs->hasdata) {
          do {
            $this->add_language( $langs->field("lang_id") );
          } while ($langs->get_next());
        }
      }
    } // multilang

    // Create our parent webpage object now. We do not do this in
    // the constructor, due to timing. Only at this point do we
    // know what type of browser (html or wml) it is, and have
    // the relevant library included to support it..
    $this->webpage();

    // Map alternative fieldnames onto standard ones..
    if (isset($user) && !isset($tbxUsername)) $tbxUsername = $user;
    if (isset($pass) && !isset($tbxPassword)) $tbxPassword = $pass;

    // Always disable buffering mode for command-line..
    if ($this->browser_type == BROWSER_TYPE_CLI) {
      $this->set_buffering_mode(UNBUFFERED);
    }

    // See if we should disable webpage buffering..
    if ($this->buffering_mode == UNBUFFERED) {
      $this->discard();
      $this->buffered = false;
    }

    // Check flag to see if we have to globalise everything in
    // sight. If this is the case, it is probably due to the website
    // being broken by setting 'register_globals = Off' in php.ini.
    if ($this->globalise_all) {
      $this->globals->globalise_all();
    }
    else {
      // By now we have the actual cookie name, so we make
      // sure it is globalised..
      $this->globalise($this->cookiename, "cookie");
    }

    // Browser announcement
    $msg = "browser: $this->browser";
    if ($this->browser_version != "") $msg .= "/$this->browser_version";
    $msg .= " ($this->browser_type)";
    debugbr($msg, DBG_DEBUG);
    debugbr("accept=[$this->accept]", DBG_DEBUG);
    debugbr("UA=[$this->user_agent]", DBG_DEBUG);

    if ($this->browser != BROWSER_NONE) {
      // HTTP AUTHENTICATION
      if ($this->auth_type == HTTP_AUTHENTICATION) {
        debugbr("HTTP Authentication", DBG_DEBUG);
        if (!isset($PHP_AUTH_USER)) {
          debugbr("NOT Got PHP_AUTH_USER - sending HTTP auth headers", DBG_DEBUG);
          header("WWW-Authenticate: Basic realm=\"" . APP_NAME . "\"");
          header("HTTP/1.0 401 Unauthorized");
          return;
        }
        else {
          // Get the HTTP authentication variables..
          debugbr("Got PHP_AUTH_USER '$PHP_AUTH_USER', PHP_AUTH_PS '$PHP_AUTH_PW'", DBG_DEBUG);
          if ($this->get_session_cookie()) {
            debugbr("We already have a valid session..", DBG_DEBUG);
            if ($this->identify_user()
              && $this->userid == "guest"
              && strtolower($PHP_AUTH_USER != "guest")
            ) {
              debugbr("But overriding guest session with supplied HTTP Auth login..", DBG_DEBUG);
              $tbxUsername = $PHP_AUTH_USER;
              if (isset($PHP_AUTH_PW)) $tbxPassword = $PHP_AUTH_PW;
              else $tbxPassword = "";
            }
          }
          else {
            // Transform it into our own kind of login sequence..
            debugbr("Transforming it into our own kind of login..", DBG_DEBUG);
            $tbxUsername = $PHP_AUTH_USER;
            if (isset($PHP_AUTH_PW)) $tbxPassword = $PHP_AUTH_PW;
            else $tbxPassword = "";
          }
        }
      } // HTTP Auth
      else {
        debugbr("FORM Authentication.", DBG_DEBUG);
      }
    }

    // IDENTIFY USER, LOGIN
    $logged_in = true;
    $retries = 2;
    do {
      // Create the user's keep if enabled..
      if ( $this->keep_enabled
        && $this->browser != BROWSER_NONE
        && $this->browser != BROWSER_PHONE
      ) {
        $this->keep = new keep("CATITKEEP" . APP_PREFIX, $this->lifetime);
        // Deal with logoff activity..
        if (isset($tbxLogoff) && $tbxLogoff == "Logoff") {
          $this->keep->forgetall();
        }
      }

      // Get the session to login the user. We expect this step
      // to succeed unless they mucked up a login sequence, or
      // someone is hacking on our website..
      if (!$this->identify_user()) {
        // Deal with authentication/session create failure..
        debugbr("response-defs: user not identified.", DBG_DEBUG);
        if (!$this->isvalid()) {
          debugbr("response-defs: user invalid.", DBG_DEBUG);
          // Make sure the cookie is removed..
          $this->delete_cookie();
          if ($this->keep_enabled) {
            $this->keep->delete();
          }

          switch ($this->auth_fail_option) {
            case AUTHFAIL_DIE_MSG:
              $this->crash(401, $this->error_message);
              break;

            case AUTHFAIL_DIE_SILENT:
              $this->crash();
              break;

            case AUTHFAIL_REDIRECT:
              header("Location: $this->auth_fail_redirect");
              break;

            // AUTHFAIL_GUEST and friends..
            default:
              if (isset($tbxUsername) && strtolower($tbxUsername) == "guest") {
                // Failed guest login should not happen if the guest user
                // exists and its 'enabled' flag is true..
                $this->crash(401, "Guest logins are not permitted.");
              }
              else {
                // Unset everything they might be logging in with, so that
                // the system will default them into guest mode..
                if (isset($tbxUsername)) {
                  debugbr("AUTHFAIL_GUEST: tbxUsername is set '$tbxUsername', setting it to guest.", DBG_DEBUG);
                  $tbxUsername = "guest";
                  $tbxPassword = "";
                }
                if (isset($authid)) unset($authid);
                $logged_in = false;
              }
              break;
          } // switch
        }
        // Otherwise we failed authentication, but the user was
        // valid - a very strange state of affairs..!
        else {
          debugbr("user valid but not auth! userid='$this->userid'", DBG_DEBUG);
          if ($this->authfail_option != AUTHFAIL_SILENT) {
            $this->crash(500, "No session!");
          }
          else die();
        }
        // Countdown for retries..
        $retries -= 1;
      } // if !identify_user
      else {
        // Affirm user identified..
        $logged_in = true;
      }
    } while (!$logged_in && $retries > 0);

    // If login failed, then we have no options left..
    if (!$logged_in) {
      if ($this->authfail_option != AUTHFAIL_SILENT) {
        $this->crash(401);
      }
      else die();
    }

    // CONTENT TYPE HEADER
    $this->set_encoding();

    // CACHE HEADERS
    switch ($this->browser_type) {
      case BROWSER_TYPE_HTML:
      case BROWSER_TYPE_XHTML:
        header("Last-Modified: " . gmdate("D, d M Y H:i:s") . " GMT");
        if ($this->page_expiry_secs == -1 || (isset($cachecontrol) && $cachecontrol == "dynamic")) {
          header("Expires: Thu, 1 Jan 1970 01:00:00 GMT");
          header("Cache-Control: no-cache, must-revalidate");
          header("Pragma: no-cache");
          debugbr("page expiry: no cache", DBG_DEBUG);
        }
        else {
          $expirysecs = time() + $this->page_expiry_secs;
          header("Expires: " . gmdate("D, d M Y H:i:s", $expirysecs) . " GMT");
          debugbr("page expiry: cache for $this->page_expiry_secs seconds", DBG_DEBUG);
        }
        break;

      case BROWSER_TYPE_WML:
      case BROWSER_TYPE_WMLUP:
        header("Expires: Mon, 26 Jul 1997 05:00:00 GMT");
        header("Last-Modified: " . gmdate("D, d M Y H:i:s T") );
        header("Cache-Control: no-cache, must revalidate");
        header("Pragma: no-cache");
        header("Accept-Ranges: none");
        break;
    } // switch

    // CONTEXT
    if ($this->browser_type != BROWSER_TYPE_CLI
        && file_exists($this->site_docroot . "/context-defs.php")) {
      include_once("context-defs.php");
      $CONTEXT = new context();
    }
    debug_trace();
  } // activate
  // ...................................................................
  /**
  * Sets the status of the multi-language flag 'multilang'. If this flag
  * is true then multi-language facilities are enabled. This allows the
  * setting of tags to indicate webpage language(s), and for inputting
  * different character encodings via form elements etc.
  * @param boolean $mode Whether we should set multi-language mode
  */
  function set_multilang($mode=true) {
    $this->multilang = $mode;
  } // set_multilang
  // ...................................................................
  /**
  * Adds another language for the current webpage. Webpages might
  * contain content in multiple languages, hence the need for a list.
  * @param integer $langid The new language ID to add for the webpage
  */
  function add_language($langid) {
    if ($this->multilang) {
      if (!in_array($langid, $this->languages)) {
        $this->languages[] = $langid;
        if (isset($this->head)) {
          $this->head->add_language($langid);
        }
      }
    }
  } // add_language
  // ...................................................................
  /**
  * Sets the content-type header characterset encoding, and (optionally)
  * allows you to override the content type itself although it should
  * be set for you automatically, so you should only need to provide the
  * second parameter for special purposes. This method will send the
  * content-type: header to the user agent (browser). It can be called
  * any number of times before send() is called.
  * @param string $charset The character encoding to use for the webpage
  * @param string $content_type The content type (defaults to proper one for browser)
  */
  function set_encoding($charset="", $content_type="") {
    if ($charset != "") {
      $this->charset = $charset;
    }
    if ($content_type == "") {
      switch ($this->browser_type) {
        case BROWSER_TYPE_HTML:
        case BROWSER_TYPE_XHTML:
          $content_type = "text/html";
          break;

        case BROWSER_TYPE_WML:
        case BROWSER_TYPE_WMLUP:
          $content_type = "text/vnd.wap.wml";
          break;

        default:
          $content_type = "text/html";
      }
    }
    // Insert prefix if not already provided..
    if ( substr(strtolower($content_type), 0, 13) != "content-type:") {
      $content_type = "content-type: $content_type";
    }
    // Set head section charset..
    if (isset($this->head)) {
      $this->head->set_charset($this->charset);
    }

    // Append charset code for header if provided..
    if ($this->charset != "") {
      $content_type .= "; charset=$this->charset";
    }
    // Also transmit a content-type header..
    if ($this->browser_type != BROWSER_TYPE_CLI) {
      header($content_type);
    }
  } // set_encoding
  // ...................................................................
  /**
  * Set globalise all
  * Flag that we should globalise all vars from anywhere. This is the
  * same as setting 'register_globals = On' in php.ini. It gives
  * backward compatibility with that setting, if you find that it has
  * been set to 'Off' and this breaks your code.
  * NB: This only takes effect when the session is activated with
  * the activate() method of this class.
  */
  function globalise_all() {
    $this->globalise_all = true;
  } // globalise_all
  // ...................................................................
  /**
  * Globalise specific vars
  * @param string $varnames Comma-delimited list of variable names to globalise
  * @param string $sources  Comma-delimited list of allowed sources for values
  */
  function globalise($varnames="", $sources="env,get,post,cookie,server") {
    $this->globals->globalise($varnames, $sources);
  } // globalise
  // ...................................................................
  /**
  * Register vars
  * Register given variables for globalisation.
  * @param string $varnames Comma-delimited list of variable names to globalise
  * @param string $sources  Comma-delimited list of allowed sources for values
  */
  function register($varnames, $sources="env,get,post,cookie,server") {
    $this->globals->register($varnames, $sources);
  } // register
  // ...................................................................
  /**
  * Set keep enable status. If the keep is enabled then there is access to
  * the remember() and forget() methods to control persistent vars using
  * Php session handling functions.
  * @see remember(), forget()
  * @param boolean $keep_enabled If true then the keep methods are enabled
  */
  function set_keep($keep_enabled=true) {
    $this->keep_enabled = $keep_enabled;
  } // set_keep
  // ...................................................................
  /**
  * Remember variables by name, across page requests using our keep
  * functionality. This utilises Php session handling to maintain the value
  * of variables named here. Pass a list of vars as a string delimited
  * by a comma.
  * @param string $varnames Comma-delimited list of variable names to keep
  */
  function remember($varnames) {
    if ($this->keep_enabled && isset($this->keep)) {
      $this->keep->remember($varnames);
    }
  } // remember
  // ...................................................................
  /**
  * Forget kept variables by name. The named variables will have been
  * passed to the remember() function beforehand, and this method causes
  * them to be forgotten.
  * @param string $varnames Comma-delimited list of variable names to forget
  */
  function forget($varnames) {
    if ($this->keep_enabled && isset($this->keep)) {
      $this->keep->forget($varnames);
    }
  } // forget
  // ...................................................................
  /**
  * Define persistent hosts list.
  * Set the list of hosts recognition strings to connect to with
  * persistent database connections. You can pass either an array of strings
  * or a single string containing a delimited list of hosts.
  * @param mixed $hosts Array or delimited list of host names to connect persistently to
  * @param string $delim Optional delimiter, defaults to a comma
  */
  function set_persistent_hosts($hosts="", $delim=",") {
    if (is_array($hosts)) {
      $this->persistent_hosts = $hosts;
    }
    else {
      if ($hosts != "") {
        $hosts = explode($delim, $hosts);
      }
    }
  } // set_persistent_hosts
  // ...................................................................
  /**
  * define a blocked IP list.
  * Set the list of IP addresses which are persona non grata for our site.
  * This might be a list of problem IPs which are hacking, for example.
  * Note that this method may not return, since the IP checking is done
  * immediately, and if the REMOTE_ADDR accessing the site right now is
  * found to match a blocked IP, then we crash and burn the response.
  *
  * @param mixed $hosts Array or delimited list of host names to connect persistently to
  * @param string $delim Optional delimiter, defaults to a comma
  */
  function set_blocked_ips($ips="", $delim=",") {
    if ($ips != "") {
      global $REMOTE_ADDR;
      if (isset($REMOTE_ADDR) && $REMOTE_ADDR != "") {
        if (!is_array($ips) && $ips != "") {
          $ips = explode($delim, $ips);
        }
        foreach ($ips as $unwelcome) {
          if (preg_match("/^$unwelcome/", $REMOTE_ADDR)) {
            $this->crash(403, "Bad address");
            exit();
          }
        }
      }
    }
  } //set_blocked_ips
  // ...................................................................
  /**
  * Set compression type
  * Set the compression type to use for content delivery. Options are:
  *  NO_COMPRESSION       Normal delivery of content
  *  BUILTIN_COMPRESSION  Builtin PHP compression (Requires Php >= v4.04)
  *  CUSTOM_COMPRESSION   Compression provided by the library
  * @param integer $type Compression type: NO_COMPRESSION, BUILTIN_COMPRESSION, CUSTOM_COMPRESSION
  */
  function set_compression_type($type=NO_COMPRESSION) {
    $this->compression_type = $type;
  } // set_compression_type
  // ...................................................................
  /**
  * Set buffering option
  * Set the webstream buffering option. Usually we want to buffer the
  * webpage output to do cool things with the content right up until
  * we send it. If you select NO_BUFFERING, then output won't be
  * buffered and you will probably be processing it yourself. If in
  * doubt, leave it as BUFFERING, the default.
  * Possible options..
  *  BUFFERED             Output webstream uses Php buffering (default)
  *  UNBUFFERED           No buffering of output webstream
  * @param bool $buffering Buffering option: NO_BUFFERING, or BUFFERING
  */
  function set_buffering_mode($mode=BUFFERED) {
    $this->buffering_mode = $mode;
  } // set_buffering_mode
  // ...................................................................
  /**
  * Set metadata mode
  * This setting determines whether Axyl will implement the metadata
  * enhancements that it has. If true, content-managed layouts will get
  * an extra button 'META', which enables page metadata to be edited
  * Also, when page generation takes place, metadata elements will be
  * procuced in the output.
  * @param bool $mode Metadata enable mode, true or false
  */
  function set_metadata_mode($mode=METADATA_ENABLED) {
    $this->metadata_mode = $mode;
  } // set_metadata_mode
  // ...................................................................
  /**
  * Set page expiry in seconds. This affects all of the pages returned by Axyl.
  * Normally this is left at -1, which causes the system to tell the user browser
  * to NOT cache the webpage, and revalidate every time. This is the usual case
  * for a dynamic website where content is always changing. If set to a positive
  * value, the user browser will (probably) keep the page cached for this time.
  *
  * @param integer $secs Time to allow website pages to live in the user browser.
  */
  function set_page_expirysecs($secs=-1) {
    $this->page_expiry_secs = $secs;
  } // set_page_expirysecs
  // ...................................................................
  /**
  * Set compression threshold
  * Set the threshold size of content before we use compression.
  * @param integer $size Pagesize in bytes below which we will not compress
  */
  function set_compression_minsize($size=0) {
    $this->compression_minsize = $size;
  } // set_compression_minsize
  // ...................................................................
  /**
  * Set website authentication
  * Set the authentication option for the website. Note that although there
  * is NO_AUTHENTICATION, this has no real effect and is mainly there so
  * you can indicate your intent not to bother with authentication. If no
  * username/password is passed then the system will always log the session
  * as a "guest" anyway, regardless of this setting. Options are:
  *  NO_AUTHENTICATION    Redundant case, no authentication
  *  HTTP_AUTHENTICATION  User sees a browser username/password popup
  *  FORM_AUTHENTICATION  Custom form fields $tbxUsername/$tbxPassword
  * @param integer $authtype Authentication option type
  */
  function set_authentication_type($authtype=HTTP_AUTHENTICATION) {
    $this->auth_type = $authtype;
  } // set_authentication_type
  // ...................................................................
  /**
  * Set authentication failure option
  * This sets the option for when authentication fails due to invalid
  * username/password. Options are:
  *  AUTHFAIL_DIE_MSG      Die, with 'not authorised' message
  *  AUTHFAIL_DIE_SILENT   Die silently
  *  AUTHFAIL_REDIRECT     Re-direct to alternate URL
  *  AUTHFAIL_GUEST        Welcome the user as a guest (the default)
  */
  function on_authentication_fail($option=AUTHFAIL_GUEST, $redirect="") {
    $this->auth_fail_option = $option;
    $this->auth_fail_redirect = $redirect;
  } // on_authentication_fail
  // ...................................................................
  /**
  * Add database
  * Add a new database to the list of datasources which are going to be used to
  * serve this response. The dbtype and the name are the only mandatory parameters.
  * @param string  $dbtype    The type of database eg: 'postgres', 'mssql' etc.
  * @param string  $name      The name of the database
  * @param string  $user      Name of a user who can access the database
  * @param string  $passwd    The password the user can access the database with
  * @param string  $host      The hostname of the machine running the database (TCP/IP)
  * @param integer $port      The port number of the database server
  * @param boolean $default   True if the database is the default database
  */
  function add_database($dbtype, $name, $user="", $passwd="", $host="", $port="", $default=false) {
    if ($this->db_backed) {
      if (!isset($this->datasource)) {
        $this->datasource = new datasources();
      }
      $this->datasource->add_database($dbtype, $name, $user, $passwd, $host, $port, $default);
    }
  } // add_database
  // ...................................................................
  /**
  * Selects a database to use
  * This will connect it if it isn't already connected. Calling this
  * with no database name will select the default one. Returns the
  * database unique identifier, or false if none was selected.
  * The named database must have been already defined. @see add_database()
  * @see datasources::add_database()
  * @param string $dbname  The name of the database to select
  * @return resource The database resource ID
  */
  function select_database($db_name="") {
    if ($this->db_backed && isset($this->datasource)) {
      return $this->datasource->select($db_name);
    }
    else {
      return false;
    }
  } // select_database
  // ...................................................................
  /**
  * Is host in peristent list
  * Returns true if the given host is in our list of persistent
  * hosts, else returns false. The persistent hosts list we hold
  * will general contain partial hostnames. We therefore check
  * to see if this partial name occurs anywhere in the given
  * hostname.
  * @param string $hostname  The name of the host to check
  */
  function InPersistentHostsList($hostname) {
    if (count($this->persistent_hosts) > 0) {
      foreach ($this->persistent_hosts as $host) {
        if (stristr($hostname, $host)) {
          return true;
        }
      }
    }
    return false;
  } // InPersistentHostsList
  // ...................................................................
  /**
  * Allowed groups
  * This defines the allowed user-groups for this response, otherwise
  * they get an error page returned. The list of allowed groups
  * should be a comma-delimited string.
  * NB: We look for globals $admin_auth_code, and $auth_code and if
  *     available, use these for authorisation.
  * @param string $allowed_groups Comma-delimited list of allowed groups
  */
  function allowed_groups($allowed_groups) {
    global $WEBMASTER_EMAIL, $WEBMASTER_PERSON;
    global $auth_code;
    global $admin_auth_code;
    if (!$this->ismemberof_group_in($allowed_groups)) {
      if (isset($admin_auth_code)) $auth = $admin_auth_code;
      elseif (isset($auth_code)) $auth = $auth_code;
      if (isset($auth)) {
        $user = new authorised_user($auth);
        if ($user->ismemberof_group_in($allowed_groups)) {
          return;
        }
      }
      // On failure follow the login settings for action..
      switch ($this->auth_fail_option) {
        case AUTHFAIL_DIE_MSG:
          $msg = "<p><center><h4>"
               . "You are not permitted to view this page.<br>"
               . "For further information, please contact "
               . "<a href=\"mailto:$WEBMASTER_EMAIL\">$WEBMASTER_PERSON</a>"
               . "</h4></center></p>"
               . "<p><center><h4><a href=\"/\">HOME</a></p></h4></center></p>"
               ;
          $this->set_template("plain");
          $this->plugin("MAIN_CONTENT", $msg);
          $this->send();
          exit;
          break;
        case AUTHFAIL_DIE_SILENT:
          $this->crash();
          break;
        case AUTHFAIL_REDIRECT:
          header("Location: $this->auth_fail_redirect");
          exit;
          break;
        default:
          $this->crash(401, $this->error_message);
          exit;
      } // switch
    }
  } // allowed_groups
  // ...................................................................
  /**
  * Check group membership
  * This check whether the user requesting the page fulfils any group
  * membership requirements. These are optionally expressed as the
  * presence of 'ax_sitepage_group' records. A set of these records
  * represents a set of groups that the user must be member of at
  * least one of, to have access to the page.
  */
  function check_group_membership() {
    // Find page and group permissions if we can..
    $q  = "SELECT * FROM ax_sitepage p, ax_sitepage_group sg, ax_group g";
    $q .= " WHERE p.page_path='" . addslashes($this->requested) . "'";
    $q .= "   AND p.page_title='" . addslashes($this->head->title) . "'";
    $q .= "   AND sg.page_id=p.page_id";
    $q .= "   AND g.group_id=sg.group_id";
    $allowed = dbrecordset($q);
    if ($allowed->hasdata) {
      $groups = array();
      do {
        $groups[] = $allowed->field("group_desc");
      } while ($allowed->get_next());
      // Check membership and die if not authorised..
      debugbr("sitepage group membership: " . implode(",", $groups), DBG_DEBUG);
      $this->allowed_groups( implode(",", $groups) );
    }
  } // check_group_membership
  // ...................................................................
  /**
  * Crash the response with message
  * A fatal error ocurred. We die with a message to the system log
  * and to the user. Code supplied is optional.
  * @param integer $code The HTTP/1.0 code to use
  * @param string  $msg  The message to display and log
  */
  function crash($code=false, $msg="") {
    // Echo debugging output, but only if
    // we are in debugging mode..
    echo debug_render();

    // For the system log..
    $logmsg = APP_PREFIX . " CRASH: ";
    if ($msg != "") $logmsg .= $msg;
    else $logmsg = "(no message)";
    if ($code) $logmsg .= " ($code)";
    error_log(APP_PREFIX . " CRASH: $msg.");
    // For the user..
    if ($code || $msg != "") {
      $umsg = "";
      if ($code) $umsg .= HTTPError($code) . " ";
      if ($msg != "") $umsg .= $msg;
      die($umsg);
    }
    else die();
  } // fatal
} // response class

// -----------------------------------------------------------------------
// CREATE THe RESPONSE OBJECT
$RESPONSE = new response();

// -----------------------------------------------------------------------
// OUTPUT GENERATION INCLUDES
switch ($RESPONSE->browser_type) {
  case BROWSER_TYPE_WML:
  case BROWSER_TYPE_WMLUP:
    include_once("wml-defs.php");
    break;
  default:
    include_once("html-defs.php");
} // switch

// -----------------------------------------------------------------------
// APPLY APPLICATION RESPONSE SETTINGS
// These settings are the ones sourced from the application.xml file
// for this particular website.
if (file_exists("$DOCUMENT_ROOT/application.xml") && isset($application) && $application->valid) {
  $firstDB = true;
  foreach ($application->settings as $setting) {
    if (isset($setting->agent) && $setting->agent != "") {
      $methodname = $setting->agent;
      $parms = array();
      foreach ($setting->parameters as $parameter) {
        $parms[] = $parameter->get_decodedvalue();
      }
      // Set the 'default=true' parameter for first database..
      if ($setting->name == "database" && $firstDB) {
        $parms[] = true;
        $firstDB = false;
      }
      // Apply the RESPONSE setting..
      call_user_method_array($methodname, $RESPONSE, $parms);
    }
  }

  // UNICODE INCLUDE..
  // If the above loop brought in the multilang setting as true
  // then we will be needing the Unicode functions..
  if ($RESPONSE->multilang) {
    include_once("unicode-defs.php");
  }

  // Default webmaster, and webmaster email..
  if ($WEBMASTER_PERSON == "") {
    $WEBMASTER_PERSON = "The " . ((APP_NAME != "") ? (APP_NAME . " ") : "") . "Webmaster";
  }
  if ($WEBMASTER_EMAIL == "") {
    $WEBMASTER_EMAIL = "webmaster@" . $this->http_host;
  }

  // Debugging control..
  $debugging = $application->getparameter("debug_on", "debug_on");
  if ($debugging) {
    debug_on($application->getparameter("debug_classes", "debug_classes"));
    debug_output($application->getparameter("debug_output", "debug_output"));
  }

  // Make some checks/adjustments..
  if ($RESPONSE->browser == BROWSER_NONE
  ||  $RESPONSE->browser == BROWSER_PHONE) {
    $RESPONSE->set_keep(false);
  }

  // Turm relative references to web docroot absolute..
  if (isset($TEMPLATESDIR) && substr($TEMPLATESDIR,0,1) != "/") {
    $TEMPLATESDIR = "/$TEMPLATESDIR";
  }
  if (isset($IMAGESDIR) && substr($IMAGESDIR,0,1) != "/") {
    $IMAGESDIR = "/$IMAGESDIR";
  }
  if (isset($CMDIR) && substr($CMDIR,0,1) != "/") {
    $CMDIR = "/$CMDIR";
  }
  if (isset($CATALOGDIR) && substr($CATALOGDIR,0,1) != "/") {
    $CATALOGDIR = "/$CATALOGDIR";
  }
  if (isset($INCDIR) && substr($INCDIR,0,1) != "/") {
    $INCDIR = "/$INCDIR";
  }

  // No longer required..
  unset($application);
  // Activate the response, determine user etc..
  $RESPONSE->activate();
}

// -----------------------------------------------------------------------
?>