<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    search-defs.php                                         */
/* Author:      Paul Waite                                              */
/* Description: Definitions for handling searches for objects in a      */
/*              generic way. This class contains functionality which    */
/*              handles search parameters, and hits etc. but has no     */
/*              search-engine-specific code to actually do the search.  */
/*                                                                      */
/*              As such, the 'search' class is therefore a virtual      */
/*              class which is intended to be inherited by a more       */
/*              specific class to implement particular modes of         */
/*              searching.                                              */
/*                                                                      */
/* ******************************************************************** */
/** @package search */

//-----------------------------------------------------------------------
/**
* The search hit class
* Something to contain a search hit which is defined as an array
* of information. The format of this array is entirely up to the
* search being performed, and is not enforced here.
* @package search
*/
class hit {
  /** Array of fieldname/value pairs - info returned by the search */
  var $fields;
  // .....................................................................
  /**
  * Constructor
  * Create a new search hit.
  * @param array $fields Assoc. array (fieldname/value) of fields returned
  */
  function hit($fields=false) {
    // The fields are an optional array of fields which were returned
    // by a search. If not passed, then we assign an empty array..
    if (!is_array($fields)) $this->fields = array();
    else $this->fields = $fields;
  }
} // hit class

//-----------------------------------------------------------------------
/**
* The search term class
* Something to contain a search term. A search term is
* basically a string of one or more words to match, together
* with the operator to apply with the match.
* @package search
*/
class term {
  /** The text of this search term */
  var $text;
  /** Join operator, one of: 'AND', 'OR', 'NOT', 'AND NOT' */
  var $operator;
  /** ID field provided for optional application use */
  var $id;
  /** True if this term is deemed valid */
  var $valid;
  // .....................................................................
  /**
  * Constructor
  * Create a new search term.
  * @param string  $text  Search term text.
  * @param integer $op    Search operator 'and', 'or', 'and not'.
  * @param string  $id    A unique ID to associate with this search term
  */
  function term($text="", $op="", $id="") {
    $this->valid = false;
    $this->id = $id;
    $this->text = trim($text);
    $op = trim($op);
    if ($op == ""
      || !strcasecmp($op, "OR")
      || !strcasecmp($op, "AND")
      || !strcasecmp($op, "NOT")
      || !strcasecmp($op, "AND NOT")) {
      $this->operator = $op;
      $this->valid = true;
    }
    debugbr("term: new search term: '$this->text' '$this->operator' '$this->id'");
  } // term
} // term class

//-----------------------------------------------------------------------
/**
* The search class is an object which can be assigned search terms, can
* execute a search, and can contain search hits having executed a seach.
* NOTE: This is a virtual class which is expected to be used by a child
* class which implements a particular search mechanism eg: swish++
* or database query searching. As such the execute() method does nothing
* and must be over-ridden in the child classes implementing searching.
* @package search
*/
class search {
  /** Title for heading of output */
  var $title = "";
  /** Query string container */
  var $query = "";
  /** Array of hits returned. Each element of this array is
      normally an array of fields returned by the search. */
  var $hit = array();
  /** Array of search terms to match */
  var $searchterm = array();
  /** Maximum results to return in query */
  var $max_results = 25;
  /** No. of results to skip in query (for paging) */
  var $skip_results = 0;
  /** Start date range for search (false means undefined). This
      variable can be in any application-specific format */
  var $date_start = false;
  /** End date range for search (false means undefined). This
      variable can be in any application-specific format */
  var $date_end = false;
  /** Name of the field to which daterange should be applied */
  var $date_fieldname = "";
  /** Whether we have run a query */
  var $executed = false;
  // .....................................................................
  /**
  * Constructor
  * Create a new search.
  * @param string $title Title/description for this search
  */
  function search($title="Search Results") {
    $this->title = $title;
    $this->initialise();
  } // search
  // .....................................................................
  /**
  * Set date range
  * Set the date range for the search. This is just recording the given
  * date information for use by child classes of this one. We do not
  * even care what the format of the dates is. They are just stored.
  * @param string $start Start date for search
  * @param string $end   End date for search
  */
  function set_daterange($start=false, $end=false, $date_fieldname="") {
    if ($start != false && $start != "")
      $this->date_start = $start;
    else $this->date_start = false;
    if ($end != false && $end != "")
      $this->date_end = $end;
    else $this->date_end = false;
    $this->date_fieldname = $date_fieldname;
  } // set_daterange
  // .....................................................................
  /**
  * Clear the date range for the search. Makes sure that the search
  * will not be filtered by a date restriction.
  */
  function clear_daterange() {
    $this->date_start = false;
    $this->date_end = false;
    $this->date_fieldname = "";
  } // clear_daterange
  // .............................................................................................
  /** Return count of searchterms present in this search.
  * @return integer Number of search terms we have currently.
  */
  function termcount() {
    return count($this->searchterm);
  } // termcount
  // .............................................................................................
  /** Return count of hits present in this search.
  * @return integer Number of hits we got.
  */
  function hitcount() {
    return count($this->hit);
  } // hitcount
  // .............................................................................................
  /** Return true if at least one of our dates is set.
  * @return boolean True if a date range is in effect.
  */
  function has_daterange() {
    return ($this->date_start || $this->date_end);
  } // has_daterange
  // .....................................................................
  /**
  * Set maximum results
  * Sets the maximum results to return from the search.
  * @param integer $max Maximum hits to return
  */
  function set_maxresults($max=0) {
    $this->max_results = $max;
    return $this;
  } // set_maxresults
  // .....................................................................
  /**
  * Set skip results
  * Sets the number of results to skip in the query. Eg. If
  * this is set to 15, then the first 15 results of the
  * query will be skipped before returning results. This can
  * be used as a method of paging a query which returns a
  * large number of results..
  * @param integer $skip Number of initial hits to skip
  */
  function set_skipresults($skip=0) {
    $this->skip_results = $skip;
    return $this;
  } // set_skipresults
  // .....................................................................
  /**
  * Add a new search term to match. Search terms can be a single word or
  * compound patterns, Each time one of these is added, it has an operator
  * associated with it - whether this term is a "may have" (OR), or a
  * "must have" (AND) term.
  * Visualize these terms as self-contained statements, with brackets
  * around each end, and which are joined to others by the given
  * operator to make the whole query.
  * @param string  $term Search term text to match.
  * @param integer $op   Joining operator: 'AND', 'OR', 'NOT, 'AND NOT'.
  * @param string  $id   An optional ID to associate with this search term.
  */
  function match($term="", $op="OR", $id="") {
    debug_trace($this);
    $this->searchterm[] = new term($term, $op, $id);
    $this->reset_search();
    debug_trace();
  } // match
  // .....................................................................
  /**
  * Define a search term which the search must match to succeed.
  * @param string $term Search term text to match.
  * @param string $id An optional ID to associate with this search term.
  */
  function must_match($term, $id="") {
    $this->match($term, "AND", $id);
  } // must_match
  // .....................................................................
  /**
  * Define a search term which the search may or may not match.
  * @param string $term Search term text to match.
  * @param string $id An optional ID to associate with this search term.
  */
  function may_match($term, $id="") {
    $this->match($term, "OR", $id);
  } // may_match
  // .....................................................................
  /**
  * Define a search term which the search must not match to succeed.
  * @param string $term Search term text to match.
  * @param string $id An optional ID to associate with this search term.
  */
  function does_not_match($term, $id="") {
    $this->match($term, "NOT", $id);
  } // does_not_match
  // .....................................................................
  /**
  * Execute search
  * This is the key functionality of a real search class
  * which is derived from this virtual class.
  * @access private
  */
  function execute() { }
  // .....................................................................
  /**
  * Query valid status
  * Simple assumption: the query string must contain
  * something to be valid. Feel free to override this!
  * @access private
  */
  function queryvalid() {
    if ($this->query == "") {
      $this->buildquery();
    }
    return ($this->query != "");
  } // queryvalid
  // .....................................................................
  /**
  * Build query syntax
  * Construct the query string. This is a generic simple method which
  * might have to be overridden in the child classes which implement
  * specific search mechanisms, eg. swish++ searching.
  * @access private
  */
  function buildquery() {
    $q = "";
    if (isset($this->searchterm)) {
      foreach ($this->searchterm as $term) {
        if ($term->valid && $term->text != "") {
          // Prefix the join operator..
          if ($q == "") {
            // Initial phrase, can only be 'not'..
            if (!strcasecmp($term->operator, "AND NOT")
             || !strcasecmp($term->operator, "NOT")) {
              $q .= "NOT ";
            }
          }
          else {
            // Just prefix the operator on..
            $q .= "$term->operator ";
          }
          $q .= "$term->text ";
        }
      }
    }
    $this->query = trim($q);
  } // buildquery
  // .....................................................................
  /**
  * Initialise everything about the search.
  * Initialise to a state which is the same as when the object is
  * instantiated.
  */
  function initialise() {
    $this->clear_search();
    $this->reset_search();
  } // initialise
  // .....................................................................
  /**
  * Clear the search terms only.
  */
  function clear_search() {
    if ($this->termcount() > 0) {
      $this->searchterm = array();
    }
  } // clear_search
  // .....................................................................
  /**
  * Reset search results
  * Reset the search results, hit-count, and
  * query executed status.
  */
  function reset_search() {
    $this->query = "";
    $this->executed = false;
    if ($this->hitcount() > 0) {
      $this->hit = array();
    }
  } // reset_search
} // search class

// ----------------------------------------------------------------------
?>