<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    swish-defs.php                                          */
/* Author:      Paul Waite                                              */
/* Description: Definitions for interfacing to the SWISH++ search       */
/*              engine system. Swish++ is a system which is optimised   */
/*              for indexing and searching a file hierarchy of text     */
/*              files of some kind. In particular it is very good at    */
/*              doing this for HTML files, and will index the content   */
/*              of META tags in the header of such files. This allows   */
/*              us to essentially cross-index these files by key data.  */
/*                                                                      */
/* ******************************************************************** */
/** @package search */

/** Generic search classes */
include_once("search-defs.php");

// ----------------------------------------------------------------------
/**
* The swish search class
* This class inherits the functionality of the generic 'search'
* class. It extends it to implement a swish++ search. To use this
* you must install the swish++ system on the web server running
* this code. There is a Debian package to do this for you or else
* retrieve the tarball from:
*   http://homepage.mac.com/pauljlucas/software/swish/
*
* NB: After installing make sure that there are links to the main
* executables: search++, index++, extract++ etc. in a directory
* on the path of the executing script, eg: /usr/bin
* @package search
*/
class swish_search extends search {
  /** Directory to do the search from */
  var $searchdir = "";
  /** Name of index file to use (defaults to 'swish++.index') */
  var $searchindex = "";
  // .....................................................................
  /**
  * Constructor
  * Create a new swish++ search
  * @param string $title     Title describing this search
  * @param string $searchdir Directory root in which to perform the search
  */
  function swish_search($title="Search Results", $searchdir="") {
    $this->search($title);
    $this->max_results = 25;
    $this->skip_results = 0;
    $this->searchdir = $searchdir;
    $this->initialise();
  }
  // .....................................................................
  /**
  * Set search directory
  * Set the root directory to go to and search from.
  * @param string $dir Directory root in which to perform the search
  */
  function set_searchdir($dir="") {
    $this->searchdir = $dir;
    return $this;
  }
  // .....................................................................
  /**
  * Set search index filename
  * Defines the index filename to use for searching. If no filename is
  * specified in the constructor (@see swish_search()) or via this
  * method, then the default is used: 'swish++.index'.
  * NB: This should be the basename only, not a path to a file.
  * @param string $fname Alternative name of index file to use in search
  */
  function set_searchindex($fname="") {
    $this->searchindex = $fname;
    return $this;
  }
  // .....................................................................
  /**
  * Execute the swish++ search
  * Here we execute a swish++  search, overriding the method
  * in the parent class. This involves piping an OS command
  * containing the search query terms. The default is to execute
  * 'search++' without any path to the executable, relying on
  * webserver PATH settings. You can over-ride this by specifying
  * a path to the executable here if required.
  * @param string $executable Alternative executable to use to search
  */
  function execute($executable="") {
    if ($executable == "") {
      $executable = "search++";
    }
    debug_trace($this);
    if ($this->queryvalid()) {
      // Assemble the commandline for the query..
      if ($this->searchdir != "") {
        $cmdline  = "cd $this->searchdir; ";
      }
      $cmdline .= "$executable ";
      if ($this->skip_results > 0) {
        $cmdline .= "-r $this->skip_results ";
      }
      //if ($this->max_results > 0) {
      //$cmdline .= "-m 10000 ";
      //}
      if ($this->searchindex != "") {
        $cmdline .= "-i $this->searchindex ";
      }
      $cmdline .= escapeshellcmd($this->query);
      debugbr("swish++ module: query: '$cmdline'");

      $fp = popen( $cmdline, "r" );
      $linecnt = 0;
      while ( !feof($fp) ) {
        $line = fgets($fp, 256);
        $fields = explode( " ", $line );
        $linecnt += 1;
        if ($fields[0] != "#" && $line != "") {
          $title = "";
          for( $j = 3; $j < count($fields); $j++) {
            if ($title != "") $title .= " ";
            $title .= $fields[$j];
          }
          $title = trim($title);
          if ($title == "") $title = "(No title)";
          $this->hit[] = new hit($fields[0], $fields[1], $fields[2], $title);
          $this->hitcount += 1;
          //if ($this->max_results > 0 && $this->hitcount >= $this->max_results) {
          //  break;
          //}
        }
        elseif ($linecnt == 1) {
          $swish_hits = trim($fields[2]);
        }
      } // while

      // Flag that we did it..
      $this->executed = true;
      debugbr("swish++ module: execution ok: returning $this->hitcount hits");
    }
    else {
      debugbr("swish++ module: invalid query: '$this->query'");
    }
    debug_trace();
  } // execute

} // swish_search class

// ----------------------------------------------------------------------
?>