<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    mail-defs.php                                           */
/* Author:      Paul Waite                                              */
/* Description: Various mail definitions.                               */
/*                                                                      */
/* ******************************************************************** */
/** @package email */

// ----------------------------------------------------------------------
/**
* Limit to the number of e-mails to send in one sitting.
* This is used in LIMIT statements to queries which return
* individuals to be e-mailed with one thing or another..
* NOTE: A value of 0 means "no limit".
*/
define("EMAIL_BULK_LIMIT",   0);

// MIME Encodings
/** 7-bit Encoding */
define("ENC_BIT7",          "7bit");
/** Base-64 Encoding */
define("ENC_BASE64",        "base64");
/** Quoted-Printable Encoding */
define("ENC_QUOTP",         "quoted-printable");

// MIME Message Content Types
/** Mixture of content: text/html/etc. */
define("MIXED",             "multipart/mixed");
/** Alternative formats of same content */
define("ALTERNATIVE",       "multipart/alternative");
/** Parts intended to be viewed all at once */
define("PARALLEL",          "multipart/parallel");
/** A digest of multiple attachments */
define("DIGEST",            "multipart/digest");

// Misc internal constants
/** No email subject string
* @access private */
define('NOSUBJECT',         "(No Subject)");
/** MIME warning message
* @access private */
define('MIME_WARNING',      "This is a multi-part message in MIME format.");
/** The default character set
* @access private */
define("DEFAULT_CHARSET",   "us-ascii");
/** MIME content is in-line
* @access private */
define("INLINE",            "inline");
/** MIME content is an attachment
* @access private */
define("ATTACH",            "attachment");
/** Carriage-return Linefeed sequence
* @access private */
define('CRLF',              "\r\n");
/** The body of the e-mail
* @access private */
define("BODY",              CRLF . "BODY" . CRLF);

// -------------------------------------------------------------------------
/**
* email class
* A class which encapsulates all the functions required to compose
* and send an e-mail message. Also caters for MIME attachments.
* @package email
*/
class email {
  // Public
  /** e-mail From address */
  var $from = "";
  /** Address to send e-mail */
  var $to = array();
  /** Cc: (copies to) header */
  var $cc = array();
  /** Bcc: (blind copies to) header */
  var $bcc = array();
  /** e-mail ReplyTo address */
  var $replyto = "";
  /** Subject of the e-mail */
  var $subject = "";
  /** e-mail body text */
  var $body = "";

  // Private
  /** Accumulated errors
      @access private */
  var $errors = array();
  /** Extra ad-hoc e-mail headers added from external
      calls. Use this if you need to add a header not
      covered by To(), From(), ReplyTo() etc. methods
      @access private */
  var $extra_headers = array();
  /** All e-mail headers end up here, apart from To: This
      array is a transient place to build all headers
      prior to sending the email
      @access private */
  var $email_headers = array();
  /** If !specified use PHP's base64
      @access private */
  var $base64_func = "";
  /** None at this time
      @access private */
  var $qp_func = "";
  /** E-mail body + headers minus From: and Subject:
      @access private */
  var $content = "";
  /** MIME content type for the e-mail
      @access private */
  var $mimecontenttype = CONTENT_TEXT;
  /** MIME boundary marker to use
      @access private */
  var $mimeboundary = "";
  /** character set to use for the e-mail
      @access private */
  var $charset = DEFAULT_CHARSET;
  /** Assembled mime attachments
      @access private */
  var $mimeparts = array();
  //.....................................................................
  /**
  * Constructor
  * Creates the basic email object.
  * @param string  $from             From: email address
  * @param string  $to               To: email address
  * @param string  $subject          Subject line of the e-mail
  * @param string  $body             Body text of the e-mail
  * @param string  $headers          Extra headers to include for this e-mail
  * @param string  $mimecontenttype  Mime content type
  * @param string  $mimeboundary     Pattern designating mime boundary
  */
  function email(
      $from="",
      $to="",
      $subject="",
      $body="",
      $headers="",
      $mimecontenttype=CONTENT_TEXT,
      $mimeboundary=""
  ) {
    debug_trace($this);
    $this->From($from);
    $this->To($to);
    $this->Subject($subject);
    $this->Body($body);

    // The headers can be passed as an array, or as a
    // string of comma-delimited headers..
    if (is_array($headers)) {
      $this->extra_headers = $headers;
    }
    elseif ($headers != "") {
      $this->extra_headers = explode(",", $headers);
    }

    // MIME settings..
    $this->mimeparts[] = "";
    $this->mimecontenttype = $mimecontenttype;
    if (empty($mimeboundary)) {
      $this->mimeboundary = "catit_" . chr(rand(65, 91)) . '------' . md5(uniqid(rand()));
    }
    else {
      $this->mimeboundary = $mimeboundary;
    }
    debug_trace();
    return;
  } // email
  // .........................................................................
  /** Set subject
  * Set the e-mail Subject: header
  * @param string $subject Subject for this email
  */
  function Subject($subject) {
    $this->subject = $subject;
  } // Subject
  // .........................................................................
  /** Set body
  * Set the e-mail body content
  * @param string $body Body content for this email
  */
  function Body($body) {
    $this->body = $body;
  } // Body
  // .........................................................................
  /** Set from
  * Set the e-mail From: e-mail addresses.
  * @param string $from e-mail address e-mail comes from
  */
  function From($from) {
    $this->from = $from;
  } // From
  // .........................................................................
  /** Set replyto
  * Set the e-mail ReplyTo: e-mail address.
  * @param string $replyto e-mail address recipient replies to
  */
  function ReplyTo($replyto) {
    $this->replyto = $replyto;
  } // ReplyTo
  // .........................................................................
  /** Set to
  * Set the e-mail To: e-mail addresses.
  * The supplied e-mail addresses can be a comma-delimited list.
  * NB: Every time this method is called, addresses are appended.
  * @param string $to List of e-mail addresses to send e-mail to
  */
  function To($to) {
    $this->add_array_or_list($to, $this->to);
  } // To
  // .........................................................................
  /** Set copies-to
  * Set the e-mail Cc: e-mail addresses for copies to.
  * The supplied e-mail addresses can be a comma-delimited list.
  * NB: Every time this method is called, addresses are appended.
  * @param string $cc List of e-mail addresses to copy e-mail to
  */
  function Cc($cc) {
    $this->add_array_or_list($cc, $this->cc);
  } // Cc
  // .........................................................................
  /** Set blind copies-to
  * Set the e-mail Bcc: e-mail addresses for blind copies to.
  * The supplied e-mail addresses can be a comma-delimited list.
  * NB: Every time this method is called, addresses are appended.
  * @param string $bcc List of e-mail addresses to blind copy e-mail to
  */
  function Bcc($bcc) {
    $this->add_array_or_list($bcc, $this->bcc);
  } // Bcc
  // .........................................................................
  /** Add an array or list of email addresses to the given variable
  * passed in by reference.
  * @param mixed $addrs Array or comma-delimited list of addresses to add
  * @param mixed $addrvar Reference to address var to assign addresses
  * @access private
  */
  function add_array_or_list($addrs, &$addrvar) {
    if (is_array($addrs)) {
      $newads = $addrs;
    }
    else {
      $newads = explode(",", $addrs);
    }
    if (is_array($newads)) {
      foreach ($newads as $newad) {
        if ($newad != "") {
          $addrvar[] = $newad;
        }
      }
    }
  } // add_array_or_list
  // .........................................................................
  /**
  * Append new content to the body of the email.
  * @param string $content New content to add to the existing email body.
  */
  function add_content($content) {
    $this->body .= $content;
  } // add_content
  // .........................................................................
  /** Set the character set encoding for the email.
  * @param string $charset_code Code of the characterset to use for the email
  */
  function charset($charset_code) {
    debug_trace($this);
    $this->charset = $charset_code;
    debug_trace();
  } // charset
  // .........................................................................
  /**
  * Generic function to add a header. We store our headers in an associative
  * array, keyed on the proper-cased header name, so we avoid duplicates.
  * @param string $header Header to add to e-mail in 'Headername: value' format
  * @deprecated the new function extra_header() is now preferred
  */
  function add_header($header) {
    debug_trace($this);
    $bits = explode(":", $header);
    $header_name = ucfirst(strtolower(trim($bits[0])));
    $header_value = trim(trim($bits[1]));
    $this->extra_headers[$header_name] = $header_value;
    debug_trace();
  } // add_header
  // .........................................................................
  /**
  * Generic method to add an extra header. This method is now preferred over
  * the depreceted 'add_header()' method above. We store our headers in an
  * associative array, keyed on the proper-cased header name.
  * @param string $headername Name of the Header to add in this build
  * @param string $headervalue Value of the Header to add in this build
  */
  function extra_header($headername, $headervalue) {
    debug_trace($this);
    if (trim($headervalue) != "") {
      $headername = ucfirst(strtolower($headername));
      $this->extra_headers[$headername] = $headervalue;
    }
    debug_trace();
  } // extra_header
  // .........................................................................
  /**
  * Method to build the header array when building the e-mail to send it. We
  * store our headers in an associative array, keyed on the header name.
  * This method is generally for internal use, and is called every time
  * the email is built for sending.
  * @param string $headername Name of the Header to add in this build
  * @param string $headervalue Value of the Header to add in this build
  * @access private
  */
  function build_header($headername, $headervalue) {
    debug_trace($this);
    if (trim($headervalue) != "") {
      $headername = ucfirst(strtolower($headername));
      $this->email_headers[$headername] = $headervalue;
    }
    debug_trace();
  } // build_header
  // .........................................................................
  /**
  * Attaches a 'file' to the e-mail message. Pass a file pathname to attach.
  * This function returns a success/failure code/key of current
  * attachment in array (+1). @see attach()
  * @param string $path         Path to the file to attach
  * @param string $contenttype  Mime content type of this attachment
  * @param string $encoding     Encoding type for this attachment
  * @param string $disp         Content disposition
  */
  function attach_file($path, $description="", $contenttype=CONTENT_OCTET, $encoding=ENC_BASE64, $disp="") {
    debug_trace($this);
    $this->mimecontenttype = MIXED;
    if (!file_exists($path)) {
      $this->errors[] = "attach_file: file '$path' does not exist";
      return false;
    }
    // Read in file
    $fp = fopen($path, "rb");  # (b)inary for Win compatability
    if (!$fp) {
      $this->errors[] = "attach_file: could not open '$path' for reading";
      return false;
    }
    $contenttype .= ";\r\n\tname=" . basename($path);
    $data = fread($fp, filesize($path));
    debug_trace();
    return $this->attach($data,
        $description,
        $contenttype,
        $encoding,
        $disp
        );
  } // attach_file
  // .........................................................................
  /**
  * Just a convenient wrapper for adding HTML attachments. Note that with
  * this call we are assuming that we are going to be sending an e-mail
  * with a plain text and an HTML equivalent. This is why we set the
  * MIME content type to "multipart/alternative" here. This can be
  * over-ridden using the parameter in the send(0 function.
  * @param string $data         The HTML data to attach
  * @param string $encoding     Encoding type for this attachment
  */
  function attach_html($data, $encoding=ENC_BASE64) {
    $this->mimecontenttype = ALTERNATIVE;
    return $this->attach_raw(
              $data,
              "",
              CONTENT_HTML,
              $encoding
              );
  } // attach_html
  // .........................................................................
  /**
  * Wrapper for general binary attachments. We assume a MIME content
  * type of multipart/mixed for these..
  * @param string $data         The binary data to attach
  * @param string $contenttype  Mime content type of this attachment
  * @param string $encoding     Encoding type for this attachment
  * @param string $disp         Content disposition
  */
  function attach($data, $description="", $contenttype=CONTENT_OCTET, $encoding=ENC_BASE64, $disp="") {
    $this->mimecontenttype = MIXED;
    return $this->attach_raw(
              $data,
              $description,
              $contenttype,
              $encoding,
              $disp
              );
  } // attach
  // .........................................................................
  /** Attach raw
  * The raw routine for attaching content to the e-mail. This method is
  * used by the other 'friendlier' attachment methods.
  * @param string $data         The binary data to attach
  * @param string $description  Forms the content-description header
  * @param string $contenttype  Mime content type of this attachment
  * @param string $encoding     Encoding type for this attachment
  * @param string $disp         Content disposition
  * @access private
  */
  function attach_raw($data, $description="", $contenttype=CONTENT_OCTET, $encoding=ENC_BASE64, $disp="") {
    if (empty($data)) {
      $this->errors[] = "attach_raw: no data to be attached";
      return false;
    }
    // Handle default encodings..
    if (trim($contenttype) == "") $contenttype = OCTET;
    if (trim($encoding) == "") $encoding = ENC_BASE64;

    // Ascii text 7-Bit standard..
    if ($encoding == ENC_BIT7) {
      $emsg = $data;
    }
    // User-defined function..
    elseif ($encoding == ENC_QUOTP) {
      $emsg = $$this->qp_func($data);
    }
    // Good ol' Base-64..
    elseif ($encoding == ENC_BASE64) {
      // Check if there is user-defined function..
      if (!$this->base64_func) {
        $emsg = base64_encode($data);
      }
      else {
        $emsg = $$this->base64_func($data);
      }
      // Splits data stream into 76 char lines
      // to fit RFC2045 semantics..
      $emsg = chunk_split($emsg);
    }

    
    //Check if content-type is text/plain and if charset is not specified append default CHARSET
    if (strstr($contenttype, CONTENT_TEXT) && !strstr(contenttype, "/;charset=/i")) {
      $contenttype .= ";\r\n\tcharset=" . $this->charset;
    }
    $msg = "";
    $msg .= "Content-Type: $contenttype" . CRLF;
    $msg .= "Content-Transfer-Encoding: $encoding" . CRLF;
    if ($disp != "") {
      $msg .= "Content-Disposition: $disp";
      if ($description != "") {
        $msg .= "; filename=\"$description\"";
      }
      $msg .= CRLF;
    }
    elseif ($description != "" && BODY != $description) {
      $msg .= "Content-Description: $description" . CRLF;
    }
    $msg .= CRLF . $emsg . CRLF;
    BODY == $description ? $this->mimeparts[0] = $msg : $this->mimeparts[] = $msg;

    return count($this->mimeparts);
  } // attach_raw
  // .........................................................................
  /**
  * Build the message body.
  * Construct mail message body and any MIME attachments. This also includes
  * assignment of the relevant headers, depending on the attachments which
  * have been defined (or not).
  * @access private
  */
  function build_body() {
    debug_trace($this);
    $msg = "";
    $nparts = count($this->mimeparts);

    // Case 1: Attachment list is there.  Therefore MIME Message header must have multipart/mixed
    if (is_array($this->mimeparts) && ($nparts > 1)) {
      $this->build_header("MIME-Version", "1.0");
      $this->build_header("Content-Type", "$this->mimecontenttype;" . CRLF . "\tboundary=\"$this->mimeboundary\"");
      $this->build_header("Content-Transfer-Encoding", ENC_BIT7);
      $msg .= CRLF . MIME_WARNING . CRLF . CRLF ;

      // Since we are here, it means we do have attachments and
      // the body must become an attachment too..
      if (!empty($this->body)) {
        $this->attach($this->body, BODY, CONTENT_TEXT, ENC_BIT7);
      }
      // Now create the MIME parts of the email
      foreach ($this->mimeparts as $mimepart) {
        if (!empty($mimepart)) {
          $msg .= CRLF . '--' . $this->mimeboundary . CRLF . $mimepart . CRLF;
        }
      }
      $msg .= "--" . $this->mimeboundary . "--" . CRLF;
    }
    else {
      $this->build_header("Content-Type", "$this->mimecontenttype;" . CRLF . "\tcharset=" . $this->charset);
      if (!empty($this->body)) {
        $msg .= CRLF . $this->body . CRLF . CRLF;
      }
    }
    debug_trace();
    return $msg;
  } // build_body
  // .........................................................................
  /**
  * Generate message content
  * Now Generate the entire Mail Message, header and body et al.
  * Note that this leaves out the To: and Subject: which are passed
  * as separate parameters when mailing the e-mail with 'mail()'.
  * @return string The full email content including headers and body
  * @access private
  */
  function generate_content() {
    $this->email_headers = $this->extra_headers;
    if (empty($this->subject)) {
      $this->subject = NOSUBJECT;
    }
    if (!empty($this->from)) {
      $this->build_header("From", $this->from);
    }
    if (!empty($this->replyto)) {
      $this->build_header("ReplyTo", $this->replyto);
    }
    if (is_array($this->cc) && count($this->cc) > 0) {
      $this->build_header("Cc", implode(",", $this->cc));
    }
    if (is_array($this->bcc) && count($this->bcc) > 0) {
      $this->build_header("Bcc", implode(",", $this->bcc));
    }
    $this->build_header("X-Mailer", "Catalyst-IT/AXYL");
    $this->content = $this->build_body();
    return $this->content;
  } // generate_content
  // .........................................................................
  /**
  * Return Printable Content
  * Returns a printable version of the e-mail. Just returns the content
  * as well as all the headers, and if for HTML, then it wraps it all in
  * some (pre) tags. Useful for debugging.
  * @param string $format Format of output, either 'text' (default) or 'html'
  */
  function printable($format="text") {
    $email =  $this->generate_content();
    $eol = "\n";
    $email = "";
    $email .= "Subject: $this->subject" . $eol;
    $email .= "To: " . implode(",", $this->to) . $eol;
    $headers = array();
    foreach ($this->email_headers as $headername => $headervalue) {
      $headers[] = "$headername: $headervalue";
    }
    $email .= implode($eol, $headers);
    $email .= $this->content;
    if ($format == "html") {
      $email = "<pre>$email</pre>";
    }
    return $email;
  } // printable
  // .........................................................................
  /**
  * Return error messages. These may have accumulated during the email
  * assembly or during the send process.
  * @return string Error message(s) which have accumulated.
  */
  function error_message() {
    $errmsg = "";
    if (count($this->errors) > 0) {
      foreach ($this->errors as $err) {
        $errmsg .= "$err\n";
      }
    }
    return $errmsg;
  } // error_message
  // .........................................................................
  /**
  * Send the email
  * Send mail via local mailer. This is usually the end-result of
  * an e-mail sequence and results in the e-mail being sent.
  * @param string $mimecontenttype Override for the email MIME content type
  * @param string $charset Override for the email character set
  * @return boolean True if email was sent successfully
  */
  function send($mimecontenttype="", $charset="") {
    debug_trace($this);
    $sentok = false;

    // Allow them to set MIME content type late..
    if ($mimecontenttype != "") {
      $this->mimecontenttype = $mimecontenttype;
    }
    $this->generate_content();
    // Must have a To address..
    if (count($this->to) == 0) {
      $this->errors[] = "send: To: header was not specified";
      return false;
    }
    // Unfurl all the headers..
    $headers = array();
    foreach ($this->email_headers as $headername => $headervalue) {
      $headers[] = "$headername: $headervalue";
    }

    // Send the e-mail on its way..
    if (!empty($this->from)) {
      // The -f parameter sets who the mail is from, by setting the email
      // address on the 'envelope' of the mail. This can only be done by a
      // user 'trusted' by sendmail, so make sure the user the webserver
      // runs as (eg. www-data) is marked as trusted in this way.
      $sentok = mail(
                implode(",", $this->to),
                $this->subject,
                $this->content,
                implode(CRLF, $headers),
                "-f$this->from"
                );
    }
    else {
      $sentok = mail(
                implode(",", $this->to),
                $this->subject,
                $this->content,
                implode(CRLF, $headers)
                );
    }
    // Debug output..
    if (debugging()) {
      debugbr( $this->printable("html"), DBG_DUMP );
    }
    debug_trace();
    if (!$sentok) {
      $this->errors[] = "send: mail() function returned fail status";
    }
    return $sentok;
  } // send
} // Class email

// -------------------------------------------------------------------------
/**
* Function to send an email using the simple Php mail() call. This is a
* very simplistic wrapper which is kept for its cuteness, but for no
* other reason than that.
* @param string $eto            The To: email address
* @param string $efrom          The From: email address
* @param string $esubject       The Subject: line
* @param string $ebody          The email body content
* @param string $emailformat    Format of e-mail 'plain', 'html' or 'mixed'
* @param string $mime_boundary  The mime boundary pattern
*/
function emailthem($eto, $eCc, $efrom, $esubject, $ebody, $emailformat="plain", $mime_boundary="") {
  $headers  = "From: $efrom\n";
  if ($eCc != "") $headers .= "Cc: $eCc\n";
  $headers .= "Reply-To: $efrom\n";
  $headers .= "Errors-To: $efrom\n";
  $headers .= "Return-Path: $efrom\n";

  switch ($emailformat) {
    case "html":
      $headers .= "Content-Type: text/html\n";
      break;
    case "plain":
      $headers .= "Content-Type: text/plain\n";
      break;
    case "mixed":
      $headers .= "MIME-Version: 1.0\n";
      $headers .= "Content-Type: multipart/alternative; boundary=\"$mime_boundary\"\n";
      break;
    default:
      $headers .= "Content-Type: text/plain\n";
  } // switch

  $headers .= "X-Mailer: PHP/" . phpversion();

  // The -f parameter sets who the mail is from, by setting the email
  // address on the 'envelope' of the mail. This can only be done by a
  // user 'trusted' by sendmail, so make sure the user the webserver
  // runs as (eg. www-data) is marked as trusted in this way.
  $sentok = mail(
            $eto,
            $esubject,
            $ebody,
            $headers,
            "-f$efrom"
            );
  // Return status of send..
  return $sentok;
} // emailthem

// -------------------------------------------------------------------------
/**
* Make a 'friendly' name from a full one. Good for "Dear... ,"
* @param string $fullname  The full name to make a friendly version of
* @return string The friendly name
*/
function friendlyName($fullname) {
  $splitname = explode(" ", $fullname);
  $mate = trim($splitname[0]);
  if ($mate == "") $mate = $fullname;
  return $mate;
} // friendlyName

// -------------------------------------------------------------------------
?>
