#!/bin/bash
# Install a POSTGRESQL database for an Axyl website.

# NOTE: This is not normally run standalone. The main DB installation
# script 'install/install-db.sh normally runs this.
# +++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++

# The database type this install script is for. This is just
# for messages and display..
DBTYPE="PostgreSQL"

# Name of this script
SCRIPTNAME=${0##*/}

# INCOMING PARAMETERS
#  $1  - INSTALLTYPE
#  $2  - DBNAME
#  $3  - DBUSER
#  $4  - DBPASSWD ('none' means a blank password)
#  $5  - DBHOST ('direct' means a local database server)
#  $6  - DBPORT
#  $7  - META ('y' or 'n') - meta-data extension
#  $8  - MICRO ('y' or 'n') - microsites extension
#  $9  - AXVER (Axyl version, '' or 'n.n.n'
#  $10 - APFX (Application prefix)
#  $11 - DOCROOT (site docroot of website)
INSTALLTYPE=""
DBNAME=""
DBUSER=""
DBPASSWD=""
DBHOST=""
DBPORT=5432
META=n
MICRO=n
AXVER="unknown"
APFX=""
DOCROOT=""

POS=0
while [ $# -gt 0 ] ; do
	POS=`expr $POS + 1`
	case $POS in
		1)  INSTALLTYPE=$1
			;;
		2)  DBNAME=$1
			;;
		3)  DBUSER=$1
			;;
		4)  DBPASSWD=$1
			;;
		5)  DBHOST=$1
			;;
		6)  DBPORT=$1
			;;
		7)  META=$1
			;;
		8)  MICRO=$1
			;;
		9)  AXVER=$1
			;;
		10)  APFX=$1
			;;
		11)  DOCROOT=$1
			;;
	esac
	shift
done

if [ "$INSTALLTYPE" = "" -o "$AXUSER" = "" -o "$AXYL_HOME" = "" -o "$AXYL_DATA" = "" ] ; then
	echo "ERROR: $DBTYPE install-db.sh: parameters."
	echo "usage: install-db.sh empty|demo dbname dbuser [dbpasswd] [dbhost] [dbport] [ax ver] [app pfx]"
	exit 1
fi

echo " $DBTYPE database - $INSTALLTYPE version"
echo " database name: $DBNAME"
echo " connecting as: $DBUSER"
[ "$DBPASSWD" != "none" ] && echo " password: $DBPASSWD"
[ "$DBHOST" != "direct" ] && echo " remote host: $DBHOST"
[ "$DBHOST" != "direct" ] && echo " port: $DBPORT"
echo ""

# Optional host settings for remotely accessed databases..
HOSTOPTS=""
[ "$DBHOST" != "direct" ] && HOSTOPTS="--host $DBHOST --port $DBPORT"

# We require Postgres to be locally installed, at least as postgresql-client
# even if no servers are created locally. This is based on the standard
# Debian location, with a few likely fallbacks.

# Detect database, and set up database vars. This set up the following
# variables:
# PG_MULTI_CLUSTER     # Eg. '8.1/main' Postgres version and cluster
# PG_VERSION           # Version of the database eg. '8.1'
# PG_VERSION_SUFFIX    # Version suffix eg. '-8.1'
# PG_BIN               # Path to the Postgres binary files
# PG_CONF              # Path to the Postgre configuration files

. ${AXYL_HOME}/db/postgres/detect-db.sh $DBNAME $DBUSER $DBPASSWD $DBHOST $DBPORT

# Now set paths to our executables
PG_INITD=/etc/init.d/postgresql${PG_VERSION_SUFFIX}
PSQL=${PG_BIN}/psql
CREATEDB=${PG_BIN}/createdb
CREATELANG=${PG_BIN}/createlang

# LOCALHOST INSTALLATION
if [ "$DBHOST" = "direct" -o "$DBHOST" = "localhost" -o "$DBHOST" = "127.0.0.1" ] ; then
	
	# Try to detect the Postgres security settings, since these will determine
	# what we can and cannot do in the way of configuration from here.
	# First, let's locate the config files..
	if [ "$PG_CONF" != "" ] ; then
		PG_HBA=${PG_CONF}/pg_hba.conf
		# Local connection settings - postgres user
		LPTRUST=`perl -n -e "m;^local[\s]+all[\s]+postgres[\s]+trust$; && print;" ${PG_HBA}`
		LPTRUST_ENTRY="local all postgres trust"
		
		# Local connection settings - database user
		LUTRUST=`perl -n -e "m;^local[\s]+all[\s]+${DBUSER}[\s]+trust$; && print;" ${PG_HBA}`
		LUTRUST_ENTRY="local all $DBUSER trust"
		
		# Remote host connection settings - postgres user
		HPTRUST=`perl -n -e "m;^host[\s]+all[\s]+postgres[\s]+127.0.0.1[\s]+255.255.255.255[\s]+trust$; && print;" ${PG_HBA}`
		HPTRUST_ENTRY="host all postgres 127.0.0.1 255.255.255.255 trust"
		
		# Remote host connection settings - database user
		HUTRUST=`perl -n -e "m;^host[\s]+${DBNAME}[\s]+${DBUSER}[\s]+127.0.0.1[\s]+255.255.255.255[\s]+trust$; && print;" ${PG_HBA}`
		HUTRUST_ENTRY="host $DBNAME $DBUSER 127.0.0.1 255.255.255.255 trust"
		  
		if [ "$LPTRUST" = "" -o "$LUTRUST" = "" -o "$HPTRUST" = "" -o "$HUTRUST" = "" ] ; then
			tell ""
			tell "Postgres Security Settings:"
			tell "Your current Postgres security settings do not have a 'trust' set up for"
			tell "local database connections for '$DBNAME'. This will probably prevent"
			tell "this script from doing its job. The script can fix this by putting"
			tell "the following into your ${PG_HBA}"
			tell "file for you, otherwise you can set something similar up yourself:"
			tell ""
			if [ "$LPTRUST" = "" ] ; then
				tell $LPTRUST_ENTRY
			fi
			if [ "$HPTRUST" = "" ] ; then
				tell $HPTRUST_ENTRY
			fi
			if [ "$LUTRUST" = "" ] ; then
				tell $LUTRUST_ENTRY
			fi
			if [ "$HUTRUST" = "" ] ; then
				tell $HUTRUST_ENTRY
			fi
			# Read ANS from commandline, or if we are in a non-interactive
			# mode then set ANS to "y"
			tell ""
			tell -n "Make postgres security changes now? [Yn]:"
			getans "y"
			[ "$ANS" = "" ] && ANS="y"
			if [ "$ANS" = "y" -o "$ANS" = "Y" -o "$ANS" = "yes" ] ; then
				PG_HBA_NEW=${PG_HBA}.new
				DONE=0
				(while read line ; do
					echo $line >> $PG_HBA_NEW
					if [ $DONE -eq 0 ] ; then
						A=`echo $line | perl -n -e "m;(^# Put your actual configuration here)|(^# TYPE ); && print;"`
						if [ "$A" != "" ] ; then
							if [ "$LPTRUST" = "" ] ; then
								echo $LPTRUST_ENTRY >> $PG_HBA_NEW
							fi
							if [ "$HPTRUST" = "" ] ; then
								echo $HPTRUST_ENTRY >> $PG_HBA_NEW
							fi
							if [ "$LUTRUST" = "" ] ; then
								echo $LUTRUST_ENTRY >> $PG_HBA_NEW
							fi
							if [ "$HUTRUST" = "" ] ; then
								echo $HUTRUST_ENTRY >> $PG_HBA_NEW
							fi
							DONE=1
						fi
					fi
				done) < $PG_HBA
				TSTAMP=`date +'%F_%H%M%S'`
				mv $PG_HBA ${PG_HBA}.${TSTAMP}
				mv $PG_HBA_NEW $PG_HBA
				tell "security changes have been made"
				tell "re-loading Postgresql with the new config.."
				$PG_INITD reload
			else
				tell "Ok, no changes - installation will proceed, but you are on your own!"
			fi
		else
			tell "Postgresql security settings are ok, no changes necessary"
		fi
	fi
	
	# The database user should have an account on the machine too..
	mkuser $DBUSER normal default any /home/$DBUSER /bin/bash Axyl DB Admin

	# Make sure all of the required database connection users are present. These
	# include the application connection user $DBUSER, but we also need the
	# users for admin and reports usage, so we assert those here. 
	DBUSERS="$DBUSER"
	for DBUSER in $DBUSERS ; do
	  USER=`$PSQL --tuples-only --username postgres --dbname template1 --command "SELECT usename FROM pg_user WHERE usename='$DBUSER'" | tr -d ' '`
	  if [ "$USER" != "$DBUSER" ] ; then
	  	# Create the user, with no password
	    tell "creating database user '${DBUSER}'.."
	    $PSQL --username postgres --dbname template1 --command "CREATE USER $DBUSER CREATEDB"
	  else
	    tell "database user '$DBUSER' present."
	  fi
	done
	
	# Set password for main connection user, if given..
	if [ "$DBPASSWD" != "" ] ; then
	  $PSQL --username postgres --dbname template1 --command "ALTER USER $DBUSER WITH ENCRYPTED PASSWORD '$DBPASSWD'"
	fi

	# Check the database exists
	DB=`$PSQL --tuples-only --username postgres --dbname template1 --command "SELECT datname FROM pg_database WHERE datname='$DBNAME'" | tr -d ' '`
	if [ "$DB" = "" ] ; then
	  tell "creating the database $DBNAME now.."
	  su - postgres -c "${CREATEDB} --username ${DBUSER} --encoding UTF-8 ${DBNAME}"
	  if [ $? -ne 0 ] ; then
	    tell "Error: failed to create PostgreSQL database '${DBNAME}'."
	    tell "Maybe a problem in your PostgreSQL configuration?"
	    exit 8
	  fi
	else
	  tell "database '$DBNAME' already exists."
	fi
	
	# Assert plpgsql language is present
	su - postgres -c "${CREATELANG} plpgsql ${DB_NAME}" || true

# REMOTE INSTALLATION  
else
	if [ -x $PSQL ] ; then
		DB=`$PSQL --tuples-only --username postgres --dbname template1 $HOSTOPTS --command "SELECT datname FROM pg_database WHERE datname='$DBNAME'" | tr -d ' '`
		if [ "$DB" = "" ] ; then
			echo "Installing on remote database server --> ${DBHOST}:${DBPORT}"
			echo "(assuming Postgresql, user and passwords are already set up)"
			if [ "$DBPASSWD" = "none" ] ; then
				$PSQL --username $DBUSER --dbname template1 $HOSTOPTS --command "CREATE DATABASE $DBNAME"
			else
				echo "Enter password, when prompted, as: $DBPASSWD"
				$PSQL --username $DBUSER --password --dbname template1 $HOSTOPTS --command "CREATE DATABASE $DBNAME"
			fi
		fi
	else
		echo "ERROR: $PSQL not found."
		echo "to set up a Postgres database remotely, you must at least have the"
		echo "'psql' utility available locally. For Debian systems this requires"
		echo "installation of the 'postgresql-client' package. For other systems,"
		echo "download the applicable RPM, or tarball from the website:"
		echo "http://www.postgresql.org/"
		exit 1
	fi
fi

# Install/upgrade the database contents (tables and data)
if [ -x $PSQL ] ; then
	DB=${AXYL_HOME}/db/postgres
	DBSCHEMA=${DB}/axyl_core.sql
	DBTRIG=${DB}/axyl_trig.sql
	DBCOREDATA=${DB}/axyl_core_data.sql
	DBMETADATA=${DB}/axyl_meta_data.sql
	DBMICRODATA=${DB}/axyl_microsite_data.sql
	DBDEMO=${DB}/axyl_demo.db
	DBTEMP=`tempfile -s axdbtmp`
	if [ ! -f $DBMETADATA ] ; then
		META=n
	fi
	if [ ! -f $DBMICRODATA ] ; then
		MICRO=n
	fi
	
	# Create database to load. We create a temporary file with the SQL in
	# it to define the database, potentially with fixed-up Lucene port..
	if [ "$LUCENE_PORT" != "" ] ; then
		CHARCNT=`echo $LUCENE_PORT | wc --chars`
		CHARCNT=`expr $CHARCNT - 1`
	fi
	
	case "$INSTALLTYPE" in
		empty)
			if [ "$LUCENE_PORT" != "" ] ; then
				sed -e "s/s:5:\"22222\";/s:$CHARCNT:\"$LUCENE_PORT\";/" $DBCOREDATA > $DBTEMP
			else
				sed -e "s/\"Lucene_Site_Indexing\";b:1;/\"Lucene_Site_Indexing\";b:0;/" $DBCOREDATA > $DBTEMP
			fi
			;;
		demo)
			if [ "$LUCENE_PORT" != "" ] ; then
				sed -e "s/s:5:\"22222\";/s:$CHARCNT:\"$LUCENE_PORT\";/" $DBDEMO > $DBTEMP
			else
				sed -e "s/\"Lucene_Site_Indexing\";b:1;/\"Lucene_Site_Indexing\";b:0;/" $DBDEMO > $DBTEMP
			fi
			;;
	esac
	
	# Now load the database, either direct or remotely..
	case "$INSTALLTYPE" in
		empty)
			# Load core schema and then core data..
			$PSQL --username $DBUSER --dbname $DBNAME $HOSTOPTS --file $DBSCHEMA
			$PSQL --username $DBUSER --dbname $DBNAME $HOSTOPTS --file $DBTRIG
			$PSQL --username $DBUSER --dbname $DBNAME $HOSTOPTS --file $DBTEMP
			if [ "$META" = "y" ] ; then
				$PSQL --username $DBUSER --dbname $DBNAME $HOSTOPTS --file $DBMETADATA
			fi
			if [ "$MICRO" = "y" ] ; then
				$PSQL --username $DBUSER --dbname $DBNAME $HOSTOPTS --file $DBMICRODATA
			fi
			# Insert the Axyl control record..
			LAST_DB_PATCH="000000000000-000100000000"
			$PSQL --username $DBUSER --dbname $DBNAME $HOSTOPTS --command "INSERT INTO ax_control (app_version,last_db_patch,updated_at) VALUES('$AXVER','$LAST_DB_PATCH',CURRENT_TIMESTAMP)"
			;;
		demo)
			# For the demo, we load a complete database dump which
			# includes the schema and data..
			$PSQL --username $DBUSER --dbname $DBNAME $HOSTOPTS --file $DBTEMP
			# Update the Axyl control record..
			LAST_DB_PATCH="000000000000-000100000000"
			$PSQL --username $DBUSER --dbname $DBNAME $HOSTOPTS --command "UPDATE ax_control SET app_version='$AXYL_VERSION',last_db_patch='${LAST_DB_PATCH}',updated_at=CURRENT_TIMESTAMP"
			;;
	esac
	rm -f $DBTEMP
	echo "finished"
else
	echo ""
	echo "ERROR: Could not import Axyl data - $PSQL not found."
	echo "to set up data, you must have the 'psql' utility available locally."
	echo "For Debian systems this requires installation of the 'postgresql-client'"
	echo "package. For other systems, download the applicable RPM, or tarball from"
	echo "the Postgres website: http://www.postgresql.org/"
	exit 1
fi

# END