<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    db-sqlite.php                                           */
/* Author:      Paul Waite                                              */
/* Description: Definitions for SQLITE database access.                 */
/*              NOTE: Only available in Php5                            */
/*                                                                      */
/* ******************************************************************** */
/** @package database */

/**
* SQLITE database interface
* This is a database interface class. It is an impedance-matcher
* between the high-level Phplib functions for accessing data, and
* the specific functions supplied by Php to access a particular
* flavour of databse such as Postgres, MS-SQL Server, Sybase etc.
* @package database
* @access private
*/
class db_sqlite extends database {
  /** Constructor */
  function db_sqlite($name="", $user="", $passwd="", $host="", $port=0, $enc="", $datestyle="") {
    $this->database($name, $user, $passwd, $host, $port, $enc, $datestyle);
    $this->type = "sqlite";
  }
  // ....................................................................
  /**
  * Connect to the database.
  * @param boolean $persistent Whether to connect persistently or not
  * @return boolean Status true if connected successfully
  */
  function connect($persistent=NOT_PERSISTENT) {
    if (!$this->connected) {
      $connstr = "";
      $connstr .= " dbname=" . $this->name;
      $connstr = trim($connstr);
      if ($persistent)
        $this->dbid = sqlite_popen("$connstr");
      else
        $this->dbid = sqlite_open("$connstr");
      if ($this->dbid) {
        if ($this->datestyle != "") $this->set_datestyle($this->datestyle);
        if ($this->enc != "") $this->set_char_encoding($this->enc);
        $this->connected = true;
      }
    }
    return $this->connected;
  }
  // ....................................................................
  /** Disconnect from the database, if connected. */
  function disconnect() {
    if (sqlite_close($this->dbid))
      $this->connected = false;
  }
  // ....................................................................
  /**
  * Execute a query on the connected database.
  * @param string $sql The SQL query to execute on the database
  * @return resource A database query resource ID, or false if query failed
  */
  function query($sql) {
    $sql = $this->convert_boolean_syntax($sql);
    $this->timer->restart();
    $rid = sqlite_query($this->dbid, $sql);
    $this->timer->stop();
    $this->executable_sql = $sql;
    $this->rid = $rid;
    $this->query_report();
    return $rid;
  }
  // ....................................................................
  /**
  * Return the number of rows returned by a SELECT query.
  * @param resource $rid The resource ID for the executed query
  * @return integer The number of rows returned by the query
  */
  function numrows($rid) {
    return sqlite_num_rows($rid);
  }
  // ....................................................................
  /**
  * Return the number of rows affected by a query.
  * @param resource $rid The resource ID for the executed query
  * @return integer The number of rows affected by the query
  */
  function affectedrows($rid) {
    return sqlite_changes($rid);
  }
  // ....................................................................
  /**
  * Free a resource.
  * @param resource $rid The resource ID for the executed query
  */
  function freeresult($rid) {
    // does nothing for sqlite
  }
  // ....................................................................
  /**
  * Return the last error message.
  * @return string The last error message which was generated
  */
  function errormessage() {
    return sqlite_error_string( sqlite_last_error($this->dbid) );
  }
  // ....................................................................
  /**
  * Return the specified row, as a standard (enumerated) array of
  * field values.
  * @param resource $rid The resource ID for the executed query
  * @param integer $rowno Row number (zero-based) of row to return
  * @return array Enumerated array of field values
  */
  function fetch_row($rid, $rowno) {
    sqlite_rewind($rid);
    for ($r=0; $r <= $rowno; $r++) {
      if (sqlite_next($rid) === false) break;
    }
    return sqlite_current($rid, SQLITE_NUM);
  }
  // ....................................................................
  /**
  * Return the specified row, as an associative array of fields
  * in a fieldname => value format.
  * @param resource $rid The resource ID for the executed query
  * @param integer $rowno Row number (zero-based) of row to return
  * @return array Associative array of field values
  */
  function fetch_array($rid, $rowno) {
    sqlite_rewind($rid);
    for ($r=0; $r <= $rowno; $r++) {
      if (sqlite_next($rid) === false) break;
    }
    return sqlite_current($rid, SQLITE_ASSOC);
  }
  // ....................................................................
  /**
  * Return a Php boolean from a database field value. The database field
  * is expected to be a container of some form of logical value. Here
  * is where we convert it according to the current database.
  * NOTE: For SQLite we are using strings 'true' and 'false' (lowercase)
  *       to denote boolean values.
  * @param mixed $dbvalue The value from the database field to convert
  * @return boolean The boolean value derived from the field value
  */
  function bool_from_db_value($dbvalue) {
    return (
      (is_string($dbvalue)  && strtolower($dbvalue) === "true")
    );
  }
  // ....................................................................
  /**
  * Return a suitable database field value to contain the value for
  * the given boolean.
  * NOTE: For SQLite we are using strings 'true' and 'false' (lowercase)
  *       to denote boolean values.
  * @param boolean $boolvalue The boolean value to convert
  * @return mixed The value suitable for the database field
  */
  function db_value_from_bool($boolvalue) {
    return ((is_bool($boolvalue) && $boolvalue === true) ? "true" : "false");
  }
  // ....................................................................
  /**
  * Return the current sequence value, given a sequence name, the table
  * and the field it applies to.
  * NOTE: For SQLite 'sequencename' is not used, only 'table' and 'column'
  *       and we return the MAX() value from the given table.column.
  * @param string $sequencename The name of the sequence to use
  * @param string $table The name of the table the sequence is for
  * @param string $column The name of the table column the sequence is for
  * @return integer The current sequence value
  */
  function current_sequencevalue($sequencename, $table, $column) {
    $seq = 0;
    $rid = $this->query("SELECT MAX($column') FROM $table" );
    if ($rid !== false) {
      $row = $this->fetch_row($rid, 0);
      $seq = $row[0];
    }
    return $seq;
  }
  // ....................................................................
  /**
  * Return the next sequence value, given a sequence name, the table
  * and the field it applies to.
  * NOTE: For SQLite 'sequencename' is not used, only 'table' and 'column'
  *       and we return the MAX() value + 1 from the given table.column.
  * @param string $sequencename The name of the sequence to use
  * @param string $table The name of the table the sequence is for
  * @param string $column The name of the table column the sequence is for
  * @return integer The next sequence value
  */
  function next_sequencevalue($sequencename, $table, $column) {
    $seq = $this->current_sequencevalue($sequencename, $table, $column);
    return ($seq + 1);
  }
  // ....................................................................
  /**
  * Set the sequence value, given a sequence name, the table
  * and the field it applies to.
  * @param integer $newval The sequence value to set
  * @param string $sequencename The name of the sequence to use
  * @param string $table The name of the table the sequence is for
  * @param string $column The name of the table column the sequence is for
  * @return boolean Whether the assignment succeeded or not
  */
  function set_sequencevalue($newval, $sequencename, $table, $column) {
    // Does nothing in SQLite
  }
  // ....................................................................
  /**
  * Set the database date style. This affect the format that dates will
  * be displayed in, and the format they are submitted in.
  * @param string $datestyle The date style code to set
  * @return boolean Whether the setting succeeded or not
  */
  function set_datestyle($datestyle) {
    // Does nothing in SQLite
  }
  // ....................................................................
  /**
  * Set the database character encoding. This affects the encoding of
  * characters in the database.
  * @param string $encoding The character encoding to set
  * @return boolean Whether the setting succeeded or not
  */
  function set_char_encoding($encoding) {
    return $this->query("PRAGMA encoding = '$encoding'");
  }
  // ....................................................................
  /**
  * Set the the lock mode on a list of tables in the database.
  * Locking in SQLite is database-wide and is either NORMAL
  * or EXCLUSIVE. Exclusive means a lock on all tables is
  * grabbed for the whole session. Normal locks it only for the
  * current transaction. Thus 'tablelist is ignored here.
  * @param string $tablelist A comma-delimited list of tables to lock
  * @param string $mode The mode to lock them in
  * @return boolean Whether the setting succeeded or not
  */
  function lock($tablelist, $mode) {
    $res = $this->query("PRAGMA " . $this->name . ".locking_mode $mode");
    return $res;
  }
  // ....................................................................
  /**
  * Given an Axyl SQL query object, build the SQL string from it
  * in suitable format for the currently connected database server.
  * @param pointer $sqlquery Pointer to an Axyl query object
  * @return string The SQL string built from the query object
  */
  function SQL(&$sqlquery) {
    $sql = "";
    switch (strtoupper($sqlquery->type)) {
      case "SELECT":
        $sql .= "SELECT ";
        if ($sqlquery->fields->total == 0) $sql .= "*";
        else $sql .= $sqlquery->fields->listed();
        $sql .= " FROM ";
        $sql .= $sqlquery->tables->listed();
        if ($sqlquery->where->total > 0) {
          $sql .= " WHERE ";
          $sql .= $sqlquery->where->listed(" ");
        }
        if ($sqlquery->groupby->total > 0) {
          $sql .= " GROUP BY ";
          $sql .= $sqlquery->groupby->listed();
        }
        if ($sqlquery->orderby->total > 0) {
          $sql .= " ORDER BY ";
          $sql .= $sqlquery->orderby->listed();
        }
        if ($sqlquery->limit > 0 || $sqlquery->offset > 0) {
          if ($sqlquery->limit > 0) {
            $sql .= " LIMIT $sqlquery->limit";
          }
          if ($sqlquery->offset > 0) {
            $sql .= " OFFSET $sqlquery->offset";
          }
        }
        break;

      case "INSERT":
        $sql .= "INSERT INTO ";
        $sql .= $sqlquery->tables->listed();
        if ($sqlquery->fields->total > 0) {
          $sql .= " (" . $sqlquery->fields->listed() . ")";
        }
        $sql .= " VALUES ";
        $sql .= "(" . $sqlquery->fields->values() . ")";
        break;

      case "DELETE":
        $sql .= "DELETE FROM ";
        $sql .= $sqlquery->tables->listed();
        if ($sqlquery->where->total > 0) {
          $sql .= " WHERE ";
          $sql .= $sqlquery->where->listed(" ");
        }
        break;

      case "UPDATE":
        $sql .= "UPDATE ";
        $sql .= $sqlquery->tables->listed();
        $sql .= " SET ";
        $sql .= $sqlquery->fields->equated();
        if ($sqlquery->where->total > 0) {
          $sql .= " WHERE ";
          $sql .= $sqlquery->where->listed(" ");
        }
        break;
    }
    // Render any NULL values..
    $SQL = str_replace("'".NULLVALUE."'", "NULL", $sql);

    // Return SQL we have built..
    return $SQL;
  }
  // ....................................................................
  /**
  * Make conversions of boolean syntax found in the SQL string and
  * return the 'standardised' SQL. This assumes that Axyl SQL will
  * be written in the form 'WHERE foo=TRUE'.
  * NOTE: For SQLite we are using strings 'true' and 'false' (lowercase)
  *       to denote boolean values.
  * @param string $sql SQL string to make conversions in
  * @return string The converted SQL string
  */
  function convert_boolean_syntax($sql) {
    $fixsql = $sql;
    // Quick check is more efficient then regexes..
    if (stristr($sql, "TRUE") || stristr($sql, "FALSE")) {
      $fixsql = preg_replace("/( WHERE.*?[\S]+=)TRUE/ie", "'\\1'.'true'", $sql);
      $fixsql = preg_replace("/( WHERE.*?[\S]+=)FALSE/ie", "'\\1'.'false'", $fixsql);
    }
    return $fixsql;
  }
}

// ----------------------------------------------------------------------
// Ensure an SQLite Php module is present..
if (!extension_loaded("sqlite3") && !extension_loaded("sqlite")) {
  if (!dl("sqlite3.so")) {
    if (!dl("sqlite.so")) {
      exit;
    }
  }
}
// ----------------------------------------------------------------------
?>