<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    dojo-editor-defs.php                                    */
/* Author:      Paul Waite                                              */
/* Description: Definitions for using the Dojo rich text editor.        */
/*                                                                      */
/* ******************************************************************** */
/** @package form */

/** Form elements */
include_once("form-defs.php");

/**
 *  NOTES: How to implement Dojo rich text editors in your forms:
 *
 *  // Include Wysiwyg classes at the top of the page..
 *  include_once("wysiwyg-editor-defs.php");
 *  // Define which plugins to load..
 *  add_wysiwyg_plugins("ContextMenu,CSS");
 *  ...
 *  // Build your form element..
 *  $Fld = new form_wysiwygfield("my_content", "Article", $my_content);
 *  // Register plugins this editor will use..
 *  $Fld->register_plugins("all");
 *  // Render the field..
 *  $myfield = $Fld->render();
 *  ...
 *  ... more as required ...
 *
 */

// ----------------------------------------------------------------------
/**
* A container class for configuration settings for the Dojo Editor
* internal configuration structure. We provide a selection of all the
* possible Dojo editor parameters, which represents a subset. This also
* renders the Javascript definition for the config @see render()
* @package form
*/
class dojo_editor_config {
  /** Width of Dojo Editor element (default 'auto') */
  var $width;
  /** Height of Dojo Editor element (default 'auto') */
  var $height;
  /** enable creation of a status bar? (default true) */
  var $statusBar;
  /** intercept ^V and use the Dojo Editor paste command
      If false, then passes ^V through to browser editor widget
      (default false) */
  var $editorPaste;
  /** maximum size of the undo queue (default 20) */
  var $undoSteps;
  /** the next parameter specifies whether the toolbar should
      be included in the size or not (default true) */
  var $sizeIncludesToolbar;
  /** if true then Dojo Editor will retrieve the full HTML, starting
      with the HTML tag (default false) */
  var $fullPage;
  /** style included in the iframe document (default nullstring) */
  var $pageStyle;
  /** set to true if you want Word code to be cleaned upon Paste
      (default true)*/
  var $killWordOnPaste;
  /** HTML tags to disallow (these have to be a regexp, or null
      if this functionality is not desired) (default null) */
  var $htmlRemoveTags;
  /** The toolbar. This takes the form of an array of named toolbar
      buttons. (defaulted to basic buttons) */
  var $toolbar;
  /** The list of font families to allow user to choose from
      (defaulted to standard list of font families) */
  var $fontnames;
  /** The list of font sizes to allow user to choose from
      (defaulted to standard list of font sizes) */
  var $fontsizes;
  /** The list of block format styles to allow user to choose from
      (defaulted to standard list of block formats) */
  var $blockformats;
  // Private
  /** A basic toolbar definition. Use for entering text in given
      font, with basic bold/italic/underline styling.
      @access private */
  var $toolbar_basic = array(
          "undo", "redo", "delete", "selectall",
          "|", "cut", "copy", "paste",
          "|", "bold", "italic", "underline",
          "|", "indent", "outdent", "justifyCenter", "justifyFull", "justifyLeft", "justifyRight"
          );
  /** A medium toolbar definition, adds colour, links, raw HTML editing,
      lists, and horizontal rule, etc.
      @access private */
  var $toolbar_medium = array(
          "undo", "redo", "delete", "selectall",
          "|", "cut", "copy", "paste",
          "|", "bold", "italic", "underline", "removeFormat",
          "|", "indent", "outdent", "justifyCenter", "justifyFull", "justifyLeft", "justifyRight",
          "|", "insertOrderedList", "insertUnorderedList"
          );
  /** A fully-populated toolbar definition for power users
      @access private */
  var $toolbar_full = array(
          "undo", "redo", "delete", "selectall",
          "|", "cut", "copy", "paste",
          "|", "bold", "italic", "underline", "strikethrough", "removeFormat",
          "|", "superscript", "subscript",
          "|", "indent", "outdent", "justifyCenter", "justifyFull", "justifyLeft", "justifyRight",
          "|", "insertOrderedList", "insertUnorderedList", "inserthtml"
          );

  // .....................................................................
  /** Constructor for new Dojo Editor configuration object  */
  function dojo_editor_config() {
  } // dojo_editor_config
  // .....................................................................
  /**
   * Set a new toolbar definition. Toolbars are an array of buttons. For
   * an example see the default setting for the toolbar in this class. You
   * can EITHER give a full toolbar definition here as an array of
   * toolbar widgets, OR give a preset toolbar ID string. Presets that
   * we have defined so far are:
   *   "basic"  - Just enough to enter text, bold/italic etc.
   *   "medium" - As above but with colour, links, raw HTML etc.
   *   "full"   - Everything available
   * @param mixed $new_toolbat Array of toolbar lines, or preset toolbar id
   */
  function set_toolbar($new_toolbar) {
    if (is_array($new_toolbar)) {
      // Trust them, they know what they are doing
      $this->toolbar = $new_toolbar;
    }
    elseif (is_string($new_toolbar)) {
      // Assign a pre-defined recipe toolbar
      switch ($new_toolbar) {
        case "basic":
          $this->toolbar = $this->toolbar_basic;
          break;
        case "medium":
          $this->toolbar = $this->toolbar_medium;
          break;
        case "full":
          $this->toolbar = $this->toolbar_full;
          break;
        default:
          $this->toolbar = $this->toolbar_full;
      } // switch
    }
  } // set_toolbar
  // .....................................................................
  /**
   * Add a new tool element to the existing toolbar array.
   * @param string $tool The name of the new tool to add to the toolbar
   */
  function add_tool($tool) {
    if (is_string($tool) && $tool != "") {
      $this->toolbar[] = $tool;
    }
  } // add_tool
} // dojo_editor_config class

// ----------------------------------------------------------------------
// PLUGINS

/**
 * The "all" list of extra plugins which can be used by Dojo Editor
 * widget. Used to facilitate the 'all' plugins requirement.
 */
$all_dojo_plugins = array(
   "AlwaysShowToolBar",
   "EnterKeyHandling",
   //"FontChoice",
   "LinkDialog",
   "TextColor",
   );

// The extra Dojo plugins we will install
$dojo_plugins = array();

/**
* Function to set plugins to be loaded. We just keep a global list
* of these which is reference by each widget.
* @param string $plugins A comma-delimited list, or array of plugins
*/
function add_wysiwyg_plugins($plugins) {
  global $dojo_plugins, $all_dojo_plugins;
  if (!is_array($plugins) && is_string($plugins)) {
    if ($plugins == "all") {
      $new_plugins = $all_dojo_plugins;
    }
    else {
      $new_plugins = explode(",", $plugins);
    }
  }
  else {
    $new_plugins = $plugins;
  }
  if (is_array($new_plugins)) {
    $dojo_plugins = $new_plugins;
  }
} // add_wysiwyg_plugins

// ----------------------------------------------------------------------
/**
* Wysiwyg Field class
* A field which renders a textarea form element as a Wysiwyg editor.
* This is based on the package 'Dojo Editor', by Mihai Bazon.
* This class leverages the standard memofield. In fact we render a standard
* memofield, and the main diference is only the setting of a classid, and
* rendering the javascript for Dojo Editor to initialisei itself.
* @package form
*/
class form_wysiwygfield extends form_memofield {
  /** Unique ID of this form element
      @access private */
  var $wysiwyg_id = "wysiwyg";
  /** Plugins we want to register for this field
      @access private */
  var $plugins = array();
  /** CSS Styles we want to display for this field
      @access private */
  var $css_styles;
  // .....................................................................
  /** Constructor - create a wysiwyg form element..
  * @param string  $name     The name of the field
  * @param string  $label    The label which can be displayed alongside the field
  * @param string  $value    The value of the field
  * @param boolean $editable Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css      CSS class or style to apply to the button
  * @param integer $width    Width of the memo field in characters
  * @param integer $rows     Height of the memo field in lines
  */
  function form_wysiwygfield($name="", $label="", $value="", $editable=EDITABLE, $css="", $width=STD_WIDTH, $rows=5) {
    $this->form_memofield($name, $label, $value, $editable, $css, $width, $rows);
    $this->wysiwyg_id = $name;
    $this->setid($this->wysiwyg_id);
    // Empty config object..
    $this->config = new dojo_editor_config();
    $this->set_attribute("dojoType", "dijit.Editor");
  } // form_wysiwygfield
  // .....................................................................
  /**
  * Set the styles to provide for this Wysiwyg editor. Styles are
  * provided as URI's of stylesheet files. These can be local (relative)
  * URI's (eg. '/foo.css') or full URL's. URI's passed either replace
  * (default action) or append to the given list (or array) of stylesheets.
  * Pass the styles list either as a comma-delimited string, or
  * an array.
  * @param mixed $new_styles Comma-delimited list or array of stylesheet URI's
  * @param string $mode If "append" add stylesheets to existing, else replace
  */
  function set_styles($new_styles, $mode="replace") {
    if (!is_array($new_styles)) {
      $new_styles = explode(",", $new_styles);
    }
    if ($mode == "replace") {
      $this->css_styles = $new_styles;
    }
    else {
      foreach ($new_styles as $style) {
        $this->css_styles[] = $style;
      }
    }
  } // set_styles
  // .....................................................................
  /** Set the Dojo Editor page style. This is the overall style setting(s)
  * which are applied to the Dojo Editor widget and so here is where you
  * determine such basic things as the font settings of the content which
  * is displayed/edited etc.
  * @param string $css_style The style setting for the editor 'page'
  */
  function set_pagestyle($css_style) {
    $this->config->pageStyle = $css_style;
  } // set_pagestyle
  // .....................................................................
  /** Set a new toolbar definition. Toolbars are arrays of arrays of lists
  * of Dojo Editor widgets. Each array is a line of these widgets. For an
  * example see the default setting for the toolbar in this class. You
  * can EITHER give a full toolbar definition here as an array of arrays
  * of toolbar widgets, OR give a preset toolbar ID string. Presets that
  * we have defined so far are:
  *   "basic"  - Just enough to enter text, bold/italic etc.
  *   "medium" - As above but with colour, links, raw HTML etc.
  *   "full"   - Everything available
  * @param mixed $new_toolbat Array of toolbar lines, or preset toolbar id
  */
  function set_toolbar($new_toolbar) {
    $this->config->set_toolbar($new_toolbar);
  } // set_toolbar
  // .....................................................................
  /** Set a new list of selectable font families. This array should be an
  * associative one in the form:
  *     "Arial" =>  "arial,helvetica,sans-serif",
  *     "Courier New" => "courier new,courier,monospace",
  *     ...
  * @param array $new_fontnames Array of font family names to use
  */
  function set_fontnames($new_fontnames) {
    if (is_array($new_fontnames)) {
      $this->config->fontnames = $new_fontnames;
    }
  } // set_fontnames
  // .....................................................................
  /** Set a new list of selectable font sizes. The sizes array should be
  * an associative one in the form:
  *     "1 (8 pt)"  => "1",
  *     "2 (10 pt)" => "2",
  *     ...
  * @param array $new_blocksizes Array of font sizes to use
  */
  function set_fontsizes($new_fontsizes) {
    if (is_array($new_fontsizes)) {
      $this->config->fontsizes = $new_fontsizes;
    }
  } // set_fontsizes
  // .....................................................................
  /** Set a new list of selectable block formats. The array should be an
  * associative one in the form:
  *     "Heading 1" => "h1",
  *     "Heading 2" => "h2",
  *     ...
  * @param array $new_blockformats Array of block formats to use
  */
  function set_blockformats($new_blockformats) {
    if (is_array($new_blockformats)) {
      $this->config->blockformats = $new_blockformats;
    }
  } // set_blockformats
  // .....................................................................
  /** Set the Dojo Editor sizing. The default is 'auto' for both width
  * and height, which gives you an editing area filling the are that
  * the original memofield widget would have filled.
  * @param string $width Width of the Dojo Editor editor
  * @param string $height Height of the Dojo Editor editor
  * @param boolean $inctoolbar Whether sizing includes the toolbar
  */
  function set_metrics($width="", $height="", $inctoolbar=true) {
    if ($width != "") {
      $this->config->width = $width;
    }
    if ($height != "") {
      $this->config->height = $height;
    }
    $this->config->sizeIncludesToolbar = ($inctoolbar === true);
  } // set_metrics
  // .....................................................................
  /** Set the regular expression to use for removing dis-allowed HTML
  * tags from the editor content. This allows you to specify a regex
  * matching those tags you don't want to ever appear in the content.
  * @param string $regex Regular expression matching disallowed tags
  */
  function set_remove_regex($regex="") {
    $this->config->htmlRemoveTags = $regex;
  } // set_remove_regex
  // .....................................................................
  /** Set the Dojo Editor undo steps maximum. The default number is 20. Use
  * this method to change that.
  * @param integer $steps Number of undo steps to record.
  */
  function set_undo_steps($steps=20) {
    $this->config->undoSteps = $steps;
  } // set_undo_steps
  // .....................................................................
  /** Set the Dojo Editor statusbar display mode, true or false.
  * @param boolean $mode Whether to display the statusbar or not
  */
  function set_statusbar($mode=true) {
    $this->config->statusBar = $mode;
  } // set_statusbar
  // .....................................................................
  /** Set the Dojo Editor Paste mode. If set to true, then the Dojo Editor
  * widget will intercept ^V and use the Dojo Editor paste command. If
  * false, then it passes ^V through to browser editor widget
  * @param boolean $mode Whether to intercept ^V or not
  */
  function set_paste_intercept($mode=true) {
    $this->config->editorPaste = $mode;
  } // set_paste_intercept
  // .....................................................................
  /**
   * Set the plugins to register for this Wysiwyg editor. For the Dojo
   * Editor we do nothing since plugins are globally defined.
   * @param string $pluginlist Comma-delimited list of plugins to register
   */
  function register_plugins($pluginlist="all") {
    return true;
  } // register_plugins
  // .....................................................................
  /**
   * Register with Dojo Editor for this editor to work. We produce the required
   * javascript. This is an internal method usually executed just prior to
   * rendering.
   * @access private
   */
  function register() {
    global $RESPONSE;
    $RESPONSE->head->add_named_script(
        "dojo.require('dijit.Editor');\n"
      . "function save_content_$this->name() {\n"
      . " dojo.byId('$this->name').value=dijit.byId('_$this->name').getValue();\n"
      . "}\n"
        ,
        "dojo",
        "javascript",
        "append",
        ADD_ONCE_ONLY
        );
  } // register
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The wysiwyg Dojo Editor editor field as HTML.
  */
  function html($name="") {
    global $RESPONSE, $dojo_plugins;
    
    $s = "";
    
    if ($name != "") {
      $this->name = $name;
      $this->wysiwyg_id = $this->name;
      $this->setid($this->wysiwyg_id);
    }
    
    // Add various the javascript bits
    $this->register();

    // Deal with the extra plugins, if any defined. Has to be done
    // before processing the toolbar below, as it affects it.
    if (count($dojo_plugins) > 0) {
      $extraplugins = array();
      foreach ($dojo_plugins as $dojo_plugin) {
        $RESPONSE->head->add_named_script(
            "dojo.require('dijit._editor.plugins.$dojo_plugin');\n",
            "dojo",
            "javascript",
            "append",
            ADD_ONCE_ONLY
            );
        switch ($dojo_plugin) {
          case "LinkDialog":
            $this->config->add_tool("{name:'dijit._editor.plugins.LinkDialog'}");
            break;
          case "TextColor":
            $this->config->add_tool("{name:'dijit._editor.plugins.TextColor'}");
            break;
        } // switch
      }
      $this->setheight("");
    }
    
    // Set the toolbar buttons
    if (is_array($this->config->toolbar)) {
      $plugins = array();
      foreach ($this->config->toolbar as $tool) {
        if (!strstr($tool, "'")) {
          $tool = "'$tool'";
        }
        $plugintools[] = $tool;
      }
      $this->set_attribute("plugins", "[" . implode(",", $plugintools) . "]");
    }
    
    // Set the stylesheets
    if (is_array($this->css_styles)) {
      $this->set_attribute("styleSheets", implode(";", $this->css_styles));
    }

    // Hidden field for saving changed content - takes the name
    // intended for this widget
    $hid = new form_hiddenfield($this->name, $this->value);
    
    // This populates our hidden field for POSTs
    $this->set_onchange("save_content_$this->name()");

    // Now we switch the name & id of this to $name prefixed by "_"
    $name = $this->name;
    $this->setname("_$name");
    $this->setid("_$name");

    // Render it all
    $s .= form_memofield::html() . $hid->html();
    
    return $s;
  } // html
} // class form_wysiwygfield

// ----------------------------------------------------------------------
?>