<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    dojo-widget-defs.php                                    */
/* Author:      Paul Waite                                              */
/* Description: Definitions for all Dojo's interface bits n pieces      */
/*              apart from form elements (in form-defs.php).            */
/*                                                                      */
/* ******************************************************************** */
/** @package form */

/** Make sure we have access to form elements too */
include_once("form-defs.php");

// ----------------------------------------------------------------------
/**
* Thumbnail image picker class.
* @package form
*/
class dojo_itemfilereadstore extends HTMLObject {
  /** URL to acquire JSON image data */
  var $data_url = "";
  // ....................................................................
  /**
  * Constructor
  * Create the object. Sets basic field attributes.
  * @param string  $name     The name of the field
  * @param string  $label    The label which can be displayed alongside the field
  * @param string  $value    The value of the field
  * @param boolean $editable Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css      CSS class or style to apply to the button
  * @param integer $width    Width of element in characters
  */
  function dojo_itemfilereadstore($name="") {
    $this->setname($name);
  } // dojo_itemfilereadstore
  // ....................................................................
  /**
   * Set the URL to acquire the data in JSON format.
   * @param string $url The URL to fetch data from
   */
  function set_data_url($url) {
    $this->data_url = $url;
  } // set_data_url
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html($name="") {
    global $RESPONSE;
    $RESPONSE->head->add_named_script(
        "dojo.require('dojo.data.ItemFileReadStore');\n",
        "dojo",
        "javascript",
        "append",
        ADD_ONCE_ONLY
        );
    $this->set_attribute("jsId", $this->name);
    $this->set_attribute("dojoType", "dojo.data.ItemFileReadStore");
    $this->set_attribute("url", $this->data_url); 
    $this->setid("");
    $this->setname("");
    $html  = "<div";
    $html .= $this->attributes();
    $html .= ">";
    $html .= "</div>\n";
    
    return $html;
  } // html
} // dojo_itemfilereadstore class

// ----------------------------------------------------------------------
/**
* Thumbnail image picker class.
* @package form
*/
class dojo_thumbpicker extends HTMLObject {
  /** The way the picker is displayed 'horizontal' or 'vertical' */
  var $orientation = "horizontal";
  /** Number of thumbnails to display */
  var $num_thumbs = 8;
  /** Whether images can have associated hyperlinks */
  var $hyperlinks_enabled = false;
  /** Optional hyperlink target if enabled */
  var $hyperlink_target = "";
  /** Whether images are clickable */
  var $is_clickable = false;
  /** Whether to show widget only on demand */
  var $show_on_demand = false;
  /** URL to acquire JSON image data */
  var $data_url = "";
  /** Query for thumbnail data request */
  var $data_query = "";
  /** Number of thumbnails to fetch each time */
  var $fetch_count = 5;
  /** Script to subscribe to picker events */
  var $subscribe_script = "";
  // ....................................................................
  /**
  * Constructor
  * Create the object. Sets basic field attributes.
  * @param string  $id   The ID of the widget
  * @param integer $size Width or height in pixels (dep. on orientation).
  * @param string  $css  CSS class or style to apply to the button
  */
  function dojo_thumbpicker($id, $size=300, $css="") {
    $this->setid($id);
    $this->setsize($size);
    $this->setcss($css);
    global $RESPONSE, $LIBDIR;
    // webpage-defs.php adds this too, but a bit late for us
    // so we get in here first
    $RESPONSE->head->add_scriptsrc(
            "$LIBDIR/js/dojo/dojo/dojo.js",
            "javascript",
            "djConfig=\"parseOnLoad:true\""
            );
    $RESPONSE->head->add_script("dojo.require('dojo.parser');\n");
  
    $RESPONSE->head->add_named_script(
        "dojo.require('dojox.image.ThumbnailPicker');\n"
      . "dojo.require('dojo.data.ItemFileReadStore');\n"
      ,
        "dojo",
        "javascript",
        "append",
        ADD_ONCE_ONLY
        );
    $RESPONSE->head->add_scriptsrc(
        "$LIBDIR/js/dojo/dojox/image/ThumbnailPicker.js",
        "javascript"
        );
    $this->set_attribute("dojoType", "dojox.image.ThumbnailPicker");
  } // dojo_thumbpicker
  // ....................................................................
  /**
   * Set the URL to acquire the data in JSON format.
   * @param string $url The URL to fetch data from
   */
  function set_data_url($url) {
    $this->data_url = $url;
  } // set_data_url
  // ....................................................................
  /**
   * Set the data store query string to acquire the data in JSON format.
   * @param string $query The query to filter fetched data
   */
  function set_data_query($query) {
    $this->data_query = $query;
  } // set_data_query
  // ....................................................................
  /**
   * Set the number of thumbnails to be fetch each time from server.
   * @param integer $fetch_count The number of thumbnails to fetch
   */
  function set_fetch_count($fetch_count) {
    $this->fetch_count = $fetch_count;
  } // set_fetch_count
  // ....................................................................
  /**
   * Set the thumbnail picker to be clickable. Thumbnails may be
   * clicked to 'do something'.
   * @param boolean $mode True (default) if widget is clickable
   */
  function set_clickable($mode=true) {
    $this->is_clickable = ($mode !== false);
  } // set_clickable
  // ....................................................................
  /**
   * Enable image thumbnails to be hyperlinked to a target. This
   * allows thumbnails to have hyperlinks off to new browser
   * windows etc.
   * @param boolean $mode True if hyperlinks are to be enabled
   * @param string $target Optional hyperlink target (eg. '_new')
   */
  function enable_hyperlinks($mode=true, $target="") {
    $this->hyperlinks_enabled = ($mode !== false);
    $this->hyperlink_target = $target;
  } // enable_hyperlinks
  // ....................................................................
  /**
   * Set up show-on-demand. In this implementation, what this means
   * if this is enabled, is that the user will see a "show" button in
   * place of the thumbpicker. When they click that, then they get
   * the thumbpicker displayed.
   * @param boolean $mode True if show-on-demand is to be enabled
   */
  function set_show_on_demand($mode=true) {
    $this->show_on_demand = ($mode !== false);
  } // set_show_on_demand
  // ....................................................................
  /**
   * Set up a subscription to external javascript function.
   * @param string $script Script to subscribe to thumbnail picker events
   */
  function subscribe_script($script) {
    $this->subscribe_script = $script;
  } // subscribe_script
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html($name="") {
    global $RESPONSE;

    $html = "";
    $this->set_attribute("numberThumbs", $this->num_thumbs);
    $this->set_attribute("isHorizontal", ($this->orientation == "horizontal" ? "true" : "false")); 
    $this->set_attribute("isClickable", ($this->is_clickable ? "true" : "false")); 

    if ($this->hyperlinks_enabled) {
      $this->set_attribute("useHyperlink", "true");
      if ($this->hyperlink_target == "") {
        $this->set_attribute("hyperlinkTarget", "this");
      }
      else {
        $this->set_attribute("hyperlinkTarget", $this->hyperlink_target);
      }
    }
    $html .= "<p>";
    $html .= "<div";
    $html .= $this->attributes();
    $html .= ">No images are available.";
    $html .= "</div>";
    $html .= "</p>";
    
    $datastorename = $this->id . "_datastore";
    $datastore = new dojo_itemfilereadstore($datastorename);
    $datastore->set_data_url($this->data_url);
    $html .= $datastore->html();
  
    // Add datastore javascript
    $RESPONSE->head->add_named_script(
          "function init_" . $datastorename . "(){\n"
        . " var " . $this->id . "_itemRequest = {\n"
        . "   query: {" . $this->data_query . "},\n"
        . "   count: $this->fetch_count, start: 0\n"
        . " };\n"
        . " var " . $this->id . "_itemNameMap = {\n"
        . "   imageThumbAttr: \"thumb\",\n"
        . "   imageLargeAttr: \"large\",\n"
        . "   titleAttr: \"title\",\n"
        . "   linkAttr: \"link\"\n"
        . " };\n"
        . " dijit.byId('$this->id').setDataStore($datastorename," . $this->id . "_itemRequest," . $this->id . "_itemNameMap);\n"
        . "}\n"
        ,
          "dojo",
          "javascript",
          "append",
          ADD_ONCE_ONLY
          );    
    $RESPONSE->head->add_named_script(
          "dojo.addOnLoad(init_" . $datastorename . ");\n",
          "dojo_onloads",
          "javascript",
          "append",
          ADD_ONCE_ONLY
          );
    if ($this->subscribe_script != "") {
      $RESPONSE->head->add_named_script(
            "function doSubscribe(){\n"
          . " dojo.subscribe(dijit.byId('$this->id').getClickTopicName(), $this->subscribe_script);\n"
          . "}\n"
          ,
            "dojo",
            "javascript",
            "append",
            ADD_ONCE_ONLY
            );
      $RESPONSE->head->add_named_script(
            "dojo.addOnLoad(doSubscribe);\n",
            "dojo_onloads",
            "javascript",
            "append",
            ADD_ONCE_ONLY
            );
    }
    return $html;
  } // html
} // dojo_thumbpicker class

// ----------------------------------------------------------------------
/**
* Tab Container class.
* @package form
*/
class dojo_contentpane extends HTMLObject {
  var $content = "";
  // ....................................................................
  function dojo_contentpane($id, $title) {
    $this->setid($id);
    $this->settitle($title);
    global $RESPONSE;
    $RESPONSE->head->add_named_script(
        "dojo.require('dijit.layout.ContentPane');\n",
        "dojo",
        "javascript",
        "append",
        ADD_ONCE_ONLY
        );
    $this->set_attribute("dojoType", "dijit.layout.ContentPane");
  } // dojo_contentpane
  // ....................................................................
  function add_content($content) {
    $this->content .= $content;
  } // add_content
  // ....................................................................
  function html() {
    $html = " <div"
          . $this->attributes()
          . ">"
          . $this->content
          . "</div>\n"
          ;
    return $html;
  } // html
} // class dojo_contentpane
// ----------------------------------------------------------------------
/**
* Tab Container class.
* @package form
*/
class dojo_tabcontainer extends HTMLObject {
  var $contentpanes = array();
  // ....................................................................
  /**
  * Constructor
  * Create the object. Sets basic field attributes.
  * @param string  $name     The name of the field
  * @param string  $label    The label which can be displayed alongside the field
  * @param string  $value    The value of the field
  * @param boolean $editable Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css      CSS class or style to apply to the button
  * @param integer $width    Width of element in characters
  */
  function dojo_tabcontainer($name="", $width="", $height="", $css="") {
    $this->setname($name);
    $this->setwidth($width);
    $this->setheight($height);
    $this->setcss($css);
    global $RESPONSE;
    $RESPONSE->head->add_named_script(
        "dojo.require('dijit.layout.TabContainer');\n",
        "dojo",
        "javascript",
        "append",
        ADD_ONCE_ONLY
        );
    $RESPONSE->head->add_named_script(
        "dojo.require('dijit.form.Button');\n",
        "dojo",
        "javascript",
        "append",
        ADD_ONCE_ONLY
        );
    $this->set_attribute("dojoType", "dijit.layout.TabContainer");
  } // dojo_tabcontainer
  // ....................................................................
  /**
  * Add tab content by id to our local array of tabs objects. It is assumed
  * to be rendered with render(). If not it is treated as literal content
  * and used as-is.
  * @param string $id The unique ID of the tab
  * @param object $tab The tab object to add
  */
  function add_contentpane($pane) {
    $this->contentpanes[] = $pane;
  } // add_contentpane
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html($name="") {
    // Set name and check ID
    if ($name != "") {
      $this->name = $name;
    }
    if ($this->id == "") {
      $this->id = $this->name;
    }
    
    $html  = "<div";
    $html .= $this->attributes();
    $html .= ">";
    foreach ($this->contentpanes as $pane) {
      if (is_object($pane)) {
        $html .= $pane->render();
      }
    } // foreach
    $html .= "</div>\n";
    
    return $html;
  } // html
} // dojo_tabcontainer class

// ----------------------------------------------------------------------
/**
* Accordion Container class.
* @package form
*/
class dojo_accordioncontainer extends dojo_tabcontainer {
  // ....................................................................
  /**
  * Constructor
  * Create the object. Sets basic field attributes.
  * @param string  $name     The name of the field
  * @param string  $label    The label which can be displayed alongside the field
  * @param string  $value    The value of the field
  * @param boolean $editable Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css      CSS class or style to apply to the button
  * @param integer $width    Width of element in characters
  */
  function dojo_accordioncontainer($name="", $width="", $height="", $css="") {
    $this->dojo_tabcontainer($name, $width, $height, $css);
    global $RESPONSE;
    switch ($RESPONSE->ui_toolkit) {
      case "dojo":
        $RESPONSE->head->add_named_script(
            "dojo.require('dijit.layout.AccordionContainer');\n",
            "dojo",
            "javascript",
            "append",
            ADD_ONCE_ONLY
            );
        $this->set_attribute("dojoType", "dijit.layout.AccordionContainer");
        break;
    } // switch
  } // dojo_accordioncontainer
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html($name="") {
    // Set name and check ID
    if ($name != "") {
      $this->name = $name;
    }
    if ($this->id == "") {
      $this->id = $name;
    }
    
    $html  = "<div";
    $html .= $this->attributes();
    $html .= ">";
    foreach ($this->tabs as $identity => $tab) {
      $bits = explode("|", $identity);
      $tabtitle = $bits[0];
      $tabid = $bits[1];
      if (is_object($tab)) {
        $tabcontent = $tab->render();
      }
      else {
        $tabcontent = $tab;
      }
      $html .= " <div id=\"$tabid\" title=\"$tabtitle\"";
      $html .= " dojoType=\"dijit.layout.AccordionPane\"";
      $html .= ">";
      $html .= $tabcontent;
      $html .= "</div>\n";
    } // foreach
    
    $html .= "</div>\n";
    
    return $html;
  } // html
} // dojo_accordioncontainer class

// ----------------------------------------------------------------------
/** Class to contain contraints
 */
class constraints {
  var $constraints = array();
  // ....................................................................
  /** Constructor */
  function constraints() { }
  // ....................................................................
  /**
   * Add a constraint, which has a name and value.
   * @param string $name Name of this constraint
   * @param string $value Value of this constraint.
   */
  function add_constraint($name, $value) {
    if ($name != "") {
      $this->constraints[$name] = $value;
    }
  }
  // ....................................................................
  /**
   * Return the constraints attribute value string.
   * @return string Constraints attributes string.
   */
  function attributes() {
    $s = "";
    $attrs = array();
    foreach ($this->constraints as $name => $value) {
      $attrs[] = "$name: '$value'";
    }
    if (count($attrs) > 0) {
      $s = "{" . implode(",", $attrs) . "}";
    }
    return $s;
  }
} // constraints class
// ----------------------------------------------------------------------
/**
 * A text field where the content has to pass validation constraints.
 * @package form
 */
class form_validationbox extends form_textfield {
  /** Constraints object - containts multiple constraints */
  var $constraints;
  /** Regular expression to validate with */
  var $regexp;
  /** Maximum value this field can contain */
  var $maximum;
  /** Minimum value this field can contain */
  var $minimum;
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name       The name of the field
  * @param string  $label      The label which can be displayed alongside the field
  * @param string  $value      The value of the field
  * @param boolean $editable   Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css        CSS class or style to apply to the button
  * @param integer $width      Width of element in characters
  */
  function form_validationbox($name="", $label="", $value="", $editable=EDITABLE, $css="", $width=STD_WIDTH) {
    $this->form_textfield($name, $label, $value, $editable, $css, $width);
    $this->constraints = new constraints();
    global $RESPONSE;
    switch ($RESPONSE->ui_toolkit) {
      case "dojo":
        $RESPONSE->head->add_named_script(
            "dojo.require('dijit.form.ValidationTextBox');\n",
            "dojo",
            "javascript",
            "append",
            ADD_ONCE_ONLY
            );
        $this->set_attribute("dojoType", "dijit.form.ValidationTextBox");
        break;
    } // switch
  } // form_validationbox
  // ....................................................................
  /**
   * Set minimum and maximum integer values as limits. Both must be
   * given here.
   * @param integer $min Minimum value for this field
   * @param integer $max Maximum value for this field
   */
  function set_minmax($min, $max) {
    if ($max < $min) {
      $max = $min + 1;
    }
    $this->maximum = $max;
    $this->minimum = $min;
    $this->add_constraint("min", $min);
    $this->add_constraint("max", $max);
  } // set_maxmin
  // ....................................................................
  /**
   * Add a new constraint to this validation box.
   * @param string $name Name of this constraint
   * @param string $value Value of this constraint.
   */
  function add_constraint($name, $value) {
    $this->constraints->add_constraint($name, $value);
  } // add_constraint
  // ....................................................................
  /**
   * Return true if this has one or more constraints.
   */
  function has_constraints() {
    return (
      isset($this->constraints) &&
      count($this->constraints->constraints) > 0
      );
  } // has_constraints
  // ....................................................................
  /**
   * Set a regular expression that must test true against the field
   * value to validate.
   * @param string $regexp The regular expression (Javascript compat)
   */
  function set_regexp($regexp) {
    $this->regexp = $regexp;
  } // set_regexp
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html($name="") {
    if ($this->id == "") {
      $this->id = $name;
    }
    global $RESPONSE;
    switch ($RESPONSE->ui_toolkit) {
      case "dojo":
        if ($this->has_constraints()) {
          $this->set_attribute("constraints", $this->constraints->attributes());
        }
        if ($this->regexp) {
          $this->set_attribute("regExp", "$this->regexp");
        }        
        break;
    } // switch
    return form_textfield::html($name);
  } // html
} // class form_validationbox

// ----------------------------------------------------------------------
/**
* This is a hybrid field. It is basically a text field where the
* value is expected to be a date. It is rendered with a javascript date
* picker widget which appears when the user clicks on the field.
* from it will populate this field.
* @package form
*/
class form_datefield extends form_validationbox {
  var $dateformat = "";
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name       The name of the field
  * @param string  $label      The label which can be displayed alongside the field
  * @param string  $value      The value of the field
  * @param boolean $editable   Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css        CSS class or style to apply to the button
  * @param integer $width      Width of element in characters
  */
  function form_datefield($name="", $label="", $value="", $editable=EDITABLE, $css="", $width=STD_WIDTH) {
    $this->form_validationbox($name, $label, $value, $editable, $css, $width);
    global $RESPONSE, $LIBDIR;
    switch ($RESPONSE->ui_toolkit) {
      case "dojo":
        $RESPONSE->head->add_named_script(
            "dojo.require('dijit.form.DateTextBox');\n",
            "dojo",
            "javascript",
            "append",
            ADD_ONCE_ONLY
            );
        $this->set_attribute("dojoType", "dijit.form.DateTextBox");
        break;
      default:
        $RESPONSE->add_scriptsrc("$LIBDIR/js/scw.js");
    } // switch
    
  } // form_datefield
  // ....................................................................
  function set_format($format) {
    $this->add_constraint("datePattern", $format);
    $this->set_attribute("promptMessage", $format);
  } // set_format
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html($name="") {
    if ($name != "") {
      $this->setname($name);
    }
    if ($this->id == "") {
      $this->id = $this->name;
    }
    global $RESPONSE;
    switch ($RESPONSE->ui_toolkit) {
      case "axyl":
        $this->set_onclick("scwShow(this,event)");
        break;
    } // switch
    return form_validationbox::html($name);
  } // html
} // form_datefield class

// ----------------------------------------------------------------------
/**
* This is a hybrid field. It is basically a text field where the
* value is expected to be a time.
* @package form
*/
class form_timefield extends form_validationbox {
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name       The name of the field
  * @param string  $label      The label which can be displayed alongside the field
  * @param string  $value      The value of the field
  * @param boolean $editable   Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css        CSS class or style to apply to the button
  * @param integer $width      Width of element in characters
  */
  function form_timefield($name="", $label="", $value="", $editable=EDITABLE, $css="", $width=STD_WIDTH) {
    $this->form_validationbox($name, $label, $value, $editable, $css, $width);
    global $RESPONSE;
    switch ($RESPONSE->ui_toolkit) {
      case "dojo":
        $RESPONSE->head->add_named_script(
            "dojo.require('dijit.form.TimeTextBox');\n",
            "dojo",
            "javascript",
            "append",
            ADD_ONCE_ONLY
            );
        $this->set_attribute("dojoType", "dijit.form.TimeTextBox");
        break;
    } // switch
  } // form_timefield
  // ....................................................................
  function set_format($format) {
    $this->add_constraint("timePattern", $format);
    $this->set_attribute("promptMessage", $format);
  } // set_format
} // form_timefield class

// ----------------------------------------------------------------------
/**
 * A number spinner field. User can spin up or down between limits.
 * @package form
 */
class form_numberspinner extends form_validationbox {
  /** Increment/decrement with each spinner click */
  var $discrete_steps = 10;
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name       The name of the field
  * @param string  $label      The label which can be displayed alongside the field
  * @param string  $value      The value of the field
  * @param boolean $editable   Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css        CSS class or style to apply to the button
  * @param integer $width      Width of element in characters
  */
  function form_numberspinner($name="", $label="", $value="", $editable=EDITABLE, $css="", $width=STD_WIDTH) {
    $this->form_validationbox($name, $label, $value, $editable, $css, $width);
    global $RESPONSE;
    switch ($RESPONSE->ui_toolkit) {
      case "dojo":
        $RESPONSE->head->add_named_script(
            "dojo.require('dijit.form.NumberSpinner');\n",
            "dojo",
            "javascript",
            "append",
            ADD_ONCE_ONLY
            );
        $this->set_attribute("dojoType", "dijit.form.NumberSpinner");
        break;
    } // switch
  } // form_numberspinner
  // ....................................................................
  /**
   * Set the number of discreate steps that the spinner can make within its
   * range. eg. if the range is 0 - 100, and this is set to 5, then there
   * will be 5 'stops' of 20.
   * @param string $steps The number of discrete steps the spinner can make
   */
  function set_discrete_steps($steps=10) {
    $this->discrete_steps = $steps;
  } // set_discrete_steps
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html($name="") {
    global $RESPONSE;
    switch ($RESPONSE->ui_toolkit) {
      case "dojo":
        $this->set_attribute("smallDelta", ceil( ($this->maximum - $this->minimum) / $this->discrete_steps ) );
        break;
    } // switch
    return form_validationbox::html($name);
  } // html
} // form_numberspinner class

// ----------------------------------------------------------------------
/**
 * A number spinner field. User can spin up or down between limits.
 * @package form
 */
class form_numberslider extends form_numberspinner {
  /** The way the slider is displayed 'horizontal' or 'vertical' */
  var $orientation = "horizontal";
  /** Whether intermediate changes are displayed */
  var $intermediate = false;
  /** Whether buttons are displayed */
  var $show_buttons = false;
  /** Whether clicking progress bar changes the slider or not */
  var $click_select = true;
  // ....................................................................
  /**
  * Constructor
  * Create a field object. Sets basic field attributes.
  * @param string  $name       The name of the field
  * @param string  $label      The label which can be displayed alongside the field
  * @param string  $value      The value of the field
  * @param boolean $editable   Editability: EDITABLE or DISPLAY_ONLY (true or false)
  * @param string  $css        CSS class or style to apply to the button
  * @param integer $width      Width of element in characters
  */
  function form_numberslider($name="", $label="", $value="", $editable=EDITABLE, $css="", $width=STD_WIDTH) {
    $this->form_numberspinner($name, $label, $value, $editable, $css, $width);
    global $RESPONSE;
    $RESPONSE->head->add_named_script(
        "dojo.require('dijit.form.Slider');\n",
        "dojo",
        "javascript",
        "append",
        ADD_ONCE_ONLY
        );
  } // form_numberslider
  // ....................................................................
  /**
   * Set the orientation of the slider as it is displayed on the page. 
   * @param string $ori Orientation of the slider: 'horizontal' or 'vertical'
   */
  function set_orientation($ori="horizontal") {
    $this->orientation = $ori;
  } // set_orientation
  // ....................................................................
  /**
   * Set whether to show intermediate slider changes or not. If true
   * this will fire onChange events very fast as it moves. If false
   * they fire only onMouseUp.
   * @param boolean $mode True if you want to see intermediate changes
   */
  function show_intermediate_changes($mode=true) {
    $this->intermediate = $mode;
  } // show_intermediate_changes
  // ....................................................................
  /**
   * Set whether to show the slider buttons or not. 
   * @param boolean $mode True if you want to see buttons
   */
  function set_show_buttons($mode=true) {
    $this->show_buttons = $mode;
  } // set_show_buttons
  // ....................................................................
  /**
   * Set whether clicking the progress bar will change the slider value. 
   * @param boolean $mode True if you want to click to change slider
   */
  function set_click_select($mode=true) {
    $this->click_select = $mode;
  } // set_click_select
  // ....................................................................
  /**
  * This renders the field as HTML.
  * @return string The field as HTML.
  */
  function html($name="") {
    switch ($this->orientation) {
      case "vertical":
        $this->set_attribute("dojoType", "dijit.form.VerticalSlider");
        break;
      default:
        $this->set_attribute("dojoType", "dijit.form.HorizontalSlider");
        break;
    } // switch

    // Add the other attributes
    if ($this->minimum != "") {
      $this->set_attribute("minimum", $this->minimum);
    }
    if ($this->maximum != "") {
      $this->set_attribute("maximum", $this->maximum);
    }
    $this->set_attribute("discreteValues", $this->discrete_steps);
    $this->set_attribute("intermediateChanges", $this->intermediate ? "true" : "false");
    $this->set_attribute("showButtons", $this->show_buttons ? "true" : "false");
    $this->set_attribute("clickSelect", $this->click_select ? "true" : "false");

    return form_textfield::html($name);
  } // html
} // form_numberslider class

// ----------------------------------------------------------------------
?>