#!/usr/bin/php -q
<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    pg-microsites-installer.php                             */
/* Author:      Paul Waite                                              */
/* Description: Scans Axyl all Axyl installations, and installs the     */
/*              symlinks required to install microsites, if any have    */
/*              been requested.                                         */
/*              NOTE: this only works for Postgresql installations.     */
/*                                                                      */
/* ******************************************************************** */
/* Script setup */
$PROGNAME      = "pg-microsites-installer";
$PROGVER       = "1.0";
$DEBUGMODE     = false;    // Verbose logging option
$ECHO_LOGGING  = false;    // echo log messages to console too
$POSTGRES_REQD = true;     // Require Postgres DB installed
$ROOT_REQD     = false;    // Require 'root' user
$USER_INPUT    = false;    // Interactive - user will be entering responses
$AXYLROOT_REQD = false;    // Script must be run in an Axyl website root


// ----------------------------------------------------------------------
// READ IN THE AXYL CONFIG - required
// ----------------------------------------------------------------------
$CONF_FILE = "/etc/axyl/axyl.conf";
$conf_content = readallfile($CONF_FILE);
if ($conf_content != "") {
  if (preg_match("/AXUSER=([\S]+)\s/", $conf_content, $matches)) {
    $AXUSER = $matches[1];
  }
  if (preg_match("/AXYL_HOME=([\S]+)\s/", $conf_content, $matches)) {
    $AXYL_HOME = $matches[1];
  }
  if (preg_match("/AXYL_DATA=([\S]+)\s/", $conf_content, $matches)) {
    $AXYL_DATA = $matches[1];
  }
  if (preg_match("/AXYL_LOGS=([\S]+)\s/", $conf_content, $matches)) {
    $AXYL_LOGS = $matches[1];
  }
}
else {
  die(
    "File $CONF_FILE was invalid. Have you installed Axyl?\n"
  . "If you have, then have you remembered to set it up properly, by\n"
  . "runing the script AXYL_HOME/install/setup-axyl.sh?\n"
  );
}
if ($AXYL_HOME != "") {
  if (!is_dir($AXYL_HOME)) {
    die("AXYL_HOME is undefined. Check your $CONF_FILE.");
  }
}
else {
  die(
    "The Axyl configuration file at $CONF_FILE does not define your\n"
  . "AXYL_HOME. This script cannot proceed without that setting.\n"
  . "Please check your Axyl configuration.\n"
  );
}

// ----------------------------------------------------------------------
// INCLUDE COMMON PRE-AMBLE & FUNCTIONS
// ----------------------------------------------------------------------
include_once("$AXYL_HOME/scripts/inc/common.inc");

// This function is required before any others are read in. It is
// used to read the Axyl configuration at the start.
function readallfile($path) {
  $content = "";
  if (file_exists($path)) {
    $fp = fopen($path, "r");
    if ($fp !== false) {
      $content = fread($fp, filesize($path));
      fclose($fp);
    }
  }
  return $content;
}

// ----------------------------------------------------------------------
// Globals..
$CMDIR = "";
$TEMPLATESDIR = "";
function get_application_paths($site_docroot) {
  global $CMDIR, $TEMPLATESDIR;
  $got= false;
  // Default them..
  $CMDIR = "/var/cm";
  $TEMPLATESDIR = "/templates";
  $APP_XML = "$site_docroot/application.xml";
  if (file_exists($APP_XML)) {
    $conf_xml = readallfile($APP_XML);
    if ($conf_xml != "") {
      if (preg_match("/<global name=\"CMDIR\">([\S]+)<\/global>/", $conf_xml, $matches)) {
        $CMDIR = $matches[1];
      }
      if ($CMDIR != "" && substr($CMDIR, 0, 1) != "/") {
        $CMDIR = "/$CMDIR";
      }
      if (preg_match("/<global name=\"TEMPLATESDIR\">([\S]+)<\/global>/", $conf_xml, $matches)) {
        $TEMPLATESDIR = $matches[1];
      }
      if ($TEMPLATESDIR != "" && substr($TEMPLATESDIR, 0, 1) != "/") {
        $TEMPLATESDIR = "/$TEMPLATESDIR";
      }
      $got = true;
    }
  }
  if ($got) {
    $CMDIR = $site_docroot . $CMDIR;
    $TEMPLATESDIR = $site_docroot . $TEMPLATESDIR;
    debug("get_application_paths: CM=$CMDIR TEMPLATES=$TEMPLATESDIR");
  }
  else {
    debug("get_application_paths: failed.");
  }
  return $got;
}

// ----------------------------------------------------------------------
// PROCESS ALL DATABASES
// Get websites..
$websites = get_axyl_websites();
if (count($websites) > 0) {
  foreach ($websites as $app_name => $website_info) {
    $dbtype       = $website_info[0];
    $dbname       = $website_info[1];
    $dbuser       = $website_info[2];
    $dbpass       = $website_info[3];
    $dbhost       = $website_info[4];
    $dbport       = $website_info[5];
    $site_docroot = $website_info[6];
    $vhost        = $website_info[7];
    $vservername  = $website_info[8];
    $admin_email  = $website_info[9];

    debug("+++");
    debug("processing application: $app_name");
    if ($dbtype == 1 || $dbtype == "postgres") {
      // Check it is an Axyl database, and of the correct version. This
      // means any with a table called 'ax_microsite'..
      $dbid = connect_pgdb(
                $dbname,
                $dbuser,
                $dbpass != "none"   ? $dbpass : "",
                $dbhost != "direct" ? $dbhost : "",
                $dbhost != "direct" ? $dbport : ""
                );
      if ($dbid !== false) {
        $chk = query_pgdb($dbid, "SELECT * FROM pg_class WHERE relname='ax_microsite'");
        if ($chk !== false) {
          $totrows = numrows_pgdb($chk);
          if ($totrows == 1) {
            debug("microsites extension detected.");
            if ($site_docroot != "" && is_dir($site_docroot)) {
              debug("site_docroot = $site_docroot");
              $got_app_dirs = false;
              // Sites requesting publish..
              $q  = "SELECT * FROM ax_microsite";
              $q .= " WHERE req_microsite_publish=TRUE";
              $pub = query_pgdb($dbid, $q);
              if ($pub !== false) {
                $totrows = numrows_pgdb($pub);
                if ($totrows > 0) {
                  // Get the CM and Templates directories..
                  $got_app_dirs = get_application_paths($site_docroot);
                  if ($got_app_dirs) {
                    for ($rowno = 0; $rowno < $totrows; $rowno++) {
                      $rowdata = fetch_array_pgdb($pub, $rowno);
                      $microsite_name = $rowdata["microsite_name"];
                      if ($microsite_name != "") {
                        debug("microsite publish request: $microsite_name");
                        // Create our symlinks, but first unlink any existing ones..
                        if (is_link("$TEMPLATESDIR/$microsite_name")) {
                          unlink("$TEMPLATESDIR/$microsite_name");
                        }
                        if (is_link("$CMDIR/$microsite_name")) {
                          unlink("$CMDIR/$microsite_name");
                        }
                        symlink( "$CMDIR/microsites/themes/$microsite_name", "$TEMPLATESDIR/$microsite_name" );
                        symlink( "$CMDIR/microsites/pages/$microsite_name",  "$CMDIR/$microsite_name" );
                        debug("microsite symlinks created.");

                        // Install an apache virtual host, if we can..
                        $microsite_domain = $rowdata["microsite_domain"];
                        $cmd = "$AXYL_HOME/install/add-apache-serveralias.sh";
                        if (is_executable($cmd)) {
                          $args = "$vservername $microsite_domain silent";
                          $ret = shell_exec("$cmd $args");
                          if ($ret != "") logit($ret);
                          debug("installed Apache ServerAlias for $microsite_domain");
                          $cmd = "$AXYL_HOME/install/prod-apache.sh";
                          if (is_executable($cmd)) {
                            $ret = shell_exec("$cmd reload");
                            if ($ret != "") logit($ret);
                            debug("apache was reloaded.");
                          }
                        }
                        else {
                          logit("ERROR: create of Apache ServerAlias for $microsite_domain failed.");
                          logit("'$cmd' is not executable.");
                        }

                        // Now update the database record..
                        $q  = "UPDATE ax_microsite SET";
                        $q .= " req_microsite_publish=FALSE,";
                        $q .= " req_microsite_remove=FALSE,";
                        $q .= " currently_installed=TRUE,";
                        $q .= " last_installed='" . timestamp_to_datetime() . "'";
                        $q .= " WHERE microsite_name='" . escape_string($microsite_name) . "'";
                        $pubup = query_pgdb($dbid, $q);
                        if ($pubup !== false) {
                          logit("installed microsite $microsite_name");
                        }
                      }
                    } // for
                  }
                } // got rows
                else {
                  debug("no microsite publish requests.");
                }
              }

              // Sites requesting removal..
              $q  = "SELECT * FROM ax_microsite";
              $q .= " WHERE req_microsite_remove=TRUE";
              $rem = query_pgdb($dbid, $q);
              if ($rem !== false) {
                $totrows = numrows_pgdb($rem);
                if ($totrows > 0) {
                  // Get the CM and Templates directories..
                  if (!$got_app_dirs) {
                    $got_app_dirs = get_application_paths($site_docroot);
                  }
                  if ($got_app_dirs) {
                    for ($rowno = 0; $rowno < $totrows; $rowno++) {
                      $rowdata = fetch_array_pgdb($rem, $rowno);
                      $microsite_name = $rowdata["microsite_name"];
                      if ($microsite_name != "") {
                        debug("microsite removal request: $microsite_name");
                        // Remove the symlinks, if they exist at all..
                        if (is_link("$TEMPLATESDIR/$microsite_name")) {
                          unlink("$TEMPLATESDIR/$microsite_name");
                        }
                        if (is_link("$CMDIR/$microsite_name")) {
                          unlink("$CMDIR/$microsite_name");
                        }
                        debug("unlinked symlinks");

                        // Install an apache virtual host, if we can..
                        $microsite_domain = $rowdata["microsite_domain"];
                        $cmd = "$AXYL_HOME/install/del-apache-serveralias.sh";
                        if (is_executable($cmd)) {
                          $args = "$vservername $microsite_domain silent";
                          $ret = shell_exec("$cmd $args");
                          if ($ret != "") logit($ret);
                          debug("removed Apache ServerAlias for $microsite_domain");
                          $cmd = "$AXYL_HOME/install/prod-apache.sh";
                          if (is_executable($cmd)) {
                            $ret = shell_exec("$cmd reload");
                            if ($ret != "") logit($ret);
                            debug("apache was reloaded.");
                          }
                        }
                        else {
                          logit("ERROR: removal of Apache ServerAlias for $microsite_domain failed.");
                          logit("'$cmd' is not executable.");
                        }

                        // Now update the database record..
                        $q  = "UPDATE ax_microsite SET";
                        $q .= " req_microsite_publish=FALSE,";
                        $q .= " req_microsite_remove=FALSE,";
                        $q .= " currently_installed=FALSE";
                        $q .= " WHERE microsite_name='" . escape_string($microsite_name) . "'";
                        $remup = query_pgdb($dbid, $q);
                        if ($remup !== false) {
                          logit("removed microsite $microsite_name");
                        }
                      }
                    } // for
                  }
                } // got rows
                else {
                  debug("no microsite removal requests.");
                }
              }
            } // got a site_docroot
            else {
              debug("site_docroot is undefined or non-existent.");
            }
          } // ax_microsite exists
          else {
            debug("no ax_microsite table found.");
          }
        }
        disconnect_pgdb($dbid);
      } // can connect to db..
    }
    else {
      debug("database type '$dbtype' is not supported yet.");
    }
  } // foreach dbname
}

?>
