#!/usr/bin/php -q
<?php
/* ******************************************************************** */
/* CATALYST PHP Source Code                                             */
/* -------------------------------------------------------------------- */
/* This program is free software; you can redistribute it and/or modify */
/* it under the terms of the GNU General Public License as published by */
/* the Free Software Foundation; either version 2 of the License, or    */
/* (at your option) any later version.                                  */
/*                                                                      */
/* This program is distributed in the hope that it will be useful,      */
/* but WITHOUT ANY WARRANTY; without even the implied warranty of       */
/* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the        */
/* GNU General Public License for more details.                         */
/*                                                                      */
/* You should have received a copy of the GNU General Public License    */
/* along with this program; if not, write to:                           */
/*   The Free Software Foundation, Inc., 59 Temple Place, Suite 330,    */
/*   Boston, MA  02111-1307  USA                                        */
/* -------------------------------------------------------------------- */
/*                                                                      */
/* Filename:    dropcon.php                                             */
/* Author:      Paul Waite                                              */
/* Description: Drop all constraints for every table by recreating the  */
/*              table entirely.                                         */
/*                                                                      */
/* ******************************************************************** */
// CLASS DEFINITIONS
// ----------------------------------------------------------------------
class optlist {
  var $opts = array();
  var $optcount = 0;
  var $progname = "";
  // .....................................................................
  /*
  * Instantiate the optlist object. This does all the work since we expect
  * that this is being done to process command line arguments/options.
  */
  function optlist() {
    global $argc, $argv;
    $this->progname = basename(array_shift($argv));
    $argc -= 1;
    if (isset($argc) && $argc > 0) {
      $optstr = implode(" ", $argv);
      $optstr = str_replace("--", "~", $optstr);
      $optstr = str_replace(" -", "~", $optstr);
      $opts = explode("~", $optstr);
      foreach ($opts as $opt) {
        if ($opt != "") {
          $pos = strpos($opt, "=");
          if ($pos > -1) {
            $optname = trim(substr($opt, 0, $pos));
            $optval  = trim(substr($opt, $pos + 1));
          }
          else {
            $pos = strpos($opt, " ");
            if ($pos > -1) {
              $optname = trim(substr($opt, 0, $pos));
              $optval  = trim(substr($opt, $pos + 1));
            }
            else {
              $optname = trim($opt);
              $optval = "";
            }
          }
          $optval = str_replace("\"", "", $optval);
          $optval = str_replace("'", "", $optval);
          $this->opts[$optname] = $optval;
        }
      }
      $this->optcount = count($this->opts);
    }
  } // optlist

  // .....................................................................
  /**
  * Return the value of the named option. Returns the string associated
  * with this option, or false if it does not exist.
  * @param string $optname Name of the option
  * @return mixed The string value of the option, or false if not valid
  */
  function opt_value($optname) {
    if (isset($this->opts[$optname])) {
      return $this->opts[$optname];
    }
    else {
      return false;
    }
  }

  // .....................................................................
  /**
  * Return status of the named option. Returns true if the option exists
  * or false if it does not exist.
  * @param string $optname Name of the option
  * @return boolean True if it exists or false if it does not.
  */
  function opt_exists($optname) {
    return (isset($this->opts[$optname]));
  }

} // optlist class

// ----------------------------------------------------------------------
/**
* List of things class
* Encapsulates lists of items. A general-purpose class for containing
* lists of things. A utility class to hold lists of things like field
* lists, lists of tablenbames, orderby lists, etc.
* @package query
*/
class listofthings {
  /** The list of things being held */
  var $things;
  /** Total things we have */
  var $total = 0;
  // ....................................................................
  /**
  * Constructor
  * Create a new listofthings object.
  */
  function listofthings() {
    $this->clear();
  }
  // ....................................................................
  /**
  * Add thing
  * @param string $thing The identifier of the thing we are adding
  * @param mixed  $val   The value of the thing we are adding
  */
  function add($thing, $val="") {
    $this->things[$thing] = $val;
    $this->total += 1;
  }
  // ....................................................................
  /**
  * Clear things
  * Clears all things from the list.
  */
  function clear() {
    if (isset($this->things)) unset($this->things);
    $this->total = 0;
  }
  // ....................................................................
  /**
  * Return list
  * @param  string $delim The delimiter for the returned list
  * @return string The delimited list of names of things
  */
  function listed($delim=",") {
    $list = "";
    if (isset($this->things)) {
      reset($this->things);
      while (list($thing, $val) = each($this->things)) {
        $list .= "$thing~^";
      }
      $list = str_replace("~^", $delim, trim($list));
      if (substr($list, -1) == $delim) {
        $list = substr($list, 0, strlen($list) - 1);
      }
    }
    return $list;
  }
  // ....................................................................
  /**
  * Return values
  * @param  string $delim The delimiter for the returned list
  * @return string The delimited list of values of things
  */
  function values($delim=",") {
    $list = "";
    if (isset($this->things)) {
      reset($this->things);
      while (list($thing, $value) = each($this->things)) {
        if ($value === "") $value = "''";
        $list .= "$value~^";
      }
      $list = str_replace("~^", $delim, trim($list));
      if (substr($list, -1) == $delim) {
        $list = substr($list, 0, strlen($list) - 1);
      }
    }
    return $list;
  }
  // ....................................................................
  /**
  * Return equates
  * Returns the things we contain in key=value format, and all
  * as a string delimited by the given character.
  * @param  string $delim The delimiter for the returned list
  * @return string The delimited list of equated things
  */
  function equated($delim=",") {
    $list = "";
    if (isset($this->things)) {
      reset($this->things);
      while (list($thing, $value) = each($this->things)) {
        if ($value === "") $value = "''";
        $list .= "$thing=$value~^";
      }
      $list = str_replace("~^", $delim, trim($list));
      if (substr($list, -1) == $delim) {
        $list = substr($list, 0, strlen($list) - 1);
      }
    }
    return $list;
  }
} // listofthings class

// ----------------------------------------------------------------------
/**
* SQLquery class
* An SQL Statement Text Container.
* This class is the parent of the main dbquery class which directs the
* query to the database. It is mainly a container of SQL query text, in
* the variable 'sql', but also offers a few basic methods for building
* queries. For complex queries however, build your own in a string and
* then just set the 'sql' variable.
* @package query
*/
class sqlquery {
  /** The formatted SQL query itself @see build() */
  var $sql = "";
  /** Type of query 'SELECT', 'INSERT' or 'UPDATE' */
  var $type = "";
  /** List of fields in the query */
  var $fields;
  /** List of tables in the query */
  var $tables;
  /** The query WHERE clause components */
  var $where;
  /** The GROUP BY clause */
  var $groupby;
  /** The ORDER BY clause */
  var $orderby;
  /** The LIMIT clause */
  var $limit;
  /** The OFFSET clause */
  var $offset;
  // ....................................................................
  /**
  * Constructor
  * Create a new SQL Query object.
  * @param string $sql The SQL statement in full
  */
  function sqlquery($sql="") {
    $this->clear();
    $this->sql = $sql;
  }
  // ....................................................................
  /**
  * Clear query
  * Wipe the current sql definition.
  * @param string $sql The SQL statement in full
  */
  function clear() {
    if (isset($this->fields))  unset($this->fields);
    if (isset($this->tables))  unset($this->tables);
    if (isset($this->where))   unset($this->where);
    if (isset($this->groupby)) unset($this->groupby);
    if (isset($this->orderby)) unset($this->orderby);
    $this->fields  = new listofthings();
    $this->tables  = new listofthings();
    $this->where   = new listofthings();
    $this->groupby = new listofthings();
    $this->orderby = new listofthings();
    $this->sql = "";
    $this->limit = 0;
    $this->offset = 0;
  }
  // ....................................................................
  /**
  * Utility function to help building list of things
  * @param listofthings  $list_of_things  listofthings to add to
  * @param mixed         $list            A simple array or a delimited list
  * @param string        $delim           Delimiter, "," default
  */
  function addlist(&$list_of_things, $list, $delim=",") {
    if (is_array($list)) {
      $items = $list;
    }
    else {
      $items = explode($delim, $list);
    }
    // Add to our existing list..
    foreach ($items as $item) {
      if ($item != "") $list_of_things->add($item);
    }
  }
  // ....................................................................
  /**
  * Define field list
  * Add a list of fields to return in query. This is a cumulative function
  * which may be called more than once to add fields. You can specify the
  * list of fields either as an array, or as a delimited list. If the latter,
  * then default delimiter is a comma, unless you specify your own.
  * Applicable to SELECT, DELETE and UPDATE.
  * @param string $field_spec The field list to add to the query
  * @param string $delim The delimter you want to separate fields with
  */
  function fieldlist($field_spec="*", $delim=",") {
    $this->addlist($this->fields, $field_spec, $delim);
  }
  // ....................................................................
  /**
  * Define table list
  * Add the table specification to our list. This is a cumulative function
  * which may be called more than once to add tables. You can specify the
  * list of tables either as an array, or as a delimited list. If the latter,
  * then default delimiter is a comma, unless you specify your own.
  * @param string $table_spec The table list to add to the query
  * @param string $delim The delimiter you want to separate tables with
  */
  function tables($table_spec, $delim=",") {
    $this->addlist($this->tables, $table_spec, $delim);
  }
  // ....................................................................
  /**
  * Define table FROM list
  * A nicer synonym for "tables()" for SELECT
  * @param string $table_spec The table list to add to the query
  * @param string $delim The delimiter you want to separate tables with
  */
  function from($table_spec, $delim=",") {
    $this->tables($table_spec, $delim);
  }
  // ....................................................................
  /**
  * Define table INSERT INTO list
  * A nicer synonym for "tables()" for INSERT
  * @param string $table_spec The table list to add to the query
  * @param string $delim The delimiter you want to separate tables with
  */
  function into($table_spec, $delim=",") {
    $this->tables($table_spec, $delim);
  }
  // ....................................................................
  /**
  * Define group by field list
  * The fields can be an array, or a delimited list. If the latter, then default delimiter is a comma,
  * unless you specify your own.
  * @param string $field_spec The field list to add to the GROUP BY. Do not include words "GROUP BY".
  * @param string $delim The delimiter you want to separate the fields with
  */
  function groupby($field_spec="", $delim=",") {
    $this->addlist($this->groupby, $field_spec, $delim);
  }
  // ....................................................................
  /**
  * Define order field list
  * Defines the Sort order field list. The fields can be an array, or a
  * delimited list. If the latter, then default delimiter is a comma,
  * unless you specify your own.
  * @param string $field_spec The field list to add to the ORDER BY. Do not include words "ORDER BY".
  * @param string $delim The delimiter you want to separate the fields with
  */
  function orderby($field_spec="", $delim=",") {
    $this->addlist($this->orderby, $field_spec, $delim);
  }
  // ....................................................................
  /**
  * Define query LIMIT
  * @param integer $limit Numeric value for limit rows to return. Do not include the word "LIMIT".
  */
  function limit($limit) {
    $this->limit = $limit;
  }
  // ....................................................................
  /**
  * Define query OFFSET
  * @param integer $offset Numeric value for start row. Do not include the word "OFFSET".
  */
  function offset($offset) {
    $this->offset = $offset;
  }
  // ....................................................................
  /**
  * Define field assignments
  * Defines the field assignment clauses for UPDATE and INSERT queries.
  * @param string $field The name of the field to assign a value to
  * @param mixed  $val   The value to assign to the field. Processed according to type.
  */
  function set($field, $val) {
    global $RESPONSE;
    // Numerics are done without quotes
    if (is_int($val) || is_float($val)) {
      $this->fields->add($field, $val);
    }
    // Boolean formats dependent on database type..
    elseif (is_bool($val)) {
      $dbtype = (isset($RESPONSE) ? $RESPONSE->datasource->dbtype() : "postgres");
      switch ($dbtype) {
        case "postgres":
          $this->fields->add($field, ($val ? "'t'" : "'f'"));
          break;
        default:
          $this->fields->add($field, ($val ? "1" : "0"));
          break;
      } // switch
    }
    // Everything else is a quoted, escaped string..
    else {
      $this->fields->add($field, "'" . escape_string(trim($val)) . "'");
    }
  }
  // ....................................................................
  /**
  * Add WHERE clause component
  *
  * This function allows you to add a WHERE clause component. An example might
  * be something like: "AND c.foo='myval'". Either call this once with the whole
  * WHERE cluase string (minus the word "WHERE"), or multiple times with
  * parts of the where clause as in the example above.
  *
  * @param string $where_clause A WHERE clause component, without the "WHERE".
  */
  function where($where_clause) {
    if ($where_clause != "") {
      $this->where->add($where_clause);
    }
  }
  // ....................................................................
  /**
  * Build the SQL query
  * This takes the various components which have been added to the object
  * and parses them to build the full SQL statement which will be sent
  * to the server. The result is stored in $this->sql.
  */
  function build() {
    $sql = "";
    switch (strtoupper($this->type)) {
      case "SELECT":
        $sql .= "SELECT ";
        if ($this->fields->total == 0) $sql .= "*";
        else $sql .= $this->fields->listed();
        $sql .= " FROM ";
        $sql .= $this->tables->listed();
        if ($this->where->total > 0) {
          $sql .= " WHERE ";
          $sql .= $this->where->listed(" ");
        }
        if ($this->groupby->total > 0) {
          $sql .= " GROUP BY ";
          $sql .= $this->groupby->listed();
        }
        if ($this->orderby->total > 0) {
          $sql .= " ORDER BY ";
          $sql .= $this->orderby->listed();
        }
        if ($this->limit > 0) {
          $sql .= " LIMIT $this->limit";
        }
        if ($this->offset > 0) {
          $sql .= " OFFSET $this->offset";
        }
        break;

      case "INSERT":
        $sql .= "INSERT INTO ";
        $sql .= $this->tables->listed();
        if ($this->fields->total > 0) {
          $sql .= " (" . $this->fields->listed() . ")";
        }
        $sql .= " VALUES ";
        $sql .= "(" . $this->fields->values() . ")";
        break;

      case "DELETE":
        $sql .= "DELETE FROM ";
        $sql .= $this->tables->listed();
        if ($this->where->total > 0) {
          $sql .= " WHERE ";
          $sql .= $this->where->listed(" ");
        }
        break;

      case "UPDATE":
        $sql .= "UPDATE ";
        $sql .= $this->tables->listed();
        $sql .= " SET ";
        $sql .= $this->fields->equated();
        if ($this->where->total > 0) {
          $sql .= " WHERE ";
          $sql .= $this->where->listed(" ");
        }
        break;
    }
    // Render any NULL values..
    $this->sql = str_replace("'".NULLVALUE."'", "NULL", $sql);
    return $this->sql;
  }
} // sqlquery class

// ----------------------------------------------------------------------
/**
* DB Query class
* This class is the one which executes queries against the
* connected database.
* @package query
*/
class dbquery extends sqlquery {
  /** Number of rows returned after execute */
  var $rowcount = 0;
  /** Current row in the query */
  var $rowno = 0;
  /** Current row resource ID */
  var $rid = false;
  /** True if query is valid, post execution */
  var $valid = false;
  /** True if data was returned, after execute */
  var $hasdata = false;
  // ....................................................................
  /**
  * Constructor
  * Create a new DB Query object.
  * @param string $sql An SQL statement in full
  */
  function dbquery($sql="") {
    $this->sqlquery($sql);
    return $this;
  }
  // ....................................................................
  /**
  * Exceute the query
  * If we have an SQL phrase, execute it now. We store
  * the result in this->valid, and also return it. If
  * a transaction is open, update the status.
  * @return bool True if query was executed successfully
  */
  function execute() {
    global $RESPONSE;

    $this->rid = false;
    if ($this->sql == "") {
      $this->build();
    }
    if ($this->sql != "") {
      $this->rid = $RESPONSE->datasource->query($this->sql);
      // Now examine the result..
      if ($this->rid != false) {
        $this->rowcount = $RESPONSE->datasource->numrows($this->rid);
        $this->rowno = 0;
        $this->hasdata = ($this->rowcount > 0);
      }
      else {
        // Log the failed query..
        $errstr = "QFAIL: " . APP_NAME . ": " . $this->sql;
        error_log($errstr, 0);
      }
    }
    $this->valid = ($this->rid != false);
    return $this->valid;
  }
  // ....................................................................
  /**
  * Set the SQL statement
  * @param string $sql An SQL statement in full
  */
  function set_sql($sql) {
    $this->tidyup();
    $this->sql = $sql;
    return $this;
  }
  // ....................................................................
  /**
  * Free resources
  * Not really necessary, but you might be that
  * fastidious kind of person.
  */
  function tidyup() {
    global $RESPONSE;
    if ($this->rid) {
      $RESPONSE->datasource->freeresult($this->rid);
      $this->clear();
      $this->rowcount = 0;
      $this->rid = false;
      $this->valid = false;
      $this->hasdata = false;
    }
  }
} // dbquery class

// ----------------------------------------------------------------------
/**
* DB Rows class
* Renders a query into data and allows access to the data either
* directly or via the usual get first,last,next,previous cursor
* navigation.
* This class returns data as "rows" which is to say a standard
* array of data. For the associative array version then please
* @see dbrecords
* NOTE: On creation, it executes the query and positions to the
* initial record (defaulted to the first).
* @package query
*/
class dbrows extends dbquery {
  /** An array containing the current DB row */
  var $current_row;
  // ....................................................................
  /**
  * Constructor
  * Create a new DB Rows object.
  * @param string $sql An SQL statement in full
  */
  function dbrows($sql="") {
    $this->dbquery($sql);
    if ($sql != "") {
      $this->execute();
    }
  }
  // ....................................................................
  /**
  * Execute query
  * Execute this query. We override the parent method here
  * simply to ensure we are positioned at the first row.
  * @return bool True if query was executed successfully
  */
  function execute() {
    dbquery::execute();
    if ($this->valid) {
      $this->get_first();
    }
    return $this->valid;
  }
  // ....................................................................
  /**
  * Set the SQL statement
  * In this case we re-execute the SQL automatically.
  * @param string $sql An SQL statement in full
  * @return bool True if query was executed successfully
  */
  function set_sql($sql) {
    $this->tidyup();
    $this->sql = $sql;
    return $this->execute();
  }
  // ....................................................................
  /**
  * Get row raw
  * Return the given database row from the resultset. This method may
  * be over-ridden in subsequent child classes.
  * @param integer $rowno The row number to return
  * @return array True if row was available
  */
  function get_row_raw($rowno) {
    global $RESPONSE;
    if ($this->rid != false) {
      return $RESPONSE->datasource->fetch_row($this->rid, $rowno);
    }
    else return false;
  }
  // ....................................................................
  /**
  * Get row
  * Return the given database row from the resultset. Uses the
  * get_row_raw() method applicable to this class.
  * @see get_row_raw()
  * @param integer $rowno The row number to return
  * @return mixed The row if it is available, else returns FALSE.
  */
  function get_row($rowno) {
    if ($this->valid && ($this->rowcount > 0)) {
      if ($rowno > ($this->rowcount - 1)) $rowno = $this->rowcount - 1;
      elseif ($rowno < 0) $rowno = 0;
      $this->current_row = $this->get_row_raw($rowno);
      if ($this->current_row) {
        $this->rowno = $rowno;
      }
    }
    else {
      if (isset($this->current_row)) unset($this->current_row);
      $this->current_row = false;
    }
    return $this->current_row;
  }
  // ....................................................................
  /**
  * Refresh the query
  * Re-run the current SQL query. If successful the row will be stored
  * in $this->current_row.
  */
  function refresh() {
    $rowno = $this->rowno;
    $this->execute();
    $this->get_row($rowno);
  }
  // ....................................................................
  /**
  * Get current row
  * If current query is invalid, try to execute it first, then do a
  * get_first(). If query is then valid, return the current row.
  * @see get_first()
  * @return mixed The row if it is available, else returns FALSE.
  */
  function get_current() {
    if (!$this->valid) {
      $this->execute();
      $this->get_first();
    }
    return $this->current_row;
  }
  // ....................................................................
  /**
  * Get current row
  * If current query is invalid, try to execute it first, then do a
  * get_first(). If query is then valid, return the current row.
  * @see get_first()
  * @return mixed The row if it is available, else returns FALSE.
  */
  function get_first() {
    if (!$this->valid) $this->execute();
    return $this->get_row(0);
  }
  // ....................................................................
  /**
  * Get next row
  * If current query is invalid, try to execute it first, then get
  * the next row from the resultset.
  * @return mixed The row if it is available, else returns FALSE.
  */
  function get_next() {
    if (!$this->valid) $this->execute();
    if ($this->rowno < ($this->rowcount - 1)) {
      return $this->get_row($this->rowno + 1);
    }
    else return false;
  }
} // dbrows class

// ----------------------------------------------------------------------
/**
* DB Record class
* Renders a query into data and allows access to the data either
* directly or via the usual get first,last,next,previous cursor
* navigation.
* This class returns data as an associative array and is thus
* the most useful of all the data access methods. It extends the
* dbrows class, and over-rides the get_row_raw method to retrieve
* data.
* @see dbrows.
* @package query
*/
class dbrecords extends dbrows {
  /**
  * Constructor
  * Create a new DB Records object.
  * @param string $sql An SQL statement in full
  */
  function dbrecords($sql="") {
    $this->dbrows($sql);
  }
  // ....................................................................
  /**
  * Get row raw
  * Return the given database row from the resultset. This over-rides
  * the parent method of the same name and returns an array.
  * @param integer $rowno The row number to return
  * @return array True if row was available
  */
  function get_row_raw($rowno) {
    global $RESPONSE;
    if ($this->rid) {
      return $RESPONSE->datasource->fetch_array($this->rid, $rowno);
    }
    else return false;
  }
  // ....................................................................
  /**
  * Get field content
  * Return the field content from the current database array (row).
  * Does not provide ANY pre/post-processing.
  * @param string $fieldname The name of the field to return value of
  * @return mixed Value of the named field
  */
  function rawfield($fieldname) {
    global $RESPONSE;
    if ($this->rid) {
      $value = $this->current_row[$fieldname];
      return $value;
    }
    else return false;
  }
  // ....................................................................
  /**
  * Get field content
  * Return the field content from the current database array (row).
  * We provide some extra functionality here too:
  *  > If the fieldname startes with "^", then the field is expected
  *    to be a datetime and is returned as such.
  *  > If fieldname is formatted: fkname@tablename%keyfield%dispfield
  *    then 'fkname' is the field name to look up from the current
  *    record. We then use the value to lookup in table 'tablename'.
  *    The keyfield is 'keyfield', and the value to return is to
  *    be found in field 'dispfield'. This is foreign key lookup.
  *  > Finally, if the value is a string, then unescape_string is
  *    done automatically for you.
  *
  * @param string $fieldname The name of the field to return value of
  * @return mixed Value of the named field
  */
  function field($fieldname) {
    global $RESPONSE;
    if ($this->rid) {
      $value = $this->rawfield($fieldname);
      if (is_string($value)) {
        $value = unescape_string($value);
      }
      return $value;
    }
    else return false;
  }
  // ....................................................................
  /** Database independent boolean handling. Returns TRUE if the named
  * field in the current row is boolean true, else false.
  * @param string $fieldname The name of the field to return boolean value of
  * @return boolean True if field contains database-dependent true value
  */
  function istrue($fieldname) {
    global $RESPONSE;
    $boolvalue = $this->field($fieldname);
    $dbtype = (isset($RESPONSE) ? $RESPONSE->datasource->dbtype() : "postgres");
    switch ($dbtype) {
      case "postgres":
        $istrue = ($boolvalue == "t");
        break;
      default:
        $istrue = ($boolvalue == 1);
        break;
    }
    // Return result..
    return $istrue;
  }
} // dbrecords class

// ----------------------------------------------------------------------
/** Connect persistent to DB */
define("PERSISTENT",         true);

/** Connect non-persistent to DB */
define("NOT_PERSISTENT",     false);

/** Default datasource for queries @see add_database() */
define("DEFAULT_DATASOURCE", true);

// ----------------------------------------------------------------------
/**
* Datasources
*
* A datasources class is just a bunch of databases. If you want
* to access a database, register it in here first, then you
* can select it to perform queries on later.
* @package database
*/
class datasources {
  /** An array of database objects. All databases we can use as datasources */
  var $database;
  /** Default database name */
  var $db_name_default = "";
  /** Name of currently selected database */
  var $db_name_selected = "";
  /**
  * Constructor
  */
  function datasources() {
  }
  // ....................................................................
  /**
  * Constructor
  * Add a new base to our list of datasources. The dbtype and the name
  * are the only mandatory parameters.
  * @param string  $dbtype    The type of database eg: 'postgres', 'mssql' etc.
  * @param string  $name      The name of the database
  * @param string  $user      Name of a user who can access the database
  * @param string  $passwd    The password the user can access the database with
  * @param string  $host      The hostname of the machine running the database (TCP/IP)
  * @param integer $port      The port number of the database server
  * @param boolean $default   True if the database is the default database
  */
  function add_database($dbtype, $name, $user="", $passwd="", $host="", $port=0, $default=false) {
    global $LIBDIR;
    switch ($dbtype) {
      case "postgres":
        $this->database[$name] = new db_postgres($name, $user, $passwd, $host, $port);
        break;
    }
    // Make sure the default database is selected..
    if ($default) {
      // Select the default DB. This tries to
      // connect to it..
      $this->set_default($name);
      $this->select($name);

      // It is a fatal application error if the default
      // database cannot be connected..
      if (!$this->connected($name)) {
        die();
      }
    }
    return $this;
  }
  // ....................................................................
  /**
  * Selects a database to use
  *
  * This will connect it if it isn't already connected. Calling this
  * with no database name will select the default one. Returns the
  * database unique identifier, or false if none was selected.
  *
  * @param string $dbname  The name of the database to select
  * @return resource The database resource ID
  */
  function select($db_name="") {
    $dbid = false;
    if ($db_name == "") {
      $db_name = $this->db_name_default;
    }
    if (isset($this->database[$db_name])) {
      $db = $this->database[$db_name];
      $this->db_name_selected = $db_name;
      if (!$db->connected) {
        $db->connect(NOT_PERSISTENT);
        $this->database[$db_name] = $db;
      }
      if ($db->connected) {
        $dbid = $db->dbid;
      }
    }
    return $dbid;
  }
  // ....................................................................
  /**
  * Sets default database
  *
  * Internal function to set the name of the default database.
  * The database must exist as a defined database already.
  * @access private
  * @param string $dbname  The name of the database
  */
  // Private method:
  // Sets the database to be used as the default.
  function set_default($db_name) {
    if (isset($this->database[$db_name])) {
      $this->db_name_default = $db_name;
      return $this;
    }
  }
  // ....................................................................
  /**
  * Database resource ID
  *
  * Returns the database resource ID of the given database name.
  * If dbname is not given, returns ID of currently selected DB.
  * @access private
  * @param string $dbname  The name of the database
  * @return resource Database resource ID
  */
  function dbid($db_name="") {
    $res = false;
    if ($db_name == "") {
      $db_name = $this->db_name_selected;
    }
    if (isset($this->database[$db_name])) {
      $db = $this->database[$db_name];
      $res = $db->dbid;
    }
    return $res;
  }
  // ....................................................................
  /**
  * Database type
  *
  * Returns the database type of the given database name.
  * If dbname is not given, returns type of DB currently selected.
  * @access private
  * @param string $dbname  The name of the database
  * @return string Database type string
  */
  // Returns the database type of the selected database.
  function dbtype($db_name="") {
    $res = false;
    if ($db_name == "") {
      $db_name = $this->db_name_selected;
    }
    if (isset($this->database[$db_name])) {
      $db = $this->database[$db_name];
      $res = $db->type;
    }
    return $res;
  }
  // ....................................................................
  /**
  * Connected status
  *
  * Returns connected status of named database, or the currently
  * selected one if no name given.
  * @access private
  * @param string $dbname  The name of the database
  * @return boolean Database connection status true or false
  */
  function connected($db_name="") {
    $res = false;
    if ($db_name == "") {
      $db_name = $this->db_name_selected;
    }
    if (isset($this->database[$db_name])) {
      $db = $this->database[$db_name];
      $res = $db->connected;
    }
    return $res;
  }
  // ....................................................................
  /**
  * Connect
  *
  * Connects to the database which has been selected in the mode
  * specified, or non-peristent otherwise.
  * @access private
  * @param boolean $persistent  Whether to connect persistently or not
  * @return boolean Whether database connection was successful
  */
  function connect($persistent=NOT_PERSISTENT) {
    $connected = false;
    if (isset($this->database[$this->db_name_selected])) {
      $db = $this->database[$this->db_name_selected];

      $db->connect($persistent);
      if ($db->connected) {
        $connected = true;
      }
      else {
        $errmsg  = "Failed to connect to database '" . $this->name . "' ";
        $errmsg .= "type='"   . $this->type . "' ";
        $errmsg .= "host='"   . $this->host . "' ";
        $errmsg .= "port='"   . $this->port . "' ";
        $errmsg .= "user='"   . $this->user . "' ";
        $errmsg .= "passwd='" . $this->passwd . "' ";
        if ($persistent) $errmsg .= "persistent=yes";
        else $errmsg .= "persistent=no";
        error_log("CONNFAIL: $errmsg");
      }
    }
    return $connected;
  }
  // ....................................................................
  /**
  * Disconnect
  *
  * Disconnect the currently selected database.
  * @access private
  */
  function disconnect() {
    if (isset($this->database[$this->db_name_selected])) {
      $db = $this->database[$this->db_name_selected];
      $db->disconnect();
    }
  }
  // ....................................................................
  /**
  * Query
  *
  * The main query interface. Send a query to the database.
  * @access private
  * @param string $sql The query string to be executed
  */
  function query($sql) {
    $rid = false;
    if (isset($this->database[$this->db_name_selected])) {
      $db = $this->database[$this->db_name_selected];
      $rid = $db->query($sql);
    }
    return $rid;
  }
  // ....................................................................
  /**
  * Return number of rows
  *
  * Returns the number of rows in the current data set.
  * @access private
  * @param resource $rid The query resource ID
  */
  function numrows($rid) {
    $rows = 0;
    if (isset($this->database[$this->db_name_selected])) {
      $db = $this->database[$this->db_name_selected];
      $rows = $db->numrows($rid);
    }
    return $rows;
  }
  // ....................................................................
  /**
  * Free result
  *
  * Free the result of a query
  * @access private
  * @param resource $rid The query resource ID
  */
  function freeresult($rid) {
    if (isset($this->database[$this->db_name_selected])) {
      $db = $this->database[$this->db_name_selected];
      $db->freeresult($rid);
    }
  }
  // ....................................................................
  /**
  * Get DB error message
  *
  * Return the last error message reported by the database
  * @access private
  * @return string The error message, if any
  */
  function errormessage() {
    $errmsg = "";
    if (isset($this->database[$this->db_name_selected])) {
      $db = $this->database[$this->db_name_selected];
      $errmsg = $db->errormessage();
    }
    return $errmsg;
  }
  // ....................................................................
  /**
  * Fetch result row
  *
  * Get the given row of the result set as an enumerated array. Note
  * that fetch_array is more useful, as it returns the data in an
  * associative array.
  * @access private
  * @see fetch_array()
  * @param resource $rid   The query resource ID
  * @param integer  $rowno The number of the row to fetch
  * @return array  The row of fields as an enumerated array
  */
  function fetch_row($rid, $rowno) {
    $rows = false;
    if (isset($this->database[$this->db_name_selected])) {
      $db = $this->database[$this->db_name_selected];
      $rows = $db->fetch_row($rid, $rowno);
    }
    return $rows;
  }
  // ....................................................................
  /**
  * Fetch array
  *
  * Get the given row of the result set as an associative array. Each
  * element is stored in the form fieldname=>value.
  * @access private
  * @see fetch_row()
  * @param resource $rid   The query resource ID
  * @param integer  $rowno The number of the row to fetch
  * @return array  The row of fields as an associative array
  */
  // Get the given row of the result set as an array..
  function fetch_array($rid, $rowno) {
    $arr = false;
    if (isset($this->database[$this->db_name_selected])) {
      $db = $this->database[$this->db_name_selected];
      $arr = $db->fetch_array($rid, $rowno);
    }
    return $arr;
  }
} // datasources class

// ----------------------------------------------------------------------
/**
* Database
*
* Define a database. This is a parent class to all of the supported
* database flavours. It holds the main data describing a database
* and it's connection. The actual functionality to connect to a
* physical database and access its data is defined in the child
* classes of this one. For example, see file 'db-postgres.php'.
* NB: Normal users of the system should not have to deal with this
* class. This class is used by the datasources class.
* @package database
* @access private
*/
class database {
  /** Type of database eg: "postgres", "mssql_server".. */
  var $type = "";
  /** Name of this database */
  var $name = "";
  /** Host server of this database */
  var $host = "";
  /** Port to access it via TCP */
  var $port = 0;
  /** Default user to connect as */
  var $user = "";
  /** Default password to connect as */
  var $passwd = "";
  /** True if we want a persistent connection */
  var $persistent = false;
  /** Flag true if database was connected ok */
  var $connected = false;
  /** Unique identifier for database access */
  var $dbid = false;
  /**
  * Constructor
  * @param string  $name    The database name
  * @param string  $user    The username of user who can access the database
  * @param string  $passwd  The user password which can access the database
  * @param string  $host    The hostname of the machine running the database
  * @param integer $port    The port number of the database server
  */
  function database($name="", $user="", $passwd="", $host="", $port=0) {
    $this->name = $name;

    // If host and port ar not specified, then
    // we assume the database is local..
    $this->host = $host;
    $this->port = $port;

    // These can be used as defaults..
    $this->user = $user;
    $this->passwd = $passwd;

  } // database

} // database class

// ----------------------------------------------------------------------
/**
* POSTGRES database interface
*
* This is a database interface class. It is an impedance-matcher
* between the high-level Phplib functions for accessing data, and
* the specific functions suplpied by Php to access a particular
* flavour of databse such as Postgres, MS-SQL Server, Sybase etc.
* @package database
* @access private
*/
class db_postgres extends database {

  function db_postgres($name="", $user="", $passwd="", $host="", $port=0) {
    $this->database($name, $user, $passwd, $host, $port);
    $this->type = "postgres";
  }

  function connect($persistent) {
    if (!$this->connected) {
      $connstr = "";
      if ($this->host != "") $connstr .= " host=" . $this->host;
      if ($this->port != 0 ) $connstr .= " port=" . $this->port;
      $connstr .= " dbname=" . $this->name;
      $connstr .= " user=" . $this->user;
      if ($this->passwd != "") $connstr .= " password=" . $this->passwd;
      $connstr = trim($connstr);
      if ($persistent)
        $this->dbid = pg_pconnect("$connstr");
      else
        $this->dbid = pg_connect("$connstr");
      if ($this->dbid) {
        $this->connected = TRUE;
      }
    }
    return $this->connected;
  }
  function disconnect() {
    if (pg_close($this->dbid))
      $this->connected = FALSE;
  }
  function query($sql) {
    return pg_exec($this->dbid, $sql);
  }
  function numrows($rid) {
    return pg_numrows($rid);
  }
  function freeresult($rid) {
    pg_freeresult($rid);
  }
  function errormessage() {
    return pg_errormessage($this->dbid);
  }
  function fetch_row($rid, $rowno) {
    return pg_fetch_row($rid, $rowno);
  }
  function fetch_array($rid, $rowno) {
    return pg_fetch_array($rid, $rowno);
  }
  function begin_transaction() {
    return $this->query("BEGIN");
  }
  function commit() {
    return $this->query("COMMIT");
  }
  function rollback() {
    return $this->query("ROLLBACK");
  }
}

// ----------------------------------------------------------------------
// Ensure Postgres Php module is present..
if (!extension_loaded("pgsql")) {
  if (!dl("pgsql.so")) {
    exit;
  }
}
// ----------------------------------------------------------------------
// Placeholder class
class response {
  var $datasource;
  function response() {}
}

// ######################################################################
// MAIN PROGRAM
// ######################################################################
//debug_on(DBG_DIAGNOSTIC|DBG_SQL);
//debug_output(DBG_O_CLI);

$VERSION = "2.00";

// Only do anything if there are args..
$dbtarg = "";
$user = "";
$password = "";

// Command-line options..
$opts = new optlist();

// MAIN
if ($opts->opt_exists("db")) {
  // Program name..
  $progname = $opts->progname;

  // Database
  $dbtarg = $opts->opt_value("db");

  // Username
  if ($opts->opt_exists("user")) {
    $user = $opts->opt_value("user");
  }
  else {
    $ubits = explode(" ", exec("who"));
    if ($ubits[0] != "") $user = $ubits[0];
    else $user = "postgres";
  }

  // Password..
  if ($opts->opt_exists("password")) {
    $password = $opts->opt_value("password");
  }

  // Get Postgres version
  $pg_version = "";
  $vstr = 7.0;
  if ($opts->opt_exists("pgversion")) {
    $vstr = $opts->opt_value("pgversion");
  }
  else {
    $pgdir = "/usr/lib/postgresql/bin";
    if (file_exists("$pgdir/pg_config")) {
      $vbits = explode(" ", shell_exec("$pgdir/pg_config --version"));
      if ($vbits[1] != "") {
        $vstr = $vbits[1];
      }
    }
    elseif (file_exists("$pgdir/psql")) {
      $vbits = explode(" ", shell_exec("$pgdir/psql --version"));
      if ($vbits[2] != "") {
        $vvbits = explode(".", $vbits[2]);
        $vstr = $vvbits[0] . "." . (isset($vvbits[1]) ? $vvbits[1] : "0");
      }
    }
  }
  $pg_version = (float) $vstr;

  //  $pg_version = 6.5;
  $PGdrop_column = ($pg_version >= 7.3);


  $RESPONSE = new response();
  $RESPONSE->datasource = new datasources();

  $RESPONSE->datasource->add_database(
    "postgres",            // Database type eg: postgres, mssql_server, oracle etc.
    $dbtarg,               // Database name
    $user,                 // Name of user with access to the database
    $password,             // Password for this user
    "",                    // Host machine name
    "",                    // Port number
    DEFAULT_DATASOURCE     // DEFAULT_DATASOURCE, or omit for other databases
    );

  $stat = $RESPONSE->datasource->connected($dbtarg);
  if (!$stat) {
    echo "connect to $dbtarg failed.\n";
    exit;
  }

  // Announce
  $tstamp = date("d/m/Y H:i:s");

  echo "----------------------------------------------------------------------------\n";
  echo "-- rmconstraints $VERSION Postgres v$pg_version\n";
  echo "-- connecting as user $user";
  if ($password != "") echo " (password supplied)";
  echo " at $tstamp\n";
  echo "--\n";
  echo "-- Instructions:\n";
  echo "-- apply this script to target database $dbtarg, This will then\n";
  echo "-- drop ALL constraints on that database.\n";
  echo "--\n";
  echo "----------------------------------------------------------------------------\n";


  // ECHO SQL TO DROP CONSTRAINTS FOR ALL TABLES..
  $tablesQ  = "SELECT pgc.relname as tablename,";
  $tablesQ .= " pgc.oid as tableoid,";
  $tablesQ .= " pgc.relhaspkey as tablehaspkey";
  $tablesQ .= "  FROM pg_class pgc";
  $tablesQ .= " WHERE pgc.relkind='r'";
  $tablesQ .= "   AND pgc.relowner > 31";
  $tablesQ .= " ORDER BY pgc.relname";
  $tables = new dbrecords($tablesQ);
  if ($tables->hasdata) {

    // Disable triggers for 7.2-
    if ($pg_version < 7.3) {
      echo "UPDATE pg_class SET reltriggers=0 WHERE relname !~ '^pg_';\n";
    }
    do {
      $tablename = $tables->field("tablename");
      $tableoid  = $tables->field("tableoid");

      echo "\n";
      echo "-- TABLE: $tablename\n";
      echo "--\n";

      // Remove ALL Constraints
      if ($pg_version < 7.3) {
        echo "CREATE TABLE temp AS SELECT * FROM $tablename;\n";
        echo "DROP TABLE $tablename;\n";
        echo "CREATE TABLE $tablename AS SELECT * FROM temp;\n";
        echo "DROP TABLE temp;\n";
      }
      else {
        // Column constraints per table..
        // Check constraints stored on-table..
        $q  = "SELECT *";
        $q .= "  FROM pg_class pgc, pg_constraint pgcon";
        $q .= " WHERE pgc.relname='$tablename'";
        $q .= "   AND pgcon.conrelid=pgc.oid";
        $q .= "   AND pgcon.contype IN ('c','p','f')";
        $checks = new dbrecords($q);
        if ($checks->hasdata) {
          do {
            $conname = $checks->field("conname");
            echo "ALTER TABLE $tablename DROP CONSTRAINT $conname CASCADE;\n";
          } while ($checks->get_next());
        }
      }
    } while ($tables->get_next());

    // Re-enable triggers for 7.2-
    if ($pg_version < 7.3) {
      echo "BEGIN TRANSACTION;\n";
      echo "CREATE TEMP TABLE tr (tmp_relname name, tmp_reltriggers smallint);\n";
      echo "INSERT INTO tr SELECT C.relname, count(T.oid) FROM pg_class C, pg_trigger T WHERE C.oid=T.tgrelid AND C.relname !~ '^pg_' GROUP BY 1;\n";
      echo "UPDATE pg_class SET reltriggers=TMP.tmp_reltriggers FROM tr TMP WHERE pg_class.relname=TMP.tmp_relname;\n";
      echo "COMMIT TRANSACTION;\n";
    }

  } // tables hasdata

} // main
else {
  echo "usage: $opts->progname --db=database_name\n";
  echo "              [--user=user] [--password=passwd]\n";
  echo "              [--pgversion=n.nn]\n";
  echo "  Returns SQL which will drop all constraints in the named database.\n";
  echo "  --pgversion forces the Postgres version\n";
}
// ----------------------------------------------------------------------
?>