/*
 *
 *  Program to scan a Bacula Volume and compare it with
 *    the catalog and optionally synchronize the catalog
 *    with the tape.
 *
 *   Kern E. Sibbald, December 2001
 *
 *
 *   Version $Id: bscan.c,v 1.70 2004/10/15 17:09:05 kerns Exp $
 */
/*
   Copyright (C) 2001-2004 Kern Sibbald and John Walker

   This program is free software; you can redistribute it and/or
   modify it under the terms of the GNU General Public License as
   published by the Free Software Foundation; either version 2 of
   the License, or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public
   License along with this program; if not, write to the Free
   Software Foundation, Inc., 59 Temple Place - Suite 330, Boston,
   MA 02111-1307, USA.

 */

#include "bacula.h"
#include "stored.h"
#include "findlib/find.h"
#include "cats/cats.h"

/* Forward referenced functions */
static void do_scan(void);
static bool record_cb(DCR *dcr, DEV_RECORD *rec);
static int  create_file_attributes_record(B_DB *db, JCR *mjcr, 
			       char *fname, char *lname, int type,
			       char *ap, DEV_RECORD *rec);
static int  create_media_record(B_DB *db, MEDIA_DBR *mr, VOLUME_LABEL *vl);
static bool update_media_record(B_DB *db, MEDIA_DBR *mr);
static int  create_pool_record(B_DB *db, POOL_DBR *pr);
static JCR *create_job_record(B_DB *db, JOB_DBR *mr, SESSION_LABEL *label, DEV_RECORD *rec);
static int  update_job_record(B_DB *db, JOB_DBR *mr, SESSION_LABEL *elabel, 
			      DEV_RECORD *rec);
static int  create_client_record(B_DB *db, CLIENT_DBR *cr);
static int  create_fileset_record(B_DB *db, FILESET_DBR *fsr);
static int  create_jobmedia_record(B_DB *db, JCR *jcr);
static JCR *create_jcr(JOB_DBR *jr, DEV_RECORD *rec, uint32_t JobId);
static int update_SIG_record(B_DB *db, char *SIGbuf, DEV_RECORD *rec, int type);


/* Global variables */
STORES *me;
#if defined(HAVE_CYGWIN) || defined(HAVE_WIN32)
int win32_client = 1;
#else
int win32_client = 0;
#endif


/* Local variables */
static DEVICE *dev = NULL;
static B_DB *db;
static JCR *bjcr;		      /* jcr for bscan */
static BSR *bsr = NULL;
static MEDIA_DBR mr;
static POOL_DBR pr;
static JOB_DBR jr;
static CLIENT_DBR cr;
static FILESET_DBR fsr;
static ATTR_DBR ar;
static FILE_DBR fr;
static SESSION_LABEL label;
static SESSION_LABEL elabel;
static ATTR *attr;

static time_t lasttime = 0;

static const char *db_name = "bacula";
static const char *db_user = "bacula";
static const char *db_password = "";
static const char *db_host = NULL;
static const char *wd = NULL;
static bool update_db = false;
static bool update_vol_info = false;
static bool list_records = false;
static int ignored_msgs = 0;

static uint64_t currentVolumeSize;
static int64_t last_pct = -1;
static bool showProgress = false;
static int num_jobs = 0;
static int num_pools = 0;
static int num_media = 0;
static int num_files = 0;

#define CONFIG_FILE "bacula-sd.conf"
char *configfile;
bool forge_on = false;


static void usage()
{
   fprintf(stderr, _(
"Copyright (C) 2001-2004 Kern Sibbald and John Walker.\n"
"\nVersion: " VERSION " (" BDATE ")\n\n"
"Usage: bscan [ options ] <bacula-archive>\n"
"       -b bootstrap      specify a bootstrap file\n"
"       -c <file>         specify configuration file\n"
"       -d <nn>           set debug level to nn\n"
"       -m                update media info in database\n"
"       -n <name>         specify the database name (default bacula)\n"
"       -u <user>         specify database user name (default bacula)\n"
"       -P <password      specify database password (default none)\n"
"       -h <host>         specify database host (default NULL)\n"
"       -p                proceed inspite of I/O errors\n"
"       -r                list records\n"
"       -s                synchronize or store in database\n"
"       -v                verbose\n"
"       -V <Volumes>      specify Volume names (separated by |)\n"
"       -w <dir>          specify working directory (default from conf file)\n"
"       -?                print this message\n\n"));
   exit(1);
}

int main (int argc, char *argv[])
{
   int ch;
   struct stat stat_buf;
   char *VolumeName = NULL;

   my_name_is(argc, argv, "bscan");
   init_msg(NULL, NULL);


   while ((ch = getopt(argc, argv, "b:c:d:h:mn:pP:rsSu:vV:w:?")) != -1) {
      switch (ch) {
      case 'S' :
	 showProgress = true;
	 break;
      case 'b':
	 bsr = parse_bsr(NULL, optarg);
	 break;

      case 'c':                    /* specify config file */
	 if (configfile != NULL) {
	    free(configfile);
	 }
	 configfile = bstrdup(optarg);
	 break;

      case 'd':                    /* debug level */
	 debug_level = atoi(optarg);
	 if (debug_level <= 0)
	    debug_level = 1; 
	 break;

      case 'h':
	 db_host = optarg;
	 break;

      case 'm':
	 update_vol_info = true;
	 break;

      case 'n':
	 db_name = optarg;
	 break;

      case 'u':
	 db_user = optarg;
	 break;

      case 'P':
	 db_password = optarg;
	 break;

      case 'p':
	 forge_on = true;
	 break;

      case 'r':
	 list_records = true;
	 break;

      case 's':
	 update_db = true;
	 break;

      case 'v':
	 verbose++;
	 break;

      case 'V':                    /* Volume name */
	 VolumeName = optarg;
	 break;

      case 'w':
	 wd = optarg;
	 break;

      case '?':
      default:
	 usage();

      }  
   }
   argc -= optind;
   argv += optind;

   if (argc != 1) {
      Pmsg0(0, _("Wrong number of arguments: \n"));
      usage();
   }

   if (configfile == NULL) {
      configfile = bstrdup(CONFIG_FILE);
   }

   parse_config(configfile);
   LockRes();
   me = (STORES *)GetNextRes(R_STORAGE, NULL);
   if (!me) {
      UnlockRes();
      Emsg1(M_ERROR_TERM, 0, _("No Storage resource defined in %s. Cannot continue.\n"), 
	 configfile);
   }
   UnlockRes();
   /* Check if -w option given, otherwise use resource for working directory */
   if (wd) { 
      working_directory = wd;
   } else if (!me->working_directory) {
      Emsg1(M_ERROR_TERM, 0, _("No Working Directory defined in %s. Cannot continue.\n"),
	 configfile);
   } else {
      working_directory = me->working_directory;
   }

   /* Check that working directory is good */
   if (stat(working_directory, &stat_buf) != 0) {
      Emsg1(M_ERROR_TERM, 0, _("Working Directory: %s not found. Cannot continue.\n"),
	 working_directory);
   }
   if (!S_ISDIR(stat_buf.st_mode)) {
      Emsg1(M_ERROR_TERM, 0, _("Working Directory: %s is not a directory. Cannot continue.\n"),
	 working_directory);
   }

   bjcr = setup_jcr("bscan", argv[0], bsr, VolumeName, 1); /* read device */
   if (!bjcr) { 
      exit(1);
   }
   dev = bjcr->dcr->dev;
   if (showProgress) {
      struct stat sb;
      fstat(dev->fd, &sb);
      currentVolumeSize = sb.st_size;
      Pmsg1(000, _("Current Volume Size = %" llu "\n"), currentVolumeSize);
   }

   if ((db=db_init_database(NULL, db_name, db_user, db_password, 
	db_host, 0, NULL, 0)) == NULL) {
      Emsg0(M_ERROR_TERM, 0, _("Could not init Bacula database\n"));
   }
   if (!db_open_database(NULL, db)) {
      Emsg0(M_ERROR_TERM, 0, db_strerror(db));
   }
   Dmsg0(200, "Database opened\n");
   if (verbose) {
      Pmsg2(000, _("Using Database: %s, User: %s\n"), db_name, db_user);
   }

   do_scan();
   printf("Records %sadded or updated in the catalog:\n%7d Media\n%7d Pool\n%7d Job\n%7d File\n",
      update_db?"":"would have been ",
      num_media, num_pools, num_jobs, num_files);

   free_jcr(bjcr);
   term_dev(dev);
   return 0;
}
  
/*  
 * We are at the end of reading a tape. Now, we simulate handling
 *   the end of writing a tape by wiffling through the attached
 *   jcrs creating jobmedia records.
 */
static bool bscan_mount_next_read_volume(DCR *dcr)
{
   DEVICE *dev = dcr->dev;
   DCR *mdcr;
   Dmsg1(100, "Walk attached jcrs. Volume=%s\n", dev->VolCatInfo.VolCatName);
   foreach_dlist(mdcr, dev->attached_dcrs) {
      JCR *mjcr = mdcr->jcr;
      if (mjcr->JobId == 0) {
	 continue;
      }
      if (verbose) {
         Pmsg1(000, _("Create JobMedia for Job %s\n"), mjcr->Job);
      }
      if (dev->state & ST_TAPE) {
	 mdcr->EndBlock = dev->EndBlock;
	 mdcr->EndFile = dev->EndFile;
      } else {
	 mdcr->EndBlock = (uint32_t)dev->file_addr;
	 mdcr->EndFile = (uint32_t)(dev->file_addr >> 32);
      }
      if (!create_jobmedia_record(db, mjcr)) {
         Pmsg2(000, _("Could not create JobMedia record for Volume=%s Job=%s\n"),
	    dev->VolCatInfo.VolCatName, mjcr->Job);
      }
   }  
   /* Now let common read routine get up next tape. Note,
    * we call mount_next... with bscan's jcr because that is where we
    * have the Volume list, but we get attached.
    */
   bool stat = mount_next_read_volume(dcr);

   if (showProgress) {
      struct stat sb;
      fstat(dev->fd, &sb);
      currentVolumeSize = sb.st_size;
      Pmsg1(000, _("Current Volume Size = %" llu "\n"), currentVolumeSize);
   }
   return stat;
}

static void do_scan()		  
{
   attr = new_attr();

   memset(&ar, 0, sizeof(ar));
   memset(&pr, 0, sizeof(pr));
   memset(&jr, 0, sizeof(jr));
   memset(&cr, 0, sizeof(cr));
   memset(&fsr, 0, sizeof(fsr));
   memset(&fr, 0, sizeof(fr));

   /* Detach bscan's jcr as we are not a real Job on the tape */

   read_records(bjcr->dcr, record_cb, bscan_mount_next_read_volume);

   free_attr(attr);
}

/*
 * Returns: true  if OK
 *	    false if error
 */
static bool record_cb(DCR *dcr, DEV_RECORD *rec)
{
   JCR *mjcr;
   char ec1[30];
   DEVICE *dev = dcr->dev;
   JCR *bjcr = dcr->jcr;
   DEV_BLOCK *block = dcr->block;

   if (rec->data_len > 0) {
      mr.VolBytes += rec->data_len + WRITE_RECHDR_LENGTH; /* Accumulate Volume bytes */
      if (showProgress) {
	 int64_t pct = (mr.VolBytes * 100) / currentVolumeSize;
	 if (pct != last_pct) {
            fprintf(stdout, "done: %" lld "\n", pct);
	    fflush(stdout);
	    last_pct = pct;
	 }
      }
   }
   
   if (list_records) {
      Pmsg5(000, _("Record: SessId=%u SessTim=%u FileIndex=%d Stream=%d len=%u\n"),
	    rec->VolSessionId, rec->VolSessionTime, rec->FileIndex, 
	    rec->Stream, rec->data_len);
   }
   /* 
    * Check for Start or End of Session Record 
    *
    */
   if (rec->FileIndex < 0) {
      bool save_update_db = update_db;

      if (verbose > 1) {
	 dump_label_record(dev, rec, 1);
      }
      switch (rec->FileIndex) {
      case PRE_LABEL:
         Pmsg0(000, _("Volume is prelabeled. This tape cannot be scanned.\n"));
	 return false;
	 break;

      case VOL_LABEL:
	 unser_volume_label(dev, rec);
	 /* Check Pool info */
	 bstrncpy(pr.Name, dev->VolHdr.PoolName, sizeof(pr.Name));
	 bstrncpy(pr.PoolType, dev->VolHdr.PoolType, sizeof(pr.PoolType));
	 num_pools++;
	 if (db_get_pool_record(bjcr, db, &pr)) {
	    if (verbose) {
               Pmsg1(000, _("Pool record for %s found in DB.\n"), pr.Name);
	    }
	 } else {
	    if (!update_db) {
               Pmsg1(000, _("VOL_LABEL: Pool record not found for Pool: %s\n"),
		  pr.Name);
	    }
	    create_pool_record(db, &pr);
	 }
	 if (strcmp(pr.PoolType, dev->VolHdr.PoolType) != 0) {
            Pmsg2(000, _("VOL_LABEL: PoolType mismatch. DB=%s Vol=%s\n"),
	       pr.PoolType, dev->VolHdr.PoolType);
	    return true;
	 } else if (verbose) {
            Pmsg1(000, _("Pool type \"%s\" is OK.\n"), pr.PoolType);
	 }

	 /* Check Media Info */
	 memset(&mr, 0, sizeof(mr));
	 bstrncpy(mr.VolumeName, dev->VolHdr.VolName, sizeof(mr.VolumeName));
	 mr.PoolId = pr.PoolId;
	 num_media++;
	 if (db_get_media_record(bjcr, db, &mr)) {
	    if (verbose) {
               Pmsg1(000, _("Media record for %s found in DB.\n"), mr.VolumeName);
	    }
	    /* Clear out some volume statistics that will be updated */
	    mr.VolJobs = mr.VolFiles = mr.VolBlocks = 0;
	    mr.VolBytes = rec->data_len + 20;
	 } else {
	    if (!update_db) {
               Pmsg1(000, _("VOL_LABEL: Media record not found for Volume: %s\n"),
		  mr.VolumeName);
	    }
	    bstrncpy(mr.MediaType, dev->VolHdr.MediaType, sizeof(mr.MediaType));
	    create_media_record(db, &mr, &dev->VolHdr);
	 }
	 if (strcmp(mr.MediaType, dev->VolHdr.MediaType) != 0) {
            Pmsg2(000, _("VOL_LABEL: MediaType mismatch. DB=%s Vol=%s\n"),
	       mr.MediaType, dev->VolHdr.MediaType);
	    return true;	      /* ignore error */
	 } else if (verbose) {
            Pmsg1(000, _("Media type \"%s\" is OK.\n"), mr.MediaType);
	 }
	 /* Reset some JCR variables */
	 foreach_dlist(dcr, dev->attached_dcrs) {
	    dcr->VolFirstIndex = dcr->FileIndex = 0;
	    dcr->StartBlock = dcr->EndBlock = 0;
	    dcr->StartFile = dcr->EndFile = 0;
	 }

         Pmsg1(000, _("VOL_LABEL: OK for Volume: %s\n"), mr.VolumeName);
	 break;

      case SOS_LABEL:
	 mr.VolJobs++;
	 num_jobs++;
	 if (ignored_msgs > 0) {
            Pmsg1(000, _("%d \"errors\" ignored before first Start of Session record.\n"), 
		  ignored_msgs);
	    ignored_msgs = 0;
	 }
	 unser_session_label(&label, rec);
	 memset(&jr, 0, sizeof(jr));
	 bstrncpy(jr.Job, label.Job, sizeof(jr.Job));
	 if (db_get_job_record(bjcr, db, &jr)) {
	    /* Job record already exists in DB */
            update_db = false;  /* don't change db in create_job_record */
	    if (verbose) {
               Pmsg1(000, _("SOS_LABEL: Found Job record for JobId: %d\n"), jr.JobId);
	    }
	 } else {
	    /* Must create a Job record in DB */
	    if (!update_db) {
               Pmsg1(000, _("SOS_LABEL: Job record not found for JobId: %d\n"),
		  jr.JobId);
	    }
	 }
	 /* Create Client record if not already there */
	    bstrncpy(cr.Name, label.ClientName, sizeof(cr.Name));
	    create_client_record(db, &cr);
	    jr.ClientId = cr.ClientId;

         /* process label, if Job record exists don't update db */
	 mjcr = create_job_record(db, &jr, &label, rec);
	 dcr = mjcr->dcr;
	 update_db = save_update_db;

	 jr.PoolId = pr.PoolId;
	 /* Set start positions into JCR */
	 if (dev->state & ST_TAPE) {
	    /*
	     * Note, we have already advanced past current block,
	     *	so the correct number is block_num - 1 
	     */ 
	    dcr->StartBlock = dev->block_num - 1;
	    dcr->StartFile = dev->file;
	 } else {
	    dcr->StartBlock = (uint32_t)dev->file_addr;
	    dcr->StartFile = (uint32_t)(dev->file_addr >> 32);
	 }
	 mjcr->start_time = jr.StartTime;
	 mjcr->JobLevel = jr.JobLevel;

	 mjcr->client_name = get_pool_memory(PM_FNAME);
	 pm_strcpy(mjcr->client_name, label.ClientName);
	 mjcr->fileset_name = get_pool_memory(PM_FNAME);
	 pm_strcpy(mjcr->fileset_name, label.FileSetName);
	 bstrncpy(dcr->pool_type, label.PoolType, sizeof(dcr->pool_type));
	 bstrncpy(dcr->pool_name, label.PoolName, sizeof(dcr->pool_name));

	 if (rec->VolSessionId != jr.VolSessionId) {
            Pmsg3(000, _("SOS_LABEL: VolSessId mismatch for JobId=%u. DB=%d Vol=%d\n"),
	       jr.JobId,
	       jr.VolSessionId, rec->VolSessionId);
	    return true;	      /* ignore error */
	 }
	 if (rec->VolSessionTime != jr.VolSessionTime) {
            Pmsg3(000, _("SOS_LABEL: VolSessTime mismatch for JobId=%u. DB=%d Vol=%d\n"),
	       jr.JobId,
	       jr.VolSessionTime, rec->VolSessionTime);
	    return true;	      /* ignore error */
	 }
	 if (jr.PoolId != pr.PoolId) {
            Pmsg3(000, _("SOS_LABEL: PoolId mismatch for JobId=%u. DB=%d Vol=%d\n"),
	       jr.JobId,
	       jr.PoolId, pr.PoolId);
	    return true;	      /* ignore error */
	 }
	 break;

      case EOS_LABEL:
	 unser_session_label(&elabel, rec);

	 /* Create FileSet record */
	 bstrncpy(fsr.FileSet, label.FileSetName, sizeof(fsr.FileSet));
	 bstrncpy(fsr.MD5, label.FileSetMD5, sizeof(fsr.MD5));
	 create_fileset_record(db, &fsr);
	 jr.FileSetId = fsr.FileSetId;

	 mjcr = get_jcr_by_session(rec->VolSessionId, rec->VolSessionTime);
	 if (!mjcr) {
            Pmsg2(000, _("Could not find SessId=%d SessTime=%d for EOS record.\n"),
		  rec->VolSessionId, rec->VolSessionTime);
	    break;
	 }

	 /* Do the final update to the Job record */
	 update_job_record(db, &jr, &elabel, rec);

	 mjcr->end_time = jr.EndTime;
	 mjcr->JobStatus = JS_Terminated;

	 /* Create JobMedia record */
	 create_jobmedia_record(db, mjcr);
	 dev->attached_dcrs->remove(mjcr->dcr);
	 free_jcr(mjcr);

	 break;

      case EOM_LABEL:
	 break;

      case EOT_LABEL:		   /* end of all tapes */
	 /* 
	  * Wiffle through all jobs still open and close
	  *   them.
	  */
	 if (update_db) {
	    DCR *mdcr;
	    foreach_dlist(mdcr, dev->attached_dcrs) {
	       JCR *mjcr = mdcr->jcr;
	       if (!mjcr || mjcr->JobId == 0) {
		  continue;
	       }
	       jr.JobId = mjcr->JobId;
	       /* Mark Job as Error Terimined */
	       jr.JobStatus = JS_ErrorTerminated;
	       jr.JobFiles = mjcr->JobFiles;
	       jr.JobBytes = mjcr->JobBytes;
	       jr.VolSessionId = mjcr->VolSessionId;
	       jr.VolSessionTime = mjcr->VolSessionTime;
	       jr.JobTDate = (utime_t)mjcr->start_time;
	       jr.ClientId = mjcr->ClientId;
	       if (!db_update_job_end_record(bjcr, db, &jr)) {
                  Pmsg1(0, _("Could not update job record. ERR=%s\n"), db_strerror(db));
	       }
	       mjcr->dcr = NULL;
	       free_jcr(mjcr);
	    }
	 }
	 mr.VolFiles = rec->File;
	 mr.VolBlocks = rec->Block;
	 mr.VolBytes += mr.VolBlocks * WRITE_BLKHDR_LENGTH; /* approx. */
	 mr.VolMounts++;
	 update_media_record(db, &mr);
         Pmsg3(0, _("End of all Volumes. VolFiles=%u VolBlocks=%u VolBytes=%s\n"), mr.VolFiles,
		    mr.VolBlocks, edit_uint64_with_commas(mr.VolBytes, ec1));
	 break;
      default:
	 break;
      } /* end switch */
      return true;
   }

   mjcr = get_jcr_by_session(rec->VolSessionId, rec->VolSessionTime);
   if (!mjcr) {
      if (mr.VolJobs > 0) {
         Pmsg2(000, _("Could not find Job for SessId=%d SessTime=%d record.\n"),
		      rec->VolSessionId, rec->VolSessionTime);
      } else {
	 ignored_msgs++;
      }
      return true;
   }
   dcr = mjcr->dcr;
   if (dcr->VolFirstIndex == 0) {
      dcr->VolFirstIndex = block->FirstIndex;
   }

   /* File Attributes stream */
   switch (rec->Stream) {
   case STREAM_UNIX_ATTRIBUTES:   
   case STREAM_UNIX_ATTRIBUTES_EX:  

      if (!unpack_attributes_record(bjcr, rec->Stream, rec->data, attr)) {
         Emsg0(M_ERROR_TERM, 0, _("Cannot continue.\n"));
      }

      if (attr->file_index != rec->FileIndex) {
         Emsg2(M_ERROR_TERM, 0, _("Record header file index %ld not equal record index %ld\n"),
	    rec->FileIndex, attr->file_index);
      }
       
      if (verbose > 1) {
	 decode_stat(attr->attr, &attr->statp, &attr->LinkFI);
	 build_attr_output_fnames(bjcr, attr);
	 print_ls_output(bjcr, attr);
      }
      fr.JobId = mjcr->JobId;
      fr.FileId = 0;
      num_files++;
      if (verbose && (num_files & 0x7FFF) == 0) {
	 char ed1[30], ed2[30], ed3[30], ed4[30];
         Pmsg4(000, _("%s file records. At file:blk=%s:%s bytes=%s\n"),
		     edit_uint64_with_commas(num_files, ed1),
		     edit_uint64_with_commas(rec->File, ed2),
		     edit_uint64_with_commas(rec->Block, ed3),
		     edit_uint64_with_commas(mr.VolBytes, ed4));
      } 
      create_file_attributes_record(db, mjcr, attr->fname, attr->lname, 
	    attr->type, attr->attr, rec);
      free_jcr(mjcr);
      break;

   /* Data stream */
   case STREAM_WIN32_DATA:
   case STREAM_FILE_DATA:
   case STREAM_SPARSE_DATA:
      mjcr->JobBytes += rec->data_len;
      if (rec->Stream == STREAM_SPARSE_DATA) {
	 mjcr->JobBytes -= sizeof(uint64_t);
      }
	 
      free_jcr(mjcr);		      /* done using JCR */
      break;

   case STREAM_GZIP_DATA:
      mjcr->JobBytes += rec->data_len; /* No correct, we should expand it */
      free_jcr(mjcr);		      /* done using JCR */
      break;

   case STREAM_SPARSE_GZIP_DATA:
      mjcr->JobBytes += rec->data_len - sizeof(uint64_t); /* No correct, we should expand it */
      free_jcr(mjcr);		      /* done using JCR */
      break;

   /* Win32 GZIP stream */
   case STREAM_WIN32_GZIP_DATA:
      mjcr->JobBytes += rec->data_len;
      free_jcr(mjcr);		      /* done using JCR */
      break;

   case STREAM_MD5_SIGNATURE:
      char MD5buf[50];
      bin_to_base64(MD5buf, (char *)rec->data, 16); /* encode 16 bytes */
      if (verbose > 1) {
         Pmsg1(000, _("Got MD5 record: %s\n"), MD5buf);
      }
      update_SIG_record(db, MD5buf, rec, MD5_SIG);
      break;

   case STREAM_SHA1_SIGNATURE:
      char SIGbuf[50];
      bin_to_base64(SIGbuf, (char *)rec->data, 20); /* encode 20 bytes */
      if (verbose > 1) {
         Pmsg1(000, _("Got SHA1 record: %s\n"), SIGbuf);
      }
      update_SIG_record(db, SIGbuf, rec, SHA1_SIG);
      break;


   case STREAM_PROGRAM_NAMES:
      if (verbose) {
         Pmsg1(000, _("Got Prog Names Stream: %s\n"), rec->data);
      }
      break;

   case STREAM_PROGRAM_DATA:
      if (verbose > 1) {
         Pmsg0(000, _("Got Prog Data Stream record.\n"));
      }
      break;
   default:
      Pmsg2(0, _("Unknown stream type!!! stream=%d data=%s\n"), rec->Stream, rec->data);
      break;
   }
   return true;
}

/*
 * Free the Job Control Record if no one is still using it.
 *  Called from main free_jcr() routine in src/lib/jcr.c so
 *  that we can do our Director specific cleanup of the jcr.
 */
static void bscan_free_jcr(JCR *jcr)
{
   Dmsg0(200, "Start dird free_jcr\n");

   if (jcr->file_bsock) {
      Dmsg0(200, "Close File bsock\n");
      bnet_close(jcr->file_bsock);
   }
   if (jcr->store_bsock) {
      Dmsg0(200, "Close Store bsock\n");
      bnet_close(jcr->store_bsock);
   }
   if (jcr->RestoreBootstrap) {
      free(jcr->RestoreBootstrap);
   }
   if (jcr->dcr) {
      free_dcr(jcr->dcr);
      jcr->dcr = NULL;
   }
   Dmsg0(200, "End dird free_jcr\n");
}

/*
 * We got a File Attributes record on the tape.  Now, lookup the Job
 *   record, and then create the attributes record.
 */
static int create_file_attributes_record(B_DB *db, JCR *mjcr,
			       char *fname, char *lname, int type,
			       char *ap, DEV_RECORD *rec)
{
   DCR *dcr = mjcr->dcr;
   ar.fname = fname;
   ar.link = lname;
   ar.ClientId = mjcr->ClientId;
   ar.JobId = mjcr->JobId;
   ar.Stream = rec->Stream;
   ar.FileIndex = rec->FileIndex;
   ar.attr = ap;
   if (dcr->VolFirstIndex == 0) {
      dcr->VolFirstIndex = rec->FileIndex;
   }
   dcr->FileIndex = rec->FileIndex;
   mjcr->JobFiles++;

   if (!update_db) {
      return 1;
   }

   if (!db_create_file_attributes_record(bjcr, db, &ar)) {
      Pmsg1(0, _("Could not create File Attributes record. ERR=%s\n"), db_strerror(db));
      return 0;
   }
   mjcr->FileId = ar.FileId;

   if (verbose > 1) {
      Pmsg1(000, _("Created File record: %s\n"), fname);   
   }
   return 1;
}

/*
 * For each Volume we see, we create a Medium record
 */
static int create_media_record(B_DB *db, MEDIA_DBR *mr, VOLUME_LABEL *vl)
{
   struct date_time dt;
   struct tm tm;

   /* We mark Vols as Archive to keep them from being re-written */
   bstrncpy(mr->VolStatus, "Archive", sizeof(mr->VolStatus));
   mr->VolRetention = 365 * 3600 * 24; /* 1 year */
   if (vl->VerNum >= 11) {
      mr->FirstWritten = btime_to_utime(vl->write_btime);
      mr->LabelDate    = btime_to_utime(vl->label_btime);
   } else {
      /* DEPRECATED DO NOT USE */
      dt.julian_day_number = vl->write_date;
      dt.julian_day_fraction = vl->write_time;
      tm_decode(&dt, &tm);
      mr->FirstWritten = mktime(&tm);
      dt.julian_day_number = vl->label_date;
      dt.julian_day_fraction = vl->label_time;
      tm_decode(&dt, &tm);
      mr->LabelDate = mktime(&tm);
   }
   lasttime = mr->LabelDate;

   if (!update_db) {
      return 1;
   }

   if (!db_create_media_record(bjcr, db, mr)) {
      Pmsg1(0, _("Could not create media record. ERR=%s\n"), db_strerror(db));
      return 0;
   }
   if (!db_update_media_record(bjcr, db, mr)) {
      Pmsg1(0, _("Could not update media record. ERR=%s\n"), db_strerror(db));
      return 0;
   }
   if (verbose) {
      Pmsg1(000, _("Created Media record for Volume: %s\n"), mr->VolumeName);
   }
   return 1;

}

/*
 * Called at end of media to update it
 */
static bool update_media_record(B_DB *db, MEDIA_DBR *mr)
{
   if (!update_db && !update_vol_info) {
      return true;
   }

   mr->LastWritten = lasttime;
   if (!db_update_media_record(bjcr, db, mr)) {
      Pmsg1(0, _("Could not update media record. ERR=%s\n"), db_strerror(db));
      return false;;
   }
   if (verbose) {
      Pmsg1(000, _("Updated Media record at end of Volume: %s\n"), mr->VolumeName);
   }
   return true;

}


static int create_pool_record(B_DB *db, POOL_DBR *pr)
{
   pr->NumVols++;
   pr->UseCatalog = 1;
   pr->VolRetention = 355 * 3600 * 24; /* 1 year */

   if (!update_db) {
      return 1;
   }
   if (!db_create_pool_record(bjcr, db, pr)) {
      Pmsg1(0, _("Could not create pool record. ERR=%s\n"), db_strerror(db));
      return 0;
   }
   if (verbose) {
      Pmsg1(000, _("Created Pool record for Pool: %s\n"), pr->Name);
   }
   return 1;

}


/*
 * Called from SOS to create a client for the current Job 
 */
static int create_client_record(B_DB *db, CLIENT_DBR *cr)
{
   if (!update_db) {
      return 1;
   }
   if (!db_create_client_record(bjcr, db, cr)) {
      Pmsg1(0, _("Could not create Client record. ERR=%s\n"), db_strerror(db));
      return 0;
   }
   if (verbose) {
      Pmsg1(000, _("Created Client record for Client: %s\n"), cr->Name);
   }
   return 1;
}

static int create_fileset_record(B_DB *db, FILESET_DBR *fsr)
{
   if (!update_db) {
      return 1;
   }
   fsr->FileSetId = 0;
   if (fsr->MD5[0] == 0) {
      fsr->MD5[0] = ' ';              /* Equivalent to nothing */
      fsr->MD5[1] = 0;
   }
   if (db_get_fileset_record(bjcr, db, fsr)) {
      if (verbose) {
         Pmsg1(000, _("Fileset \"%s\" already exists.\n"), fsr->FileSet);
      }
   } else {
      if (!db_create_fileset_record(bjcr, db, fsr)) {
         Pmsg2(0, _("Could not create FileSet record \"%s\". ERR=%s\n"), 
	    fsr->FileSet, db_strerror(db));
	 return 0;
      }
      if (verbose) {
         Pmsg1(000, _("Created FileSet record \"%s\"\n"), fsr->FileSet);
      }
   }
   return 1;
}

/*
 * Simulate the two calls on the database to create
 *  the Job record and to update it when the Job actually
 *  begins running.
 */
static JCR *create_job_record(B_DB *db, JOB_DBR *jr, SESSION_LABEL *label, 
			     DEV_RECORD *rec)
{
   JCR *mjcr;
   struct date_time dt;
   struct tm tm;

   jr->JobId = label->JobId;
   jr->JobType = label->JobType;
   jr->JobLevel = label->JobLevel;
   jr->JobStatus = JS_Created;
   bstrncpy(jr->Name, label->JobName, sizeof(jr->Name));
   bstrncpy(jr->Job, label->Job, sizeof(jr->Job));
   if (label->VerNum >= 11) {
      jr->SchedTime = btime_to_unix(label->write_btime);
   } else {
      dt.julian_day_number = label->write_date;
      dt.julian_day_fraction = label->write_time;
      tm_decode(&dt, &tm);
      jr->SchedTime = mktime(&tm);
   }

   jr->StartTime = jr->SchedTime;
   jr->JobTDate = (utime_t)jr->SchedTime;
   jr->VolSessionId = rec->VolSessionId;
   jr->VolSessionTime = rec->VolSessionTime;

   /* Now create a JCR as if starting the Job */
   mjcr = create_jcr(jr, rec, label->JobId);

   if (!update_db) {
      return mjcr;
   }

   /* This creates the bare essentials */
   if (!db_create_job_record(bjcr, db, jr)) {
      Pmsg1(0, _("Could not create JobId record. ERR=%s\n"), db_strerror(db));
      return mjcr;
   }

   /* This adds the client, StartTime, JobTDate, ... */
   if (!db_update_job_start_record(bjcr, db, jr)) {
      Pmsg1(0, _("Could not update job start record. ERR=%s\n"), db_strerror(db));
      return mjcr;
   }
   Pmsg2(000, _("Created new JobId=%u record for original JobId=%u\n"), jr->JobId, 
	 label->JobId);
   mjcr->JobId = jr->JobId;	      /* set new JobId */
   return mjcr;
}

/* 
 * Simulate the database call that updates the Job 
 *  at Job termination time.
 */
static int update_job_record(B_DB *db, JOB_DBR *jr, SESSION_LABEL *elabel,
			      DEV_RECORD *rec)
{
   struct date_time dt;
   struct tm tm;
   JCR *mjcr;

   mjcr = get_jcr_by_session(rec->VolSessionId, rec->VolSessionTime);
   if (!mjcr) {
      Pmsg2(000, _("Could not find SessId=%d SessTime=%d for EOS record.\n"),
		   rec->VolSessionId, rec->VolSessionTime);
      return 0;
   }
   if (elabel->VerNum >= 11) {
      jr->EndTime = btime_to_unix(elabel->write_btime);
   } else {
      dt.julian_day_number = elabel->write_date;
      dt.julian_day_fraction = elabel->write_time;
      tm_decode(&dt, &tm);
      jr->EndTime = mktime(&tm);
   }
   lasttime = jr->EndTime;
   mjcr->end_time = jr->EndTime;

   jr->JobId = mjcr->JobId;
   jr->JobStatus = elabel->JobStatus;
   mjcr->JobStatus = elabel->JobStatus;
   jr->JobFiles = elabel->JobFiles;
   jr->JobBytes = elabel->JobBytes;
   jr->VolSessionId = rec->VolSessionId;
   jr->VolSessionTime = rec->VolSessionTime;
   jr->JobTDate = (utime_t)mjcr->start_time;
   jr->ClientId = mjcr->ClientId;

   if (!update_db) {
      free_jcr(mjcr);
      return 1;
   }
   
   if (!db_update_job_end_record(bjcr, db, jr)) {
      Pmsg2(0, _("Could not update JobId=%u record. ERR=%s\n"), jr->JobId,  db_strerror(db));
      free_jcr(mjcr);
      return 0;
   }
   if (verbose) {
      Pmsg2(000, _("Updated Job termination record for JobId=%u TermStat=%c\n"), jr->JobId,
	 jr->JobStatus);
   }
   if (verbose > 1) {
      const char *term_msg;
      static char term_code[70];
      char sdt[50], edt[50];
      char ec1[30], ec2[30], ec3[30];

      switch (mjcr->JobStatus) {
      case JS_Terminated:
         term_msg = _("Backup OK");
	 break;
      case JS_FatalError:
      case JS_ErrorTerminated:
         term_msg = _("*** Backup Error ***");
	 break;
      case JS_Canceled:
         term_msg = _("Backup Canceled");
	 break;
      default:
	 term_msg = term_code;
         sprintf(term_code, _("Job Termination code: %d"), mjcr->JobStatus);
	 break;
      }
      bstrftime(sdt, sizeof(sdt), mjcr->start_time);
      bstrftime(edt, sizeof(edt), mjcr->end_time);
      Pmsg14(000,  _("%s\n\
JobId:                  %d\n\
Job:                    %s\n\
FileSet:                %s\n\
Backup Level:           %s\n\
Client:                 %s\n\
Start time:             %s\n\
End time:               %s\n\
Files Written:          %s\n\
Bytes Written:          %s\n\
Volume Session Id:      %d\n\
Volume Session Time:    %d\n\
Last Volume Bytes:      %s\n\
Termination:            %s\n\n"),
	edt,
	mjcr->JobId,
	mjcr->Job,
	mjcr->fileset_name,
	job_level_to_str(mjcr->JobLevel),
	mjcr->client_name,
	sdt,
	edt,
	edit_uint64_with_commas(mjcr->JobFiles, ec1),
	edit_uint64_with_commas(mjcr->JobBytes, ec2),
	mjcr->VolSessionId,
	mjcr->VolSessionTime,
	edit_uint64_with_commas(mr.VolBytes, ec3),
	term_msg);
   }
   free_jcr(mjcr);
   return 1;
}

static int create_jobmedia_record(B_DB *db, JCR *mjcr)
{
   JOBMEDIA_DBR jmr;
   DCR *dcr = mjcr->dcr;

   if (dev->state & ST_TAPE) {
      dcr->EndBlock = dev->EndBlock;
      dcr->EndFile  = dev->EndFile;
   } else {
      dcr->EndBlock = (uint32_t)dev->file_addr;
      dcr->EndFile = (uint32_t)(dev->file_addr >> 32);
   }

   memset(&jmr, 0, sizeof(jmr));
   jmr.JobId = mjcr->JobId;
   jmr.MediaId = mr.MediaId;
   jmr.FirstIndex = dcr->VolFirstIndex;
   jmr.LastIndex = dcr->FileIndex;
   jmr.StartFile = dcr->StartFile;
   jmr.EndFile = dcr->EndFile;
   jmr.StartBlock = dcr->StartBlock;
   jmr.EndBlock = dcr->EndBlock;


   if (!update_db) {
      return 1;
   }

   if (!db_create_jobmedia_record(bjcr, db, &jmr)) {
      Pmsg1(0, _("Could not create JobMedia record. ERR=%s\n"), db_strerror(db));
      return 0;
   }
   if (verbose) {
      Pmsg2(000, _("Created JobMedia record JobId %d, MediaId %d\n"), 
		jmr.JobId, jmr.MediaId);
   }
   return 1;
}

/* 
 * Simulate the database call that updates the MD5/SHA1 record
 */
static int update_SIG_record(B_DB *db, char *SIGbuf, DEV_RECORD *rec, int type)
{
   JCR *mjcr;

   mjcr = get_jcr_by_session(rec->VolSessionId, rec->VolSessionTime);
   if (!mjcr) {
      if (mr.VolJobs > 0) {
         Pmsg2(000, _("Could not find SessId=%d SessTime=%d for MD5/SHA1 record.\n"),
		      rec->VolSessionId, rec->VolSessionTime);
      } else {
	 ignored_msgs++;
      }
      return 0;
   }

   if (!update_db || mjcr->FileId == 0) {
      free_jcr(mjcr);
      return 1;
   }
   
   if (!db_add_SIG_to_file_record(bjcr, db, mjcr->FileId, SIGbuf, type)) {
      Pmsg1(0, _("Could not add MD5/SHA1 to File record. ERR=%s\n"), db_strerror(db));
      free_jcr(mjcr);
      return 0;
   }
   if (verbose > 1) {
      Pmsg0(000, _("Updated MD5/SHA1 record\n"));
   }
   free_jcr(mjcr);
   return 1;
}


/* 
 * Create a JCR as if we are really starting the job
 */
static JCR *create_jcr(JOB_DBR *jr, DEV_RECORD *rec, uint32_t JobId)
{
   JCR *jobjcr;
   /*
    * Transfer as much as possible to the Job JCR. Most important is
    *	the JobId and the ClientId.
    */
   jobjcr = new_jcr(sizeof(JCR), bscan_free_jcr);
   jobjcr->JobType = jr->JobType;
   jobjcr->JobLevel = jr->JobLevel;
   jobjcr->JobStatus = jr->JobStatus;
   bstrncpy(jobjcr->Job, jr->Job, sizeof(jobjcr->Job));
   jobjcr->JobId = JobId;      /* this is JobId on tape */
   jobjcr->sched_time = jr->SchedTime;
   jobjcr->start_time = jr->StartTime;
   jobjcr->VolSessionId = rec->VolSessionId;
   jobjcr->VolSessionTime = rec->VolSessionTime;
   jobjcr->ClientId = jr->ClientId;
// attach_jcr_to_device(dev, jobjcr);
   new_dcr(jobjcr, dev);
   return jobjcr;
}

/* Dummies to replace askdir.c */
bool	dir_get_volume_info(DCR *dcr, enum get_vol_info_rw  writing) { return 1;}
bool	dir_find_next_appendable_volume(DCR *dcr) { return 1;}
bool	dir_update_volume_info(DCR *dcr, bool relabel) { return 1; }
bool	dir_create_jobmedia_record(DCR *dcr) { return 1; }
bool	dir_ask_sysop_to_create_appendable_volume(DCR *dcr) { return 1; }
bool	dir_update_file_attributes(DCR *dcr, DEV_RECORD *rec) { return 1;}
bool	dir_send_job_status(JCR *jcr) {return 1;}


bool dir_ask_sysop_to_mount_volume(DCR *dcr)
{
   DEVICE *dev = dcr->dev;
   Dmsg0(20, "Enter dir_ask_sysop_to_mount_volume\n");
   /* Close device so user can use autochanger if desired */
   if (dev_cap(dev, CAP_OFFLINEUNMOUNT)) {
      offline_dev(dev);
   }
   force_close_dev(dev);
   fprintf(stderr, "Mount Volume \"%s\" on device %s and press return when ready: ",
	 dcr->VolumeName, dev_name(dev));
   getchar();	
   return true;
}
