/***************************************************************************
 *            bb_util.c
 *
 *  Tue Jul  5 14:55:21 2005
 *  Copyright  2005  Fabio Marzocca
 *  Email	<thesaltydog@gmail.com>
 ****************************************************************************/

#include <config.h>

#include <glib.h>
#include <glib/gstdio.h>
#include <stdlib.h>
#include <sys/stat.h>
#include <string.h>
#include <monetary.h>
#include  <glib/gi18n.h>
#include <glib/gprintf.h>
#include <gtk/gtk.h>
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <libgnomevfs/gnome-vfs-mime-handlers.h>
#include <glibtop/mountlist.h>
#include <glibtop/fsusage.h>
 
#include "baobab.h"
#include "bb_util.h"
#include "tv.h"
#include "callbacks.h"

static int radio_group_get_active(GtkRadioButton * );
static void radio_group_set_active(GtkRadioButton * rb,int btn);
static void set_search_opt(GladeXML* dlg_xml);
static void save_search_opt(GladeXML* dlg_xml);



/* This is an internally used function to find pixmap files. */
static gchar*
find_pixmap_file                       (const gchar     *filename)
{
	return g_build_path(G_DIR_SEPARATOR_S,
			    PIX_DIR,
			    filename,
			    NULL);
}


void
baobab_get_filesystem(baobab_fs *fs)
{
	size_t i;
	glibtop_mountlist mountlist;
	glibtop_mountentry *mountentries;

	memset(fs, 0, sizeof *fs);

	mountentries = glibtop_get_mountlist(&mountlist, FALSE);

	for (i = 0; i < mountlist.number; ++i) {
		glibtop_fsusage fsusage;
		glibtop_get_fsusage(&fsusage, mountentries[i].mountdir);
		/* FIXME: check if not an "excluded" mount point
		   v.1.1.1 changed bavail with bfree) */
		if (!g_slist_find_custom(baobab.bbExcludedDirs,
					 mountentries[i].mountdir,
					 list_find)) {
			fs->total += fsusage.blocks * fsusage.block_size;
			fs->avail += fsusage.bfree * fsusage.block_size;
			fs->used  += (fsusage.blocks - fsusage.bfree) * fsusage.block_size;
		}
	}

	g_free(mountentries);
}


void
set_statusbar(gchar * text)
{
	GtkWidget* sb;
	guint context=1;

	text=g_filename_display_name(text);
	sb = glade_xml_get_widget(baobab.main_xml, "statusbar1");
	gtk_statusbar_pop(GTK_STATUSBAR (sb), context);
	gtk_statusbar_push(GTK_STATUSBAR(sb),context,text);
	while (gtk_events_pending()) { gtk_main_iteration(); }
	g_free(text);
}

/* GtkFileChooser to select a directory to scan
*/
gchar*
dir_select (gboolean SEARCH, GtkWidget* parent)
{
	GtkWidget *dialog;
	GtkWidget * toggle;
	
	dialog = gtk_file_chooser_dialog_new (_("Select a directory"),
				      GTK_WINDOW (parent),
				      GTK_FILE_CHOOSER_ACTION_SELECT_FOLDER,
				      GTK_STOCK_CANCEL, GTK_RESPONSE_CANCEL,
				      GTK_STOCK_OPEN, GTK_RESPONSE_ACCEPT,
				      NULL);

	gtk_file_chooser_set_show_hidden(GTK_FILE_CHOOSER (dialog),TRUE);
	gtk_file_chooser_set_current_folder(GTK_FILE_CHOOSER (dialog), 
								g_get_home_dir ());
	/* add extra widget */
	toggle = gtk_check_button_new_with_label (_("Show hidden directories"));
  	gtk_widget_show (toggle);
	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(toggle),TRUE);
  	g_signal_connect ((gpointer) toggle, "toggled",
                    G_CALLBACK (on_toggled),
                    dialog);
	gtk_file_chooser_set_extra_widget (GTK_FILE_CHOOSER (dialog), toggle);
	
	
	if (gtk_dialog_run (GTK_DIALOG (dialog)) == GTK_RESPONSE_ACCEPT)
	{
		char *filename;
    		filename = gtk_file_chooser_get_uri (GTK_FILE_CHOOSER (dialog));
		gtk_widget_destroy (dialog);
		if (!SEARCH) {
    		start_proc_on_dir(filename);
    		g_free (filename);
			return NULL;
			}
		else {
			return(filename);
		}
  	}
	else {
		gtk_widget_destroy (dialog);
		return NULL;
	}
}


void
on_toggled (GtkToggleButton   *togglebutton, gpointer         dialog)
{
	gtk_file_chooser_set_show_hidden(GTK_FILE_CHOOSER (dialog),
		!gtk_file_chooser_get_show_hidden(GTK_FILE_CHOOSER (dialog)));
}



/* change busy icon
*/

void
change_icon(gchar *filename)
{
	GdkPixbufAnimation* anim;
	GError *err=NULL;
	GString *pathname ;
	GtkWidget * image;
	
	pathname=g_string_new(PIX_DIR);
	g_string_append(pathname,"/");
	g_string_append(pathname,filename);
	image = glade_xml_get_widget(baobab.main_xml, "busyimage");

	anim= gdk_pixbuf_animation_new_from_file (pathname->str, &err);
	if(err!=NULL) {
		g_print("%s\n",err->message);
		g_error_free(err);
	}

	g_string_free (pathname,TRUE);
	gtk_image_set_from_animation(GTK_IMAGE(image),anim); 
	
}


GdkPixbuf * 
set_bar(gfloat perc)
{
	
	GdkPixbuf * srcpxb=NULL, *destpxb;
	gint height,width;
	
	if (perc <= 33.0f ) {
			srcpxb = baobab.green_bar;
		}
	else if (perc > 33.0f && perc <= 60.0f) {
			srcpxb = baobab.yellow_bar;
		}
	else if (perc > 60.0f) {
			srcpxb = baobab.red_bar;
		}
		
	destpxb=baobab_load_pixbuf("white.png");
	height = gdk_pixbuf_get_height(destpxb);
	width = gdk_pixbuf_get_width(destpxb)-1; 
	width = (gint)(gfloat)((width*perc)/100.0f);         /* if bar width ==50 --> $width = $perc/2 */
	
	gdk_pixbuf_copy_area(srcpxb,1,1,width-2,height-2,destpxb,1,1);
                                        
	return destpxb;
}

/* menu & toolbar sensitivity */
void
check_menu_sens(gboolean scanning)
{
	GtkWidget * tbstop;
	GtkWidget * tbscanall;
	GtkWidget * tbscandir;
	GtkWidget * menuallfs;
	GtkWidget * menuscandir;
	GtkWidget * tb_search;

	if (scanning) {

		while (gtk_events_pending()) { gtk_main_iteration();}
		set_statusbar(_("Scanning..."));
		tbstop = glade_xml_get_widget(baobab.main_xml, "tbstop");
		gtk_widget_set_sensitive(tbstop,TRUE);
		tbscanall = glade_xml_get_widget(baobab.main_xml, "tbscanall");
		gtk_widget_set_sensitive(tbscanall,FALSE);
		tbscandir = glade_xml_get_widget(baobab.main_xml, "tbscandir");
		gtk_widget_set_sensitive(tbscandir,FALSE);
		menuallfs = glade_xml_get_widget(baobab.main_xml, "menuallfs");
		gtk_widget_set_sensitive(menuallfs,FALSE);
		menuscandir = glade_xml_get_widget(baobab.main_xml, "menuscandir");
		gtk_widget_set_sensitive(menuscandir,FALSE);
		tb_search = glade_xml_get_widget(baobab.main_xml, "tb_search");
		gtk_widget_set_sensitive(tb_search,FALSE);
		gtk_widget_set_sensitive(glade_xml_get_widget(baobab.main_xml, "preferenze1"),FALSE);
		gtk_widget_set_sensitive(glade_xml_get_widget(baobab.main_xml, "menu_scan_rem"),FALSE);
		gtk_widget_set_sensitive(glade_xml_get_widget(baobab.main_xml, "tb_scan_remote"),FALSE);
		gtk_widget_set_sensitive(glade_xml_get_widget(baobab.main_xml, "search_for_a_file"),FALSE);
		gtk_widget_set_sensitive(glade_xml_get_widget(baobab.main_xml, "tbprops"),FALSE);
		gtk_widget_set_sensitive(glade_xml_get_widget(baobab.main_xml, "ck_allocated"),FALSE);		
		if (get_NB_page()==VIEW_SEARCH) {
					gtk_widget_set_sensitive(glade_xml_get_widget(baobab.main_xml, "label1"),FALSE);
			}
		}
	else {
		tbstop = glade_xml_get_widget(baobab.main_xml, "tbstop");
		gtk_widget_set_sensitive(tbstop,FALSE);
		if (get_NB_page()==VIEW_TREE && baobab.is_local) {
			gtk_widget_set_sensitive(glade_xml_get_widget(baobab.main_xml, "ck_allocated"),TRUE);
			}
		tbscanall = glade_xml_get_widget(baobab.main_xml, "tbscanall");
		gtk_widget_set_sensitive(tbscanall,TRUE);
		tbscandir = glade_xml_get_widget(baobab.main_xml, "tbscandir");
		gtk_widget_set_sensitive(tbscandir,TRUE);
		menuallfs = glade_xml_get_widget(baobab.main_xml, "menuallfs");
		gtk_widget_set_sensitive(menuallfs,TRUE);
		menuscandir = glade_xml_get_widget(baobab.main_xml, "menuscandir");
		gtk_widget_set_sensitive(menuscandir,TRUE);
		tb_search = glade_xml_get_widget(baobab.main_xml, "tb_search");
		gtk_widget_set_sensitive(tb_search,TRUE);
		gtk_widget_set_sensitive(glade_xml_get_widget(baobab.main_xml, "tb_scan_remote"),TRUE);
		gtk_widget_set_sensitive(glade_xml_get_widget(baobab.main_xml, "menu_scan_rem"),TRUE);
		gtk_widget_set_sensitive(glade_xml_get_widget(baobab.main_xml, "search_for_a_file"),TRUE);
		gtk_widget_set_sensitive(glade_xml_get_widget(baobab.main_xml, "label1"),TRUE);
		gtk_widget_set_sensitive(glade_xml_get_widget(baobab.main_xml, "preferenze1"),TRUE);
		gtk_widget_set_sensitive(glade_xml_get_widget(baobab.main_xml, "tbprops"),TRUE);		

			}
}


void
stop_scan(void)
{
	if(get_NB_page()==VIEW_TREE) {
		set_statusbar( _("Calculating percentage bars..."));
		gtk_tree_model_foreach((GtkTreeModel*)baobab.model,show_bars,NULL);
		gtk_tree_view_columns_autosize((GtkTreeView *)baobab.tree_view);
	}
	change_icon("done.png");
	check_menu_sens(FALSE);
	set_statusbar( _("Ready"));	
}


gboolean
show_bars (GtkTreeModel *mdl,
           GtkTreePath *path,
           GtkTreeIter *iter,
           gpointer data)
{
	GtkTreeIter parent;
	gfloat perc;
	gint readelements, size_col;
	guint64 refsize,size;
	gchar textperc[10];
	char* sizecstr=NULL;

	if (baobab.show_allocated)  size_col= (gint)COL_H_ALLOCSIZE;
	else size_col= (gint)COL_H_SIZE;
	
	if(gtk_tree_model_iter_parent (mdl,&parent,iter)) {
		gtk_tree_model_get(mdl,&parent,COL_H_ELEMENTS,&readelements,-1);
		if (readelements==-1) {return TRUE;}
		gtk_tree_model_get(mdl,iter,COL_H_ELEMENTS,&readelements,-1);
		if (readelements==-1) {return TRUE;}
		gtk_tree_model_get(mdl,&parent,size_col,&refsize,-1);
		if (refsize==0) {return FALSE;}		
		gtk_tree_model_get(mdl,iter,size_col,&size,-1);
	    	perc =((gfloat)size*100)/(gfloat)refsize;
		g_sprintf(textperc," %.1f %%",perc);
		sizecstr = gnome_vfs_format_file_size_for_display(size);

		gtk_tree_store_set((GtkTreeStore*)mdl,iter,
					COL_BAR,set_bar(perc),
					COL_DIR_SIZE,sizecstr,
					COL_H_PERC,perc,
					COL_PERC,textperc,
					-1);
		g_free(sizecstr);
		}
	else {
		gtk_tree_model_get(mdl,iter,COL_H_ELEMENTS,&readelements,-1);
		if (readelements !=-1) {
			gtk_tree_model_get(mdl,iter,size_col,&size,-1);
			sizecstr = gnome_vfs_format_file_size_for_display(size);
			gtk_tree_store_set((GtkTreeStore*)mdl,iter,
				COL_BAR,set_bar(100.0),
				COL_H_PERC,100.0,
				COL_PERC,"100 %",
				COL_DIR_SIZE,sizecstr,
				-1);
			g_free(sizecstr);
			}
		}
	return FALSE;	
}



void
message(gchar* messaggio,GtkWidget* parent)
{
	GtkWidget * dialog;
	dialog = gtk_message_dialog_new(GTK_WINDOW(parent),
                                  GTK_DIALOG_DESTROY_WITH_PARENT,
                                  GTK_MESSAGE_INFO,
                                  GTK_BUTTONS_OK,
                                  "%s",messaggio);	
	gtk_message_dialog_set_markup((GtkMessageDialog*)dialog,messaggio);
	gtk_dialog_run (GTK_DIALOG (dialog));
    	gtk_widget_destroy (dialog);
	
}


gint
messageyesno(gchar* messaggio,GtkWidget* parent)
{
	GtkWidget * dialog;
	gint response;
	dialog = gtk_message_dialog_new(GTK_WINDOW(parent),
                                  GTK_DIALOG_DESTROY_WITH_PARENT,
                                  GTK_MESSAGE_QUESTION,
                                  GTK_BUTTONS_YES_NO,
                                  "%s",messaggio);	
	gtk_message_dialog_set_markup((GtkMessageDialog*)dialog,messaggio);
	response=gtk_dialog_run (GTK_DIALOG (dialog));
    	gtk_widget_destroy (dialog);
	return response;
}



gboolean
baobab_check_dir(const gchar * dirname)
{
	char * error_msg = NULL;
	GnomeVFSFileInfo *info;
	GnomeVFSResult result;
	gboolean bret = TRUE;
	
	info = gnome_vfs_file_info_new();
	result=gnome_vfs_get_file_info(dirname, info,GNOME_VFS_FILE_INFO_DEFAULT);

	if (info->type == GNOME_VFS_FILE_TYPE_SYMBOLIC_LINK)
		error_msg = g_strdup_printf(_("%s is a symbolic link"), dirname);
	else if (info->type != GNOME_VFS_FILE_TYPE_DIRECTORY)
		error_msg = g_strdup_printf(_("%s is not a valid directory"), dirname);

	if (error_msg) {
		message(error_msg, baobab.window);
		g_free(error_msg);
		bret = FALSE;
	}

	gnome_vfs_file_info_unref(info);
	return bret;

}


void
popupmenu_list(GtkTreePath *path, GdkEventButton *event)
{
	GtkWidget * pmenu, *open, *trash,*sep,*allfiles, *graph_map;
	GtkTreeIter  iter;
	gchar * path_to_string;
	GtkWidget * image;

	if (baobab.selected_path) {
		g_free(baobab.selected_path);
		baobab.selected_path = NULL;
	}

	gtk_tree_model_get_iter(GTK_TREE_MODEL(baobab.model), &iter, path);
	gtk_tree_model_get(GTK_TREE_MODEL(baobab.model), &iter,
			   COL_H_FULLPATH, &baobab.selected_path,
			   -1);

	if (strcmp(baobab.selected_path, "") == 0) 
		return;
		
	/* path_to_string is freed in callback function */
	path_to_string= gtk_tree_path_to_string(path);

	pmenu = gtk_menu_new();
	open = gtk_image_menu_item_new_from_stock("gtk-open", NULL);
	trash = gtk_image_menu_item_new_from_stock("gtk-delete", NULL);
	graph_map=gtk_image_menu_item_new_with_label(_("Folder graphical map"));
	image=gtk_image_new_from_stock("gtk-select-color",GTK_ICON_SIZE_MENU);
	gtk_image_menu_item_set_image (GTK_IMAGE_MENU_ITEM(graph_map),image);
	sep = gtk_separator_menu_item_new();
	allfiles = gtk_image_menu_item_new_with_label(_("List all files in folder"));
	image = gtk_image_new_from_pixbuf(baobab_load_pixbuf("searchall.png"));
	gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(allfiles), image);


	g_signal_connect(open, "activate",
			 G_CALLBACK(open_file_cb),
			 NULL);

	g_signal_connect(trash, "activate",
			 G_CALLBACK(trash_dir_cb),
			 NULL);

	g_signal_connect(allfiles, "activate",
			 G_CALLBACK(list_all_cb),
			 NULL);
			 
	g_signal_connect(graph_map,"activate",
			 G_CALLBACK (graph_map_cb),
			 path_to_string);

	
	gtk_container_add(GTK_CONTAINER(pmenu), open);
	gtk_container_add(GTK_CONTAINER(pmenu), allfiles);
	gtk_container_add(GTK_CONTAINER(pmenu), graph_map);
	gtk_container_add(GTK_CONTAINER(pmenu), sep);
	gtk_container_add(GTK_CONTAINER(pmenu), trash);
	gtk_widget_show_all(pmenu);
	gtk_menu_popup(GTK_MENU(pmenu), NULL, NULL, NULL, NULL,
		       event->button, event->time);
	
}


void
popupmenu_list_search(GtkTreePath *path, GdkEventButton *event)
{
	GtkWidget *pmenu, *trash, *open, *sep;
	GtkTreeIter iter;

	if (baobab.selected_path) {
		g_free(baobab.selected_path);
		baobab.selected_path = NULL;
	}

	gtk_tree_model_get_iter(GTK_TREE_MODEL(baobab.model_search), &iter, path);
	gtk_tree_model_get(GTK_TREE_MODEL(baobab.model_search), &iter,
			   COL_FULLPATH, &baobab.selected_path,
			   -1);


	if (strcmp(baobab.selected_path, "") == 0) 
		return;
		
	pmenu = gtk_menu_new();
	open = gtk_image_menu_item_new_from_stock("gtk-open",NULL);
	trash = gtk_image_menu_item_new_from_stock("gtk-delete",NULL);
	sep = gtk_separator_menu_item_new();


	g_signal_connect(open, "activate",
			 G_CALLBACK(open_file_cb),
			 NULL);

	g_signal_connect(trash, "activate",
			 G_CALLBACK(trash_file_cb),
			 NULL);

	
	gtk_container_add(GTK_CONTAINER(pmenu), open);
	gtk_widget_show(open);
	gtk_container_add(GTK_CONTAINER(pmenu),sep);
	gtk_widget_show(sep);
	gtk_container_add(GTK_CONTAINER(pmenu),trash);
	gtk_widget_show(trash);
	gtk_menu_popup(GTK_MENU(pmenu), NULL, NULL, NULL, NULL,
		       event->button, event->time);
	
}




/* search dialog */
void
dialog_search(void)
{
	GString * search;
	GtkWidget* dlg ;
	GladeXML *dlg_xml;
	GtkWidget * rb_allfs, *btn_select;
	
	/* Glade stuff */
	dlg_xml = glade_xml_new(baobab.glade_path,"dlg_search",NULL);
	glade_xml_signal_autoconnect(dlg_xml);
	dlg=glade_xml_get_widget(dlg_xml,"dlg_search");
	
	rb_allfs = glade_xml_get_widget(dlg_xml,"radio_allfs");
	btn_select = glade_xml_get_widget(dlg_xml,"btn_select_search");
	
	gtk_window_set_transient_for(GTK_WINDOW(dlg),GTK_WINDOW(baobab.window));
	gtk_window_set_icon(GTK_WINDOW(dlg),baobab_get_icon());
	set_search_opt(dlg_xml);
			
	/* connect signals */
	g_signal_connect(rb_allfs, "clicked",
					 G_CALLBACK(on_radio_allfs_clicked),NULL);
	
	g_signal_connect(btn_select, "clicked",
					 G_CALLBACK(on_btn_select_search_clicked),NULL);
					 
	
	while (gtk_dialog_run (GTK_DIALOG (dlg)) != GTK_RESPONSE_CANCEL)
	{			
		save_search_opt(dlg_xml);
		if (strcmp(bbSearchOpt.filename->str,"")==0) {
			message(_("Please provide a file name to search for!"),dlg);
		}
		else {	
			search = g_string_new(bbSearchOpt.filename->str);
   			gtk_widget_destroy (dlg);
			if (!bbSearchOpt.exact) {
				g_string_prepend(search,"*");
				g_string_append(search,"*");	
				}
			if (bbSearchOpt.search_whole){
				start_search(search->str,NULL);
				}
			else {
				start_search(search->str,bbSearchOpt.dir->str);
				}
			g_string_free(search,TRUE);
			return;	/* exits from while loop */
		}
    }
	
	g_object_unref(dlg_xml);	
	gtk_widget_destroy (dlg);
	
}

void
save_search_opt(GladeXML* dlg_xml)
{
	GtkEntry * entry_searchname;
	GtkWidget* rb_exact;
	GtkEntry * entry_sel_dir;
	GtkWidget * rb_allfs;

	entry_searchname = (GtkEntry*)glade_xml_get_widget(dlg_xml,"entry1");
	g_string_assign(bbSearchOpt.filename,gtk_entry_get_text(GTK_ENTRY(entry_searchname)));
	rb_exact = glade_xml_get_widget(dlg_xml,"radio_exact");
	bbSearchOpt.exact = gtk_toggle_button_get_active((GtkToggleButton*)rb_exact);
	entry_sel_dir = (GtkEntry*)glade_xml_get_widget(dlg_xml,"entry2");
	g_string_assign(bbSearchOpt.dir,gtk_entry_get_text(GTK_ENTRY(entry_sel_dir)));
	rb_allfs = glade_xml_get_widget(dlg_xml,"radio_allfs");
	bbSearchOpt.search_whole = gtk_toggle_button_get_active((GtkToggleButton*)rb_allfs);
	bbSearchOpt.mod_date = radio_group_get_active(
						(GtkRadioButton*)glade_xml_get_widget(dlg_xml,"radio_mod_unk"));
	bbSearchOpt.size_limit = radio_group_get_active(
						(GtkRadioButton*)glade_xml_get_widget(dlg_xml,"radio_size_unk"));
	
}


void
set_search_opt(GladeXML* dlg_xml)
{
	GtkEntry * entry_searchname;
	GtkWidget* rb_exact;
	GtkEntry * entry_sel_dir;
	GtkWidget * rb_allfs;
	GtkWidget* expander;
	
	entry_searchname = (GtkEntry*)glade_xml_get_widget(dlg_xml,"entry1");
	gtk_entry_set_text(entry_searchname,bbSearchOpt.filename->str);
	rb_exact = glade_xml_get_widget(dlg_xml,"radio_exact");
	gtk_toggle_button_set_active((GtkToggleButton*)rb_exact,bbSearchOpt.exact);
	entry_sel_dir = (GtkEntry*)glade_xml_get_widget(dlg_xml,"entry2");
	gtk_entry_set_text(entry_sel_dir,bbSearchOpt.dir->str);
	rb_allfs = glade_xml_get_widget(dlg_xml,"radio_allfs");
	gtk_toggle_button_set_active((GtkToggleButton*)rb_allfs,bbSearchOpt.search_whole);
	
	radio_group_set_active((GtkRadioButton*)glade_xml_get_widget(dlg_xml,"radio_mod_unk"),
							bbSearchOpt.mod_date);

	radio_group_set_active((GtkRadioButton*)glade_xml_get_widget(dlg_xml,"radio_size_unk"),
							bbSearchOpt.size_limit);
	expander = glade_xml_get_widget(dlg_xml,"expander1");
	gtk_expander_set_expanded((GtkExpander*)expander, 
							  (bbSearchOpt.mod_date != NONE || 
							   bbSearchOpt.size_limit != NONE));
}

void
switch_view(gint view)
{
	GtkWidget * nb;
	nb = glade_xml_get_widget(baobab.main_xml,"notebook1");
	gtk_notebook_set_current_page(GTK_NOTEBOOK(nb), view);
	
}

gint
get_NB_page(void) {
	GtkWidget * nb;
	nb = glade_xml_get_widget(baobab.main_xml,"notebook1");
	return (gtk_notebook_get_current_page(GTK_NOTEBOOK(nb))); 
}

int
radio_group_get_active(GtkRadioButton * rb)
{
	GSList *list, *l;
	int i;
	
	list = gtk_radio_button_get_group (rb);
	i = 0;
	for (l = list; l; l = l->next) {
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (l->data)))
	    break;
		i++;
	 }
	i = g_slist_length (list) - i;
	
	 return i;
}


void
radio_group_set_active(GtkRadioButton * rb,int btn)
{
	GSList *list, *l;
	int i;
	
	list = gtk_radio_button_get_group (rb);
	btn = g_slist_length (list) - btn;
	i=0;
	for (l = list; l; l = l->next) {
		if (i==btn) {
	    	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(l->data),TRUE);
			break;
			}
		i++;
	 	}
}


void set_label_scan (baobab_fs *fs)
{
	gchar *markup, *size;
	GString * total;
	GString * used;
	GString * avail;

	if (baobab.label_scan) {
			g_free(baobab.label_scan);
	}
	
	size = gnome_vfs_format_file_size_for_display(fs->total);
	total = g_string_new(size);
	g_free(size);
	size = gnome_vfs_format_file_size_for_display(fs->used);
	used = g_string_new(size);
	g_free(size);

	size = gnome_vfs_format_file_size_for_display(fs->avail);
	avail = g_string_new(size);
	g_free(size);
 	

		markup = g_markup_printf_escaped ("<small>%s <b>%s</b> (%s %s %s %s )</small>", 
								_("Total filesystem capacity:"),
								total->str,
								_("used:"),
								used->str,
								_("available:"),
								avail->str);
	baobab.label_scan = strdup(markup);
	
	g_string_free(total,TRUE);
	g_string_free(used,TRUE);
	g_string_free(avail,TRUE);
	g_free (markup);
	
}


void
show_label(gint view)
{
	GtkWidget* label = glade_xml_get_widget(baobab.main_xml, "label1");

	if (view == VIEW_TREE) {
		gtk_label_set_markup(GTK_LABEL(label), baobab.label_scan);
	}
	else if (view == VIEW_SEARCH) {
		gtk_label_set_markup(GTK_LABEL(label), baobab.label_search);
	}
	else {
		g_assert_not_reached();
	}
}


void 
set_label_search(gint numfiles,guint64 totalsize)
{
	gchar *markup, *size;
	if (baobab.label_search) {
			g_free(baobab.label_search);
	}
	
	size = gnome_vfs_format_file_size_for_display(totalsize);
	markup = g_markup_printf_escaped ("<small>%s <b>%d</b> %s %s <b>%s</b></small>", 
								_("Found:"),
								numfiles,
								(numfiles==1 ? _("file") : _("files")),
								_("for total:"),
								size);
	baobab.label_search = strdup(markup);
	g_free(markup);
	g_free(size);
}

void
open_file_with_application(gchar* file)
{
	GnomeVFSMimeApplication * application;
	gchar * mime;
	GnomeVFSFileInfo *info;
	GnomeVFSURI *vfs_uri;
	
	info = gnome_vfs_file_info_new();
	vfs_uri = gnome_vfs_uri_new (file);
	
	gnome_vfs_get_file_info(file, info,GNOME_VFS_FILE_INFO_GET_MIME_TYPE);
	mime=info->mime_type;

	application = gnome_vfs_mime_get_default_application (mime);

	if (!application) {
		if ((g_file_test (file, G_FILE_TEST_IS_EXECUTABLE)) &&
	    	(g_ascii_strcasecmp (mime, "application/x-executable") == 0) &&
	    	gnome_vfs_uri_is_local(vfs_uri)) { 
			g_spawn_command_line_async (file, NULL);
			}
		else {
		message( _("There is no installed viewer capable "
					"of displaying the document."),baobab.window);
		}
	}
	else {
	GList *uris = NULL;
	gchar *uri;
		uri = gnome_vfs_uri_to_string(vfs_uri,GNOME_VFS_URI_HIDE_NONE);
		uris = g_list_append (uris, uri);
		gnome_vfs_mime_application_launch (application, uris);	
	
		g_list_free (uris);
		g_free (uri);		
	}
	gnome_vfs_uri_unref(vfs_uri);
	gnome_vfs_mime_application_free (application);
	gnome_vfs_file_info_unref(info);
}


gchar *
get_trash_path (const gchar * file)
{
	GnomeVFSURI * trash_uri;
	GnomeVFSURI * uri;
	gchar       * filename;

	filename = gnome_vfs_escape_path_string (file);
	uri = gnome_vfs_uri_new (file);
	g_free (filename);

	gnome_vfs_find_directory (uri,
				  GNOME_VFS_DIRECTORY_KIND_TRASH,
				  &trash_uri,
				  TRUE,
				  TRUE,
				  0777);
	gnome_vfs_uri_unref (uri);

	if (trash_uri == NULL) {
		return NULL;
	}
	else {
		gchar * trash_path;
		trash_path = gnome_vfs_uri_to_string (trash_uri, GNOME_VFS_URI_HIDE_TOPLEVEL_METHOD);
		gnome_vfs_uri_unref (trash_uri);
		return trash_path;
	}
}


gboolean
trash_file(const gchar*  filename)
{
	gchar * trash_path, * basename, *destination;
	GnomeVFSResult result;
	
	
	trash_path = get_trash_path (filename);
	
	if (trash_path == NULL) {
			message(_("Cannot find the Trash on this system!"),baobab.window);
			g_free (trash_path);
			return FALSE;
		}
		
	if ((!g_file_test (filename, G_FILE_TEST_EXISTS)) &&
		    (!g_file_test (filename, G_FILE_TEST_IS_SYMLINK))) {
				message (_("The document does not exist."),baobab.window);
				g_free (trash_path);
				return FALSE;
			}
			
	basename = g_path_get_basename(filename);
	destination = g_build_filename (trash_path, basename, NULL);
				
	result = gnome_vfs_move (filename, destination, TRUE);
	if (result != GNOME_VFS_OK) {
		gchar * mess;

		mess = g_strdup_printf (_("Moving <b>%s</b> to trash failed: %s."),
				                           g_path_get_basename (filename),
				                           gnome_vfs_result_to_string (result));
		message (mess,baobab.window);
		g_free (mess);
		g_free(basename);			
		g_free (trash_path);
		g_free(destination);
		return FALSE;	
		}
	if (strcmp(destination,filename)==0) {
		gchar* mess;
		gint response;
		
		mess = g_strdup_printf (_("Do you want to delete <b>%s</b> permanently?"),
	                           g_path_get_basename (filename));
		response = messageyesno(mess,baobab.window);
		g_free(mess);
		if (response == GTK_RESPONSE_YES) {
			GnomeVFSResult result;
			if (!g_file_test (filename, G_FILE_TEST_IS_DIR)) {
					result=gnome_vfs_unlink (filename);
				}
				else {
					result=gnome_vfs_remove_directory (filename);
				}
				
			if (result != GNOME_VFS_OK) {
			gchar * mess;

			mess = g_strdup_printf (_("Deleting <b>%s</b> failed: %s."),
				                           g_path_get_basename (filename),
				                           gnome_vfs_result_to_string (result));
			message (mess,baobab.window);
			g_free (mess);
			g_free(basename);			
			g_free (trash_path);
			g_free(destination);
			return FALSE;	
				}
			}
		else {
			g_free(basename);			
			g_free (trash_path);
			g_free(destination);
			return FALSE;
			}
		}
		
	
	
	g_free(basename);			
	g_free (trash_path);
	g_free(destination);
	return TRUE;
}

void
contents_changed(void)
{
	baobab_get_filesystem(&g_fs);
	set_label_scan(&g_fs);
	show_label(VIEW_TREE);

	if (messageyesno(_("The content of your home directory has changed.\n"
						"Do you want to rescan the last tree to update the folder branch details?"),
							baobab.window) == GTK_RESPONSE_YES) {
			
				start_proc_on_dir(baobab.last_scan_command->str);
			
		}
	

}

GdkPixbuf*
baobab_load_pixbuf(const gchar *filename)
{
  gchar *pathname = NULL;
  GdkPixbuf *pixbuf;
  GError *error = NULL;

  if (!filename || !filename[0])
      return NULL;

  pathname = find_pixmap_file (filename);

  if (!pathname)
    {
      g_warning (_("Couldn't find pixmap file: %s"), filename);
      return NULL;
    }

  pixbuf = gdk_pixbuf_new_from_file (pathname, &error);
  if (!pixbuf)
    {
      fprintf (stderr, "Failed to load pixbuf file: %s: %s\n",
               pathname, error->message);
      g_error_free (error);
    }
  g_free (pathname);
  return pixbuf;
}


GdkPixbuf * baobab_get_icon(void)
{
	GdkPixbuf * buf = gtk_icon_theme_load_icon(gtk_icon_theme_get_default(),
						   "baobab.png",
						   48,
						   0,
						   NULL);

	if (!buf) {
		GError *error = NULL;
		char *path;

		path = g_build_path(G_DIR_SEPARATOR_S,
				    BAOBAB_DATA_DIR,
				    "pixmaps",
				    "baobab.png",
				    NULL);

		buf = gdk_pixbuf_new_from_file(path, &error);

		if (!buf)
		{
			g_warning("Failed to load pixbuf file: %s: %s\n",
				  path,
				  error->message);
			g_error_free(error);
		}

		g_free(path);
	}

	return buf;
}
