/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastient.laout@tuxfamily.org                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef BASKET_H
#define BASKET_H

#include <qwidget.h>
#include <qclipboard.h>
#include <qptrlist.h>

class QFrame;
class QVBoxLayout;
class QString;
class QColor;
class QPixmap;
class QAction;

class QDomDocument;
class QDomElement;

class Item;
class addItemDialog;

/**Basket that contain some Items.
  *@author Sbastien Laot
  */
class Basket : public QWidget
{
  Q_OBJECT
  public:
  friend class addItemDialog;
	/** Construtor and destructor */
	Basket(QWidget *parent = 0, const char *name = 0, WFlags fl = 0);
	~Basket();
  private:
	void initActions(const char *name);
  public:
	/** Basket properties */
	inline QString icon()                   { return m_icon;                   } // Properties ::
	inline bool    isCheckable()            { return m_checkable;              }
	inline bool    isShowAnnotations()      { return m_showAnnotations;        }
	inline int     hAlign()                 { return m_hAlign;                 }
	inline int     vAlign()                 { return m_vAlign;                 }
	inline bool    isLocked()               { return m_isLocked;               }
	inline bool    useColor()               { return m_useColor;               } // Colors ::
	inline QColor  color()                  { return m_color;                  }
	inline bool    useAltColor()            { return m_useAltColor;            }
	inline QColor  altColor()               { return m_altColor;               }
	inline int     newItemPlacement()       { return m_newItemPlacement;       } // Add item policy ::
	inline bool    allowFreeDrop()          { return m_allowFreeDrop;          }
	inline bool    pasteSelAtCursorPos()    { return m_pasteSelAtCursorPos;    }
	inline int     systrayStackFrom()       { return m_systrayStackFrom;       }
	inline bool    deleteAfterSystrayDrag() { return m_deleteAfterSystrayDrag; }
	void setName(const char *name);
	void setIcon(const QString &path);
	void setCheckable(bool checkable);
	void setShowAnnotations(bool show);
	void setAlign(int hAlign, int vAlign);
	void setLocked(bool lock);
	void setColor(bool useColor, const QColor &color, bool useAltColor, const QColor &altColor);
	void setNewItemPlacement(int newItemPlacement)        { m_insertTo = m_newItemPlacement = newItemPlacement; }
	void setAllowFreeDrop(bool allowFreeDrop)             { m_allowFreeDrop          = allowFreeDrop;       }
	void setPasteSelAtCursorPos(bool pasteSelAtCursorPos) { m_pasteSelAtCursorPos    = pasteSelAtCursorPos; }
	void setSystrayStackFrom(int systrayStackFrom)        { m_systrayStackFrom       = systrayStackFrom;    }
	void setDeleteAfterSystrayDrag(bool enable)           { m_deleteAfterSystrayDrag = enable;              }
	/** Save work */
	inline QString savePath()            { return m_savePath;            }
	void setSavePath(const QString &path = QString()); // Set the basket save path, and :
	                                                   //  LOAD the path content
	                                                   //  SAVE to the path if it doesn't exist
	                                                   //  CHOOSE automatically a path if isNull and SAVE to it
	                                                   //  Must be run only after creation (bad conception : arrange that later)
	void chooseNewSavePath(); // Choose automatically a new path
	/** XML save & load work */
	static QDomElement getXmlElement(const QDomElement &startElement, const QString &elementPath);
	static QString getXmlElementText(const QDomElement &startElement, const QString &elementPath, const QString &defaultTxt = "");
	static bool    trueOrFalse(const QString &value, bool defaultValue = true);
	static QString trueOrFalse(bool value);
	void deleteFiles();
  public: // For redirected event !!
	virtual void mousePressEvent(QMouseEvent *qme);
  protected:
	virtual void contextMenuEvent(QContextMenuEvent *e);
  public: // FIXME: If Item doesn't accept event, it call Basket events !
	/** Drag and drop functions */
	virtual void dragEnterEvent(QDragEnterEvent *event);
	virtual void dragLeaveEvent(QDragLeaveEvent*);
	virtual void dragMoveEvent(QDragMoveEvent*);
	virtual void dropEvent(QDropEvent *event);
  public:
	bool  isEmpty()           { return  m_itemsList.isEmpty();                         }
	bool  canDragItem()       { return (m_systrayStackFrom && !m_itemsList.isEmpty()); }
	Item* currentStackItem();
	void  dragStackItem();
	void  dragNotice(QWidget *item, int offset);
	Item* lastInsertedItem();
  signals:
	void nameChanged(Basket *basket, const QString &name);
	void iconChanged(Basket *basket, const QString &icon);
  public slots:
	void linkLookChanged();
	/** Items manipulation */
	void addItemDialog();
	void delItem(Item *item);
	void pasteItem(QClipboard::Mode mode = QClipboard::Clipboard);
	void recolorizeItems();
	/** Save & load work */
	void saveItemsContent();
	void save();
	void load();
	void loadItems(const QDomElement &items);
	/** Insert item in this basket */
	void insertItem(Item *item);

  private:
	QPtrList<Item>  m_itemsList;

	QVBoxLayout    *m_layout;
	//QSpacerItem  *m_vSpacer;

	/** Basket properties */
	QString  m_icon;
	bool     m_checkable;
	bool     m_showAnnotations;
	int      m_hAlign;
	int      m_vAlign;
	bool     m_isLocked;
	QString  m_savePath;
	// Add item policy ::
	int      m_newItemPlacement;
	bool     m_allowFreeDrop;
	bool     m_pasteSelAtCursorPos;
	int      m_systrayStackFrom;
	bool     m_deleteAfterSystrayDrag;
	// Colors ::
	bool     m_useColor;
	QColor   m_color;
	bool     m_useAltColor;
	QColor   m_altColor;

  public:
	int      m_insertTo;
	void resetInsertTo() { m_insertTo = m_newItemPlacement; }
  private:
	QFrame  *m_frameInsertTo;
	Item    *m_lastInsertedItem;

	int      m_contentOnClickAction;
	int      m_fileOnClickAction;
	bool     m_runOverrideClick;
  public:
	int  contentOnClickAction() { return m_contentOnClickAction; }
	int  fileOnClickAction()    { return m_fileOnClickAction;    }
	bool isRunOverrideClick()   { return m_runOverrideClick;     }
	void setContentOnClickAction(int action) { m_contentOnClickAction = action; }
	void setFileOnClickAction(int action)    { m_fileOnClickAction    = action; }
	void setRunOverrideClick(bool over)      { m_runOverrideClick     = over;   }
	QAction *actAddItem;
	QAction *actPaste;
	QAction *actProp;
	QAction *actLock;
};

#endif // BASKET_H
