/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastient.laout@tuxfamily.org                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qtabwidget.h>
#include <qlayout.h>
#include <qtooltip.h>

#include <qpoint.h>
#include <qpixmap.h>
#include <qinputdialog.h>
#include <kpopupmenu.h>
#include <kiconloader.h>
#include <qiconset.h>
#include <kaction.h>
#include <kapp.h>
#include <klocale.h>

#include <kstandarddirs.h>
#include <qdir.h>
#include <qstringlist.h>
#include <kmessagebox.h>
#include <qtimer.h>
#include <qaction.h>
#include <kstdaccel.h>
#include <kglobalaccel.h>
#include <kkeydialog.h>
#include <kpassivepopup.h>

#include <kmessagebox.h>

#include "container.h"
#include "basket.h"
#include "basketproperties.h"
#include "item.h"
#include "appsettings.h"
#include "debugwindow.h"

/** ContainerSystemTray */

ContainerSystemTray::ContainerSystemTray(QWidget *parent, const char *name)
 : KSystemTray(parent, name), m_parentContainer((Container*)parent)
{
	setAcceptDrops(true);
	m_showTimer = new QTimer(this);
	connect( m_showTimer, SIGNAL(timeout()), m_parentContainer, SLOT(setActive()) );

	updateToolTip();
}

ContainerSystemTray::~ContainerSystemTray()
{
}

void ContainerSystemTray::mousePressEvent(QMouseEvent *event)
{
	if (event->button() & Qt::LeftButton) {          // Prepare drag
		m_pressPos = event->globalPos();
		m_canDrag  = true;
		event->accept();
	} else if (event->button() & Qt::MidButton) {    // Paste
		m_parentContainer->currentBasket()->pasteItem(QClipboard::Selection);
		if (AppSettings::usePassivePopup())
			AppSettings::mainContainer->showPassiveDropped(i18n("Pasted selection (%1) to basket <i>%2</i>"));
		event->accept();
	} else if (event->button() & Qt::RightButton) { // Popup menu
		KIconLoader icol;
		QIconSet    icos;
		QPixmap     ico;

		KPopupMenu menu(this);
		menu.insertTitle( icol.loadIcon("basket", KIcon::NoGroup, KIcon::SizeSmall), "BasKet" );

		QPtrList<Basket> baskets = m_parentContainer->listBaskets();
		Basket *cur;
		for (uint i = 0; i < baskets.count(); ++i ) {
			cur = baskets.at(i);
			if ( ! cur->icon().isEmpty() ) {
				icos = QIconSet( icol.loadIcon( cur->icon(), KIcon::NoGroup, KIcon::SizeSmall ) );
				menu.insertItem(icos, cur->name(), i + 10);
			} else
				menu.insertItem(      cur->name(), i + 10);
		}
		menu.setItemChecked( m_parentContainer->currentBasketIndex() + 10, true );
		connect( &menu, SIGNAL(activated(int)), this, SLOT(showBasket(int)) );
		menu.insertSeparator();

		m_parentContainer->actNewBasket->addTo(&menu);
		menu.insertSeparator();
		m_parentContainer->actAppConfig->addTo(&menu);
		m_parentContainer->actConfigGlobalShortcuts->addTo(&menu);
		m_parentContainer->actQuit->addTo(&menu);

		menu.exec(event->globalPos());
		event->accept();
	} else
		event->ignore();
}

void ContainerSystemTray::mouseMoveEvent(QMouseEvent *event)
{
	Basket *current = m_parentContainer->currentBasket();
	if ( m_canDrag && current->canDragItem() &&
	     (m_pressPos - event->globalPos()).manhattanLength() > KApplication::startDragDistance() ) {
		current->dragStackItem();
		if (current->deleteAfterSystrayDrag())
			current->delItem(current->currentStackItem());
		event->accept();
	} else
		event->ignore();
}

void ContainerSystemTray::mouseReleaseEvent(QMouseEvent *event)
{
	m_canDrag = false;

	if (event->button() == Qt::LeftButton)         // Show / hide main window
		if ( rect().contains(event->pos()) ) {     // Accept only if released in systemTray
			m_parentContainer->changeActive();
			event->accept();
		} else
			event->ignore();
}

void ContainerSystemTray::dragEnterEvent(QDragEnterEvent *event)
{
	if (AppSettings::dropTimeToShow() == 0) // Directly show main Window
		m_parentContainer->setActive(true);
	else if ( ! m_showTimer->isActive() )   // If timer not already started, start it
		m_showTimer->start( AppSettings::dropTimeToShow() * 100, true);

	event->accept();
}

void ContainerSystemTray::dragLeaveEvent(QDragLeaveEvent*)
{
	m_canDrag = false;
	m_showTimer->stop();
}

void ContainerSystemTray::dropEvent(QDropEvent *event)
{
	m_showTimer->stop();
	m_parentContainer->currentBasket()->dropEvent(event);

	if (AppSettings::usePassivePopup())
		AppSettings::mainContainer->showPassiveDropped(i18n("Dropped (%1) to basket <i>%2</i>"));

}

void ContainerSystemTray::showBasket(int index)
{
	if (index >= 10)
		m_parentContainer->setCurrentBasket(index - 10);
}

void ContainerSystemTray::updateToolTip()
{
	if ( ! AppSettings::showSystrayToolTip() ) {
		QToolTip::remove(this);
		return;
	}

	Basket *basket = m_parentContainer->currentBasket();
	Item   *item   = basket->currentStackItem();

	bool useHtml;
	if (item)
		useHtml = (item->type() == Item::Html) && !(item->isShowSource()); // Use HTML in tooltip
	else
		useHtml = false;
	QString tip  = (useHtml ? "<qt>" : ""  ); // Interprete HTML in others lines than the first
	QString endL = (useHtml ? "<br>" : "\n"); // Use correct endline with the format
	QString endT = (useHtml ? (item->html().startsWith("<html>") ? "" : "<br>") // End title
	                                 : "\n"); // Because "<html>" will cause a new line return (with "<br>" : two !!)
		// FIXME: endT is useless : create itemHtml->body(); For enhanced work !

	tip += i18n("Current basket : %1").arg(basket->name());

	if (basket->canDragItem()) {
		tip += endL + i18n("%1 item to drag (%2)")
		       .arg( basket->systrayStackFrom() == 1 ? i18n("First") : i18n("Last") )
		       .arg( item->contentTypeName() );
		if (item->type() != Item::Image) {
			tip += " :" + endT;
			tip += item->toString();
		}
	} else if (basket->isEmpty())
		tip += endL + i18n("(Empty)");

	QToolTip::add(this, tip);
//	*AppSettings::debugWindow << "<b>Info :</b>\tSystem tray icon tooltip updated : " + tip;
}

void ContainerSystemTray::wheelEvent(QWheelEvent *event)
{
	m_parentContainer->wheelEvent(event);
}

/** ContainerTabBar */

ContainerTabBar::ContainerTabBar(QWidget *parent, const char *name)
 : QTabBar(parent, name)
{
	setAcceptDrops(true);
	m_showTimer = new QTimer(this);
	connect( m_showTimer, SIGNAL(timeout()), this, SLOT(timeout()) );
}

ContainerTabBar::~ContainerTabBar()
{
}

void ContainerTabBar::contextMenuEvent(QContextMenuEvent *e)
{
	QTab *tab;

	if (e->reason() == QContextMenuEvent::Mouse)     // With mouse : find which tab is under the mouse
		for ( int i = 0; i < count(); ++i ) {
			tab = tabAt(i);
			if (tab->rect().contains(e->pos())) {
				emit wantContextMenu(i, e->globalPos());
				break;
			}
		}
	else                                             // With keyboard : use the current tab
		emit wantContextMenu( indexOf(currentTab()), // Show popup menu on middle of the current tab
		                      mapToGlobal(tabAt(indexOf(currentTab()))->rect().center()) );
}

void ContainerTabBar::dragEnterEvent(QDragEnterEvent *event)
{
	QTab *tab;
	bool isHoverTab = false; // If a tab exist under cursor (to accept event or not)

	for ( int i = 0; i < count(); ++i ) {           // Show the tab under the mouse
		tab = tabAt(i);
		if (tab->rect().contains(event->pos())) {
			isHoverTab = true;
			if (AppSettings::dropTimeToShow() == 0) // Directly show main Window
				emit wantShowPage(i);
			else if ( ! m_showTimer->isActive() ) { // If timer not already started, start it
				m_showTimer->start( AppSettings::dropTimeToShow() * 100, true);
				m_basketToShow = i;
			}
			break;
		}
	}

	event->accept(isHoverTab);
}

void ContainerTabBar::dragMoveEvent(QDragMoveEvent *event) // FIXME: Duplicate code (3 times)
// TODO: DragLeaveEvent : showPage(theLastShownBefore_dragEnterEvent);
{
	QTab *tab;
	bool isHoverTab = false; // If a tab exist under cursor

	for ( int i = 0; i < count(); ++i ) {
		tab = tabAt(i);
		if (tab->rect().contains(event->pos())) {
			isHoverTab = true;
			if (i != m_basketToShow) { // If mouse point a new basket, restart timer for this basket
				m_showTimer->stop();
				m_showTimer->start( AppSettings::dropTimeToShow() * 100, true);
				m_basketToShow = i;
			}
			break;
		}
	}

	event->accept(isHoverTab);
}

void ContainerTabBar::dragLeaveEvent(QDragLeaveEvent*)
{
	m_showTimer->stop();
}

void ContainerTabBar::dropEvent(QDropEvent *event)
{
	m_showTimer->stop();

	AppSettings::mainContainer->basketAt(m_basketToShow)->dropEvent(event);
}

void ContainerTabBar::timeout()
{
	emit wantShowPage(m_basketToShow);
}

void ContainerTabBar::mousePressEvent(QMouseEvent *event)
{
	QTabBar::mousePressEvent(event);

	if (event->button() & Qt::MidButton) {
		QTab *tab;

		for ( int i = 0; i < count(); ++i ) {
			tab = tabAt(i);
			if (tab->rect().contains(event->pos())) {
				emit wantPaste(i, QClipboard::Selection);
				event->accept();
				return;
			}
		}
	}

	event->ignore();
}

void ContainerTabBar::mouseDoubleClickEvent(QMouseEvent *event)
{
	if (event->button() & Qt::LeftButton) {
		QTab *tab;

		for ( int i = 0; i < count(); ++i ) {
			tab = tabAt(i);
			if (tab->rect().contains(event->pos())) {
				emit doubleClicked(i);
				break;
			}
		}
	} else
		event->ignore();
}

/** ContainerTabWidget */

ContainerTabWidget::ContainerTabWidget(QWidget *parent, const char *name) : QTabWidget(parent, name)
{
	ContainerTabBar *tabBar = new ContainerTabBar(this);
	setTabBar(tabBar);

	connect( tabBar, SIGNAL(doubleClicked(int)),                  this, SLOT(slotDoubleClicked(int))               );
	connect( tabBar, SIGNAL(wantPaste(int, QClipboard::Mode)),    this, SLOT(slotWantPaste(int, QClipboard::Mode)) );
	connect( tabBar, SIGNAL(wantContextMenu(int, const QPoint&)), this, SLOT(slotContextMenu(int, const QPoint&))  );
	connect( tabBar, SIGNAL(wantShowPage(int)),                   this, SLOT(setCurrentPage(int))                  );
	connect( this,   SIGNAL(currentChanged(QWidget*)),            this, SLOT(currentBasketChanged(QWidget*))       );
}

ContainerTabWidget::~ContainerTabWidget()
{
}

void ContainerTabWidget::slotDoubleClicked(int tabIndex)
{
	emit doubleClicked(tabIndex);
}

void ContainerTabWidget::slotContextMenu(int tabIndex, const QPoint &globalPos)
{
	emit contextMenu(tabIndex, globalPos);
}

void ContainerTabWidget::currentBasketChanged(QWidget*)
{
	if (AppSettings::tray)
		AppSettings::tray->updateToolTip();
}

void ContainerTabWidget::slotWantPaste(int tabIndex, QClipboard::Mode mode)
{
	emit wantPaste(tabIndex, mode);
}

/** Container */

Container::Container(QWidget *parent, const char *name)
 : KMainWindow(parent, name)
{
	setCentralWidget(new QWidget(this));
	m_layout = new QVBoxLayout(centralWidget());
	m_tab    = new ContainerTabWidget(centralWidget());
	m_layout->addWidget(m_tab);

	initActions();

	AppSettings::mainContainer = this; // FIXME: Needed for the uggly hack in Basket::initActions() :-/
	loadBaskets();

	connect( m_tab, SIGNAL(doubleClicked(int)),               this, SLOT(propBasket(int))                      );
	connect( m_tab, SIGNAL(contextMenu(int, const QPoint&)),  this, SLOT(contextMenu(int, const QPoint&))      );
	connect( m_tab, SIGNAL(wantPaste(int, QClipboard::Mode)), this, SLOT(pasteToBasket(int, QClipboard::Mode)) );
	connect( m_tab, SIGNAL(currentChanged(QWidget*)),         this, SLOT(tabChanged(QWidget*))                 );
}

Container::~Container()
{
}

void Container::initActions()
{
	KIconLoader icol;
	QPixmap     ico;

	// TODO: KDE Wide hotkeys to create baskets, items, items from clipboard !
	// TODO: Configurable shortcuts / hotkeys
	ico = icol.loadIcon( "tab_new",   KIcon::Toolbar, KIcon::SizeSmall );
	actNewBasket = new QAction( i18n("New basket..."),       ico, i18n("&New basket..."),
	                            i18n("New basket", "Ctrl+B"),         this );
	connect( actNewBasket, SIGNAL(activated()) , this, SLOT(askNewBasket())          );

	ico = icol.loadIcon( "configure", KIcon::Toolbar, KIcon::SizeSmall );
	actAppConfig = new QAction( i18n("Configure BasKet..."), ico, i18n("&Configure BasKet..."), 0, this );
	connect( actAppConfig, SIGNAL(activated()) , this, SLOT(showAppSettingsDialog()) );

	ico = icol.loadIcon( "configure_shortcuts", KIcon::Toolbar, KIcon::SizeSmall );
	actConfigGlobalShortcuts = new QAction( i18n("Configure global s&hortcuts..."), ico,
	                                        i18n("Configure &global shortcuts..."), 0, this );
	connect( actConfigGlobalShortcuts, SIGNAL(activated()) , this, SLOT(showGlobalShortcutsSettingsDialog()) );

	ico = icol.loadIcon( "exit",      KIcon::Toolbar, KIcon::SizeSmall );
	actQuit      = new QAction( i18n("Quit"),                ico, i18n("&Quit"),
	                            KStdAccel::shortcut(KStdAccel::Quit), this );
	connect( actQuit,      SIGNAL(activated()) , this, SLOT(askForQuit())            );

	QAction* actHide      = new QAction( i18n("Hide main window"),    ico, i18n("&Hide main window"),
	                            i18n("Hide main window", "Escape"),   this );
	connect( actHide,      SIGNAL(activated()) , this, SLOT(hideOnEscape())          );
}

void Container::loadBaskets()
{
	// FIXME: it isn't the place but I will clear that later)
	KStandardDirs stdDirs;    // use KGroup ?
	AppSettings::appPath = stdDirs.saveLocation("data", "basket/");
	QDir dir(AppSettings::appPath, QString::null, QDir::Name | QDir::IgnoreCase, QDir::Dirs | QDir::Hidden | QDir::NoSymLinks);

	QStringList list = dir.entryList();
	if ( list.count() > 2 ) { // Pass "." and ".."
		for ( QStringList::Iterator it = list.begin(); it != list.end(); ++it )
			if ( *it != "." && *it != ".." )
				newBasket(QString(), AppSettings::appPath + *it + "/");
	} else // TODO: Create a basket of examples (tutorial) or link to a short help
		newBasket( i18n("Basket %1").arg(1) );

	m_tab->setCurrentPage(AppSettings::lastBasket());
}

Basket* Container::basketAt(int index)
{
	return (Basket*)m_tab->page(index);
}

void Container::pasteToBasket(int index, QClipboard::Mode mode)
{
	basketAt(index)->pasteItem(mode);
}

void Container::contextMenu(int tabIndex, const QPoint &globalPos)
{
	KPopupMenu *menu = new KPopupMenu(this);

	KIconLoader icol;
	QPixmap     ico;

	m_contextIndex = tabIndex;

	basketAt(tabIndex)->actProp->addTo(menu);
	ico = icol.loadIcon( "tab_remove",                       KIcon::Toolbar, KIcon::SizeSmall    );
	menu->insertItem(    ico, i18n("&Delete \"%1\"...").arg(basketAt(tabIndex)->name()),
	                     this, SLOT(delBasket()),       0, 1 );
	basketAt(tabIndex)->actLock->addTo(menu);

	menu->insertSeparator();
	actNewBasket->addTo(menu);
	menu->insertSeparator();
	actAppConfig->addTo(menu);
	actConfigGlobalShortcuts->addTo(menu);
	actQuit->addTo(menu);

	if ( basketAt(tabIndex)->isLocked() ) {
		menu->setItemEnabled(0, false);
		menu->setItemEnabled(1, false);
		menu->setItemChecked(2, true);
	}

	menu->exec(globalPos);
}

void Container::propBasket(int index)
{
	m_contextIndex = index;           // FIXME: m_contextIndex is VERY VERY VERY uggly !
	propBasket();
}

void Container::propBasket()
{
	BasketPropertiesDialog *bProp = new BasketPropertiesDialog(basketAt(m_contextIndex), this);
	bProp->exec();
}

void Container::delBasket()
{
	Basket  *basket = basketAt(m_contextIndex);

	int really = KMessageBox::questionYesNo( this,
		i18n("<qt>Do you really want to delete <b>%1</b> ?</qt>").arg(basket->name()), i18n("Delete confirm") );
	if (really == KMessageBox::No)
		return;

	basket->deleteFiles();

	m_tab->removePage(basket);
	delete basket;

	if (m_tab->count() == 0)
		newBasket();
}

void Container::slotLockClicked()
{
	Basket *cur = basketAt(m_contextIndex);

	lockBasket();
	cur->save();
}

void Container::lockBasket()
{
	Basket *cur = basketAt(m_contextIndex);

	cur->setLocked( ! cur->isLocked());
}

void Container::showAppSettingsDialog()
{
	AppSettingsDialog *appS = new AppSettingsDialog(this);
	appS->exec();
	delete appS;
}

void Container::showGlobalShortcutsSettingsDialog()
{
	KKeyDialog::configure(AppSettings::globalAccel);
	//.setCaption(..)
	AppSettings::globalAccel->writeSettings();
}

void Container::askNewBasket()
{
	static int number = 0;
	bool ok;

	QString name = QInputDialog::getText(
		i18n("New basket - BasKet"), i18n("New basket name :"),
		QLineEdit::Normal, i18n("Basket %1").arg(QString::number(++number)), &ok, this );
	if ( ok && !name.isEmpty() ) {
		newBasket(name);
		setActive(true);
	}
}

void Container::newBasket(const QString &name, const QString &savePath)
{
	Basket *basket = new Basket(m_tab, name);
	m_tab->addTab(basket, name);
	m_tab->showPage(basket);
	//m_tab->setCurrentPage( m_tab->indexOf(basket) );

	connect( basket, SIGNAL(nameChanged(Basket*, const QString&)), this, SLOT(basketNameChanged(Basket*, const QString&)) );
	connect( basket, SIGNAL(iconChanged(Basket*, const QString&)), this, SLOT(basketIconChanged(Basket*, const QString&)) );

	basket->setSavePath(savePath); // Will be saved if path is empty, or loaded instead
}

void Container::basketNameChanged(Basket *basket, const QString &name)
{
	m_tab->setTabLabel(basket, name);
}

void Container::basketIconChanged(Basket *basket, const QString &path)
{
	KIconLoader	icol;
	QIconSet	icos;

	icos = icol.loadIconSet( path, KIcon::NoGroup, KIcon::SizeSmall, true );
	m_tab->setTabIconSet( basket, icos );
}

void Container::linkLookChanged()
{
	for ( int i = 0; i < m_tab->count(); ++i )
		basketAt(i)->linkLookChanged();
}

void Container::tabPlacementChanged(bool onTop)
{
	m_tab->setTabPosition( onTop ? QTabWidget::Top : QTabWidget::Bottom );
}

void Container::tabChanged(QWidget*)
{
	AppSettings::setLastBasket( m_tab->currentPageIndex() );
	m_contextIndex = m_tab->currentPageIndex(); // For the QAction to show Properties or lock basket
}

QPtrList<Basket> Container::listBaskets()
{
	QPtrList<Basket> list;

	for ( int i = 0; i < m_tab->count(); ++i )
		list.append( basketAt(i) );

	return list;
}

int Container::currentBasketIndex()
{
	return m_tab->currentPageIndex();
}

Basket* Container::currentBasket()
{
	return (Basket*) m_tab->currentPage();
}

void Container::setCurrentBasket(int index)
{
	m_tab->setCurrentPage(index);
	setActive(true);
}

void Container::setActive(bool active)
{
	if (active) {
		if (isMinimized())
			hide();        // If minimized, show() doesn't work !
		show();            // Show it
		showNormal();      // If it was minimized [Needed by KWin 3.2]
		raise();           // Raise it on top
		setActiveWindow(); // And set it the active window
	} else
		hide();

	AppSettings::setWasShown( AppSettings::mainContainer->isShown() );
}

void Container::changeActive()
{
	if ( ! AppSettings::useSystray() )
		return;

	if (AppSettings::useCleverShowHide()) {
		/* Clever show / hide algorithm */
		setActive( ! isActiveWindow() );
	} else {
		/* Standard algorithm */
		setShown( ! isShown() );
		AppSettings::setWasShown( AppSettings::mainContainer->isShown() );
	}
}

bool Container::askForQuit()
{
	int really = KMessageBox::questionYesNo( this,
		i18n("Do you really want to quit BasKet ?"), i18n("Quit confirm") );

	if (really == KMessageBox::No)
		return false;

	kapp->quit();
	return true;
}

void Container::closeEvent(QCloseEvent *e)
{
	if (AppSettings::useSystray() && AppSettings::hideOnClose()) {
		KMessageBox::information(this,
			i18n("Closing the main window will keep BasKet running in the system tray. "
			     "Use Quit from the baskets popup menu to quit the application."),
			i18n("Docking in System Tray"), "hideOnCloseInfo");
		setActive(false);
		e->ignore();
	} else
		if (askForQuit())
			e->accept();
		else
			e->ignore();
}

void Container::hideOnEscape()
{
	if (AppSettings::useSystray() && AppSettings::hideOnEscape())
		setActive(false);
}

void Container::hideEvent(QHideEvent *)
{
	if (AppSettings::useSystray() && AppSettings::hideOnMinimize() && isMinimized())
//		hide();
		setActive(false);
}

void Container::wheelEvent(QWheelEvent *e)
{
	int delta = (e->delta() > 0) - (e->delta() < 0); // 1 or -1
	int index = (m_tab->currentPageIndex() + delta) % m_tab->count();
	if (index < 0)
		index = m_tab->count() - 1;
	m_tab->setCurrentPage(index);

	if (AppSettings::usePassivePopup())
		showPassiveContent();

	e->accept();
}

void Container::contextMenuEvent(QContextMenuEvent *e)
{
	KPopupMenu *menu = new KPopupMenu(this);

	actNewBasket->addTo(menu);
	menu->insertSeparator();
	actAppConfig->addTo(menu);
	actConfigGlobalShortcuts->addTo(menu);
	actQuit->addTo(menu);

	menu->exec(e->globalPos());
}

void Container::newItemInCurrentBasket()
{
	currentBasket()->addItemDialog();
}

void Container::pasteInCurrentBasket()
{
	currentBasket()->pasteItem();

	if (AppSettings::usePassivePopup())
		showPassiveDropped(i18n("Clipboard content (%1) pasted to basket <i>%2</i>"));
}

void Container::pasteSelInCurrentBasket()
{
	currentBasket()->pasteItem(QClipboard::Selection);

	if (AppSettings::usePassivePopup())
		showPassiveDropped(i18n("Selection (%1) pasted to basket <i>%2</i>"));
}

void Container::goToPreviousBasket()
{
	int index = (m_tab->currentPageIndex() - 1);
	if (index < 0)
		index = m_tab->count() - 1;
	m_tab->setCurrentPage(index);

	if (AppSettings::usePassivePopup())
		showPassiveContent();
}

void Container::goToNextBasket()
{
	int index = (m_tab->currentPageIndex() + 1) % m_tab->count();
	m_tab->setCurrentPage(index);

	if (AppSettings::usePassivePopup())
		showPassiveContent();
}

void Container::showPassiveDropped(const QString &title)
{
	KPassivePopup::message(
		title.arg(currentBasket()->lastInsertedItem()->contentTypeName())
		     .arg(currentBasket()->name()),
		currentBasket()->lastInsertedItem()->toString(),
		kapp->iconLoader()->loadIcon(currentBasket()->icon(), KIcon::NoGroup, 16, KIcon::DefaultState, 0L, true),
		AppSettings::useSystray() ? (QWidget*)AppSettings::tray : (QWidget*)AppSettings::mainContainer);
}

void Container::showPassiveContent()
{
	// FIXME: Duplicate code (2 times)
	QString message;
	if (currentBasket()->canDragItem()) {
		Item *item = currentBasket()->currentStackItem();
		message = i18n("%1 item to drag (%2)")
				.arg( currentBasket()->systrayStackFrom() == 1 ? i18n("First") : i18n("Last") )
				.arg( item->contentTypeName() );
		if (item->type() != Item::Image) {
			message += " :\n";
			message += item->toString();
		}
	} else if (currentBasket()->isEmpty())
		message += "\n" + i18n("(Empty)");

	KPassivePopup::message(
		i18n("Current basket  : <i>%1</i>")
			.arg(currentBasket()->name()),
		message,
		kapp->iconLoader()->loadIcon(currentBasket()->icon(), KIcon::NoGroup, 16, KIcon::DefaultState, 0L, true),
		AppSettings::useSystray() ? (QWidget*)AppSettings::tray : (QWidget*)AppSettings::mainContainer);
}

// TODO: Show basket icon for those 4 passive popups

#include "container.moc"
