/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastient.laout@tuxfamily.org                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#ifndef CONTAINER_H
#define CONTAINER_H

#include <kmainwindow.h>
#include <qtabbar.h>
#include <qtabwidget.h>
#include <ksystemtray.h>
#include <qptrlist.h>
#include <qpoint.h>
#include <qclipboard.h>

class QWidget;
class QPoint;
class QTimer;
class QAction;

class Basket;
class Container;

/**This class provide a personalized system tray icon.
  *@author Sbastien Laot
  */

class ContainerSystemTray : public KSystemTray
{
  Q_OBJECT
  public:
	ContainerSystemTray(QWidget *parent = 0, const char *name = 0);
	~ContainerSystemTray();
  protected:
	void mousePressEvent(QMouseEvent *event);
	void mouseMoveEvent(QMouseEvent *event);
	void mouseReleaseEvent(QMouseEvent *event);
	void dragEnterEvent(QDragEnterEvent *event);
	void dragLeaveEvent(QDragLeaveEvent*);
	void dropEvent(QDropEvent *event);
	void wheelEvent(QWheelEvent *event);
  protected slots:
	void showBasket(int index);
  public slots:
	void updateToolTip();
  private:
	QTimer    *m_showTimer;
	bool       m_canDrag;
	QPoint     m_pressPos;
	Container *m_parentContainer;
};

/**This class provide a QTabBar with a popup menu for the baskets
  *and drag and drop possibilities.
  *@author Sbastien Laot
  */

class ContainerTabBar : public QTabBar {
  Q_OBJECT
  public:
	ContainerTabBar(QWidget *parent = 0, const char *name = 0);
	~ContainerTabBar();
  protected:
	virtual void dragEnterEvent(QDragEnterEvent *event);
	virtual void dragLeaveEvent(QDragLeaveEvent*);
	virtual void dropEvent(QDropEvent *event);
	virtual void mousePressEvent(QMouseEvent *event);
	virtual void mouseDoubleClickEvent(QMouseEvent *event);
  signals:
	void doubleClicked(int tabIndex);
	void wantContextMenu(int tabIndex, const QPoint &globalPos);
	void wantShowPage(int tabIndex);
	void wantPaste(int tabIndex, QClipboard::Mode mode);
  private:
	virtual void contextMenuEvent(QContextMenuEvent *e);
	virtual void dragMoveEvent(QDragMoveEvent *e);
  private slots:
	void timeout();

  private:
	QTimer *m_showTimer;
	int     m_basketToShow; // Number of the basket to show after timeout
};

/**This class provide a QTabWidget with a popup menu for the tabs.
  *@author Sbastien Laot
  */

class ContainerTabWidget : public QTabWidget {
  Q_OBJECT
  public:
	ContainerTabWidget(QWidget *parent = 0, const char *name = 0);
	~ContainerTabWidget();
  signals:
	void doubleClicked(int tabIndex);
	void contextMenu(int tabIndex, const QPoint &globalPos);
	void wantPaste(int tabIndex, QClipboard::Mode mode);
  private slots:
	void slotWantPaste(int tabIndex, QClipboard::Mode mode);
	void slotDoubleClicked(int tabIndex);
	void slotContextMenu(int tabIndex, const QPoint &globalPos);
	void currentBasketChanged(QWidget*);
};

/**The window that contain baskets, organized by tabs.
  *@author Sbastien Laot
  */

class Container : public KMainWindow {
  Q_OBJECT
  public:
	Container(QWidget *parent = 0, const char *name = 0);
	~Container();
  private:
	void initActions();
  public slots:
	void loadBaskets();
	/** Baskets manipulations */
	void propBasket(int index);
	void propBasket();
	void delBasket();
	void lockBasket();
	void askNewBasket();
	void newBasket(const QString &name = QString(), const QString &savePath = QString());
		// Create new basket in this container, and
		//  Save it in a new folder or
		//  Load it if savePath is defined
	void pasteToBasket(int index, QClipboard::Mode mode = QClipboard::Clipboard);
	/** Display updates */
	void basketNameChanged(Basket *basket, const QString &name);
	void basketIconChanged(Basket *basket, const QString &path);
	/** Global config changed */
	void linkLookChanged();
	void tabPlacementChanged(bool onTop);
	/** User actions */
	void showAppSettingsDialog();
	void showGlobalShortcutsSettingsDialog();
	void setActive(bool active = true);
	void changeActive();
	bool askForQuit();
  public:
	/** Necessary to the System tray popup menu, and others... */
	QPtrList<Basket> listBaskets();
	int currentBasketIndex();
	Basket* currentBasket();
	void setCurrentBasket(int index);
	Basket* basketAt(int index);
  public slots:
	/** Global shortcuts */
	void newItemInCurrentBasket();
	void pasteInCurrentBasket();
	void pasteSelInCurrentBasket();
	void goToPreviousBasket();
	void goToNextBasket();
	void showPassiveDropped(const QString &title); // %1 will be replaced by contentTypeName()
	                                               // %2 with currentBasket()->name()
	void showPassiveContent();
  protected:
	void closeEvent(QCloseEvent *e);
	void hideEvent(QHideEvent *);
	void contextMenuEvent(QContextMenuEvent *e);
  public: // To redirect SystemTray wheelEvent
	void wheelEvent(QWheelEvent *e);
  private slots:
	/** User actions */
	void hideOnEscape();
	void tabChanged(QWidget*);
	void contextMenu(int tabIndex, const QPoint &globalPos);
  public slots: // Uggly hack to allow a Basket to lock itself !!
	void slotLockClicked();
  private:
	QVBoxLayout        *m_layout;
	ContainerTabWidget *m_tab;
	int                 m_contextIndex;
  public:
	QAction *actNewBasket;
	QAction *actAppConfig;
	QAction *actConfigGlobalShortcuts;
	QAction *actQuit;
};

#endif // CONTAINER_H
