/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastient.laout@tuxfamily.org                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

/** Interface */
#include <qlayout.h>
#include <qlabel.h>
#include <qfont.h>
#include <qinputdialog.h>
#include <kiconloader.h>
#include <ktempfile.h>
#include <qcursor.h>
#include <klocale.h>
#include <kapplication.h>
#include <qdragobject.h>
#include <kpopupmenu.h>
#include <qtooltip.h>
#include <kurldrag.h>
#include <qaction.h>

/** Open URLs */
#include <kstandarddirs.h>
#include <krun.h>
#include <qfile.h>
#include <qdir.h>
#include <kfiledialog.h>
#include <qfile.h>
#include <kio/jobclasses.h>

#include "basket.h"
#include "item.h"
#include "itemedit.h"
#include "linklabel.h"
#include "itemfactory.h"
#include "appsettings.h"
#include "onclickaction.h"

#include "itemdrag.h"

#include "debugwindow.h"

/** AlignableCheckBox */

AlignableCheckBox::AlignableCheckBox(const QString &text, QWidget *parent, const char *name)
 : QWidget(parent, name), m_check(this)
{
	m_layout = new QVBoxLayout(this);
	m_spacer = new QSpacerItem(0, 0, QSizePolicy::Minimum, QSizePolicy::Expanding);
	m_check.setText(text);
	m_layout->addWidget(&m_check);
}

AlignableCheckBox::~AlignableCheckBox()
{
}

void AlignableCheckBox::setAlignment(int align)
{
	m_layout->remove(&m_check);
	m_layout->removeItem(m_spacer);

	switch (align) {
		case 0:
			m_layout->addWidget(&m_check);
			m_layout->addItem(m_spacer);
			break;
		case 1:
			m_layout->addWidget(&m_check);
			break;
		case 2:
			m_layout->addItem(m_spacer);
			m_layout->addWidget(&m_check);
			break;
	}
}

/** Item */

Item::Item(const QString &text, int fontType, const QColor &fontColor,
           const QString &annotations, bool checked, const QString &runCommand,
           Basket *parent, const QString &fileName)
 : QFrame(parent), m_parentBasket(parent), /*m_annotations(annotations),*/
   m_runCommand(runCommand), m_fileName(fileName)
{
	m_type = Text;
	initItem(checked);
	setText(text, fontType, fontColor);
	setAnnotations(annotations);
}

Item::Item(const QString &html, bool showSource,
           const QString &annotations, bool checked, const QString &runCommand,
           Basket *parent, const QString &fileName)
 : QFrame(parent), m_parentBasket(parent), /*m_annotations(annotations),*/
   m_runCommand(runCommand), m_fileName(fileName)
{
	m_type = Html;
	initItem(checked);
	setHtml(html, showSource);
	setAnnotations(annotations);
}

Item::Item(const KURL &url, const QString &title, const QString &icon,
           bool autoTitle, bool autoIcon,
           const QString &annotations, bool checked, const QString &runCommand,
           Basket *parent)
 : QFrame(parent), m_parentBasket(parent), /*m_annotations(annotations),*/
   m_runCommand(runCommand), m_url(0)
{
	m_type = Url;
	initItem(checked);
	setUrl(url, title, icon);
	setAuto(autoTitle, autoIcon);
	setAnnotations(annotations);
}

Item::Item(const QPixmap &image,
           const QString &annotations, bool checked, const QString &runCommand,
           Basket *parent, const QString &fileName)
 : QFrame(parent), m_parentBasket(parent), /*m_annotations(annotations),*/
   m_runCommand(runCommand), m_fileName(fileName)
{
	m_type = Image;
	initItem(checked);
	setPixmap(image);
	setAnnotations(annotations);
}

Item::Item(const QColor  &color,
           const QString &annotations, bool checked, const QString &runCommand,
           Basket *parent)
 : QFrame(parent), m_parentBasket(parent), /*m_annotations(annotations),*/
   m_runCommand(runCommand), m_color(0)
{
	m_type = Color;
	initItem(checked);
	setColor(color);
	setAnnotations(annotations);
}

/* FIXME: URGENT: Create an updateTooTip() and run it in constructors.
 *        Because ToolTip soon will'nt show ONLY annotations !!!!!!!!!
 */

Item::~Item()
{
	if (m_editProcess.isRunning())
		m_editProcess.kill();
}

void Item::initItem(bool checked)
{
	m_layout = new QHBoxLayout(this);

	m_check = new AlignableCheckBox(QString(), this);
	m_check->setShown( m_parentBasket->isCheckable() );
	m_check->setSizePolicy( QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Expanding, 0, 0, false) );
	m_check->setEnabled( ! m_parentBasket->isLocked() );
	m_check->checkBox()->setChecked( checked );
	m_layout->addWidget(m_check);

	if (useLinkLabel()) {
		m_linkLabel = new LinkLabel(m_parentBasket->hAlign(), m_parentBasket->vAlign(), this);
		m_layout->addWidget(m_linkLabel);
	} else {
		m_item = new QLabel(this);
		m_layout->addWidget(m_item);
	}

	alignChanged( m_parentBasket->hAlign(), m_parentBasket->vAlign() );

	connect( m_check->checkBox(), SIGNAL(clicked()), this, SLOT(slotChecked()) );
	setAcceptDrops(true);
	m_canDrag = false;
}

QString Item::contentTypeName()
{
	switch(m_type) {
		case Text:  return i18n("Text");
		case Html:  return i18n("Rich Text (HTML)");
		case Url:   return i18n("Link");
		case Image: return i18n("Image");
		case Color: return i18n("Color");
	}
	return QString();
}

QString Item::toString()
{
	switch(m_type) {
		case Text:  return text();
		case Html:  return html();
		case Url:   return url().prettyURL();
		case Image: return QString();
		case Color: return color().name();
	}
	return QString();
}

/** Events */

int Item::onClickActionPolicy()
{
	if (m_type == Url)
		return m_parentBasket->fileOnClickAction();
	else
		return m_parentBasket->contentOnClickAction();
}


void Item::contextMenuEvent(QContextMenuEvent *e)
{
	KPopupMenu *popupMenu = new KPopupMenu(this);

	KIconLoader icol;
	QPixmap     ico;
	QString     txt;

	txt = i18n("&Edit...%1")
	      .arg(OnClickAction::hotkeyForAction(OnClickAction::Edit,
	           onClickActionPolicy(), m_parentBasket->isRunOverrideClick(), !m_runCommand.isEmpty()));
	ico = icol.loadIcon(   "edit",                    KIcon::Toolbar, KIcon::SizeSmall    );
	popupMenu->insertItem( ico, txt,                  this, SLOT(slotEdit()),        0, 0 );
	txt = i18n("Edit a&nnotations...%1")
	      .arg(OnClickAction::hotkeyForAction(OnClickAction::EditAnnot,
	           onClickActionPolicy(), m_parentBasket->isRunOverrideClick(), !m_runCommand.isEmpty()));
	ico = icol.loadIcon(   "attach",                  KIcon::Toolbar, KIcon::SizeSmall    );
	popupMenu->insertItem( ico, txt,                  this, SLOT(slotEditAnnotations()), 0, 1 );
	ico = icol.loadIcon(   "editdelete",              KIcon::Toolbar, KIcon::SizeSmall    );
	popupMenu->insertItem( ico, i18n("&Delete"),      this, SLOT(slotDelete()),      0, 2 );
	m_parentBasket->actAddItem->addTo(popupMenu);
	popupMenu->insertSeparator();
	ico = icol.loadIcon(   "editcut",                 KIcon::Toolbar, KIcon::SizeSmall    );
	popupMenu->insertItem( ico, i18n("C&ut"),         this, SLOT(slotCut()),         0, 4 );
	ico = icol.loadIcon(   "editcopy",                KIcon::Toolbar, KIcon::SizeSmall    );
	txt = i18n("&Copy%1")
	      .arg(OnClickAction::hotkeyForAction(OnClickAction::Copy,
	           onClickActionPolicy(), m_parentBasket->isRunOverrideClick(), !m_runCommand.isEmpty()));
	popupMenu->insertItem( ico, txt,                  this, SLOT(slotCopy()),        0, 5 );
	m_parentBasket->actPaste->addTo(popupMenu);

	// FIXME: Use KPopupMenu but disabled items icons are not alpha-ed !

	if (m_type == Url) {
		popupMenu->insertSeparator();
		txt = i18n("&Open%1")
		      .arg(OnClickAction::hotkeyForAction(OnClickAction::Open,
		           onClickActionPolicy(), m_parentBasket->isRunOverrideClick(), !m_runCommand.isEmpty()));
		ico = icol.loadIcon(   "fileopen",                          KIcon::Toolbar, KIcon::SizeSmall    );
		popupMenu->insertItem( ico, txt,                            this, SLOT(slotOpen()),        0, 7 );
		txt = i18n("Open &with...%1")
		      .arg(OnClickAction::hotkeyForAction(OnClickAction::OpenWith,
		           onClickActionPolicy(), m_parentBasket->isRunOverrideClick(), !m_runCommand.isEmpty()));
		ico = icol.loadIcon (  "fileopen",                          KIcon::Toolbar, KIcon::SizeSmall    );
		popupMenu->insertItem( ico, txt,                            this, SLOT(slotOpenWith()),    0, 8 );
		ico = icol.loadIcon (  "filesaveas",                        KIcon::Toolbar, KIcon::SizeSmall    );
		popupMenu->insertItem( ico, i18n("&Save target as..."),     this, SLOT(slotSaveAs()),      0, 9 );
	} else {
		if (m_type != Color) {
			popupMenu->insertSeparator();
			ico = icol.loadIcon (  "filesaveas",                    KIcon::Toolbar, KIcon::SizeSmall  );
			popupMenu->insertItem( ico, i18n("&Save a copy as..."), this, SLOT(slotSaveAs()),   0, 10 );
		}
		/*if (m_type == Image) {
			ico = icol.loadIcon (  "background",                   KIcon::Toolbar, KIcon::SizeSmall     );
			popupMenu->insertItem( ico, i18n("Set as wall&paper"), this, SLOT(slotSetWallpaper()), 0, 11 );
		}*/
	}

	if ( m_parentBasket->isLocked() ) {
		popupMenu->setItemEnabled(0, false);
		popupMenu->setItemEnabled(1, false);
		popupMenu->setItemEnabled(2, false);
		popupMenu->setItemEnabled(3, false);
		popupMenu->setItemEnabled(4, false);
		popupMenu->setItemEnabled(6, false);
	}

	if (m_editProcess.isRunning())
		popupMenu->setItemEnabled( 0, false );

	popupMenu->popup(e->globalPos());
	e->accept();
}

void Item::setPaletteBackgroundColor(const QColor &color)
{
	QFrame::setPaletteBackgroundColor(color);
	if (useLinkLabel())
		m_linkLabel->setPaletteBackgroundColor(color);
}

void Item::mousePressEvent(QMouseEvent *qme)
{
	if (qme->button() & Qt::LeftButton) {
		m_pressPos = qme->globalPos();
		m_canDrag  = true;
		qme->accept();
	} else
		qme->ignore(); // Basket must receive the event to paste selection
}

void Item::mouseReleaseEvent(QMouseEvent *qme)
{
	m_canDrag = false;

	// Exec an action only if left button is pressed
	//  BUT ALSO IF ALT PRESSED to allow Alt+middleClick to launch actions
	//  Because KWin handle Alt+click.
	//  So it's a temporary hack to provide an alternative to alt+click !
	if ( ! (qme->button() & Qt::LeftButton) &&   // If button is different of LeftButton
	     ! (qme->button() & Qt::MidButton)     ) //  and also different of MiddleButton
		return;                                  // => No action to do
	if ( (qme->button() & Qt::MidButton) && !(qme->state() & Qt::KeyButtonMask) ) // If button is Middle and without keys
		return;                                  // => It's a paste selection => No action to do

	switch ( OnClickAction::actionForButton(qme->state(), onClickActionPolicy(),
	         m_parentBasket->isRunOverrideClick(), !m_runCommand.isEmpty())      ) {
		case OnClickAction::Copy:
			*AppSettings::debugWindow << "<b>Action :</b>\tCopy to clipboard";
			slotCopy();
			break;
		case OnClickAction::CopyToBoth:
			*AppSettings::debugWindow << "<b>Action :</b>\tCopy to clipboard &";
			slotCopy();
			// No break because we run run both
		case OnClickAction::CopyToSel:
			*AppSettings::debugWindow << "<b>Action :</b>\tCopy to selection";
			slotCopySelection();
			break;
		case OnClickAction::Edit:
			*AppSettings::debugWindow << "<b>Action :</b>\tEdit";
			slotEdit();
			break;
		case OnClickAction::EditAnnot:
			*AppSettings::debugWindow << "<b>Action :</b>\tEdit annotations";
			slotEditAnnotations();
			break;
		case OnClickAction::Open:
			*AppSettings::debugWindow << "<b>Action :</b>\tOpen";
			slotOpen();
			break;
		case OnClickAction::OpenWith:
			*AppSettings::debugWindow << "<b>Action :</b>\tOpen with...";
			slotOpenWith();
			break;
		case OnClickAction::RunCommand:
			*AppSettings::debugWindow << "<b>Action :</b>\tRun command";
			slotRunCommand();
			break;
		case OnClickAction::NoAction:
			*AppSettings::debugWindow << "<b>Action :</b>\tNo action !";
			break;
	}
}

void Item::mouseMoveEvent(QMouseEvent *qme)
{
	if ( m_canDrag && (m_pressPos - qme->globalPos()).manhattanLength() > KApplication::startDragDistance() )
		dragItem();
}

void Item::dragItem()
{
	m_canDrag = false;

	QDragObject *d = new ItemDrag(*this, m_parentBasket/*this*/); // will be deleted by QT

	if ( d->drag() && ! m_parentBasket->isLocked() )
		emit wantDelete(this);
}

// FIXME: Since Item handle d'n'd events, Basket cannot receive them :

/* Retun 0 to insert before (if the mouse is on the top half of this item) or 1 for after
 */
void Item::dragMoveEvent(QDragMoveEvent *e)
{
	m_parentBasket->dragNotice(this, e->pos().y() - (height() / 2) > 0); // FIXME: emit a signal
}

void Item::dragLeaveEvent(QDragLeaveEvent *event)
{
	m_canDrag = false;
	m_parentBasket->dragLeaveEvent(event);
}

void Item::dragEnterEvent(QDragEnterEvent *event)
{
	m_parentBasket->dragEnterEvent(event);
}

void Item::dropEvent(QDropEvent *event)
{
	m_canDrag = false;
	m_parentBasket->dropEvent(event);
}

/** Slots */

void Item::editImage()
{
	/* If we are already editing, do not edit it again */
	if (m_editProcess.isRunning())
		return;

	/* Execute the edit command and wait... */
	m_editProcess.clearArguments();
	m_editProcess << "kpaint" << "--caption" << "Edit the basket image" << m_parentBasket->savePath() + fileName(); // m_editTmpFile;
	m_editProcess.start();       // TODO: Use app settings for edit image command (ex: gimp...)
	connect( &m_editProcess, SIGNAL(processExited(KProcess*)), this, SLOT(slotEndEdit(KProcess*)) );
}

void Item::slotEdit(bool editAnnotations)
{
	if ( m_parentBasket->isLocked() )
		return;

	ItemEditDialog *dialog = new ItemEditDialog(this, editAnnotations, this);
	dialog->exec();
	delete dialog;
}

void Item::slotEditAnnotations()
{
	slotEdit(true);
}

void Item::slotDelete()
{
	emit wantDelete(this);
}

void Item::slotNew()
{
	emit wantNew();
}

void Item::slotCut()
{
	slotCopy();
	slotDelete();
}

void Item::slotCopy(bool toSelection)
{
	QClipboard *cb = KApplication::clipboard();
	QClipboard::Mode mode = (toSelection ? QClipboard::Selection : QClipboard::Clipboard);

	cb->setData( new ItemDrag(*this), mode ); // ItemDrag will be deleted by QT
}

void Item::slotCopySelection()
{
	slotCopy(true);
}

void Item::slotPaste()
{
	emit wantPaste(QClipboard::Clipboard);
}

void Item::slotOpen()
{
	if (m_type != Url)  // TODO: Open other items as files
		return;

	if ( ! m_url->isEmpty()) {
		KRun *run = new KRun(*m_url);
		run->setAutoDelete(true);
	} else if ( ! runCommand().isEmpty())
		KRun::runCommand(runCommand());
}

void Item::slotOpenWith()
{
	if (m_type != Url)  // TODO: Open other items as files
		return;

	KRun::displayOpenWithDialog(*m_url);
}

void Item::slotSaveAs()
{
	QString fileName;
	QString filters;
	QString caption;

	switch(m_type) {
		case Text:  filters  = "text/plain"; break;
		case Html:  filters  = "text/html";  break;
		case Url:   filters  = "*";          break; // TODO:  Get MIME type of the url target
		                                            // FIXME: If isDir() : return
		case Color: return;                  break;
		case Image: filters  = "image/png";  break; // TODO: Offer more types
	}

	caption  = (m_type == Url ? i18n("Save target") : i18n("Save a copy"));
	fileName = KFileDialog::getSaveFileName(QString::null, filters, this, caption);
	// TODO: Ask to overwrite !
	if (fileName.isEmpty())
		return;

	saveAs(fileName);
}

void Item::slotEndEdit(KProcess*)
{
	pixmap()->load(m_parentBasket->savePath() + fileName());
	update();	// FIXME: If the image has been resized, the label keep its old size constraint (min) !
}

void Item::slotRunCommand()
{
	KRun::runCommand(runCommand());
}

void Item::slotChecked()
{
	saveProperties();
}

void Item::saveProperties()
{
	m_parentBasket->save();
}

void Item::saveContent()
{
	if (useFile())
		saveAs( KURL(m_parentBasket->savePath() + m_fileName) );
}

void Item::loadContent()
{
	if (useFile())
		loadFrom( KURL(m_parentBasket->savePath() + m_fileName) );
}

void Item::saveAs(const KURL &dest)
{
	QString fileName = urlWithoutProtocol(dest);

	if (m_type == Text || m_type == Html) {
		QFile file(fileName);	// TODO: FIXME: Use a KDE QFile quivalent !
		if ( file.open(IO_WriteOnly) ) {
			QTextStream stream( &file );
			if (m_type == Text)
				stream << text();
			if (m_type == Html)
				stream << html();
			file.close();
		}
	}

	if (m_type == Image)
		pixmap()->save(fileName, "PNG");	// TODO: FIXME: Use a KDE QFile quivalent !

	if (m_type == Url)	// Copy file insteid of saving as
		KIO::FileCopyJob fileCopy( url(), dest, /*int permissions*/0666, false, true, true, true);
}

void Item::loadFrom(const KURL &source)	// To ItemFactory ????
{
	QString fileName = source.prettyURL().right(source.prettyURL().length() - source.protocol().length() - 1);
	// Because KURL provide a "file:/path/name" insteid of the QT wanted "/path/name" :-/

	if (m_type == Text || m_type == Html) {
		QFile file(fileName);	// TODO: FIXME: Use a KDE QFile quivalent !
		if ( file.open(IO_ReadOnly) ) {
			QTextStream stream( &file );
			QString content, tmp;
			while ( tmp = stream.readLine() ) {
				if (content.isEmpty())
					content = tmp;
				else
					content += "\n" + tmp;
			}
			if (m_type == Text)
				setText(content);
			if (m_type == Html)
				setHtml(content);
			file.close();
		}
	}
	if (m_type == Image)
		pixmap()->load(fileName);	// TODO: FIXME: Use a KDE QFile quivalent !

//	if (m_type == File)	// Copy file !
//		KIO::FileCopyJob fileCopy( source, url(), /*int permissions*/0666, false, true, true, true);
}

bool Item::setFileName(const QString &fileName)
{
	if (useFile()) {
		/* Re-save content (move file) */
		QDir dir;
		if ( !  dir.exists(m_parentBasket->savePath() + fileName)     // Do not erase existing file
		     && dir.rename(m_parentBasket->savePath() + m_fileName,   // Move file
		                   m_parentBasket->savePath() + fileName) ) {
			m_fileName = fileName;                                    // Success
			return true;
		}
	}

	return false;
}

void Item::setRunCommand(const QString &runCommand)
{
	m_runCommand = runCommand;
}

bool Item::isChecked()
{
	return m_check->checkBox()->isChecked();
}

void Item::setChecked(bool check)
{
	m_check->checkBox()->setChecked(check);
}

QString Item::annotations()
{
	return m_annotations;
}

void Item::setAnnotations(const QString &annotations)
{
	m_annotations = annotations;

	if ( m_parentBasket->isShowAnnotations() )
		QToolTip::add(this, m_annotations);
}

void Item::showAnnotationsChanged(bool show)
{
	if (show)
		QToolTip::add(this, m_annotations);
	else
		QToolTip::add(this, "");
}

void Item::checkableChanged(bool checkable)
{
	m_check->setShown(checkable);
}

void Item::alignChanged(int hAlign, int vAlign)
{
	int hFlag, vFlag;

	switch (hAlign) {
		default:
		case 0: hFlag = Qt::AlignLeft;    break;
		case 1: hFlag = Qt::AlignHCenter; break;
		case 2: hFlag = Qt::AlignRight;   break;
	}
	switch (vAlign) {
		case 0: vFlag = Qt::AlignTop;     break;
		default:
		case 1: vFlag = Qt::AlignVCenter; break;
		case 2: vFlag = Qt::AlignBottom;  break;
	}

	m_layout->remove(m_check);
	m_layout->remove(m_item);
	m_layout->remove(m_linkLabel);
	m_layout->setResizeMode(QLayout::Minimum);

	if (useLinkLabel()) {
		m_layout->insertWidget(0, m_linkLabel);
		m_linkLabel->setAlign(hAlign, vAlign);
	} else {
		m_layout->insertWidget(0, m_item);
		m_item->setAlignment( hFlag | vFlag | Qt::WordBreak );
	}
	m_layout->insertWidget( -(hFlag == Qt::AlignRight), m_check );  // insert to index 0 or -1
	m_check->setAlignment(vAlign);
}

void Item::linkLookChanged()
{
	if (m_type == Url)
		m_linkLabel->setLink( *m_title, *m_icon, LinkLook::lookForURL(*m_url) );
}

void Item::lockedChanged(bool lock)
{
	m_check->setEnabled( ! lock);
}

/** Various functions to change contents (according to the type of the item) */

// Text

QString Item::text()
{
	return m_item->text();
}

int Item::textFontType()
{
	return m_textFontType;
}

void Item::setText(const QString &text)
{
	m_item->setTextFormat(Qt::PlainText);
	m_item->setText(text);
}

void Item::setText(const QString &text, int type, QColor fontColor)
{
	setText(text);
	setTextFontType(type);
	setPaletteForegroundColor(fontColor);
}

void Item::setTextFontType(int type)
{
	m_textFontType = type;
	m_item->setFont( ItemFactory::fontForFontType(type) );
}

// Html

QString Item::html()
{
	return m_item->text();
}

bool Item::isShowSource()
{
	return m_showSource;
}

void Item::setHtml(const QString &html)
{
	m_item->setText(html);
}

void Item::setHtml(const QString &html, bool showSource)
{
	m_showSource = showSource;
	m_item->setTextFormat( showSource ? Qt::PlainText : Qt::RichText );
	m_item->setText(html);
}

void Item::setShowSource(bool show)
{
	m_showSource = show;
	m_item->setTextFormat( show ? Qt::PlainText : Qt::RichText );
}

// Url

KURL Item::url()
{
	return *m_url;
}

QString Item::title()
{
	return *m_title;
}

QString Item::icon()
{
	return *m_icon;
}

void Item::setUrl(const KURL &url, const QString &title, const QString &icon)
{
	if (m_url) {
		delete m_url;
		delete m_title;
		delete m_icon;
	}

	m_url   = new KURL(url);
	m_title = new QString(title);
	m_icon  = new QString(icon);

	m_linkLabel->setLink(*m_title, *m_icon, LinkLook::lookForURL(*m_url));
}

// Image

QPixmap* Item::pixmap()
{
	return m_item->pixmap();
}

void Item::setPixmap(const QPixmap &pixmap)
{
	m_item->setPixmap(pixmap);
	/*m_item->update();
	m_item->setMinimumSize(pixmap.size());
	m_item->adjustSize();
	m_item->updateGeometry();*/
}

// Color

QColor Item::color()
{
	return QColor(*m_color);
}

void Item::setColor(QColor color)
{
	if (m_color)
		delete m_color;
	m_color = new QColor(color);

	setPaletteForegroundColor(color);
	QFont *font = new QFont();
	font->setBold(true);
	setFont(*font);

	setText(color.name());

}

/*void Item::reparent(Basket *parent, const QPoint &point, bool showIt)
{
	// m_parentBasket->remove
	// QFrame::reparent(parent, 0, point, showIt);
}*/

// Item::ColorFormat :
/*QString( "%1 \"%2\" (%3,%4,%5)%6 {%7 %8 %9}" ).
arg( name ).
arg( color.name().upper() ).
arg( r ).arg( g ).arg( b ).
arg( isWebColor( color ) ? " web" : "" ).
arg( r / 255.0, 1, 'f', 3 ).
arg( g / 255.0, 1, 'f', 3 ).
arg( b / 255.0, 1, 'f', 3 )
);*/

/*bool MainForm::isWebColor( QColor color )		// From QT examples
{
	int r = color.red();            // The 216 web colors are those colors whose RGB (Red, Green, Blue)
	int g = color.green();          //  values are all in the set (0, 51, 102, 153, 204, 255).
	int b = color.blue();

	return ( ( r ==   0 || r ==  51 || r == 102 ||
			   r == 153 || r == 204 || r == 255 ) &&
			 ( g ==   0 || g ==  51 || g == 102 ||
			   g == 153 || g == 204 || g == 255 ) &&
			 ( b ==   0 || b ==  51 || b == 102 ||
			   b == 153 || b == 204 || b == 255 ) );
}*/

#include "item.moc"
