/***************************************************************************
 *   Copyright (C) 2003 by Sbastien Laot                                 *
 *   sebastient.laout@tuxfamily.org                                        *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qlayout.h>
#include <qlabel.h>
#include <qpushbutton.h>
#include <kurlrequester.h>
#include <kurl.h>
#include <klineedit.h>
#include <qcombobox.h>
#include <qradiobutton.h>
#include <kcolorbutton.h>
#include <kcolorcombo.h>
#include <qcheckbox.h>
#include <qtoolbutton.h>
#include <qtextedit.h>
#include <qlineedit.h>
#include <kfontcombo.h>
#include <klocale.h>
#include <kicondialog.h>
#include <kiconloader.h>
#include <qiconset.h>
#include <kopenwith.h>

#include "item.h"
#include "itemedit.h"
#include "itemfactory.h"
#include "basket.h"
#include "linklabel.h"
#include "appsettings.h"
#include "variouswidgets.h"

#include <qmessagebox.h>

/** ItemEditWidgetBase */

ItemEditWidgetBase::ItemEditWidgetBase(Item *item, QWidget *parent, const char *name, WFlags fl)
 : QWidget(parent, name, fl)
{
	m_item = item;
	m_isAlreadyFirstFocused = false;
}

ItemEditWidgetBase::~ItemEditWidgetBase()
{
}

void ItemEditWidgetBase::saveChanges()
{
}

void ItemEditWidgetBase::goFirstFocus()
{
	if ( ! m_isAlreadyFirstFocused ) {
		doGoFirstFocus();
		m_isAlreadyFirstFocused = true;
	}
}

void ItemEditWidgetBase::doGoFirstFocus()
{
}

/** ItemTextEditWidget */

ItemTextEditWidget::ItemTextEditWidget(Item *item, QWidget *parent, const char *name, WFlags fl)
 : ItemEditWidgetBase(item, parent, name, fl)
{
	QVBoxLayout *vLayout = new QVBoxLayout(this, 7, 6);

	QWidget      *topWid    = new QWidget(this);
	QHBoxLayout  *topLayout = new QHBoxLayout(topWid, 0, 6);
	m_font                  = new QComboBox(topWid);
	m_color                 = new KColorCombo(topWid);
	QLabel       *topLbl    = new QLabel(m_font, i18n("A&ppearance :"), topWid);
	m_font->insertItem("Default");
	m_font->insertItem("Sans serif");
	m_font->insertItem("Serif");
	m_font->insertItem("Monospace");
	m_font->setCurrentItem(item->textFontType());
	m_color->setColor(m_item->paletteForegroundColor());
	topLayout->addWidget(topLbl);
	topLayout->addWidget(m_font);
	topLayout->addWidget(m_color);
	vLayout->addWidget(topWid);
	topLbl->setSizePolicy( QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed, 0, 0, false) );

	m_text = new QTextEdit(this);
	m_text->setTextFormat(Qt::PlainText);
	m_text->setText(m_item->text());
	changeFont(item->textFontType());
	changeColor(m_item->paletteForegroundColor());
	vLayout->addWidget(m_text);

	connect( m_font,  SIGNAL(activated(int)),           this, SLOT(changeFont(int))            );
	connect( m_color, SIGNAL(activated(const QColor&)), this, SLOT(changeColor(const QColor&)) );
}

ItemTextEditWidget::~ItemTextEditWidget()
{
}

void ItemTextEditWidget::doGoFirstFocus()
{
	m_text->setFocus();
	m_text->moveCursor(QTextEdit::MoveEnd, false);
}

void ItemTextEditWidget::changeFont(int fontType)
{
	m_text->selectAll(true);
	m_text->setFont(ItemFactory::fontForFontType(fontType));
	m_text->selectAll(false);
}

void ItemTextEditWidget::changeColor(const QColor &color)
{
	m_text->selectAll(true);
	m_text->setColor(color);
	m_text->selectAll(false);
}

void ItemTextEditWidget::saveChanges()
{
	m_item->setText(m_text->text());
	m_item->setPaletteForegroundColor(m_color->color());
	m_item->setTextFontType(m_font->currentItem());
}

/** ItemHtmlEditWidget */

ItemHtmlEditWidget::ItemHtmlEditWidget(Item *item, QWidget *parent, const char *name, WFlags fl)
 : ItemEditWidgetBase(item, parent, name, fl)
{
	QVBoxLayout *vLayout = new QVBoxLayout(this, 7, 6);
	KIconLoader icol;
	QIconSet    icosb;
	QIconSet    icosi;
	QIconSet    icosu;
	QIconSet    icosl;
	QIconSet    icosc;
	QIconSet    icosr;
	QIconSet    icosj;

	/*QWidget **/wid  = new QWidget(this);
	QHBoxLayout *hlay = new QHBoxLayout(wid);
	m_font            = new KFontCombo(wid);
	m_color           = new KColorCombo(wid);
	hlay->addWidget(m_font);
	hlay->addSpacing(6);
	hlay->addWidget(m_color);

	hlay->addSpacing(6);

	icosb = QIconSet( icol.loadIcon( "text_bold",   KIcon::Toolbar, KIcon::SizeSmall ) );
	icosi = QIconSet( icol.loadIcon( "text_italic", KIcon::Toolbar, KIcon::SizeSmall ) );
	icosu = QIconSet( icol.loadIcon( "text_under",  KIcon::Toolbar, KIcon::SizeSmall ) );
	m_tbb = new QToolButton(wid);
	m_tbb->setIconSet(icosb);
	m_tbb->setAccel(QKeySequence(i18n("Bold", "Ctrl+B")));
	m_tbb->setTextLabel(i18n("Bold"));
	m_tbb->setToggleButton(true);
	m_tbi = new QToolButton(wid);
	m_tbi->setIconSet(icosi);
	m_tbi->setAccel(QKeySequence(i18n("Italic", "Ctrl+I")));
	m_tbi->setTextLabel(i18n("Italic"));
	m_tbi->setToggleButton(true);
	m_tbu = new QToolButton(wid);
	m_tbu->setIconSet(icosu);
	m_tbu->setAccel(QKeySequence(i18n("Underline", "Ctrl+U")));
	m_tbu->setTextLabel(i18n("Underline"));
	m_tbu->setToggleButton(true);
	hlay->addWidget(m_tbb);
	hlay->addWidget(m_tbi);
	hlay->addWidget(m_tbu);

/*
	QTextEdit::VerticalAlignment
		This enum is used to set the vertical alignment of the text.
			QTextEdit::AlignNormal      - Normal alignment
			QTextEdit::AlignSuperScript - Superscript
			QTextEdit::AlignSubScript   - Subscript
	icosSub = QIconSet( icol.loadIcon( "text_sub",   KIcon::Toolbar, KIcon::SizeSmall ) );
	icosSup = QIconSet( icol.loadIcon( "text_super", KIcon::Toolbar, KIcon::SizeSmall ) );
	"text_strike"
	"indent"
	"unindent"
*/

	hlay->addSpacing(6);

	icosl = QIconSet( icol.loadIcon( "text_left",   KIcon::Toolbar, KIcon::SizeSmall ) );
	icosc = QIconSet( icol.loadIcon( "text_center", KIcon::Toolbar, KIcon::SizeSmall ) );
	icosr = QIconSet( icol.loadIcon( "text_right",  KIcon::Toolbar, KIcon::SizeSmall ) );
	icosj = QIconSet( icol.loadIcon( "text_block",  KIcon::Toolbar, KIcon::SizeSmall ) );
	m_tbal  = new QToolButton(wid);
	m_tbal->setIconSet(icosl);
	m_tbal->setTextLabel(i18n("Align to left"));
	m_tbal->setToggleButton(true);
	m_tbac  = new QToolButton(wid);
	m_tbac->setIconSet(icosc);
	m_tbac->setTextLabel(i18n("Centered"));
	m_tbac->setToggleButton(true);
	m_tbar  = new QToolButton(wid);
	m_tbar->setIconSet(icosr);
	m_tbar->setTextLabel(i18n("Align to right"));
	m_tbar->setToggleButton(true);
	m_tbaj  = new QToolButton(wid);
	m_tbaj->setIconSet(icosj);
	m_tbaj->setTextLabel(i18n("Justifyed"));
	m_tbaj->setToggleButton(true);
	hlay->addWidget(m_tbal);
	hlay->addWidget(m_tbac);
	hlay->addWidget(m_tbar);
	hlay->addWidget(m_tbaj);
	hlay->insertStretch(-1);

	vLayout->addWidget(wid);

	m_text = new QTextEdit(this);
	m_text->setTextFormat( item->isShowSource() ? Qt::PlainText : Qt::RichText );
	m_text->setText(m_item->html());
	vLayout->addWidget(m_text);

	m_showHtmlSource = new QCheckBox(i18n("&Show HTML source"), this);
	m_showHtmlSource->setChecked(m_text->textFormat() == Qt::PlainText);
	vLayout->addWidget(m_showHtmlSource);
	wid->setEnabled( ! m_showHtmlSource->isChecked() );

	cursorPositionChanged(0, 0);

	connect( m_font,  SIGNAL(textChanged(const QString&)), this,   SLOT(slotChangeFont(const QString&))  );
	connect( m_color, SIGNAL(activated(const QColor&)),    m_text, SLOT(setColor(const QColor&))         );

	connect( m_tbb,   SIGNAL(toggled(bool)), m_text, SLOT(setBold(bool))      );
	connect( m_tbi,   SIGNAL(toggled(bool)), m_text, SLOT(setItalic(bool))    );
	connect( m_tbu,   SIGNAL(toggled(bool)), m_text, SLOT(setUnderline(bool)) );

	connect( m_tbal,  SIGNAL(clicked()), this, SLOT(slotAlignLeft())    );
	connect( m_tbac,  SIGNAL(clicked()), this, SLOT(slotAlignCenter())  );
	connect( m_tbar,  SIGNAL(clicked()), this, SLOT(slotAlignRight())   );
	connect( m_tbaj,  SIGNAL(clicked()), this, SLOT(slotAlignJustify()) );

	connect( m_text,  SIGNAL(cursorPositionChanged(int, int)), this, SLOT(cursorPositionChanged(int, int)) );
	connect( m_text,  SIGNAL(clicked(int, int)),               this, SLOT(cursorPositionChanged(int, int)) );

	connect( m_showHtmlSource, SIGNAL(toggled(bool)), this, SLOT(slotShowSource(bool)) );
}

ItemHtmlEditWidget::~ItemHtmlEditWidget()
{
}

void ItemHtmlEditWidget::doGoFirstFocus()
{
	m_text->setFocus();
	m_text->moveCursor(QTextEdit::MoveEnd, false);
}

void ItemHtmlEditWidget::slotShowSource(bool on)
{
	QString str = m_text->text();
	m_text->setTextFormat( on ? Qt::PlainText : Qt::RichText );
	m_text->setText(str);

	wid->setEnabled( ! on );
}

void ItemHtmlEditWidget::slotChangeFont(const QString &font)
{
	m_text->setCurrentFont(font);
}

void ItemHtmlEditWidget::slotAlignLeft()
{
	m_text->setAlignment(Qt::AlignLeft);
	untoggleAllAlign();
	m_tbal->setOn(true);
}

void ItemHtmlEditWidget::slotAlignCenter()
{
	m_text->setAlignment(Qt::AlignCenter);
	untoggleAllAlign();
	m_tbac->setOn(true);
}

void ItemHtmlEditWidget::slotAlignRight()
{
	m_text->setAlignment(Qt::AlignRight);
	untoggleAllAlign();
	m_tbar->setOn(true);
}

void ItemHtmlEditWidget::slotAlignJustify()
{
	m_text->setAlignment(Qt::AlignJustify);
	untoggleAllAlign();
	m_tbaj->setOn(true);
}

void ItemHtmlEditWidget::untoggleAllAlign()
{
	m_tbal->setOn(false);
	m_tbac->setOn(false);
	m_tbar->setOn(false);
	m_tbaj->setOn(false);
}

void ItemHtmlEditWidget::cursorPositionChanged(int/* para*/, int/* pos*/)
{
	if (m_text->hasSelectedText())
		return;

	m_font->setCurrentFont(m_text->currentFont().family());
	m_color->setColor(m_text->color());
	m_tbb->setOn(m_text->bold());
	m_tbi->setOn(m_text->italic());
	m_tbu->setOn(m_text->underline());

	untoggleAllAlign();
	switch (m_text->alignment()) {
		case Qt::AlignLeft:    m_tbal->setOn(true); break;
		case Qt::AlignCenter:  m_tbac->setOn(true); break;
		case Qt::AlignRight:   m_tbar->setOn(true); break;
		case Qt::AlignJustify: m_tbaj->setOn(true); break;
	};
}

void ItemHtmlEditWidget::saveChanges()
{
	m_item->setHtml(m_text->text(), m_showHtmlSource->isChecked());
}

/** ItemUrlEditWidget */

ItemUrlEditWidget::ItemUrlEditWidget(Item *item, QWidget *parent, const char *name, WFlags fl)
 : ItemEditWidgetBase(item, parent, name, fl)
{
	QGridLayout *layout = new QGridLayout(this, 3, 2, 7, 6);

	m_url = new KURLRequester(m_item->url().url(), this);

	QWidget *wid1 = new QWidget(this);
	QHBoxLayout *titleLay = new QHBoxLayout(wid1, 0, 6);
	m_title = new QLineEdit(m_item->title(), wid1);
	m_autoTitle = new QPushButton(i18n("Auto"), wid1);
	m_autoTitle->setToggleButton(true);
	m_autoTitle->setOn(item->autoTitle());
	titleLay->addWidget(m_title);
	titleLay->addWidget(m_autoTitle);

	QWidget *wid = new QWidget(this);
	QHBoxLayout *hLay = new QHBoxLayout(wid, 0, 6);
	m_icon = new KIconButton(wid);
	QLabel *label3 = new QLabel(m_icon, i18n("&Icon :"), this);
	m_icon->setIconType ( KIcon::NoGroup, KIcon::MimeType );
	m_icon->setIconSize( LinkLook::lookForURL(KURL( m_url->lineEdit()->text() ))->iconSize() );
	if ( ! item->icon().isEmpty() )
		m_icon->setIcon(item->icon());
	/*m_icon->setIcon(item->icon());
	m_icon->show();
	m_icon->setMinimumSize( m_icon->size() );
	if ( item->icon().isEmpty() )
		m_icon->resetIcon();*/
	QPushButton *button2 = new QPushButton(i18n("Non&e"),  wid);
	m_autoIcon = new QPushButton(i18n("Auto"), wid);
	m_autoIcon->setToggleButton(true);
	m_autoIcon->setOn(item->autoIcon());
	hLay->addWidget(m_icon);
	hLay->addWidget(button2);
	hLay->addWidget(m_autoIcon);
	hLay->addStretch();

	//m_url->setShowLocalProtocol(true);
	QLabel      *label1 = new QLabel(m_url,   i18n("&URL :"),    this);
	QLabel      *label2 = new QLabel(m_title, i18n("&Title :"),  this);
	layout->addWidget(label1,  0, 0, Qt::AlignVCenter);
	layout->addWidget(label2,  1, 0, Qt::AlignVCenter);
	layout->addWidget(label3,  2, 0, Qt::AlignVCenter);
	layout->addWidget(m_url,   0, 1, Qt::AlignVCenter);
	layout->addWidget(wid1,    1, 1, Qt::AlignVCenter);
	layout->addWidget(wid,     2, 1, Qt::AlignVCenter);

	m_isAutoModified = false;
	connect( m_url,   SIGNAL(textChanged(const QString&)), this, SLOT(urlChanged(const QString&))     );
	connect( m_title, SIGNAL(textChanged(const QString&)), this, SLOT(doNotAutoTitle(const QString&)) );
	connect( m_icon,  SIGNAL(iconChanged(QString))       , this, SLOT(doNotAutoIcon(QString))         );
	connect( m_autoTitle, SIGNAL(clicked()), this, SLOT(guessTitle()) );
	connect( m_autoIcon,  SIGNAL(clicked()), this, SLOT(guessIcon())  );
	connect( button2,     SIGNAL(pressed()), this, SLOT(emptyIcon())  );

	urlChanged("");
}

ItemUrlEditWidget::~ItemUrlEditWidget()
{
}

void ItemUrlEditWidget::doGoFirstFocus()
{
	if (m_url->lineEdit()->text().isEmpty()) {
		m_url->setFocus();
		m_url->lineEdit()->end(false);
	} else {
		m_title->setFocus();
		m_title->end(false);
	}
}

void ItemUrlEditWidget::emptyIcon()
{
	m_autoIcon->setOn(false);
	m_icon->resetIcon();
}

void ItemUrlEditWidget::urlChanged(const QString&)
{
	m_isAutoModified = true;
	guessTitle();
	guessIcon();
}

void ItemUrlEditWidget::doNotAutoTitle(const QString&)
{
	if (m_isAutoModified)
		m_isAutoModified = false;
	else
		m_autoTitle->setOn(false);
}

void ItemUrlEditWidget::doNotAutoIcon(QString)
{
	m_autoIcon->setOn(false);
}

void ItemUrlEditWidget::guessIcon()
{
	if (m_autoIcon->isOn())
		m_icon->setIcon( ItemFactory::iconForURL( KURL(m_url->url()) ) );
}

void ItemUrlEditWidget::guessTitle()
{
	if (m_autoTitle->isOn())
		m_title->setText( ItemFactory::titleForURL( KURL(m_url->url()) ) );
}

void ItemUrlEditWidget::saveChanges()
{
	m_item->setUrl(  KURL(m_url->url()), m_title->text(), m_icon->icon() );
	m_item->setAuto( m_autoTitle->isOn(), m_autoIcon->isOn() );

	/* Change icon size if link look have changed */
	LinkLook *linkLook = LinkLook::lookForURL(KURL(m_url->lineEdit()->text()));
	QString icon = m_icon->icon();             // When we change size, icon isn't changed and keep it's old size
	m_icon->setIconSize(linkLook->iconSize()); //  So I store it's name and reload it after size change !
	if (icon != "")                            //  But if icon is empty it will load the "unknow" icon
		m_icon->setIcon(icon);                 //  So I reload it only if needed.
}

/** ItemImageEditWidget */

ItemImageEditWidget::ItemImageEditWidget(Item *item, QWidget *parent, const char *name, WFlags fl)
 : ItemEditWidgetBase(item, parent, name, fl)
{
	QVBoxLayout *vLayout   = new QVBoxLayout(this, 7, 6);

	m_edit                 = new QPushButton(i18n("&Edit"), this);
	/*QHBoxLayout *importLay = new QHBoxLayout(0, 0, 6);
	QPushButton *pbIcon    = new QPushButton(i18n("&Import an icon of"), this);
	m_iconSize             = new IconSizeCombo(false, this);
	m_iconSize->setSizePolicy( QSizePolicy(QSizePolicy::Fixed, QSizePolicy::Fixed) );*/
	vLayout->addWidget(m_edit);
	/*importLay->addWidget(pbIcon);
	importLay->addWidget(m_iconSize);
	vLayout->addItem(importLay);*/

	connect( m_edit, SIGNAL(clicked()), item, SLOT(editImage())  );
	/*connect( pbIcon, SIGNAL(clicked()), this, SLOT(chooseIcon()) );*/
}

ItemImageEditWidget::~ItemImageEditWidget()
{
}

void  ItemImageEditWidget::doGoFirstFocus()
{
	m_edit->setFocus();
}

/*void ItemImageEditWidget::chooseIcon()
{
	QString iconName;

	iconName = KIconDialog::getIcon(KIcon::Desktop, KIcon::Application, false, m_iconSize->iconSize());
	if (iconName.isEmpty())
		return;
	KIconLoader icol;
	QPixmap     pix;
	pix = icol.loadIcon(iconName, KIcon::Desktop, m_iconSize->iconSize());

	m_item->setPixmap(pix);
	m_item->saveContent();
}*/

void ItemImageEditWidget::saveChanges()
{
}

/** ItemColorEditWidget */

ItemColorEditWidget::ItemColorEditWidget(Item *item, QWidget *parent, const char *name, WFlags fl)
 : ItemEditWidgetBase(item, parent, name, fl)
{
	QHBoxLayout *layout = new QHBoxLayout(this, 7, 6);

/*
	m_button[0]          = new QRadioButton(i18n("Hex (#RRVVBB)"), group);
	m_button[1]          = new QRadioButton(i18n("RVB (R,V,B)"),   group);
	m_button[2]          = new QRadioButton(i18n("HSV (H,S,V)"),   group);
	m_button[3]          = new QRadioButton(i18n("Name (blue)"),   group);
*/
	m_color = new KColorButton(this);
	m_color->setColor(item->color());
	layout->addWidget(m_color);
}

ItemColorEditWidget::~ItemColorEditWidget()
{
}

void ItemColorEditWidget::doGoFirstFocus()
{
	m_color->setFocus();
}

void ItemColorEditWidget::saveChanges()
{
	m_item->setColor( m_color->color() );
}

/** ItemEditWidget */

ItemEditWidget::ItemEditWidget(Item *item, QWidget *parent, const char *name, WFlags fl)
 : ItemEditWidgetBase(item, parent, name, fl)
{
	QVBoxLayout *layout = new QVBoxLayout(this, 7, 6);

	m_checked     = new QCheckBox(i18n("Chec&ked item"), this);
	m_annotations = new QTextEdit(this);
	QLabel *lbl   = new QLabel(m_annotations, i18n("Anno&tations :"), this);
	m_annotations->setTextFormat(Qt::PlainText);
	m_annotations->setText(m_item->annotations());
	layout->addWidget(m_checked);
	layout->addWidget(lbl);
	layout->addWidget(m_annotations);
	m_checked->setChecked(item->isChecked());

	QHBoxLayout *rLay = new QHBoxLayout(0, 0, 6);
	m_runCommand      = new QLineEdit(item->runCommand(), this);
	QLabel *lbl2      = new QLabel(m_runCommand, i18n("On click &run command :"), this);
	QPushButton *pb   = new QPushButton(i18n("&Browse..."), this);
	rLay->addWidget(lbl2);
	rLay->addWidget(m_runCommand);
	rLay->addWidget(pb);
	layout->addLayout(rLay);

	connect( pb,   SIGNAL(clicked()),    this, SLOT(slotSelCommand()) );

	if (0 && /** TODO: I must add file abstraction support for this : **/ item->useFile()) {
		QHBoxLayout *hLay = new QHBoxLayout(0, 0, 6);
		m_fileName = new QLineEdit(item->fileName(), this);
		QLabel *lbl3 = new QLabel(m_fileName, i18n("&File name :"), this);
		hLay->addWidget(lbl3);
		hLay->addWidget(m_fileName);
		layout->addLayout(hLay);
	}
}

ItemEditWidget::~ItemEditWidget()
{
}

void ItemEditWidget::doGoFirstFocus()
{
	m_annotations->setFocus();
	m_annotations->moveCursor(QTextEdit::MoveEnd, false);
}

void ItemEditWidget::slotSelCommand()
{
	QString message;
	if ( ((Basket *)(m_item->parent()))->isRunOverrideClick() )
		message = i18n("Please choose witch application or command to run "
		               "at a click (or Ctrl+Alt+click) on the item.");
	else
		message = i18n("Please choose witch application or command to run "
		               "at a Ctrl+Alt+click on the item.");

	KOpenWithDlg *dlg =  new KOpenWithDlg(KURL::List(), message, m_runCommand->text(), this);
	dlg->exec();
	if ( ! dlg->text().isEmpty() )
		m_runCommand->setText(dlg->text());
}

void ItemEditWidget::saveChanges()
{
	m_item->setChecked(m_checked->isChecked());
	m_item->setAnnotations(m_annotations->text());
	m_item->setRunCommand(m_runCommand->text());
/*	if (m_item->useFile())
		if ( ! m_item->setFileName(m_fileName->text()) ) // Rename item file name
			m_fileName->setText(m_item->fileName());     // Reload file name if failed
*/
}

/** ItemEditDialog */

ItemEditDialog::ItemEditDialog(Item *item, bool editAnnotations, QWidget *parent, const char *name, WFlags fl)
 : QTabDialog(parent, name, true, fl)
{
	QString contentType;

	if (item->type() == Item::Text) {
		m_editContent = new ItemTextEditWidget(item, this);
		contentType = i18n("Text");
	}
	if (item->type() == Item::Html) {
		m_editContent = new ItemHtmlEditWidget(item, this);
		contentType = i18n("Rich text (HTML)");
	}
	if (item->type() == Item::Url) {
		m_editContent = new ItemUrlEditWidget(item, this);
		contentType = i18n("Link");
	}
	if (item->type() == Item::Image) {
		m_editContent = new ItemImageEditWidget(item, this);
		contentType = i18n("Image");
	}
	if (item->type() == Item::Color) {
		m_editContent = new ItemColorEditWidget(item, this);
		contentType = i18n("Color");
	}
	insertTab(m_editContent, i18n("Co&ntent : %1").arg(contentType));

	m_itemEdit = new ItemEditWidget(item, this);
	insertTab(m_itemEdit, i18n("&Meta-data"));

	m_item = item;

	setOkButton( i18n("&OK") );
	setApplyButton( i18n("&Apply") );
	setCancelButton( i18n("&Cancel") );
	setCaption(i18n("Edit item and properties"));

	if (editAnnotations)
		showPage(m_itemEdit);

	connect( this, SIGNAL(aboutToShow()),            this, SLOT(execGoFirstFocus())         );
	connect( this, SIGNAL(currentChanged(QWidget*)), this, SLOT(execGoFirstFocus(QWidget*)) );
	connect( this, SIGNAL(applyButtonPressed()),     this, SLOT(apply())                    );
}

ItemEditDialog::~ItemEditDialog()
{
}

void ItemEditDialog::execGoFirstFocus(QWidget *page)
{
	if (page == 0)
		page = currentPage();
	((ItemEditWidgetBase*)page)->goFirstFocus();
}

void ItemEditDialog::apply()
{
	m_editContent->saveChanges();
	m_itemEdit->saveChanges();

	m_item->saveContent();
	m_item->saveProperties();

	if (AppSettings::tray)
		AppSettings::tray->updateToolTip();
}

#include "itemedit.moc"
