/* cmd-revisions.c
 *
 * vim:smartindent ts=8:sts=2:sta:et:ai:shiftwidth=2
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/cmd/main.h"
#include "libarch/patch-logs.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/my.h"
#include "libarch/archive.h"
#include "commands/cmdutils.h"
#include "commands/revisions.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] [[archive/]version]");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) \
  OP (opt_reverse, "r", "reverse", 0, \
      N_("sort from newest to oldest")) \
  OP (opt_full, "f", "full", 0, \
      N_("list fully qualified names")) \
  OP (opt_summary, "s", "summary", 0, \
      N_("print a summary of each patch")) \
  OP (opt_creator, "c", "creator", 0, \
      N_("print the creator of each patch")) \
  OP (opt_date, "D", "date", 0, \
      N_("print the date of each patch"))


t_uchar arch_cmd_revisions_help[] = N_("list the revisions in an archive version\n"
                                     "Print a list of revisions within an archive version.\n"
                                     "\n"
                                     "The list is ordinarily sorted from oldest to newest,\n"
                                     "but the order can be changed with -r (--reverse).\n"
                                     "\n"
                                     "With optional arguments specifying patches, list only those\n"
                                     "patches, if they exist.  If a listed patch does not exist,\n"
                                     "exit with status 1.  The -r (--reverse) flag has no effect\n"
                                     "with optional arguments.\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_revisions (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  int reverse;
  int full;
  int summary;
  int creator;
  int date;
  int any_headers;


  reverse = 0;
  full = 0;
  summary = 0;
  creator = 0;
  date = 0;
  any_headers = 0;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_revisions_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_reverse:
          {
            reverse = 1;
            break;
          }

        case opt_full:
          {
            full = 1;
            break;
          }

        case opt_summary:
          {
            summary = 1;
            any_headers = 1;
            break;
          }
        case opt_date:
          {
            date = 1;
            any_headers = 1;
            break;
          }
        case opt_creator:
          {
            creator = 1;
            any_headers = 1;
            break;
          }
        }
    }

  if (argc > 2)
    goto usage_error;


  {
    t_uchar * version_spec = 0;
    t_uchar * archive = 0;
    t_uchar * version = 0;
    struct arch_archive * arch = 0;
    rel_table revisions = 0;
    
    if (argc == 2)
      {
        version_spec = str_save (0, argv[1]);
      }
    else
      {
	t_uchar * default_archive = arch_my_default_archive (NULL);
        version_spec = arch_try_tree_version (program_name);
	version_spec = str_replace (version_spec, arch_fully_qualify (default_archive, version_spec));
	lim_free (0, default_archive);
      }

    arch = arch_archive_connect_branch (version_spec, &version);
    
    if (!arch)
      {
	safe_printfmt (2, _("Could not connect to archive %s\n"), version_spec);
	exit (2);
      }

    if (!arch_valid_package_name (version, arch_maybe_archive, arch_req_version, 0))
      {
        safe_printfmt (2, "%s: invalid version name (%s)\n",
                       argv[0], version);
        exit (2);
      }
    
    version = str_replace (version, arch_parse_package_name (arch_ret_package_version, 0, version));

    if (argc == 2)
      arch_check_for (arch, arch_req_version, version);
    revisions = arch_archive_revisions (arch, version, full);

    if (reverse)
      {
        if (full)
          arch_sort_table_by_name_field (1, revisions, 0);
        else
          arch_sort_table_by_patch_level_field (1, revisions, 0);
      }

    if (!any_headers)
      rel_print_table (1, revisions);
    else
      {
        int x;

        for (x = 0; x < rel_n_records (revisions); ++x)
          {
            t_uchar * package_patch_level = 0;
            t_uchar * log;
            assoc_table headers = 0;
            t_uchar * body = 0;

            if (full)
              package_patch_level = arch_parse_package_name (arch_ret_non_archive, 0, revisions[x][0]);
            else
              package_patch_level = str_alloc_cat_many (0, version, "--", revisions[x][0], str_end);

            log = arch_archive_log (arch, package_patch_level);

            arch_parse_log (0, &headers, &body, log);

            safe_printfmt (1, "%s\n", revisions[x][0]);
            if (date)
              {
                t_uchar * d;
                d = assoc_ref (headers, "standard-date");
                safe_printfmt (1, "    %s\n", (d ? d : (t_uchar *)"<no standard-date: header>"));
              }
            if (creator)
              {
                t_uchar * c;
                c = assoc_ref (headers, "creator");
                safe_printfmt (1, "    %s\n", (c ? c : (t_uchar *)"<no creator: header>"));
              }
            if (summary)
              {
                t_uchar * s;
                s = assoc_ref (headers, "summary");
                safe_printfmt (1, "    %s\n", (s ? s : (t_uchar *)"<no summary: header>"));
              }

            safe_flush (1);

            lim_free (0, package_patch_level);
            free_assoc_table (headers);
            lim_free (0, body);
          }


      }

    lim_free (0, version_spec);
    lim_free (0, archive);
    lim_free (0, version);
    arch_archive_close (arch);
    rel_free_table (revisions);
  }

  return 0;
}




/* tag: Tom Lord Tue May 20 15:39:49 2003 (revisions.c)
 */
