#include "beep.h"
#include "main.h"
#include <time.h>
#include "libbeep/util.h"
#include "libbeep/configfile.h"
#include <sys/stat.h>
#include <unistd.h>
#include "playlistwin.h"
#include "pixmaps.h"
#include <string.h>
#include <beep_mini.xpm>

#define FILENAME_COL 3


static gboolean click_play = FALSE;
static void bmp_db_set_icon(GtkWidget * win);

GtkTreePath *current;
gchar *playlist_current_name = NULL;
GtkItemFactory *bmp_db_popup;

GtkWidget *progressbar_db;

static void bmp_db_menu_callback(gpointer data, guint action,
				     GtkWidget * widget);
static gboolean bmp_db_popup_menu(GtkWidget * widget,
				      GdkEventButton * event,
				      gpointer user_data);

enum {
    DIRBROWSER_COLLAPSE,
    DIRBROWSER_COLLAPSE_SEL,
    DIRBROWSER_EXPAND,
    DIRBROWSER_ENQUEUE,
    DIRBROWSER_PLAY,
    DIRBROWSER_NEWPLIST,
    DIRBROWSER_DETACH,
    DIRBROWSER_REMOVE,
    DIRBROWSER_RENAME
};

enum {
    IS_ARTIST,
    IS_ALBUM,
    IS_TRACK
};


GtkAccelGroup *db_accel;


GtkItemFactoryEntry bmp_db_menu_entries[] = {
    {N_("/_Enqueue"), "<alt>e", bmp_db_menu_callback,
     DIRBROWSER_ENQUEUE, "<StockItem>", GTK_STOCK_ADD},
    {N_("/Pl_ay"), "<alt>a", bmp_db_menu_callback, DIRBROWSER_PLAY,
     "<ImageItem>", play_pixbuf},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/E_xpand"), "<alt>x", bmp_db_menu_callback, DIRBROWSER_EXPAND,
     "<StockItem>", GTK_STOCK_GO_FORWARD},
    {N_("/_Collapse"), "<alt>c", bmp_db_menu_callback,
     DIRBROWSER_COLLAPSE_SEL, "<StockItem>", GTK_STOCK_GO_BACK},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/Detach"), NULL, bmp_db_menu_callback, DIRBROWSER_DETACH,
     "<Item>",},
    {N_("/Remove"), NULL, bmp_db_menu_callback, DIRBROWSER_REMOVE,
     "<Item>",},
    {N_("/Rename"), NULL, bmp_db_menu_callback, DIRBROWSER_RENAME,
     "<Item>",},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/Collapse Tree"), NULL, bmp_db_menu_callback,
     DIRBROWSER_COLLAPSE, "<StockItem>", GTK_STOCK_GOTO_TOP},
    {N_("/-"), NULL, NULL, 0, "<Separator>"},
    {N_("/New Playlist"), NULL, bmp_db_menu_callback,
     DIRBROWSER_NEWPLIST, "<StockItem>", GTK_STOCK_DELETE},

};

static const int bmp_db_menu_entries_num =
    sizeof(bmp_db_menu_entries) / sizeof(bmp_db_menu_entries[0]);


typedef struct {

    gchar *name;
    gchar *path_real;
    GdkPixbuf *image;

} db_node;

GPtrArray *nodes;
guint ctr_array = 0;


static gchar *bmp_db_sanify_tag(gchar * value);
static GtkTreeModel *bmp_db_return_filestore_model_fresh(GList * paths);

/* The function adds a track to the model, specified by artist, album
 * and finally track. It creates artist and album nodes on the fly as
 * needed. Think of it as a "mkdir -p". */

static void bmp_db_mdb_add_track(gchar * arg_artist_p,
				  gchar * arg_album_p, gchar * arg_track_p,
				  guint arg_tracknum, gchar * arg_path,
				  GtkTreeModel * model_sort)
{

    /* DISCLAIMER: This code is intentionally fscked up and repeats
     * everything all over. */

    /* FIXME: should use a hash table for faster and cleaner searches */

    gchar *arg_artist = bmp_db_sanify_tag(arg_artist_p);
    gchar *arg_album = bmp_db_sanify_tag(arg_album_p);
    gchar *arg_track = bmp_db_sanify_tag(arg_track_p);

    GdkPixbuf *buf_artist;
    GdkPixbuf *buf_album;
    GdkPixbuf *buf_file;

    GtkTreeIter iter;
    GtkTreeIter iter_albums;
    GtkTreeIter iter_tracks;

    GtkTreeModel *model =
	gtk_tree_model_sort_get_model(GTK_TREE_MODEL_SORT(model_sort));

    gboolean found = FALSE;
    gboolean valid = TRUE;

    gchar *value;

    /* First find the artist */
    if (gtk_tree_model_get_iter_first(model, &iter)) {
	while (valid) {
	    gtk_tree_model_get(model, &iter, 1, &value, -1);

	    if (g_ascii_strcasecmp(value, arg_artist) == 0) {
		found = TRUE;
		break;
	    }

	    valid = gtk_tree_model_iter_next(model, &iter);
	}
    }

    /* The artist doesn't exist yet */
    if (!found) {
	buf_artist = gdk_pixbuf_new_from_inline(sizeof(artist_pixbuf),
						artist_pixbuf, TRUE, NULL);

	gtk_tree_store_append(GTK_TREE_STORE(model), &iter, NULL);
	gtk_tree_store_set(GTK_TREE_STORE(model), &iter,
			   0, buf_artist,
			   1, arg_artist,
			   2, NULL, 3, NULL, 4, IS_ARTIST, -1);

	g_object_unref(buf_artist);
    }
    /* Reset/re-use found status */
    found = FALSE;
    valid = TRUE;

    if (gtk_tree_model_iter_children(model, &iter_albums, &iter)) {
	while (valid) {
	    gtk_tree_model_get(model, &iter_albums, 1, &value, -1);

	    if (g_ascii_strcasecmp(value, arg_album) == 0) {
		g_free(value);
		found = TRUE;
		break;
	    }

	    g_free(value);
	    valid = gtk_tree_model_iter_next(model, &iter_albums);
	}
    }
    /* The album doesn't exist yet */
    if (!found) {
	buf_album = gdk_pixbuf_new_from_inline(sizeof(cda_pixbuf),
					       cda_pixbuf, TRUE, NULL);

	gtk_tree_store_append(GTK_TREE_STORE(model), &iter_albums, &iter);
	gtk_tree_store_set(GTK_TREE_STORE(model), &iter_albums,
			   0, buf_album,
			   1, arg_album,
			   2, NULL, 3, NULL, 4, IS_ALBUM, -1);

	g_object_unref(buf_album);

    }

    /* Reset/re-use found status */
    found = FALSE;
    valid = FALSE;

    if (gtk_tree_model_iter_children(model, &iter_tracks, &iter_albums)) {
	while (valid) {
	    gtk_tree_model_get(model, &iter_tracks, 1, &value, -1);

	    if (g_ascii_strcasecmp(value, arg_track) == 0) {
		g_free(value);
		found = TRUE;
		break;
	    }

	    g_free(value);
	    valid = gtk_tree_model_iter_next(model, &iter_tracks);
	}
    }

    /* The track doesn't exist yet */
    if (!found) {
	buf_file = gdk_pixbuf_new_from_inline(sizeof(audio_pixbuf),
					      audio_pixbuf, TRUE, NULL);

	gtk_tree_store_append(GTK_TREE_STORE(model), &iter_tracks,
			      &iter_albums);
	gtk_tree_store_set(GTK_TREE_STORE(model), &iter_tracks, 0,
			   buf_file, 1, arg_track, 2, arg_tracknum, 3,
			   arg_path, 4, IS_TRACK, -1);

	g_object_unref(buf_file);

    }
}


static gboolean bmp_db_check_file(gchar * filename)
{

    /* If this isn't a directory or regular file return failure */	
    if (!g_file_test(filename, G_FILE_TEST_IS_REGULAR | G_FILE_TEST_IS_DIR)) {
        return FALSE;
    }

    /* Check if any plugin would accept this file */
    if (!input_check_file(filename, FALSE)) {
	return FALSE;
    }

    return TRUE;
}

static guint64 bmp_db_get_nodeidx_from_iter(GtkTreeModel * model,
					    GtkTreeIter * iter,
					    guint nodehidx,
					    gboolean * error)
{

    GtkTreePath *path = gtk_tree_model_get_path(model, iter);

    guint64 node_idx;
    gchar **branches;
    error = FALSE;

    branches = g_strsplit(gtk_tree_path_to_string(path), ":", 0);
    node_idx = g_ascii_strtoull(branches[nodehidx], NULL, 10);

    return node_idx;
}

static gchar *bmp_db_sanify_tag(gchar * value)
{

    if (!value)
	return "(unknown)";

    gchar *frag = NULL;
    gchar **branches;
    guint ctr = 0;
    gboolean bracket_open = FALSE;

    GString *chunk = g_string_new(NULL);
    branches = g_strsplit(g_strstrip(value), " ", 0);

    while (frag || (ctr == 0)) {

	if (ctr != 0) {
	    g_string_append_c(chunk, ' ');
	}

	frag = branches[ctr];

	if (frag) {
	    gchar *tmp = g_ascii_strdown(frag, -1);
	    tmp[0] = g_ascii_toupper(tmp[0]);

	    if ((bracket_open)
		&& (g_strstr_len(tmp, strlen(tmp), ")") != NULL)) {
		bracket_open = FALSE;
	    } else if (g_strstr_len(tmp, strlen(tmp), "(") == NULL) {
		g_string_append(chunk, tmp);
	    } else {
		bracket_open = TRUE;
	    }
	}

	ctr++;

    }

    /* FIXME: is chunk freed somewhere? */
    return g_strstrip(chunk->str);
}


static GList *bmp_db_get_paths_from_selection(GtkTreeView * data)
{
    GtkTreeSelection *selection;
    GtkTreeModel *model;
    selection = gtk_tree_view_get_selection(data);
    model = gtk_tree_view_get_model(data);
    return gtk_tree_selection_get_selected_rows(selection, &model);
}

static gchar *bmp_db_get_path_real(gchar * path, GtkTreeModel * model,
				       GtkTreeIter * iter)
{

    GtkTreePath *current;
    guint64 node_idx;
    db_node *node;
    gboolean error;

    current = gtk_tree_model_get_path(model, iter);
    node_idx = bmp_db_get_nodeidx_from_iter(model, iter, 0, &error);
    node = g_ptr_array_index(nodes, node_idx);

    return g_build_filename(node->path_real, path, NULL);
}


static void bmp_db_create_dirtree(GtkTreeModel * model, gchar * from,
				      GtkTreeIter parent_iter)
{

    GList *dirs = NULL;
    GList *files = NULL;
    DIR *dir;
    struct dirent *dirent;
    struct stat statbuf;

    GtkTreeIter iter;
    GtkTreeIter iter_child;
    GList *list = NULL;
    gboolean first = TRUE;
    gchar *from_real;

    from = g_locale_from_utf8(from, -1, NULL, NULL, NULL);
    from_real = bmp_db_get_path_real(from, model, &parent_iter);
    g_free(from);
    
    /* It's not really a waste creating both beforehand since we will
     * certainly need the folder one at least once, and most probably
     * the audio one too anyway */
    GdkPixbuf *buf_dir = gdk_pixbuf_new_from_inline(sizeof(folder_pixbuf),
						    folder_pixbuf,
						    TRUE,
						    NULL);

    GdkPixbuf *buf_file = gdk_pixbuf_new_from_inline(sizeof(audio_pixbuf),
						     audio_pixbuf,
						     TRUE,
						     NULL);

    GdkPixbuf *buf_cda = gdk_pixbuf_new_from_inline(sizeof(cda_pixbuf),
						    cda_pixbuf,
						    TRUE,
						    NULL);


    stat(from_real, &statbuf);
    if ((statbuf.st_mode & S_IFDIR) != S_IFDIR)
	return;

    if ((list = input_scan_dir(from_real)) != NULL) {
	/* We enter a directory that has been "hijacked" by an
	 * input-plugin. This is used by the CDDA plugin */

	/* Sort of simplified  */
	files = list;
    } else {
	if ((dir = opendir(from_real)) != NULL) {
	    while ((dirent = readdir(dir)) != NULL) {
		char *name_real;
		/* FIXME: Excludes hidden dirs too (do we want them anyway?) */
		if (dirent->d_name[0] == '.')
		    continue;
		name_real =
		    g_build_filename(from_real, dirent->d_name, NULL);
		stat(name_real, &statbuf);

		if (S_ISDIR(statbuf.st_mode)) {
		    dirs =
			g_list_append(dirs,
				      g_locale_to_utf8(dirent->d_name, -1,
						       NULL, NULL, NULL));
		}

	    }
	    closedir(dir);
	}

    }

    if (dirs)
	dirs = g_list_sort(dirs, (GCompareFunc) g_ascii_strcasecmp);

    /* The following code is kinda wacky but i rather perform all
     * these check than to see the browser fsck up because of
     * something... */

    while (dirs) {
	if (first) {
	    if (gtk_tree_model_iter_has_child(model, &parent_iter)) {
		gtk_tree_model_iter_nth_child(model,
					      &iter, &parent_iter, 0);
	    } else {
		gtk_tree_store_append(GTK_TREE_STORE(model), &iter,
				      &parent_iter);
	    }
	    first = FALSE;
	} else {
	    gtk_tree_store_append(GTK_TREE_STORE(model), &iter,
				  &parent_iter);
	}

	gtk_tree_store_set(GTK_TREE_STORE(model), &iter,
			   0, buf_dir, 1, dirs->data, -1);

	gtk_tree_store_append(GTK_TREE_STORE(model), &iter_child, &iter);
	gtk_tree_store_set(GTK_TREE_STORE(model), &iter_child,
			   0, NULL, 1, NULL, -1);

	dirs = dirs->next;
    }


    /* We're still first, nothing has been added, so remove that empty row */
    if (first) {
	gtk_tree_model_iter_nth_child(model, &iter, &parent_iter, 0);

	gtk_tree_store_remove(GTK_TREE_STORE(model), &iter);

    }

    g_list_free(list);
    g_list_free(files);
    g_list_free(dirs);

    g_object_unref(buf_dir);
    g_object_unref(buf_file);
    g_object_unref(buf_cda);

}


static GtkTreeModel *bmp_db_create_model(void)
{
    GtkTreeStore *model;
    GtkTreeIter iter;
    db_node *node;
    gchar *filename;
    FILE *file;
    gchar *buffer, **lines, **frags;
    struct stat stats;
    guint i;
    GdkPixbuf *buf;

    model = gtk_tree_store_new(2, GDK_TYPE_PIXBUF, G_TYPE_STRING);
    buf = gdk_pixbuf_new_from_inline(sizeof(harddisk_pixbuf),
				     harddisk_pixbuf, TRUE, NULL);

    /* create root node */
    /* add data to the tree store */

    nodes = g_ptr_array_new();

    filename =
	g_build_filename(g_get_home_dir(), BMP_RCPATH, "dbrc", NULL);

    if (!(file = fopen(filename, "r"))) {
	/* Seems the file doesn't exist so give at least the root filesystem */
	if (!(file = fopen(filename, "w"))) {
	    /* Ok now we really give up */
	    return NULL;
	}
	fprintf(file, "/:/\n");
	/*  not so cool  */
	fclose(file);
	file = fopen(filename, "r");
    }

    lstat(filename, &stats);
    buffer = g_malloc(stats.st_size + 1);
    if (fread(buffer, 1, stats.st_size, file) != stats.st_size) {
	g_free(buffer);
	fclose(file);
	return NULL;
    }
    fclose(file);
    buffer[stats.st_size] = '\0';
    lines = g_strsplit(buffer, "\n", 0);
    g_free(buffer);

    i = 0;
    while (lines[i]) {
	if (strlen(lines[i])) {

	    frags = g_strsplit(lines[i], ":", 0);

	    node = g_malloc0(sizeof(db_node));
	    node->name = g_strdup(frags[0]);
	    node->path_real = g_strdup(frags[1]);
	    node->image = buf;
	    g_ptr_array_add(nodes, node);
	    ctr_array++;
	    gtk_tree_store_append(model, &iter, NULL);
	    gtk_tree_store_set(model, &iter,
			       0, node->image, 1, node->name, -1);
	    /* caveat, the root in bmp_db_create_dirtree is relative */
	    bmp_db_create_dirtree(GTK_TREE_MODEL(model),
				      /* starting at */ "/", iter);
	    g_strfreev(frags);
	}
	i++;
    }

    g_strfreev(lines);

    g_object_unref(buf);
    return GTK_TREE_MODEL(model);
}

static void bmp_db_add_columns(GtkTreeView * treeview)
{
    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;


    renderer = gtk_cell_renderer_pixbuf_new();
    column = gtk_tree_view_column_new();
    gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
    gtk_tree_view_column_pack_start(column, renderer, FALSE);
    gtk_tree_view_column_set_attributes(column, renderer, "pixbuf", 0,
					NULL);
    gtk_tree_view_append_column(treeview, column);

    gtk_tree_view_column_set_spacing(column, 4);
    renderer = gtk_cell_renderer_text_new();
    gtk_tree_view_column_pack_start(column, renderer, TRUE);
    gtk_tree_view_column_set_attributes(column, renderer, "text", 1, NULL);



}


static gchar *bmp_db_path_get_full(GtkTreeModel * model,
				       const gchar * path,
				       GtkTreeIter iter_current)
{

    GtkTreeIter iter_parent;
    GtkTreeIter iter_root;
    db_node *node;

    gchar *str_parent = NULL;

    if (gtk_tree_model_iter_parent(model, &iter_parent, &iter_current)) {
	if (!gtk_tree_model_iter_parent(model, &iter_root, &iter_parent)) {

	    node = g_ptr_array_index(nodes, 0);

	    if (node)
		str_parent = g_strdup(node->path_real);

	} else {
	    gtk_tree_model_get(model, &iter_parent, 1, &str_parent, -1);
	}

	if (str_parent) {
	    if (str_parent[0] == '/')
		return g_build_filename("/", path, NULL);
	    else
		return
		    g_build_filename(bmp_db_path_get_full
				     (model, str_parent, iter_parent),
				     path, NULL);
	}
    }

    return g_strdup("");
}


static void bmp_db_row_expanded(GtkTreeView * treeview,
				    GtkTreeIter * arg1,
				    GtkTreePath * arg2, gpointer user_data)
{
    GtkTreeIter iter_current;
    GtkTreeIter iter_child;
    GtkTreeModel *model;
    gchar *str;
    gchar *node_name;
    gchar *path;

    g_free(current);

    current = gtk_tree_path_copy(arg2);
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(treeview));

    gtk_tree_model_get_iter(GTK_TREE_MODEL(model), &iter_current, arg2);
    if (gtk_tree_model_iter_has_child(GTK_TREE_MODEL(model),
				      &iter_current)) {

	gtk_tree_model_iter_children(GTK_TREE_MODEL(model),
				     &iter_child, &iter_current);

	gtk_tree_model_get(GTK_TREE_MODEL(model), &iter_child, 1, &str,
			   -1);

	if (!str) {
	    gtk_tree_model_get(GTK_TREE_MODEL(model), &iter_current, 1,
			       &node_name, -1);
	    path =
		bmp_db_path_get_full(GTK_TREE_MODEL(model), node_name,
					 iter_current);
	    bmp_db_create_dirtree(GTK_TREE_MODEL(model), path,
				      iter_current);
	}

    }
}

static void bmp_db_clear_clicked(GtkWidget * w, gpointer data)
{
    gtk_tree_selection_unselect_all(gtk_tree_view_get_selection
				    (GTK_TREE_VIEW(data)));
}


static void bmp_db_row_activated(GtkTreeSelection * sel,
				     gpointer user_data)
{
    GtkTreeIter iter;
    GList *paths = NULL;
    GList *paths_f = NULL;

    gchar *node_name;
    gchar *path;
    gchar *path_real;

    GtkTreeModel *model;
    GtkTreeView *treeview;

    treeview = gtk_tree_selection_get_tree_view(sel);
    model = gtk_tree_view_get_model(treeview);

    /* FIXME: paths is not freed */
    paths = bmp_db_get_paths_from_selection(treeview);

    gtk_widget_set_sensitive(user_data, FALSE);

    while (paths) {
	gtk_tree_model_get_iter(GTK_TREE_MODEL(model), &iter, paths->data);
	gtk_tree_model_get(GTK_TREE_MODEL(model), &iter, 1, &node_name,
			   -1);
	path =
	    bmp_db_path_get_full(GTK_TREE_MODEL(model), node_name,
				     iter);

	/* FIXME: path is not freed */
	path = g_locale_from_utf8(path, -1, NULL, NULL, NULL);
	path_real = bmp_db_get_path_real(path, model, &iter);

	/* FIXME: paths_f is not freed */
	paths_f = g_list_append(paths_f, path_real);

	paths = paths->next;
    }

    GtkWidget *store =
	GTK_WIDGET(bmp_db_return_filestore_model_fresh(paths_f));

    gtk_tree_view_set_model(user_data, GTK_TREE_MODEL(store));
    gtk_widget_set_sensitive(user_data, TRUE);
}


static void bmp_db_filestore_row_activated(GtkTreeView * treeview,
					   GtkTreePath * arg1,
					   GtkTreeViewColumn * arg2,
					   gpointer user_data)
{



    GtkTreeIter iter;
    GtkTreeIter iter_child;
    GList *paths = NULL;

    gchar *node_name;
    gint entry_type = -1;

    struct stat statbuf;
    GtkTreeModel *model;

    model = gtk_tree_view_get_model(treeview);
    paths = bmp_db_get_paths_from_selection(treeview);
    gtk_tree_model_get_iter(GTK_TREE_MODEL(model), &iter, paths->data);
    gtk_tree_model_get(GTK_TREE_MODEL(model), &iter, FILENAME_COL,
		       &node_name, 4, &entry_type, -1);

    if ((node_name != NULL) && (entry_type == IS_TRACK)) {

	if (click_play)
	    playlist_clear();

	stat(node_name, &statbuf);
	if (S_ISREG(statbuf.st_mode)) {
	    playlist_add_url_string(node_name);
	}

	if (click_play)
	    playlist_play();

	gtk_tree_selection_unselect_all(gtk_tree_view_get_selection
					(treeview));

    } else {

	if (entry_type == IS_ALBUM) {
	    if (gtk_tree_model_iter_children(model, &iter_child, &iter)) {
		if (click_play)
		    playlist_clear();
		gboolean valid = TRUE;
		while (valid) {
		    gtk_tree_model_get(GTK_TREE_MODEL(model), &iter_child,
				       FILENAME_COL, &node_name, -1);
		    if (node_name != NULL) {
			stat(node_name, &statbuf);
			if (S_ISREG(statbuf.st_mode)) {
			    playlist_add_url_string(node_name);
			}
			valid =
			    gtk_tree_model_iter_next(model, &iter_child);
		    }
		}
		if (click_play)
		    playlist_play();
	    }
	}
	gtk_tree_selection_unselect_all(gtk_tree_view_get_selection
					(treeview));
    }
}


static void bmp_db_add_clicked(GtkWidget * w, gpointer data)
{

    GtkTreeIter iter;
    GList *paths = NULL;
    GtkTreeModel *model;

    gchar *node_name;
    gchar *path;
    gchar *path_real;

    model = gtk_tree_view_get_model(data);

    paths = bmp_db_get_paths_from_selection(data);

    while (paths) {
	struct stat statbuf;

	gtk_tree_model_get_iter(GTK_TREE_MODEL(model), &iter, paths->data);
	gtk_tree_model_get(GTK_TREE_MODEL(model), &iter, 1, &node_name,
			   -1);

	path =
	    bmp_db_path_get_full(GTK_TREE_MODEL(model), node_name,
				     iter);
	path = g_locale_from_utf8(path, -1, NULL, NULL, NULL);
	path_real = bmp_db_get_path_real(path, model, &iter);

	stat(path, &statbuf);
	if (S_ISDIR(statbuf.st_mode)) {
	    playlist_add_dir(path_real);
	} else {
	    playlist_add_url_string(path_real);
	}

	paths = paths->next;

    }

    g_free(paths);
    bmp_db_clear_clicked(NULL, data);

}

static void bmp_db_expand_selected(GtkWidget * data)
{
    GList *paths = NULL;
    /* paths is not freed */
    paths = bmp_db_get_paths_from_selection(GTK_TREE_VIEW(data));
    while (paths) {
	gtk_tree_view_expand_to_path(GTK_TREE_VIEW(data), paths->data);
	paths = paths->next;
    }

}

static void bmp_db_detatch_selected_entry_kp(GtkWidget * w,
						 GtkDialog * dialog)
{
    gtk_dialog_response(dialog, GTK_RESPONSE_ACCEPT);
}

static void bmp_db_detach_selected(GtkWidget * data)
{

    GtkWidget *dialog;
    GtkWidget *entry;
    GtkLabel *variable;

    GtkTreeIter iter;
    GList *paths = NULL;
    GtkTreeSelection *selection;
    GtkTreeModel *model;

    PangoAttrList *attrs;
    PangoAttribute *attr;

    GdkPixbuf *buf;

    gchar *node_name;
    gchar *path;
    gchar *path_real;

    struct stat statbuf;

    attrs = pango_attr_list_new();
    attr = pango_attr_weight_new(PANGO_WEIGHT_BOLD);
    attr->start_index = 0;
    attr->end_index = -1;
    pango_attr_list_insert(attrs, attr);

    buf = gdk_pixbuf_new_from_inline(sizeof(harddisk_pixbuf),
				     harddisk_pixbuf, TRUE, NULL);

    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(data));
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(data));

    paths =
	gtk_tree_selection_get_selected_rows(GTK_TREE_SELECTION(selection),
					     &model);

    variable = GTK_LABEL(gtk_label_new("N/A"));
    gtk_misc_set_alignment(GTK_MISC(variable), 0.0, 0.0);
    gtk_label_set_attributes(GTK_LABEL(variable), attrs);
    gtk_label_set_attributes(GTK_LABEL(variable), attrs);

    dialog = gtk_dialog_new_with_buttons("Choose a Name",
					 GTK_WINDOW(mainwin),
					 GTK_DIALOG_NO_SEPARATOR,
					 GTK_STOCK_OK, GTK_RESPONSE_ACCEPT,
					 NULL);

    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox),
		       GTK_WIDGET(variable), TRUE, TRUE, 8);
    gtk_widget_show(GTK_WIDGET(variable));

    entry = gtk_entry_new();
    g_signal_connect(entry, "activate",
		     G_CALLBACK(bmp_db_detatch_selected_entry_kp),
		     dialog);
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), entry, TRUE,
		       TRUE, 8);
    gtk_entry_set_editable(GTK_ENTRY(entry), TRUE);
    gtk_widget_show(GTK_WIDGET(entry));
    gtk_window_set_position(GTK_WINDOW(dialog), GTK_WIN_POS_CENTER);
    gtk_widget_set_size_request(GTK_WIDGET(dialog), 400, 160);

    while (paths) {
	gtk_tree_model_get_iter(GTK_TREE_MODEL(model), &iter, paths->data);
	gtk_tree_model_get(GTK_TREE_MODEL(model), &iter, 1, &node_name,
			   -1);

	path =
	    bmp_db_path_get_full(GTK_TREE_MODEL(model), node_name,
				     iter);
	path_real = bmp_db_get_path_real(path, model, &iter);

	stat(path_real, &statbuf);
	if (S_ISDIR(statbuf.st_mode)) {

	    db_node *node;
	    gchar *dir = g_strdup(path_real);
	    gchar *name = g_strdup(path_real);

	    node = g_malloc0(sizeof(db_node));
	    node->path_real = dir;

	    gtk_label_set_text(variable,
			       g_strconcat("Choose a Name for:\n", name,
					   NULL));
	    gtk_dialog_run(GTK_DIALOG(dialog));
	    node->name = g_strdup(gtk_entry_get_text(GTK_ENTRY(entry)));
	    if ((node->name == NULL) || (node->name[0] == '\0'))
		node->name = name;
	    gtk_widget_destroy(dialog);

	    node->image = buf;
	    g_ptr_array_add(nodes, node);
	    ctr_array++;

	    if (gtk_tree_model_get_iter_from_string(model, &iter, "0")) {
		gtk_tree_store_append(GTK_TREE_STORE(model), &iter, NULL);
		gtk_tree_store_set(GTK_TREE_STORE(model), &iter,
				   0, node->image, 1, node->name, -1);
		bmp_db_create_dirtree(GTK_TREE_MODEL(model),
					  /* starting at */ "/", iter);
	    } else {
		/* FIXME: response to failed to get iter */
	    }

	} else {
	    /* FIXME: response to is a file, not dir */
	}

	paths = paths->next;
    }

    g_free(paths);
    g_object_unref(buf);

}


static void bmp_db_remove_attached(GtkWidget * data)
{

    GtkTreeIter iter;
    GList *paths = NULL;
    GtkTreeSelection *selection;
    GtkTreeModel *model;
    gboolean error;

    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(data));
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(data));

    paths =
	gtk_tree_selection_get_selected_rows(GTK_TREE_SELECTION(selection),
					     &model);

    while (paths) {

	GtkTreeIter parent;
	guint64 node_idx;
	g_assert(gtk_tree_model_get_iter
		 (GTK_TREE_MODEL(model), &iter, paths->data));
	node_idx = bmp_db_get_nodeidx_from_iter(model, &iter, 0, &error);


	if (!gtk_tree_model_iter_parent(model, &parent, &iter)) {
	    if (gtk_tree_model_get_iter_from_string
		(model, &iter, gtk_tree_path_to_string(paths->data))) {
		db_node *node;
		gtk_tree_store_remove(GTK_TREE_STORE(model), &iter);

		node = g_ptr_array_index(nodes, node_idx);
		g_ptr_array_remove_index(nodes, node_idx);
		ctr_array--;
		g_free(node);

	    } else {
		/* FIXME: response to failed to get iter */
	    }

	} else {
	    /* FIXME: response to is not a root node */
	}

	paths = paths->next;
    }

}


static void bmp_db_collapse_selected(GtkWidget * data)
{
    GList *paths = NULL;
    GtkTreeSelection *selection;
    GtkTreeModel *model;

    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(data));
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(data));

    paths =
	gtk_tree_selection_get_selected_rows(GTK_TREE_SELECTION(selection),
					     &model);

    while (paths) {
	gtk_tree_view_collapse_row(GTK_TREE_VIEW(data), paths->data);
	paths = paths->next;
    }
    g_free(paths);
}

static void bmp_db_play_clicked(GtkWidget * w, gpointer data)
{

    GtkTreeIter iter;
    GList *paths = NULL;
    GtkTreeSelection *selection;
    GtkTreeModel *model;
    selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(data));
    model = gtk_tree_view_get_model(GTK_TREE_VIEW(data));

    /* paths is not freed */
    paths =
	gtk_tree_selection_get_selected_rows(GTK_TREE_SELECTION(selection),
					     &model);

    playlist_clear();

    while (paths) {

	char *node_name;
	char *path;
	char *path_real;
	struct stat statbuf;

	gtk_tree_model_get_iter(GTK_TREE_MODEL(model), &iter, paths->data);
	gtk_tree_model_get(GTK_TREE_MODEL(model), &iter, 1, &node_name,
			   -1);

	path =
	    bmp_db_path_get_full(GTK_TREE_MODEL(model), node_name,
				     iter);
	path_real = bmp_db_get_path_real(path, model, &iter);

	stat(path, &statbuf);
	if (S_ISDIR(statbuf.st_mode)) {
	    playlist_add_dir(path_real);
	} else {
	    playlist_add_url_string(path_real);
	}

	paths = paths->next;

    }

    bmp_db_clear_clicked(NULL, data);
    playlist_play();

}

static void rb_enq_clicked(GtkWidget * w, gpointer data)
{
    click_play = FALSE;
}

static void rb_play_clicked(GtkWidget * w, gpointer data)
{
    click_play = TRUE;
}

static void bmp_db_add_expanded_paths(GtkWidget * treeview,
				    GtkTreePath * path, gpointer data)
{
    data = g_list_append(data, gtk_tree_path_to_string(path));
}

static void bmp_db_save_setup(GtkWidget * w, gpointer data)
{

    gchar *filename;
    FILE *file;
    db_node *node;
    guint i;

    filename =
	g_build_filename(g_get_home_dir(), BMP_RCPATH, "dbrc", NULL);

    if (!(file = fopen(filename, "w")))
	return;

    i = 0;

    while (i < ctr_array) {

	node = g_ptr_array_index(nodes, i);
	fprintf(file, "%s:%s\n", node->name, node->path_real);
	i++;
    }

    g_ptr_array_free(nodes, TRUE);
    fclose(file);
}

static gboolean bmp_db_apply_setup(GtkTreeView * treeview)
{
    /* I don't think it's reasonable to write this into the main config file */

/*     gchar * filename; */
/*     filename = g_strconcat(g_get_home_dir(), "/.beep/dbrc", NULL);	 */

/*     FILE *file; */
/*     gchar *buffer, **lines; */
/*     struct stat stats; */
/*     guint i; */

/*     if (lstat(filename, &stats) == -1) */
/* 	 return NULL; */
/*     if (!(file = fopen(filename, "r"))) */
/* 	 return NULL; */

/*     buffer = g_malloc(stats.st_size + 1); */
/*     if (fread(buffer, 1, stats.st_size, file) != stats.st_size) */
/*     { */
/* 	 g_free(buffer); */
/* 	 fclose(file); */
/* 	 return NULL; */
/*     } */
/*     fclose(file); */
/*     buffer[stats.st_size] = '\0'; */
/*     lines = g_strsplit(buffer, "\n", 0); */
/*     g_free(buffer); */

/*     current = gtk_tree_path_new_from_string(lines[0]); */
/*     gtk_tree_view_scroll_to_cell    (GTK_TREE_VIEW(treeview), */
/* 				     current, */
/* 				     gtk_tree_view_get_column(GTK_TREE_VIEW(treeview),0), */
/* 				     TRUE, */
/* 				     0.1, */
/* 				     0.0); */
/*     i = 1; */
/*     while (lines[i]) { */
/* 	 if (strlen(lines[i])) { */
/* 	      gtk_tree_view_expand_to_path(GTK_TREE_VIEW(treeview), */
/* 					   gtk_tree_path_new_from_string(lines[i])); */
/* 	      while(gtk_events_pending()){gtk_main_iteration();}; */
/* 	 } */
/* 	 i++; */
/*     } */

/*     g_strfreev(lines); */

    return TRUE;
}


static void bmp_db_rescan_tree(GtkWidget * w, gpointer data)
{

    GtkTreeModel *model_new;
    model_new = bmp_db_create_model();
    gtk_tree_view_set_model(GTK_TREE_VIEW(data),
			    GTK_TREE_MODEL(model_new));
    gtk_widget_set_sensitive(data, TRUE);
}

static GtkWidget *bmp_db_return_treeview_fresh(void)
{

    GtkTreeModel *model;
    GtkWidget *treeview;
    GtkTreeSelection *l_selection;

    model = bmp_db_create_model();

    treeview = gtk_tree_view_new_with_model(model);
    g_object_unref(model);

    bmp_db_add_columns(GTK_TREE_VIEW(treeview));

    l_selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
    gtk_tree_selection_set_mode(l_selection, GTK_SELECTION_MULTIPLE);

    g_object_set(G_OBJECT(treeview), "headers-visible", FALSE, NULL);
    g_signal_connect(G_OBJECT(treeview), "row-expanded",
		     G_CALLBACK(bmp_db_row_expanded), nodes);
    g_signal_connect(G_OBJECT(treeview), "button-press-event",
		     G_CALLBACK(bmp_db_popup_menu), treeview);

    gtk_tree_view_expand_row(GTK_TREE_VIEW(treeview),
			     gtk_tree_path_new_from_string("0"), FALSE);

    gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(treeview), FALSE);
    return treeview;

}

static gint bmp_db_keypress(GtkWidget * grab_widget,
				GdkEventKey * event, gpointer data)
{
    if (event && (event->keyval == GDK_Escape)) {
	gtk_widget_hide(data);
    } else {
	return FALSE;
    }

    return TRUE;
}



static gint bmp_db_mdb_sort_func(GtkTreeModel * model,
			   GtkTreeIter * a, GtkTreeIter * b,
			   gpointer user_data)
{

    gchar *cmp_a;
    gchar *cmp_b;

    guint track_a;
    guint track_b;

    gtk_tree_model_get(model, a, 1, &cmp_a, -1);
    gtk_tree_model_get(model, b, 1, &cmp_b, -1);

    gtk_tree_model_get(model, a, 2, &track_a, -1);
    gtk_tree_model_get(model, b, 2, &track_b, -1);

    if ((track_a != 0) && (track_b != 0)) {
	return (track_a - track_b);
    }

    if ((track_a != 0) && (track_b == 0)) {
	return 1;
    }

    if ((track_a == 0) && (track_b != 0)) {
	return -1;
    }

    /* FIXME: Is this here right? */
    if ((cmp_a == NULL) && (cmp_b == NULL))
	return 0;
    if ((cmp_a == NULL) && (cmp_b != NULL))
	return 1;
    if ((cmp_a != NULL) && (cmp_b == NULL))
	return -1;

    return g_ascii_strcasecmp(cmp_a, cmp_b);
}


static GList *bmp_db_return_files_from_path(gchar * path)
{

    DIR *dir;
    struct dirent *dirent;
    struct stat statbuf;
    GList *list = NULL;
    GList *files = NULL;

    stat(path, &statbuf);
    if ((statbuf.st_mode & S_IFDIR) != S_IFDIR) {
	/* FIXME: Add action for no directory */
	return NULL;
    }

    if ((list = input_scan_dir(path)) != NULL) {
	/*
	 * We enter a directory that has been "hijacked" by an
	 * input-plugin. This is used by the CDDA plugin
	 */

	/* Sort of simplified  */
	files = list;
    } else {
	if ((dir = opendir(path)) != NULL) {
	    while ((dirent = readdir(dir)) != NULL) {
		char *name_real;
		/* FIXME: Excludes hidden dirs too (do we want them anyway?) */
		if (dirent->d_name[0] == '.') {
		    continue;
		}
		name_real =
		    g_strdup(g_strconcat(path, "/", dirent->d_name, NULL));
		stat(name_real, &statbuf);
		if (S_ISREG(statbuf.st_mode)) {
		    if (bmp_db_check_file(name_real)) {
			files = g_list_append(files,
					      g_strdup(g_filename_to_utf8
						       (name_real, -1,
							NULL, NULL,
							NULL)));
		    }
		}
	    }
	    closedir(dir);
	}

    }

    return files;
}


static GtkTreeModel *bmp_db_return_filestore_model_fresh(GList * paths)
{

    GtkTreeStore *file_store;


    file_store =
	gtk_tree_store_new(5, GDK_TYPE_PIXBUF, G_TYPE_STRING, G_TYPE_INT,
			   G_TYPE_STRING, G_TYPE_INT);
    GtkTreeModel *sort_model =
	gtk_tree_model_sort_new_with_model(GTK_TREE_MODEL(file_store));


    gtk_tree_sortable_set_default_sort_func(GTK_TREE_SORTABLE(sort_model),
					    bmp_db_mdb_sort_func, NULL, NULL);


    gdouble length_pl = g_list_length(paths);
    gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(progressbar_db), 0.0);

    int ctr = 0;

    while (paths) {

	gchar *path = paths->data;
	ctr++;

	GList *files = bmp_db_return_files_from_path(path);

	while (files) {
	    gchar *title;
	    gchar *artist;
	    gchar *album;
	    gchar *genre;
	    guint tracknumber;

	    input_return_tag_value_by_name(files->data, &artist, &title,
					   &album, &genre, &tracknumber);

	    if (title == NULL) {

		title = g_strdup(g_basename(files->data));
		title[strlen(title) - 4] = 0;

	    }

	    bmp_db_mdb_add_track(artist, album, title, tracknumber,
				  files->data, sort_model);

	    files = files->next;
	}

	gdouble frac = 1 / length_pl;
	gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(progressbar_db),
				      frac * ctr);
	while (gtk_events_pending()) {
	    gtk_main_iteration();
	};
	paths = paths->next;
    }

    return sort_model;
}

static GtkWidget *bmp_db_return_filestore_fresh(GList * paths)
{

    GtkWidget *treeview;

    GtkCellRenderer *renderer;
    GtkTreeViewColumn *column;

    GtkTreeModel *fstore;

    fstore = bmp_db_return_filestore_model_fresh(paths);

    treeview = gtk_tree_view_new_with_model(GTK_TREE_MODEL(fstore));
    gtk_tree_view_set_rules_hint(GTK_TREE_VIEW(treeview), FALSE);

    renderer = gtk_cell_renderer_pixbuf_new();
    column = gtk_tree_view_column_new();
    gtk_tree_view_column_set_sizing(column, GTK_TREE_VIEW_COLUMN_AUTOSIZE);
    gtk_tree_view_column_pack_start(column, renderer, FALSE);
    gtk_tree_view_column_set_attributes(column, renderer, "pixbuf", 0,
					NULL);
    gtk_tree_view_append_column(GTK_TREE_VIEW(treeview), column);

    gtk_tree_view_column_set_spacing(column, 4);
    renderer = gtk_cell_renderer_text_new();
    gtk_tree_view_column_pack_start(column, renderer, TRUE);
    gtk_tree_view_column_set_attributes(column, renderer, "text", 1, NULL);

    g_object_set(G_OBJECT(treeview), "headers-visible", FALSE, NULL);
    return treeview;

}

GtkWidget *bmp_db_create(gchar * current_path)
{
    static GtkWidget *window;
    GtkWidget *vbox_right;
    GtkWidget *vbox_left;
    GtkWidget *frame;
    GtkWidget *sw, *sw_right;
    GtkWidget *treeview = NULL;
    GtkWidget *label_header;
    GtkWidget *paned;

    if (!window) {

	/* create window, etc */
	/* yes this is all messed up */

	PangoAttrList *attrs;
	PangoAttribute *attr;

	window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
	gtk_window_set_title(GTK_WINDOW(window), "bmp :: Add Files...");
	gtk_container_set_border_width(GTK_CONTAINER(window), 8);
	gtk_window_set_destroy_with_parent(GTK_WINDOW(window), TRUE);
	frame = gtk_frame_new(NULL);
	gtk_frame_set_shadow_type(GTK_FRAME(frame), GTK_SHADOW_ETCHED_IN);

	vbox_left = gtk_vbox_new(FALSE, 0);
	vbox_right = gtk_vbox_new(FALSE, 0);

	attrs = pango_attr_list_new();
	attr = pango_attr_weight_new(PANGO_WEIGHT_BOLD);
	attr->start_index = 0;
	attr->end_index = -1;
	pango_attr_list_insert(attrs, attr);

	label_header = gtk_label_new("Filesystem:");
	gtk_label_set_attributes(GTK_LABEL(label_header), attrs);
	gtk_misc_set_alignment(GTK_MISC(label_header), 0.0, 0.0);


	treeview = bmp_db_return_treeview_fresh();


	sw = gtk_scrolled_window_new(NULL, NULL);
	sw_right = gtk_scrolled_window_new(NULL, NULL);

	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_ALWAYS);
	gtk_scrolled_window_set_shadow_type
	    (GTK_SCROLLED_WINDOW(sw), GTK_SHADOW_ETCHED_IN);

	gtk_scrolled_window_set_policy(GTK_SCROLLED_WINDOW(sw_right),
				       GTK_POLICY_AUTOMATIC,
				       GTK_POLICY_ALWAYS);
	gtk_scrolled_window_set_shadow_type
	    (GTK_SCROLLED_WINDOW(sw_right), GTK_SHADOW_ETCHED_IN);


	progressbar_db = gtk_progress_bar_new();
	gtk_progress_bar_set_orientation
	    (GTK_PROGRESS_BAR(progressbar_db), GTK_PROGRESS_LEFT_TO_RIGHT);
	gtk_box_pack_start(GTK_BOX(vbox_left), sw, TRUE, TRUE, 4);
	gtk_box_pack_start(GTK_BOX(vbox_right), sw_right, TRUE, TRUE, 4);
	gtk_box_pack_start(GTK_BOX(vbox_right), progressbar_db, FALSE,
			   FALSE, 4);

	paned = gtk_hpaned_new();
	gtk_paned_add1(GTK_PANED(paned), vbox_left);
	gtk_paned_add2(GTK_PANED(paned), vbox_right);
	gtk_container_add(GTK_CONTAINER(frame), paned);


	gtk_container_add(GTK_CONTAINER(window), frame);


	gtk_widget_set_events(window,
			      GDK_FOCUS_CHANGE_MASK |
			      GDK_BUTTON_MOTION_MASK |
			      GDK_BUTTON_PRESS_MASK |
			      GDK_BUTTON_RELEASE_MASK | GDK_SCROLL_MASK |
			      GDK_KEY_PRESS_MASK |
			      GDK_VISIBILITY_NOTIFY_MASK);

	g_signal_connect(G_OBJECT(window), "key_press_event",
			 G_CALLBACK(bmp_db_keypress), window);

	gtk_container_add(GTK_CONTAINER(sw), treeview);
	GtkWidget *fstore;

	/* FIXME: clean up variables, move them to start */

	/* FIXME: paths is not freed, and is resetted every iteration */
	GList *paths = NULL;
	paths = g_list_append(paths, g_strdup(g_get_home_dir()));
	fstore = bmp_db_return_filestore_fresh(paths);
	gtk_container_add(GTK_CONTAINER(sw_right), fstore);

	GtkTreeSelection *sel =
	    gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
	g_signal_connect(G_OBJECT(sel), "changed",
			 G_CALLBACK(bmp_db_row_activated), fstore);
	g_signal_connect(G_OBJECT(fstore), "row-activated",
			 G_CALLBACK(bmp_db_filestore_row_activated), NULL);

/* 	bmp_db_apply_setup(GTK_TREE_VIEW(treeview)); */

	gtk_window_set_default_size(GTK_WINDOW(window), 650, 600);
	gtk_window_set_position(GTK_WINDOW(window), GTK_WIN_POS_CENTER);

    }
    gtk_widget_realize(window);
    bmp_db_set_icon(window);

    db_accel = gtk_accel_group_new();

    bmp_db_popup = gtk_item_factory_new(GTK_TYPE_MENU,
					    "<Main>", db_accel);

    gtk_item_factory_create_items(GTK_ITEM_FACTORY(bmp_db_popup),
				  bmp_db_menu_entries_num,
				  bmp_db_menu_entries, NULL);

    gtk_window_add_accel_group(GTK_WINDOW(window), db_accel);

    return window;
}

static void bmp_db_set_icon(GtkWidget * win)
{
    static GdkPixmap *icon;
    static GdkBitmap *mask;
    GdkAtom icon_atom;
    glong data[2];

    if (!icon)
	icon =
	    gdk_pixmap_create_from_xpm_d(win->window, &mask,
					 &win->style->bg[GTK_STATE_NORMAL],
					 beep_mini_xpm);
    data[0] = GDK_WINDOW_XWINDOW(icon);
    data[1] = GDK_WINDOW_XWINDOW(mask);

    icon_atom = gdk_atom_intern((const gchar *) "KWM_WIN_ICON", FALSE);
    gdk_property_change(win->window, icon_atom, icon_atom, 32,
			GDK_PROP_MODE_REPLACE, (guchar *) data, 2);
    gdk_window_set_icon(win->window, NULL, icon, mask);
    gdk_window_set_icon_name(win->window, "bmp: Add Files...");
    gdk_window_set_title(win->window, "bmp: Add Files...");
    gdk_window_set_group(win->window, mainwin->window);
}

static gboolean bmp_db_popup_menu(GtkWidget * widget,
				      GdkEventButton * event,
				      gpointer user_data)
{



    if (event->button == 3) {
	gtk_item_factory_popup_with_data(bmp_db_popup,
					 user_data,
					 NULL,
					 event->x_root,
					 event->y_root,
					 event->button, event->time);
	return TRUE;
    }

    return FALSE;
}

static void bmp_db_menu_callback(gpointer data, guint action,
				     GtkWidget * widget)
{

    GtkWidget *treeview = gtk_item_factory_popup_data(bmp_db_popup);


    switch (action) {
    case DIRBROWSER_REMOVE:
	bmp_db_remove_attached(treeview);
    case DIRBROWSER_DETACH:
	bmp_db_detach_selected(treeview);
	break;
    case DIRBROWSER_COLLAPSE:
	gtk_tree_view_collapse_all(GTK_TREE_VIEW(treeview));
	break;
    case DIRBROWSER_COLLAPSE_SEL:
	bmp_db_collapse_selected(treeview);
	break;
    case DIRBROWSER_EXPAND:
	bmp_db_expand_selected(treeview);
	break;
    case DIRBROWSER_ENQUEUE:
	bmp_db_add_clicked(NULL, treeview);
	break;
    case DIRBROWSER_PLAY:
	bmp_db_play_clicked(NULL, treeview);
	break;
    case DIRBROWSER_NEWPLIST:
	playlist_clear();
	break;
    default:
	break;
    }
}
