/* ****************************************************************************

 * eID Middleware Project.
 * Copyright (C) 2008-2009 FedICT.
 *
 * This is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License version
 * 3.0 as published by the Free Software Foundation.
 *
 * This software is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this software; if not, see
 * http://www.gnu.org/licenses/.

**************************************************************************** */
#ifdef WIN32
#include "windows.h"
#include "SCardErr.h"
#else
#include <wintypes.h>
#include <winscard.h>
#endif

#include "UnitTest++/src/UnitTest++.h"
#include "../eidlib/eidlib.h"
#include "../eidlib/eidlibException.h"
#include "../eidlibC/eidlibC.h"
#include "../eidlibC/eidlibCdefines.h"
#include "../common/Util.h"

#ifndef SCARD_E_INVALID_CHV
#define SCARD_E_INVALID_CHV				0x8010002AL
#endif

#ifndef SCARD_E_NO_READERS_AVAILABLE
#define SCARD_E_NO_READERS_AVAILABLE	0x8010002EL
#endif

#ifndef SCARD_W_WRONG_CHV
#define SCARD_W_WRONG_CHV				0x8010006BL
#endif

#ifndef SCARD_W_CHV_BLOCKED
#define SCARD_W_CHV_BLOCKED				0x8010006CL
#endif

#ifndef SCARD_W_CANCELLED_BY_USER
#define SCARD_W_CANCELLED_BY_USER		0x8010006EL
#endif

#ifndef SCARD_W_CARD_NOT_AUTHENTICATED
#define SCARD_W_CARD_NOT_AUTHENTICATED	0x8010006FL
#endif

using namespace eIDMW;

enum
{
	  IDX_READER_TEST_CARD_V1		// emulated reader with test card version 1 v1_test_1
	, IDX_READER_NO_CARD			// emulated reader without a card
	, IDX_READER_EID				// emulated reader with EID card v1_test_2
	, IDX_READER_TEST_CARD_V2		// emulated reader with test card version 2 v2_test_1
	, IDX_READER_SIS				// emulated reader with SIS card 
	, IDX_READER_REVOKED_CERT		// emulated reader with revoked certificates v1_test_3
	, NRREADERS
};

BEID_Bytes BEID_FILE_ID				= {(BYTE*)"\x3F\x00\xDF\x01\x40\x31", 6};
BEID_Bytes BEID_FILE_ID_SIGN		= {(BYTE*)"\x3F\x00\xDF\x01\x40\x32", 6};
BEID_Bytes BEID_FILE_ADDRESS		= {(BYTE*)"\x3F\x00\xDF\x01\x40\x33", 6};
BEID_Bytes BEID_FILE_ADDRESS_SIGN	= {(BYTE*)"\x3F\x00\xDF\x01\x40\x34", 6};
BEID_Bytes BEID_FILE_PHOTO			= {(BYTE*)"\x3F\x00\xDF\x01\x40\x35", 6};
BEID_Bytes BEID_FILE_PREF			= {(BYTE*)"\x3F\x00\xDF\x01\x40\x39", 6};
BEID_Bytes BEID_FILE_TOKENINFO		= {(BYTE*)"\x3F\x00\xDF\x00\x50\x32", 6};
BEID_Bytes BEID_FILE_RRN			= {(BYTE*)"\x3F\x00\xDF\x00\x50\x3C", 6};
BEID_Bytes BEID_FILE_RRNCA			= {(BYTE*)"\x3F\x00\xDF\x00\x50\x3D", 6};
//BEID_Bytes SIS_FILE_ID				= {(BYTE*)		"data"

#define BEID_MAX_URL_LENGTH		255

TEST(BEIDInitExit)
{
#define NRREADERS  8
	//-------------------------------------
	// get a readerlist 
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	
	//-------------------------------------
	// nr of readers must be correct
	//-------------------------------------
	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	const char*	pszReaderName;
	BEID_Status status;

	//-------------------------------------
	// Select a reader by index
	// set parameters and init
	//-------------------------------------
	
	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V1).getName();

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	lOCSP	 = BEID_OCSP_CRL_NOT_USED;
	lCRL	 = BEID_OCSP_CRL_NOT_USED;


	status	= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);

	CHECK_EQUAL(BEID_OK,status.general);

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);

	//-------------------------------------
	// set parameters and do the test
	//-------------------------------------
	ppList		  = ReaderSet.readerList();
	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V1).getName();

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	lOCSP		= BEID_OCSP_CRL_OPTIONAL;
	lCRL		= BEID_OCSP_CRL_OPTIONAL;
	lHandle		= 0;

	status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);

	CHECK_EQUAL(BEID_OK,status.general);

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);

	//-------------------------------------
	// set parameters and do the test
	//-------------------------------------
	ppList		  = ReaderSet.readerList();
	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V1).getName();

	lOCSP		= BEID_OCSP_CRL_MANDATORY;
	lCRL		= BEID_OCSP_CRL_MANDATORY;
	lHandle		= 0;

	status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);

	CHECK_EQUAL(BEID_E_BAD_PARAM,status.general);

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);

	//-------------------------------------
	// select an inexistent card reader
	//-------------------------------------
	ppList		  = ReaderSet.readerList();
	pszReaderName = "John_Doe";

	lOCSP	= BEID_OCSP_CRL_NOT_USED;
	lCRL	= BEID_OCSP_CRL_NOT_USED;
	lHandle	= 0;

	status	= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);

	CHECK_EQUAL(BEID_E_PCSC,status.general);
	CHECK_EQUAL(SCARD_E_NO_READERS_AVAILABLE,status.pcsc);

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);

}

TEST(BEIDGetID)
{
#define NRREADERS  8
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	const char*			pszReaderName = NULL; 
	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V1).getName();

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
	CHECK_EQUAL(BEID_OK,status.general);

	//-------------------------------------
	// set parameters to illegal values
	//-------------------------------------
	BEID_ID_Data*		ptData		  = NULL;
	BEID_Certif_Check*	ptCertifCheck = NULL;

	status = BEID_GetID(ptData, ptCertifCheck);
	CHECK_EQUAL(BEID_E_BAD_PARAM,status.general);

	BEID_ID_Data		Data;
	BEID_Certif_Check	CertifCheck;
	CertifCheck.certificatesLength=BEID_MAX_CERT_NUMBER;

	//-------------------------------------
	// use correct parameters
	// we'll use a test card though, so version will be NULL
	//-------------------------------------

	status = BEID_GetID(&Data, &CertifCheck);
	CHECK_EQUAL(BEID_OK,status.general);
	CHECK_EQUAL(0,Data.version);

	//-------------------------------------
	// use correct parameters
	// we'll use a reader with a normal EID card
	//-------------------------------------
	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_EID).getName();

	status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
	CHECK_EQUAL(BEID_OK,status.general);

	CertifCheck.certificatesLength=BEID_MAX_CERT_NUMBER;
	status = BEID_GetID(&Data, &CertifCheck);
	CHECK_EQUAL(BEID_OK,status.general);
	CHECK_EQUAL(0,Data.version);
	CHECK_EQUAL("590445074725",Data.cardNumber);
	CHECK_EQUAL("534C494E336600296CFF2623660B0826",Data.chipNumber);
	CHECK_EQUAL("20061024",Data.validityDateBegin);
	CHECK_EQUAL("20111024",Data.validityDateEnd);
	CHECK_EQUAL("Holsbeek",Data.municipality);
	CHECK_EQUAL("73040102749",Data.nationalNumber);
	CHECK_EQUAL("Hoeben",Data.name);
	CHECK_EQUAL("Stefan",Data.firstName1);	// this is not trimmed!!!
	CHECK_EQUAL("",Data.firstName2);
	CHECK_EQUAL("",Data.firstName3);
	CHECK_EQUAL("be",Data.nationality);
	CHECK_EQUAL("Bree",Data.birthLocation);
	CHECK_EQUAL("19730401",Data.birthDate);	// this is actually wrong!!!
	CHECK_EQUAL("M",Data.sex);
	CHECK_EQUAL("",Data.nobleCondition);
	CHECK_EQUAL(1,Data.documentType);
	CHECK_EQUAL(0,Data.whiteCane);
	CHECK_EQUAL(0,Data.yellowCane);
	CHECK_EQUAL(0,Data.extendedMinority);
	//CHECK_ARRAY_EQUAL(0,Data.hashPhoto);


	//-------------------------------------
	// Check if certificates came out OK
	// and signature check went ok
	// also data should be checked
	//-------------------------------------
	CHECK_EQUAL(1,CertifCheck.certificatesLength);
	CHECK_EQUAL(BEID_SIGNATURE_VALID,CertifCheck.signatureCheck);


	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);

	//-------------------------------------
	// use correct parameters
	// we'll use a reader without a card
	//-------------------------------------
	ppList = ReaderSet.readerList();

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_NO_CARD).getName();
	CHECK_EQUAL(BEID_OK,status.general);

	status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
	CHECK_EQUAL(BEID_E_PCSC,status.general);
	CHECK_EQUAL(SCARD_E_NO_SMARTCARD,status.pcsc);

	status = BEID_GetID(&Data, &CertifCheck);
	CHECK_EQUAL(BEID_E_PCSC,status.general);
	CHECK_EQUAL(SCARD_E_NO_SMARTCARD,status.pcsc);

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);
}

TEST(BEIDGetAddress)
{
	BEID_Status status;
	BEID_Address		Address;
	BEID_Certif_Check	CertifCheck;

	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList			= ReaderSet.readerList();
	const char*			pszReaderName	= NULL; 
	long				lOCSP			= BEID_OCSP_CRL_NOT_USED;
	long				lCRL			= BEID_OCSP_CRL_NOT_USED;
	long				lHandle			= 0;

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	//-------------------------------------
	// use correct parameters
	// no card available
	//-------------------------------------
	lOCSP	= BEID_OCSP_CRL_NOT_USED;
	lCRL	= BEID_OCSP_CRL_NOT_USED;
	lHandle	= 0;

	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_NO_CARD).getName();
	status		  = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
	CHECK_EQUAL(BEID_E_PCSC,status.general);
	CHECK_EQUAL(SCARD_E_NO_SMARTCARD,status.pcsc);

	status = BEID_GetAddress(&Address, &CertifCheck);
	CHECK_EQUAL(BEID_E_PCSC,status.general);
	CHECK_EQUAL(SCARD_E_NO_SMARTCARD,status.pcsc);

	//-------------------------------------
	// use correct parameters
	// use a test card though, so version will be NULL
	//-------------------------------------
	lOCSP	= BEID_OCSP_CRL_NOT_USED;
	lCRL	= BEID_OCSP_CRL_NOT_USED;
	lHandle	= 0;

	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V1).getName();
	status		  = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
	CHECK_EQUAL(BEID_OK,status.general);

	CertifCheck.certificatesLength = BEID_MAX_CERT_NUMBER;
	status = BEID_GetAddress(&Address, &CertifCheck);
	CHECK_EQUAL(BEID_OK,status.general);
	CHECK_EQUAL(0,Address.version);

	//-------------------------------------
	// use correct parameters
	// use a good card
	//-------------------------------------

	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_EID).getName();

	lOCSP	= BEID_OCSP_CRL_NOT_USED;
	lCRL	= BEID_OCSP_CRL_NOT_USED;
	lHandle	= 0;

	status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
	CHECK_EQUAL(BEID_OK,status.general);

	status = BEID_GetAddress(&Address, &CertifCheck);
	CHECK_EQUAL(BEID_OK,status.general);

	//-------------------------------------
	// Check if certificates came out OK
	// and signature check went ok
	//-------------------------------------
	CHECK_EQUAL(1,CertifCheck.certificatesLength);
	CHECK_EQUAL(BEID_SIGNATURE_VALID,CertifCheck.signatureCheck);

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);
}

TEST(BEIDPicture)
{
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_EID).getName();

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
	CHECK_EQUAL(BEID_OK,status.general);

	//-------------------------------------
	// try with a buffer of length NULL
	//-------------------------------------
	BEID_Bytes			pict;
	BEID_Certif_Check	CertifCheck;

	pict.length = 0;

	status = BEID_GetPicture(&pict, &CertifCheck);
	CHECK_EQUAL(BEID_E_BAD_PARAM,status.general);

	//-------------------------------------
	// try with a non allocated buffer
	//-------------------------------------

	pict.length = BEID_MAX_PICTURE_LEN;
	pict.data	= NULL;

	status = BEID_GetPicture(&pict, &CertifCheck);
	CHECK_EQUAL(BEID_E_BAD_PARAM,status.general);

	//-------------------------------------
	// try with a buffer equal to the maximum
	//-------------------------------------

	pict.length = BEID_MAX_PICTURE_LEN;
	pict.data	= new BYTE [pict.length];

	CertifCheck.certificatesLength = BEID_MAX_CERT_NUMBER;
	status = BEID_GetPicture(&pict, &CertifCheck);
	CHECK_EQUAL(BEID_OK,status.general);

	//-------------------------------------
	// Check if certificates came out OK
	// and signature check went ok
	//-------------------------------------
	CHECK_EQUAL(1,CertifCheck.certificatesLength);
	CHECK_EQUAL(BEID_SIGNATURE_VALID,CertifCheck.signatureCheck);

	delete [] pict.data;

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);

}

TEST(BEIDSelectApplication)
{
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_EID).getName();

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
	CHECK_EQUAL(BEID_OK,status.general);

	BEID_Bytes application;

	application.data		 = NULL;
	application.length		 = 0;
	application.neededlength = 0;

	status = BEID_SelectApplication(&application);
	CHECK_EQUAL(BEID_E_BAD_PARAM,status.general);

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);

}
TEST(BEIDFlushCache)
{
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	//-------------------------------------
	// use correct parameters
	// no card available. Should not matter because files
	// are flushed independent of cards
	//-------------------------------------
	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_NO_CARD).getName();

	status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
	CHECK_EQUAL(BEID_E_PCSC,status.general);
	CHECK_EQUAL(SCARD_E_NO_SMARTCARD,status.pcsc);

	status = BEID_FlushCache();
	CHECK_EQUAL(BEID_OK,status.general);

	//-------------------------------------
	// use correct parameters
	// test card
	//-------------------------------------
	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V1).getName();

	status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
	CHECK_EQUAL(BEID_OK,status.general);

	status = BEID_FlushCache();
	CHECK_EQUAL(BEID_OK,status.general);

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);

}

TEST(BEIDSendAPDU)
{
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	//-------------------------------------
	// use correct parameters
	// Use parameter errors
	//-------------------------------------
	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_EID).getName();

	status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
	CHECK_EQUAL(BEID_OK,status.general);

	{
		BEID_Bytes CmdAPDU;
		//::BEID_Pin   PinData;
		BEID_Bytes RespAPDU;

		status = BEID_SendAPDU(&CmdAPDU, NULL, &RespAPDU);
		CHECK_EQUAL(BEID_E_BAD_PARAM,status.general);
	}
	{
		//BEID_Bytes CmdAPDU;
		::BEID_Pin   PinData;
		BEID_Bytes RespAPDU;

		status = BEID_SendAPDU(NULL, &PinData, &RespAPDU);
		CHECK_EQUAL(BEID_E_BAD_PARAM,status.general);
	}
	{
		BEID_Bytes CmdAPDU;
		::BEID_Pin   PinData;
		//BEID_Bytes RespAPDU;

		status = BEID_SendAPDU(&CmdAPDU, &PinData, NULL);
		CHECK_EQUAL(BEID_E_BAD_PARAM,status.general);
	}

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);

}

TEST(BEIDGetVersionInfo)
{
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	//-------------------------------------
	// use correct parameters
	// EID card
	//-------------------------------------
	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_EID).getName();

	status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
	CHECK_EQUAL(BEID_OK,status.general);

	BEID_VersionInfo VersionInfo;
	BOOL			 Signature = false;
	BEID_Bytes		 SignedStatus;

	status = BEID_GetVersionInfo(&VersionInfo, Signature, &SignedStatus);
	CHECK_EQUAL(BEID_OK,status.general);

	BYTE serNr[BEID_MAX_SERIALNR_LEN];
	serNr[0]=0x53;
	serNr[1]=0x4C;
	serNr[2]=0x49;
	serNr[3]=0x4E;
	serNr[4]=0x33;
	serNr[5]=0x66;
	serNr[6]=0x00;
	serNr[7]=0x29;
	serNr[8]=0x6C;
	serNr[9]=0xFF;
	serNr[10]=0x26;
	serNr[11]=0x23;
	serNr[12]=0x66;
	serNr[13]=0x0B;
	serNr[14]=0x08;
	serNr[15]=0x26;

	CHECK_ARRAY_EQUAL(serNr,VersionInfo.SerialNumber,BEID_MAX_SERIALNR_LEN);
	CHECK_EQUAL(0xa5,(unsigned char)VersionInfo.ComponentCode);
	CHECK_EQUAL(0x03,(unsigned char)VersionInfo.OSNumber);
	CHECK_EQUAL(0x01,(unsigned char)VersionInfo.OSVersion);
	CHECK_EQUAL(0x01,(unsigned char)VersionInfo.SoftmaskNumber);
	CHECK_EQUAL(0x01,(unsigned char)VersionInfo.SoftmaskVersion);
	CHECK_EQUAL(0x11,(unsigned char)VersionInfo.AppletVersion);
	CHECK_EQUAL(0x02,(unsigned char)VersionInfo.GlobalOSVersion);
	CHECK_EQUAL(0x00,(unsigned char)VersionInfo.AppletInterfaceVersion);
	CHECK_EQUAL(0x01,(unsigned char)VersionInfo.PKCS1Support);
	CHECK_EQUAL(0x01,(unsigned char)VersionInfo.KeyExchangeVersion);
	CHECK_EQUAL(0x0f,(unsigned char)VersionInfo.ApplicationLifeCycle);
	CHECK_EQUAL(0x04,(unsigned char)VersionInfo.GraphPerso);
	CHECK_EQUAL(0x02,(unsigned char)VersionInfo.ElecPerso);
	CHECK_EQUAL(0x00,(unsigned char)VersionInfo.ElecPersoInterface);

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);

}

TEST(BEIDGetPINs)
{
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;


	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_TEST_CARD_V1 card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V1).getName();
		status		  = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Pins	pins;

		status = BEID_GetPINs(&pins);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(1,pins.pinsLength);

		for (long pinIdx = 0; pinIdx<pins.pinsLength; pinIdx++)
		{
			CHECK_EQUAL(0,pins.pins[pinIdx].pinType);
			CHECK_EQUAL(1,pins.pins[pinIdx].id);
			CHECK_EQUAL(BEID_PIN_USG_AUTH,pins.pins[pinIdx].usageCode);
			CHECK_EQUAL(0,pins.pins[pinIdx].triesLeft);
			CHECK_EQUAL(0x30,pins.pins[pinIdx].flags);
		}

	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_NO_CARD card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_NO_CARD).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_E_PCSC,status.general);
		CHECK_EQUAL(SCARD_E_NO_SMARTCARD,status.pcsc);

		::BEID_Pins	pins;
		pins.pinsLength = BEID_MAX_PINS;

		status = BEID_GetPINs(&pins);
		CHECK_EQUAL(BEID_E_PCSC,status.general);
		CHECK_EQUAL(SCARD_E_NO_SMARTCARD,status.pcsc);
	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_EID card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_EID).getName();
		status		  = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Pins	pins;
		pins.pinsLength = BEID_MAX_PINS;

		status = BEID_GetPINs(&pins);
		CHECK_EQUAL(BEID_OK,status.general);

		CHECK_EQUAL(pins.pinsLength,1);

		for (long pinIdx = 0; pinIdx<pins.pinsLength; pinIdx++)
		{
			CHECK_EQUAL(0,pins.pins[pinIdx].pinType);
			CHECK_EQUAL(1,pins.pins[pinIdx].id);
			CHECK_EQUAL(BEID_PIN_USG_AUTH,pins.pins[pinIdx].usageCode);
			CHECK_EQUAL(0,pins.pins[pinIdx].triesLeft);
			CHECK_EQUAL(0x30,pins.pins[pinIdx].flags);
		}
	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_TEST_CARD_V2 card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V2).getName();
		status		  = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Pins	pins;
		pins.pinsLength = BEID_MAX_PINS;

		status = BEID_GetPINs(&pins);
		CHECK_EQUAL(BEID_OK,status.general);

		CHECK_EQUAL(pins.pinsLength,2);

		for (long pinIdx = 0; pinIdx<pins.pinsLength; pinIdx++)
		{
			CHECK_EQUAL(0, pins.pins[pinIdx].pinType);
			CHECK_EQUAL(pinIdx+1,pins.pins[pinIdx].id);
			if(pinIdx==0)
				CHECK_EQUAL(BEID_PIN_USG_AUTH,pins.pins[pinIdx].usageCode);
			else if(pinIdx==1)
				CHECK_EQUAL(BEID_PIN_USG_SIGN,pins.pins[pinIdx].usageCode);
			CHECK_EQUAL(3, pins.pins[pinIdx].triesLeft);
			CHECK_EQUAL(0x30,pins.pins[pinIdx].flags);
		}

	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_SIS card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_SIS).getName();
		status		  = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Pins	pins;
		pins.pinsLength = BEID_MAX_PINS;

		status = BEID_GetPINs(&pins);
		CHECK_EQUAL(BEID_E_UNKNOWN_CARD,status.general);

		CHECK_EQUAL(pins.pinsLength,0);
	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_REVOKED_CERT card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_REVOKED_CERT).getName();
		status		  = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Pins	pins;
		pins.pinsLength = BEID_MAX_PINS;

		status = BEID_GetPINs(&pins);
		CHECK_EQUAL(BEID_OK,status.general);

		CHECK_EQUAL(pins.pinsLength,1);

		for (long pinIdx = 0; pinIdx<pins.pinsLength; pinIdx++)
		{
			CHECK_EQUAL(0,pins.pins[pinIdx].pinType);
			CHECK_EQUAL(pinIdx+1,pins.pins[pinIdx].id);
			CHECK_EQUAL(BEID_PIN_USG_AUTH,pins.pins[pinIdx].usageCode);
			CHECK_EQUAL(0,pins.pins[pinIdx].triesLeft);
			CHECK_EQUAL(0x30,pins.pins[pinIdx].flags);
		}
	}

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);
} //TEST(BEIDGetPINs)

TEST(BEIDGetPINStatus)
{
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	{
		//-------------------------------------
		// use correct parameters
		// EID card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_EID).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		//-------------------------------------
		// get the PIN status. The old EID interface will always take the first pin
		//-------------------------------------
		::BEID_Pin	pin;
		long		triesLeft	= -1;
		BOOL		bSignature	= false /*true*/; //Doesn't work with v1
		BEID_Bytes	signedStatus;

		status = BEID_GetPINStatus(&pin, &triesLeft, bSignature, &signedStatus);

		CHECK_EQUAL(BEID_E_CARD,status.general);
	}

	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_TEST_CARD_V1 card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V1).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		BOOL		bSignature = false /*true*/; //Doesn't work with v1
		long		triesLeft  = -1;
		BEID_Bytes	signedStatus;
		::BEID_Pin	pin;

		//-------------------------------------
		// get the PIN status. The old EID interface will always take the first pin
		//-------------------------------------
		status = BEID_GetPINStatus(&pin, &triesLeft, bSignature, &signedStatus);

		CHECK_EQUAL(BEID_E_CARD,status.general);
		CHECK_EQUAL(-1,triesLeft);
		CHECK_EQUAL(0,pin.pinType);
		CHECK_EQUAL(1,pin.id);
		CHECK_EQUAL(1,pin.usageCode);
		//CHECK_EQUAL(pin.shortUsage,NULL);
		//CHECK_EQUAL(pin.longUsage,NULL);
	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_NO_CARD card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_NO_CARD).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_E_PCSC,status.general);
		CHECK_EQUAL(SCARD_E_NO_SMARTCARD,status.pcsc); 

		BOOL		bSignature = true; 
		long		triesLeft  = -1;
		BEID_Bytes	signedStatus;
		::BEID_Pin	pin;

		status = BEID_GetPINStatus(&pin, &triesLeft, bSignature, &signedStatus);

		CHECK_EQUAL(BEID_E_PCSC,status.general);
		CHECK_EQUAL(SCARD_E_NO_SMARTCARD,status.pcsc); 
	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_TEST_CARD_V2 card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V2).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		BOOL		bSignature = true; 
		long		triesLeft  = -1;
		::BEID_Pin	pin;

		BEID_Bytes	signedStatus;
		signedStatus.data		 = NULL;
		signedStatus.length		 = 0;
		signedStatus.neededlength = 0;


		status = BEID_GetPINStatus(&pin, &triesLeft, bSignature, &signedStatus);
		CHECK_EQUAL(BEID_E_INSUFFICIENT_BUFFER,status.general);

		signedStatus.data		 = new BYTE[signedStatus.neededlength];
		signedStatus.length		 = signedStatus.neededlength;
		signedStatus.neededlength = 0;

		status = BEID_GetPINStatus(&pin, &triesLeft, bSignature, &signedStatus);
		CHECK_EQUAL(BEID_OK,status.general);

		CHECK_EQUAL(3,triesLeft);
		CHECK_EQUAL(0,pin.pinType);
		CHECK_EQUAL(1,pin.id);
		CHECK_EQUAL(1,pin.usageCode);
		//CHECK_EQUAL(pin.shortUsage,NULL);
		//CHECK_EQUAL(pin.longUsage,NULL);

		if(signedStatus.data) delete[] signedStatus.data;
	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_SIS card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_SIS).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		BOOL		bSignature = true;
		long		triesLeft  = -1;
		BEID_Bytes	signedStatus;
		::BEID_Pin	pin;

		status = BEID_GetPINStatus(&pin, &triesLeft, bSignature, &signedStatus);

		CHECK_EQUAL(BEID_E_UNKNOWN_CARD,status.general);
	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_REVOKED_CERT card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_REVOKED_CERT).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		BOOL		bSignature = false /*true*/; //Doesn't work with v1
		long		triesLeft  = -1;
		BEID_Bytes	signedStatus;
		::BEID_Pin	pin;

		status = BEID_GetPINStatus(&pin, &triesLeft, bSignature, &signedStatus);

		CHECK_EQUAL(BEID_E_CARD,status.general);
		CHECK_EQUAL(-1,triesLeft);
		CHECK_EQUAL(0,pin.pinType);
		CHECK_EQUAL(1,pin.id);
		CHECK_EQUAL(1,pin.usageCode);
		//CHECK_EQUAL(pin.shortUsage,NULL);
		//CHECK_EQUAL(pin.longUsage,NULL);
	}

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);

} // TEST(BEIDGetPINStatus)
TEST(BEIDVerifyPIN)
{
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	{
		//-------------------------------------
		// use correct parameters
		// EID card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_EID).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Pins pins;
		status = BEID_GetPINs(&pins);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(pins.pinsLength,1);

		for (long pinIdx = 0; pinIdx < pins.pinsLength; pinIdx++)
		{
			::BEID_Pin	pin;
			pin.pinType   = pins.pins[pinIdx].pinType;
			pin.id		  = pins.pins[pinIdx].id;
			pin.usageCode = pins.pins[pinIdx].usageCode;
			pin.shortUsage = NULL;
			pin.longUsage  = NULL;
			
			char		pinCode[5];;
			long		triesLeft;

			//-------------------------------------
			// check correct PIN
			//-------------------------------------
			strcpy_s(pinCode,sizeof(pinCode),"1234");
			status = BEID_VerifyPIN(&pin, pinCode, &triesLeft);
			CHECK_EQUAL(BEID_OK,status.general);

			//-------------------------------------
			// check incorrect PIN
			//-------------------------------------
			strcpy_s(pinCode,sizeof(pinCode),"4321");
			status = BEID_VerifyPIN(&pin, pinCode, &triesLeft);
			CHECK_EQUAL(BEID_E_PCSC,status.general);
			CHECK_EQUAL(SCARD_E_INVALID_CHV,status.pcsc); 
			CHECK_EQUAL(2,triesLeft);
		}

	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_TEST_CARD_V1 card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V1).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Pins pins;
		status = BEID_GetPINs(&pins);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(pins.pinsLength,1);

		for (long pinIdx = 0; pinIdx < pins.pinsLength; pinIdx++)
		{
			::BEID_Pin	pin;
			pin.pinType   = pins.pins[pinIdx].pinType;
			pin.id		  = pins.pins[pinIdx].id;
			pin.usageCode = pins.pins[pinIdx].usageCode;
			pin.shortUsage = NULL;
			pin.longUsage  = NULL;

			char		pinCode[5];;
			long		triesLeft;

			//-------------------------------------
			// check correct PIN
			//-------------------------------------
			strcpy_s(pinCode,sizeof(pinCode),"1234");
			status = BEID_VerifyPIN(&pin, pinCode, &triesLeft);
			CHECK_EQUAL(BEID_OK,status.general);

			//-------------------------------------
			// check incorrect PIN
			//-------------------------------------
			strcpy_s(pinCode,sizeof(pinCode),"4321");
			status = BEID_VerifyPIN(&pin, pinCode, &triesLeft);
			CHECK_EQUAL(BEID_E_PCSC,status.general);
			CHECK_EQUAL(SCARD_E_INVALID_CHV,status.pcsc); 
			CHECK_EQUAL(2,triesLeft);
		}
	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_NO_CARD card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_NO_CARD).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_E_PCSC,status.general);
		CHECK_EQUAL(SCARD_E_NO_SMARTCARD,status.pcsc); 

		::BEID_Pins pins;
		status = BEID_GetPINs(&pins);
		CHECK_EQUAL(BEID_E_PCSC,status.general);
		CHECK_EQUAL(SCARD_E_NO_SMARTCARD,status.pcsc); 
		CHECK_EQUAL(0,pins.pinsLength);
	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_TEST_CARD_V2 card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V2).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Pins pins;
		status = BEID_GetPINs(&pins);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(2,pins.pinsLength);

		for (long pinIdx = 0; pinIdx < pins.pinsLength; pinIdx++)
		{
			::BEID_Pin	pin;
			pin.pinType   = pins.pins[pinIdx].pinType;
			pin.id		  = pins.pins[pinIdx].id;
			pin.usageCode = pins.pins[pinIdx].usageCode;
			pin.shortUsage = NULL;
			pin.longUsage  = NULL;

			char		pinCode[5];;
			long		triesLeft;

			//-------------------------------------
			// check correct PIN
			//-------------------------------------
			strcpy_s(pinCode,sizeof(pinCode),"1234");
			status = BEID_VerifyPIN(&pin, pinCode, &triesLeft);
			CHECK_EQUAL(BEID_OK,status.general);

			//-------------------------------------
			// check incorrect PIN
			//-------------------------------------
			strcpy_s(pinCode,sizeof(pinCode),"4321");
			status = BEID_VerifyPIN(&pin, pinCode, &triesLeft);
			CHECK_EQUAL(BEID_E_PCSC,status.general);
			CHECK_EQUAL(SCARD_E_INVALID_CHV,status.pcsc); 
			CHECK_EQUAL(2,triesLeft);
		}
	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_SIS card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_SIS).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Pins pins;
		status = BEID_GetPINs(&pins);
		CHECK_EQUAL(BEID_E_UNKNOWN_CARD,status.general);
		CHECK_EQUAL(0,pins.pinsLength);
	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_REVOKED_CERT card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_REVOKED_CERT).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Pins pins;
		status = BEID_GetPINs(&pins);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(pins.pinsLength,1);

		for (long pinIdx = 0; pinIdx < pins.pinsLength; pinIdx++)
		{
			::BEID_Pin	pin;
			pin.pinType   = pins.pins[pinIdx].pinType;
			pin.id		  = pins.pins[pinIdx].id;
			pin.usageCode = pins.pins[pinIdx].usageCode;
			pin.shortUsage = NULL;
			pin.longUsage  = NULL;

			char		pinCode[5];;
			long		triesLeft;

			//-------------------------------------
			// check correct PIN
			//-------------------------------------
			strcpy_s(pinCode,sizeof(pinCode),"1234");
			status = BEID_VerifyPIN(&pin, pinCode, &triesLeft);
			CHECK_EQUAL(BEID_OK,status.general);

			//-------------------------------------
			// check incorrect PIN
			//-------------------------------------
			strcpy_s(pinCode,sizeof(pinCode),"4321");
			status = BEID_VerifyPIN(&pin, pinCode, &triesLeft);
			CHECK_EQUAL(BEID_E_PCSC,status.general);
			CHECK_EQUAL(SCARD_E_INVALID_CHV,status.pcsc); 
			CHECK_EQUAL(2,triesLeft);
		}
	}

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);
}

TEST(BEIDChangePIN)
{
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	{
		//-------------------------------------
		// use correct parameters
		// EID card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_EID).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Pins pins;
		status = BEID_GetPINs(&pins);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(pins.pinsLength,1);
		for (long pinIdx = 0; pinIdx < pins.pinsLength; pinIdx++)
		{
			::BEID_Pin	pin;
			pin.pinType   = pins.pins[pinIdx].pinType;
			pin.id		  = pins.pins[pinIdx].id;
			pin.usageCode = pins.pins[pinIdx].usageCode;
			pin.shortUsage = NULL;
			pin.longUsage  = NULL;

			char		oldPin[5];
			char		newPin[5];
			long		triesLeft  = -1;

			//-------------------------------------
			// change the PIN code
			//-------------------------------------
			strcpy_s(oldPin,sizeof(oldPin),"1234");
			strcpy_s(newPin,sizeof(newPin),"4321");
			status = BEID_ChangePIN(&pin, oldPin, newPin, &triesLeft);
			CHECK_EQUAL(BEID_OK,status.general);

			//-------------------------------------
			// verify the new PIN
			//-------------------------------------
			status = BEID_VerifyPIN(&pin, newPin, &triesLeft);
			CHECK_EQUAL(BEID_OK,status.general);

			//-------------------------------------
			// verify if the old PIN code is invalid
			//-------------------------------------
			status = BEID_VerifyPIN(&pin, oldPin, &triesLeft);
			CHECK_EQUAL(BEID_E_PCSC,status.general);
			CHECK_EQUAL(SCARD_E_INVALID_CHV,status.pcsc); 
			CHECK_EQUAL(2,triesLeft);

			//-------------------------------------
			// replace the pin to its initial value
			//-------------------------------------
			status = BEID_ChangePIN(&pin, newPin, oldPin, &triesLeft);
			CHECK_EQUAL(BEID_OK,status.general);
		}
	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_TEST_CARD_V1 card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V1).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Pins pins;
		status = BEID_GetPINs(&pins);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(pins.pinsLength,1);
		for (long pinIdx = 0; pinIdx < pins.pinsLength; pinIdx++)
		{
			::BEID_Pin	pin;
			pin.pinType   = pins.pins[pinIdx].pinType;
			pin.id		  = pins.pins[pinIdx].id;
			pin.usageCode = pins.pins[pinIdx].usageCode;
			pin.shortUsage = NULL;
			pin.longUsage  = NULL;

			char		oldPin[5];
			char		newPin[5];
			long		triesLeft;

			//-------------------------------------
			// check PIN change
			//-------------------------------------
			strcpy_s(oldPin,sizeof(oldPin),"1234");
			strcpy_s(newPin,sizeof(newPin),"4321");
			status = BEID_ChangePIN(&pin, oldPin, newPin, &triesLeft);
			CHECK_EQUAL(BEID_OK,status.general);

			status = BEID_VerifyPIN(&pin, newPin, &triesLeft);
			CHECK_EQUAL(BEID_OK,status.general);

			//-------------------------------------
			// replace the pin to its initial value
			//-------------------------------------
			status = BEID_ChangePIN(&pin, newPin, oldPin, &triesLeft);
			CHECK_EQUAL(BEID_OK,status.general);
		}
	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_NO_CARD card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_NO_CARD).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_E_PCSC,status.general);
		CHECK_EQUAL(SCARD_E_NO_SMARTCARD,status.pcsc); 

		::BEID_Pins pins;
		status = BEID_GetPINs(&pins);
		CHECK_EQUAL(BEID_E_PCSC,status.general);
		CHECK_EQUAL(SCARD_E_NO_SMARTCARD,status.pcsc); 
		CHECK_EQUAL(0,pins.pinsLength);
	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_TEST_CARD_V1 card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V2).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Pins pins;
		status = BEID_GetPINs(&pins);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(pins.pinsLength,2);
		for (long pinIdx = 0; pinIdx < pins.pinsLength; pinIdx++)
		{
			::BEID_Pin	pin;
			pin.pinType   = pins.pins[pinIdx].pinType;
			pin.id		  = pins.pins[pinIdx].id;
			pin.usageCode = pins.pins[pinIdx].usageCode;
			pin.shortUsage = NULL;
			pin.longUsage  = NULL;

			char		oldPin[5];
			char		newPin[5];
			long		triesLeft;

			//-------------------------------------
			// check PIN change
			//-------------------------------------
			strcpy_s(oldPin,sizeof(oldPin),"1234");
			strcpy_s(newPin,sizeof(newPin),"4321");
			status = BEID_ChangePIN(&pin, oldPin, newPin, &triesLeft);
			CHECK_EQUAL(BEID_OK,status.general);

			status = BEID_VerifyPIN(&pin, newPin, &triesLeft);
			CHECK_EQUAL(BEID_OK,status.general);

			//-------------------------------------
			// replace the pin to its initial value
			//-------------------------------------
			status = BEID_ChangePIN(&pin, newPin, oldPin, &triesLeft);
			CHECK_EQUAL(BEID_OK,status.general);
		}
	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_SIS card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_SIS).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Pins pins;
		status = BEID_GetPINs(&pins);
		CHECK_EQUAL(BEID_E_UNKNOWN_CARD,status.general);
		CHECK_EQUAL(0,pins.pinsLength);
	}
	{
		//-------------------------------------
		// use correct parameters
		// IDX_READER_REVOKED_CERT card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_REVOKED_CERT).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Pins pins;
		status = BEID_GetPINs(&pins);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(pins.pinsLength,1);
		for (long pinIdx = 0; pinIdx < pins.pinsLength; pinIdx++)
		{
			::BEID_Pin	pin;
			pin.pinType   = pins.pins[pinIdx].pinType;
			pin.id		  = pins.pins[pinIdx].id;
			pin.usageCode = pins.pins[pinIdx].usageCode;
			pin.shortUsage = NULL;
			pin.longUsage  = NULL;

			char		oldPin[5];
			char		newPin[5];
			long		triesLeft;

			//-------------------------------------
			// check PIN change
			//-------------------------------------
			strcpy_s(oldPin,sizeof(oldPin),"1234");
			strcpy_s(newPin,sizeof(newPin),"4321");
			status = BEID_ChangePIN(&pin, oldPin, newPin, &triesLeft);
			CHECK_EQUAL(BEID_OK,status.general);

			status = BEID_VerifyPIN(&pin, newPin, &triesLeft);
			CHECK_EQUAL(BEID_OK,status.general);

			//-------------------------------------
			// replace the pin to its initial value
			//-------------------------------------
			status = BEID_ChangePIN(&pin, newPin, oldPin, &triesLeft);
			CHECK_EQUAL(BEID_OK,status.general);
		}
	}
	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);
}

TEST(BEIDReadFile_t1v1)
{
	//----------------------------------------------
	// data file that can be read
	// test card 1 version 1
	//----------------------------------------------
	unsigned char tuc3F00DF014031[] = {
		0x01,0x0c,0x35,0x39,0x30,0x34,0x35,0x36,0x37,0x38,0x39,0x30,0x31,0x32,0x02,0x10,
		0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x61,0x62,0x63,0x64,0x65,0x66,
		0x03,0x0a,0x30,0x31,0x2e,0x30,0x31,0x2e,0x32,0x30,0x30,0x37,0x04,0x0a,0x30,0x31,
		0x2e,0x30,0x31,0x2e,0x32,0x30,0x31,0x32,0x05,0x09,0x42,0x72,0x75,0x78,0x65,0x6c,
		0x6c,0x65,0x73,0x06,0x0b,0x38,0x30,0x30,0x31,0x32,0x35,0x31,0x31,0x32,0x36,0x35,
		0x07,0x05,0x6e,0x61,0x6d,0x65,0x30,0x08,0x06,0x67,0x6e,0x61,0x6d,0x65,0x30,0x09,
		0x00,0x0a,0x05,0x42,0x65,0x6c,0x67,0x65,0x0b,0x07,0x49,0x78,0x65,0x6c,0x6c,0x65,
		0x73,0x0c,0x0b,0x32,0x35,0x20,0x4a,0x41,0x4e,0x20,0x31,0x39,0x38,0x30,0x0d,0x01,
		0x4d,0x0e,0x06,0x50,0x72,0x69,0x6e,0x63,0x65,0x0f,0x01,0x31,0x10,0x01,0x30,0x11,
		0x14,0xdc,0x9d,0xb2,0xc8,0x3d,0xc1,0xf1,0xf7,0xd6,0x96,0x4c,0xae,0x35,0xda,0x0c,
		0xb9,0x2e,0x22,0x09,0xf0,
	};
	static long size_tuc3F00DF014031 = sizeof(tuc3F00DF014031);

	unsigned char tuc3F00DF014032[] = {
		0x47,0x4d,0x1d,0xec,0x14,0x22,0x8f,0x84,0xf4,0x12,0xac,0x32,0x89,0x55,0x16,0xda,
		0xbb,0x31,0x5f,0x79,0x5a,0x08,0x40,0x36,0x42,0x9b,0x11,0xf3,0x7e,0x63,0xcc,0xf4,
		0xe6,0x6f,0x85,0xf5,0x3b,0xae,0xcc,0x18,0x46,0x68,0xcd,0x1a,0x79,0xfd,0x81,0x0b,
		0xdb,0x30,0x3a,0x9a,0x0f,0xd4,0xad,0x54,0x67,0xc6,0x64,0xd8,0xbe,0xfb,0x51,0x36,
		0x1c,0xb5,0x99,0x0f,0x39,0x77,0x14,0x1c,0x33,0x80,0x51,0xd6,0xab,0xb3,0xab,0x56,
		0x1f,0xed,0x3d,0x07,0xc6,0x76,0x9b,0xc1,0x94,0xd3,0xdd,0x27,0x87,0x85,0x91,0x1b,
		0x06,0x4f,0xec,0x95,0xd9,0x1a,0x71,0x42,0x28,0x60,0x6b,0x3d,0xde,0x2c,0xa4,0x82,
		0xbb,0x33,0xfb,0xc6,0xe6,0x4e,0x50,0xed,0xce,0xde,0x88,0xcd,0x2c,0x99,0xc5,0x9d,
	};
	static long size_tuc3F00DF014032 = sizeof(tuc3F00DF014032);

	unsigned char tuc3F00DF014033[] = {
		0x01,0x0b,0x56,0x69,0x6c,0x61,0x6c,0x61,0x61,0x6e,0x20,0x31,0x33,0x02,0x04,0x31,
		0x36,0x30,0x31,0x03,0x09,0x52,0x75,0x69,0x73,0x62,0x72,0x6f,0x65,0x6b,
	};
	static long size_tuc3F00DF014033 = sizeof(tuc3F00DF014033);

	unsigned char tuc3F00DF014034[] = {
		0x51,0x44,0x70,0xe9,0xf2,0x5e,0x77,0xa5,0xd5,0xc8,0x7a,0x0d,0xe6,0xce,0x88,0xc0,
		0x5e,0xe1,0xef,0xac,0xe7,0xf0,0x0e,0x30,0x97,0x13,0xd9,0x06,0x1a,0xa5,0xb3,0xb6,
		0xbb,0x00,0xd0,0x17,0x2b,0xb4,0xad,0x98,0xca,0x70,0xf0,0xfe,0xd5,0xb6,0xb1,0xab,
		0x6d,0xc2,0xce,0x4f,0x15,0xac,0xea,0x81,0x3c,0xe9,0x38,0x85,0xc9,0xcb,0xe3,0x5d,
		0x07,0x4e,0xb9,0x94,0x65,0x15,0xab,0xa8,0x51,0x5e,0x03,0x4d,0x9e,0xc0,0xb2,0x00,
		0x29,0xd6,0x1d,0x07,0x8d,0xd0,0xb1,0x3b,0x9d,0xae,0x5d,0x6d,0x1a,0x38,0x75,0xd5,
		0x94,0xe3,0x42,0x43,0x9c,0x42,0x3e,0x2d,0x54,0x63,0xb6,0x51,0x45,0x4e,0x86,0xd0,
		0xcd,0x20,0xf0,0x02,0x93,0x29,0x38,0xf2,0xa3,0xd2,0x82,0x1a,0xf8,0x6d,0x27,0x62,
	};
	static long size_tuc3F00DF014034 = sizeof(tuc3F00DF014034);

	unsigned char tuc3F00DF014035[] = {
		0xff,0xd8,0xff,0xe0,0x00,0x10,0x4a,0x46,0x49,0x46,0x00,0x01,0x02,0x01,0x01,0x2c,
		0x01,0x2c,0x00,0x00,0xff,0xdb,0x00,0x43,0x00,0x1c,0x13,0x15,0x18,0x15,0x11,0x1c,
		0x18,0x16,0x18,0x1f,0x1d,0x1c,0x21,0x29,0x45,0x2d,0x29,0x26,0x26,0x29,0x54,0x3c,
		0x40,0x32,0x45,0x64,0x58,0x69,0x67,0x62,0x58,0x60,0x5f,0x6e,0x7c,0x9e,0x86,0x6e,
		0x75,0x96,0x77,0x5f,0x60,0x8a,0xbb,0x8b,0x96,0xa3,0xa9,0xb1,0xb3,0xb1,0x6b,0x84,
		0xc2,0xd0,0xc1,0xac,0xce,0x9e,0xae,0xb1,0xaa,0xff,0xc0,0x00,0x0b,0x08,0x00,0xc8,
		0x00,0x8c,0x01,0x01,0x11,0x00,0xff,0xc4,0x00,0xd2,0x00,0x00,0x01,0x05,0x01,0x01,
		0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x03,0x04,
		0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x10,0x00,0x02,0x01,0x03,0x03,0x02,0x04,0x03,
		0x05,0x05,0x04,0x04,0x00,0x00,0x01,0x7d,0x01,0x02,0x03,0x00,0x04,0x11,0x05,0x12,
		0x21,0x31,0x41,0x06,0x13,0x51,0x61,0x07,0x22,0x71,0x14,0x32,0x81,0x91,0xa1,0x08,
		0x23,0x42,0xb1,0xc1,0x15,0x52,0xd1,0xf0,0x24,0x33,0x62,0x72,0x82,0x09,0x0a,0x16,
		0x17,0x18,0x19,0x1a,0x25,0x26,0x27,0x28,0x29,0x2a,0x34,0x35,0x36,0x37,0x38,0x39,
		0x3a,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4a,0x53,0x54,0x55,0x56,0x57,0x58,0x59,
		0x5a,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0x6a,0x73,0x74,0x75,0x76,0x77,0x78,0x79,
		0x7a,0x83,0x84,0x85,0x86,0x87,0x88,0x89,0x8a,0x92,0x93,0x94,0x95,0x96,0x97,0x98,
		0x99,0x9a,0xa2,0xa3,0xa4,0xa5,0xa6,0xa7,0xa8,0xa9,0xaa,0xb2,0xb3,0xb4,0xb5,0xb6,
		0xb7,0xb8,0xb9,0xba,0xc2,0xc3,0xc4,0xc5,0xc6,0xc7,0xc8,0xc9,0xca,0xd2,0xd3,0xd4,
		0xd5,0xd6,0xd7,0xd8,0xd9,0xda,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,0xe9,0xea,
		0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,0xff,0xda,0x00,0x08,0x01,0x01,
		0x00,0x00,0x3f,0x00,0xdf,0xc9,0xf5,0x34,0x64,0xfa,0x9a,0x32,0x7d,0x4d,0x19,0x3e,
		0xa6,0x8c,0x9f,0x53,0x46,0x4f,0xa9,0xa3,0x27,0xd4,0xd2,0x16,0xc0,0xc9,0x6c,0x7e,
		0x35,0x1f,0xda,0xa1,0x1d,0x67,0x4f,0xfb,0xec,0x53,0xd6,0x55,0x7f,0xb9,0x20,0x6f,
		0xa1,0xcd,0x3b,0x27,0xd4,0xd1,0x93,0xea,0x68,0xc9,0xf5,0x34,0x64,0xfa,0x9a,0x32,
		0x7d,0x4d,0x19,0x3e,0xa6,0x8c,0x9f,0x53,0x46,0x4f,0xa9,0xa4,0xa2,0x8a,0x28,0xaa,
		0x73,0x6a,0x31,0x46,0xfb,0x57,0xe6,0xc1,0xe4,0xe7,0x8a,0xab,0x73,0xa9,0x3b,0x8f,
		0xdc,0xfc,0xab,0x8c,0x93,0x9e,0x6b,0x2a,0x4b,0xd2,0xe7,0x25,0x99,0x88,0xee,0xd9,
		0x26,0xa3,0x17,0x32,0x72,0x36,0x82,0x3d,0x0a,0xff,0x00,0xf5,0xaa,0x58,0xe7,0x72,
		0x7e,0x50,0x8c,0x7b,0x00,0x7f,0xc9,0xab,0xd0,0xea,0x32,0x21,0xc1,0xde,0x87,0xb8,
		0x3f,0x30,0xfd,0x6b,0x52,0x0b,0xc8,0xe6,0xc0,0xe8,0x4f,0xa7,0x23,0xff,0x00,0xad,
		0xf8,0xd5,0x8a,0x28,0xa2,0x8a,0x28,0xa2,0x8a,0x46,0x65,0x45,0x2c,0xc4,0x05,0x51,
		0x92,0x4f,0x40,0x2b,0x9b,0xd4,0x35,0x69,0x2e,0x58,0xa4,0x04,0xac,0x59,0xe0,0x63,
		0xaf,0xb9,0xac,0xf5,0xf9,0x9c,0xf9,0xb2,0x1c,0x0e,0xde,0xb4,0x34,0xd6,0xd9,0xc7,
		0x97,0x9c,0x0e,0xb9,0x34,0x79,0xf0,0x63,0xfd,0x58,0xfc,0xa9,0x37,0xc2,0xdd,0x13,
		0x6f,0xb8,0x6c,0x7f,0x3a,0x9d,0x77,0x60,0x79,0x6e,0x19,0x4f,0x69,0x3f,0xc4,0x54,
		0xc9,0x37,0x96,0x42,0x48,0x0c,0x64,0xf4,0x0f,0x82,0xbf,0x81,0xab,0x03,0x83,0xc8,
		0x1c,0x76,0x35,0xa9,0xa7,0xcf,0xbd,0x7c,0xb2,0x78,0x51,0xf2,0x83,0xd4,0x7b,0x55,
		0xda,0x28,0xa2,0x8a,0x28,0xa2,0xb1,0xb5,0xcb,0x96,0x91,0x96,0xce,0x16,0xc7,0xf1,
		0x4a,0x47,0x61,0xd8,0x7f,0x5f,0xca,0xb2,0x7e,0x44,0x52,0xa9,0x8f,0x73,0x55,0x27,
		0xdb,0x9f,0x94,0xf3,0xf5,0xa8,0x83,0x7f,0x7b,0x9a,0x4c,0x7a,0x52,0xab,0x76,0x65,
		0xc8,0xa9,0x51,0x33,0xcc,0x2c,0x43,0x7a,0x67,0x07,0xf3,0xa9,0x96,0xea,0x44,0x1b,
		0x26,0x40,0xeb,0xdd,0x58,0x55,0x88,0x64,0x08,0x99,0xb7,0x6d,0xf1,0x8e,0xa8,0x7a,
		0xaf,0xd3,0xfc,0x2a,0xe4,0x32,0x2b,0x61,0xd4,0xed,0xc1,0xe1,0x87,0x05,0x4d,0x6c,
		0xda,0xdc,0x79,0xf1,0xfc,0xd8,0x12,0x0f,0xbc,0xa3,0xf9,0xd4,0xf4,0x51,0x45,0x14,
		0x53,0x26,0x90,0x43,0x0b,0xca,0xc3,0x21,0x14,0xb6,0x07,0x7c,0x76,0xae,0x62,0xea,
		0x46,0x3b,0x9d,0x8e,0x19,0xdb,0x73,0x9f,0xe8,0x2b,0x36,0x49,0x4b,0x70,0x0e,0x05,
		0x0a,0x85,0xce,0x31,0x9a,0x9d,0x2c,0x9d,0x97,0x27,0x8a,0x63,0x5b,0xb2,0x9f,0xeb,
		0x48,0x21,0xdc,0xa7,0xa0,0x61,0xda,0x98,0x51,0x94,0xf4,0x23,0x15,0x34,0x72,0xab,
		0x0d,0xb3,0x82,0x47,0xf7,0x87,0x51,0x4a,0x55,0xed,0xc8,0x92,0x32,0x1b,0xdc,0x74,
		0x61,0xe8,0x45,0x59,0x86,0x65,0x90,0x87,0x8f,0xe4,0x27,0x86,0x52,0x7a,0xff,0x00,
		0xf5,0xfd,0x3d,0x6b,0x4e,0xd6,0x53,0x1b,0x2c,0x89,0x93,0xb7,0x82,0xbe,0xde,0x95,
		0xb0,0x08,0x20,0x10,0x72,0x0f,0x20,0xd2,0xd1,0x45,0x14,0x55,0x0d,0x5e,0x60,0x96,
		0xeb,0x19,0x07,0x0e,0x77,0x13,0x8e,0x81,0x70,0x7f,0x9e,0x3f,0x5a,0xe5,0xae,0x65,
		0x2e,0xdc,0xfa,0xd3,0x60,0x84,0xca,0xe0,0x0a,0xdc,0xb4,0xb0,0x0a,0x01,0x22,0xb4,
		0x05,0xaa,0xec,0xe9,0xda,0xaa,0x5c,0xd8,0xe3,0x90,0x2b,0x32,0x68,0xb6,0x7b,0x1e,
		0xc7,0xd2,0xab,0x82,0xa3,0x83,0xf4,0xe3,0xb1,0xa8,0xa5,0x5e,0xeb,0x4d,0x8e,0x52,
		0xbc,0x75,0x0d,0xd4,0x52,0x9f,0x91,0xb7,0xaf,0xdd,0x61,0x86,0xad,0x7b,0x09,0xcb,
		0x27,0x27,0x2e,0xbc,0x30,0xfe,0x55,0xb9,0x68,0x73,0x16,0xdc,0xe4,0x0e,0x9f,0x4a,
		0x9e,0x8a,0x28,0xa2,0xb0,0xf5,0xf9,0x71,0x26,0xcd,0xd9,0x01,0x00,0x23,0xd0,0xf2,
		0x4f,0xfe,0xcb,0x5c,0xf0,0xcb,0x35,0x6d,0x69,0x76,0xe0,0x28,0x62,0x3a,0xfb,0x56,
		0xd2,0x2f,0x4e,0x95,0x30,0xe9,0x41,0x00,0x8e,0x6a,0x9d,0xdd,0x92,0xcc,0x84,0xa8,
		0xc3,0x7f,0x3a,0xe7,0x2f,0x2d,0xde,0x19,0x09,0xda,0x45,0x55,0x0e,0x7a,0x1e,0x94,
		0xd7,0x03,0x39,0x1d,0xe9,0xc8,0xdf,0xc3,0xd8,0x8f,0xce,0xac,0xd8,0xc9,0xe5,0x4e,
		0x37,0x74,0xe8,0x7e,0x95,0xd2,0xd8,0xb9,0x0d,0xb0,0x9e,0x6a,0xf5,0x14,0x51,0x45,
		0x72,0x7a,0xa4,0xbe,0x65,0xc5,0xce,0x58,0x13,0xe6,0x1f,0xc8,0x1c,0x0f,0xd0,0x0a,
		0xcf,0x8c,0x66,0x45,0x1e,0xa7,0x15,0xd2,0xd9,0xae,0xd8,0xc0,0x1c,0x55,0xf5,0xcf,
		0xa5,0x4a,0x29,0xd8,0xa3,0x15,0x4e,0xf6,0xc5,0x6e,0x54,0xe3,0x01,0xbd,0x7d,0x6b,
		0x9a,0xbd,0xb1,0x92,0xd9,0xce,0x54,0xe2,0xaa,0x88,0xcb,0x02,0x54,0xfe,0x14,0xcc,
		0x11,0xd7,0x8a,0x9e,0x32,0x37,0x29,0x3d,0x09,0xda,0x6b,0xa1,0xb3,0x90,0xe6,0x16,
		0x63,0xce,0x02,0xb1,0xf7,0xad,0x7a,0x28,0xa2,0x82,0xc1,0x41,0x66,0x38,0x50,0x32,
		0x7e,0x95,0xc2,0x33,0x92,0xe4,0xb1,0xc9,0x27,0x9f,0x7a,0x7d,0xa2,0xef,0xb9,0x41,
		0xfe,0xd6,0x6b,0xa7,0x84,0x04,0x51,0xb8,0x81,0xc7,0x7a,0x9c,0x4f,0x1a,0xf5,0x3c,
		0x7a,0xd4,0xab,0x2c,0x6d,0xf7,0x5d,0x4f,0xe3,0x52,0x02,0x28,0xcd,0x04,0x81,0xd6,
		0xa1,0x9e,0x38,0xe5,0x8c,0x89,0x30,0x47,0xbd,0x61,0xdd,0xe9,0x91,0x07,0x0d,0x6f,
		0x28,0x27,0xfb,0xbd,0x6a,0x85,0xcd,0xac,0x85,0x0b,0x18,0xc6,0x57,0xba,0x90,0x73,
		0xf8,0x55,0x64,0xf9,0xb3,0x9e,0xe3,0x15,0xb9,0x6e,0xe3,0x64,0x64,0x7f,0x7f,0xfa,
		0x56,0xf2,0x1d,0xc8,0xa7,0xd4,0x66,0x96,0x8a,0x2a,0xa6,0xaa,0xfb,0x34,0xd9,0x88,
		0xea,0x46,0xde,0xb8,0xea,0x70,0x7f,0x4c,0xd7,0x35,0x6d,0xb6,0x58,0x96,0x29,0x23,
		0x4d,0xac,0xa4,0x86,0x1f,0x7b,0x3e,0xb4,0xba,0x4c,0x3b,0xae,0x86,0x46,0x71,0xde,
		0xb6,0xe5,0x75,0x56,0xcd,0x40,0xfa,0x84,0x2a,0x30,0xcb,0xc1,0xef,0xc6,0x2a,0xb9,
		0xbc,0x80,0xb8,0xf2,0x98,0xe4,0xfa,0x73,0x57,0xac,0xef,0x43,0x10,0x37,0x86,0x07,
		0x8c,0xe6,0xb4,0x41,0xc8,0xcd,0x55,0xba,0xb8,0xf2,0xc1,0x19,0x22,0xb2,0xe5,0xba,
		0xe7,0xe6,0x66,0x23,0xbf,0xa0,0xa9,0x12,0xee,0xdd,0x40,0xda,0xd1,0xb1,0xf4,0xde,
		0x09,0xa7,0x49,0x24,0x53,0x29,0xf9,0x70,0x7d,0xeb,0x04,0xaf,0x97,0x3b,0x27,0xa1,
		0xad,0x2b,0x39,0x0f,0x93,0xc9,0xfb,0xae,0x31,0xfe,0x7f,0x1a,0xe9,0x2d,0x8e,0x6d,
		0xd0,0xfe,0x15,0x25,0x14,0x56,0x57,0x88,0xa4,0xd9,0x62,0xab,0xdd,0x9b,0x3f,0x80,
		0x18,0xfe,0xa2,0xb2,0x2c,0x88,0x7b,0x55,0x66,0x1f,0xea,0x8b,0x28,0xfe,0x7f,0xd6,
		0xae,0xe9,0x4a,0x3c,0xc7,0x60,0x31,0x56,0xee,0x2d,0xcc,0x8b,0x90,0x71,0x59,0x46,
		0xc0,0x2b,0xc8,0x25,0x52,0xe1,0xc6,0x03,0x77,0x43,0xf4,0xf4,0xa5,0xb2,0xb4,0x95,
		0xa4,0x41,0x74,0x77,0xc5,0x16,0x4a,0x20,0xf5,0x20,0x0f,0xcb,0x81,0x57,0x52,0xcb,
		0x6b,0x96,0x04,0xf2,0x78,0xc9,0xe7,0xf1,0xad,0x78,0x87,0xee,0x86,0x7a,0xe2,0xa8,
		0xdd,0x41,0xe6,0xcb,0x82,0x71,0x59,0xf2,0xe9,0xa8,0xc8,0xe8,0xea,0xdb,0xf1,0xf2,
		0x3a,0x9e,0x14,0xfd,0x2a,0xbc,0x16,0x52,0x64,0x9b,0xa4,0x32,0x32,0xae,0xc4,0x50,
		0x46,0x31,0xef,0x56,0xac,0x6c,0x1e,0x25,0xc3,0x1e,0x3d,0x3d,0x3e,0x95,0x9f,0xaa,
		0x43,0xe4,0x5e,0x86,0x1f,0x75,0xd7,0xfc,0xff,0x00,0x4a,0x4b,0x47,0xca,0x14,0xf7,
		0x26,0xba,0x6d,0x35,0xf7,0x5b,0x0e,0xb5,0x6e,0x8a,0x2b,0x03,0xc4,0x8d,0xf3,0x80,
		0x1b,0x8d,0xa0,0x63,0xd0,0xe4,0x9f,0xf0,0xaa,0x1a,0x4b,0x06,0x12,0xc0,0xcd,0x8c,
		0xe0,0x8c,0x7e,0xbf,0xd2,0xb5,0x6c,0xd0,0xc7,0x3b,0x29,0xe9,0x8e,0x0f,0xad,0x68,
		0x22,0xe7,0xad,0x23,0xc0,0x0f,0x6a,0x44,0xb7,0x51,0xfc,0x38,0xa7,0xf9,0x63,0x77,
		0xa6,0x2a,0x55,0xe0,0x53,0x08,0xc9,0xc1,0xa4,0x68,0x43,0x0f,0x7f,0x5a,0x6f,0xd9,
		0xc6,0x73,0x4e,0x29,0xb5,0x6b,0x0f,0x5f,0x50,0x22,0x88,0xf1,0x9d,0xdf,0xd2,0xb2,
		0xed,0x9b,0x13,0x2e,0x7b,0x9c,0x57,0x4b,0xa3,0xb6,0x63,0x2b,0xd8,0x7e,0x95,0xa5,
		0x45,0x15,0xcd,0x78,0x83,0x22,0x79,0x33,0xd0,0xba,0xe3,0xfe,0xf9,0x15,0x99,0x04,
		0xa2,0x0b,0x84,0x98,0x03,0x81,0x8c,0x8f,0x62,0x39,0xad,0xc8,0xe5,0x5f,0xb4,0x23,
		0x2b,0x82,0x8c,0x7e,0x52,0x0d,0x69,0x23,0x7c,0xc4,0x54,0xe2,0x8e,0x2a,0x17,0x62,
		0xcc,0x42,0xf4,0x15,0x3a,0xf2,0x2a,0x27,0x38,0x3f,0x2f,0x24,0x53,0xe3,0x70,0xeb,
		0x91,0xf9,0x53,0xce,0x2a,0x29,0x5b,0x0b,0x5c,0xce,0xb7,0x36,0xfb,0x94,0x41,0xd1,
		0x41,0x3f,0xe7,0xf2,0xac,0xf8,0xdb,0x0c,0xa7,0xd0,0xd6,0xfe,0x87,0x3f,0xef,0xca,
		0x9f,0xe2,0x18,0xad,0xda,0x28,0xae,0x6b,0xc4,0x21,0x85,0xc4,0x99,0x3c,0x16,0x52,
		0x07,0xb6,0xd0,0x3f,0xa1,0xac,0x86,0xc1,0x40,0x79,0xe9,0x49,0x0b,0x88,0xe7,0x47,
		0x27,0x85,0x60,0xc7,0xf3,0xae,0xb1,0x5c,0x70,0xd5,0x65,0x1f,0x8a,0x79,0xf9,0x85,
		0x52,0xba,0x8a,0xe5,0x43,0x79,0x2d,0x8c,0xf7,0xe0,0xff,0x00,0x3a,0x8a,0x3b,0xa9,
		0x91,0x42,0x30,0xcb,0x1e,0x0e,0x05,0x3a,0x34,0xbb,0x33,0xee,0xdf,0xf2,0x9e,0xab,
		0xc6,0x3f,0xc6,0xae,0xa2,0x95,0x39,0x14,0xad,0x20,0x15,0x5a,0xe6,0x5f,0x93,0xad,
		0x72,0xd7,0xcf,0xbe,0xee,0x43,0x9c,0xe0,0xe2,0xa2,0x5c,0x71,0xf5,0xe6,0xaf,0xe9,
		0xd2,0x94,0x99,0x48,0x38,0x65,0x39,0x15,0xd7,0xfd,0x3a,0x51,0x45,0x61,0x78,0x96,
		0x03,0xfb,0xb9,0xf9,0x2a,0x46,0xc3,0xec,0x46,0x48,0xfe,0x67,0xf2,0xac,0x40,0x84,
		0xdb,0x67,0xd0,0xd4,0x1d,0xfe,0xb5,0xbf,0xa7,0xdc,0x19,0xad,0x14,0xb1,0xc9,0x1c,
		0x1f,0xad,0x68,0x44,0x4f,0x63,0x53,0x89,0x31,0x4b,0x24,0xd1,0xa2,0xe5,0xdc,0x28,
		0xf7,0x35,0x02,0xdc,0xda,0xb9,0xe2,0x64,0xe3,0xf0,0x34,0xe5,0xbc,0xb5,0xdd,0x81,
		0x32,0xe7,0xdf,0x8a,0x90,0xb8,0x61,0x90,0x47,0xb1,0x1c,0xd4,0x72,0x9f,0x6a,0xce,
		0xbb,0x90,0x24,0x6c,0xc4,0xf0,0xa3,0x35,0xce,0x9c,0x93,0x93,0xde,0x95,0x4e,0x08,
		0xab,0x16,0xe7,0x0e,0xac,0xa7,0x90,0x6b,0xb2,0xb5,0x7f,0x32,0xda,0x36,0xff,0x00,
		0x64,0x0a,0x96,0x8a,0xa3,0xac,0xc5,0xe6,0xe9,0xb2,0x71,0x92,0xa4,0x30,0xfc,0xff,
		0x00,0xc0,0x9a,0xe7,0xfc,0xb0,0x96,0x59,0x3d,0x73,0x59,0xec,0x30,0xc7,0xd8,0xd5,
		0xcd,0x2e,0x63,0x1c,0xcd,0x19,0x3f,0x2b,0x0c,0xfe,0x35,0xb9,0x0c,0x82,0x9f,0x26,
		0x5f,0xa3,0x11,0x54,0xe6,0xb1,0x0c,0xfb,0xdd,0xde,0x5f,0x66,0x34,0x82,0xd6,0xd0,
		0xf5,0x49,0x01,0xf6,0x63,0x4e,0x36,0x56,0xee,0x30,0x91,0x91,0xee,0xc4,0xe6,0xac,
		0x41,0x6c,0x96,0xe0,0x08,0xcb,0x63,0xd3,0x71,0x34,0xf9,0xa4,0x01,0x71,0x58,0x9a,
		0xa4,0xe0,0x81,0x12,0x9e,0x4f,0x2d,0xcd,0x67,0x01,0x4b,0x8c,0xb0,0xa9,0xa1,0x8d,
		0x98,0xe5,0x06,0x70,0x09,0x3f,0x4a,0xec,0x2c,0x10,0x25,0xa2,0x61,0x98,0x86,0x19,
		0xc1,0xe8,0x0f,0x7c,0x55,0x8a,0x29,0x0a,0x86,0x52,0xac,0x32,0x08,0xc1,0x1e,0xa2,
		0xb9,0x8d,0x48,0xa8,0x41,0x1a,0x0e,0x17,0x82,0x7d,0xeb,0x2d,0x86,0x5b,0xeb,0x8a,
		0x20,0x6d,0x92,0xa9,0x15,0xb3,0x1c,0x98,0xe0,0xfe,0x75,0x7a,0x12,0x1d,0x78,0x3c,
		0xd4,0xeb,0x06,0xe1,0x8c,0x8a,0x43,0x69,0x4e,0xfb,0x38,0x41,0x51,0xc8,0xc2,0x31,
		0xcf,0x4a,0xce,0xba,0xb8,0xda,0xa7,0x1c,0x91,0x58,0x45,0x8b,0xbe,0xe6,0xe4,0x93,
		0x4e,0xcf,0x24,0xf6,0xa1,0x3a,0x56,0xae,0x8f,0x0b,0xc8,0xec,0x14,0x91,0xbd,0x76,
		0x9c,0x77,0x04,0xf3,0xfc,0xab,0xa7,0x55,0x0a,0xa1,0x47,0x40,0x31,0x4b,0x45,0x55,
		0xbd,0xb9,0xf2,0xe3,0x28,0x8d,0x87,0xc7,0x27,0xd0,0x57,0x39,0x70,0xdb,0x99,0xff,
		0x00,0x31,0x54,0xdb,0xa2,0x9c,0x7b,0x54,0x47,0x82,0x78,0xad,0x68,0xce,0xe8,0x55,
		0x87,0x71,0x4e,0x8e,0xe1,0xa1,0x39,0x3d,0x2b,0x46,0x1b,0xf8,0xd8,0x73,0x80,0x6a,
		0x71,0x76,0x84,0x75,0x1f,0x9d,0x47,0x25,0xec,0x60,0x7d,0xe1,0xf9,0xd6,0x74,0xd7,
		0x2f,0x70,0xd8,0x40,0x70,0x7b,0xd4,0x53,0xc7,0xb2,0xdd,0x8e,0x0e,0x48,0xac,0x71,
		0x4a,0x4f,0x18,0xa7,0xa9,0xc2,0xf4,0xae,0x83,0xc3,0xd2,0xc6,0xa8,0xc1,0x8e,0x19,
		0xb0,0x01,0xf6,0xff,0x00,0xf5,0xd6,0xe5,0x15,0x4a,0x5b,0x8d,0x89,0x9b,0x83,0xb5,
		0xba,0x84,0x15,0x97,0x35,0xc1,0x6d,0xc4,0x93,0x8e,0xb8,0x3f,0xce,0xb3,0x2e,0x1b,
		0x18,0x3d,0x7a,0xe6,0xa2,0x5f,0x98,0x91,0x9e,0xf9,0xa8,0xd8,0x10,0x2b,0x42,0xc5,
		0xb7,0x43,0xb4,0xf6,0x35,0x2b,0xaf,0x04,0x54,0x0f,0x1f,0xa7,0x5a,0x66,0xd9,0xfa,
		0x29,0xfd,0x6a,0x68,0xad,0x89,0x20,0xbf,0x35,0x76,0x28,0x80,0xc6,0x05,0x45,0x78,
		0x7f,0x74,0xc0,0x7a,0x56,0x29,0x5c,0x73,0x48,0x54,0x83,0x46,0xe2,0x48,0x1e,0xf5,
		0x3c,0x57,0x32,0xc5,0xb5,0x91,0xc8,0x60,0x30,0x3b,0xf1,0xe9,0x5a,0x56,0xda,0xdc,
		0xf1,0x80,0xb2,0x63,0x6f,0xa8,0x15,0xb2,0x97,0x37,0x45,0x41,0x30,0x02,0x0f,0x20,
		0x8e,0x84,0x57,0x3e,0xad,0x31,0x3f,0x3c,0x81,0x7f,0x53,0x51,0x4d,0x2e,0x1f,0xcb,
		0x53,0xc7,0x73,0xea,0x6a,0xac,0xaf,0xbd,0xcf,0xa0,0xa6,0x2b,0xe1,0xf3,0xda,0xa4,
		0x73,0xb9,0x73,0xde,0xac,0x58,0x3e,0x19,0x97,0xd7,0x15,0xa1,0xd4,0x50,0x17,0x3d,
		0x45,0x48,0xb1,0x8e,0xf4,0xed,0xa0,0x70,0x3a,0xd3,0x88,0xc0,0xcd,0x57,0x9f,0x95,
		0xc5,0x65,0xc8,0x98,0x72,0xb8,0xc7,0x26,0xa4,0xb5,0x84,0x4d,0x87,0x73,0x80,0x2a,
		0x9b,0x10,0x64,0x24,0x70,0xb9,0xe3,0xe9,0x46,0x79,0xa3,0x71,0xe2,0xa6,0x4b,0x99,
		0x95,0x40,0x59,0x5c,0x01,0xd8,0x31,0xa5,0x33,0x9c,0x1d,0xbd,0x4f,0x15,0x13,0x1d,
		0xa0,0x8c,0xfc,0xc7,0xaf,0xb5,0x47,0x40,0x1c,0xd3,0xc1,0x3b,0x6a,0x6b,0x73,0xb6,
		0x40,0x7d,0xeb,0x52,0x36,0xe3,0x1e,0xd5,0x20,0xf5,0xa5,0xc9,0xa7,0x03,0x4a,0x49,
		0x35,0x11,0x5c,0xf6,0xaa,0x97,0x10,0xe0,0x87,0xc7,0x43,0x93,0x55,0x25,0x76,0x84,
		0x3c,0x4a,0x4e,0x09,0xcd,0x56,0xef,0x41,0xe2,0x90,0x52,0xe7,0x14,0xed,0xea,0x83,
		0xe5,0x19,0x3e,0xa6,0x98,0x49,0x34,0x0a,0x72,0x7d,0xef,0xa5,0x03,0xbd,0x5a,0xb7,
		0x4c,0xb6,0x71,0xe9,0x5a,0x08,0x3d,0xbb,0x54,0xa0,0x0c,0xf1,0x4b,0x4f,0x00,0xd3,
		0xd6,0x33,0xdc,0x53,0xd2,0x1c,0x9a,0x65,0xd4,0x4b,0x1c,0x6c,0xcf,0xc0,0x03,0x9a,
		0xe7,0x70,0x25,0xde,0xc4,0x90,0xdc,0x6d,0x18,0xce,0x7b,0x53,0x18,0x15,0x7d,0xa7,
		0xa8,0x38,0x34,0xc3,0x9c,0xd2,0xd1,0x49,0x45,0x14,0xbf,0x4f,0xc6,0x9c,0x80,0xb1,
		0x00,0x75,0x35,0xb3,0x69,0x6d,0xfb,0xbc,0x8e,0xe7,0xf4,0xab,0x9f,0x67,0x38,0xfa,
		0x0a,0x55,0x87,0x27,0x9a,0x77,0x91,0xcd,0x38,0x42,0x41,0xe9,0x53,0xa4,0x27,0xd2,
		0xa5,0x54,0x0a,0x39,0x15,0x8b,0xe2,0x0b,0x9d,0xa1,0x60,0x46,0xe5,0x8e,0x5b,0x07,
		0xb5,0x61,0x06,0x65,0x39,0x04,0x83,0xea,0x0d,0x27,0x42,0x3e,0xb4,0x1e,0xb4,0x51,
		0x9a,0x4a,0x01,0xc1,0xcd,0x2e,0x68,0x03,0xbd,0x28,0x24,0x74,0x35,0xd3,0xe9,0x92,
		0x25,0xc5,0xb2,0xc8,0xa3,0x04,0x7c,0xac,0x3d,0x0d,0x5f,0x08,0x29,0x42,0x73,0x9e,
		0xb4,0xbe,0x58,0xa5,0x08,0x01,0xa7,0xd4,0x53,0xca,0xb1,0x44,0xce,0xc4,0x00,0x06,
		0x49,0x35,0xc7,0x5d,0xdc,0x1b,0x8b,0x87,0x94,0xe7,0xe6,0x3c,0x67,0xb0,0xed,0x50,
		0x0a,0x3b,0x8a,0x28,0xa5,0x15,0x3f,0xf6,0x75,0xf7,0xfc,0xf9,0xdc,0x7f,0xdf,0xa6,
		0xff,0x00,0x0a,0x4f,0xec,0xeb,0xef,0xf9,0xf3,0xb8,0xff,0x00,0xbf,0x4d,0xfe,0x14,
		0xa3,0x4f,0xbd,0xff,0x00,0x9f,0x3b,0x8f,0xfb,0xf4,0xdf,0xe1,0x4b,0xfd,0x9f,0x7d,
		0xff,0x00,0x3e,0x77,0x1f,0xf7,0xe9,0xbf,0xc2,0x81,0xa7,0xdf,0x7f,0xcf,0x9d,0xc7,
		0xfd,0xfa,0x6f,0xf0,0xad,0x2d,0x16,0x2b,0xcb,0x6b,0x83,0x1c,0x96,0xb3,0x88,0xe4,
		0xee,0x63,0x38,0x04,0x56,0xf8,0x46,0xc7,0xdc,0x6f,0xca,0x9c,0x11,0xbf,0xba,0xdf,
		0x95,0x3b,0x6b,0x63,0xee,0x9f,0xca,0x82,0xad,0xfd,0xd3,0xf9,0x53,0x08,0x7e,0x7e,
		0x46,0xfc,0xab,0x27,0x5a,0x17,0x4f,0x00,0x8a,0x08,0x26,0x6d,0xe7,0xe6,0x2a,0x84,
		0xe0,0x56,0x4a,0x68,0xf7,0xb2,0x23,0xb7,0x90,0xea,0x55,0x37,0xe1,0x90,0x8c,0xfb,
		0x0f,0x7f,0x6a,0x87,0xfb,0x3a,0xf7,0xfe,0x7c,0xee,0x3f,0xef,0xdb,0x7f,0x85,0x27,
		0xf6,0x75,0xee,0x7f,0xe3,0xce,0xe3,0xfe,0xfd,0x37,0xf8,0x51,0xfd,0x9d,0x7d,0xff,
		0x00,0x3e,0x77,0x1f,0xf7,0xe9,0xbf,0xc2,0x8f,0xec,0xfb,0xef,0xf9,0xf3,0xb8,0xff,
		0x00,0xbf,0x4d,0xfe,0x14,0xbf,0xd9,0xf7,0xbf,0xf3,0xe7,0x71,0xff,0x00,0x7e,0x9b,
		0xfc,0x2b,0xff,0xd9,
	};
	static long size_tuc3F00DF014035 = sizeof(tuc3F00DF014035);

	unsigned char tuc3F00DF005032[] = {
		0x30,0x27,0x02,0x01,0x00,0x04,0x10,0x53,0x4c,0x49,0x4e,0x33,0x66,0x00,0x29,0x6c,
		0xff,0x26,0x23,0x00,0x00,0x00,0xe1,0x80,0x06,0x42,0x45,0x4c,0x50,0x49,0x43,0x03,
		0x02,0x04,0x30,0x9e,0x04,0x04,0x02,0x00,0x00,
	};
	static long size_tuc3F00DF005032 = sizeof(tuc3F00DF005032);

	unsigned char tuc3F00DF00503C[] = {
		0x30,0x82,0x03,0xd6,0x30,0x82,0x02,0xbe,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,0x01,
		0x00,0x00,0x00,0x00,0x00,0xfb,0x7c,0x59,0x10,0x24,0x00,0x00,0xce,0xc0,0x02,0x30,
		0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x30,0x3b,
		0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x0e,0x30,
		0x0c,0x06,0x03,0x55,0x04,0x0a,0x13,0x05,0x5a,0x45,0x54,0x45,0x53,0x31,0x1c,0x30,
		0x1a,0x06,0x03,0x55,0x04,0x03,0x13,0x13,0x5a,0x45,0x54,0x45,0x53,0x20,0x47,0x6f,
		0x76,0x65,0x72,0x6e,0x6d,0x65,0x6e,0x74,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,0x30,
		0x34,0x30,0x33,0x32,0x33,0x31,0x30,0x31,0x30,0x30,0x32,0x5a,0x17,0x0d,0x30,0x39,
		0x30,0x33,0x32,0x33,0x31,0x30,0x31,0x30,0x30,0x32,0x5a,0x30,0x21,0x31,0x0b,0x30,
		0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x12,0x30,0x10,0x06,0x03,
		0x55,0x04,0x03,0x13,0x09,0x5a,0x45,0x54,0x45,0x53,0x20,0x52,0x4e,0x20,0x30,0x81,
		0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,
		0x03,0x81,0x8d,0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb4,0xb8,0x1d,0xed,0xc5,
		0x2d,0x80,0x31,0x42,0x08,0x76,0x44,0xac,0x96,0xde,0x35,0x7c,0xbc,0xc6,0xcd,0xec,
		0x48,0x73,0xfd,0x84,0x0e,0x19,0x90,0x18,0x5a,0x18,0x15,0x18,0x6f,0xca,0xab,0x63,
		0x10,0xea,0x7a,0x0d,0x5d,0x4b,0x74,0x01,0xbe,0xcb,0x18,0x1b,0x97,0x2d,0x7a,0x41,
		0x08,0x04,0x0f,0xbb,0x61,0xf7,0xb7,0xc1,0x8d,0xe1,0x00,0xb5,0xc9,0x8f,0x1b,0x3f,
		0xdf,0x9a,0x1f,0x88,0xe6,0x07,0x80,0xb9,0xf8,0x2b,0xb1,0x69,0xc2,0xed,0x3b,0xfb,
		0xc2,0x47,0xea,0x08,0x2b,0xae,0xd2,0x80,0x43,0xdd,0xc0,0x23,0x1e,0x0a,0x4e,0xe7,
		0x0d,0x3d,0x41,0x97,0xb7,0x20,0x22,0xd9,0x40,0x6b,0x85,0xe2,0x45,0x5a,0x90,0x8f,
		0xbf,0xc7,0x04,0xa8,0xa7,0x69,0x32,0xb0,0x1e,0x7d,0xfb,0x02,0x03,0x01,0x00,0x01,
		0xa3,0x82,0x01,0x72,0x30,0x82,0x01,0x6e,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,
		0x01,0xff,0x04,0x04,0x03,0x02,0x06,0xc0,0x30,0x1d,0x06,0x03,0x55,0x1d,0x0e,0x04,
		0x16,0x04,0x14,0x82,0xce,0x78,0xd3,0x3f,0x6f,0x41,0x41,0xce,0xa1,0xec,0x04,0x89,
		0xcf,0x2d,0xb8,0xcb,0xc8,0x5a,0xc9,0x30,0x42,0x06,0x03,0x55,0x1d,0x20,0x04,0x3b,
		0x30,0x39,0x30,0x37,0x06,0x07,0x60,0x38,0x01,0x01,0x01,0x03,0x01,0x30,0x2c,0x30,
		0x2a,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x02,0x01,0x16,0x1e,0x68,0x74,0x74,
		0x70,0x3a,0x2f,0x2f,0x72,0x65,0x70,0x6f,0x73,0x69,0x74,0x6f,0x72,0x79,0x2e,0x65,
		0x69,0x64,0x2e,0x7a,0x65,0x74,0x65,0x73,0x2e,0x62,0x65,0x30,0x3c,0x06,0x03,0x55,
		0x1d,0x1f,0x04,0x35,0x30,0x33,0x30,0x31,0xa0,0x2f,0xa0,0x2d,0x86,0x2b,0x68,0x74,
		0x74,0x70,0x3a,0x2f,0x2f,0x63,0x72,0x6c,0x2e,0x65,0x69,0x64,0x2e,0x7a,0x65,0x74,
		0x65,0x73,0x2e,0x62,0x65,0x2f,0x62,0x65,0x6c,0x67,0x69,0x75,0x6d,0x2f,0x62,0x65,
		0x6c,0x67,0x69,0x75,0x6d,0x2e,0x63,0x72,0x6c,0x30,0x7c,0x06,0x08,0x2b,0x06,0x01,
		0x05,0x05,0x07,0x01,0x01,0x04,0x70,0x30,0x6e,0x30,0x3f,0x06,0x08,0x2b,0x06,0x01,
		0x05,0x05,0x07,0x30,0x02,0x86,0x33,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x77,0x77,
		0x77,0x2e,0x5a,0x65,0x74,0x65,0x73,0x43,0x61,0x72,0x64,0x73,0x2e,0x62,0x65,0x2f,
		0x64,0x75,0x6d,0x6d,0x79,0x2f,0x63,0x65,0x72,0x74,0x69,0x66,0x63,0x61,0x74,0x65,
		0x73,0x2f,0x74,0x65,0x73,0x74,0x2e,0x63,0x72,0x74,0x30,0x2b,0x06,0x08,0x2b,0x06,
		0x01,0x05,0x05,0x07,0x30,0x01,0x86,0x1f,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x6f,
		0x63,0x73,0x70,0x2e,0x5a,0x65,0x74,0x65,0x73,0x43,0x61,0x72,0x64,0x73,0x2e,0x62,
		0x65,0x2f,0x64,0x75,0x6d,0x6d,0x79,0x30,0x11,0x06,0x09,0x60,0x86,0x48,0x01,0x86,
		0xf8,0x42,0x01,0x01,0x04,0x04,0x03,0x02,0x04,0xb0,0x30,0x09,0x06,0x03,0x55,0x1d,
		0x13,0x04,0x02,0x30,0x00,0x30,0x1f,0x06,0x03,0x55,0x1d,0x23,0x04,0x18,0x30,0x16,
		0x80,0x14,0x46,0xad,0x98,0xa1,0xc0,0x9a,0x48,0x5e,0x61,0xa0,0xbc,0x7e,0x0b,0x50,
		0x02,0x82,0x03,0xb8,0x4d,0x46,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
		0x01,0x01,0x05,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x23,0xab,0x0b,0xf7,0xb9,0x25,
		0x13,0x94,0xd9,0x35,0xb6,0x72,0xc8,0xa2,0x6e,0xe5,0x39,0x6d,0x40,0xcd,0xf6,0x3b,
		0x40,0x17,0x25,0x41,0x3e,0x06,0x39,0x75,0x56,0x9c,0x8d,0x41,0x13,0x09,0xcd,0x44,
		0x70,0xfc,0xb0,0x2d,0x10,0x2d,0x8a,0x8a,0xca,0x18,0xe1,0x47,0x7f,0x63,0xf6,0xbb,
		0x02,0x67,0xf5,0xc9,0xf1,0x10,0x12,0xad,0xeb,0x3b,0xaa,0xbe,0x18,0x67,0x35,0x52,
		0xb6,0xe9,0xe9,0xf7,0x36,0xd8,0x31,0xc2,0x1e,0x29,0x34,0x29,0x48,0x56,0x79,0x54,
		0xdc,0xd9,0xa9,0x8c,0xc4,0x67,0x70,0x97,0xbd,0xa6,0x8e,0xe3,0xb8,0x82,0x9b,0x54,
		0x8f,0xe5,0x92,0x8a,0xea,0x11,0x69,0x24,0x0f,0xc7,0x26,0x3a,0x7f,0x5d,0x42,0xa5,
		0x4f,0x17,0x05,0xed,0x1f,0x3b,0xc0,0x01,0x73,0x57,0x4e,0xf2,0x67,0x18,0x49,0x12,
		0xc5,0xc5,0x78,0x59,0x4d,0x2d,0x6a,0x37,0x4f,0x7f,0x1d,0xbe,0x17,0x90,0xad,0x8b,
		0xa7,0x85,0x96,0xa7,0x9d,0xc0,0x28,0x50,0xf6,0x15,0xf9,0xd2,0x2c,0xb8,0x54,0x93,
		0x1b,0xa3,0x57,0xc3,0xf8,0x7c,0x75,0xef,0x64,0xb6,0xbd,0x7b,0xd4,0x8d,0xb2,0x76,
		0x04,0x68,0x23,0x09,0xfe,0xd5,0xef,0x36,0x9b,0x96,0x89,0x91,0xb9,0x08,0x68,0x3f,
		0xca,0x39,0x8b,0x98,0x54,0x2e,0x27,0x0d,0x8f,0x5b,0x2e,0x9a,0x20,0xed,0xf8,0x2e,
		0xf5,0x81,0xf4,0xcb,0x61,0xd7,0x22,0xb9,0x34,0x1e,0xe6,0x6d,0x0f,0xe9,0x52,0x23,
		0x9b,0xa2,0x80,0x0b,0x50,0x47,0xd1,0x26,0xea,0xb8,0x32,0x72,0x82,0x27,0xfb,0x61,
		0x0d,0x4f,0xfa,0x99,0x88,0xad,0x72,0x36,0x33,0x26,
	};
	static long size_tuc3F00DF00503C = sizeof(tuc3F00DF00503C);

	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	//-------------------------------------
	// use correct parameters
	// test card 1 version 1
	//-------------------------------------
	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V1).getName();

	status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
	CHECK_EQUAL(BEID_OK,status.general);

	//-------------------------------------
	// get the PINs from the card
	//-------------------------------------
	::BEID_Pins pins;
	status = BEID_GetPINs(&pins);
	CHECK_EQUAL(BEID_OK,status.general);
	CHECK_EQUAL(1,pins.pinsLength);

	// for all the pins...
	for (long pinIdx=0;pinIdx<pins.pinsLength;pinIdx++)
	{
		::BEID_Pin  pin;
		pin.pinType = pins.pins[pinIdx].pinType;
		pin.id		= pins.pins[pinIdx].id;
		pin.usageCode= pins.pins[pinIdx].usageCode;
		long		triesLeft = pins.pins[pinIdx].triesLeft;

		//-------------------------------------
		// read file BEID_FILE_ID and test
		//-------------------------------------
		BEID_Bytes outData;
		outData.length		 = size_tuc3F00DF014031;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_ID, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF014031);
		CHECK_ARRAY_EQUAL(tuc3F00DF014031,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_ID_SIGN and test
		//-------------------------------------

		outData.length		 = size_tuc3F00DF014032;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_ID_SIGN, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF014032);
		CHECK_ARRAY_EQUAL(tuc3F00DF014032,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_ADDRESS and test
		//-------------------------------------

		outData.length		 = size_tuc3F00DF014033;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_ADDRESS, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF014033);
		CHECK_ARRAY_EQUAL(tuc3F00DF014033,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_ADDRESS_SIGN and test
		//-------------------------------------

		outData.length		 = size_tuc3F00DF014034;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_ADDRESS_SIGN, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF014034);
		CHECK_ARRAY_EQUAL(tuc3F00DF014034,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_PHOTO and test
		//-------------------------------------

		outData.length		 = size_tuc3F00DF014035;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_PHOTO, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF014035);
		CHECK_ARRAY_EQUAL(tuc3F00DF014035,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_TOKENINFO and test
		//-------------------------------------

		outData.length		 = size_tuc3F00DF005032;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_TOKENINFO, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF005032);
		CHECK_ARRAY_EQUAL(tuc3F00DF005032,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_RRN and test
		//-------------------------------------

		outData.length		 = size_tuc3F00DF00503C;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_RRN, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF00503C);
		CHECK_ARRAY_EQUAL(tuc3F00DF00503C,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;
	}
	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);
}

TEST(BEIDReadFile_t2v1)
{
	//----------------------------------------------
	// data file that can be read
	// test card 1 version 1
	//----------------------------------------------
	unsigned char tuc3F00DF014031[] = {
		0x01,0x0c,0x35,0x39,0x30,0x34,0x34,0x35,0x30,0x37,0x34,0x37,0x32,0x35,0x02,0x10,
		0x53,0x4c,0x49,0x4e,0x33,0x66,0x00,0x29,0x6c,0xff,0x26,0x23,0x66,0x0b,0x08,0x26,
		0x03,0x0a,0x32,0x34,0x2e,0x31,0x30,0x2e,0x32,0x30,0x30,0x36,0x04,0x0a,0x32,0x34,
		0x2e,0x31,0x30,0x2e,0x32,0x30,0x31,0x31,0x05,0x08,0x48,0x6f,0x6c,0x73,0x62,0x65,
		0x65,0x6b,0x06,0x0b,0x37,0x33,0x30,0x34,0x30,0x31,0x30,0x32,0x37,0x34,0x39,0x07,
		0x06,0x48,0x6f,0x65,0x62,0x65,0x6e,0x08,0x06,0x53,0x74,0x65,0x66,0x61,0x6e,0x09,
		0x00,0x0a,0x04,0x42,0x65,0x6c,0x67,0x0b,0x04,0x42,0x72,0x65,0x65,0x0c,0x0c,0x30,
		0x31,0x20,0x41,0x50,0x52,0x20,0x20,0x31,0x39,0x37,0x33,0x0d,0x01,0x4d,0x0e,0x00,
		0x0f,0x01,0x31,0x10,0x01,0x30,0x11,0x14,0x03,0xcc,0x6c,0x50,0x84,0x85,0x82,0x54,
		0xcc,0x72,0x3c,0xb4,0xe0,0x8e,0x45,0xa8,0x85,0x33,0x68,0x7a,
	};
	static long size_tuc3F00DF014031 = sizeof(tuc3F00DF014031);

	unsigned char tuc3F00DF014032[] = {
		0x3d,0x88,0x5c,0x02,0x01,0x24,0x2e,0xd0,0x6d,0x1a,0xe9,0x0c,0xf1,0xde,0x7e,0x75,
		0xe0,0x1e,0x6b,0x12,0x9c,0xa5,0x35,0x53,0xf4,0x79,0x4f,0xb3,0xea,0xc3,0x58,0x2a,
		0x67,0xc7,0xaa,0xf0,0x9d,0x75,0x68,0x75,0x45,0x0c,0xc1,0x59,0x1b,0x5b,0x20,0x7e,
		0xa2,0x6a,0x55,0x15,0xd5,0x40,0x25,0xf3,0x91,0x98,0x48,0x94,0xf8,0x51,0x4e,0xfb,
		0xdb,0x07,0x69,0xc5,0x86,0xd1,0x92,0x29,0xb9,0xf0,0x46,0x00,0xdc,0xad,0x28,0x1c,
		0x67,0x1a,0xb5,0x5e,0xa0,0xbb,0x0e,0xeb,0x7e,0xce,0x2c,0x57,0xbb,0xcf,0xc8,0x7b,
		0xb3,0xd2,0x19,0x6a,0x36,0x3b,0xb0,0xd7,0xa3,0x3f,0x52,0x46,0xd5,0x9d,0x85,0xda,
		0x9d,0xff,0x11,0x9a,0xa7,0x8e,0x25,0xa2,0x0d,0xa9,0x21,0x5a,0xee,0x96,0x93,0x65,
	};
	static long size_tuc3F00DF014032 = sizeof(tuc3F00DF014032);

	unsigned char tuc3F00DF014033[] = {
		0x01,0x0d,0x5a,0x61,0x6e,0x64,0x73,0x74,0x72,0x61,0x61,0x74,0x20,0x35,0x33,0x02,
		0x04,0x33,0x32,0x32,0x31,0x03,0x08,0x48,0x6f,0x6c,0x73,0x62,0x65,0x65,0x6b,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,
	};
	static long size_tuc3F00DF014033 = sizeof(tuc3F00DF014033);

	unsigned char tuc3F00DF014034[] = {
		0x66,0x75,0xfe,0xf7,0xce,0x5b,0x09,0x39,0xed,0xfd,0x15,0x21,0x09,0x29,0x70,0xef,
		0x6c,0xb3,0x85,0x79,0x4a,0xf7,0xce,0x6d,0x54,0x6a,0x41,0xea,0x0c,0x74,0x70,0xd1,
		0xcf,0xda,0x8e,0x38,0xcf,0xea,0x96,0xb1,0x8e,0x14,0xf9,0xba,0xf6,0x32,0x75,0x75,
		0xc3,0x23,0x58,0xf2,0x6a,0x1a,0x1a,0x72,0x7e,0x60,0x7e,0x43,0x99,0xb6,0x74,0x41,
		0xe2,0x29,0x19,0xbd,0x18,0x22,0xa4,0xd3,0xeb,0x8a,0x5b,0xe4,0xb8,0x0c,0xe2,0xf7,
		0xe1,0xd1,0x4f,0xe2,0x26,0x4d,0xe5,0x2f,0x1b,0x71,0xfc,0x1c,0xac,0x2d,0xff,0x55,
		0xa7,0xd3,0xe6,0x5d,0x6d,0xe8,0x49,0x50,0x5c,0x42,0xd5,0x20,0x35,0x8b,0xae,0x57,
		0xb4,0x37,0x6f,0xbc,0x4c,0x42,0xfb,0x99,0xd1,0xc1,0xd9,0xf3,0x5c,0x89,0x64,0xad,
	};
	static long size_tuc3F00DF014034 = sizeof(tuc3F00DF014034);

	unsigned char tuc3F00DF014035[] = {
		0xff,0xd8,0xff,0xe0,0x00,0x10,0x4a,0x46,0x49,0x46,0x00,0x01,0x02,0x01,0x01,0x2c,
		0x01,0x2c,0x00,0x00,0xff,0xdb,0x00,0x43,0x00,0x0f,0x0a,0x0b,0x0d,0x0b,0x09,0x0f,
		0x0d,0x0c,0x0d,0x11,0x10,0x0f,0x11,0x16,0x25,0x18,0x16,0x14,0x14,0x16,0x2d,0x20,
		0x22,0x1b,0x25,0x35,0x2f,0x38,0x37,0x34,0x2f,0x34,0x33,0x3b,0x42,0x55,0x48,0x3b,
		0x3f,0x50,0x3f,0x33,0x34,0x4a,0x64,0x4b,0x50,0x57,0x5a,0x5f,0x60,0x5f,0x39,0x47,
		0x68,0x6f,0x67,0x5c,0x6e,0x55,0x5d,0x5f,0x5b,0xff,0xc0,0x00,0x0b,0x08,0x00,0xc8,
		0x00,0x8c,0x01,0x01,0x11,0x00,0xff,0xc4,0x00,0xd2,0x00,0x00,0x01,0x05,0x01,0x01,
		0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x03,0x04,
		0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x10,0x00,0x02,0x01,0x03,0x03,0x02,0x04,0x03,
		0x05,0x05,0x04,0x04,0x00,0x00,0x01,0x7d,0x01,0x02,0x03,0x00,0x04,0x11,0x05,0x12,
		0x21,0x31,0x41,0x06,0x13,0x51,0x61,0x07,0x22,0x71,0x14,0x32,0x81,0x91,0xa1,0x08,
		0x23,0x42,0xb1,0xc1,0x15,0x52,0xd1,0xf0,0x24,0x33,0x62,0x72,0x82,0x09,0x0a,0x16,
		0x17,0x18,0x19,0x1a,0x25,0x26,0x27,0x28,0x29,0x2a,0x34,0x35,0x36,0x37,0x38,0x39,
		0x3a,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4a,0x53,0x54,0x55,0x56,0x57,0x58,0x59,
		0x5a,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0x6a,0x73,0x74,0x75,0x76,0x77,0x78,0x79,
		0x7a,0x83,0x84,0x85,0x86,0x87,0x88,0x89,0x8a,0x92,0x93,0x94,0x95,0x96,0x97,0x98,
		0x99,0x9a,0xa2,0xa3,0xa4,0xa5,0xa6,0xa7,0xa8,0xa9,0xaa,0xb2,0xb3,0xb4,0xb5,0xb6,
		0xb7,0xb8,0xb9,0xba,0xc2,0xc3,0xc4,0xc5,0xc6,0xc7,0xc8,0xc9,0xca,0xd2,0xd3,0xd4,
		0xd5,0xd6,0xd7,0xd8,0xd9,0xda,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,0xe9,0xea,
		0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,0xff,0xda,0x00,0x08,0x01,0x01,
		0x00,0x00,0x3f,0x00,0xef,0xb7,0x1f,0x53,0x40,0x27,0x1d,0x4d,0x21,0x63,0xea,0x68,
		0xdc,0x7d,0x4d,0x1b,0x8e,0x3a,0x9f,0xce,0x97,0x27,0xd4,0xd1,0x96,0x1d,0xcd,0x1b,
		0x8f,0xf7,0x8f,0xe7,0x4d,0x33,0x28,0x20,0x17,0xe4,0xf4,0xe7,0xad,0x29,0x93,0x03,
		0x3b,0xbf,0x5a,0x6a,0x4e,0xaf,0xc2,0x3e,0x48,0xeb,0xcd,0x3b,0x71,0xf5,0x34,0x9b,
		0x9b,0xd4,0xfe,0x74,0x6e,0x6f,0xef,0x1f,0xce,0x93,0x73,0x7f,0x78,0xfe,0x74,0xbb,
		0x9b,0xfb,0xc7,0xf3,0xa3,0x73,0x7a,0x9f,0xce,0x8d,0xcd,0xea,0x7f,0x3a,0x5c,0x52,
		0xd2,0x1c,0x51,0x49,0x50,0x5e,0x5e,0xc1,0x65,0x03,0x4d,0x71,0x22,0xa2,0x0e,0xe4,
		0xf5,0xae,0x43,0x53,0xf1,0xb3,0x07,0x2b,0x64,0x17,0x68,0xfe,0x22,0xbc,0xd6,0x63,
		0xf8,0xc2,0x69,0xa1,0x6f,0x38,0xb3,0x48,0x3e,0xee,0x0e,0xd5,0x3f,0x50,0x2b,0x22,
		0x2d,0x4d,0x9a,0xe9,0x66,0x9a,0x67,0xce,0x79,0x24,0x9e,0x05,0x5e,0xff,0x00,0x84,
		0xa2,0xee,0x30,0xf1,0x45,0x71,0x21,0x4c,0xf0,0x5b,0xe6,0xc0,0xfc,0x6a,0xb4,0x7a,
		0xe5,0xdc,0x13,0x89,0xa1,0xb8,0x90,0x3f,0x76,0xdd,0x9c,0xfd,0x6b,0x66,0xc3,0xc6,
		0xf7,0xb1,0x95,0xfb,0x4a,0x24,0xeb,0xf4,0xda,0x7f,0x4a,0xea,0xec,0x7c,0x43,0x65,
		0x79,0x66,0x6e,0x43,0x98,0xd5,0x78,0x60,0xdd,0x45,0x69,0xc7,0x22,0x4a,0x81,0xd1,
		0x83,0x2b,0x0c,0x82,0x3a,0x53,0xbb,0xd2,0xd1,0x45,0x3e,0x90,0xd1,0x4c,0x92,0x44,
		0x89,0x0b,0xc8,0xc1,0x54,0x72,0x49,0x38,0x02,0xb9,0x6d,0x73,0xc6,0x50,0x5a,0x86,
		0x8f,0x4f,0x2b,0x34,0x9f,0xdf,0x3f,0x74,0x7f,0x8d,0x70,0xfa,0x86,0xb3,0x79,0xa8,
		0x49,0xbe,0xe6,0x56,0x90,0xe3,0x8c,0xf1,0x8f,0xa0,0xaa,0x04,0x92,0x73,0xed,0xd6,
		0x98,0x84,0x9c,0x82,0x08,0xfa,0x8a,0x52,0x0e,0xdf,0x9b,0x39,0xc7,0x6a,0x01,0xc1,
		0xc1,0xa5,0xc7,0x39,0xa7,0x27,0xde,0x1c,0xfe,0x35,0x37,0x99,0x22,0x70,0x84,0xf2,
		0x7a,0x7a,0xd7,0x55,0xe1,0xcf,0x13,0xfd,0x8a,0x18,0xed,0xee,0x81,0x68,0x87,0x1b,
		0x81,0xc9,0x1c,0xd7,0x77,0x0c,0xa9,0x34,0x6b,0x24,0x6c,0x19,0x18,0x64,0x11,0xdc,
		0x54,0x94,0xb4,0x53,0xe9,0x0d,0x35,0x98,0x2a,0x92,0x48,0x00,0x57,0x0d,0xe2,0xcf,
		0x11,0x8b,0x82,0xd6,0x76,0x8d,0xfb,0xb5,0xce,0xe7,0xcf,0xde,0xfa,0x57,0x10,0xec,
		0x4b,0x1f,0x9b,0x23,0xb6,0x6a,0x12,0xc7,0xa8,0xe0,0x91,0xf9,0xd3,0xfe,0xf0,0xf6,
		0xef,0x49,0xef,0x9c,0x67,0x81,0x4e,0xed,0x9e,0xde,0xd4,0x80,0xe4,0xe7,0x1f,0x86,
		0x29,0x73,0xf3,0x01,0xc9,0xfa,0x53,0xd4,0xed,0x56,0x1d,0x0d,0x2a,0xc8,0x3a,0xb0,
		0xfd,0x29,0xe0,0xb0,0x60,0x41,0xf9,0x4d,0x77,0x9e,0x0e,0xd7,0x61,0x92,0x04,0xb0,
		0x95,0xb6,0xc8,0xbc,0x26,0x7f,0x88,0x7a,0x57,0x5a,0x29,0x45,0x2d,0x3e,0x83,0xd2,
		0xb9,0xdf,0x17,0xea,0xa2,0xc2,0xc7,0xca,0x5e,0x64,0x94,0x10,0x3d,0x87,0x15,0xe6,
		0x13,0xca,0x64,0x6c,0xf3,0x51,0x06,0xcf,0xe1,0x48,0xab,0xf7,0x48,0xcf,0x4a,0x7e,
		0xd2,0x07,0x20,0xfb,0x71,0x46,0xd3,0xd7,0x69,0xfc,0x05,0x4d,0x15,0xb3,0xbf,0x62,
		0x2a,0xd2,0xe9,0x72,0x75,0xd8,0xde,0xb4,0xf8,0xb4,0x79,0x09,0x04,0xa9,0xab,0xd1,
		0xe8,0x99,0x18,0x7c,0xe3,0xda,0xa9,0x5f,0x68,0xb2,0x43,0x97,0x8c,0x31,0x51,0xc9,
		0xac,0xe4,0xc2,0xfc,0xaf,0x91,0x83,0xc8,0x3d,0x2a,0xdd,0xb4,0xa6,0x0b,0x88,0xe4,
		0x8c,0x72,0x8c,0x18,0x57,0xaa,0x68,0xf7,0xdf,0xda,0x1a,0x7c,0x77,0x18,0x0a,0x5b,
		0xa8,0x07,0x38,0x35,0x7e,0x83,0x4f,0x34,0x86,0xbc,0xe3,0xc7,0x77,0x7e,0x6e,0xaa,
		0x63,0xce,0x44,0x48,0x17,0x19,0xee,0x79,0xff,0x00,0x0a,0xe4,0x58,0x74,0x1e,0xbd,
		0xea,0x6b,0x3b,0x63,0x2b,0xe4,0xfd,0xd1,0x5a,0x49,0x6a,0xbd,0x00,0xab,0x09,0x64,
		0x8f,0x8c,0x8a,0xb3,0x1d,0x8c,0x4b,0xc6,0xd1,0xcd,0x5d,0x86,0xd2,0x15,0x1c,0x20,
		0xab,0xab,0x12,0x6d,0xe1,0x45,0x3b,0xcb,0x52,0x06,0x45,0x28,0x40,0x3a,0x52,0x94,
		0x5c,0x1c,0x8e,0xbe,0xd5,0xcb,0xf8,0x8b,0x4e,0x11,0x11,0x3c,0x6b,0x85,0x6e,0x08,
		0xf4,0x35,0x8e,0x8d,0xb4,0x6d,0x39,0x1c,0xf7,0xae,0xdb,0xc0,0x97,0xa5,0x7c,0xcb,
		0x3c,0x65,0x4b,0x6f,0x07,0xd3,0x80,0x3f,0xa5,0x76,0xa0,0xd2,0xe6,0x9e,0x45,0x43,
		0x77,0x3a,0x5b,0x5b,0x49,0x34,0x87,0x0a,0x8a,0x58,0xd7,0x8f,0xea,0xf7,0x4f,0x7f,
		0x7d,0x2d,0xc3,0x8e,0x59,0x8b,0x7d,0x2a,0x8e,0x32,0x45,0x6d,0xda,0x5b,0x88,0x2d,
		0x97,0xfb,0xc7,0xad,0x4a,0x0f,0x39,0xab,0x90,0x01,0xe9,0x56,0xd1,0x01,0xc7,0x15,
		0x32,0x29,0xcf,0xd6,0xac,0x01,0x81,0x40,0xcd,0x4b,0x1a,0x9c,0x50,0xca,0x73,0xd2,
		0xaa,0xde,0xdb,0x8b,0x8b,0x77,0x8d,0x86,0x43,0x0c,0x57,0x13,0x2c,0x5e,0x59,0x78,
		0xd8,0x1f,0x94,0x91,0x5b,0x7e,0x11,0x9f,0xca,0xd6,0xe1,0xeb,0x87,0xca,0xf1,0xf4,
		0xaf,0x4b,0x5e,0x94,0xb5,0x2d,0x64,0x78,0xa0,0xb8,0xd0,0xae,0x76,0x0c,0x9d,0xbd,
		0xbe,0xb5,0xe4,0x93,0xb9,0x32,0x1e,0xb4,0xfb,0x38,0xfc,0xc9,0xd7,0x3e,0xb5,0xbd,
		0x20,0xc4,0x6b,0x51,0x01,0xf3,0x55,0xc8,0x3a,0x8a,0xbc,0x8c,0x05,0x4e,0x87,0x8e,
		0x95,0x20,0xa9,0x11,0x38,0xe6,0xad,0x46,0xa0,0x20,0xe2,0x99,0x22,0xe4,0xe4,0x54,
		0x12,0x8c,0x57,0x27,0xaf,0x40,0x12,0xe9,0x9c,0x0e,0x1f,0x9a,0xbb,0xe0,0xcb,0x65,
		0x9f,0x54,0x0d,0x92,0x0c,0x5f,0x30,0xc5,0x7a,0x3a,0xf4,0x14,0xa6,0xa4,0x35,0x5e,
		0xf6,0x05,0xb8,0xb3,0x96,0x17,0xfb,0xae,0x85,0x4f,0xe3,0x5e,0x33,0x79,0x16,0xcb,
		0x86,0x5c,0xfd,0xd2,0x47,0x27,0xd2,0xac,0xe9,0x89,0x99,0x46,0x79,0xad,0x67,0xfe,
		0x55,0x08,0x38,0x3c,0x55,0x98,0x4e,0x0e,0x4f,0x5a,0xb6,0x8f,0x93,0xc1,0xab,0x50,
		0xb5,0x4e,0x32,0x79,0xa9,0xd0,0x00,0xa3,0x26,0xa7,0x0c,0x36,0x8e,0x45,0x34,0xbf,
		0x18,0xa8,0x66,0xae,0x6b,0x5e,0x4c,0x9c,0x9f,0xc2,0xac,0xf8,0x11,0x5b,0xfb,0x56,
		0x5e,0x3e,0x51,0x1f,0x27,0xdf,0x35,0xe8,0x42,0x97,0xf0,0xa9,0x31,0x4d,0x71,0x95,
		0x22,0xbc,0x77,0x56,0xb6,0x31,0xea,0x33,0xc4,0xeb,0x82,0x92,0x15,0xc7,0xe3,0x52,
		0x69,0xaa,0x03,0xf4,0xe6,0xb4,0x25,0x1c,0x1c,0x55,0x60,0xd9,0x38,0xc5,0x4f,0x19,
		0x27,0x8e,0x6a,0xe4,0x23,0x91,0x57,0x61,0x4e,0x2a,0xc2,0x8e,0xd5,0x2c,0x63,0x24,
		0x62,0xac,0xf9,0x5c,0x53,0x19,0x48,0x15,0x13,0xf3,0xda,0xb0,0x75,0xe8,0xb3,0x16,
		0xf1,0xd0,0x75,0xab,0x5e,0x03,0x1f,0xe9,0x37,0x27,0x1d,0x94,0x7f,0x3a,0xee,0x96,
		0x9d,0x8a,0x92,0x90,0xf4,0xae,0x07,0xc7,0xb6,0x2b,0x0d,0xec,0x57,0x51,0xae,0x3c,
		0xe0,0x43,0x9f,0x71,0x8f,0xe9,0xfc,0xab,0x03,0x4d,0x19,0x63,0xed,0x57,0x66,0xe1,
		0x32,0x6a,0xbe,0x40,0x1c,0xd2,0xad,0xc2,0x8c,0x62,0xaf,0x5b,0x4c,0xad,0x5a,0x30,
		0x38,0x3f,0x95,0x58,0x89,0xb3,0xd7,0xbd,0x3a,0x39,0x02,0x0c,0x93,0xd2,0x89,0x75,
		0x58,0x61,0xe1,0xe4,0x51,0x55,0x1b,0x5d,0x80,0x92,0x00,0x63,0xee,0x2a,0x44,0xd4,
		0xa1,0x94,0x60,0xe5,0x73,0xdc,0xd5,0x7d,0x4a,0x35,0x96,0xce,0x50,0x39,0x18,0xaa,
		0xfe,0x0f,0x96,0x5b,0x79,0x27,0x75,0x00,0xa1,0x20,0x1c,0xd7,0xa0,0x2f,0x23,0x34,
		0xec,0x54,0x94,0xd6,0x38,0x04,0xfa,0x57,0x9f,0xf8,0x87,0x52,0x9a,0xfd,0x19,0x1b,
		0x63,0x44,0xa4,0x95,0xda,0x39,0x15,0x8b,0xa6,0x8c,0x33,0xd4,0xd7,0x72,0xec,0x4c,
		0x01,0x96,0xac,0xc1,0xe6,0x39,0x07,0x9f,0xc2,0xac,0x08,0x26,0xd9,0xbf,0x63,0x74,
		0xa8,0x16,0xe6,0x68,0x9b,0x90,0x54,0xe7,0xb5,0x69,0x5a,0x6a,0x7c,0x61,0xf8,0x35,
		0xb3,0x63,0x2f,0xda,0x53,0xe5,0xea,0x47,0x14,0xcb,0x85,0x96,0x33,0xc8,0x38,0xf5,
		0xac,0x1b,0xa5,0x8c,0xca,0xde,0x6d,0xc0,0x5e,0x79,0xef,0x53,0xdb,0x43,0x60,0x40,
		0x26,0xe8,0xe0,0xf7,0xce,0x2b,0x5a,0x2b,0x4b,0x79,0x50,0x18,0xa5,0xdc,0x31,0xc7,
		0x39,0xab,0x09,0x01,0x5b,0x73,0x1e,0x73,0xc6,0x39,0xa8,0xf4,0x88,0x19,0x2c,0x94,
		0x01,0x8c,0xcb,0xd2,0xbb,0x78,0xc7,0xca,0x29,0xe6,0xa4,0x34,0xd6,0x19,0x04,0x57,
		0x9d,0x6a,0x16,0x8d,0x65,0x79,0x3c,0x2c,0x38,0x56,0xe3,0x3d,0xc7,0x6a,0xa3,0x6d,
		0x0f,0x96,0xae,0xff,0x00,0xde,0x3c,0x55,0x79,0xce,0x5b,0x9a,0xaa,0xf3,0x49,0xbb,
		0x10,0x8c,0x7b,0xe2,0x89,0x92,0xee,0x52,0x82,0x39,0x1c,0x8e,0xe3,0x9a,0x8b,0x50,
		0xb6,0xf2,0x82,0xb0,0x0d,0x90,0x3e,0x62,0xde,0xb4,0xdb,0x30,0xe4,0x8c,0xf3,0x9a,
		0xeb,0xf4,0x80,0x23,0x55,0xc5,0x68,0xde,0x45,0xbd,0x36,0x8e,0xe2,0xb9,0xd9,0xb4,
		0xa6,0x69,0x18,0x6c,0x1f,0x37,0xb5,0x5b,0xd3,0x74,0x6f,0xb3,0x02,0x40,0x4c,0xbf,
		0x04,0x91,0xc8,0x1e,0xd5,0xa1,0x0e,0x8f,0x0c,0x27,0x74,0x43,0x61,0xf6,0xe2,0xa7,
		0x75,0x0a,0x98,0x1d,0xaa,0x7d,0x38,0x89,0x24,0x54,0xc0,0xe1,0x85,0x74,0x6a,0x38,
		0xa5,0xa9,0x29,0x2b,0x95,0xf1,0xad,0xb1,0x10,0x25,0xd2,0x0e,0x9f,0x2b,0x1f,0xe5,
		0x5c,0xb5,0xa2,0xb0,0xb6,0xc9,0xe7,0x24,0x9a,0x86,0x54,0x25,0xaa,0x5b,0x7b,0x75,
		0x3d,0x45,0x5d,0xd9,0xe5,0xa7,0x1c,0x56,0x56,0xa0,0x9b,0x8e,0xe2,0x6a,0x2b,0x64,
		0x0a,0x33,0xc5,0x6f,0x69,0x6c,0x4e,0x0d,0x6c,0x06,0x27,0x19,0xa7,0xaa,0x2b,0x8e,
		0x45,0x38,0x7c,0x84,0x00,0x72,0x2a,0x6d,0xc0,0xa9,0xe9,0x54,0xe6,0x04,0x66,0xa5,
		0xd2,0x57,0x17,0x41,0xbd,0x5a,0xba,0x31,0xd2,0x96,0xa4,0xa2,0xa8,0x6b,0x36,0xa2,
		0xef,0x4d,0x9a,0x22,0x32,0x4a,0xe4,0x7d,0x47,0x35,0xc2,0x34,0x5e,0x48,0x28,0x01,
		0xf9,0x6a,0x9c,0xa7,0x9a,0x92,0x27,0xc7,0x41,0x56,0x0b,0x65,0x79,0x15,0x99,0xa8,
		0x49,0x8e,0x05,0x45,0x6c,0x0b,0x46,0x0e,0x38,0x35,0xbb,0xa5,0x83,0xc7,0x6a,0xda,
		0xda,0xc0,0x00,0x47,0x38,0xa6,0x24,0x85,0x58,0xa9,0xa9,0x43,0x67,0x1c,0x54,0x84,
		0x9c,0x55,0x79,0x8e,0x41,0xab,0x3a,0x52,0xfe,0xf6,0x3f,0xa9,0xad,0xf1,0xd2,0x96,
		0xa4,0xa2,0x9a,0xe3,0x2b,0x5c,0x1e,0xa7,0x6e,0xf6,0xf7,0xb3,0x46,0xe3,0xa1,0xc8,
		0x3e,0xa2,0xb2,0x26,0xe0,0xe2,0x92,0x16,0x19,0xe4,0x54,0xcf,0x26,0x13,0xde,0xb1,
		0xae,0xdc,0xb4,0x84,0x9e,0xb4,0xf8,0x2e,0x56,0x28,0xb6,0xbf,0x51,0xd3,0x15,0xab,
		0xa6,0x5f,0x8c,0x81,0xd3,0xd8,0xd6,0xc4,0x77,0xce,0xf2,0xaa,0x24,0x4c,0xcb,0x8e,
		0x5b,0x3d,0x2a,0xcc,0x88,0x4e,0x1d,0x7a,0x8e,0xa2,0x9f,0x1b,0x03,0x8c,0xd4,0xdb,
		0xb2,0x3d,0xaa,0x07,0xe8,0x7d,0xeb,0x67,0x4d,0xb2,0x68,0x40,0x79,0x46,0x1b,0xa0,
		0x1e,0x95,0xa3,0x45,0x49,0x45,0x21,0xe9,0x59,0xfa,0xd5,0x94,0x77,0x36,0x32,0x12,
		0x80,0xc8,0x8a,0x4a,0xb6,0x39,0xe3,0x9a,0xf3,0xcb,0xd4,0xfd,0xe1,0xf4,0xcd,0x45,
		0x08,0xc3,0x63,0x35,0x63,0x03,0x93,0xda,0xb3,0xae,0x10,0x34,0x99,0xc5,0x42,0xd0,
		0x16,0xfb,0xa3,0x35,0xa0,0x60,0xf2,0xad,0xad,0x9b,0x67,0xcd,0xbf,0x9a,0xdf,0x80,
		0x1f,0x2c,0x3e,0x31,0x9a,0xb0,0x92,0x76,0xcf,0x34,0xe2,0x0e,0xef,0x6a,0x7a,0xb7,
		0x6a,0x58,0xb0,0xf7,0x31,0x2f,0x62,0xe3,0xf9,0xd7,0x4c,0x29,0x68,0xa9,0x28,0xa2,
		0x91,0x86,0x54,0x83,0x5e,0x71,0xac,0xc1,0xe4,0x5d,0x48,0x98,0xfb,0xac,0x46,0x6b,
		0x35,0x0e,0xd6,0xfa,0x55,0x98,0x81,0x94,0x63,0xa5,0x50,0xbf,0x63,0x0c,0x81,0x23,
		0x5c,0x9c,0x75,0x27,0x8a,0x2d,0x3c,0xd7,0x61,0xf3,0x28,0x1e,0xc3,0x35,0xad,0x14,
		0x72,0x91,0x83,0x20,0x23,0xd0,0x8a,0xbb,0x6f,0x14,0xac,0xbc,0x4d,0x8f,0x6c,0x54,
		0xa6,0xd6,0x70,0xa4,0xf9,0xab,0x9e,0xdc,0x53,0x2d,0xee,0x25,0x2d,0xe5,0xcc,0x9b,
		0x58,0x1c,0x67,0xb5,0x5c,0x1c,0x55,0xad,0x2d,0x0c,0xba,0x82,0x7a,0x20,0xdc,0x6b,
		0xa1,0xc5,0x18,0xa5,0xc5,0x3e,0x8a,0x28,0xae,0x17,0xc4,0x80,0x7f,0x6a,0x4e,0x98,
		0xf4,0x23,0xf2,0xae,0x6e,0x53,0xb5,0xbd,0xe9,0xf0,0xdc,0x79,0x6a,0x79,0xa8,0x1e,
		0x60,0xec,0x4b,0x0a,0xb5,0x14,0x71,0x85,0xdd,0xd2,0x9f,0x1c,0x25,0xe4,0x3e,0x59,
		0x20,0x7a,0xe6,0xb4,0xec,0x2d,0xda,0x2e,0xa4,0x90,0x4d,0x69,0x6f,0x00,0x73,0x50,
		0x4a,0xeb,0xb8,0x71,0x4f,0x57,0x3b,0x79,0xad,0xfd,0x12,0xdc,0xc7,0x6f,0xe6,0xb8,
		0xc3,0x4b,0xc8,0xfa,0x76,0xad,0x2a,0x29,0x4d,0x3a,0x8a,0x28,0xae,0x13,0xc5,0x87,
		0x6e,0xad,0x2f,0xd0,0x7f,0x21,0x5c,0xe4,0xc3,0x77,0x3d,0xea,0xb4,0x84,0x80,0x00,
		0xe2,0xa3,0x1c,0x74,0xab,0x91,0xc8,0x4a,0x81,0xd8,0xd6,0x9d,0x9a,0xf1,0x5a,0xd1,
		0xa9,0x28,0x0f,0xa5,0x2b,0x70,0x2a,0xbc,0xd2,0x63,0xfc,0x6a,0xe6,0x93,0x00,0xbd,
		0xbc,0x58,0xcf,0xdc,0x03,0x73,0x7b,0x8f,0x4a,0xeb,0x40,0x00,0x01,0x4b,0x4b,0x8a,
		0x52,0x29,0x68,0xa2,0xa8,0xdf,0xeb,0x16,0x36,0x1c,0x4f,0x3a,0x87,0xec,0x8b,0xcb,
		0x7e,0x55,0xc3,0xeb,0xf7,0xa9,0x7d,0x7c,0xf3,0xc6,0x0a,0xab,0x63,0x01,0xba,0xf1,
		0x58,0x8e,0xc7,0xad,0x46,0xdf,0x31,0xeb,0x8a,0x61,0x4e,0xc3,0xb5,0x3e,0x16,0x28,
		0x70,0xdd,0xab,0x5e,0xde,0x55,0x2a,0x0a,0xd5,0xd8,0xef,0x55,0x17,0x19,0xc7,0xd6,
		0xa3,0x6b,0xe5,0x3c,0x2b,0x64,0xf6,0x02,0x9d,0x14,0x0f,0x33,0x6e,0x90,0x91,0xed,
		0x5d,0x17,0x87,0xa2,0x0b,0x3c,0x8c,0x06,0x36,0xa8,0x1f,0x9f,0xff,0x00,0xaa,0xb7,
		0xf1,0x46,0x3f,0x9d,0x2d,0x2d,0x14,0xc9,0x65,0x8e,0x18,0xcb,0xca,0xe1,0x54,0x75,
		0x24,0xd7,0x27,0xe2,0x5f,0x14,0x14,0x8c,0xc1,0x60,0xc4,0x6e,0x1c,0xc9,0xd0,0xfe,
		0x15,0xc8,0x09,0x18,0xb6,0xf7,0x25,0x9d,0xba,0x92,0x6a,0x47,0x62,0xc3,0xf0,0xaa,
		0xce,0x0d,0x46,0x43,0x02,0x09,0xa5,0x0d,0x83,0xeb,0x4b,0x8d,0xdd,0x6a,0x68,0x97,
		0x8e,0x37,0x7b,0x55,0x88,0xe0,0x2e,0xc3,0x20,0xe3,0xeb,0x5a,0x56,0xd6,0xea,0x84,
		0x10,0x2a,0xfc,0x49,0xc5,0x58,0xb5,0xd5,0x93,0x4e,0xbd,0x10,0x90,0x09,0x98,0x64,
		0x03,0xed,0x5d,0x05,0xb6,0xa7,0x04,0xf8,0xc9,0xd8,0x7d,0x1a,0xad,0x87,0x56,0xfb,
		0xa4,0x1f,0xa5,0x28,0x34,0xb5,0x85,0x7d,0xe2,0x35,0x50,0x56,0xcd,0x37,0x7f,0xb6,
		0xdd,0x3f,0x01,0x5c,0xe6,0xa1,0x7b,0x3d,0xd7,0x32,0x3b,0x31,0x1e,0xa6,0xb1,0xef,
		0x54,0xc8,0xa1,0xfa,0xe3,0xad,0x54,0x1d,0x45,0x4a,0x32,0x47,0xad,0x28,0x8c,0xb0,
		0xe4,0x54,0x9f,0x66,0x27,0xb5,0x35,0xad,0x18,0x1e,0x99,0xa7,0x25,0xb1,0xc8,0xc8,
		0xab,0x91,0x41,0xc7,0xdd,0xab,0x31,0xc4,0x46,0x30,0x2a,0xe4,0x30,0xb1,0x23,0x3d,
		0x2a,0xd8,0x8f,0x0b,0x59,0x1a,0xcc,0x67,0x02,0x74,0x1f,0x3c,0x3c,0xfd,0x47,0x7a,
		0xd5,0xb7,0x93,0x7c,0x28,0xeb,0xfc,0x40,0x11,0x53,0xa4,0xa5,0x94,0x8c,0x91,0xcf,
		0x23,0x35,0xa1,0x6f,0xaa,0xb2,0x00,0x25,0x5d,0xc0,0x77,0xcf,0x35,0x75,0x75,0x1b,
		0x72,0xb9,0xdc,0x47,0xb1,0x15,0xca,0x3a,0x0e,0x40,0xe4,0x8f,0x4a,0xab,0x02,0x79,
		0x83,0x9e,0xf5,0x11,0xb6,0x01,0xd9,0x18,0x7c,0xad,0x54,0xa7,0xb0,0xdb,0xd0,0x73,
		0x55,0x52,0x22,0x1b,0x04,0x74,0xab,0x70,0xa5,0x5b,0x11,0xf1,0xd2,0x9f,0xe4,0x64,
		0x76,0xa7,0xac,0x38,0xea,0x2a,0xc4,0x51,0xae,0x2a,0xca,0x44,0xbe,0x95,0x3a,0xa7,
		0x1d,0x30,0x28,0x71,0xc1,0xaa,0x57,0x11,0x79,0x8a,0xea,0x47,0xde,0x52,0x29,0x34,
		0xd3,0xbb,0x4d,0x84,0xfa,0x0c,0x7f,0x4a,0xb0,0x4e,0xd9,0x33,0xd9,0xa9,0x09,0x23,
		0xe9,0x4f,0x59,0x06,0x28,0x8e,0xce,0xeb,0x39,0x36,0xd2,0x8c,0xff,0x00,0xb0,0x6a,
		0xbd,0xbd,0x85,0xdc,0x73,0x3a,0xfd,0x96,0x6d,0xb9,0xe0,0xf9,0x67,0x15,0x3c,0xba,
		0x7d,0xcb,0x0c,0x8b,0x69,0x73,0xfe,0xe1,0xa8,0xdf,0x4d,0xb9,0x74,0xff,0x00,0x8f,
		0x69,0x41,0xff,0x00,0xae,0x66,0xa8,0x49,0xa3,0xde,0x6f,0xc8,0xb4,0x9c,0xff,0x00,
		0xdb,0x33,0xfe,0x14,0xab,0xa5,0xde,0x81,0xff,0x00,0x1e,0x93,0xff,0x00,0xdf,0xb3,
		0xfe,0x15,0x3c,0x7a,0x7d,0xe6,0x39,0xb4,0x9c,0x7f,0xdb,0x33,0xfe,0x15,0x38,0xb0,
		0xba,0xc7,0xfc,0x7a,0xcd,0x9f,0xfa,0xe6,0x69,0xe2,0xc6,0xe7,0xbd,0xb4,0xdf,0xf7,
		0xc1,0xa9,0xa3,0xb2,0x9c,0x75,0xb6,0x93,0xfe,0xf8,0x35,0x38,0xb5,0x98,0x63,0xf7,
		0x12,0x7f,0xdf,0x06,0x94,0xdb,0x5c,0x7f,0xcf,0x19,0x3f,0xef,0x93,0x4d,0x6b,0x6b,
		0x83,0xff,0x00,0x2c,0x24,0xff,0x00,0xbe,0x0d,0x31,0xec,0xa7,0x11,0xb1,0x10,0x48,
		0x4e,0x0f,0x1b,0x0d,0x51,0xd2,0xec,0x6f,0x23,0xd3,0x15,0x1e,0xd6,0x65,0x61,0x9e,
		0x0c,0x64,0x1e,0xbf,0x4a,0xb6,0xd6,0x77,0x25,0x07,0xee,0x25,0xcf,0xfb,0x86,0x81,
		0x69,0x72,0xcb,0xcd,0xbc,0xbf,0xf7,0xc1,0xa8,0x8d,0x95,0xd0,0x24,0x7d,0x9a,0x53,
		0xff,0x00,0x00,0x35,0xff,0xd9,
	};
	static long size_tuc3F00DF014035 = sizeof(tuc3F00DF014035);

	unsigned char tuc3F00DF005032[] = {
		0x30,0x27,0x02,0x01,0x00,0x04,0x10,0x53,0x4c,0x49,0x4e,0x33,0x66,0x00,0x29,0x6c,
		0xff,0x26,0x23,0x66,0x0b,0x08,0x26,0x80,0x06,0x42,0x45,0x4c,0x50,0x49,0x43,0x03,
		0x02,0x04,0x30,0x9e,0x04,0x04,0x02,0x00,0x00,
	};
	static long size_tuc3F00DF005032 = sizeof(tuc3F00DF005032);

	unsigned char tuc3F00DF00503C[] = {
		0x30,0x82,0x03,0x2d,0x30,0x82,0x02,0x15,0xa0,0x03,0x02,0x01,0x02,0x02,0x0b,0x04,
		0x00,0x00,0x00,0x00,0x01,0x07,0xe6,0x0b,0xfc,0xdc,0x30,0x0d,0x06,0x09,0x2a,0x86,
		0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x30,0x27,0x31,0x0b,0x30,0x09,0x06,
		0x03,0x55,0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x18,0x30,0x16,0x06,0x03,0x55,0x04,
		0x03,0x13,0x0f,0x42,0x65,0x6c,0x67,0x69,0x75,0x6d,0x20,0x52,0x6f,0x6f,0x74,0x20,
		0x43,0x41,0x30,0x1e,0x17,0x0d,0x30,0x35,0x31,0x32,0x30,0x31,0x31,0x31,0x30,0x30,
		0x30,0x30,0x5a,0x17,0x0d,0x31,0x32,0x30,0x38,0x30,0x31,0x31,0x30,0x30,0x30,0x30,
		0x30,0x5a,0x30,0x29,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x42,
		0x45,0x31,0x0c,0x30,0x0a,0x06,0x03,0x55,0x04,0x0a,0x13,0x03,0x52,0x52,0x4e,0x31,
		0x0c,0x30,0x0a,0x06,0x03,0x55,0x04,0x03,0x13,0x03,0x52,0x52,0x4e,0x30,0x81,0x9f,
		0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,
		0x81,0x8d,0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xe5,0xdc,0x8b,0xbb,0x67,0xfc,
		0xf7,0x73,0x96,0x28,0x23,0xb3,0x42,0x13,0xdf,0x9d,0x31,0x28,0x8b,0x7d,0xed,0x60,
		0xd9,0x11,0xba,0x0c,0x43,0x9c,0xcc,0xe0,0xe6,0x51,0x7d,0xb4,0x71,0xd9,0x00,0xea,
		0x1b,0x5f,0x6f,0x2d,0x17,0xd7,0xb2,0x5a,0x58,0x80,0x3c,0x93,0x70,0x9c,0xa9,0x40,
		0x4b,0x97,0xab,0xd2,0x34,0x17,0xee,0x79,0xcb,0x08,0xdd,0xde,0x12,0x50,0x82,0x24,
		0x73,0xd1,0xf9,0xcb,0x2e,0x4a,0x57,0xc4,0x3a,0x52,0x77,0x97,0x73,0xcc,0x37,0x55,
		0xf0,0x8b,0x8e,0x59,0xaa,0xb7,0x2b,0x11,0xa5,0x06,0x07,0x02,0x32,0xfa,0x42,0x40,
		0x67,0xe0,0x85,0xf1,0xf6,0xed,0xbe,0x62,0xf9,0x24,0x09,0x90,0x90,0x90,0xec,0x3a,
		0x55,0x97,0xbe,0x20,0x8f,0x04,0x87,0x93,0x81,0xcf,0x02,0x03,0x01,0x00,0x01,0xa3,
		0x81,0xdb,0x30,0x81,0xd8,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,0xff,0x04,
		0x04,0x03,0x02,0x06,0xc0,0x30,0x43,0x06,0x03,0x55,0x1d,0x20,0x04,0x3c,0x30,0x3a,
		0x30,0x38,0x06,0x06,0x60,0x38,0x01,0x01,0x01,0x04,0x30,0x2e,0x30,0x2c,0x06,0x08,
		0x2b,0x06,0x01,0x05,0x05,0x07,0x02,0x01,0x16,0x20,0x68,0x74,0x74,0x70,0x3a,0x2f,
		0x2f,0x72,0x65,0x70,0x6f,0x73,0x69,0x74,0x6f,0x72,0x79,0x2e,0x65,0x69,0x64,0x2e,
		0x62,0x65,0x6c,0x67,0x69,0x75,0x6d,0x2e,0x62,0x65,0x30,0x1d,0x06,0x03,0x55,0x1d,
		0x0e,0x04,0x16,0x04,0x14,0x4f,0xa9,0x6f,0xbc,0xd1,0xe4,0x3f,0xb8,0xc4,0x0e,0x7a,
		0xc2,0x00,0x14,0xcc,0x85,0xf2,0x93,0xb2,0x41,0x30,0x36,0x06,0x03,0x55,0x1d,0x1f,
		0x04,0x2f,0x30,0x2d,0x30,0x2b,0xa0,0x29,0xa0,0x27,0x86,0x25,0x68,0x74,0x74,0x70,
		0x3a,0x2f,0x2f,0x63,0x72,0x6c,0x2e,0x65,0x69,0x64,0x2e,0x62,0x65,0x6c,0x67,0x69,
		0x75,0x6d,0x2e,0x62,0x65,0x2f,0x62,0x65,0x6c,0x67,0x69,0x75,0x6d,0x2e,0x63,0x72,
		0x6c,0x30,0x09,0x06,0x03,0x55,0x1d,0x13,0x04,0x02,0x30,0x00,0x30,0x1f,0x06,0x03,
		0x55,0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0x10,0xf0,0x0c,0x56,0x9b,0x61,0xea,
		0x57,0x3a,0xb6,0x35,0x97,0x6d,0x9f,0xdd,0xb9,0x14,0x8e,0xdb,0xe6,0x30,0x0d,0x06,
		0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x82,0x01,0x01,
		0x00,0xa9,0x38,0xb3,0x32,0xf2,0x4d,0xb0,0x13,0x55,0x6c,0x08,0x1b,0x06,0xf4,0xbb,
		0x97,0x4d,0xd0,0xbb,0xe1,0xb1,0x43,0x49,0xf8,0x4f,0x62,0x4f,0xdb,0x25,0x06,0x86,
		0x2c,0x31,0xa3,0x27,0xb4,0x61,0xd0,0x9f,0xd2,0x9a,0x30,0x23,0x28,0x5d,0x5c,0x47,
		0xbe,0xe6,0x3b,0xd4,0x7d,0x3b,0x68,0x35,0xde,0x1a,0xcd,0xd0,0xe0,0x54,0x50,0x61,
		0x42,0xf1,0xc7,0x23,0x29,0x49,0x63,0xb5,0x8b,0x09,0x25,0x87,0x4b,0x3a,0xb3,0x56,
		0xd6,0xb6,0xa2,0xe0,0x47,0xbd,0xc5,0x51,0x68,0xef,0x79,0x65,0x62,0x9b,0x95,0xf2,
		0xf3,0x20,0xdd,0xbc,0x27,0x86,0xf8,0x93,0xda,0x01,0xd1,0x7e,0x7f,0x76,0x5e,0xa1,
		0x10,0x7d,0x07,0xfd,0x27,0x98,0x72,0x79,0x32,0x8b,0xbc,0xe4,0xc9,0x1a,0xfd,0x85,
		0x83,0xf7,0x91,0x09,0xc8,0x74,0xff,0xfa,0x0f,0x2a,0x28,0xbb,0xbc,0x1e,0x85,0x86,
		0xc0,0x4f,0x91,0x16,0xfa,0x1b,0x18,0x84,0xae,0xd5,0x24,0x1a,0x00,0xb9,0x0b,0xae,
		0xe2,0x20,0x1f,0xa7,0x30,0xaf,0xb5,0x85,0x51,0xc9,0x8e,0x07,0x21,0x78,0x31,0x7d,
		0x51,0x97,0x6c,0xcc,0x8d,0xab,0xab,0xfa,0x0a,0x94,0x69,0xfc,0x76,0x56,0x0e,0xd2,
		0xa2,0x14,0x64,0x17,0x01,0x6e,0xbe,0x43,0x2d,0x57,0x0f,0x53,0xc5,0x48,0x52,0x54,
		0x65,0x3b,0x05,0x23,0xaa,0x91,0x58,0x15,0x86,0xdc,0x0f,0x0d,0x8f,0x1b,0x47,0xd0,
		0xe3,0x58,0xb1,0x70,0xcf,0x31,0x7d,0x2c,0x92,0xdd,0xc0,0xda,0xff,0x1e,0xb6,0xa7,
		0xda,0xce,0x86,0xad,0xc4,0x01,0x8b,0x32,0x97,0x81,0xe0,0x40,0x7f,0x3a,0xd5,0x72,
		0x1f,
	};
	static long size_tuc3F00DF00503C = sizeof(tuc3F00DF00503C);

	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	//-------------------------------------
	// use correct parameters
	// test card 2 version 1 (==IDX_READER_EID)
	//-------------------------------------
	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_EID).getName();

	status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
	CHECK_EQUAL(BEID_OK,status.general);

	//-------------------------------------
	// get the PINs from the card
	//-------------------------------------
	::BEID_Pins pins;
	status = BEID_GetPINs(&pins);
	CHECK_EQUAL(BEID_OK,status.general);
	CHECK_EQUAL(1,pins.pinsLength);

	// for all the pins...
	for (long pinIdx=0;pinIdx<pins.pinsLength;pinIdx++)
	{
		::BEID_Pin  pin;
		pin.pinType = pins.pins[pinIdx].pinType;
		pin.id		= pins.pins[pinIdx].id;
		pin.usageCode= pins.pins[pinIdx].usageCode;
		long		triesLeft = pins.pins[pinIdx].triesLeft;
		//-------------------------------------
		// read file BEID_FILE_ID and test
		//-------------------------------------
		BEID_Bytes outData;
		outData.length		 = size_tuc3F00DF014031;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_ID, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF014031);
		CHECK_ARRAY_EQUAL(tuc3F00DF014031,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_ID_SIGN and test
		//-------------------------------------

		outData.length		 = size_tuc3F00DF014032;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_ID_SIGN, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF014032);
		CHECK_ARRAY_EQUAL(tuc3F00DF014032,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_ADDRESS and test
		//-------------------------------------

		outData.length		 = size_tuc3F00DF014033;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_ADDRESS, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF014033);
		CHECK_ARRAY_EQUAL(tuc3F00DF014033,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_ADDRESS_SIGN and test
		//-------------------------------------

		outData.length		 = size_tuc3F00DF014034;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_ADDRESS_SIGN, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF014034);
		CHECK_ARRAY_EQUAL(tuc3F00DF014034,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_PHOTO and test
		//-------------------------------------

		outData.length		 = size_tuc3F00DF014035;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_PHOTO, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF014035);
		CHECK_ARRAY_EQUAL(tuc3F00DF014035,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_TOKENINFO and test
		//-------------------------------------

		outData.length		 = size_tuc3F00DF005032;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_TOKENINFO, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF005032);
		CHECK_ARRAY_EQUAL(tuc3F00DF005032,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_RRN and test
		//-------------------------------------

		outData.length		 = size_tuc3F00DF00503C;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_RRN, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF00503C);
		CHECK_ARRAY_EQUAL(tuc3F00DF00503C,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;
	}
	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);
}

TEST(BEIDReadFile_t1v2)
{
	//----------------------------------------------
	// data file that can be read
	// test card 1 version 2
	//----------------------------------------------
	unsigned char tuc3F00DF014031[] = {
		0x01,0x0c,0x35,0x39,0x30,0x34,0x35,0x36,0x37,0x38,0x39,0x30,0x31,0x32,0x02,0x10,
		0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x61,0x62,0x63,0x64,0x65,0x66,
		0x03,0x0a,0x30,0x31,0x2e,0x30,0x31,0x2e,0x32,0x30,0x30,0x37,0x04,0x0a,0x30,0x31,
		0x2e,0x30,0x31,0x2e,0x32,0x30,0x31,0x32,0x05,0x09,0x42,0x72,0x75,0x78,0x65,0x6c,
		0x6c,0x65,0x73,0x06,0x0b,0x38,0x30,0x30,0x31,0x32,0x35,0x31,0x31,0x32,0x36,0x35,
		0x07,0x05,0x6e,0x61,0x6d,0x65,0x31,0x08,0x06,0x67,0x6e,0x61,0x6d,0x65,0x31,0x09,
		0x00,0x0a,0x05,0x42,0x65,0x6c,0x67,0x65,0x0b,0x07,0x49,0x78,0x65,0x6c,0x6c,0x65,
		0x73,0x0c,0x0b,0x32,0x35,0x20,0x4a,0x41,0x4e,0x20,0x31,0x39,0x38,0x30,0x0d,0x01,
		0x4d,0x0e,0x06,0x50,0x72,0x69,0x6e,0x63,0x65,0x0f,0x01,0x31,0x10,0x01,0x30,0x11,
		0x14,0xdc,0x9d,0xb2,0xc8,0x3d,0xc1,0xf1,0xf7,0xd6,0x96,0x4c,0xae,0x35,0xda,0x0c,
		0xb9,0x2e,0x22,0x09,0xf0,
	};

	static long size_tuc3F00DF014031 = sizeof(tuc3F00DF014031);

	unsigned char tuc3F00DF014032[] = {
		0x8f,0x0b,0x38,0xf1,0xd7,0x34,0x65,0x73,0xed,0xff,0xca,0x2c,0x74,0x43,0xf6,0x65,
		0x10,0x4c,0x39,0xe6,0xf5,0xb8,0xf2,0x0c,0x37,0x32,0xbe,0xc5,0x3b,0x42,0xcb,0x73,
		0xe9,0xd5,0x9e,0x77,0xbf,0x7b,0xed,0x64,0x2c,0x47,0x5a,0x80,0xb5,0x2f,0xdf,0xeb,
		0xac,0x34,0xd6,0x1a,0x57,0x93,0x71,0xf3,0x3d,0xc5,0x9f,0x1b,0xa4,0x0c,0xc7,0x64,
		0x84,0x72,0xd5,0x82,0x35,0x4f,0xb8,0xcb,0x7a,0x3e,0xff,0x7a,0x23,0xc2,0x11,0x63,
		0x07,0x68,0x05,0x8a,0xf5,0x80,0xa1,0xdc,0x78,0xbe,0x4d,0x94,0x3a,0x21,0xc6,0x59,
		0x9f,0xcb,0x32,0xee,0x07,0x5d,0xb3,0x30,0xf1,0xa2,0x35,0xf1,0xf7,0x10,0x21,0x00,
		0xfc,0x3d,0x5c,0x77,0xed,0xe7,0x6e,0xcf,0x18,0x25,0xf7,0xc6,0x3a,0x57,0x0a,0x7d,
	};
	static long size_tuc3F00DF014032 = sizeof(tuc3F00DF014032);

	unsigned char tuc3F00DF014033[] = {
		0x01,0x0b,0x56,0x69,0x6c,0x61,0x6c,0x61,0x61,0x6e,0x20,0x31,0x33,0x02,0x04,0x31,
		0x36,0x30,0x31,0x03,0x09,0x52,0x75,0x69,0x73,0x62,0x72,0x6f,0x65,0x6b,
	};
	static long size_tuc3F00DF014033 = sizeof(tuc3F00DF014033);

	unsigned char tuc3F00DF014034[] = {
		0x8b,0xa8,0x2b,0xa6,0x05,0x7b,0x47,0xda,0x27,0xec,0xc6,0x59,0xfe,0x5c,0x5e,0xb4,
		0xf9,0x2e,0xf9,0xd9,0xae,0x30,0x46,0xa0,0x70,0x92,0xc3,0x58,0x69,0x0f,0x9f,0x6a,
		0x9c,0x10,0xb3,0x15,0x37,0x86,0x81,0x06,0x5c,0xf1,0x69,0x8a,0x92,0x40,0xc5,0x05,
		0xd3,0x23,0xca,0x71,0x6a,0x30,0x0d,0x2b,0x75,0x59,0xc7,0x0d,0xe0,0xf8,0xe8,0x56,
		0x9f,0xcd,0x1b,0x43,0x6f,0xa8,0xff,0x5c,0x66,0xd6,0xc1,0xa3,0x2a,0xdb,0xe2,0xe0,
		0x72,0x0f,0x5f,0xd2,0x2b,0x15,0xff,0x21,0x50,0x7c,0x2e,0xe6,0x16,0x2e,0xf6,0x0c,
		0x91,0xc5,0x99,0x4b,0xb5,0x23,0x47,0x18,0xfb,0x41,0xe3,0x5f,0xd8,0xd5,0xe0,0x1b,
		0xe1,0x67,0x61,0x86,0x79,0xdc,0xe2,0xbc,0x69,0x69,0x9e,0x51,0xe4,0x7b,0xf4,0xc1,
	};
	static long size_tuc3F00DF014034 = sizeof(tuc3F00DF014034);

	unsigned char tuc3F00DF014035[] = {
		0xff,0xd8,0xff,0xe0,0x00,0x10,0x4a,0x46,0x49,0x46,0x00,0x01,0x02,0x01,0x01,0x2c,
		0x01,0x2c,0x00,0x00,0xff,0xdb,0x00,0x43,0x00,0x1c,0x13,0x15,0x18,0x15,0x11,0x1c,
		0x18,0x16,0x18,0x1f,0x1d,0x1c,0x21,0x29,0x45,0x2d,0x29,0x26,0x26,0x29,0x54,0x3c,
		0x40,0x32,0x45,0x64,0x58,0x69,0x67,0x62,0x58,0x60,0x5f,0x6e,0x7c,0x9e,0x86,0x6e,
		0x75,0x96,0x77,0x5f,0x60,0x8a,0xbb,0x8b,0x96,0xa3,0xa9,0xb1,0xb3,0xb1,0x6b,0x84,
		0xc2,0xd0,0xc1,0xac,0xce,0x9e,0xae,0xb1,0xaa,0xff,0xc0,0x00,0x0b,0x08,0x00,0xc8,
		0x00,0x8c,0x01,0x01,0x11,0x00,0xff,0xc4,0x00,0xd2,0x00,0x00,0x01,0x05,0x01,0x01,
		0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x03,0x04,
		0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x10,0x00,0x02,0x01,0x03,0x03,0x02,0x04,0x03,
		0x05,0x05,0x04,0x04,0x00,0x00,0x01,0x7d,0x01,0x02,0x03,0x00,0x04,0x11,0x05,0x12,
		0x21,0x31,0x41,0x06,0x13,0x51,0x61,0x07,0x22,0x71,0x14,0x32,0x81,0x91,0xa1,0x08,
		0x23,0x42,0xb1,0xc1,0x15,0x52,0xd1,0xf0,0x24,0x33,0x62,0x72,0x82,0x09,0x0a,0x16,
		0x17,0x18,0x19,0x1a,0x25,0x26,0x27,0x28,0x29,0x2a,0x34,0x35,0x36,0x37,0x38,0x39,
		0x3a,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4a,0x53,0x54,0x55,0x56,0x57,0x58,0x59,
		0x5a,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0x6a,0x73,0x74,0x75,0x76,0x77,0x78,0x79,
		0x7a,0x83,0x84,0x85,0x86,0x87,0x88,0x89,0x8a,0x92,0x93,0x94,0x95,0x96,0x97,0x98,
		0x99,0x9a,0xa2,0xa3,0xa4,0xa5,0xa6,0xa7,0xa8,0xa9,0xaa,0xb2,0xb3,0xb4,0xb5,0xb6,
		0xb7,0xb8,0xb9,0xba,0xc2,0xc3,0xc4,0xc5,0xc6,0xc7,0xc8,0xc9,0xca,0xd2,0xd3,0xd4,
		0xd5,0xd6,0xd7,0xd8,0xd9,0xda,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,0xe9,0xea,
		0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,0xff,0xda,0x00,0x08,0x01,0x01,
		0x00,0x00,0x3f,0x00,0xdf,0xc9,0xf5,0x34,0x64,0xfa,0x9a,0x32,0x7d,0x4d,0x19,0x3e,
		0xa6,0x8c,0x9f,0x53,0x46,0x4f,0xa9,0xa3,0x27,0xd4,0xd2,0x16,0xc0,0xc9,0x6c,0x7e,
		0x35,0x1f,0xda,0xa1,0x1d,0x67,0x4f,0xfb,0xec,0x53,0xd6,0x55,0x7f,0xb9,0x20,0x6f,
		0xa1,0xcd,0x3b,0x27,0xd4,0xd1,0x93,0xea,0x68,0xc9,0xf5,0x34,0x64,0xfa,0x9a,0x32,
		0x7d,0x4d,0x19,0x3e,0xa6,0x8c,0x9f,0x53,0x46,0x4f,0xa9,0xa4,0xa2,0x8a,0x28,0xaa,
		0x73,0x6a,0x31,0x46,0xfb,0x57,0xe6,0xc1,0xe4,0xe7,0x8a,0xab,0x73,0xa9,0x3b,0x8f,
		0xdc,0xfc,0xab,0x8c,0x93,0x9e,0x6b,0x2a,0x4b,0xd2,0xe7,0x25,0x99,0x88,0xee,0xd9,
		0x26,0xa3,0x17,0x32,0x72,0x36,0x82,0x3d,0x0a,0xff,0x00,0xf5,0xaa,0x58,0xe7,0x72,
		0x7e,0x50,0x8c,0x7b,0x00,0x7f,0xc9,0xab,0xd0,0xea,0x32,0x21,0xc1,0xde,0x87,0xb8,
		0x3f,0x30,0xfd,0x6b,0x52,0x0b,0xc8,0xe6,0xc0,0xe8,0x4f,0xa7,0x23,0xff,0x00,0xad,
		0xf8,0xd5,0x8a,0x28,0xa2,0x8a,0x28,0xa2,0x8a,0x46,0x65,0x45,0x2c,0xc4,0x05,0x51,
		0x92,0x4f,0x40,0x2b,0x9b,0xd4,0x35,0x69,0x2e,0x58,0xa4,0x04,0xac,0x59,0xe0,0x63,
		0xaf,0xb9,0xac,0xf5,0xf9,0x9c,0xf9,0xb2,0x1c,0x0e,0xde,0xb4,0x34,0xd6,0xd9,0xc7,
		0x97,0x9c,0x0e,0xb9,0x34,0x79,0xf0,0x63,0xfd,0x58,0xfc,0xa9,0x37,0xc2,0xdd,0x13,
		0x6f,0xb8,0x6c,0x7f,0x3a,0x9d,0x77,0x60,0x79,0x6e,0x19,0x4f,0x69,0x3f,0xc4,0x54,
		0xc9,0x37,0x96,0x42,0x48,0x0c,0x64,0xf4,0x0f,0x82,0xbf,0x81,0xab,0x03,0x83,0xc8,
		0x1c,0x76,0x35,0xa9,0xa7,0xcf,0xbd,0x7c,0xb2,0x78,0x51,0xf2,0x83,0xd4,0x7b,0x55,
		0xda,0x28,0xa2,0x8a,0x28,0xa2,0xb1,0xb5,0xcb,0x96,0x91,0x96,0xce,0x16,0xc7,0xf1,
		0x4a,0x47,0x61,0xd8,0x7f,0x5f,0xca,0xb2,0x7e,0x44,0x52,0xa9,0x8f,0x73,0x55,0x27,
		0xdb,0x9f,0x94,0xf3,0xf5,0xa8,0x83,0x7f,0x7b,0x9a,0x4c,0x7a,0x52,0xab,0x76,0x65,
		0xc8,0xa9,0x51,0x33,0xcc,0x2c,0x43,0x7a,0x67,0x07,0xf3,0xa9,0x96,0xea,0x44,0x1b,
		0x26,0x40,0xeb,0xdd,0x58,0x55,0x88,0x64,0x08,0x99,0xb7,0x6d,0xf1,0x8e,0xa8,0x7a,
		0xaf,0xd3,0xfc,0x2a,0xe4,0x32,0x2b,0x61,0xd4,0xed,0xc1,0xe1,0x87,0x05,0x4d,0x6c,
		0xda,0xdc,0x79,0xf1,0xfc,0xd8,0x12,0x0f,0xbc,0xa3,0xf9,0xd4,0xf4,0x51,0x45,0x14,
		0x53,0x26,0x90,0x43,0x0b,0xca,0xc3,0x21,0x14,0xb6,0x07,0x7c,0x76,0xae,0x62,0xea,
		0x46,0x3b,0x9d,0x8e,0x19,0xdb,0x73,0x9f,0xe8,0x2b,0x36,0x49,0x4b,0x70,0x0e,0x05,
		0x0a,0x85,0xce,0x31,0x9a,0x9d,0x2c,0x9d,0x97,0x27,0x8a,0x63,0x5b,0xb2,0x9f,0xeb,
		0x48,0x21,0xdc,0xa7,0xa0,0x61,0xda,0x98,0x51,0x94,0xf4,0x23,0x15,0x34,0x72,0xab,
		0x0d,0xb3,0x82,0x47,0xf7,0x87,0x51,0x4a,0x55,0xed,0xc8,0x92,0x32,0x1b,0xdc,0x74,
		0x61,0xe8,0x45,0x59,0x86,0x65,0x90,0x87,0x8f,0xe4,0x27,0x86,0x52,0x7a,0xff,0x00,
		0xf5,0xfd,0x3d,0x6b,0x4e,0xd6,0x53,0x1b,0x2c,0x89,0x93,0xb7,0x82,0xbe,0xde,0x95,
		0xb0,0x08,0x20,0x10,0x72,0x0f,0x20,0xd2,0xd1,0x45,0x14,0x55,0x0d,0x5e,0x60,0x96,
		0xeb,0x19,0x07,0x0e,0x77,0x13,0x8e,0x81,0x70,0x7f,0x9e,0x3f,0x5a,0xe5,0xae,0x65,
		0x2e,0xdc,0xfa,0xd3,0x60,0x84,0xca,0xe0,0x0a,0xdc,0xb4,0xb0,0x0a,0x01,0x22,0xb4,
		0x05,0xaa,0xec,0xe9,0xda,0xaa,0x5c,0xd8,0xe3,0x90,0x2b,0x32,0x68,0xb6,0x7b,0x1e,
		0xc7,0xd2,0xab,0x82,0xa3,0x83,0xf4,0xe3,0xb1,0xa8,0xa5,0x5e,0xeb,0x4d,0x8e,0x52,
		0xbc,0x75,0x0d,0xd4,0x52,0x9f,0x91,0xb7,0xaf,0xdd,0x61,0x86,0xad,0x7b,0x09,0xcb,
		0x27,0x27,0x2e,0xbc,0x30,0xfe,0x55,0xb9,0x68,0x73,0x16,0xdc,0xe4,0x0e,0x9f,0x4a,
		0x9e,0x8a,0x28,0xa2,0xb0,0xf5,0xf9,0x71,0x26,0xcd,0xd9,0x01,0x00,0x23,0xd0,0xf2,
		0x4f,0xfe,0xcb,0x5c,0xf0,0xcb,0x35,0x6d,0x69,0x76,0xe0,0x28,0x62,0x3a,0xfb,0x56,
		0xd2,0x2f,0x4e,0x95,0x30,0xe9,0x41,0x00,0x8e,0x6a,0x9d,0xdd,0x92,0xcc,0x84,0xa8,
		0xc3,0x7f,0x3a,0xe7,0x2f,0x2d,0xde,0x19,0x09,0xda,0x45,0x55,0x0e,0x7a,0x1e,0x94,
		0xd7,0x03,0x39,0x1d,0xe9,0xc8,0xdf,0xc3,0xd8,0x8f,0xce,0xac,0xd8,0xc9,0xe5,0x4e,
		0x37,0x74,0xe8,0x7e,0x95,0xd2,0xd8,0xb9,0x0d,0xb0,0x9e,0x6a,0xf5,0x14,0x51,0x45,
		0x72,0x7a,0xa4,0xbe,0x65,0xc5,0xce,0x58,0x13,0xe6,0x1f,0xc8,0x1c,0x0f,0xd0,0x0a,
		0xcf,0x8c,0x66,0x45,0x1e,0xa7,0x15,0xd2,0xd9,0xae,0xd8,0xc0,0x1c,0x55,0xf5,0xcf,
		0xa5,0x4a,0x29,0xd8,0xa3,0x15,0x4e,0xf6,0xc5,0x6e,0x54,0xe3,0x01,0xbd,0x7d,0x6b,
		0x9a,0xbd,0xb1,0x92,0xd9,0xce,0x54,0xe2,0xaa,0x88,0xcb,0x02,0x54,0xfe,0x14,0xcc,
		0x11,0xd7,0x8a,0x9e,0x32,0x37,0x29,0x3d,0x09,0xda,0x6b,0xa1,0xb3,0x90,0xe6,0x16,
		0x63,0xce,0x02,0xb1,0xf7,0xad,0x7a,0x28,0xa2,0x82,0xc1,0x41,0x66,0x38,0x50,0x32,
		0x7e,0x95,0xc2,0x33,0x92,0xe4,0xb1,0xc9,0x27,0x9f,0x7a,0x7d,0xa2,0xef,0xb9,0x41,
		0xfe,0xd6,0x6b,0xa7,0x84,0x04,0x51,0xb8,0x81,0xc7,0x7a,0x9c,0x4f,0x1a,0xf5,0x3c,
		0x7a,0xd4,0xab,0x2c,0x6d,0xf7,0x5d,0x4f,0xe3,0x52,0x02,0x28,0xcd,0x04,0x81,0xd6,
		0xa1,0x9e,0x38,0xe5,0x8c,0x89,0x30,0x47,0xbd,0x61,0xdd,0xe9,0x91,0x07,0x0d,0x6f,
		0x28,0x27,0xfb,0xbd,0x6a,0x85,0xcd,0xac,0x85,0x0b,0x18,0xc6,0x57,0xba,0x90,0x73,
		0xf8,0x55,0x64,0xf9,0xb3,0x9e,0xe3,0x15,0xb9,0x6e,0xe3,0x64,0x64,0x7f,0x7f,0xfa,
		0x56,0xf2,0x1d,0xc8,0xa7,0xd4,0x66,0x96,0x8a,0x2a,0xa6,0xaa,0xfb,0x34,0xd9,0x88,
		0xea,0x46,0xde,0xb8,0xea,0x70,0x7f,0x4c,0xd7,0x35,0x6d,0xb6,0x58,0x96,0x29,0x23,
		0x4d,0xac,0xa4,0x86,0x1f,0x7b,0x3e,0xb4,0xba,0x4c,0x3b,0xae,0x86,0x46,0x71,0xde,
		0xb6,0xe5,0x75,0x56,0xcd,0x40,0xfa,0x84,0x2a,0x30,0xcb,0xc1,0xef,0xc6,0x2a,0xb9,
		0xbc,0x80,0xb8,0xf2,0x98,0xe4,0xfa,0x73,0x57,0xac,0xef,0x43,0x10,0x37,0x86,0x07,
		0x8c,0xe6,0xb4,0x41,0xc8,0xcd,0x55,0xba,0xb8,0xf2,0xc1,0x19,0x22,0xb2,0xe5,0xba,
		0xe7,0xe6,0x66,0x23,0xbf,0xa0,0xa9,0x12,0xee,0xdd,0x40,0xda,0xd1,0xb1,0xf4,0xde,
		0x09,0xa7,0x49,0x24,0x53,0x29,0xf9,0x70,0x7d,0xeb,0x04,0xaf,0x97,0x3b,0x27,0xa1,
		0xad,0x2b,0x39,0x0f,0x93,0xc9,0xfb,0xae,0x31,0xfe,0x7f,0x1a,0xe9,0x2d,0x8e,0x6d,
		0xd0,0xfe,0x15,0x25,0x14,0x56,0x57,0x88,0xa4,0xd9,0x62,0xab,0xdd,0x9b,0x3f,0x80,
		0x18,0xfe,0xa2,0xb2,0x2c,0x88,0x7b,0x55,0x66,0x1f,0xea,0x8b,0x28,0xfe,0x7f,0xd6,
		0xae,0xe9,0x4a,0x3c,0xc7,0x60,0x31,0x56,0xee,0x2d,0xcc,0x8b,0x90,0x71,0x59,0x46,
		0xc0,0x2b,0xc8,0x25,0x52,0xe1,0xc6,0x03,0x77,0x43,0xf4,0xf4,0xa5,0xb2,0xb4,0x95,
		0xa4,0x41,0x74,0x77,0xc5,0x16,0x4a,0x20,0xf5,0x20,0x0f,0xcb,0x81,0x57,0x52,0xcb,
		0x6b,0x96,0x04,0xf2,0x78,0xc9,0xe7,0xf1,0xad,0x78,0x87,0xee,0x86,0x7a,0xe2,0xa8,
		0xdd,0x41,0xe6,0xcb,0x82,0x71,0x59,0xf2,0xe9,0xa8,0xc8,0xe8,0xea,0xdb,0xf1,0xf2,
		0x3a,0x9e,0x14,0xfd,0x2a,0xbc,0x16,0x52,0x64,0x9b,0xa4,0x32,0x32,0xae,0xc4,0x50,
		0x46,0x31,0xef,0x56,0xac,0x6c,0x1e,0x25,0xc3,0x1e,0x3d,0x3d,0x3e,0x95,0x9f,0xaa,
		0x43,0xe4,0x5e,0x86,0x1f,0x75,0xd7,0xfc,0xff,0x00,0x4a,0x4b,0x47,0xca,0x14,0xf7,
		0x26,0xba,0x6d,0x35,0xf7,0x5b,0x0e,0xb5,0x6e,0x8a,0x2b,0x03,0xc4,0x8d,0xf3,0x80,
		0x1b,0x8d,0xa0,0x63,0xd0,0xe4,0x9f,0xf0,0xaa,0x1a,0x4b,0x06,0x12,0xc0,0xcd,0x8c,
		0xe0,0x8c,0x7e,0xbf,0xd2,0xb5,0x6c,0xd0,0xc7,0x3b,0x29,0xe9,0x8e,0x0f,0xad,0x68,
		0x22,0xe7,0xad,0x23,0xc0,0x0f,0x6a,0x44,0xb7,0x51,0xfc,0x38,0xa7,0xf9,0x63,0x77,
		0xa6,0x2a,0x55,0xe0,0x53,0x08,0xc9,0xc1,0xa4,0x68,0x43,0x0f,0x7f,0x5a,0x6f,0xd9,
		0xc6,0x73,0x4e,0x29,0xb5,0x6b,0x0f,0x5f,0x50,0x22,0x88,0xf1,0x9d,0xdf,0xd2,0xb2,
		0xed,0x9b,0x13,0x2e,0x7b,0x9c,0x57,0x4b,0xa3,0xb6,0x63,0x2b,0xd8,0x7e,0x95,0xa5,
		0x45,0x15,0xcd,0x78,0x83,0x22,0x79,0x33,0xd0,0xba,0xe3,0xfe,0xf9,0x15,0x99,0x04,
		0xa2,0x0b,0x84,0x98,0x03,0x81,0x8c,0x8f,0x62,0x39,0xad,0xc8,0xe5,0x5f,0xb4,0x23,
		0x2b,0x82,0x8c,0x7e,0x52,0x0d,0x69,0x23,0x7c,0xc4,0x54,0xe2,0x8e,0x2a,0x17,0x62,
		0xcc,0x42,0xf4,0x15,0x3a,0xf2,0x2a,0x27,0x38,0x3f,0x2f,0x24,0x53,0xe3,0x70,0xeb,
		0x91,0xf9,0x53,0xce,0x2a,0x29,0x5b,0x0b,0x5c,0xce,0xb7,0x36,0xfb,0x94,0x41,0xd1,
		0x41,0x3f,0xe7,0xf2,0xac,0xf8,0xdb,0x0c,0xa7,0xd0,0xd6,0xfe,0x87,0x3f,0xef,0xca,
		0x9f,0xe2,0x18,0xad,0xda,0x28,0xae,0x6b,0xc4,0x21,0x85,0xc4,0x99,0x3c,0x16,0x52,
		0x07,0xb6,0xd0,0x3f,0xa1,0xac,0x86,0xc1,0x40,0x79,0xe9,0x49,0x0b,0x88,0xe7,0x47,
		0x27,0x85,0x60,0xc7,0xf3,0xae,0xb1,0x5c,0x70,0xd5,0x65,0x1f,0x8a,0x79,0xf9,0x85,
		0x52,0xba,0x8a,0xe5,0x43,0x79,0x2d,0x8c,0xf7,0xe0,0xff,0x00,0x3a,0x8a,0x3b,0xa9,
		0x91,0x42,0x30,0xcb,0x1e,0x0e,0x05,0x3a,0x34,0xbb,0x33,0xee,0xdf,0xf2,0x9e,0xab,
		0xc6,0x3f,0xc6,0xae,0xa2,0x95,0x39,0x14,0xad,0x20,0x15,0x5a,0xe6,0x5f,0x93,0xad,
		0x72,0xd7,0xcf,0xbe,0xee,0x43,0x9c,0xe0,0xe2,0xa2,0x5c,0x71,0xf5,0xe6,0xaf,0xe9,
		0xd2,0x94,0x99,0x48,0x38,0x65,0x39,0x15,0xd7,0xfd,0x3a,0x51,0x45,0x61,0x78,0x96,
		0x03,0xfb,0xb9,0xf9,0x2a,0x46,0xc3,0xec,0x46,0x48,0xfe,0x67,0xf2,0xac,0x40,0x84,
		0xdb,0x67,0xd0,0xd4,0x1d,0xfe,0xb5,0xbf,0xa7,0xdc,0x19,0xad,0x14,0xb1,0xc9,0x1c,
		0x1f,0xad,0x68,0x44,0x4f,0x63,0x53,0x89,0x31,0x4b,0x24,0xd1,0xa2,0xe5,0xdc,0x28,
		0xf7,0x35,0x02,0xdc,0xda,0xb9,0xe2,0x64,0xe3,0xf0,0x34,0xe5,0xbc,0xb5,0xdd,0x81,
		0x32,0xe7,0xdf,0x8a,0x90,0xb8,0x61,0x90,0x47,0xb1,0x1c,0xd4,0x72,0x9f,0x6a,0xce,
		0xbb,0x90,0x24,0x6c,0xc4,0xf0,0xa3,0x35,0xce,0x9c,0x93,0x93,0xde,0x95,0x4e,0x08,
		0xab,0x16,0xe7,0x0e,0xac,0xa7,0x90,0x6b,0xb2,0xb5,0x7f,0x32,0xda,0x36,0xff,0x00,
		0x64,0x0a,0x96,0x8a,0xa3,0xac,0xc5,0xe6,0xe9,0xb2,0x71,0x92,0xa4,0x30,0xfc,0xff,
		0x00,0xc0,0x9a,0xe7,0xfc,0xb0,0x96,0x59,0x3d,0x73,0x59,0xec,0x30,0xc7,0xd8,0xd5,
		0xcd,0x2e,0x63,0x1c,0xcd,0x19,0x3f,0x2b,0x0c,0xfe,0x35,0xb9,0x0c,0x82,0x9f,0x26,
		0x5f,0xa3,0x11,0x54,0xe6,0xb1,0x0c,0xfb,0xdd,0xde,0x5f,0x66,0x34,0x82,0xd6,0xd0,
		0xf5,0x49,0x01,0xf6,0x63,0x4e,0x36,0x56,0xee,0x30,0x91,0x91,0xee,0xc4,0xe6,0xac,
		0x41,0x6c,0x96,0xe0,0x08,0xcb,0x63,0xd3,0x71,0x34,0xf9,0xa4,0x01,0x71,0x58,0x9a,
		0xa4,0xe0,0x81,0x12,0x9e,0x4f,0x2d,0xcd,0x67,0x01,0x4b,0x8c,0xb0,0xa9,0xa1,0x8d,
		0x98,0xe5,0x06,0x70,0x09,0x3f,0x4a,0xec,0x2c,0x10,0x25,0xa2,0x61,0x98,0x86,0x19,
		0xc1,0xe8,0x0f,0x7c,0x55,0x8a,0x29,0x0a,0x86,0x52,0xac,0x32,0x08,0xc1,0x1e,0xa2,
		0xb9,0x8d,0x48,0xa8,0x41,0x1a,0x0e,0x17,0x82,0x7d,0xeb,0x2d,0x86,0x5b,0xeb,0x8a,
		0x20,0x6d,0x92,0xa9,0x15,0xb3,0x1c,0x98,0xe0,0xfe,0x75,0x7a,0x12,0x1d,0x78,0x3c,
		0xd4,0xeb,0x06,0xe1,0x8c,0x8a,0x43,0x69,0x4e,0xfb,0x38,0x41,0x51,0xc8,0xc2,0x31,
		0xcf,0x4a,0xce,0xba,0xb8,0xda,0xa7,0x1c,0x91,0x58,0x45,0x8b,0xbe,0xe6,0xe4,0x93,
		0x4e,0xcf,0x24,0xf6,0xa1,0x3a,0x56,0xae,0x8f,0x0b,0xc8,0xec,0x14,0x91,0xbd,0x76,
		0x9c,0x77,0x04,0xf3,0xfc,0xab,0xa7,0x55,0x0a,0xa1,0x47,0x40,0x31,0x4b,0x45,0x55,
		0xbd,0xb9,0xf2,0xe3,0x28,0x8d,0x87,0xc7,0x27,0xd0,0x57,0x39,0x70,0xdb,0x99,0xff,
		0x00,0x31,0x54,0xdb,0xa2,0x9c,0x7b,0x54,0x47,0x82,0x78,0xad,0x68,0xce,0xe8,0x55,
		0x87,0x71,0x4e,0x8e,0xe1,0xa1,0x39,0x3d,0x2b,0x46,0x1b,0xf8,0xd8,0x73,0x80,0x6a,
		0x71,0x76,0x84,0x75,0x1f,0x9d,0x47,0x25,0xec,0x60,0x7d,0xe1,0xf9,0xd6,0x74,0xd7,
		0x2f,0x70,0xd8,0x40,0x70,0x7b,0xd4,0x53,0xc7,0xb2,0xdd,0x8e,0x0e,0x48,0xac,0x71,
		0x4a,0x4f,0x18,0xa7,0xa9,0xc2,0xf4,0xae,0x83,0xc3,0xd2,0xc6,0xa8,0xc1,0x8e,0x19,
		0xb0,0x01,0xf6,0xff,0x00,0xf5,0xd6,0xe5,0x15,0x4a,0x5b,0x8d,0x89,0x9b,0x83,0xb5,
		0xba,0x84,0x15,0x97,0x35,0xc1,0x6d,0xc4,0x93,0x8e,0xb8,0x3f,0xce,0xb3,0x2e,0x1b,
		0x18,0x3d,0x7a,0xe6,0xa2,0x5f,0x98,0x91,0x9e,0xf9,0xa8,0xd8,0x10,0x2b,0x42,0xc5,
		0xb7,0x43,0xb4,0xf6,0x35,0x2b,0xaf,0x04,0x54,0x0f,0x1f,0xa7,0x5a,0x66,0xd9,0xfa,
		0x29,0xfd,0x6a,0x68,0xad,0x89,0x20,0xbf,0x35,0x76,0x28,0x80,0xc6,0x05,0x45,0x78,
		0x7f,0x74,0xc0,0x7a,0x56,0x29,0x5c,0x73,0x48,0x54,0x83,0x46,0xe2,0x48,0x1e,0xf5,
		0x3c,0x57,0x32,0xc5,0xb5,0x91,0xc8,0x60,0x30,0x3b,0xf1,0xe9,0x5a,0x56,0xda,0xdc,
		0xf1,0x80,0xb2,0x63,0x6f,0xa8,0x15,0xb2,0x97,0x37,0x45,0x41,0x30,0x02,0x0f,0x20,
		0x8e,0x84,0x57,0x3e,0xad,0x31,0x3f,0x3c,0x81,0x7f,0x53,0x51,0x4d,0x2e,0x1f,0xcb,
		0x53,0xc7,0x73,0xea,0x6a,0xac,0xaf,0xbd,0xcf,0xa0,0xa6,0x2b,0xe1,0xf3,0xda,0xa4,
		0x73,0xb9,0x73,0xde,0xac,0x58,0x3e,0x19,0x97,0xd7,0x15,0xa1,0xd4,0x50,0x17,0x3d,
		0x45,0x48,0xb1,0x8e,0xf4,0xed,0xa0,0x70,0x3a,0xd3,0x88,0xc0,0xcd,0x57,0x9f,0x95,
		0xc5,0x65,0xc8,0x98,0x72,0xb8,0xc7,0x26,0xa4,0xb5,0x84,0x4d,0x87,0x73,0x80,0x2a,
		0x9b,0x10,0x64,0x24,0x70,0xb9,0xe3,0xe9,0x46,0x79,0xa3,0x71,0xe2,0xa6,0x4b,0x99,
		0x95,0x40,0x59,0x5c,0x01,0xd8,0x31,0xa5,0x33,0x9c,0x1d,0xbd,0x4f,0x15,0x13,0x1d,
		0xa0,0x8c,0xfc,0xc7,0xaf,0xb5,0x47,0x40,0x1c,0xd3,0xc1,0x3b,0x6a,0x6b,0x73,0xb6,
		0x40,0x7d,0xeb,0x52,0x36,0xe3,0x1e,0xd5,0x20,0xf5,0xa5,0xc9,0xa7,0x03,0x4a,0x49,
		0x35,0x11,0x5c,0xf6,0xaa,0x97,0x10,0xe0,0x87,0xc7,0x43,0x93,0x55,0x25,0x76,0x84,
		0x3c,0x4a,0x4e,0x09,0xcd,0x56,0xef,0x41,0xe2,0x90,0x52,0xe7,0x14,0xed,0xea,0x83,
		0xe5,0x19,0x3e,0xa6,0x98,0x49,0x34,0x0a,0x72,0x7d,0xef,0xa5,0x03,0xbd,0x5a,0xb7,
		0x4c,0xb6,0x71,0xe9,0x5a,0x08,0x3d,0xbb,0x54,0xa0,0x0c,0xf1,0x4b,0x4f,0x00,0xd3,
		0xd6,0x33,0xdc,0x53,0xd2,0x1c,0x9a,0x65,0xd4,0x4b,0x1c,0x6c,0xcf,0xc0,0x03,0x9a,
		0xe7,0x70,0x25,0xde,0xc4,0x90,0xdc,0x6d,0x18,0xce,0x7b,0x53,0x18,0x15,0x7d,0xa7,
		0xa8,0x38,0x34,0xc3,0x9c,0xd2,0xd1,0x49,0x45,0x14,0xbf,0x4f,0xc6,0x9c,0x80,0xb1,
		0x00,0x75,0x35,0xb3,0x69,0x6d,0xfb,0xbc,0x8e,0xe7,0xf4,0xab,0x9f,0x67,0x38,0xfa,
		0x0a,0x55,0x87,0x27,0x9a,0x77,0x91,0xcd,0x38,0x42,0x41,0xe9,0x53,0xa4,0x27,0xd2,
		0xa5,0x54,0x0a,0x39,0x15,0x8b,0xe2,0x0b,0x9d,0xa1,0x60,0x46,0xe5,0x8e,0x5b,0x07,
		0xb5,0x61,0x06,0x65,0x39,0x04,0x83,0xea,0x0d,0x27,0x42,0x3e,0xb4,0x1e,0xb4,0x51,
		0x9a,0x4a,0x01,0xc1,0xcd,0x2e,0x68,0x03,0xbd,0x28,0x24,0x74,0x35,0xd3,0xe9,0x92,
		0x25,0xc5,0xb2,0xc8,0xa3,0x04,0x7c,0xac,0x3d,0x0d,0x5f,0x08,0x29,0x42,0x73,0x9e,
		0xb4,0xbe,0x58,0xa5,0x08,0x01,0xa7,0xd4,0x53,0xca,0xb1,0x44,0xce,0xc4,0x00,0x06,
		0x49,0x35,0xc7,0x5d,0xdc,0x1b,0x8b,0x87,0x94,0xe7,0xe6,0x3c,0x67,0xb0,0xed,0x50,
		0x0a,0x3b,0x8a,0x28,0xa5,0x15,0x3f,0xf6,0x75,0xf7,0xfc,0xf9,0xdc,0x7f,0xdf,0xa6,
		0xff,0x00,0x0a,0x4f,0xec,0xeb,0xef,0xf9,0xf3,0xb8,0xff,0x00,0xbf,0x4d,0xfe,0x14,
		0xa3,0x4f,0xbd,0xff,0x00,0x9f,0x3b,0x8f,0xfb,0xf4,0xdf,0xe1,0x4b,0xfd,0x9f,0x7d,
		0xff,0x00,0x3e,0x77,0x1f,0xf7,0xe9,0xbf,0xc2,0x81,0xa7,0xdf,0x7f,0xcf,0x9d,0xc7,
		0xfd,0xfa,0x6f,0xf0,0xad,0x2d,0x16,0x2b,0xcb,0x6b,0x83,0x1c,0x96,0xb3,0x88,0xe4,
		0xee,0x63,0x38,0x04,0x56,0xf8,0x46,0xc7,0xdc,0x6f,0xca,0x9c,0x11,0xbf,0xba,0xdf,
		0x95,0x3b,0x6b,0x63,0xee,0x9f,0xca,0x82,0xad,0xfd,0xd3,0xf9,0x53,0x08,0x7e,0x7e,
		0x46,0xfc,0xab,0x27,0x5a,0x17,0x4f,0x00,0x8a,0x08,0x26,0x6d,0xe7,0xe6,0x2a,0x84,
		0xe0,0x56,0x4a,0x68,0xf7,0xb2,0x23,0xb7,0x90,0xea,0x55,0x37,0xe1,0x90,0x8c,0xfb,
		0x0f,0x7f,0x6a,0x87,0xfb,0x3a,0xf7,0xfe,0x7c,0xee,0x3f,0xef,0xdb,0x7f,0x85,0x27,
		0xf6,0x75,0xee,0x7f,0xe3,0xce,0xe3,0xfe,0xfd,0x37,0xf8,0x51,0xfd,0x9d,0x7d,0xff,
		0x00,0x3e,0x77,0x1f,0xf7,0xe9,0xbf,0xc2,0x8f,0xec,0xfb,0xef,0xf9,0xf3,0xb8,0xff,
		0x00,0xbf,0x4d,0xfe,0x14,0xbf,0xd9,0xf7,0xbf,0xf3,0xe7,0x71,0xff,0x00,0x7e,0x9b,
		0xfc,0x2b,0xff,0xd9,
	};
	static long size_tuc3F00DF014035 = sizeof(tuc3F00DF014035);

	unsigned char tuc3F00DF005032[] = {
		0x30,0x27,0x02,0x01,0x00,0x04,0x10,0x53,0x4c,0x49,0x4e,0x33,0x66,0x00,0x38,0x7b,
		0xff,0x24,0x72,0xb0,0x18,0x16,0x39,0x80,0x06,0x42,0x45,0x4c,0x50,0x49,0x43,0x03,
		0x02,0x04,0x30,0x9e,0x04,0x03,0x02,0x00,0x00,
	};
	static long size_tuc3F00DF005032 = sizeof(tuc3F00DF005032);

	unsigned char tuc3F00DF00503C[] = {
		0x30,0x82,0x03,0xd6,0x30,0x82,0x02,0xbe,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,0x01,
		0x00,0x00,0x00,0x00,0x00,0xfb,0x7c,0x59,0x10,0x24,0x00,0x00,0xce,0xc0,0x02,0x30,
		0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x30,0x3b,
		0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x0e,0x30,
		0x0c,0x06,0x03,0x55,0x04,0x0a,0x13,0x05,0x5a,0x45,0x54,0x45,0x53,0x31,0x1c,0x30,
		0x1a,0x06,0x03,0x55,0x04,0x03,0x13,0x13,0x5a,0x45,0x54,0x45,0x53,0x20,0x47,0x6f,
		0x76,0x65,0x72,0x6e,0x6d,0x65,0x6e,0x74,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,0x30,
		0x34,0x30,0x33,0x32,0x33,0x31,0x30,0x31,0x30,0x30,0x32,0x5a,0x17,0x0d,0x30,0x39,
		0x30,0x33,0x32,0x33,0x31,0x30,0x31,0x30,0x30,0x32,0x5a,0x30,0x21,0x31,0x0b,0x30,
		0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x12,0x30,0x10,0x06,0x03,
		0x55,0x04,0x03,0x13,0x09,0x5a,0x45,0x54,0x45,0x53,0x20,0x52,0x4e,0x20,0x30,0x81,
		0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,
		0x03,0x81,0x8d,0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb4,0xb8,0x1d,0xed,0xc5,
		0x2d,0x80,0x31,0x42,0x08,0x76,0x44,0xac,0x96,0xde,0x35,0x7c,0xbc,0xc6,0xcd,0xec,
		0x48,0x73,0xfd,0x84,0x0e,0x19,0x90,0x18,0x5a,0x18,0x15,0x18,0x6f,0xca,0xab,0x63,
		0x10,0xea,0x7a,0x0d,0x5d,0x4b,0x74,0x01,0xbe,0xcb,0x18,0x1b,0x97,0x2d,0x7a,0x41,
		0x08,0x04,0x0f,0xbb,0x61,0xf7,0xb7,0xc1,0x8d,0xe1,0x00,0xb5,0xc9,0x8f,0x1b,0x3f,
		0xdf,0x9a,0x1f,0x88,0xe6,0x07,0x80,0xb9,0xf8,0x2b,0xb1,0x69,0xc2,0xed,0x3b,0xfb,
		0xc2,0x47,0xea,0x08,0x2b,0xae,0xd2,0x80,0x43,0xdd,0xc0,0x23,0x1e,0x0a,0x4e,0xe7,
		0x0d,0x3d,0x41,0x97,0xb7,0x20,0x22,0xd9,0x40,0x6b,0x85,0xe2,0x45,0x5a,0x90,0x8f,
		0xbf,0xc7,0x04,0xa8,0xa7,0x69,0x32,0xb0,0x1e,0x7d,0xfb,0x02,0x03,0x01,0x00,0x01,
		0xa3,0x82,0x01,0x72,0x30,0x82,0x01,0x6e,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,
		0x01,0xff,0x04,0x04,0x03,0x02,0x06,0xc0,0x30,0x1d,0x06,0x03,0x55,0x1d,0x0e,0x04,
		0x16,0x04,0x14,0x82,0xce,0x78,0xd3,0x3f,0x6f,0x41,0x41,0xce,0xa1,0xec,0x04,0x89,
		0xcf,0x2d,0xb8,0xcb,0xc8,0x5a,0xc9,0x30,0x42,0x06,0x03,0x55,0x1d,0x20,0x04,0x3b,
		0x30,0x39,0x30,0x37,0x06,0x07,0x60,0x38,0x01,0x01,0x01,0x03,0x01,0x30,0x2c,0x30,
		0x2a,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x02,0x01,0x16,0x1e,0x68,0x74,0x74,
		0x70,0x3a,0x2f,0x2f,0x72,0x65,0x70,0x6f,0x73,0x69,0x74,0x6f,0x72,0x79,0x2e,0x65,
		0x69,0x64,0x2e,0x7a,0x65,0x74,0x65,0x73,0x2e,0x62,0x65,0x30,0x3c,0x06,0x03,0x55,
		0x1d,0x1f,0x04,0x35,0x30,0x33,0x30,0x31,0xa0,0x2f,0xa0,0x2d,0x86,0x2b,0x68,0x74,
		0x74,0x70,0x3a,0x2f,0x2f,0x63,0x72,0x6c,0x2e,0x65,0x69,0x64,0x2e,0x7a,0x65,0x74,
		0x65,0x73,0x2e,0x62,0x65,0x2f,0x62,0x65,0x6c,0x67,0x69,0x75,0x6d,0x2f,0x62,0x65,
		0x6c,0x67,0x69,0x75,0x6d,0x2e,0x63,0x72,0x6c,0x30,0x7c,0x06,0x08,0x2b,0x06,0x01,
		0x05,0x05,0x07,0x01,0x01,0x04,0x70,0x30,0x6e,0x30,0x3f,0x06,0x08,0x2b,0x06,0x01,
		0x05,0x05,0x07,0x30,0x02,0x86,0x33,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x77,0x77,
		0x77,0x2e,0x5a,0x65,0x74,0x65,0x73,0x43,0x61,0x72,0x64,0x73,0x2e,0x62,0x65,0x2f,
		0x64,0x75,0x6d,0x6d,0x79,0x2f,0x63,0x65,0x72,0x74,0x69,0x66,0x63,0x61,0x74,0x65,
		0x73,0x2f,0x74,0x65,0x73,0x74,0x2e,0x63,0x72,0x74,0x30,0x2b,0x06,0x08,0x2b,0x06,
		0x01,0x05,0x05,0x07,0x30,0x01,0x86,0x1f,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x6f,
		0x63,0x73,0x70,0x2e,0x5a,0x65,0x74,0x65,0x73,0x43,0x61,0x72,0x64,0x73,0x2e,0x62,
		0x65,0x2f,0x64,0x75,0x6d,0x6d,0x79,0x30,0x11,0x06,0x09,0x60,0x86,0x48,0x01,0x86,
		0xf8,0x42,0x01,0x01,0x04,0x04,0x03,0x02,0x04,0xb0,0x30,0x09,0x06,0x03,0x55,0x1d,
		0x13,0x04,0x02,0x30,0x00,0x30,0x1f,0x06,0x03,0x55,0x1d,0x23,0x04,0x18,0x30,0x16,
		0x80,0x14,0x46,0xad,0x98,0xa1,0xc0,0x9a,0x48,0x5e,0x61,0xa0,0xbc,0x7e,0x0b,0x50,
		0x02,0x82,0x03,0xb8,0x4d,0x46,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
		0x01,0x01,0x05,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x23,0xab,0x0b,0xf7,0xb9,0x25,
		0x13,0x94,0xd9,0x35,0xb6,0x72,0xc8,0xa2,0x6e,0xe5,0x39,0x6d,0x40,0xcd,0xf6,0x3b,
		0x40,0x17,0x25,0x41,0x3e,0x06,0x39,0x75,0x56,0x9c,0x8d,0x41,0x13,0x09,0xcd,0x44,
		0x70,0xfc,0xb0,0x2d,0x10,0x2d,0x8a,0x8a,0xca,0x18,0xe1,0x47,0x7f,0x63,0xf6,0xbb,
		0x02,0x67,0xf5,0xc9,0xf1,0x10,0x12,0xad,0xeb,0x3b,0xaa,0xbe,0x18,0x67,0x35,0x52,
		0xb6,0xe9,0xe9,0xf7,0x36,0xd8,0x31,0xc2,0x1e,0x29,0x34,0x29,0x48,0x56,0x79,0x54,
		0xdc,0xd9,0xa9,0x8c,0xc4,0x67,0x70,0x97,0xbd,0xa6,0x8e,0xe3,0xb8,0x82,0x9b,0x54,
		0x8f,0xe5,0x92,0x8a,0xea,0x11,0x69,0x24,0x0f,0xc7,0x26,0x3a,0x7f,0x5d,0x42,0xa5,
		0x4f,0x17,0x05,0xed,0x1f,0x3b,0xc0,0x01,0x73,0x57,0x4e,0xf2,0x67,0x18,0x49,0x12,
		0xc5,0xc5,0x78,0x59,0x4d,0x2d,0x6a,0x37,0x4f,0x7f,0x1d,0xbe,0x17,0x90,0xad,0x8b,
		0xa7,0x85,0x96,0xa7,0x9d,0xc0,0x28,0x50,0xf6,0x15,0xf9,0xd2,0x2c,0xb8,0x54,0x93,
		0x1b,0xa3,0x57,0xc3,0xf8,0x7c,0x75,0xef,0x64,0xb6,0xbd,0x7b,0xd4,0x8d,0xb2,0x76,
		0x04,0x68,0x23,0x09,0xfe,0xd5,0xef,0x36,0x9b,0x96,0x89,0x91,0xb9,0x08,0x68,0x3f,
		0xca,0x39,0x8b,0x98,0x54,0x2e,0x27,0x0d,0x8f,0x5b,0x2e,0x9a,0x20,0xed,0xf8,0x2e,
		0xf5,0x81,0xf4,0xcb,0x61,0xd7,0x22,0xb9,0x34,0x1e,0xe6,0x6d,0x0f,0xe9,0x52,0x23,
		0x9b,0xa2,0x80,0x0b,0x50,0x47,0xd1,0x26,0xea,0xb8,0x32,0x72,0x82,0x27,0xfb,0x61,
		0x0d,0x4f,0xfa,0x99,0x88,0xad,0x72,0x36,0x33,0x26,
	};
	static long size_tuc3F00DF00503C = sizeof(tuc3F00DF00503C);

	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	//-------------------------------------
	// use correct parameters
	// test card 1 version 2
	//-------------------------------------
	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V2).getName();

	status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
	CHECK_EQUAL(BEID_OK,status.general);

	::BEID_Pins pins;
	status = BEID_GetPINs(&pins);
	CHECK_EQUAL(BEID_OK,status.general);
	CHECK_EQUAL(pins.pinsLength,2);

	for (long pinIdx = 0; pinIdx < pins.pinsLength; pinIdx++)
	{
		::BEID_Pin	pin;
		pin.pinType   = pins.pins[pinIdx].pinType;
		pin.id		  = pins.pins[pinIdx].id;
		pin.usageCode = pins.pins[pinIdx].usageCode;
		pin.shortUsage = NULL;
		pin.longUsage  = NULL;

		//-------------------------------------
		// read file BEID_FILE_ID and test
		//-------------------------------------
		BEID_Bytes outData;
		outData.length		 = size_tuc3F00DF014031;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_ID, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF014031);
		CHECK_ARRAY_EQUAL(tuc3F00DF014031,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_ID_SIGN and test
		//-------------------------------------

		outData.length		 = size_tuc3F00DF014032;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_ID_SIGN, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF014032);
		CHECK_ARRAY_EQUAL(tuc3F00DF014032,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_ADDRESS and test
		//-------------------------------------

		outData.length		 = size_tuc3F00DF014033;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_ADDRESS, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF014033);
		CHECK_ARRAY_EQUAL(tuc3F00DF014033,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_ADDRESS_SIGN and test
		//-------------------------------------

		outData.length		 = size_tuc3F00DF014034;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_ADDRESS_SIGN, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF014034);
		CHECK_ARRAY_EQUAL(tuc3F00DF014034,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_PHOTO and test
		//-------------------------------------

		outData.length		 = size_tuc3F00DF014035;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_PHOTO, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF014035);
		CHECK_ARRAY_EQUAL(tuc3F00DF014035,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_TOKENINFO and test
		//-------------------------------------

		outData.length		 = size_tuc3F00DF005032;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_TOKENINFO, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF005032);
		CHECK_ARRAY_EQUAL(tuc3F00DF005032,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_RRN and test
		//-------------------------------------

		outData.length		 = size_tuc3F00DF00503C;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_RRN, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF00503C);
		CHECK_ARRAY_EQUAL(tuc3F00DF00503C,outData.data,outData.length);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;
	}

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);
}

TEST(BEIDWriteFile)
{
	unsigned char tuc3F00DF014039[] = {
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x00,0x00,0x00,0x00,
	};
	long size_tuc3F00DF014039 = sizeof(tuc3F00DF014039);

	unsigned char new_tuc3F00DF014039[] = {
		0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,
		0x01,0x00,0x00,0x00,
	};
	long size_new_tuc3F00DF014039 = sizeof(new_tuc3F00DF014039);

	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	//-------------------------------------
	// use correct parameters
	// test card 1 version 1 (==IDX_READER_TEST_CARD_V1)
	//-------------------------------------
	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V1).getName();

	status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
	CHECK_EQUAL(BEID_OK,status.general);

	//-------------------------------------
	// get the PINs from the card
	//-------------------------------------
	::BEID_Pins pins;
	status = BEID_GetPINs(&pins);
	CHECK_EQUAL(BEID_OK,status.general);
	CHECK_EQUAL(1,pins.pinsLength);

	// for all the pins...
	for (long pinIdx=0;pinIdx<pins.pinsLength;pinIdx++)
	{
		::BEID_Pin  pin;
		pin.pinType = pins.pins[pinIdx].pinType;
		pin.id		= pins.pins[pinIdx].id;
		pin.usageCode= pins.pins[pinIdx].usageCode;
		long		triesLeft = pins.pins[pinIdx].triesLeft;

		//-------------------------------------
		// write file BEID_FILE_ID and test
		//-------------------------------------

		BEID_Bytes outData;
		outData.length		 = size_tuc3F00DF014039;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		// first read the file and check if OK
		status = BEID_ReadFile(&BEID_FILE_PREF, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_tuc3F00DF014039);
		CHECK_ARRAY_EQUAL(tuc3F00DF014039,outData.data,outData.length);

		BEID_Bytes old_fileData;
		old_fileData.data   = (unsigned char *)tuc3F00DF014039;
		old_fileData.length = (unsigned long)size_tuc3F00DF014039;

		BEID_Bytes new_fileData;
		new_fileData.data   = (unsigned char *)new_tuc3F00DF014039;
		new_fileData.length = (unsigned long)size_new_tuc3F00DF014039;

		// WRITE the file and check if OK
		status = BEID_WriteFile(&BEID_FILE_PREF,&new_fileData,&pin);
		CHECK_EQUAL(BEID_OK,status.general);

		// read the file that is modified and check if OK
		outData.length		 = size_new_tuc3F00DF014039;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		status = BEID_ReadFile(&BEID_FILE_PREF, &outData, &pin);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,size_new_tuc3F00DF014039);
		CHECK_ARRAY_EQUAL(new_tuc3F00DF014039,outData.data,outData.length);

		//Replace the file to its initial value
		status = BEID_WriteFile(&BEID_FILE_PREF,&old_fileData,&pin);
		CHECK_EQUAL(BEID_OK,status.general);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;
	}
	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);
}

TEST(BEIDGetRawData)
{
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 
	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	BEID_Status status;

	{
		pszReaderName	= ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V1).getName();
		status			= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		BEID_Raw	rawData;

		status = BEID_GetRawData(&rawData);
		CHECK_EQUAL(BEID_OK,status.general);
	}
	{
		pszReaderName	= ReaderSet.getReaderByNum(IDX_READER_NO_CARD).getName();
		status			= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_E_PCSC,status.general);
		CHECK_EQUAL(SCARD_E_NO_SMARTCARD,status.pcsc); 

		BEID_Raw	rawData;

		status = BEID_GetRawData(&rawData);
		CHECK_EQUAL(status.general,BEID_E_PCSC);
		CHECK_EQUAL(status.pcsc,SCARD_E_NO_SMARTCARD); 
	}
	{
		pszReaderName	= ReaderSet.getReaderByNum(IDX_READER_EID).getName();
		status			= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		BEID_Raw	rawData;

		status = BEID_GetRawData(&rawData);
		CHECK_EQUAL(BEID_OK,status.general);
	}
	{
		pszReaderName	= ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V2).getName();
		status			= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		BEID_Raw	rawData;

		status = BEID_GetRawData(&rawData);
		CHECK_EQUAL(BEID_OK,status.general);
	}
	{
		pszReaderName	= ReaderSet.getReaderByNum(IDX_READER_SIS).getName();
		status			= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		BEID_Raw	rawData;

		status = BEID_GetRawData(&rawData);
		CHECK_EQUAL(BEID_E_UNKNOWN_CARD,status.general);
	}
	{
		pszReaderName	= ReaderSet.getReaderByNum(IDX_READER_REVOKED_CERT).getName();
		status			= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		BEID_Raw	rawData;

		status = BEID_GetRawData(&rawData);
		CHECK_EQUAL(BEID_OK,status.general);
	}

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);

}
TEST(BEIDSetRawData)
{
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 
	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	BEID_Status status;

	{
		pszReaderName	= ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V1).getName();
		status			= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		BEID_Raw	rawData;

		status = BEID_GetRawData(&rawData);
		CHECK_EQUAL(BEID_OK,status.general);

		status			= BEID_Init("VIRTUAL", lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		status = BEID_SetRawData(&rawData);
		CHECK_EQUAL(BEID_OK,status.general);
	}
	{
		pszReaderName	= ReaderSet.getReaderByNum(IDX_READER_EID).getName();
		status			= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		BEID_Raw	rawData;

		status = BEID_GetRawData(&rawData);
		CHECK_EQUAL(BEID_OK,status.general);

		status			= BEID_Init("VIRTUAL", lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		status = BEID_SetRawData(&rawData);
		CHECK_EQUAL(BEID_OK,status.general);
	}
	{
		pszReaderName	= ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V2).getName();
		status			= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		BEID_Raw	rawData;

		status = BEID_GetRawData(&rawData);
		CHECK_EQUAL(BEID_OK,status.general);

		status			= BEID_Init("VIRTUAL", lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		status = BEID_SetRawData(&rawData);
		CHECK_EQUAL(BEID_OK,status.general);
	}

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);

}
TEST(BEIDGetCertificates)
{
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 
	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	BEID_Status status;

	{
		// EF(Cert#2) (authentication)
    unsigned char tuc3F00DF005038[] = {
        0x30,0x82,0x03,0xe1,0x30,0x82,0x02,0xc9,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,0x10,
        0x00,0x00,0x00,0x00,0x00,0xf8,0xa1,0x31,0x5b,0xc7,0x86,0x24,0x4d,0x3b,0xc6,0x30,
        0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x30,0x33,
        0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x13,0x30,
        0x11,0x06,0x03,0x55,0x04,0x03,0x13,0x0a,0x43,0x69,0x74,0x69,0x7a,0x65,0x6e,0x20,
        0x43,0x41,0x31,0x0f,0x30,0x0d,0x06,0x03,0x55,0x04,0x05,0x13,0x06,0x32,0x30,0x30,
        0x36,0x32,0x30,0x30,0x1e,0x17,0x0d,0x30,0x36,0x31,0x30,0x32,0x35,0x31,0x39,0x32,
        0x33,0x30,0x37,0x5a,0x17,0x0d,0x31,0x31,0x31,0x30,0x32,0x34,0x32,0x33,0x35,0x39,
        0x35,0x39,0x5a,0x30,0x6e,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,
        0x42,0x45,0x31,0x27,0x30,0x25,0x06,0x03,0x55,0x04,0x03,0x13,0x1e,0x53,0x74,0x65,
        0x66,0x61,0x6e,0x20,0x48,0x6f,0x65,0x62,0x65,0x6e,0x20,0x28,0x41,0x75,0x74,0x68,
        0x65,0x6e,0x74,0x69,0x63,0x61,0x74,0x69,0x6f,0x6e,0x29,0x31,0x0f,0x30,0x0d,0x06,
        0x03,0x55,0x04,0x04,0x13,0x06,0x48,0x6f,0x65,0x62,0x65,0x6e,0x31,0x0f,0x30,0x0d,
        0x06,0x03,0x55,0x04,0x2a,0x13,0x06,0x53,0x74,0x65,0x66,0x61,0x6e,0x31,0x14,0x30,
        0x12,0x06,0x03,0x55,0x04,0x05,0x13,0x0b,0x37,0x33,0x30,0x34,0x30,0x31,0x30,0x32,
        0x37,0x34,0x39,0x30,0x81,0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
        0x01,0x01,0x01,0x05,0x00,0x03,0x81,0x8d,0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,
        0x85,0x42,0xd3,0xa2,0x97,0x62,0x72,0xc9,0x55,0x30,0xdd,0x68,0x4a,0x49,0x2d,0x54,
        0x6b,0x21,0x09,0x9f,0x0b,0x4f,0x3e,0x6c,0x2b,0x96,0xa5,0x45,0x05,0x7f,0x48,0xd9,
        0x98,0x98,0xd1,0xe1,0x3f,0x5d,0x59,0x7b,0x9a,0x63,0x77,0x93,0x19,0xbf,0x00,0x23,
        0x7c,0x79,0xc1,0x18,0xbe,0x02,0xae,0xd9,0x1a,0xd1,0x1f,0xe1,0x85,0x70,0xe9,0x5a,
        0xdb,0x3b,0x0c,0xe4,0x48,0xc1,0xab,0x90,0x75,0x79,0xeb,0x99,0xbe,0x62,0xf8,0xf6,
        0x4f,0xcb,0x34,0xc6,0xac,0xf5,0xd1,0x02,0xd0,0x70,0x62,0x1e,0x2f,0x5c,0x1e,0xe7,
        0xe4,0x02,0x73,0x5b,0x98,0x4b,0x8f,0xe4,0x37,0xe7,0x2f,0xed,0xa3,0x20,0xfb,0x47,
        0x7c,0xef,0xf0,0xf8,0x8b,0xe1,0xba,0x97,0x39,0x19,0xa1,0x7f,0xd4,0x25,0xcc,0xc1,
        0x02,0x03,0x01,0x00,0x01,0xa3,0x82,0x01,0x38,0x30,0x82,0x01,0x34,0x30,0x44,0x06,
        0x03,0x55,0x1d,0x20,0x04,0x3d,0x30,0x3b,0x30,0x39,0x06,0x07,0x60,0x38,0x01,0x01,
        0x01,0x02,0x02,0x30,0x2e,0x30,0x2c,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x02,
        0x01,0x16,0x20,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x72,0x65,0x70,0x6f,0x73,0x69,
        0x74,0x6f,0x72,0x79,0x2e,0x65,0x69,0x64,0x2e,0x62,0x65,0x6c,0x67,0x69,0x75,0x6d,
        0x2e,0x62,0x65,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,0xff,0x04,0x04,0x03,
        0x02,0x07,0x80,0x30,0x1f,0x06,0x03,0x55,0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,
        0x07,0x93,0x19,0x45,0x26,0xaf,0x58,0x85,0x94,0xbf,0xb1,0xb4,0xb4,0x5e,0x9e,0xc2,
        0x7b,0x86,0xda,0x70,0x30,0x39,0x06,0x03,0x55,0x1d,0x1f,0x04,0x32,0x30,0x30,0x30,
        0x2e,0xa0,0x2c,0xa0,0x2a,0x86,0x28,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x63,0x72,
        0x6c,0x2e,0x65,0x69,0x64,0x2e,0x62,0x65,0x6c,0x67,0x69,0x75,0x6d,0x2e,0x62,0x65,
        0x2f,0x65,0x69,0x64,0x63,0x32,0x30,0x30,0x36,0x32,0x30,0x2e,0x63,0x72,0x6c,0x30,
        0x11,0x06,0x09,0x60,0x86,0x48,0x01,0x86,0xf8,0x42,0x01,0x01,0x04,0x04,0x03,0x02,
        0x05,0xa0,0x30,0x6d,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x01,0x01,0x04,0x61,
        0x30,0x5f,0x30,0x35,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x30,0x02,0x86,0x29,
        0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x63,0x65,0x72,0x74,0x73,0x2e,0x65,0x69,0x64,
        0x2e,0x62,0x65,0x6c,0x67,0x69,0x75,0x6d,0x2e,0x62,0x65,0x2f,0x62,0x65,0x6c,0x67,
        0x69,0x75,0x6d,0x72,0x73,0x2e,0x63,0x72,0x74,0x30,0x26,0x06,0x08,0x2b,0x06,0x01,
        0x05,0x05,0x07,0x30,0x01,0x86,0x1a,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x6f,0x63,
        0x73,0x70,0x2e,0x65,0x69,0x64,0x2e,0x62,0x65,0x6c,0x67,0x69,0x75,0x6d,0x2e,0x62,
        0x65,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,
        0x03,0x82,0x01,0x01,0x00,0x68,0x07,0xbe,0x78,0x15,0xc0,0x32,0x52,0x4e,0x09,0x8f,
        0x87,0x77,0x31,0xf1,0x39,0xcc,0x6f,0x97,0xd4,0xbf,0x4f,0x10,0xd1,0xcb,0x77,0x43,
        0xee,0x4f,0x40,0x56,0x8a,0xbb,0x08,0x2d,0x97,0xac,0xdf,0xe8,0x1d,0xfe,0xc1,0xab,
        0xe6,0xaa,0x74,0x83,0x2f,0xe6,0x5b,0x92,0x51,0xf3,0x83,0x45,0x12,0x5e,0xf7,0xff,
        0x21,0xc9,0x71,0xb1,0xd0,0x97,0x99,0xdc,0x88,0xe4,0x04,0x48,0x72,0x56,0x14,0x4f,
        0xf5,0x23,0xc9,0x42,0x5f,0x7a,0x75,0x45,0x25,0x75,0xa2,0x42,0x3e,0xdf,0xd5,0xaa,
        0x7e,0x81,0xe1,0x2a,0x43,0x17,0x82,0xc9,0xce,0x7e,0xe6,0x97,0x49,0x0f,0x55,0x95,
        0xe8,0x8e,0x51,0x83,0x4f,0x31,0x84,0x50,0x5b,0xda,0x16,0x84,0x32,0x3c,0x7a,0x18,
        0x5d,0x0f,0x29,0xa4,0xbc,0x78,0xa1,0x2d,0x0c,0x78,0xb3,0x57,0x1a,0x38,0x7b,0x93,
        0x1e,0xe2,0xcc,0x25,0xda,0xc3,0x5b,0x4c,0x2c,0x5a,0xe2,0xb6,0x92,0x5c,0x3d,0x91,
        0x37,0x47,0xdd,0xd0,0x48,0xca,0xdd,0x77,0x20,0x35,0xe8,0xda,0x40,0x62,0x1a,0x0f,
        0x79,0x66,0x5b,0xb9,0x3c,0x33,0x66,0x33,0x6e,0xe3,0x74,0x2a,0x8a,0xfd,0xd0,0xbd,
        0xc1,0x05,0x09,0x60,0x33,0x9a,0xd3,0x65,0xfc,0xf9,0x73,0xc0,0xa7,0x86,0xfa,0x1b,
        0x7b,0xa2,0x0b,0x7b,0x70,0xf8,0x69,0xf3,0x6e,0xb8,0xf1,0xc8,0x5e,0x52,0x9c,0x04,
        0x50,0x6e,0xb7,0x46,0xcd,0x18,0xbe,0x2e,0xb1,0x00,0xf7,0x17,0x77,0x29,0xb8,0x3c,
        0x4d,0x9f,0xa6,0xdb,0xf2,0xe3,0xa0,0xc2,0xd6,0xcc,0x0e,0x1e,0x73,0xfe,0x50,0x54,
        0x82,0x20,0xd2,0x4c,0x03,
    };

		unsigned long size_tuc3F00DF005038 = sizeof(tuc3F00DF005038);
		// EF(Cert#3) (non-repudiation)
    unsigned char tuc3F00DF005039[] = {
        0x30,0x82,0x03,0xf6,0x30,0x82,0x02,0xde,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,0x10,
        0x00,0x00,0x00,0x00,0x00,0x30,0xef,0xb4,0x1b,0x4f,0xf6,0xd6,0x60,0xe4,0x63,0x30,
        0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x30,0x33,
        0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x13,0x30,
        0x11,0x06,0x03,0x55,0x04,0x03,0x13,0x0a,0x43,0x69,0x74,0x69,0x7a,0x65,0x6e,0x20,
        0x43,0x41,0x31,0x0f,0x30,0x0d,0x06,0x03,0x55,0x04,0x05,0x13,0x06,0x32,0x30,0x30,
        0x36,0x32,0x30,0x30,0x1e,0x17,0x0d,0x30,0x36,0x31,0x30,0x32,0x35,0x31,0x39,0x32,
        0x33,0x30,0x38,0x5a,0x17,0x0d,0x31,0x31,0x31,0x30,0x32,0x34,0x32,0x33,0x35,0x39,
        0x35,0x39,0x5a,0x30,0x69,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,
        0x42,0x45,0x31,0x22,0x30,0x20,0x06,0x03,0x55,0x04,0x03,0x13,0x19,0x53,0x74,0x65,
        0x66,0x61,0x6e,0x20,0x48,0x6f,0x65,0x62,0x65,0x6e,0x20,0x28,0x53,0x69,0x67,0x6e,
        0x61,0x74,0x75,0x72,0x65,0x29,0x31,0x0f,0x30,0x0d,0x06,0x03,0x55,0x04,0x04,0x13,
        0x06,0x48,0x6f,0x65,0x62,0x65,0x6e,0x31,0x0f,0x30,0x0d,0x06,0x03,0x55,0x04,0x2a,
        0x13,0x06,0x53,0x74,0x65,0x66,0x61,0x6e,0x31,0x14,0x30,0x12,0x06,0x03,0x55,0x04,
        0x05,0x13,0x0b,0x37,0x33,0x30,0x34,0x30,0x31,0x30,0x32,0x37,0x34,0x39,0x30,0x81,
        0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,
        0x03,0x81,0x8d,0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0x8c,0x4c,0x0f,0x82,0xac,
        0x6c,0x92,0xc7,0x60,0xc9,0xe0,0x8a,0x14,0x85,0x32,0xc1,0x8f,0xf0,0x8b,0xec,0x22,
        0x67,0x06,0x2c,0x1c,0x85,0xd8,0xf6,0xa9,0x25,0xee,0x8c,0xa9,0xfb,0x5e,0xc0,0x30,
        0x03,0x51,0x62,0x3a,0xce,0x62,0x39,0x27,0x8d,0x1d,0x85,0xcf,0xd4,0x1c,0x97,0xca,
        0xc9,0xb4,0x30,0xaf,0xb3,0x39,0xea,0x67,0xee,0x34,0x8e,0xae,0x51,0xa8,0xbf,0x41,
        0x4c,0x46,0xbd,0x1a,0x77,0x14,0x48,0x30,0x34,0x1d,0x47,0x52,0x26,0xa8,0x7a,0xc8,
        0x7b,0xf0,0x78,0x3f,0x21,0x1e,0x34,0xe4,0x91,0x93,0x67,0x94,0x84,0x59,0xd6,0xe2,
        0x19,0x04,0xbe,0x33,0xc0,0x66,0xcd,0xf7,0x22,0xf1,0xd1,0x41,0x11,0xa1,0x59,0x0a,
        0x2a,0x05,0x6c,0x5b,0x4b,0xc8,0x7f,0x1e,0xf1,0xb2,0xab,0x02,0x03,0x01,0x00,0x01,
        0xa3,0x82,0x01,0x52,0x30,0x82,0x01,0x4e,0x30,0x44,0x06,0x03,0x55,0x1d,0x20,0x04,
        0x3d,0x30,0x3b,0x30,0x39,0x06,0x07,0x60,0x38,0x01,0x01,0x01,0x02,0x01,0x30,0x2e,
        0x30,0x2c,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x02,0x01,0x16,0x20,0x68,0x74,
        0x74,0x70,0x3a,0x2f,0x2f,0x72,0x65,0x70,0x6f,0x73,0x69,0x74,0x6f,0x72,0x79,0x2e,
        0x65,0x69,0x64,0x2e,0x62,0x65,0x6c,0x67,0x69,0x75,0x6d,0x2e,0x62,0x65,0x30,0x0e,
        0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,0xff,0x04,0x04,0x03,0x02,0x06,0x40,0x30,0x1f,
        0x06,0x03,0x55,0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0x07,0x93,0x19,0x45,0x26,
        0xaf,0x58,0x85,0x94,0xbf,0xb1,0xb4,0xb4,0x5e,0x9e,0xc2,0x7b,0x86,0xda,0x70,0x30,
        0x39,0x06,0x03,0x55,0x1d,0x1f,0x04,0x32,0x30,0x30,0x30,0x2e,0xa0,0x2c,0xa0,0x2a,
        0x86,0x28,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x63,0x72,0x6c,0x2e,0x65,0x69,0x64,
        0x2e,0x62,0x65,0x6c,0x67,0x69,0x75,0x6d,0x2e,0x62,0x65,0x2f,0x65,0x69,0x64,0x63,
        0x32,0x30,0x30,0x36,0x32,0x30,0x2e,0x63,0x72,0x6c,0x30,0x11,0x06,0x09,0x60,0x86,
        0x48,0x01,0x86,0xf8,0x42,0x01,0x01,0x04,0x04,0x03,0x02,0x05,0x20,0x30,0x6d,0x06,
        0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x01,0x01,0x04,0x61,0x30,0x5f,0x30,0x35,0x06,
        0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x30,0x02,0x86,0x29,0x68,0x74,0x74,0x70,0x3a,
        0x2f,0x2f,0x63,0x65,0x72,0x74,0x73,0x2e,0x65,0x69,0x64,0x2e,0x62,0x65,0x6c,0x67,
        0x69,0x75,0x6d,0x2e,0x62,0x65,0x2f,0x62,0x65,0x6c,0x67,0x69,0x75,0x6d,0x72,0x73,
        0x2e,0x63,0x72,0x74,0x30,0x26,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x30,0x01,
        0x86,0x1a,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x6f,0x63,0x73,0x70,0x2e,0x65,0x69,
        0x64,0x2e,0x62,0x65,0x6c,0x67,0x69,0x75,0x6d,0x2e,0x62,0x65,0x30,0x18,0x06,0x08,
        0x2b,0x06,0x01,0x05,0x05,0x07,0x01,0x03,0x04,0x0c,0x30,0x0a,0x30,0x08,0x06,0x06,
        0x04,0x00,0x8e,0x46,0x01,0x01,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
        0x01,0x01,0x05,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x51,0xac,0x8e,0x4b,0x6a,0x2d,
        0xd8,0x6a,0x85,0x9a,0x09,0x92,0xc4,0x1c,0x34,0x9c,0xe0,0x4c,0x91,0x5e,0x36,0x18,
        0x1a,0x0d,0x0a,0x9e,0x08,0xeb,0xc9,0xd8,0xf4,0x08,0x90,0x05,0x4f,0xa9,0x2d,0xef,
        0xef,0x32,0xf4,0xd3,0x50,0x36,0x1b,0x3f,0x93,0xe1,0xba,0xa5,0x94,0x62,0x26,0x63,
        0xbd,0x2e,0x81,0x3e,0xc8,0x5a,0x88,0x8a,0x84,0x9f,0x01,0x66,0xbe,0x5f,0x03,0x53,
        0xd7,0x40,0x68,0xce,0x13,0xeb,0x42,0x67,0x09,0x11,0xe2,0xc0,0xde,0x64,0x7f,0xf9,
        0x2b,0x72,0x8d,0x9c,0xdb,0x1d,0x1f,0xf9,0xbf,0x63,0x36,0xbc,0xc2,0xc9,0xed,0x2d,
        0xad,0xc2,0xac,0x42,0x38,0xb3,0x5a,0xe1,0x3c,0x34,0x06,0x93,0x4b,0xc0,0x9e,0x4c,
        0xc6,0xb8,0xc7,0x71,0x55,0x37,0x8f,0x8d,0xf8,0x74,0x31,0x09,0x1a,0x77,0x24,0x50,
        0x28,0xc2,0xd6,0x62,0x73,0x63,0xb9,0xc5,0x96,0xdc,0xa9,0x25,0xe9,0xdb,0x69,0xef,
        0xbb,0xaa,0x57,0x12,0xa9,0x01,0x31,0x38,0xbd,0x58,0xa4,0x24,0xfa,0x6d,0xad,0x7f,
        0x0d,0xbd,0x5d,0x14,0xba,0x64,0x14,0xbb,0x0a,0x9d,0x98,0x2a,0x00,0x07,0x11,0x17,
        0xdf,0x24,0x21,0x3f,0x05,0xb5,0xfa,0x0a,0xc7,0x73,0x89,0xd4,0x53,0x69,0x11,0x91,
        0xc0,0x7c,0x2c,0x17,0x78,0x10,0x2c,0xa5,0xfd,0x00,0xbf,0x6a,0x19,0xe5,0xbb,0xda,
        0x89,0xf1,0xb1,0x16,0xc4,0x60,0xd0,0x58,0x62,0xf1,0x8b,0xfd,0x0b,0xf6,0x07,0x59,
        0x1b,0x45,0xc8,0xca,0x57,0x8c,0x59,0x11,0xc7,0xa5,0xf3,0xa6,0x09,0x54,0xb7,0x06,
        0xd3,0xfe,0xc8,0x2b,0x7a,0x9e,0xf3,0xba,0x71,0xaa,
    };
		unsigned long size_tuc3F00DF005039 = sizeof(tuc3F00DF005039);
		// EF(Cert#4) (CA)
    unsigned char tuc3F00DF00503A[] = {
        0x30,0x82,0x03,0xdc,0x30,0x82,0x02,0xc4,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,0x37,
        0x7a,0x2a,0x69,0x18,0x27,0xc6,0xbb,0x56,0x8a,0x3b,0x53,0xb4,0xca,0x09,0x00,0x30,
        0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x30,0x27,
        0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x18,0x30,
        0x16,0x06,0x03,0x55,0x04,0x03,0x13,0x0f,0x42,0x65,0x6c,0x67,0x69,0x75,0x6d,0x20,
        0x52,0x6f,0x6f,0x74,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,0x30,0x35,0x31,0x32,0x30,
        0x31,0x31,0x31,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x31,0x32,0x30,0x38,0x30,0x31,
        0x31,0x30,0x30,0x30,0x30,0x30,0x5a,0x30,0x33,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,
        0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x13,0x30,0x11,0x06,0x03,0x55,0x04,0x03,0x13,
        0x0a,0x43,0x69,0x74,0x69,0x7a,0x65,0x6e,0x20,0x43,0x41,0x31,0x0f,0x30,0x0d,0x06,
        0x03,0x55,0x04,0x05,0x13,0x06,0x32,0x30,0x30,0x36,0x32,0x30,0x30,0x82,0x01,0x22,
        0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,
        0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,0x82,0x01,0x01,0x00,0xaf,0x29,0xad,
        0xf8,0x97,0x18,0xbc,0xea,0x93,0xe4,0xcc,0x8e,0xf3,0xb6,0xfc,0xf5,0x2b,0x36,0x60,
        0x47,0x86,0xa4,0x78,0x93,0x1c,0xac,0xc7,0xcc,0x34,0xbe,0xb1,0xa2,0x95,0x85,0xeb,
        0xfa,0xf9,0x8d,0x4a,0xad,0x4f,0x70,0xeb,0x32,0x1f,0x61,0x34,0x87,0xc8,0xa6,0x03,
        0xf9,0x61,0x62,0xd0,0x80,0x83,0x8b,0xbd,0x62,0x86,0x24,0x49,0x37,0xf3,0x24,0x92,
        0x0e,0xca,0xd5,0x66,0xbc,0x9f,0xb6,0x24,0xf0,0xd9,0x91,0xae,0xfa,0x24,0x74,0xc7,
        0xfd,0xa1,0x71,0x60,0xe0,0xb0,0xfe,0xad,0xaf,0x1d,0xc9,0x10,0x87,0xdb,0x50,0xb8,
        0xaf,0xe5,0x26,0xf5,0x05,0x50,0xaa,0x5c,0xe8,0xf5,0x03,0xa1,0x7b,0x5f,0x96,0xa5,
        0x66,0x91,0xf4,0x73,0x0e,0xe8,0xa6,0x5a,0xb4,0x0b,0xf0,0xf0,0xf9,0xe3,0x60,0xe9,
        0x53,0xc7,0x14,0x95,0xec,0x1b,0xcb,0x22,0xdf,0x75,0x16,0x65,0x7b,0x52,0xf4,0xf0,
        0xa4,0x1d,0x18,0x25,0x17,0xa0,0xd5,0x18,0xf5,0xc8,0x14,0x25,0xe1,0x23,0xa8,0xb0,
        0xe5,0x08,0xf1,0x38,0xd6,0x97,0x2e,0xec,0x1e,0xff,0x5a,0x19,0x23,0xc0,0x53,0xba,
        0xb8,0xdc,0xcf,0x08,0xed,0xe5,0x70,0x56,0x79,0xa7,0x96,0x3e,0x3a,0x8a,0x19,0xf9,
        0x20,0x5f,0x62,0x44,0x1d,0x88,0xf3,0xa4,0xd2,0x88,0x4f,0x48,0xe5,0x1a,0x73,0xcd,
        0x36,0xe0,0x5a,0xb3,0x12,0xc8,0xfd,0xd6,0x4b,0x95,0xac,0x3e,0x85,0x5c,0x15,0x31,
        0x60,0x52,0xa2,0x8f,0xa0,0x28,0x1a,0x6e,0x2c,0x32,0x8f,0x21,0x5c,0xc3,0xa6,0xf6,
        0x55,0xd2,0xaa,0x1d,0xa4,0x98,0xdd,0x2f,0xcd,0xd4,0x76,0x5f,0x91,0x02,0x03,0x01,
        0x00,0x01,0xa3,0x81,0xf7,0x30,0x81,0xf4,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,
        0x01,0xff,0x04,0x04,0x03,0x02,0x01,0x06,0x30,0x12,0x06,0x03,0x55,0x1d,0x13,0x01,
        0x01,0xff,0x04,0x08,0x30,0x06,0x01,0x01,0xff,0x02,0x01,0x00,0x30,0x43,0x06,0x03,
        0x55,0x1d,0x20,0x04,0x3c,0x30,0x3a,0x30,0x38,0x06,0x06,0x60,0x38,0x01,0x01,0x01,
        0x02,0x30,0x2e,0x30,0x2c,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x02,0x01,0x16,
        0x20,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x72,0x65,0x70,0x6f,0x73,0x69,0x74,0x6f,
        0x72,0x79,0x2e,0x65,0x69,0x64,0x2e,0x62,0x65,0x6c,0x67,0x69,0x75,0x6d,0x2e,0x62,
        0x65,0x30,0x1d,0x06,0x03,0x55,0x1d,0x0e,0x04,0x16,0x04,0x14,0x07,0x93,0x19,0x45,
        0x26,0xaf,0x58,0x85,0x94,0xbf,0xb1,0xb4,0xb4,0x5e,0x9e,0xc2,0x7b,0x86,0xda,0x70,
        0x30,0x36,0x06,0x03,0x55,0x1d,0x1f,0x04,0x2f,0x30,0x2d,0x30,0x2b,0xa0,0x29,0xa0,
        0x27,0x86,0x25,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x63,0x72,0x6c,0x2e,0x65,0x69,
        0x64,0x2e,0x62,0x65,0x6c,0x67,0x69,0x75,0x6d,0x2e,0x62,0x65,0x2f,0x62,0x65,0x6c,
        0x67,0x69,0x75,0x6d,0x2e,0x63,0x72,0x6c,0x30,0x11,0x06,0x09,0x60,0x86,0x48,0x01,
        0x86,0xf8,0x42,0x01,0x01,0x04,0x04,0x03,0x02,0x00,0x07,0x30,0x1f,0x06,0x03,0x55,
        0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0x10,0xf0,0x0c,0x56,0x9b,0x61,0xea,0x57,
        0x3a,0xb6,0x35,0x97,0x6d,0x9f,0xdd,0xb9,0x14,0x8e,0xdb,0xe6,0x30,0x0d,0x06,0x09,
        0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x82,0x01,0x01,0x00,
        0x0f,0x20,0x18,0xcc,0x94,0x9d,0xb3,0xf9,0xd5,0xb8,0x28,0xcd,0x64,0xc2,0x8b,0x61,
        0xbc,0x8a,0x34,0xe3,0xac,0x9c,0x4c,0x61,0x42,0x1f,0x23,0x0d,0x12,0x6e,0xcd,0xce,
        0x97,0xa5,0x9a,0xbc,0x3d,0x51,0xbf,0x93,0xb2,0xbe,0x35,0x9a,0x20,0xe1,0x66,0x93,
        0x58,0xac,0x3f,0xa4,0x90,0xd1,0x81,0xfa,0xed,0x63,0x3a,0xc4,0xc4,0x0b,0x81,0x4d,
        0xb0,0xec,0x37,0xb4,0x51,0xc2,0xf9,0xd3,0x96,0x66,0xb4,0x61,0x15,0xee,0x9b,0x34,
        0x4b,0xae,0x3d,0xaa,0x63,0x47,0x83,0xa0,0x7a,0xd4,0x03,0x8e,0x6a,0x56,0x9e,0xb3,
        0x27,0xf9,0x56,0x77,0xb6,0x56,0x3b,0x5d,0x21,0xb8,0x59,0x64,0xef,0x2e,0x38,0xaf,
        0x63,0xce,0xcd,0x00,0xe2,0x92,0x19,0x33,0x86,0x74,0x1e,0x9e,0xf2,0x1c,0x52,0x92,
        0xa5,0xd3,0x52,0x70,0xb1,0x8d,0x65,0x5e,0xe8,0x6f,0x39,0xf7,0xfc,0x86,0x58,0x06,
        0xdd,0xfb,0xcc,0x95,0x2a,0xa0,0x01,0x7b,0xba,0x06,0xc8,0xed,0x6f,0x8a,0x09,0xff,
        0x5d,0x90,0x2d,0x0e,0x52,0xaa,0xc8,0x37,0xc0,0xe8,0x36,0x19,0xcd,0x14,0x98,0xaf,
        0x94,0x22,0xf5,0xa7,0xe7,0x9e,0x07,0xea,0xf0,0x3c,0xfc,0x7f,0x3a,0xcf,0x5e,0x9a,
        0x2c,0x5f,0xd1,0x2c,0x97,0xa5,0x88,0xf7,0xbb,0xa4,0x1f,0xd8,0x8c,0x43,0x48,0x09,
        0x45,0xeb,0xb2,0xdc,0x6b,0x9a,0x2f,0x66,0xc7,0x99,0x7b,0x63,0x5c,0x17,0x3b,0x28,
        0x97,0xab,0x76,0x3d,0x9e,0x4e,0xba,0xd3,0x38,0x15,0x79,0x2f,0x64,0x2e,0x46,0xb9,
        0x25,0xdf,0xd4,0x98,0x79,0x69,0xd1,0x49,0xc7,0x06,0x89,0x72,0x8f,0x0b,0xfc,0xf1,
    };
 		unsigned long size_tuc3F00DF00503A = sizeof(tuc3F00DF00503A);

		// EF(Cert#6) (root)
    unsigned char tuc3F00DF00503B[] = {
        0x30,0x82,0x03,0x94,0x30,0x82,0x02,0x7c,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,0x58,
        0x0b,0x05,0x6c,0x53,0x24,0xdb,0xb2,0x50,0x57,0x18,0x5f,0xf9,0xe5,0xa6,0x50,0x30,
        0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x30,0x27,
        0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x18,0x30,
        0x16,0x06,0x03,0x55,0x04,0x03,0x13,0x0f,0x42,0x65,0x6c,0x67,0x69,0x75,0x6d,0x20,
        0x52,0x6f,0x6f,0x74,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,0x30,0x33,0x30,0x31,0x32,
        0x36,0x32,0x33,0x30,0x30,0x30,0x30,0x5a,0x17,0x0d,0x31,0x34,0x30,0x31,0x32,0x36,
        0x32,0x33,0x30,0x30,0x30,0x30,0x5a,0x30,0x27,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,
        0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x18,0x30,0x16,0x06,0x03,0x55,0x04,0x03,0x13,
        0x0f,0x42,0x65,0x6c,0x67,0x69,0x75,0x6d,0x20,0x52,0x6f,0x6f,0x74,0x20,0x43,0x41,
        0x30,0x82,0x01,0x22,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,
        0x01,0x05,0x00,0x03,0x82,0x01,0x0f,0x00,0x30,0x82,0x01,0x0a,0x02,0x82,0x01,0x01,
        0x00,0xc8,0xa1,0x71,0xe9,0x1c,0x46,0x42,0x79,0x78,0x71,0x6f,0x9d,0xae,0xa9,0xa8,
        0xab,0x28,0xb7,0x4d,0xc7,0x20,0xeb,0x30,0x91,0x5a,0x75,0xf5,0xe2,0xd2,0xcf,0xc8,
        0x4c,0x14,0x98,0x42,0x58,0xad,0xc7,0x11,0xc5,0x40,0x40,0x6a,0x5a,0xf9,0x74,0x12,
        0x27,0x87,0xe9,0x9c,0xe5,0x71,0x4e,0x22,0x2c,0xd1,0x12,0x18,0xaa,0x30,0x5e,0xa2,
        0x21,0xb9,0xd9,0xbb,0xff,0xf6,0x74,0xeb,0x31,0x01,0xe7,0x3b,0x7e,0x58,0x0f,0x91,
        0x16,0x4d,0x76,0x89,0xa8,0x01,0x4f,0xad,0x22,0x66,0x70,0xfa,0x4b,0x1d,0x95,0xc1,
        0x30,0x58,0xea,0xbc,0xd9,0x65,0xd8,0x9a,0xb4,0x88,0xeb,0x49,0x46,0x52,0xdf,0xd2,
        0x53,0x15,0x76,0xcb,0x14,0x5d,0x19,0x49,0xb1,0x6f,0x6a,0xd3,0xd3,0xfd,0xbc,0xc2,
        0x2d,0xec,0x45,0x3f,0x09,0x3f,0x58,0xbe,0xfc,0xd4,0xef,0x00,0x8c,0x81,0x35,0x72,
        0xbf,0xf7,0x18,0xea,0x96,0x62,0x7d,0x2b,0x28,0x7f,0x15,0x6c,0x63,0xd2,0xca,0xca,
        0x7d,0x05,0xac,0xc8,0x6d,0x07,0x6d,0x32,0xbe,0x68,0xb8,0x05,0x40,0xae,0x54,0x98,
        0x56,0x3e,0x66,0xf1,0x30,0xe8,0xef,0xc4,0xab,0x93,0x5e,0x07,0xde,0x32,0x8f,0x12,
        0x74,0xaa,0x5b,0x34,0x23,0x54,0xc0,0xea,0x6c,0xce,0xfe,0x36,0x92,0xa8,0x09,0x17,
        0xea,0xa1,0x2d,0xcf,0x6c,0xe3,0x84,0x1d,0xde,0x87,0x2e,0x33,0x0b,0x3c,0x74,0xe2,
        0x21,0x50,0x38,0x95,0x2e,0x5c,0xe0,0xe5,0xc6,0x31,0xf9,0xdb,0x40,0xfa,0x6a,0xa1,
        0xa4,0x8a,0x93,0x9b,0xa7,0x21,0x06,0x87,0x1d,0x27,0xd3,0xc4,0xa1,0xc9,0x4c,0xb0,
        0x6f,0x02,0x03,0x01,0x00,0x01,0xa3,0x81,0xbb,0x30,0x81,0xb8,0x30,0x0e,0x06,0x03,
        0x55,0x1d,0x0f,0x01,0x01,0xff,0x04,0x04,0x03,0x02,0x01,0x06,0x30,0x0f,0x06,0x03,
        0x55,0x1d,0x13,0x01,0x01,0xff,0x04,0x05,0x30,0x03,0x01,0x01,0xff,0x30,0x42,0x06,
        0x03,0x55,0x1d,0x20,0x04,0x3b,0x30,0x39,0x30,0x37,0x06,0x05,0x60,0x38,0x01,0x01,
        0x01,0x30,0x2e,0x30,0x2c,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x02,0x01,0x16,
        0x20,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x72,0x65,0x70,0x6f,0x73,0x69,0x74,0x6f,
        0x72,0x79,0x2e,0x65,0x69,0x64,0x2e,0x62,0x65,0x6c,0x67,0x69,0x75,0x6d,0x2e,0x62,
        0x65,0x30,0x1d,0x06,0x03,0x55,0x1d,0x0e,0x04,0x16,0x04,0x14,0x10,0xf0,0x0c,0x56,
        0x9b,0x61,0xea,0x57,0x3a,0xb6,0x35,0x97,0x6d,0x9f,0xdd,0xb9,0x14,0x8e,0xdb,0xe6,
        0x30,0x11,0x06,0x09,0x60,0x86,0x48,0x01,0x86,0xf8,0x42,0x01,0x01,0x04,0x04,0x03,
        0x02,0x00,0x07,0x30,0x1f,0x06,0x03,0x55,0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,
        0x10,0xf0,0x0c,0x56,0x9b,0x61,0xea,0x57,0x3a,0xb6,0x35,0x97,0x6d,0x9f,0xdd,0xb9,
        0x14,0x8e,0xdb,0xe6,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,
        0x05,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0xc8,0x6d,0x22,0x51,0x8a,0x61,0xf8,0x0f,
        0x96,0x6e,0xd5,0x20,0xb2,0x81,0xf8,0xc6,0xdc,0xa3,0x16,0x00,0xda,0xcd,0x6a,0xe7,
        0x6b,0x2a,0xfa,0x59,0x48,0xa7,0x4c,0x49,0x37,0xd7,0x73,0xa1,0x6a,0x01,0x65,0x5e,
        0x32,0xbd,0xe7,0x97,0xd3,0xd0,0x2e,0x3c,0x73,0xd3,0x8c,0x7b,0x83,0xef,0xd6,0x42,
        0xc1,0x3f,0xa8,0xa9,0x5d,0x0f,0x37,0xba,0x76,0xd2,0x40,0xbd,0xcc,0x2d,0x3f,0xd3,
        0x44,0x41,0x49,0x9c,0xfd,0x5b,0x29,0xf4,0x02,0x23,0x22,0x5b,0x71,0x1b,0xbf,0x58,
        0xd9,0x28,0x4e,0x2d,0x45,0xf4,0xda,0xe7,0xb5,0x63,0x45,0x44,0x11,0x0d,0x2a,0x7f,
        0x33,0x7f,0x36,0x49,0xb4,0xce,0x6e,0xa9,0x02,0x31,0xae,0x5c,0xfd,0xc8,0x89,0xbf,
        0x42,0x7b,0xd7,0xf1,0x60,0xf2,0xd7,0x87,0xf6,0x57,0x2e,0x7a,0x7e,0x6a,0x13,0x80,
        0x1d,0xdc,0xe3,0xd0,0x63,0x1e,0x3d,0x71,0x31,0xb1,0x60,0xd4,0x9e,0x08,0xca,0xab,
        0xf0,0x94,0xc7,0x48,0x75,0x54,0x81,0xf3,0x1b,0xad,0x77,0x9c,0xe8,0xb2,0x8f,0xdb,
        0x83,0xac,0x8f,0x34,0x6b,0xe8,0xbf,0xc3,0xd9,0xf5,0x43,0xc3,0x64,0x55,0xeb,0x1a,
        0xbd,0x36,0x86,0x36,0xba,0x21,0x8c,0x97,0x1a,0x21,0xd4,0xea,0x2d,0x3b,0xac,0xba,
        0xec,0xa7,0x1d,0xab,0xbe,0xb9,0x4a,0x9b,0x35,0x2f,0x1c,0x5c,0x1d,0x51,0xa7,0x1f,
        0x54,0xed,0x12,0x97,0xff,0xf2,0x6e,0x87,0x7d,0x46,0xc9,0x74,0xd6,0xef,0xeb,0x3d,
        0x7d,0xe6,0x59,0x6e,0x06,0x94,0x04,0xe4,0xa2,0x55,0x87,0x38,0x28,0x6a,0x22,0x5e,
        0xe2,0xbe,0x74,0x12,0xb0,0x04,0x43,0x2a,
    };
		unsigned long size_tuc3F00DF00503B = sizeof(tuc3F00DF00503B);

    // EF(Cert#8) (RN)
    unsigned char tuc3F00DF00503C[] = {
        0x30,0x82,0x03,0x2d,0x30,0x82,0x02,0x15,0xa0,0x03,0x02,0x01,0x02,0x02,0x0b,0x04,
        0x00,0x00,0x00,0x00,0x01,0x07,0xe6,0x0b,0xfc,0xdc,0x30,0x0d,0x06,0x09,0x2a,0x86,
        0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x30,0x27,0x31,0x0b,0x30,0x09,0x06,
        0x03,0x55,0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x18,0x30,0x16,0x06,0x03,0x55,0x04,
        0x03,0x13,0x0f,0x42,0x65,0x6c,0x67,0x69,0x75,0x6d,0x20,0x52,0x6f,0x6f,0x74,0x20,
        0x43,0x41,0x30,0x1e,0x17,0x0d,0x30,0x35,0x31,0x32,0x30,0x31,0x31,0x31,0x30,0x30,
        0x30,0x30,0x5a,0x17,0x0d,0x31,0x32,0x30,0x38,0x30,0x31,0x31,0x30,0x30,0x30,0x30,
        0x30,0x5a,0x30,0x29,0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x42,
        0x45,0x31,0x0c,0x30,0x0a,0x06,0x03,0x55,0x04,0x0a,0x13,0x03,0x52,0x52,0x4e,0x31,
        0x0c,0x30,0x0a,0x06,0x03,0x55,0x04,0x03,0x13,0x03,0x52,0x52,0x4e,0x30,0x81,0x9f,
        0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,0x03,
        0x81,0x8d,0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xe5,0xdc,0x8b,0xbb,0x67,0xfc,
        0xf7,0x73,0x96,0x28,0x23,0xb3,0x42,0x13,0xdf,0x9d,0x31,0x28,0x8b,0x7d,0xed,0x60,
        0xd9,0x11,0xba,0x0c,0x43,0x9c,0xcc,0xe0,0xe6,0x51,0x7d,0xb4,0x71,0xd9,0x00,0xea,
        0x1b,0x5f,0x6f,0x2d,0x17,0xd7,0xb2,0x5a,0x58,0x80,0x3c,0x93,0x70,0x9c,0xa9,0x40,
        0x4b,0x97,0xab,0xd2,0x34,0x17,0xee,0x79,0xcb,0x08,0xdd,0xde,0x12,0x50,0x82,0x24,
        0x73,0xd1,0xf9,0xcb,0x2e,0x4a,0x57,0xc4,0x3a,0x52,0x77,0x97,0x73,0xcc,0x37,0x55,
        0xf0,0x8b,0x8e,0x59,0xaa,0xb7,0x2b,0x11,0xa5,0x06,0x07,0x02,0x32,0xfa,0x42,0x40,
        0x67,0xe0,0x85,0xf1,0xf6,0xed,0xbe,0x62,0xf9,0x24,0x09,0x90,0x90,0x90,0xec,0x3a,
        0x55,0x97,0xbe,0x20,0x8f,0x04,0x87,0x93,0x81,0xcf,0x02,0x03,0x01,0x00,0x01,0xa3,
        0x81,0xdb,0x30,0x81,0xd8,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,0x01,0xff,0x04,
        0x04,0x03,0x02,0x06,0xc0,0x30,0x43,0x06,0x03,0x55,0x1d,0x20,0x04,0x3c,0x30,0x3a,
        0x30,0x38,0x06,0x06,0x60,0x38,0x01,0x01,0x01,0x04,0x30,0x2e,0x30,0x2c,0x06,0x08,
        0x2b,0x06,0x01,0x05,0x05,0x07,0x02,0x01,0x16,0x20,0x68,0x74,0x74,0x70,0x3a,0x2f,
        0x2f,0x72,0x65,0x70,0x6f,0x73,0x69,0x74,0x6f,0x72,0x79,0x2e,0x65,0x69,0x64,0x2e,
        0x62,0x65,0x6c,0x67,0x69,0x75,0x6d,0x2e,0x62,0x65,0x30,0x1d,0x06,0x03,0x55,0x1d,
        0x0e,0x04,0x16,0x04,0x14,0x4f,0xa9,0x6f,0xbc,0xd1,0xe4,0x3f,0xb8,0xc4,0x0e,0x7a,
        0xc2,0x00,0x14,0xcc,0x85,0xf2,0x93,0xb2,0x41,0x30,0x36,0x06,0x03,0x55,0x1d,0x1f,
        0x04,0x2f,0x30,0x2d,0x30,0x2b,0xa0,0x29,0xa0,0x27,0x86,0x25,0x68,0x74,0x74,0x70,
        0x3a,0x2f,0x2f,0x63,0x72,0x6c,0x2e,0x65,0x69,0x64,0x2e,0x62,0x65,0x6c,0x67,0x69,
        0x75,0x6d,0x2e,0x62,0x65,0x2f,0x62,0x65,0x6c,0x67,0x69,0x75,0x6d,0x2e,0x63,0x72,
        0x6c,0x30,0x09,0x06,0x03,0x55,0x1d,0x13,0x04,0x02,0x30,0x00,0x30,0x1f,0x06,0x03,
        0x55,0x1d,0x23,0x04,0x18,0x30,0x16,0x80,0x14,0x10,0xf0,0x0c,0x56,0x9b,0x61,0xea,
        0x57,0x3a,0xb6,0x35,0x97,0x6d,0x9f,0xdd,0xb9,0x14,0x8e,0xdb,0xe6,0x30,0x0d,0x06,
        0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x03,0x82,0x01,0x01,
        0x00,0xa9,0x38,0xb3,0x32,0xf2,0x4d,0xb0,0x13,0x55,0x6c,0x08,0x1b,0x06,0xf4,0xbb,
        0x97,0x4d,0xd0,0xbb,0xe1,0xb1,0x43,0x49,0xf8,0x4f,0x62,0x4f,0xdb,0x25,0x06,0x86,
        0x2c,0x31,0xa3,0x27,0xb4,0x61,0xd0,0x9f,0xd2,0x9a,0x30,0x23,0x28,0x5d,0x5c,0x47,
        0xbe,0xe6,0x3b,0xd4,0x7d,0x3b,0x68,0x35,0xde,0x1a,0xcd,0xd0,0xe0,0x54,0x50,0x61,
        0x42,0xf1,0xc7,0x23,0x29,0x49,0x63,0xb5,0x8b,0x09,0x25,0x87,0x4b,0x3a,0xb3,0x56,
        0xd6,0xb6,0xa2,0xe0,0x47,0xbd,0xc5,0x51,0x68,0xef,0x79,0x65,0x62,0x9b,0x95,0xf2,
        0xf3,0x20,0xdd,0xbc,0x27,0x86,0xf8,0x93,0xda,0x01,0xd1,0x7e,0x7f,0x76,0x5e,0xa1,
        0x10,0x7d,0x07,0xfd,0x27,0x98,0x72,0x79,0x32,0x8b,0xbc,0xe4,0xc9,0x1a,0xfd,0x85,
        0x83,0xf7,0x91,0x09,0xc8,0x74,0xff,0xfa,0x0f,0x2a,0x28,0xbb,0xbc,0x1e,0x85,0x86,
        0xc0,0x4f,0x91,0x16,0xfa,0x1b,0x18,0x84,0xae,0xd5,0x24,0x1a,0x00,0xb9,0x0b,0xae,
        0xe2,0x20,0x1f,0xa7,0x30,0xaf,0xb5,0x85,0x51,0xc9,0x8e,0x07,0x21,0x78,0x31,0x7d,
        0x51,0x97,0x6c,0xcc,0x8d,0xab,0xab,0xfa,0x0a,0x94,0x69,0xfc,0x76,0x56,0x0e,0xd2,
        0xa2,0x14,0x64,0x17,0x01,0x6e,0xbe,0x43,0x2d,0x57,0x0f,0x53,0xc5,0x48,0x52,0x54,
        0x65,0x3b,0x05,0x23,0xaa,0x91,0x58,0x15,0x86,0xdc,0x0f,0x0d,0x8f,0x1b,0x47,0xd0,
        0xe3,0x58,0xb1,0x70,0xcf,0x31,0x7d,0x2c,0x92,0xdd,0xc0,0xda,0xff,0x1e,0xb6,0xa7,
        0xda,0xce,0x86,0xad,0xc4,0x01,0x8b,0x32,0x97,0x81,0xe0,0x40,0x7f,0x3a,0xd5,0x72,
        0x1f,
    };
		unsigned long size_tuc3F00DF00503C = sizeof(tuc3F00DF00503C);

		pszReaderName	= ReaderSet.getReaderByNum(IDX_READER_EID).getName();
		status			= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		BEID_Certif_Check	certs;

		unsigned char*	  check_certificates[]	={tuc3F00DF00503C,	                tuc3F00DF005038,	              tuc3F00DF005039,	                tuc3F00DF00503A,                  tuc3F00DF00503B};
		size_t			  check_certifLengths[]	={size_tuc3F00DF00503C,             size_tuc3F00DF005038,             size_tuc3F00DF005039,             size_tuc3F00DF00503A,             size_tuc3F00DF00503B};
		char*			  check_certifLabels[]	={"RN",                             "Authentication",	              "Signature",                      "CA",                             "Root"};
		int				  check_certifStatus[]	={BEID_CERTSTATUS_CERT_NOT_VALIDATED,BEID_CERTSTATUS_CERT_NOT_VALIDATED,BEID_CERTSTATUS_CERT_NOT_VALIDATED,BEID_CERTSTATUS_CERT_NOT_VALIDATED,BEID_CERTSTATUS_CERT_NOT_VALIDATED};

		status = BEID_GetCertificates(&certs);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(5,certs.certificatesLength);
		for (int certIdx=0;certIdx<certs.certificatesLength;certIdx++)
		{
			CHECK_EQUAL(check_certifLengths[certIdx],certs.certificates[certIdx].certifLength);
			CHECK_ARRAY_EQUAL(check_certificates[certIdx],certs.certificates[certIdx].certif,certs.certificates[certIdx].certifLength);
			CHECK_EQUAL(check_certifLabels[certIdx],certs.certificates[certIdx].certifLabel);
			CHECK_EQUAL(check_certifStatus[certIdx],certs.certificates[certIdx].certifStatus);
		}

	}
 	{
 		pszReaderName	= ReaderSet.getReaderByNum(IDX_READER_NO_CARD).getName();
 		status			= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_E_PCSC,status.general);
		CHECK_EQUAL(SCARD_E_NO_SMARTCARD,status.pcsc); 
 
		BEID_Certif_Check	certs;
 
 		status = BEID_GetCertificates(&certs);
 		CHECK_EQUAL(status.general,BEID_E_PCSC);
 		CHECK_EQUAL(status.pcsc,SCARD_E_NO_SMARTCARD); 
 	}
// 	{
// 		pszReaderName	= ReaderSet.getReaderByNum(IDX_READER_EID).getName();
// 		status			= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
// 		CHECK_EQUAL(BEID_OK,status.general);
// 
// 		BEID_Raw	rawData;
// 
// 		status = BEID_SetRawData(&rawData);
// 		CHECK_EQUAL(BEID_OK,status.general);
// 	}
// 	{
// 		pszReaderName	= ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V2).getName();
// 		status			= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
// 		CHECK_EQUAL(BEID_OK,status.general);
// 
// 		BEID_Raw	rawData;
// 
// 		status = BEID_SetRawData(&rawData);
// 		CHECK_EQUAL(BEID_OK,status.general);
// 	}
	{
		pszReaderName	= ReaderSet.getReaderByNum(IDX_READER_SIS).getName();
		status			= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		BEID_Certif_Check	certs;

		status = BEID_GetCertificates(&certs);
		CHECK_EQUAL(BEID_E_UNKNOWN_CARD,status.general);
	}
// 	{
// 		pszReaderName	= ReaderSet.getReaderByNum(IDX_READER_REVOKED_CERT).getName();
// 		status			= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
// 		CHECK_EQUAL(BEID_OK,status.general);
// 
// 		BEID_Raw	rawData;
// 
// 		status = BEID_SetRawData(&rawData);
// 		CHECK_EQUAL(BEID_OK,status.general);
// 	}

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);
}

TEST(BEIDGetRawFile)
{
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 
	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	BEID_Status status;

	{
		pszReaderName	= ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V1).getName();
		status			= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Bytes raw;
		raw.data		 = NULL;
		raw.length		 = 0;
		raw.neededlength = 0;

		status = BEID_GetRawFile(&raw);
		CHECK_EQUAL(BEID_E_INSUFFICIENT_BUFFER,status.general);
//		CHECK_EQUAL(false,raw.neededlength==0);
	}

	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);
}

TEST(BEIDSetRawFile)
{
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 
	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	BEID_Status status;

	//{
	//	pszReaderName	= ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V1).getName();
	//	status			= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
	//	CHECK_EQUAL(BEID_OK,status.general);

	//	::BEID_Bytes raw;
	//	raw.data		 = NULL;
	//	raw.length		 = 0;
	//	raw.neededlength = 0;

	//	status = BEID_GetRawFile(&raw);
	//	CHECK_EQUAL(BEID_E_INSUFFICIENT_BUFFER,status.general);

	//	raw.data		 = new BYTE[raw.neededlength];
	//	raw.length		 = raw.neededlength;
	//	raw.neededlength = 0;

	//	status = BEID_GetRawFile(&raw);
	//	CHECK_EQUAL(BEID_OK,status.general);

	//	status			= BEID_Init("VIRTUAL", lOCSP, lCRL, &lHandle);
	//	CHECK_EQUAL(BEID_OK,status.general);
	//	status = BEID_SetRawFile(&raw);
	//	CHECK_EQUAL(BEID_OK,status.general);

	//	if(raw.data) delete[] raw.data;
	//}
	{
		pszReaderName	= ReaderSet.getReaderByNum(IDX_READER_EID).getName();
		status			= BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Bytes raw;
		raw.data		 = NULL;
		raw.length		 = 0;
		raw.neededlength = 0;

		status = BEID_GetRawFile(&raw);
		CHECK_EQUAL(BEID_E_INSUFFICIENT_BUFFER,status.general);

		raw.data		 = new BYTE[raw.neededlength];
		raw.length		 = raw.neededlength;
		raw.neededlength = 0;

		status = BEID_GetRawFile(&raw);
		CHECK_EQUAL(BEID_OK,status.general);

		status			= BEID_Init("VIRTUAL", lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		status = BEID_SetRawFile(&raw);
		CHECK_EQUAL(BEID_OK,status.general);

		if(raw.data) delete[] raw.data;
	}
	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);
}
TEST(BEIDReadBinary)
{
	//----------------------------------------------
	// data file that can be read
	// test card 1 version 1
	//----------------------------------------------
	unsigned char tuc3F00DF014031[] = {
		0x01,0x0c,0x35,0x39,0x30,0x34,0x35,0x36,0x37,0x38,0x39,0x30,0x31,0x32,0x02,0x10,
		0x30,0x31,0x32,0x33,0x34,0x35,0x36,0x37,0x38,0x39,0x61,0x62,0x63,0x64,0x65,0x66,
		0x03,0x0a,0x30,0x31,0x2e,0x30,0x31,0x2e,0x32,0x30,0x30,0x37,0x04,0x0a,0x30,0x31,
		0x2e,0x30,0x31,0x2e,0x32,0x30,0x31,0x32,0x05,0x09,0x42,0x72,0x75,0x78,0x65,0x6c,
		0x6c,0x65,0x73,0x06,0x0b,0x38,0x30,0x30,0x31,0x32,0x35,0x31,0x31,0x32,0x36,0x35,
		0x07,0x05,0x6e,0x61,0x6d,0x65,0x30,0x08,0x06,0x67,0x6e,0x61,0x6d,0x65,0x30,0x09,
		0x00,0x0a,0x05,0x42,0x65,0x6c,0x67,0x65,0x0b,0x07,0x49,0x78,0x65,0x6c,0x6c,0x65,
		0x73,0x0c,0x0b,0x32,0x35,0x20,0x4a,0x41,0x4e,0x20,0x31,0x39,0x38,0x30,0x0d,0x01,
		0x4d,0x0e,0x06,0x50,0x72,0x69,0x6e,0x63,0x65,0x0f,0x01,0x31,0x10,0x01,0x30,0x11,
		0x14,0xdc,0x9d,0xb2,0xc8,0x3d,0xc1,0xf1,0xf7,0xd6,0x96,0x4c,0xae,0x35,0xda,0x0c,
		0xb9,0x2e,0x22,0x09,0xf0,
	};
	static long size_tuc3F00DF014031 = sizeof(tuc3F00DF014031);

	unsigned char tuc3F00DF014032[] = {
		0x47,0x4d,0x1d,0xec,0x14,0x22,0x8f,0x84,0xf4,0x12,0xac,0x32,0x89,0x55,0x16,0xda,
		0xbb,0x31,0x5f,0x79,0x5a,0x08,0x40,0x36,0x42,0x9b,0x11,0xf3,0x7e,0x63,0xcc,0xf4,
		0xe6,0x6f,0x85,0xf5,0x3b,0xae,0xcc,0x18,0x46,0x68,0xcd,0x1a,0x79,0xfd,0x81,0x0b,
		0xdb,0x30,0x3a,0x9a,0x0f,0xd4,0xad,0x54,0x67,0xc6,0x64,0xd8,0xbe,0xfb,0x51,0x36,
		0x1c,0xb5,0x99,0x0f,0x39,0x77,0x14,0x1c,0x33,0x80,0x51,0xd6,0xab,0xb3,0xab,0x56,
		0x1f,0xed,0x3d,0x07,0xc6,0x76,0x9b,0xc1,0x94,0xd3,0xdd,0x27,0x87,0x85,0x91,0x1b,
		0x06,0x4f,0xec,0x95,0xd9,0x1a,0x71,0x42,0x28,0x60,0x6b,0x3d,0xde,0x2c,0xa4,0x82,
		0xbb,0x33,0xfb,0xc6,0xe6,0x4e,0x50,0xed,0xce,0xde,0x88,0xcd,0x2c,0x99,0xc5,0x9d,
	};
	static long size_tuc3F00DF014032 = sizeof(tuc3F00DF014032);

	unsigned char tuc3F00DF014033[] = {
		0x01,0x0b,0x56,0x69,0x6c,0x61,0x6c,0x61,0x61,0x6e,0x20,0x31,0x33,0x02,0x04,0x31,
		0x36,0x30,0x31,0x03,0x09,0x52,0x75,0x69,0x73,0x62,0x72,0x6f,0x65,0x6b,
	};
	static long size_tuc3F00DF014033 = sizeof(tuc3F00DF014033);

	unsigned char tuc3F00DF014034[] = {
		0x51,0x44,0x70,0xe9,0xf2,0x5e,0x77,0xa5,0xd5,0xc8,0x7a,0x0d,0xe6,0xce,0x88,0xc0,
		0x5e,0xe1,0xef,0xac,0xe7,0xf0,0x0e,0x30,0x97,0x13,0xd9,0x06,0x1a,0xa5,0xb3,0xb6,
		0xbb,0x00,0xd0,0x17,0x2b,0xb4,0xad,0x98,0xca,0x70,0xf0,0xfe,0xd5,0xb6,0xb1,0xab,
		0x6d,0xc2,0xce,0x4f,0x15,0xac,0xea,0x81,0x3c,0xe9,0x38,0x85,0xc9,0xcb,0xe3,0x5d,
		0x07,0x4e,0xb9,0x94,0x65,0x15,0xab,0xa8,0x51,0x5e,0x03,0x4d,0x9e,0xc0,0xb2,0x00,
		0x29,0xd6,0x1d,0x07,0x8d,0xd0,0xb1,0x3b,0x9d,0xae,0x5d,0x6d,0x1a,0x38,0x75,0xd5,
		0x94,0xe3,0x42,0x43,0x9c,0x42,0x3e,0x2d,0x54,0x63,0xb6,0x51,0x45,0x4e,0x86,0xd0,
		0xcd,0x20,0xf0,0x02,0x93,0x29,0x38,0xf2,0xa3,0xd2,0x82,0x1a,0xf8,0x6d,0x27,0x62,
	};
	static long size_tuc3F00DF014034 = sizeof(tuc3F00DF014034);

	unsigned char tuc3F00DF014035[] = {
		0xff,0xd8,0xff,0xe0,0x00,0x10,0x4a,0x46,0x49,0x46,0x00,0x01,0x02,0x01,0x01,0x2c,
		0x01,0x2c,0x00,0x00,0xff,0xdb,0x00,0x43,0x00,0x1c,0x13,0x15,0x18,0x15,0x11,0x1c,
		0x18,0x16,0x18,0x1f,0x1d,0x1c,0x21,0x29,0x45,0x2d,0x29,0x26,0x26,0x29,0x54,0x3c,
		0x40,0x32,0x45,0x64,0x58,0x69,0x67,0x62,0x58,0x60,0x5f,0x6e,0x7c,0x9e,0x86,0x6e,
		0x75,0x96,0x77,0x5f,0x60,0x8a,0xbb,0x8b,0x96,0xa3,0xa9,0xb1,0xb3,0xb1,0x6b,0x84,
		0xc2,0xd0,0xc1,0xac,0xce,0x9e,0xae,0xb1,0xaa,0xff,0xc0,0x00,0x0b,0x08,0x00,0xc8,
		0x00,0x8c,0x01,0x01,0x11,0x00,0xff,0xc4,0x00,0xd2,0x00,0x00,0x01,0x05,0x01,0x01,
		0x01,0x01,0x01,0x01,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x01,0x02,0x03,0x04,
		0x05,0x06,0x07,0x08,0x09,0x0a,0x0b,0x10,0x00,0x02,0x01,0x03,0x03,0x02,0x04,0x03,
		0x05,0x05,0x04,0x04,0x00,0x00,0x01,0x7d,0x01,0x02,0x03,0x00,0x04,0x11,0x05,0x12,
		0x21,0x31,0x41,0x06,0x13,0x51,0x61,0x07,0x22,0x71,0x14,0x32,0x81,0x91,0xa1,0x08,
		0x23,0x42,0xb1,0xc1,0x15,0x52,0xd1,0xf0,0x24,0x33,0x62,0x72,0x82,0x09,0x0a,0x16,
		0x17,0x18,0x19,0x1a,0x25,0x26,0x27,0x28,0x29,0x2a,0x34,0x35,0x36,0x37,0x38,0x39,
		0x3a,0x43,0x44,0x45,0x46,0x47,0x48,0x49,0x4a,0x53,0x54,0x55,0x56,0x57,0x58,0x59,
		0x5a,0x63,0x64,0x65,0x66,0x67,0x68,0x69,0x6a,0x73,0x74,0x75,0x76,0x77,0x78,0x79,
		0x7a,0x83,0x84,0x85,0x86,0x87,0x88,0x89,0x8a,0x92,0x93,0x94,0x95,0x96,0x97,0x98,
		0x99,0x9a,0xa2,0xa3,0xa4,0xa5,0xa6,0xa7,0xa8,0xa9,0xaa,0xb2,0xb3,0xb4,0xb5,0xb6,
		0xb7,0xb8,0xb9,0xba,0xc2,0xc3,0xc4,0xc5,0xc6,0xc7,0xc8,0xc9,0xca,0xd2,0xd3,0xd4,
		0xd5,0xd6,0xd7,0xd8,0xd9,0xda,0xe1,0xe2,0xe3,0xe4,0xe5,0xe6,0xe7,0xe8,0xe9,0xea,
		0xf1,0xf2,0xf3,0xf4,0xf5,0xf6,0xf7,0xf8,0xf9,0xfa,0xff,0xda,0x00,0x08,0x01,0x01,
		0x00,0x00,0x3f,0x00,0xdf,0xc9,0xf5,0x34,0x64,0xfa,0x9a,0x32,0x7d,0x4d,0x19,0x3e,
		0xa6,0x8c,0x9f,0x53,0x46,0x4f,0xa9,0xa3,0x27,0xd4,0xd2,0x16,0xc0,0xc9,0x6c,0x7e,
		0x35,0x1f,0xda,0xa1,0x1d,0x67,0x4f,0xfb,0xec,0x53,0xd6,0x55,0x7f,0xb9,0x20,0x6f,
		0xa1,0xcd,0x3b,0x27,0xd4,0xd1,0x93,0xea,0x68,0xc9,0xf5,0x34,0x64,0xfa,0x9a,0x32,
		0x7d,0x4d,0x19,0x3e,0xa6,0x8c,0x9f,0x53,0x46,0x4f,0xa9,0xa4,0xa2,0x8a,0x28,0xaa,
		0x73,0x6a,0x31,0x46,0xfb,0x57,0xe6,0xc1,0xe4,0xe7,0x8a,0xab,0x73,0xa9,0x3b,0x8f,
		0xdc,0xfc,0xab,0x8c,0x93,0x9e,0x6b,0x2a,0x4b,0xd2,0xe7,0x25,0x99,0x88,0xee,0xd9,
		0x26,0xa3,0x17,0x32,0x72,0x36,0x82,0x3d,0x0a,0xff,0x00,0xf5,0xaa,0x58,0xe7,0x72,
		0x7e,0x50,0x8c,0x7b,0x00,0x7f,0xc9,0xab,0xd0,0xea,0x32,0x21,0xc1,0xde,0x87,0xb8,
		0x3f,0x30,0xfd,0x6b,0x52,0x0b,0xc8,0xe6,0xc0,0xe8,0x4f,0xa7,0x23,0xff,0x00,0xad,
		0xf8,0xd5,0x8a,0x28,0xa2,0x8a,0x28,0xa2,0x8a,0x46,0x65,0x45,0x2c,0xc4,0x05,0x51,
		0x92,0x4f,0x40,0x2b,0x9b,0xd4,0x35,0x69,0x2e,0x58,0xa4,0x04,0xac,0x59,0xe0,0x63,
		0xaf,0xb9,0xac,0xf5,0xf9,0x9c,0xf9,0xb2,0x1c,0x0e,0xde,0xb4,0x34,0xd6,0xd9,0xc7,
		0x97,0x9c,0x0e,0xb9,0x34,0x79,0xf0,0x63,0xfd,0x58,0xfc,0xa9,0x37,0xc2,0xdd,0x13,
		0x6f,0xb8,0x6c,0x7f,0x3a,0x9d,0x77,0x60,0x79,0x6e,0x19,0x4f,0x69,0x3f,0xc4,0x54,
		0xc9,0x37,0x96,0x42,0x48,0x0c,0x64,0xf4,0x0f,0x82,0xbf,0x81,0xab,0x03,0x83,0xc8,
		0x1c,0x76,0x35,0xa9,0xa7,0xcf,0xbd,0x7c,0xb2,0x78,0x51,0xf2,0x83,0xd4,0x7b,0x55,
		0xda,0x28,0xa2,0x8a,0x28,0xa2,0xb1,0xb5,0xcb,0x96,0x91,0x96,0xce,0x16,0xc7,0xf1,
		0x4a,0x47,0x61,0xd8,0x7f,0x5f,0xca,0xb2,0x7e,0x44,0x52,0xa9,0x8f,0x73,0x55,0x27,
		0xdb,0x9f,0x94,0xf3,0xf5,0xa8,0x83,0x7f,0x7b,0x9a,0x4c,0x7a,0x52,0xab,0x76,0x65,
		0xc8,0xa9,0x51,0x33,0xcc,0x2c,0x43,0x7a,0x67,0x07,0xf3,0xa9,0x96,0xea,0x44,0x1b,
		0x26,0x40,0xeb,0xdd,0x58,0x55,0x88,0x64,0x08,0x99,0xb7,0x6d,0xf1,0x8e,0xa8,0x7a,
		0xaf,0xd3,0xfc,0x2a,0xe4,0x32,0x2b,0x61,0xd4,0xed,0xc1,0xe1,0x87,0x05,0x4d,0x6c,
		0xda,0xdc,0x79,0xf1,0xfc,0xd8,0x12,0x0f,0xbc,0xa3,0xf9,0xd4,0xf4,0x51,0x45,0x14,
		0x53,0x26,0x90,0x43,0x0b,0xca,0xc3,0x21,0x14,0xb6,0x07,0x7c,0x76,0xae,0x62,0xea,
		0x46,0x3b,0x9d,0x8e,0x19,0xdb,0x73,0x9f,0xe8,0x2b,0x36,0x49,0x4b,0x70,0x0e,0x05,
		0x0a,0x85,0xce,0x31,0x9a,0x9d,0x2c,0x9d,0x97,0x27,0x8a,0x63,0x5b,0xb2,0x9f,0xeb,
		0x48,0x21,0xdc,0xa7,0xa0,0x61,0xda,0x98,0x51,0x94,0xf4,0x23,0x15,0x34,0x72,0xab,
		0x0d,0xb3,0x82,0x47,0xf7,0x87,0x51,0x4a,0x55,0xed,0xc8,0x92,0x32,0x1b,0xdc,0x74,
		0x61,0xe8,0x45,0x59,0x86,0x65,0x90,0x87,0x8f,0xe4,0x27,0x86,0x52,0x7a,0xff,0x00,
		0xf5,0xfd,0x3d,0x6b,0x4e,0xd6,0x53,0x1b,0x2c,0x89,0x93,0xb7,0x82,0xbe,0xde,0x95,
		0xb0,0x08,0x20,0x10,0x72,0x0f,0x20,0xd2,0xd1,0x45,0x14,0x55,0x0d,0x5e,0x60,0x96,
		0xeb,0x19,0x07,0x0e,0x77,0x13,0x8e,0x81,0x70,0x7f,0x9e,0x3f,0x5a,0xe5,0xae,0x65,
		0x2e,0xdc,0xfa,0xd3,0x60,0x84,0xca,0xe0,0x0a,0xdc,0xb4,0xb0,0x0a,0x01,0x22,0xb4,
		0x05,0xaa,0xec,0xe9,0xda,0xaa,0x5c,0xd8,0xe3,0x90,0x2b,0x32,0x68,0xb6,0x7b,0x1e,
		0xc7,0xd2,0xab,0x82,0xa3,0x83,0xf4,0xe3,0xb1,0xa8,0xa5,0x5e,0xeb,0x4d,0x8e,0x52,
		0xbc,0x75,0x0d,0xd4,0x52,0x9f,0x91,0xb7,0xaf,0xdd,0x61,0x86,0xad,0x7b,0x09,0xcb,
		0x27,0x27,0x2e,0xbc,0x30,0xfe,0x55,0xb9,0x68,0x73,0x16,0xdc,0xe4,0x0e,0x9f,0x4a,
		0x9e,0x8a,0x28,0xa2,0xb0,0xf5,0xf9,0x71,0x26,0xcd,0xd9,0x01,0x00,0x23,0xd0,0xf2,
		0x4f,0xfe,0xcb,0x5c,0xf0,0xcb,0x35,0x6d,0x69,0x76,0xe0,0x28,0x62,0x3a,0xfb,0x56,
		0xd2,0x2f,0x4e,0x95,0x30,0xe9,0x41,0x00,0x8e,0x6a,0x9d,0xdd,0x92,0xcc,0x84,0xa8,
		0xc3,0x7f,0x3a,0xe7,0x2f,0x2d,0xde,0x19,0x09,0xda,0x45,0x55,0x0e,0x7a,0x1e,0x94,
		0xd7,0x03,0x39,0x1d,0xe9,0xc8,0xdf,0xc3,0xd8,0x8f,0xce,0xac,0xd8,0xc9,0xe5,0x4e,
		0x37,0x74,0xe8,0x7e,0x95,0xd2,0xd8,0xb9,0x0d,0xb0,0x9e,0x6a,0xf5,0x14,0x51,0x45,
		0x72,0x7a,0xa4,0xbe,0x65,0xc5,0xce,0x58,0x13,0xe6,0x1f,0xc8,0x1c,0x0f,0xd0,0x0a,
		0xcf,0x8c,0x66,0x45,0x1e,0xa7,0x15,0xd2,0xd9,0xae,0xd8,0xc0,0x1c,0x55,0xf5,0xcf,
		0xa5,0x4a,0x29,0xd8,0xa3,0x15,0x4e,0xf6,0xc5,0x6e,0x54,0xe3,0x01,0xbd,0x7d,0x6b,
		0x9a,0xbd,0xb1,0x92,0xd9,0xce,0x54,0xe2,0xaa,0x88,0xcb,0x02,0x54,0xfe,0x14,0xcc,
		0x11,0xd7,0x8a,0x9e,0x32,0x37,0x29,0x3d,0x09,0xda,0x6b,0xa1,0xb3,0x90,0xe6,0x16,
		0x63,0xce,0x02,0xb1,0xf7,0xad,0x7a,0x28,0xa2,0x82,0xc1,0x41,0x66,0x38,0x50,0x32,
		0x7e,0x95,0xc2,0x33,0x92,0xe4,0xb1,0xc9,0x27,0x9f,0x7a,0x7d,0xa2,0xef,0xb9,0x41,
		0xfe,0xd6,0x6b,0xa7,0x84,0x04,0x51,0xb8,0x81,0xc7,0x7a,0x9c,0x4f,0x1a,0xf5,0x3c,
		0x7a,0xd4,0xab,0x2c,0x6d,0xf7,0x5d,0x4f,0xe3,0x52,0x02,0x28,0xcd,0x04,0x81,0xd6,
		0xa1,0x9e,0x38,0xe5,0x8c,0x89,0x30,0x47,0xbd,0x61,0xdd,0xe9,0x91,0x07,0x0d,0x6f,
		0x28,0x27,0xfb,0xbd,0x6a,0x85,0xcd,0xac,0x85,0x0b,0x18,0xc6,0x57,0xba,0x90,0x73,
		0xf8,0x55,0x64,0xf9,0xb3,0x9e,0xe3,0x15,0xb9,0x6e,0xe3,0x64,0x64,0x7f,0x7f,0xfa,
		0x56,0xf2,0x1d,0xc8,0xa7,0xd4,0x66,0x96,0x8a,0x2a,0xa6,0xaa,0xfb,0x34,0xd9,0x88,
		0xea,0x46,0xde,0xb8,0xea,0x70,0x7f,0x4c,0xd7,0x35,0x6d,0xb6,0x58,0x96,0x29,0x23,
		0x4d,0xac,0xa4,0x86,0x1f,0x7b,0x3e,0xb4,0xba,0x4c,0x3b,0xae,0x86,0x46,0x71,0xde,
		0xb6,0xe5,0x75,0x56,0xcd,0x40,0xfa,0x84,0x2a,0x30,0xcb,0xc1,0xef,0xc6,0x2a,0xb9,
		0xbc,0x80,0xb8,0xf2,0x98,0xe4,0xfa,0x73,0x57,0xac,0xef,0x43,0x10,0x37,0x86,0x07,
		0x8c,0xe6,0xb4,0x41,0xc8,0xcd,0x55,0xba,0xb8,0xf2,0xc1,0x19,0x22,0xb2,0xe5,0xba,
		0xe7,0xe6,0x66,0x23,0xbf,0xa0,0xa9,0x12,0xee,0xdd,0x40,0xda,0xd1,0xb1,0xf4,0xde,
		0x09,0xa7,0x49,0x24,0x53,0x29,0xf9,0x70,0x7d,0xeb,0x04,0xaf,0x97,0x3b,0x27,0xa1,
		0xad,0x2b,0x39,0x0f,0x93,0xc9,0xfb,0xae,0x31,0xfe,0x7f,0x1a,0xe9,0x2d,0x8e,0x6d,
		0xd0,0xfe,0x15,0x25,0x14,0x56,0x57,0x88,0xa4,0xd9,0x62,0xab,0xdd,0x9b,0x3f,0x80,
		0x18,0xfe,0xa2,0xb2,0x2c,0x88,0x7b,0x55,0x66,0x1f,0xea,0x8b,0x28,0xfe,0x7f,0xd6,
		0xae,0xe9,0x4a,0x3c,0xc7,0x60,0x31,0x56,0xee,0x2d,0xcc,0x8b,0x90,0x71,0x59,0x46,
		0xc0,0x2b,0xc8,0x25,0x52,0xe1,0xc6,0x03,0x77,0x43,0xf4,0xf4,0xa5,0xb2,0xb4,0x95,
		0xa4,0x41,0x74,0x77,0xc5,0x16,0x4a,0x20,0xf5,0x20,0x0f,0xcb,0x81,0x57,0x52,0xcb,
		0x6b,0x96,0x04,0xf2,0x78,0xc9,0xe7,0xf1,0xad,0x78,0x87,0xee,0x86,0x7a,0xe2,0xa8,
		0xdd,0x41,0xe6,0xcb,0x82,0x71,0x59,0xf2,0xe9,0xa8,0xc8,0xe8,0xea,0xdb,0xf1,0xf2,
		0x3a,0x9e,0x14,0xfd,0x2a,0xbc,0x16,0x52,0x64,0x9b,0xa4,0x32,0x32,0xae,0xc4,0x50,
		0x46,0x31,0xef,0x56,0xac,0x6c,0x1e,0x25,0xc3,0x1e,0x3d,0x3d,0x3e,0x95,0x9f,0xaa,
		0x43,0xe4,0x5e,0x86,0x1f,0x75,0xd7,0xfc,0xff,0x00,0x4a,0x4b,0x47,0xca,0x14,0xf7,
		0x26,0xba,0x6d,0x35,0xf7,0x5b,0x0e,0xb5,0x6e,0x8a,0x2b,0x03,0xc4,0x8d,0xf3,0x80,
		0x1b,0x8d,0xa0,0x63,0xd0,0xe4,0x9f,0xf0,0xaa,0x1a,0x4b,0x06,0x12,0xc0,0xcd,0x8c,
		0xe0,0x8c,0x7e,0xbf,0xd2,0xb5,0x6c,0xd0,0xc7,0x3b,0x29,0xe9,0x8e,0x0f,0xad,0x68,
		0x22,0xe7,0xad,0x23,0xc0,0x0f,0x6a,0x44,0xb7,0x51,0xfc,0x38,0xa7,0xf9,0x63,0x77,
		0xa6,0x2a,0x55,0xe0,0x53,0x08,0xc9,0xc1,0xa4,0x68,0x43,0x0f,0x7f,0x5a,0x6f,0xd9,
		0xc6,0x73,0x4e,0x29,0xb5,0x6b,0x0f,0x5f,0x50,0x22,0x88,0xf1,0x9d,0xdf,0xd2,0xb2,
		0xed,0x9b,0x13,0x2e,0x7b,0x9c,0x57,0x4b,0xa3,0xb6,0x63,0x2b,0xd8,0x7e,0x95,0xa5,
		0x45,0x15,0xcd,0x78,0x83,0x22,0x79,0x33,0xd0,0xba,0xe3,0xfe,0xf9,0x15,0x99,0x04,
		0xa2,0x0b,0x84,0x98,0x03,0x81,0x8c,0x8f,0x62,0x39,0xad,0xc8,0xe5,0x5f,0xb4,0x23,
		0x2b,0x82,0x8c,0x7e,0x52,0x0d,0x69,0x23,0x7c,0xc4,0x54,0xe2,0x8e,0x2a,0x17,0x62,
		0xcc,0x42,0xf4,0x15,0x3a,0xf2,0x2a,0x27,0x38,0x3f,0x2f,0x24,0x53,0xe3,0x70,0xeb,
		0x91,0xf9,0x53,0xce,0x2a,0x29,0x5b,0x0b,0x5c,0xce,0xb7,0x36,0xfb,0x94,0x41,0xd1,
		0x41,0x3f,0xe7,0xf2,0xac,0xf8,0xdb,0x0c,0xa7,0xd0,0xd6,0xfe,0x87,0x3f,0xef,0xca,
		0x9f,0xe2,0x18,0xad,0xda,0x28,0xae,0x6b,0xc4,0x21,0x85,0xc4,0x99,0x3c,0x16,0x52,
		0x07,0xb6,0xd0,0x3f,0xa1,0xac,0x86,0xc1,0x40,0x79,0xe9,0x49,0x0b,0x88,0xe7,0x47,
		0x27,0x85,0x60,0xc7,0xf3,0xae,0xb1,0x5c,0x70,0xd5,0x65,0x1f,0x8a,0x79,0xf9,0x85,
		0x52,0xba,0x8a,0xe5,0x43,0x79,0x2d,0x8c,0xf7,0xe0,0xff,0x00,0x3a,0x8a,0x3b,0xa9,
		0x91,0x42,0x30,0xcb,0x1e,0x0e,0x05,0x3a,0x34,0xbb,0x33,0xee,0xdf,0xf2,0x9e,0xab,
		0xc6,0x3f,0xc6,0xae,0xa2,0x95,0x39,0x14,0xad,0x20,0x15,0x5a,0xe6,0x5f,0x93,0xad,
		0x72,0xd7,0xcf,0xbe,0xee,0x43,0x9c,0xe0,0xe2,0xa2,0x5c,0x71,0xf5,0xe6,0xaf,0xe9,
		0xd2,0x94,0x99,0x48,0x38,0x65,0x39,0x15,0xd7,0xfd,0x3a,0x51,0x45,0x61,0x78,0x96,
		0x03,0xfb,0xb9,0xf9,0x2a,0x46,0xc3,0xec,0x46,0x48,0xfe,0x67,0xf2,0xac,0x40,0x84,
		0xdb,0x67,0xd0,0xd4,0x1d,0xfe,0xb5,0xbf,0xa7,0xdc,0x19,0xad,0x14,0xb1,0xc9,0x1c,
		0x1f,0xad,0x68,0x44,0x4f,0x63,0x53,0x89,0x31,0x4b,0x24,0xd1,0xa2,0xe5,0xdc,0x28,
		0xf7,0x35,0x02,0xdc,0xda,0xb9,0xe2,0x64,0xe3,0xf0,0x34,0xe5,0xbc,0xb5,0xdd,0x81,
		0x32,0xe7,0xdf,0x8a,0x90,0xb8,0x61,0x90,0x47,0xb1,0x1c,0xd4,0x72,0x9f,0x6a,0xce,
		0xbb,0x90,0x24,0x6c,0xc4,0xf0,0xa3,0x35,0xce,0x9c,0x93,0x93,0xde,0x95,0x4e,0x08,
		0xab,0x16,0xe7,0x0e,0xac,0xa7,0x90,0x6b,0xb2,0xb5,0x7f,0x32,0xda,0x36,0xff,0x00,
		0x64,0x0a,0x96,0x8a,0xa3,0xac,0xc5,0xe6,0xe9,0xb2,0x71,0x92,0xa4,0x30,0xfc,0xff,
		0x00,0xc0,0x9a,0xe7,0xfc,0xb0,0x96,0x59,0x3d,0x73,0x59,0xec,0x30,0xc7,0xd8,0xd5,
		0xcd,0x2e,0x63,0x1c,0xcd,0x19,0x3f,0x2b,0x0c,0xfe,0x35,0xb9,0x0c,0x82,0x9f,0x26,
		0x5f,0xa3,0x11,0x54,0xe6,0xb1,0x0c,0xfb,0xdd,0xde,0x5f,0x66,0x34,0x82,0xd6,0xd0,
		0xf5,0x49,0x01,0xf6,0x63,0x4e,0x36,0x56,0xee,0x30,0x91,0x91,0xee,0xc4,0xe6,0xac,
		0x41,0x6c,0x96,0xe0,0x08,0xcb,0x63,0xd3,0x71,0x34,0xf9,0xa4,0x01,0x71,0x58,0x9a,
		0xa4,0xe0,0x81,0x12,0x9e,0x4f,0x2d,0xcd,0x67,0x01,0x4b,0x8c,0xb0,0xa9,0xa1,0x8d,
		0x98,0xe5,0x06,0x70,0x09,0x3f,0x4a,0xec,0x2c,0x10,0x25,0xa2,0x61,0x98,0x86,0x19,
		0xc1,0xe8,0x0f,0x7c,0x55,0x8a,0x29,0x0a,0x86,0x52,0xac,0x32,0x08,0xc1,0x1e,0xa2,
		0xb9,0x8d,0x48,0xa8,0x41,0x1a,0x0e,0x17,0x82,0x7d,0xeb,0x2d,0x86,0x5b,0xeb,0x8a,
		0x20,0x6d,0x92,0xa9,0x15,0xb3,0x1c,0x98,0xe0,0xfe,0x75,0x7a,0x12,0x1d,0x78,0x3c,
		0xd4,0xeb,0x06,0xe1,0x8c,0x8a,0x43,0x69,0x4e,0xfb,0x38,0x41,0x51,0xc8,0xc2,0x31,
		0xcf,0x4a,0xce,0xba,0xb8,0xda,0xa7,0x1c,0x91,0x58,0x45,0x8b,0xbe,0xe6,0xe4,0x93,
		0x4e,0xcf,0x24,0xf6,0xa1,0x3a,0x56,0xae,0x8f,0x0b,0xc8,0xec,0x14,0x91,0xbd,0x76,
		0x9c,0x77,0x04,0xf3,0xfc,0xab,0xa7,0x55,0x0a,0xa1,0x47,0x40,0x31,0x4b,0x45,0x55,
		0xbd,0xb9,0xf2,0xe3,0x28,0x8d,0x87,0xc7,0x27,0xd0,0x57,0x39,0x70,0xdb,0x99,0xff,
		0x00,0x31,0x54,0xdb,0xa2,0x9c,0x7b,0x54,0x47,0x82,0x78,0xad,0x68,0xce,0xe8,0x55,
		0x87,0x71,0x4e,0x8e,0xe1,0xa1,0x39,0x3d,0x2b,0x46,0x1b,0xf8,0xd8,0x73,0x80,0x6a,
		0x71,0x76,0x84,0x75,0x1f,0x9d,0x47,0x25,0xec,0x60,0x7d,0xe1,0xf9,0xd6,0x74,0xd7,
		0x2f,0x70,0xd8,0x40,0x70,0x7b,0xd4,0x53,0xc7,0xb2,0xdd,0x8e,0x0e,0x48,0xac,0x71,
		0x4a,0x4f,0x18,0xa7,0xa9,0xc2,0xf4,0xae,0x83,0xc3,0xd2,0xc6,0xa8,0xc1,0x8e,0x19,
		0xb0,0x01,0xf6,0xff,0x00,0xf5,0xd6,0xe5,0x15,0x4a,0x5b,0x8d,0x89,0x9b,0x83,0xb5,
		0xba,0x84,0x15,0x97,0x35,0xc1,0x6d,0xc4,0x93,0x8e,0xb8,0x3f,0xce,0xb3,0x2e,0x1b,
		0x18,0x3d,0x7a,0xe6,0xa2,0x5f,0x98,0x91,0x9e,0xf9,0xa8,0xd8,0x10,0x2b,0x42,0xc5,
		0xb7,0x43,0xb4,0xf6,0x35,0x2b,0xaf,0x04,0x54,0x0f,0x1f,0xa7,0x5a,0x66,0xd9,0xfa,
		0x29,0xfd,0x6a,0x68,0xad,0x89,0x20,0xbf,0x35,0x76,0x28,0x80,0xc6,0x05,0x45,0x78,
		0x7f,0x74,0xc0,0x7a,0x56,0x29,0x5c,0x73,0x48,0x54,0x83,0x46,0xe2,0x48,0x1e,0xf5,
		0x3c,0x57,0x32,0xc5,0xb5,0x91,0xc8,0x60,0x30,0x3b,0xf1,0xe9,0x5a,0x56,0xda,0xdc,
		0xf1,0x80,0xb2,0x63,0x6f,0xa8,0x15,0xb2,0x97,0x37,0x45,0x41,0x30,0x02,0x0f,0x20,
		0x8e,0x84,0x57,0x3e,0xad,0x31,0x3f,0x3c,0x81,0x7f,0x53,0x51,0x4d,0x2e,0x1f,0xcb,
		0x53,0xc7,0x73,0xea,0x6a,0xac,0xaf,0xbd,0xcf,0xa0,0xa6,0x2b,0xe1,0xf3,0xda,0xa4,
		0x73,0xb9,0x73,0xde,0xac,0x58,0x3e,0x19,0x97,0xd7,0x15,0xa1,0xd4,0x50,0x17,0x3d,
		0x45,0x48,0xb1,0x8e,0xf4,0xed,0xa0,0x70,0x3a,0xd3,0x88,0xc0,0xcd,0x57,0x9f,0x95,
		0xc5,0x65,0xc8,0x98,0x72,0xb8,0xc7,0x26,0xa4,0xb5,0x84,0x4d,0x87,0x73,0x80,0x2a,
		0x9b,0x10,0x64,0x24,0x70,0xb9,0xe3,0xe9,0x46,0x79,0xa3,0x71,0xe2,0xa6,0x4b,0x99,
		0x95,0x40,0x59,0x5c,0x01,0xd8,0x31,0xa5,0x33,0x9c,0x1d,0xbd,0x4f,0x15,0x13,0x1d,
		0xa0,0x8c,0xfc,0xc7,0xaf,0xb5,0x47,0x40,0x1c,0xd3,0xc1,0x3b,0x6a,0x6b,0x73,0xb6,
		0x40,0x7d,0xeb,0x52,0x36,0xe3,0x1e,0xd5,0x20,0xf5,0xa5,0xc9,0xa7,0x03,0x4a,0x49,
		0x35,0x11,0x5c,0xf6,0xaa,0x97,0x10,0xe0,0x87,0xc7,0x43,0x93,0x55,0x25,0x76,0x84,
		0x3c,0x4a,0x4e,0x09,0xcd,0x56,0xef,0x41,0xe2,0x90,0x52,0xe7,0x14,0xed,0xea,0x83,
		0xe5,0x19,0x3e,0xa6,0x98,0x49,0x34,0x0a,0x72,0x7d,0xef,0xa5,0x03,0xbd,0x5a,0xb7,
		0x4c,0xb6,0x71,0xe9,0x5a,0x08,0x3d,0xbb,0x54,0xa0,0x0c,0xf1,0x4b,0x4f,0x00,0xd3,
		0xd6,0x33,0xdc,0x53,0xd2,0x1c,0x9a,0x65,0xd4,0x4b,0x1c,0x6c,0xcf,0xc0,0x03,0x9a,
		0xe7,0x70,0x25,0xde,0xc4,0x90,0xdc,0x6d,0x18,0xce,0x7b,0x53,0x18,0x15,0x7d,0xa7,
		0xa8,0x38,0x34,0xc3,0x9c,0xd2,0xd1,0x49,0x45,0x14,0xbf,0x4f,0xc6,0x9c,0x80,0xb1,
		0x00,0x75,0x35,0xb3,0x69,0x6d,0xfb,0xbc,0x8e,0xe7,0xf4,0xab,0x9f,0x67,0x38,0xfa,
		0x0a,0x55,0x87,0x27,0x9a,0x77,0x91,0xcd,0x38,0x42,0x41,0xe9,0x53,0xa4,0x27,0xd2,
		0xa5,0x54,0x0a,0x39,0x15,0x8b,0xe2,0x0b,0x9d,0xa1,0x60,0x46,0xe5,0x8e,0x5b,0x07,
		0xb5,0x61,0x06,0x65,0x39,0x04,0x83,0xea,0x0d,0x27,0x42,0x3e,0xb4,0x1e,0xb4,0x51,
		0x9a,0x4a,0x01,0xc1,0xcd,0x2e,0x68,0x03,0xbd,0x28,0x24,0x74,0x35,0xd3,0xe9,0x92,
		0x25,0xc5,0xb2,0xc8,0xa3,0x04,0x7c,0xac,0x3d,0x0d,0x5f,0x08,0x29,0x42,0x73,0x9e,
		0xb4,0xbe,0x58,0xa5,0x08,0x01,0xa7,0xd4,0x53,0xca,0xb1,0x44,0xce,0xc4,0x00,0x06,
		0x49,0x35,0xc7,0x5d,0xdc,0x1b,0x8b,0x87,0x94,0xe7,0xe6,0x3c,0x67,0xb0,0xed,0x50,
		0x0a,0x3b,0x8a,0x28,0xa5,0x15,0x3f,0xf6,0x75,0xf7,0xfc,0xf9,0xdc,0x7f,0xdf,0xa6,
		0xff,0x00,0x0a,0x4f,0xec,0xeb,0xef,0xf9,0xf3,0xb8,0xff,0x00,0xbf,0x4d,0xfe,0x14,
		0xa3,0x4f,0xbd,0xff,0x00,0x9f,0x3b,0x8f,0xfb,0xf4,0xdf,0xe1,0x4b,0xfd,0x9f,0x7d,
		0xff,0x00,0x3e,0x77,0x1f,0xf7,0xe9,0xbf,0xc2,0x81,0xa7,0xdf,0x7f,0xcf,0x9d,0xc7,
		0xfd,0xfa,0x6f,0xf0,0xad,0x2d,0x16,0x2b,0xcb,0x6b,0x83,0x1c,0x96,0xb3,0x88,0xe4,
		0xee,0x63,0x38,0x04,0x56,0xf8,0x46,0xc7,0xdc,0x6f,0xca,0x9c,0x11,0xbf,0xba,0xdf,
		0x95,0x3b,0x6b,0x63,0xee,0x9f,0xca,0x82,0xad,0xfd,0xd3,0xf9,0x53,0x08,0x7e,0x7e,
		0x46,0xfc,0xab,0x27,0x5a,0x17,0x4f,0x00,0x8a,0x08,0x26,0x6d,0xe7,0xe6,0x2a,0x84,
		0xe0,0x56,0x4a,0x68,0xf7,0xb2,0x23,0xb7,0x90,0xea,0x55,0x37,0xe1,0x90,0x8c,0xfb,
		0x0f,0x7f,0x6a,0x87,0xfb,0x3a,0xf7,0xfe,0x7c,0xee,0x3f,0xef,0xdb,0x7f,0x85,0x27,
		0xf6,0x75,0xee,0x7f,0xe3,0xce,0xe3,0xfe,0xfd,0x37,0xf8,0x51,0xfd,0x9d,0x7d,0xff,
		0x00,0x3e,0x77,0x1f,0xf7,0xe9,0xbf,0xc2,0x8f,0xec,0xfb,0xef,0xf9,0xf3,0xb8,0xff,
		0x00,0xbf,0x4d,0xfe,0x14,0xbf,0xd9,0xf7,0xbf,0xf3,0xe7,0x71,0xff,0x00,0x7e,0x9b,
		0xfc,0x2b,0xff,0xd9,
	};
	static long size_tuc3F00DF014035 = sizeof(tuc3F00DF014035);

	unsigned char tuc3F00DF005032[] = {
		0x30,0x27,0x02,0x01,0x00,0x04,0x10,0x53,0x4c,0x49,0x4e,0x33,0x66,0x00,0x29,0x6c,
		0xff,0x26,0x23,0x00,0x00,0x00,0xe1,0x80,0x06,0x42,0x45,0x4c,0x50,0x49,0x43,0x03,
		0x02,0x04,0x30,0x9e,0x04,0x04,0x02,0x00,0x00,
	};
	static long size_tuc3F00DF005032 = sizeof(tuc3F00DF005032);

	unsigned char tuc3F00DF00503C[] = {
		0x30,0x82,0x03,0xd6,0x30,0x82,0x02,0xbe,0xa0,0x03,0x02,0x01,0x02,0x02,0x10,0x01,
		0x00,0x00,0x00,0x00,0x00,0xfb,0x7c,0x59,0x10,0x24,0x00,0x00,0xce,0xc0,0x02,0x30,
		0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x05,0x05,0x00,0x30,0x3b,
		0x31,0x0b,0x30,0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x0e,0x30,
		0x0c,0x06,0x03,0x55,0x04,0x0a,0x13,0x05,0x5a,0x45,0x54,0x45,0x53,0x31,0x1c,0x30,
		0x1a,0x06,0x03,0x55,0x04,0x03,0x13,0x13,0x5a,0x45,0x54,0x45,0x53,0x20,0x47,0x6f,
		0x76,0x65,0x72,0x6e,0x6d,0x65,0x6e,0x74,0x20,0x43,0x41,0x30,0x1e,0x17,0x0d,0x30,
		0x34,0x30,0x33,0x32,0x33,0x31,0x30,0x31,0x30,0x30,0x32,0x5a,0x17,0x0d,0x30,0x39,
		0x30,0x33,0x32,0x33,0x31,0x30,0x31,0x30,0x30,0x32,0x5a,0x30,0x21,0x31,0x0b,0x30,
		0x09,0x06,0x03,0x55,0x04,0x06,0x13,0x02,0x42,0x45,0x31,0x12,0x30,0x10,0x06,0x03,
		0x55,0x04,0x03,0x13,0x09,0x5a,0x45,0x54,0x45,0x53,0x20,0x52,0x4e,0x20,0x30,0x81,
		0x9f,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,0x01,0x01,0x01,0x05,0x00,
		0x03,0x81,0x8d,0x00,0x30,0x81,0x89,0x02,0x81,0x81,0x00,0xb4,0xb8,0x1d,0xed,0xc5,
		0x2d,0x80,0x31,0x42,0x08,0x76,0x44,0xac,0x96,0xde,0x35,0x7c,0xbc,0xc6,0xcd,0xec,
		0x48,0x73,0xfd,0x84,0x0e,0x19,0x90,0x18,0x5a,0x18,0x15,0x18,0x6f,0xca,0xab,0x63,
		0x10,0xea,0x7a,0x0d,0x5d,0x4b,0x74,0x01,0xbe,0xcb,0x18,0x1b,0x97,0x2d,0x7a,0x41,
		0x08,0x04,0x0f,0xbb,0x61,0xf7,0xb7,0xc1,0x8d,0xe1,0x00,0xb5,0xc9,0x8f,0x1b,0x3f,
		0xdf,0x9a,0x1f,0x88,0xe6,0x07,0x80,0xb9,0xf8,0x2b,0xb1,0x69,0xc2,0xed,0x3b,0xfb,
		0xc2,0x47,0xea,0x08,0x2b,0xae,0xd2,0x80,0x43,0xdd,0xc0,0x23,0x1e,0x0a,0x4e,0xe7,
		0x0d,0x3d,0x41,0x97,0xb7,0x20,0x22,0xd9,0x40,0x6b,0x85,0xe2,0x45,0x5a,0x90,0x8f,
		0xbf,0xc7,0x04,0xa8,0xa7,0x69,0x32,0xb0,0x1e,0x7d,0xfb,0x02,0x03,0x01,0x00,0x01,
		0xa3,0x82,0x01,0x72,0x30,0x82,0x01,0x6e,0x30,0x0e,0x06,0x03,0x55,0x1d,0x0f,0x01,
		0x01,0xff,0x04,0x04,0x03,0x02,0x06,0xc0,0x30,0x1d,0x06,0x03,0x55,0x1d,0x0e,0x04,
		0x16,0x04,0x14,0x82,0xce,0x78,0xd3,0x3f,0x6f,0x41,0x41,0xce,0xa1,0xec,0x04,0x89,
		0xcf,0x2d,0xb8,0xcb,0xc8,0x5a,0xc9,0x30,0x42,0x06,0x03,0x55,0x1d,0x20,0x04,0x3b,
		0x30,0x39,0x30,0x37,0x06,0x07,0x60,0x38,0x01,0x01,0x01,0x03,0x01,0x30,0x2c,0x30,
		0x2a,0x06,0x08,0x2b,0x06,0x01,0x05,0x05,0x07,0x02,0x01,0x16,0x1e,0x68,0x74,0x74,
		0x70,0x3a,0x2f,0x2f,0x72,0x65,0x70,0x6f,0x73,0x69,0x74,0x6f,0x72,0x79,0x2e,0x65,
		0x69,0x64,0x2e,0x7a,0x65,0x74,0x65,0x73,0x2e,0x62,0x65,0x30,0x3c,0x06,0x03,0x55,
		0x1d,0x1f,0x04,0x35,0x30,0x33,0x30,0x31,0xa0,0x2f,0xa0,0x2d,0x86,0x2b,0x68,0x74,
		0x74,0x70,0x3a,0x2f,0x2f,0x63,0x72,0x6c,0x2e,0x65,0x69,0x64,0x2e,0x7a,0x65,0x74,
		0x65,0x73,0x2e,0x62,0x65,0x2f,0x62,0x65,0x6c,0x67,0x69,0x75,0x6d,0x2f,0x62,0x65,
		0x6c,0x67,0x69,0x75,0x6d,0x2e,0x63,0x72,0x6c,0x30,0x7c,0x06,0x08,0x2b,0x06,0x01,
		0x05,0x05,0x07,0x01,0x01,0x04,0x70,0x30,0x6e,0x30,0x3f,0x06,0x08,0x2b,0x06,0x01,
		0x05,0x05,0x07,0x30,0x02,0x86,0x33,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x77,0x77,
		0x77,0x2e,0x5a,0x65,0x74,0x65,0x73,0x43,0x61,0x72,0x64,0x73,0x2e,0x62,0x65,0x2f,
		0x64,0x75,0x6d,0x6d,0x79,0x2f,0x63,0x65,0x72,0x74,0x69,0x66,0x63,0x61,0x74,0x65,
		0x73,0x2f,0x74,0x65,0x73,0x74,0x2e,0x63,0x72,0x74,0x30,0x2b,0x06,0x08,0x2b,0x06,
		0x01,0x05,0x05,0x07,0x30,0x01,0x86,0x1f,0x68,0x74,0x74,0x70,0x3a,0x2f,0x2f,0x6f,
		0x63,0x73,0x70,0x2e,0x5a,0x65,0x74,0x65,0x73,0x43,0x61,0x72,0x64,0x73,0x2e,0x62,
		0x65,0x2f,0x64,0x75,0x6d,0x6d,0x79,0x30,0x11,0x06,0x09,0x60,0x86,0x48,0x01,0x86,
		0xf8,0x42,0x01,0x01,0x04,0x04,0x03,0x02,0x04,0xb0,0x30,0x09,0x06,0x03,0x55,0x1d,
		0x13,0x04,0x02,0x30,0x00,0x30,0x1f,0x06,0x03,0x55,0x1d,0x23,0x04,0x18,0x30,0x16,
		0x80,0x14,0x46,0xad,0x98,0xa1,0xc0,0x9a,0x48,0x5e,0x61,0xa0,0xbc,0x7e,0x0b,0x50,
		0x02,0x82,0x03,0xb8,0x4d,0x46,0x30,0x0d,0x06,0x09,0x2a,0x86,0x48,0x86,0xf7,0x0d,
		0x01,0x01,0x05,0x05,0x00,0x03,0x82,0x01,0x01,0x00,0x23,0xab,0x0b,0xf7,0xb9,0x25,
		0x13,0x94,0xd9,0x35,0xb6,0x72,0xc8,0xa2,0x6e,0xe5,0x39,0x6d,0x40,0xcd,0xf6,0x3b,
		0x40,0x17,0x25,0x41,0x3e,0x06,0x39,0x75,0x56,0x9c,0x8d,0x41,0x13,0x09,0xcd,0x44,
		0x70,0xfc,0xb0,0x2d,0x10,0x2d,0x8a,0x8a,0xca,0x18,0xe1,0x47,0x7f,0x63,0xf6,0xbb,
		0x02,0x67,0xf5,0xc9,0xf1,0x10,0x12,0xad,0xeb,0x3b,0xaa,0xbe,0x18,0x67,0x35,0x52,
		0xb6,0xe9,0xe9,0xf7,0x36,0xd8,0x31,0xc2,0x1e,0x29,0x34,0x29,0x48,0x56,0x79,0x54,
		0xdc,0xd9,0xa9,0x8c,0xc4,0x67,0x70,0x97,0xbd,0xa6,0x8e,0xe3,0xb8,0x82,0x9b,0x54,
		0x8f,0xe5,0x92,0x8a,0xea,0x11,0x69,0x24,0x0f,0xc7,0x26,0x3a,0x7f,0x5d,0x42,0xa5,
		0x4f,0x17,0x05,0xed,0x1f,0x3b,0xc0,0x01,0x73,0x57,0x4e,0xf2,0x67,0x18,0x49,0x12,
		0xc5,0xc5,0x78,0x59,0x4d,0x2d,0x6a,0x37,0x4f,0x7f,0x1d,0xbe,0x17,0x90,0xad,0x8b,
		0xa7,0x85,0x96,0xa7,0x9d,0xc0,0x28,0x50,0xf6,0x15,0xf9,0xd2,0x2c,0xb8,0x54,0x93,
		0x1b,0xa3,0x57,0xc3,0xf8,0x7c,0x75,0xef,0x64,0xb6,0xbd,0x7b,0xd4,0x8d,0xb2,0x76,
		0x04,0x68,0x23,0x09,0xfe,0xd5,0xef,0x36,0x9b,0x96,0x89,0x91,0xb9,0x08,0x68,0x3f,
		0xca,0x39,0x8b,0x98,0x54,0x2e,0x27,0x0d,0x8f,0x5b,0x2e,0x9a,0x20,0xed,0xf8,0x2e,
		0xf5,0x81,0xf4,0xcb,0x61,0xd7,0x22,0xb9,0x34,0x1e,0xe6,0x6d,0x0f,0xe9,0x52,0x23,
		0x9b,0xa2,0x80,0x0b,0x50,0x47,0xd1,0x26,0xea,0xb8,0x32,0x72,0x82,0x27,0xfb,0x61,
		0x0d,0x4f,0xfa,0x99,0x88,0xad,0x72,0x36,0x33,0x26,
	};
	static long size_tuc3F00DF00503C = sizeof(tuc3F00DF00503C);

	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	//-------------------------------------
	// use correct parameters
	// test card 1 version 1
	//-------------------------------------
	pszReaderName = ReaderSet.getReaderByNum(IDX_READER_TEST_CARD_V1).getName();

	status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
	CHECK_EQUAL(BEID_OK,status.general);

	{
		//-------------------------------------
		// read file BEID_FILE_ID and test
		// read the complete file
		//-------------------------------------

		int iLenToRead		 = size_tuc3F00DF014031;

		BEID_Bytes outData;
		outData.length		 = iLenToRead;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		int iOffset = 0;
		int iCount  = outData.length;

		status = BEID_ReadBinary(&BEID_FILE_ID, iOffset, iCount, &outData);

		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,iLenToRead);
		CHECK_ARRAY_EQUAL(&tuc3F00DF014031[iOffset],outData.data,iLenToRead);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_ID and test
		// read part of the file
		//-------------------------------------
		iLenToRead			 = 100;
		outData.length		 = iLenToRead;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		iOffset = 0;
		iCount  = outData.length;

		status = BEID_ReadBinary(&BEID_FILE_ID, iOffset, iCount, &outData);

		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,iLenToRead);
		CHECK_ARRAY_EQUAL(&tuc3F00DF014031[iOffset],outData.data,iCount);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;

		//-------------------------------------
		// read file BEID_FILE_ID and test
		// read part of the file
		//-------------------------------------
		iLenToRead			 = 100;
		outData.length		 = iLenToRead;
		outData.data		 = new BYTE [outData.length];
		outData.neededlength = 0;

		iOffset = 50;
		iCount  = outData.length;

		status = BEID_ReadBinary(&BEID_FILE_ID, iOffset, iCount, &outData);

		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_EQUAL(outData.length,iLenToRead);
		CHECK_ARRAY_EQUAL(&tuc3F00DF014031[iOffset],outData.data,iCount);

		delete [] outData.data;
		outData.data = NULL;
		outData.length = 0;
		outData.neededlength = 0;
		//-------------------------------------
		// read file BEID_FILE_ID_SIGN and test
		//-------------------------------------

// 		fileID.data   = (unsigned char *)BEID_FILE_ID_SIGN;
// 		fileID.length = (unsigned long)strlen(BEID_FILE_ID_SIGN);
// 		outData;
// 
// 		outData.length		 = size_tuc3F00DF014032;
// 		outData.data		 = new BYTE [outData.length];
// 		outData.neededlength = 0;
// 
// 		status = BEID_ReadFile(&fileID, &outData, &pin);
// 		CHECK_EQUAL(BEID_OK,status.general);
// 		CHECK_EQUAL(outData.length,size_tuc3F00DF014032);
// 		CHECK_ARRAY_EQUAL(tuc3F00DF014032,outData.data,outData.length);
// 
// 		delete [] outData.data;
// 		outData.data = NULL;
// 		outData.length = 0;
// 		outData.neededlength = 0;
// 
// 		//-------------------------------------
// 		// read file BEID_FILE_ADDRESS and test
// 		//-------------------------------------
// 
// 		fileID.data   = (unsigned char *)BEID_FILE_ADDRESS;
// 		fileID.length = (unsigned long)strlen(BEID_FILE_ADDRESS);
// 		outData;
// 
// 		outData.length		 = size_tuc3F00DF014033;
// 		outData.data		 = new BYTE [outData.length];
// 		outData.neededlength = 0;
// 
// 		status = BEID_ReadFile(&fileID, &outData, &pin);
// 		CHECK_EQUAL(BEID_OK,status.general);
// 		CHECK_EQUAL(outData.length,size_tuc3F00DF014033);
// 		CHECK_ARRAY_EQUAL(tuc3F00DF014033,outData.data,outData.length);
// 
// 		delete [] outData.data;
// 		outData.data = NULL;
// 		outData.length = 0;
// 		outData.neededlength = 0;
// 
// 		//-------------------------------------
// 		// read file BEID_FILE_ADDRESS_SIGN and test
// 		//-------------------------------------
// 
// 		fileID.data   = (unsigned char *)BEID_FILE_ADDRESS_SIGN;
// 		fileID.length = (unsigned long)strlen(BEID_FILE_ADDRESS_SIGN);
// 		outData;
// 
// 		outData.length		 = size_tuc3F00DF014034;
// 		outData.data		 = new BYTE [outData.length];
// 		outData.neededlength = 0;
// 
// 		status = BEID_ReadFile(&fileID, &outData, &pin);
// 		CHECK_EQUAL(BEID_OK,status.general);
// 		CHECK_EQUAL(outData.length,size_tuc3F00DF014034);
// 		CHECK_ARRAY_EQUAL(tuc3F00DF014034,outData.data,outData.length);
// 
// 		delete [] outData.data;
// 		outData.data = NULL;
// 		outData.length = 0;
// 		outData.neededlength = 0;
// 
// 		//-------------------------------------
// 		// read file BEID_FILE_PHOTO and test
// 		//-------------------------------------
// 
// 		fileID.data   = (unsigned char *)BEID_FILE_PHOTO;
// 		fileID.length = (unsigned long)strlen(BEID_FILE_PHOTO);
// 		outData;
// 
// 		outData.length		 = size_tuc3F00DF014035;
// 		outData.data		 = new BYTE [outData.length];
// 		outData.neededlength = 0;
// 
// 		status = BEID_ReadFile(&fileID, &outData, &pin);
// 		CHECK_EQUAL(BEID_OK,status.general);
// 		CHECK_EQUAL(outData.length,size_tuc3F00DF014035);
// 		CHECK_ARRAY_EQUAL(tuc3F00DF014035,outData.data,outData.length);
// 
// 		delete [] outData.data;
// 		outData.data = NULL;
// 		outData.length = 0;
// 		outData.neededlength = 0;
// 
// 		//-------------------------------------
// 		// read file BEID_FILE_TOKENINFO and test
// 		//-------------------------------------
// 
// 		fileID.data   = (unsigned char *)BEID_FILE_TOKENINFO;
// 		fileID.length = (unsigned long)strlen(BEID_FILE_TOKENINFO);
// 		outData;
// 
// 		outData.length		 = size_tuc3F00DF005032;
// 		outData.data		 = new BYTE [outData.length];
// 		outData.neededlength = 0;
// 
// 		status = BEID_ReadFile(&fileID, &outData, &pin);
// 		CHECK_EQUAL(BEID_OK,status.general);
// 		CHECK_EQUAL(outData.length,size_tuc3F00DF005032);
// 		CHECK_ARRAY_EQUAL(tuc3F00DF005032,outData.data,outData.length);
// 
// 		delete [] outData.data;
// 		outData.data = NULL;
// 		outData.length = 0;
// 		outData.neededlength = 0;
// 
// 		//-------------------------------------
// 		// read file BEID_FILE_RRN and test
// 		//-------------------------------------
// 
// 		fileID.data   = (unsigned char *)BEID_FILE_RRN;
// 		fileID.length = (unsigned long)strlen(BEID_FILE_RRN);
// 		outData;
// 
// 		outData.length		 = size_tuc3F00DF00503C;
// 		outData.data		 = new BYTE [outData.length];
// 		outData.neededlength = 0;
// 
// 		status = BEID_ReadFile(&fileID, &outData, &pin);
// 		CHECK_EQUAL(BEID_OK,status.general);
// 		CHECK_EQUAL(outData.length,size_tuc3F00DF00503C);
// 		CHECK_ARRAY_EQUAL(tuc3F00DF00503C,outData.data,outData.length);
// 
// 		delete [] outData.data;
// 		outData.data = NULL;
// 		outData.length = 0;
// 		outData.neededlength = 0;
	}
	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);
}
TEST(BEIDGetCRLUrl)
{
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	{
		//-------------------------------------
		// use correct parameters
		// EID card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_EID).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Certif_Check certs;
		certs.certificatesLength = BEID_MAX_CERT_NUMBER;
		certs.signatureCheck	 = false;

		status = BEID_GetCertificates(&certs);
		CHECK_EQUAL(BEID_OK,status.general);
		

		::BEID_Certif cert;
		BEID_Bytes    url;
		url.data	= new BYTE [BEID_MAX_URL_LENGTH+1];
		url.length  = BEID_MAX_URL_LENGTH;

		//-------------------------------------
		// copy a certificate
		//-------------------------------------
		int	certNr = 1;
		memcpy(cert.certif,certs.certificates[certNr].certif,certs.certificates[certNr].certifLength);
		cert.certifLength = certs.certificates[certNr].certifLength;
		strcpy_s(cert.certifLabel,sizeof(cert.certifLabel),certs.certificates[certNr].certifLabel);
		cert.certifStatus = certs.certificates[certNr].certifStatus;

		//-------------------------------------
		// get the URL
		//-------------------------------------
		status = BEID_GetCRLUrl(&cert, &url);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_ARRAY_EQUAL("http://crl.eid.belgium.be/eidc200620.crl",url.data,(int)strlen((const char*)url.data));

		delete [] url.data;
		url.data   = NULL;
		url.length = 0;

		url.data	= new BYTE [BEID_MAX_URL_LENGTH+1];
		url.length  = BEID_MAX_URL_LENGTH;
		//-------------------------------------
		// give an inexistent certificate
		//-------------------------------------
		memset(cert.certif,0,BEID_MAX_CERT_LEN);
		cert.certifLength = BEID_MAX_CERT_LEN;
		strcpy_s(cert.certifLabel,sizeof(cert.certifLabel),"john_doe");

		status = BEID_GetCRLUrl(&cert, &url);
		CHECK_EQUAL(BEID_E_INTERNAL,status.general);

		delete [] url.data;
		url.data   = NULL;
		url.length = 0;
	}
	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);
}
TEST(BEIDGetOCSPUrl)
{
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	{
		//-------------------------------------
		// use correct parameters
		// EID card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_EID).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		::BEID_Certif_Check certs;
		certs.certificatesLength = BEID_MAX_CERT_NUMBER;
		certs.signatureCheck	 = false;

		status = BEID_GetCertificates(&certs);
		CHECK_EQUAL(BEID_OK,status.general);
		

		::BEID_Certif cert;
		BEID_Bytes    url;
		url.data	= new BYTE [BEID_MAX_URL_LENGTH+1];
		url.length  = BEID_MAX_URL_LENGTH;

		//-------------------------------------
		// copy a certificate
		//-------------------------------------
		int	certNr = 1;
		memcpy(cert.certif,certs.certificates[certNr].certif,certs.certificates[certNr].certifLength);
		cert.certifLength = certs.certificates[certNr].certifLength;
		strcpy_s(cert.certifLabel,sizeof(cert.certifLabel),certs.certificates[certNr].certifLabel);
		cert.certifStatus = certs.certificates[certNr].certifStatus;

		//-------------------------------------
		// get the URL
		//-------------------------------------
		status = BEID_GetOCSPUrl(&cert, &url);
		CHECK_EQUAL(BEID_OK,status.general);
		CHECK_ARRAY_EQUAL("http://ocsp.eid.belgium.be",url.data,(int)strlen((const char*)url.data));

		delete [] url.data;
		url.data   = NULL;
		url.length = 0;

		url.data	= new BYTE [BEID_MAX_URL_LENGTH+1];
		url.length  = BEID_MAX_URL_LENGTH;
		//-------------------------------------
		// give an inexistent certificate
		//-------------------------------------
		memset(cert.certif,0,BEID_MAX_CERT_LEN);
		cert.certifLength = BEID_MAX_CERT_LEN;
		strcpy_s(cert.certifLabel,sizeof(cert.certifLabel),"john_doe");

		status = BEID_GetOCSPUrl(&cert, &url);
		CHECK_EQUAL(BEID_E_INTERNAL,status.general);

		delete [] url.data;
		url.data   = NULL;
		url.length = 0;
	}
	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);
}
TEST(BEIDBeginTransaction)
{
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	{
		//-------------------------------------
		// use correct parameters
		// EID card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_EID).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		status = BEID_BeginTransaction();
		CHECK_EQUAL(BEID_OK,status.general);
	}
	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);
}

TEST(BEIDEndTransaction)
{
	//-------------------------------------
	// get a readerlist and select a reader
	//-------------------------------------
	const char* const*	ppList = ReaderSet.readerList();
	const char*			pszReaderName = NULL; 

	CHECK_EQUAL( NRREADERS, ReaderSet.readerCount());

	BEID_Status status;

	long lOCSP		= BEID_OCSP_CRL_NOT_USED;
	long lCRL		= BEID_OCSP_CRL_NOT_USED;
	long lHandle	= 0;

	{
		//-------------------------------------
		// use correct parameters
		// EID card
		//-------------------------------------
		pszReaderName = ReaderSet.getReaderByNum(IDX_READER_EID).getName();

		status = BEID_Init((char*)pszReaderName, lOCSP, lCRL, &lHandle);
		CHECK_EQUAL(BEID_OK,status.general);

		//-------------------------------------
		// EndTransaction without beginTransaction
		//-------------------------------------
		status = BEID_EndTransaction();
		CHECK_EQUAL(BEID_OK,status.general);

		//-------------------------------------
		// beginTransaction followed by EndTransaction 
		//-------------------------------------
		status = BEID_BeginTransaction();
		CHECK_EQUAL(BEID_OK,status.general);
		status = BEID_EndTransaction();
		CHECK_EQUAL(BEID_OK,status.general);

		//-------------------------------------
		// beginTransaction followed read a file then EndTransaction 
		//-------------------------------------
		status = BEID_BeginTransaction();
		CHECK_EQUAL(BEID_OK,status.general);

		BEID_Address		Address;
		BEID_Certif_Check	CertifCheck;
		status = BEID_GetAddress(&Address, &CertifCheck);
		CHECK_EQUAL(BEID_OK,status.general);

		status = BEID_EndTransaction();
		CHECK_EQUAL(BEID_OK,status.general);
	}
	status = BEID_Exit();
	CHECK_EQUAL(BEID_OK,status.general);
}

TEST(BEIDDefaultReader)
{
    long h = 0;
    BEID_Init("", 0, 0, &h);
    
    BEID_Certif_Check CertifCheck;
    BEID_Status bs = BEID_GetCertificates(&CertifCheck);

    CHECK_EQUAL( BEID_OK,bs.general);

    BEID_Exit();
}
