// Connection_Mono.cpp: implementation of the CConnection_Mono class. 
// 
////////////////////////////////////////////////////////////////////// 
 
#ifdef _WIN32 
#include <Winsock2.h> 
#endif //_WIN32 
 
#ifdef linux 
#include <stdio.h> 
#include <string.h> 
#include <netdb.h> 
#include <sys/socket.h> 
#include <unistd.h> 
#endif // linux 
 
#ifdef __SCO__ 
#include <stddef.h> 
#include <unistd.h> 
#include <string.h> 
#include <netdb.h> 
#include <stdio.h> 
#endif // __SCO__ 
 
#ifdef __HPUX__ 
#include <stddef.h> 
#include <unistd.h> 
#include <string.h> 
#include <netdb.h> 
#include <stdio.h> 
#endif // __HPUX__ 
 
#include "../../Include/Comm/Connection_Mono.h" 
 
#include "../../Include/Comm/MessageStack.h" 
#include "../../Include/Comm/ServerSession.h" 
#include "../../Include/Comm/ConnectionSocket.h" 
#include "../../Include/Comm/Message.h" 
#include "../../Include/Base/StringToObjectCollection.h" 
#include "../../Include/Base/StringContainer.h" 
#include "../../Include/Base/ULongContainer.h" 
#include "../../Include/Base/LoggerManager.h" 

 
#ifdef PASS_SSL 
#include <openssl/err.h> 
#endif // PASS_SSL 
 
////////////////////////////////////////////////////////////////////// 
// Construction/Destruction 
////////////////////////////////////////////////////////////////////// 
 
CConnection_Mono::CConnection_Mono() : CConnection ("Mono") 
{ 
	m_ConnectionSocket = NULL; 
	m_ulMessageIDCounter = 1; 
	m_bServerConnection = FALSE; 
	m_SendStack = new CMessageStack; 
	m_ReceiveStack = new CMessageStack; 
	m_ServerSession = NULL; 
	m_bConnectionBreak = FALSE; 
	m_WaitMessageTimeOut = 60000; // Max 60 Sec default 
	m_bConnectionOpened = FALSE; 
	memset (m_szServerName,0,256); 
	memset (m_szServerAddress,0,20); 
	m_usServerPort = 0; 
 
#ifdef PASS_SSL 
	m_bSecuredConnection = FALSE; 
	m_szCertificateFile = NULL; 
	m_szPrivateKeyFile = NULL; 
	memset (m_szPrivateKeyPassword,0,sizeof(m_szPrivateKeyPassword)); 
	m_SSLctx = NULL; 
	m_SSLssl = NULL; 
	m_SSLmeth = NULL; 
#endif //PASS_SSL 

	PASS_INITLOCK (&lock); 
} 
 
CConnection_Mono::~CConnection_Mono() 
{ 
	// Clean stacks 
	CMessage *message; 
 
	while ((message = m_SendStack->GetMessage()) != NULL) 
	{ 
		delete message; 
	} 
 
	while ((message = m_ReceiveStack->GetMessage()) != NULL) 
	{ 
		delete message; 
	} 
 
	delete m_SendStack; 
	delete m_ReceiveStack; 
 
	// Now stop and delete server session 
	if (m_ServerSession != NULL) 
	{ 
		m_ServerSession->Stop(); 
 
		delete m_ServerSession; 
	} 
 
#ifdef PASS_SSL 
        if (m_szCertificateFile != NULL) 
        { 
        	delete [] m_szCertificateFile; 
        } 
 
        if (m_szPrivateKeyFile != NULL) 
        { 
        	delete [] m_szPrivateKeyFile; 
        } 
 
#endif // PASS_SSL 

	PASS_DESTROYLOCK (&lock); 
} 
 
void CConnection_Mono::SetSocket (CConnectionSocket *aSocket) 
{ 
	m_ConnectionSocket = aSocket; 
} 

CConnectionSocket *CConnection_Mono::GetSocket (void) 
{ 
	return m_ConnectionSocket; 
}
 
CMessageStack *CConnection_Mono::GetReceiveStack (void) 
{ 
	return m_ReceiveStack; 
}  
 
CMessageStack *CConnection_Mono::GetSendStack (void) 
{ 
	return m_SendStack; 
} 
                
BOOL CConnection_Mono::Open (void) 
{ 
#ifdef PASS_SSL 
	if (m_bSecuredConnection == TRUE) 
	{ 
        // Initialize SSL 
   
        /* SSL preliminaries. We keep the certificate and key with the context. */ 
 
        SSL_load_error_strings(); 
        SSLeay_add_ssl_algorithms(); 
        m_SSLmeth = SSLv2_client_method(); 
        m_SSLctx = SSL_CTX_new (m_SSLmeth); 
	} 
#endif // PASS_SSL 
 
	// Open the socket to remote server 
	if (m_bConnectionOpened == TRUE) 
	{ 
		// Connection already opened, just return TRUE 
		return TRUE; 
	} 
 
	if ((strlen (m_szServerAddress) == 0 && strlen (m_szServerName) == 0) || m_usServerPort == 0) 
	{ 
		m_bConnectionOpened = FALSE; 
		return FALSE; 
	} 
 
	int sockfd = 0; 
    struct sockaddr_in servaddr; 
 
#ifdef PASS_EXCEPTION 
	try 
#endif //PASS_EXCEPTION 
	{ 
		sockfd = socket (AF_INET, SOCK_STREAM, 0); 
 
#ifdef _WIN32 
		if (sockfd == INVALID_SOCKET) 
#endif //_WIN32 
#ifdef linux 
		if (sockfd == -1) 
#endif //linux 
#ifdef __SCO__ 
		if (sockfd == -1) 
#endif // __SCO__ 
#ifdef __HPUX__ 
		if (sockfd == -1) 
#endif // __HPUX__ 
		{ 
			m_bConnectionOpened = FALSE; 
			return FALSE; 
		} 
 
		memset (&servaddr,0,sizeof(servaddr)); 
 
		servaddr.sin_family = AF_INET; 
		servaddr.sin_port = htons (m_usServerPort); 
 
		// if server name != 0 convert it into IP address 
		if (strlen (m_szServerName) != 0) 
		{ 
			struct hostent *host_info = gethostbyname (m_szServerName); 
 
			if (host_info == NULL) 
			{ 
				// Can't resolve server address 
				m_bConnectionOpened = FALSE; 
				return FALSE; 
			} 
 
			// Get IP address 
			char *serverIP = host_info->h_addr_list[0]; 
 
			unsigned char serverIP0 = serverIP[0]; 
			unsigned char serverIP1 = serverIP[1]; 
			unsigned char serverIP2 = serverIP[2]; 
			unsigned char serverIP3 = serverIP[3]; 
 
			// copy this IP address to m_szServerAddress 
#ifdef _WIN32 
			_snprintf (m_szServerAddress,20,"%d.%d.%d.%d",serverIP0,serverIP1,serverIP2,serverIP3); 
#endif // _WIN32 
#ifdef linux 
			snprintf (m_szServerAddress,20,"%d.%d.%d.%d",serverIP0,serverIP1,serverIP2,serverIP3); 
#endif // linux 
#ifdef __SCO__
			snprintf (m_szServerAddress,20,"%d.%d.%d.%d",serverIP0,serverIP1,serverIP2,serverIP3); 
#endif // __SCO__ 
#ifdef __HPUX__
			snprintf (m_szServerAddress,20,"%d.%d.%d.%d",serverIP0,serverIP1,serverIP2,serverIP3); 
#endif // __HPUX__ 

		} 
 
		unsigned long inaddr = inet_addr (m_szServerAddress); 
		memcpy (&servaddr.sin_addr,&inaddr,sizeof (unsigned long)); 
 
#ifdef _WIN32 
		if ((connect (sockfd, (struct sockaddr *)&servaddr,sizeof(servaddr))) == SOCKET_ERROR) 
#endif //_WIN32 
#ifdef linux 
		if ((connect (sockfd, (struct sockaddr *)&servaddr,sizeof(servaddr))) == -1) 
#endif //linux 
#ifdef __SCO__ 
		if ((connect (sockfd, (struct sockaddr *)&servaddr,sizeof(servaddr))) == -1) 
#endif // __SCO__ 
#ifdef __HPUX__ 
		if ((connect (sockfd, (struct sockaddr *)&servaddr,sizeof(servaddr))) == -1) 
#endif // __HPUX__ 
        { 
#ifdef _WIN32 
			int error = WSAGetLastError(); 
#endif 
 
        	m_bConnectionOpened = FALSE; 
#ifdef _WIN32 
			closesocket (sockfd); 
#endif //_WIN32 
#ifdef linux 
			close (sockfd); 
#endif //linux 
#ifdef __SCO__ 
			close (sockfd); 
#endif // __SCO__ 
#ifdef __HPUX__ 
			close (sockfd); 
#endif // __HPUX__ 
            
        	return FALSE; 
        } 
	} 
#ifdef PASS_EXCEPTION 
	catch (...) 
	{ 
		// Error 
		return FALSE; 
	} 
#endif //PASS_EXCEPTION 
 
#ifdef PASS_SSL 
	if (m_bSecuredConnection == TRUE) 
	{ 
		int err; 
 
        if (!m_SSLctx) 
        { 
        	m_SSLctx = NULL; 
			return FALSE; 
        } 
 
		if (m_szCertificateFile != NULL) 
		{ 
			if (SSL_CTX_use_certificate_file(m_SSLctx, m_szCertificateFile, SSL_FILETYPE_PEM) <= 0) 
			{ 
        		m_SSLctx = NULL; 
				return FALSE; 
			} 
		} 
 
  		if (m_SSLctx == NULL) 
  		{ 
  			Trace (__FILE__,__LINE__,LOGMASK_COMM_EX,"Can't create new SSL handle : m_SSLctx NULL"); 
  			return FALSE; 
  		} 
 
        m_SSLssl = SSL_new (m_SSLctx); 
        SSL_set_fd (m_SSLssl, sockfd); 
 
	  	Trace (__FILE__,__LINE__,LOGMASK_COMM_EX,"Before SSL_connect"); 
       	err = SSL_connect (m_SSLssl); 
		Trace (__FILE__,__LINE__,LOGMASK_COMM_EX,"After SSL_connect"); 
 
        if (err == -1) 
        { 
			char bufError[256]; 
			memset (bufError,0,256); 
//			char *returnError = ERR_error_string(ERR_get_error(),bufError); 
        	return FALSE; 
        } 
	} 
 
#endif //PASS_SSL 
 
	// Now create connection socket 
	m_ConnectionSocket = new CConnectionSocket (sockfd,servaddr); 
	m_ConnectionSocket->SetSendStack (m_SendStack); 
	m_ConnectionSocket->SetReceiveStack (m_ReceiveStack); 
	m_ConnectionSocket->SetConnection (this); 
	m_ConnectionSocket->SetTickTime (2); 
 
#ifdef PASS_SSL 
	// Connection used as client connection, set client certificate for authentification 
	m_ConnectionSocket->SetCertificateFile (m_szCertificateFile); 
	m_ConnectionSocket->SetSecuredConnection (m_bSecuredConnection); 
	m_ConnectionSocket->SetSSLSSL (m_SSLssl); 
#endif // PASS_SSL 
 
	m_ConnectionSocket->Start(); 
 
	m_bConnectionOpened = TRUE; 
    m_bConnectionBreak = FALSE;

	return TRUE; 
} 
 
BOOL CConnection_Mono::Close (void) 
{ 
	if (m_ConnectionSocket == NULL) 
	{ 
		return FALSE; 
	} 
 
	// Wait for all message be set to the server 
	while (m_SendStack->IsMessageAvailable() == TRUE) 
	{ 
		PASS_MILLISLEEP (10);
	} 
 
	// Set close flag to CConnectionSocket 
	m_ConnectionSocket->Close(); 
 
	m_ConnectionSocket->Stop(); 

	// Now wait for CConnectionSocket to close 
	unsigned long count = 0; 
	BOOL socketClosed = FALSE; 
 
	while (count < 15 && socketClosed == FALSE)
	{ 
		if (m_ConnectionSocket->CanBeDeleted() == TRUE) 
		{ 
			socketClosed = TRUE; 
		}
		PASS_MILLISLEEP (100); 
		count++; 
	} 
 
	if (socketClosed == TRUE) 
	{ 
		// Delete socket 
		delete m_ConnectionSocket; 
		m_ConnectionSocket = NULL; 
	} 
	else 
	{ 
		// Force socket to stop 
		m_ConnectionSocket->Kill(); 
		delete m_ConnectionSocket; 
		m_ConnectionSocket = NULL; 
	} 
 
	m_bConnectionOpened = FALSE; 
 
	return TRUE; 
} 
 
CMessage *CConnection_Mono::_GetMessageForID (unsigned long messageID) 
{ 
#ifdef PASS_EXCEPTION 
	try 
#endif // PASS_EXCEPTION 
	{ 
		return m_ReceiveStack->GetMessageForID (messageID); 
	} 
#ifdef PASS_EXCEPTION 
	catch (...) 
	{ 
		// Exception, try more 
		return NULL; 
	} 
#endif // PASS_EXCEPTION
} 
 
CMessage *CConnection_Mono::GetMessageForID (unsigned long messageID) 
{ 
	PASS_LOCK (&lock); 
 
	CMessage *localMessage = NULL; 
	localMessage = _GetMessageForID (messageID); 
 
	PASS_UNLOCK (&lock); 
	return localMessage; 
} 
 
CMessage *CConnection_Mono::WaitMessageForID (unsigned long messageID) 
{ 
	PASS_LOCK (&lock); 
 
	if (messageID == 0) 
	{ 
		PASS_UNLOCK (&lock); 
		return NULL; 
	} 
 
	CMessage *returnMessage = NULL; 
	unsigned long countTimeOut = 0; 
 
#ifdef PASS_EXCEPTION 
	try  
#endif //PASS_EXCEPTION 
	{ 
		while ((returnMessage = _GetMessageForID(messageID)) == NULL) 
		{ 
			if (countTimeOut++ > m_WaitMessageTimeOut || m_bConnectionBreak == TRUE) 
			{ 
				returnMessage = NULL; 
				break; 
			} 
 
			PASS_MILLISLEEP (1); 
		} 
	} 
#ifdef PASS_EXCEPTION 
	catch (...) 
	{ 
		PASS_UNLOCK (&lock); 
		return NULL; 
	} 
#endif //PASS_EXCEPTION 
 
	PASS_UNLOCK (&lock); 
	return returnMessage; 
} 
 
unsigned long CConnection_Mono::SendMessage (CMessage *aMessage) 
{         
	// Is the message valid ? 
	if (aMessage == NULL) 
	{ 
		// Invalid message return 0; 
		// Set Error 
		return 0; 
	} 
 
	// Is the connection open ? 
	if (m_bConnectionOpened == FALSE && m_bServerConnection == FALSE) 
	{ 
		// Connection not yet opened, return 0 and set error 
		// Set Error 
		return 0; 
	} 
        		 
    if (aMessage->GetMessageID() == 0) 
    { 
		// Message ID not yet set 
        aMessage->SetMessageID (m_ulMessageIDCounter++); 
		m_SendStack->PutMessage (aMessage); 
        return m_ulMessageIDCounter - 1; 
    } 
 
	m_SendStack->PutMessage (aMessage); 
 
 	return aMessage->GetMessageID(); 
} 
 
void CConnection_Mono::SetServerSession (CServerSession *aSession) 
{ 
	m_ServerSession = aSession; 
} 
 
BOOL CConnection_Mono::SetConnectionParam (CStringToObjectCollection *param) 
{ 
	// Needed params : StringContainer - 'ServerAddress' and ULongContainer - 'ServerPort' 
	// No optional params 
 
	if (param == NULL) 
	{ 
		// Set error 
		return FALSE; 
	} 
 
	if (param->GetName() == NULL || param->GetType() == NULL) 
	{ 
		// Set Error 
		return FALSE; 
	} 
 
	if (strcmp (param->GetName(),"StringToObjectCollection") != 0 || strcmp (param->GetType(),"Collection") != 0) 
	{ 
		// Set Error 
		return FALSE; 
	} 
 
	// Now get server address and server port 
 
	// server address need to be a stringContainer 
	CStringContainer *serverAddress = NULL; 
	CStringContainer *serverName = NULL; 
 
	serverAddress = (CStringContainer *)param->Lookup ("ServerAddress"); 
	serverName = (CStringContainer *)param->Lookup ("ServerName"); 
 
	if (serverAddress == NULL && serverName == NULL) 
	{ 
		// Can't find server address 
		// Set error 
		return FALSE; 
	} 
 
	if (serverAddress != NULL) 
	{ 
		if (serverAddress->GetName() == NULL || serverAddress->GetType() == NULL) 
		{ 
			// no name or type 
			//  Set errror 
			return FALSE; 
		} 
 
		if (strcmp(serverAddress->GetName(),"StringContainer") != 0 || strcmp(serverAddress->GetType(),"Container") != 0) 
		{ 
			// Not a string container 
			// Set error 
			return FALSE; 
		} 
 
		strncpy (m_szServerAddress,serverAddress->GetString(),20); 
	} 
 
	if (serverName != NULL) 
	{ 
		if (serverName->GetName() == NULL || serverName->GetType() == NULL) 
		{ 
			// no name or type 
			//  Set errror 
			return FALSE; 
		} 
 
		if (strcmp(serverName->GetName(),"StringContainer") != 0 || strcmp(serverName->GetType(),"Container") != 0) 
		{ 
			// Not a string container 
			// Set error 
			return FALSE; 
		} 
 
		strncpy (m_szServerName,serverName->GetString(),256); 
	} 
 
	// Now get the server port 
 
	// server port needss to be a ULongContainer 
	CLongContainer *serverPort = NULL; 
 
	serverPort = (CLongContainer *)param->Lookup ("ServerPort"); 
 
	if (serverPort == NULL) 
	{ 
		// Can't find server address 
		// Set error 
		return FALSE; 
	} 
 
	if (serverPort->GetName() == NULL || serverPort->GetType() == NULL) 
	{ 
		// no name or type 
		//  Set errror 
		return FALSE; 
	} 
 
	if (strcmp(serverPort->GetName(),"ULongContainer") != 0 || strcmp(serverPort->GetType(),"Container") != 0) 
	{ 
		// Not a string container 
		// Set error 
		return FALSE; 
	} 
 
	m_usServerPort = (unsigned short) serverPort->GetULong(); 
 
	return TRUE; 
} 
 
void CConnection_Mono::Tick (void) 
{ 
// Check if the socket can be deleted 
 
	if (m_bServerConnection == TRUE) 
	{ 
		if (m_bConnectionBreak == TRUE) 
		{ 
			// The connection was broken, stop socket (if needed) 
			if (m_ConnectionSocket->CanBeDeleted() == TRUE) 
			{ 
				unsigned long ulCounter = 0;

				// Wait that receive stack is empty
				while (m_ReceiveStack->IsMessageAvailable() == TRUE && ulCounter < 3000)
				{
					PASS_MILLISLEEP (10);
					ulCounter++;
				}

				// Stop the server session 
				if (m_ServerSession != NULL) 
				{ 
					m_ServerSession->Stop();

					ulCounter = 0;

					while (m_ServerSession->CanBeDeleted() == FALSE && ulCounter < 1000)
					{
						PASS_MILLISLEEP (10);
						ulCounter++;
					}

					delete m_ServerSession; 
					m_ServerSession = NULL; 
				} 
 
				// the socket is ready to be deleted 
				delete m_ConnectionSocket; 
				m_ConnectionSocket = NULL; 
 
#ifdef _WIN32 
				SetEvent(m_hCloseEvent); 
#endif //_WIN32 
 
#ifdef linux 
				pthread_mutex_lock (&m_InternalLock); 
				m_CloseFlag = TRUE; 
				pthread_mutex_unlock (&m_InternalLock); 
#endif //linux 
 
#ifdef __SCO__ 
				mutex_lock (&m_InternalLock); 
				m_CloseFlag = TRUE; 
				mutex_unlock (&m_InternalLock); 
#endif // __SCO__ 
#ifdef __HPUX__ 
				PASS_LOCK (&m_InternalLock); 
				m_CloseFlag = TRUE; 
				PASS_UNLOCK (&m_InternalLock); 
#endif // __HPUX__ 
			} 
		} 
	} 
} 
 
#ifdef PASS_SSL 
void CConnection_Mono::SetSecuredConnection (BOOL flag) 
{ 
    m_bSecuredConnection = flag; 
} 
 
void CConnection_Mono::SetSSLCTX (SSL_CTX *ctx) 
{ 
    m_SSLctx = ctx; 
} 
 
void CConnection_Mono::SetCertificateFile (char *file) 
{ 
    if (file != NULL) 
    { 
        if (m_szCertificateFile != NULL) 
        { 
        	delete [] m_szCertificateFile; 
        } 
        m_szCertificateFile = new char [(strlen (file)) + 1]; 
        strcpy (m_szCertificateFile,file); 
    } 
} 
 
char *CConnection_Mono::GetCertificateFile (void) 
{ 
    return m_szCertificateFile; 
} 
 
void CConnection_Mono::SetPrivateKeyFile (char *file) 
{ 
    if (file != NULL) 
    { 
        if (m_szPrivateKeyFile != NULL) 
        { 
        	delete [] m_szPrivateKeyFile; 
        } 
        m_szPrivateKeyFile = new char [(strlen (file)) + 1]; 
        strcpy (m_szPrivateKeyFile,file); 
    } 
} 
 
void CConnection_Mono::SetPrivateKeyPassword (char *pass) 
{ 
	strncpy (m_szPrivateKeyPassword,pass,sizeof(m_szPrivateKeyPassword)); 
} 
 
char *CConnection_Mono::GetPrivateKeyPassword (void) 
{ 
	return m_szPrivateKeyPassword; 
} 
 
#endif //PASS_SSL 
 
