// BEIDApp.cpp mentation of the CBEIDApp class.
//
//////////////////////////////////////////////////////////////////////

/******************************************
 Change these consts when a card version changes
******************************************/
#define EIDLIB_CURRENT_APPLET_ITF_VERSION 0x00
#define EIDLIB_CURRENT_ELECTRO_PERSO_ITF 0x00
#define EIDLIB_CURRENT_EF_ID 0x00
#define EIDLIB_CURRENT_EF_ADDRESS 0x00

#include "wx/wxprec.h"

#ifdef __BORLANDC__
    #pragma hdrstop
#endif

// Include wxWindows' headers

#ifndef WX_PRECOMP
    #include <wx/wx.h>
#endif

#include "BEIDApp.h"
#include "OpenSCReader.h"
#include "VirtualReader.h"
#include "CommandAPDU.h"
#include "TLVBuffer.h"
#include "CertifManager.h"
#include "Verify.h"
#include "Config.h"
#include "wxwalldlg.h"
#include "wx/filename.h"
#include "AutoUpdate.h"

#ifdef _WIN32
    #include "processapi.h"
#endif


#ifdef __LOGGING_CXX
    #include <log4cxx/patternlayout.h>
    #include <log4cxx/rollingfileappender.h>
    #define MYLOG4CXX_DEBUG(this, message) { \
        LOG4CXX_DEBUG(this->m_Logger, message) \
    }

#ifdef _DEBUG
        #pragma comment( lib, "./lib/log4cxxd.lib" )
#else
        #pragma comment( lib, "./lib/log4cxx.lib" )
#endif
#else
    #define MYLOG4CXX_DEBUG(this, message)
#endif


#ifndef SC_NO_ERROR
    #define SC_NO_ERROR 0
#endif

#define TOKENINFO_OFFSET    37

#include "wxmainapp.h"


#ifdef _WIN32
  IMPLEMENT_APP_NO_MAIN(MyApp)
#else
  eidlib::MyApp *gpApp = NULL;
#endif

wxCriticalSection CBEIDApp::m_cs;
BOOL CBEIDApp::m_bDownload = FALSE;

/* Helper Functions */
unsigned short BytesToShort(unsigned char *pucData, int iOffset, bool msbFirst = true) 
{
    if(msbFirst)
	    return (256 * (((unsigned short)pucData[iOffset]) & 255) + (((int)pucData[iOffset+1]) & 255));
    else
	    return ((((unsigned short)pucData[iOffset]) & 255) + 256 * (((int)pucData[iOffset+1]) & 255));
}

#ifndef _WIN32
    #include <gtk/gtk.h>
#endif

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////

CBEIDApp::CBEIDApp(char *pszReaderName, int iOCSP, int iCRL)
{
    memset(m_szProcessName, 0, sizeof(m_szProcessName));
    memset(m_szReaderName, 0, sizeof(m_szReaderName));
    if(pszReaderName != NULL)
    {
        strcpy(m_szReaderName, pszReaderName);
    }
    m_iOCSP = iOCSP;
    m_iCRL = iCRL;
    m_pReader = NULL;
    m_pCertifManager = NULL;
    memset(&m_Application, 0, sizeof(BEID_Bytes));
    m_Application.data = new unsigned char[64];
    memset(m_Application.data, 0, 64);
    memcpy(m_Application.data, CCertifManager::m_ucAID, 12);
    m_Application.length = 12;
    m_bShowRootWarning = true;
    m_pAutoUpdate = NULL;
    m_bDownload = FALSE;
#ifdef __LOGGING_CXX
    m_Logger = log4cxx::Logger::getLogger(_T("CBEIDApp"));
    m_Logger->addAppender(new log4cxx::RollingFileAppender(new log4cxx::PatternLayout("%d{%d/%m/%Y %H:%M:%S} %-5p %c - %m%n"), "C:\\eidlib.log", false));
#endif

#ifdef _WIN32
    wxCreateApp();
    wxApp::Initialize();
    wxGetApp().OnInit();
#else
    if(!wxTheApp)
    {	
        gpApp = (eidlib::MyApp *)wxTheApp = (wxApp *)new eidlib::MyApp;
    }
 /*   int argc = 1;
    char** argv = new char*[2];
    argv[0] = new char[8];
    memset(argv[0], 0, 8);
    strcpy(argv[0], "eidlib");
    argv[1] = NULL;
*/
    gtk_set_locale();
    
   // gtk_init( &argc, &argv );
    gtk_init( 0, NULL );

    if(gpApp != NULL)
    {
        wxApp::Initialize();
        gpApp->OnInit();
    }
#endif

    eidlib::CConfig::Load();
    FillProcessName();   
    
    // AutoUpdate
     if(eidlib::CConfig::AllowAutoUpdate() )
     {
        m_pAutoUpdate = new CAutoUpdate(wxTHREAD_JOINABLE );
        if ( m_pAutoUpdate->Create() != wxTHREAD_NO_ERROR )
        {
            m_pAutoUpdate = NULL;
        }
     }
}   

CBEIDApp::~CBEIDApp()
{
    if(m_pAutoUpdate != NULL)
    {
#ifdef _WIN32
        m_pAutoUpdate->Wait();
#endif
        delete m_pAutoUpdate;
        m_pAutoUpdate = NULL;
    }

    eidlib::CConfig::Unload(); 
    if (m_Application.data != NULL)
    {
        delete [] m_Application.data;
    }
#ifdef _WIN32
        wxGetApp().OnExit();
        wxApp::CleanUp();
#else
    if(gpApp != NULL)
    {
        gpApp->OnExit();
      	delete gpApp;
        gpApp = NULL;
        wxTheApp = NULL;
    }
#endif
}

CCardReader *CBEIDApp::CreateReader()
{
    if(m_pReader != NULL)
    {
        delete m_pReader;
        m_pReader= NULL;
    }
    if(0 == strcmp(m_szReaderName, "VIRTUAL"))
    {
        m_pReader = new CVirtualReader();
    }
    else
    {
        m_pReader = new COpenSCReader();
    }
    return m_pReader;
}

BOOL CBEIDApp::Init(long *plHandle, BEID_Status *ptStatus)
{
#ifdef _WIN32
    wxCriticalSectionLocker locker(m_cs);
#endif
    BOOL bRet = FALSE;
    *plHandle = 0;
    m_bShowRootWarning = true;
    MYLOG4CXX_DEBUG(this, "*** Begin Init ***");

    CreateReader();
    if(m_pReader != NULL)
    {
        if(SC_NO_ERROR == m_pReader->EstablishContext(ptStatus))
        {
            MYLOG4CXX_DEBUG(this, "EstablishContext OK");
            if(SC_NO_ERROR == m_pReader->Connect(m_szReaderName, ptStatus))
            {
                MYLOG4CXX_DEBUG(this, "Connect OK");
                *plHandle = m_pReader->GetHandle();
                m_pCertifManager = new CCertifManager();
                m_pCertifManager->SetCardReader(m_pReader);
                CVerify::SetCertifManager(m_pCertifManager); 
                bRet = TRUE;
            }
        }
    }
    MYLOG4CXX_DEBUG(this, "*** End Init ***");
    if(bRet && m_pAutoUpdate != NULL)
    {
        BEID_VersionInfo tVersionInfo = {0};
        BEID_Status tStatus = {0};
        if(GetVersionInfo(&tVersionInfo, FALSE, NULL, &tStatus))
        {
            CheckVersion(&tVersionInfo);
        }
        m_pAutoUpdate->Run();
#ifndef _WIN32
        m_pAutoUpdate->Wait();
        if(m_bDownload)
        {
            m_pAutoUpdate->RetrieveNewVersion(false);
        }
#endif
    }
    return bRet;
}

BOOL CBEIDApp::Exit(BEID_Status *ptStatus)
{
    wxCriticalSectionLocker locker(m_cs);
    BOOL bRet = TRUE;
    MYLOG4CXX_DEBUG(this, "*** Begin Exit ***");

    if(m_pCertifManager != NULL)
    {
        m_pCertifManager->Cleanup(); 
        delete m_pCertifManager;
        m_pCertifManager = NULL;
        CVerify::SetCertifManager(NULL); 
    }

    if(m_pReader != NULL)
    {
        if(SC_NO_ERROR == m_pReader->Disconnect(ptStatus))
        {	
            m_pReader->ReleaseContext(ptStatus);
            delete m_pReader;
            m_pReader= NULL;
        }
    }
    m_bShowRootWarning = true;
    MYLOG4CXX_DEBUG(this, "*** End Exit ***");
    return bRet;
}

BOOL CBEIDApp::GetID(BEID_ID_Data *ptData, BEID_Certif_Check *ptCertifCheck, BEID_Status *ptStatus, BEID_Raw *pRawData /* NULL */)
{
    wxCriticalSectionLocker locker(m_cs);
    long lRet = SC_NO_ERROR;
    MYLOG4CXX_DEBUG(this, "*** Begin GetID ***");
    // Firewall Check
    if(!FireWallAllowed("id"))
    {
        ptCertifCheck->signatureCheck = BEID_SIGNATURE_INVALID;
        MYLOG4CXX_DEBUG(this, "Firewall blocked request");
        MYLOG4CXX_DEBUG(this, "*** Exit GetID ***");
        return FALSE;
    }

    if(m_pReader != NULL)
    {
        unsigned char ucID[BEID_MAX_RAW_ID_LEN] = {0};
        unsigned long ulLen = BEID_MAX_RAW_ID_LEN;
        unsigned char ucFileID[6] = {0x3F, 0x00, 0xDF, 0x01, 0x40, 0x31};

        // Try to read from cache
        BEID_Bytes tTempBytes = {0};
        tTempBytes.data = ucID;
        tTempBytes.length = BEID_MAX_RAW_ID_LEN;
        char *pszFile = "\x40\x31";
        BOOL bCache = FALSE;
        if(SC_NO_ERROR != m_pReader->ReadCachedFile(pszFile, &tTempBytes))
        {
            MYLOG4CXX_DEBUG(this, "ReadCachedFile ID NOK");
            if(SC_NO_ERROR == (lRet = m_pReader->ReadFile(ucFileID, sizeof(ucFileID), ucID, &ulLen, ptStatus)))
            {
                MYLOG4CXX_DEBUG(this, "ReadFile ID OK");
                bCache = TRUE;
            }
        }
        else
        {
            MYLOG4CXX_DEBUG(this, "ReadCachedFile ID OK");
            ulLen = tTempBytes.length;
        }
        if(SC_NO_ERROR == lRet)
        {
            if(SC_NO_ERROR == (lRet = ReadCertificates(ptCertifCheck, ptStatus)))
            {
                MYLOG4CXX_DEBUG(this, "ReadCertificates OK");
                lRet =VerifySignatureID(ucID, ulLen, ptCertifCheck, ptStatus, pRawData);
                BOOL bFill = FALSE;
                if(ptCertifCheck->signatureCheck == BEID_SIGNATURE_VALID)
                {
                    bFill = TRUE;
                    MYLOG4CXX_DEBUG(this, "Valid Signature");
                }
                else if(ptCertifCheck->signatureCheck == BEID_SIGNATURE_VALID_WRONG_RRNCERT)
                {
                    MYLOG4CXX_DEBUG(this, "Valid Signature But Wrong RN Cert");
                    if(eidlib::CConfig::AllowTestRoot())
                    {
                        MYLOG4CXX_DEBUG(this, "AllowTestRoot Enabled");
                        // Warning
                        if(m_bShowRootWarning)
                        {
                            if(wxYES == wxMessageBox(_("The Root Certificate is not correct.\nThis may be a test card.\n\nDo you want to accept it ?"),
                                                    _("Belgian EID Middleware"), wxYES_NO | wxNO_DEFAULT | wxICON_WARNING | wxCENTRE))
                            {
                                bFill = TRUE;
                                m_bShowRootWarning = false;
                                MYLOG4CXX_DEBUG(this, "Test Card dialogbox accepted");
                            }
                        }
                        else
                        {
                            bFill = TRUE;
                        }
                    }
                }

                if(!bFill)
                {
                    ptStatus->general = BEID_E_INVALID_ROOT_CERT;
                    lRet = -1;
                }
                if(bFill && SC_NO_ERROR == VerifyCertificates(ptCertifCheck))
                {
                    MYLOG4CXX_DEBUG(this, "VerifyCertificates OK");
                    if(pRawData != NULL)
                    {
                        memcpy(pRawData->idData, ucID, ulLen);
                        pRawData->idLength = ulLen;
                    }
                    // Parse TLV
                    CTLVBuffer oTLVBuffer;
                    oTLVBuffer.ParseTLV(ucID, ulLen);
                    FillIDData(oTLVBuffer, ptData);
                    // Write cache
                    if(bCache)
                    {
                        m_pReader->WriteCachedFile(pszFile, ucID, ulLen);
                    }
                }
                m_pCertifManager->FillCertifs(ptCertifCheck);
                MYLOG4CXX_DEBUG(this, "ID retrieved and validated OK");
            }
        }
    }


    MYLOG4CXX_DEBUG(this, "*** Exit GetID ***");
    return lRet == SC_NO_ERROR ? TRUE : FALSE;
}

BOOL CBEIDApp::GetAddress(BEID_Address *ptAddress, BEID_Certif_Check *ptCertifCheck, BEID_Status *ptStatus, BEID_Raw *pRawData /* NULL */)
{
    wxCriticalSectionLocker locker(m_cs);
    long lRet = SC_NO_ERROR;
    if(!FireWallAllowed("address"))
    {
        ptCertifCheck->signatureCheck = BEID_SIGNATURE_INVALID;
        return FALSE;
    }
    if(m_pReader != NULL)
    {        
        unsigned char ucAddress[BEID_MAX_RAW_ADDRESS_LEN] = {0};
        unsigned long ulLen = BEID_MAX_RAW_ADDRESS_LEN;
        unsigned char ucFileID[6] = {0x3F, 0x00, 0xDF, 0x01, 0x40, 0x33};

        // Try to read from cache
        BEID_Bytes tTempBytes = {0};
        tTempBytes.data = ucAddress;
        tTempBytes.length = BEID_MAX_RAW_ADDRESS_LEN;
        char *pszFile = "\x40\x33";
        BOOL bCache = FALSE;
        if(SC_NO_ERROR != m_pReader->ReadCachedMemFile(pszFile, &tTempBytes))
        {
            if(SC_NO_ERROR == (lRet = m_pReader->ReadFile(ucFileID, sizeof(ucFileID), ucAddress, &ulLen, ptStatus)))
            {
                bCache = TRUE;
            }
        }
        else
        {
            ulLen = tTempBytes.length;
        }

        if(SC_NO_ERROR == lRet)
        {
            if(SC_NO_ERROR == (lRet = ReadCertificates(ptCertifCheck, ptStatus)))
            {
                // Parse TLV
                CTLVBuffer oTLVBuffer;
                oTLVBuffer.ParseTLV(ucAddress, ulLen);
                unsigned long ulRealLen = oTLVBuffer.GetLengthForSignature(); 
                lRet =VerifySignatureAddress(ucAddress, ulRealLen, ptCertifCheck, ptStatus, pRawData);
                BOOL bFill = FALSE;
                if(ptCertifCheck->signatureCheck == BEID_SIGNATURE_VALID)
                {
                    bFill = TRUE;
                }
                else if(ptCertifCheck->signatureCheck == BEID_SIGNATURE_VALID_WRONG_RRNCERT)
                {
                    if(eidlib::CConfig::AllowTestRoot())
                    {
                        // Warning
                        if(m_bShowRootWarning)
                        {
                            if(wxYES == wxMessageBox(_("The Root Certificate is not correct.\nThis may be a test card.\n\nDo you want to accept it ?"),
                                                    _("Belgian EID Middleware"), wxYES_NO | wxNO_DEFAULT | wxICON_WARNING | wxCENTRE))
                            {
                                bFill = TRUE;
                                m_bShowRootWarning = false;
                                MYLOG4CXX_DEBUG(this, "Test Card dialogbox accepted");
                            }
                        }
                        else
                        {
                                bFill = TRUE;
                        }
                    }
                }

                if(!bFill)
                {
                    ptStatus->general = BEID_E_INVALID_ROOT_CERT;
                    lRet = -1;
                }
                if(bFill && SC_NO_ERROR == VerifyCertificates(ptCertifCheck))
                {
                    if(pRawData != NULL)
                    {
                        memcpy(pRawData->addrData, ucAddress, ulLen);
                        pRawData->addrLength = ulLen;
                    }

                    FillAddressData(oTLVBuffer, ptAddress);
                    // Write cache
                    if(bCache)
                    {
                        m_pReader->WriteCachedMemFile(pszFile, ucAddress, ulLen);
                    }
                }
                m_pCertifManager->FillCertifs(ptCertifCheck);
            }
        }
    }
    
    return lRet == SC_NO_ERROR ? TRUE : FALSE;
}

BOOL CBEIDApp::GetPicture(BEID_Bytes *ptPicture, BEID_Certif_Check *ptCertifCheck, BEID_Status *ptStatus, BEID_Raw *pRawData /* NULL */)
{
    wxCriticalSectionLocker locker(m_cs);
    long lRet = SC_NO_ERROR;
    if(!FireWallAllowed("photo"))
    {
        ptCertifCheck->signatureCheck = BEID_SIGNATURE_INVALID;
        return FALSE;
    }
    if(m_pReader != NULL)
    {        
        unsigned char ucPicture[BEID_MAX_PICTURE_LEN] = {0};
        unsigned long ulLen = BEID_MAX_PICTURE_LEN;
        unsigned char ucFileID[6] = {0x3F, 0x00, 0xDF, 0x01, 0x40, 0x35};

        // Try to read from cache
        BEID_Bytes tTempBytes = {0};
        tTempBytes.data = ucPicture;
        tTempBytes.length = BEID_MAX_PICTURE_LEN;
        char *pszFile = "\x40\x35";
        BOOL bCache = FALSE;
        if(SC_NO_ERROR != m_pReader->ReadCachedFile(pszFile, &tTempBytes))
        {
            if(SC_NO_ERROR == (lRet = m_pReader->ReadFile(ucFileID, sizeof(ucFileID), ucPicture, &ulLen, ptStatus)))
            {
                bCache = TRUE;
            }
        }
        else
        {
            ulLen = tTempBytes.length;
        }
        if(SC_NO_ERROR == lRet)
        {
            if(ptPicture->length < ulLen)
            {
                // Buffer too small
                ptStatus->general = BEID_E_INSUFFICIENT_BUFFER;
                return FALSE;
            }
            if(SC_NO_ERROR == (lRet = ReadCertificates(ptCertifCheck, ptStatus)))
            {
                lRet = VerifyHashPicture(ucPicture, ulLen,  ptCertifCheck, ptStatus, pRawData);
                BOOL bFill = FALSE;
                if(ptCertifCheck->signatureCheck == BEID_SIGNATURE_VALID)
                {
                    bFill = TRUE;
                }
                else if(ptCertifCheck->signatureCheck == BEID_SIGNATURE_VALID_WRONG_RRNCERT)
                {
                    if(eidlib::CConfig::AllowTestRoot())
                    {
                        // Warning
                        if(m_bShowRootWarning)
                        {
                            if(wxYES == wxMessageBox(_("The Root Certificate is not correct.\nThis may be a test card.\n\nDo you want to accept it ?"),
                                                    _("Belgian EID Middleware"), wxYES_NO | wxNO_DEFAULT | wxICON_WARNING | wxCENTRE))
                            {
                                bFill = TRUE;
                                m_bShowRootWarning = false;
                                MYLOG4CXX_DEBUG(this, "Test Card dialogbox accepted");
                            }
                        }
                        else
                        {
                                bFill = TRUE;
                        }
                    }
                }

                if(!bFill)
                {
                    ptStatus->general = BEID_E_INVALID_ROOT_CERT;
                    lRet = -1;
                }
                if(bFill && SC_NO_ERROR == VerifyCertificates(ptCertifCheck))
                {
                    if(pRawData != NULL)
                    {
                        memcpy(pRawData->pictureData, ucPicture, ulLen);
                        pRawData->pictureLength = ulLen;
                    }

                    memcpy(ptPicture->data, ucPicture, ulLen);
                    ptPicture->length = ulLen;
                    // Write cache
                    if(bCache)
                    {
                        m_pReader->WriteCachedFile(pszFile, ucPicture, ulLen);
                    }
                }
                else
                {
                     ptPicture->length = 0;
                }
                m_pCertifManager->FillCertifs(ptCertifCheck);
            }
        }
    }
    
    return lRet == SC_NO_ERROR ? TRUE : FALSE;
}

void CBEIDApp::FillIDData(CTLVBuffer & oTLVBuffer, BEID_ID_Data *ptData)
{
    ptData->version = 0;
    oTLVBuffer.FillShortData(0x00, &ptData->version);
    if(ptData->version > EIDLIB_CURRENT_EF_ID)
    {
        if(m_pAutoUpdate != NULL)
        {
            m_pAutoUpdate->RetrieveNewVersion(true);
        }
        return;
    }

    if (ptData->version == 0)
    {
        oTLVBuffer.FillASCIIData(0x01, ptData->cardNumber);
        oTLVBuffer.FillBinaryStringData(0x02, ptData->chipNumber);
        oTLVBuffer.FillDateData(0x03, ptData->validityDateBegin);
        oTLVBuffer.FillDateData(0x04, ptData->validityDateEnd);
        oTLVBuffer.FillUTF8Data(0x05,  ptData->municipality);
        oTLVBuffer.FillASCIIData(0x06, ptData->nationalNumber);
        oTLVBuffer.FillUTF8Data(0x07,  ptData->name);
        oTLVBuffer.FillUTF8Data(0x08,  ptData->firstName1);
        oTLVBuffer.FillUTF8Data(0x09,  ptData->firstName3);
        strcpy(ptData->nationality, "be");
        oTLVBuffer.FillUTF8Data(0x0B,  ptData->birthLocation);
        oTLVBuffer.FillDateData(0x0C,  ptData->birthDate);
        CTLV *pTagData = NULL;
        if(NULL != (pTagData = oTLVBuffer.GetTagData(0x0D)))
        {
            unsigned char *pData = pTagData->GetData(); 
            if(pData[0] == 'M')
                strcpy(ptData->sex, "M");
            else
                strcpy(ptData->sex, "F");
        }
        else
        {
            oTLVBuffer.FillASCIIData(0x0D, ptData->sex);
        }
        oTLVBuffer.FillUTF8Data(0x0E,  ptData->nobleCondition);
        oTLVBuffer.FillLongData(0x0F, &ptData->documentType);
        ptData->extendedMinority = 0;
        ptData->whiteCane = 0;
        ptData->yellowCane = 0;
        int iStatus = 0;
        oTLVBuffer.FillIntData(0x10, &iStatus);
        switch(iStatus)
        {
            case 1:
                ptData->whiteCane = TRUE;
            break;
            case 2:
                ptData->extendedMinority = TRUE;
            break;
            case 3:
                ptData->whiteCane = TRUE;
                ptData->extendedMinority = TRUE;
            break;
            case 4:
                ptData->yellowCane = TRUE;
            break;
            case 5:
                ptData->yellowCane = TRUE;
                ptData->extendedMinority = TRUE;
            break;
        }
        oTLVBuffer.FillBinaryData(0x11, ptData->hashPhoto);
    }
}

void CBEIDApp::FillAddressData(CTLVBuffer & oTLVBuffer, BEID_Address *ptData)
{
    ptData->version = 0;
    oTLVBuffer.FillShortData(0x00, &ptData->version);
    if(ptData->version > EIDLIB_CURRENT_EF_ADDRESS)
    {
        if(m_pAutoUpdate != NULL)
        {
            m_pAutoUpdate->RetrieveNewVersion(true);
        }
        return;
    }

    if (ptData->version == 0)
    {
        oTLVBuffer.FillUTF8Data(0x01, ptData->street);
        oTLVBuffer.FillASCIIData(0x02, ptData->zip);
        oTLVBuffer.FillUTF8Data(0x03, ptData->municipality);
    }
}

long CBEIDApp::VerifySignatureID(unsigned char *pData, unsigned long ulLenData, BEID_Certif_Check *ptCertifCheck, BEID_Status *ptStatus, BEID_Raw *pRawData /* NULL */)
{
    MYLOG4CXX_DEBUG(this, "*** Begin VerifySignatureID ***");
    long lRet = SC_NO_ERROR;
    unsigned char ucSGNID[BEID_MAX_SIGNATURE_LEN] = {0};
    unsigned long ulSGNIDLen = BEID_MAX_SIGNATURE_LEN;
    unsigned char ucFileSGNID[6] = {0x3F, 0x00, 0xDF, 0x01, 0x40, 0x32};
    // Try to read from cache
    BEID_Bytes tTempBytes = {0};
    tTempBytes.data = ucSGNID;
    tTempBytes.length = BEID_MAX_SIGNATURE_LEN;
    char *pszFile = "\x40\x32";
    if(SC_NO_ERROR != m_pReader->ReadCachedFile(pszFile, &tTempBytes))
    {
        MYLOG4CXX_DEBUG(this, "ReadCachedFile SIGID NOK");
        if(SC_NO_ERROR == (lRet = m_pReader->ReadFile(ucFileSGNID, sizeof(ucFileSGNID), ucSGNID, &ulSGNIDLen, ptStatus)))
        {
            MYLOG4CXX_DEBUG(this, "ReadFile SIGID OK");
            m_pReader->WriteCachedFile(pszFile, ucSGNID, ulSGNIDLen);
        }
    }
    else
    {
        MYLOG4CXX_DEBUG(this, "ReadCachedFile SIGID OK");
        ulSGNIDLen = tTempBytes.length;
    }
    if(SC_NO_ERROR == lRet)
    {
        ptCertifCheck->signatureCheck = CVerify::VerifySignature(pData, ulLenData, ucSGNID, ulSGNIDLen); 
        if(BEID_SIGNATURE_PROCESSING_ERROR == ptCertifCheck->signatureCheck)
        {
             MYLOG4CXX_DEBUG(this, "VerifySignature ID Processing Error");
             ptStatus->general = BEID_E_VERIFICATION;
        }
        else
        {
            if(pRawData != NULL)
            {
                memcpy(pRawData->idSigData, ucSGNID, ulSGNIDLen);
                pRawData->idSigLength = ulSGNIDLen;
            }
        }
    }
    MYLOG4CXX_DEBUG(this, "*** End VerifySignatureID ***");
    return lRet;
}

long CBEIDApp::VerifySignatureAddress(unsigned char *pData, unsigned long ulLenData, BEID_Certif_Check *ptCertifCheck, BEID_Status *ptStatus, BEID_Raw *pRawData /* NULL */)
{
    long lRet = SC_NO_ERROR;
    unsigned char ucSGNAddress[BEID_MAX_SIGNATURE_LEN] = {0};
    unsigned long ulSGNAddressLen = BEID_MAX_SIGNATURE_LEN;
    unsigned char ucFileSGNID[6] = {0x3F, 0x00, 0xDF, 0x01, 0x40, 0x34};
    // Try to read from cache
    BEID_Bytes tTempBytesAddr = {0};
    tTempBytesAddr.data = ucSGNAddress;
    tTempBytesAddr.length = ulSGNAddressLen;
    char *pszFileAddr = "\x40\x34";
    if(SC_NO_ERROR != m_pReader->ReadCachedMemFile(pszFileAddr, &tTempBytesAddr))
    {
        if(SC_NO_ERROR == (lRet = m_pReader->ReadFile(ucFileSGNID, sizeof(ucFileSGNID), ucSGNAddress, &ulSGNAddressLen, ptStatus)))
        {
            m_pReader->WriteCachedMemFile(pszFileAddr, ucSGNAddress, ulSGNAddressLen);
        }
    }
    else
    {
        ulSGNAddressLen = tTempBytesAddr.length;
    }

    if(SC_NO_ERROR == lRet)
    {
        // Read EF(SGN#ID)
        unsigned char ucSGNID[BEID_MAX_SIGNATURE_LEN] = {0};
        unsigned long ulSGNIDLen = BEID_MAX_SIGNATURE_LEN;
        unsigned char ucFileSGNID[6] = {0x3F, 0x00, 0xDF, 0x01, 0x40, 0x32};

        // Try to read from cache
        BEID_Bytes tTempBytes = {0};
        tTempBytes.data = ucSGNID;
        tTempBytes.length = BEID_MAX_SIGNATURE_LEN;
        char *pszFile = "\x40\x32";
        if(SC_NO_ERROR != m_pReader->ReadCachedFile(pszFile, &tTempBytes))
        {
            if(SC_NO_ERROR == (lRet = m_pReader->ReadFile(ucFileSGNID, sizeof(ucFileSGNID), ucSGNID, &ulSGNIDLen, ptStatus)))
            {
                m_pReader->WriteCachedFile(pszFile, ucSGNID, ulSGNIDLen);
            }
        }
        else
        {
            ulSGNIDLen = tTempBytes.length;
        }
        if(SC_NO_ERROR == lRet)
        {
            unsigned char szRealBuf[BEID_MAX_RAW_ADDRESS_LEN + BEID_MAX_SIGNATURE_LEN] = {0};
            memcpy(szRealBuf, pData, ulLenData);
            memcpy(szRealBuf + ulLenData, ucSGNID, ulSGNIDLen);
            ptCertifCheck->signatureCheck = CVerify::VerifySignature(szRealBuf, ulLenData + ulSGNIDLen, ucSGNAddress, ulSGNAddressLen); 
            if(BEID_SIGNATURE_PROCESSING_ERROR == ptCertifCheck->signatureCheck)
            {
                 ptStatus->general = BEID_E_VERIFICATION;
            }
            else
            {
                if(pRawData != NULL)
                {
                    memcpy(pRawData->addrSigData, ucSGNAddress, ulSGNAddressLen);
                    pRawData->addrSigLength = ulSGNAddressLen;
                }
            }
        }
    }
    return lRet;
}

long CBEIDApp::VerifyHashPicture(unsigned char *pData, unsigned long ulLenData, BEID_Certif_Check *ptCertifCheck, BEID_Status *ptStatus, BEID_Raw *pRawData /* NULL */)
{
    long lRet = SC_NO_ERROR;
    unsigned char ucHash[BEID_MAX_HASH_PICTURE_LEN] = {0};
    lRet = ReadHashData(ucHash, ptStatus);
    if(SC_NO_ERROR == lRet)
    {
        ptCertifCheck->signatureCheck = CVerify::VerifyHash(pData, ulLenData, ucHash); 
        if(BEID_SIGNATURE_PROCESSING_ERROR == ptCertifCheck->signatureCheck)
        {
            ptStatus->general = BEID_E_VERIFICATION;
        }
    }
    return lRet;
}

long CBEIDApp::ReadCertificates(BEID_Certif_Check *ptCertifCheck, BEID_Status *ptStatus)
{
    long lRet = SC_NO_ERROR;
    MYLOG4CXX_DEBUG(this, "*** Begin ReadCertificates ***");

    if(m_pCertifManager != NULL)
    {
         if(m_pCertifManager->GetCertifCount() == 0)
        {
            MYLOG4CXX_DEBUG(this, "Read All Certificates (Nothing in cache)");
            BOOL bAll = TRUE;
            if(-1 == m_iOCSP && -1 == m_iCRL)
            {
                MYLOG4CXX_DEBUG(this, "Skipped Read All Certificates");
                bAll = FALSE;
            }
            lRet = m_pCertifManager->ReadCertifsP15(ptStatus, bAll);
        }
    }
    MYLOG4CXX_DEBUG(this, "*** End ReadCertificates ***");
    return lRet;
}

long CBEIDApp::ReadHashData(unsigned char ucPicHash[20], BEID_Status *ptStatus)
{
    long lRet = SC_NO_ERROR;
    memset(ucPicHash, 0, sizeof(ucPicHash));
    if(m_pReader != NULL)
    {  
        unsigned char ucID[BEID_MAX_RAW_ID_LEN] = {0};
        unsigned long ulLen = BEID_MAX_RAW_ID_LEN;
        unsigned char ucFileID[6] = {0x3F, 0x00, 0xDF, 0x01, 0x40, 0x31};

        // Try to read from cache
        BEID_Bytes tTempBytes = {0};
        tTempBytes.data = ucID;
        tTempBytes.length = BEID_MAX_RAW_ID_LEN;
        char *pszFile = "\x40\x31";
        if(SC_NO_ERROR != m_pReader->ReadCachedFile(pszFile, &tTempBytes))
        {
            if(SC_NO_ERROR == (lRet = m_pReader->ReadFile(ucFileID, sizeof(ucFileID), ucID, &ulLen, ptStatus)))
            {
                m_pReader->WriteCachedFile(pszFile, ucID, ulLen);
            }
        }
        else
        {
            ulLen = tTempBytes.length;
        }
        if(SC_NO_ERROR == lRet)
        {
            // Parse TLV
            CTLVBuffer oTLVBuffer;
            oTLVBuffer.ParseTLV(ucID, ulLen);
            oTLVBuffer.FillBinaryData(0x11, ucPicHash);
        }
    }
    return lRet;
}

BOOL CBEIDApp::BeginTransaction(BEID_Status *ptStatus)
{
    wxCriticalSectionLocker locker(m_cs);
    long lRet = SC_NO_ERROR;
    if(m_pReader != NULL)
    {
        lRet = m_pReader->BeginTransaction(ptStatus);
    }
    return lRet == SC_NO_ERROR ? TRUE : FALSE;
}

BOOL CBEIDApp::EndTransaction(BEID_Status *ptStatus)
{
    long lRet = SC_NO_ERROR;
    if(m_pReader != NULL)
    {
        lRet = m_pReader->EndTransaction(ptStatus);
    }
    return lRet == SC_NO_ERROR ? TRUE : FALSE;
}
        
BOOL CBEIDApp::SelectApplication(BEID_Bytes *ptApplication, BEID_Status *ptStatus)
{
#ifdef _WIN32
    wxCriticalSectionLocker locker(m_cs);
#endif
    long lRet = SC_NO_ERROR;
    if(m_pReader != NULL)
    {
        unsigned char ucMF[2] = {0x3F, 0x00};
        if(SC_NO_ERROR == (lRet = m_pReader->BeginTransaction(ptStatus)))
        {
            lRet = m_pReader->SelectFile(ucMF, sizeof(ucMF), 0x02, ptStatus);
            if(lRet == SC_NO_ERROR)
            {
                lRet = m_pReader->SelectFile(ptApplication->data , ptApplication->length, 0x04, ptStatus);
                if(SC_NO_ERROR == lRet)
                {
                    m_Application.length = ptApplication->length;
                    memcpy(m_Application.data, ptApplication->data, ptApplication->length);
                }
            }
            m_pReader->EndTransaction(ptStatus);
        }
    }
    return lRet == SC_NO_ERROR ? TRUE : FALSE;
}

BOOL CBEIDApp::FlushCache(BEID_Status *ptStatus)
{
    wxCriticalSectionLocker locker(m_cs);
    long lRet = SC_NO_ERROR;
    if(m_pCertifManager != NULL)
    {
        m_pCertifManager->Cleanup(); 
    }
    if(m_pReader != NULL)
    {
        m_pReader->FlushCache();
    }
    return lRet == SC_NO_ERROR ? TRUE : FALSE;
}

BOOL CBEIDApp::SendAPDU(BEID_Bytes *ptCmdAPDU, BEID_Pin *pPinData, BEID_Bytes *ptRespAPDU, BEID_Status *ptStatus)
{
    wxCriticalSectionLocker locker(m_cs);
    long lRet = SC_NO_ERROR;
    // Firewall Check
    if(!FireWallAllowed("sendapdu"))
    {
        return FALSE;
    }
    unsigned char ucPinReference = pPinData->id; 
    if(m_pReader != NULL)
    {
        unsigned char ucStatus[2] = {0};
        unsigned long ulRespLen = ptRespAPDU->length;
        lRet = m_pReader->Transmit(ptCmdAPDU->data, ptCmdAPDU->length, ptRespAPDU->data, &ptRespAPDU->length, ptStatus);  
        if(SC_NO_ERROR == lRet)
        {
            // Check for 'Security Status not satisfied' error
            if(ptRespAPDU->length > 1 && ucPinReference > 0x00)
            {
                ucStatus[0] = ptRespAPDU->data[ptRespAPDU->length - 2];
                ucStatus[1] = ptRespAPDU->data[ptRespAPDU->length - 1];
                if(0x69 == ucStatus[0] && 0x82 == ucStatus[1])
                {
                    long iLeft = -1;
                    lRet = VerifyPIN(pPinData, NULL, &iLeft, ptStatus);
                    if(SC_NO_ERROR == lRet)
                    {
                        // Retry command
                        ptRespAPDU->length = ulRespLen;
                        memset(ptRespAPDU->data, 0, ulRespLen);
                        lRet = m_pReader->Transmit(ptCmdAPDU->data, ptCmdAPDU->length, ptRespAPDU->data, &ptRespAPDU->length, ptStatus);  
                    }
                }
            }
        }
    }
    return lRet == SC_NO_ERROR ? TRUE : FALSE;
}


BOOL CBEIDApp::GetVersionInfo(BEID_VersionInfo *ptVersionInfo, BOOL bSignature, BEID_Bytes *ptSignedStatus, BEID_Status *ptStatus, BEID_Raw *pRawData /* NULL */)
{
#ifdef _WIN32
    wxCriticalSectionLocker locker(m_cs);
#endif
    long lRet = SC_NO_ERROR;
    if(m_pReader != NULL)
    {
        unsigned char ucOutData[BEID_MAX_CARD_DATA_SIG_LEN] = {0};
        unsigned long ulOutLen = sizeof(ucOutData);
        // Get Card Data
        lRet = m_pReader->GetCardData(ucOutData, &ulOutLen, bSignature ? true : false, ptStatus);  
        if(SC_NO_ERROR == lRet && ulOutLen > 0)
        {
            // Get GlobalOSVersion
            unsigned short usGlobalOSVersion = BytesToShort(ucOutData, 22, true);
            memcpy(ptVersionInfo, ucOutData, BEID_MAX_CARD_DATA_LEN);
            ptVersionInfo->GlobalOSVersion = usGlobalOSVersion; 
            if(bSignature)
            {
                if(ptSignedStatus->length < ulOutLen - BEID_MAX_CARD_DATA_LEN)
                {
                    // Buffer too small
                    ptStatus->general = BEID_E_INSUFFICIENT_BUFFER;
                    return FALSE;
                }
                memcpy(ptSignedStatus->data, ucOutData + BEID_MAX_CARD_DATA_LEN, ulOutLen - BEID_MAX_CARD_DATA_LEN);
                ptSignedStatus->length = ulOutLen - BEID_MAX_CARD_DATA_LEN;
            }
            if(pRawData != NULL)
            {
                memcpy(pRawData->cardData, ucOutData, ulOutLen);
                pRawData->cardDataLength = ulOutLen;
            }
            // TokenInfo
            lRet = GetTokenInfo(ptVersionInfo, ptStatus, pRawData);
        }
    }
    return lRet == SC_NO_ERROR ? TRUE : FALSE;
}

long CBEIDApp::GetTokenInfo(BEID_VersionInfo *ptVersionInfo, BEID_Status *ptStatus, BEID_Raw *pRawData /* NULL */)
{
    long lRet = SC_NO_ERROR;
    if(m_pReader != NULL)
    {
        BEID_Bytes tAID = {(unsigned char *)"\xA0\x00\x00\x01\x77\x50\x4B\x43\x53\x2D\x31\x35", 12};
        if(SC_NO_ERROR == (lRet = m_pReader->BeginTransaction(ptStatus)))
        {
            if (SelectApplication(&tAID, ptStatus))
            {
                // Select Tokeninfo file
                unsigned char ucTokenInfo[2] = {0x50, 0x32};
                unsigned char ucToken[256] = {0};
                unsigned long ulLen = sizeof(ucToken);
                lRet = m_pReader->SelectFile(ucTokenInfo, sizeof(ucTokenInfo), 0x02, ptStatus);
                if(lRet == SC_NO_ERROR)
                {
                    lRet = m_pReader->ReadBinary(ucToken, &ulLen, ptStatus);
                    if(SC_NO_ERROR == lRet)
                    {
                        if(ulLen >= TOKENINFO_OFFSET + 4)
                        {
                            ptVersionInfo->GraphPerso = ucToken[TOKENINFO_OFFSET];
                            ptVersionInfo->ElecPerso = ucToken[TOKENINFO_OFFSET+1];
                            ptVersionInfo->ElecPersoInterface = ucToken[TOKENINFO_OFFSET+2];
                            ptVersionInfo->Reserved = ucToken[TOKENINFO_OFFSET+3];
                        }
                        if(pRawData != NULL)
                        {
                            memcpy(pRawData->tokenInfo, ucToken, ulLen);
                            pRawData->tokenInfoLength = ulLen;
                        }
                    }                        
                }
            }
            m_pReader->EndTransaction(ptStatus);
        }
    }
    return lRet;
}

BOOL CBEIDApp::GetPINStatus(BEID_Pin *pPinData, long *piTriesLeft, BOOL bSignature, BEID_Bytes *ptSignedStatus, BEID_Status *ptStatus)
{
    wxCriticalSectionLocker locker(m_cs);
    long lRet = SC_NO_ERROR;
    unsigned char ucPinReference = pPinData->id; 
    *piTriesLeft = -1;
    if(m_pReader != NULL)
    {
        unsigned char ucOutData[BEID_MAX_SIGNATURE_LEN + 1] = {0};
        unsigned long ulOutLen = sizeof(ucOutData);
        // Get Card Data
        lRet = m_pReader->GetPINStatus(ucOutData, &ulOutLen, ucPinReference, bSignature ? true : false, ptStatus);  
        if(SC_NO_ERROR == lRet)
        {
            if(ulOutLen > 0)
            {
                *piTriesLeft = ucOutData[0]; 
                if(bSignature)
                {
                    if(ptSignedStatus->length < ulOutLen - 1)
                    {
                        // Buffer too small
                        ptStatus->general = BEID_E_INSUFFICIENT_BUFFER;
                        return FALSE;
                    }
                    memcpy(ptSignedStatus->data, ucOutData + 1, ulOutLen - 1);
                    ptSignedStatus->length = ulOutLen - 1;
                }
            }
            else
            {
                // Card Error
                if (0 == memcmp(ptStatus->cardSW, "\x6D\x00", 2))
                {
                    ptStatus->general = BEID_E_UNSUPPORTED_FUNCTION;
                } 
            }
        }
    }
    return lRet == SC_NO_ERROR ? TRUE : FALSE;
}

BOOL CBEIDApp::VerifyPIN(BEID_Pin *pPinData, const char *pszPin, long *piTriesLeft, BEID_Status *ptStatus)
{
    wxCriticalSectionLocker locker(m_cs);
    long lRet = SC_NO_ERROR;
    *piTriesLeft = -1;
    if(m_pReader != NULL)
    {
        lRet = m_pReader->VerifyPIN(pPinData, pszPin, piTriesLeft, ptStatus);
    }

    return lRet == SC_NO_ERROR ? TRUE : FALSE;
}

BOOL CBEIDApp::ChangePIN(BEID_Pin *pPinData, char *pszOldPin, char *pszNewPin, long *piTriesLeft, BEID_Status *ptStatus)
{
    wxCriticalSectionLocker locker(m_cs);
    long lRet = SC_NO_ERROR;
    *piTriesLeft = -1;
    if(m_pReader != NULL)
    {
        lRet = m_pReader->ChangePIN(pPinData, pszOldPin, pszNewPin, piTriesLeft, ptStatus);
    }

    return lRet == SC_NO_ERROR ? TRUE : FALSE;
}

BOOL CBEIDApp::ReadFile(BEID_Bytes *ptFileID, BEID_Bytes *ptOutData, BEID_Pin *pPinData, BEID_Status *ptStatus)
{
    wxCriticalSectionLocker locker(m_cs);
    long lRet = SC_NO_ERROR;
    // Firewall Check
    if(!FireWallAllowed("readfile"))
    {
        return FALSE;
    }

    unsigned char ucPinReference = pPinData->id; 
    if(m_pReader != NULL)
    {
        unsigned long ulOutDataLen = ptOutData->length;
        if(SC_NO_ERROR == (lRet = m_pReader->BeginTransaction(ptStatus)))
        {
            lRet = m_pReader->SelectFile(ptFileID->data, ptFileID->length, 0x02, ptStatus);
            if(SC_NO_ERROR == lRet)
            {
                lRet = m_pReader->ReadBinary(ptOutData->data, &ptOutData->length, ptStatus);   
            }

            if(ucPinReference > 0x00)
            {
                // Check for 'Security Status not satisfied' error
                if(0x69 == ptStatus->cardSW[0] && 0x82 == ptStatus->cardSW[1])
                {
                    long iLeft = -1;
                    lRet = VerifyPIN(pPinData, NULL, &iLeft, ptStatus);
                    if(SC_NO_ERROR == lRet)
                    {
                        ptOutData->length = ulOutDataLen;
                        memset(ptOutData->data, 0, ulOutDataLen);
                        lRet = m_pReader->ReadBinary(ptOutData->data, &ptOutData->length, ptStatus);   
                    }
                }
            }
            m_pReader->EndTransaction(ptStatus);     
        }
    }
    return lRet == SC_NO_ERROR ? TRUE : FALSE;
}

BOOL CBEIDApp::WriteFile(BEID_Bytes *ptFileID, BEID_Bytes *ptInData, BEID_Pin *pPinData, BEID_Status *ptStatus)
{
    wxCriticalSectionLocker locker(m_cs);
    long lRet = SC_NO_ERROR;
    // Firewall Check
    if(!FireWallAllowed("writefile"))
    {
        return FALSE;
    }
    unsigned char ucPinReference = pPinData->id;
    if(m_pReader != NULL)
    {
        if(SC_NO_ERROR == (lRet = m_pReader->BeginTransaction(ptStatus)))
        {
            lRet = m_pReader->SelectFile(ptFileID->data, ptFileID->length, 0x02, ptStatus);
            if(SC_NO_ERROR == lRet)
            {
                lRet = m_pReader->UpdateBinary(ptInData->data, ptInData->length, ptStatus);   
            }

            if(ucPinReference > 0x00)
            {
                // Check for 'Security Status not satisfied' error
                if(0x69 == ptStatus->cardSW[0] && 0x82 == ptStatus->cardSW[1])
                {
                    long iLeft = -1;
                    lRet = VerifyPIN(pPinData, NULL, &iLeft, ptStatus);
                    if(SC_NO_ERROR == lRet)
                    {
                        // Retry command
                        lRet = m_pReader->UpdateBinary(ptInData->data, ptInData->length, ptStatus);   
                    }
                }
            }
            m_pReader->EndTransaction(ptStatus);
        }
    }
    return lRet == SC_NO_ERROR ? TRUE : FALSE;
}

long CBEIDApp::VerifyCertificates(BEID_Certif_Check *ptCertifCheck)
{
    long lRet = SC_NO_ERROR;

    ptCertifCheck->usedPolicy = BEID_POLICY_NONE;

    // First check the mandatory
    if (BEID_OCSP_CRL_MANDATORY == m_iOCSP || (BEID_OCSP_CRL_OPTIONAL == m_iOCSP && BEID_OCSP_CRL_OPTIONAL == m_iCRL))
    {
        ptCertifCheck->usedPolicy = BEID_POLICY_OCSP;
        lRet = m_pCertifManager->VerifyCertsOCSP();
    }
    else if(BEID_OCSP_CRL_MANDATORY == m_iCRL)
    {
        ptCertifCheck->usedPolicy = BEID_POLICY_CRL;
        lRet = m_pCertifManager->VerifyCertsCRL(); 
    }

    // In case OCSP or CRL checking failed with error, return
    if(SC_NO_ERROR != lRet && (BEID_OCSP_CRL_MANDATORY == m_iOCSP || BEID_OCSP_CRL_MANDATORY == m_iCRL))
    {
        return lRet;
    }

    // First check if some certificates are not validated
    if(ptCertifCheck->usedPolicy != BEID_POLICY_NONE && m_pCertifManager->FindNotValidated())
    {
        lRet = -1;
    }

    if(SC_NO_ERROR != lRet || ptCertifCheck->usedPolicy == BEID_POLICY_NONE)
    {
        // Check the optional
        if (BEID_OCSP_CRL_OPTIONAL == m_iCRL)
        {
            ptCertifCheck->usedPolicy |= BEID_POLICY_CRL;            
            lRet = m_pCertifManager->VerifyCertsCRL(); 
        }
        else if(BEID_OCSP_CRL_OPTIONAL == m_iOCSP)
        {
            ptCertifCheck->usedPolicy |= BEID_POLICY_OCSP;            
            lRet = m_pCertifManager->VerifyCertsOCSP();
        }
        else
        {
            lRet = SC_NO_ERROR;
        }
    }
    return lRet;
}

BOOL CBEIDApp::FireWallAllowed(char *pszType)
{
    BOOL bRet = FALSE; 
    MYLOG4CXX_DEBUG(this, "*** Begin FireWallAllowed ***");

    // Check Application
    if(strlen(m_szProcessName) > 0)
    {
        std::string strTemp("Checking process ");
        strTemp.append(m_szProcessName); 
        MYLOG4CXX_DEBUG(this, strTemp);
        if(eidlib::CConfig::GetProgramAccess(m_szProcessName, pszType))
        {
            MYLOG4CXX_DEBUG(this, "Access granted");
            bRet = TRUE;
        }
        else
        {
            CWallDialog dialog(NULL);
            dialog.FillData(m_szProcessName, pszType); 
            dialog.CentreOnParent();
            int iRetVal = dialog.ShowModal();
            switch(iRetVal)
            {
                case ID_BUTTON_YES : 
                    bRet = TRUE;
                    MYLOG4CXX_DEBUG(this, "YES Clicked");
                    break;
                case ID_BUTTON_ALWAYS :
                    bRet = TRUE;
                    MYLOG4CXX_DEBUG(this, "ALWAYS Clicked");
                    eidlib::CConfig::UpdateProgramAccess(m_szProcessName, pszType);
                    break;
                case ID_BUTTON_ALWAYS_ALL :
                    bRet = TRUE;
                    MYLOG4CXX_DEBUG(this, "ALWAYS ALL Clicked");
                    eidlib::CConfig::UpdateProgramAccess(m_szProcessName, "*");
                    break;
            }
        }
    }
    MYLOG4CXX_DEBUG(this, "*** End FireWallAllowed ***");
    return bRet;
}

BOOL CBEIDApp::GetRawData(BEID_Raw *pRawData, BEID_Status *ptStatus)
{
#ifdef _WIN32
    wxCriticalSectionLocker locker(m_cs);
#endif
    long lRet = SC_NO_ERROR;
    if(m_pReader != NULL)
    {
        // ID data
        BEID_ID_Data idData = {0};
        BEID_Certif_Check tCheck = {0};
        if(!GetID(&idData, &tCheck, ptStatus, pRawData))
        {
            return FALSE;
        }
        // Address Data
        memset(&tCheck, 0, sizeof(BEID_Certif_Check));
        BEID_Address addrData = {0};
        if(!GetAddress(&addrData, &tCheck, ptStatus, pRawData))
        {
            return FALSE;
        }
        // Picture
        memset(&tCheck, 0, sizeof(BEID_Certif_Check));
        BYTE buffer[4096] = {0};
        BEID_Bytes tBytes = {0};
        tBytes.length = 4096;
        tBytes.data = buffer;
        if(!GetPicture(&tBytes, &tCheck, ptStatus, pRawData))
        {
            return FALSE;
        }
        // VersionInfo
        BEID_VersionInfo tVersion = {0};
        if(!GetVersionInfo(&tVersion, FALSE, NULL, ptStatus, pRawData))
        {
            return FALSE;
        }
        // RN Cert
        eidlib::CCertif *pCertifRN = m_pCertifManager->GetCertif("RN");
        if(pCertifRN != NULL)
        {
            memcpy(pRawData->certRN, pCertifRN->GetData(), pCertifRN->GetLength());
            pRawData->certRNLength = pCertifRN->GetLength();  
        }

        // Challenge Response
        unsigned char ucChallenge[BEID_MAX_CHALLENGE_LEN] = {0};
        unsigned long ulChLen = sizeof(ucChallenge);
        unsigned char ucResponse[BEID_MAX_RESPONSE_LEN] = {0};
        unsigned long ulRespLen = sizeof(ucResponse);
        if(SC_NO_ERROR == (lRet = m_pReader->GetChallengeResponse(ucChallenge, &ulChLen, ucResponse, &ulRespLen, ptStatus)))
        {
            memcpy(pRawData->challenge, ucChallenge, ulChLen);
            pRawData->challengeLength = ulChLen;
            memcpy(pRawData->response, ucResponse, ulRespLen);
            pRawData->responseLength = ulRespLen;
        }
    }
    return lRet == SC_NO_ERROR ? TRUE : FALSE;
}

BOOL CBEIDApp::SetRawData(BEID_Raw *pRawData, BEID_Status *ptStatus)
{
    wxCriticalSectionLocker locker(m_cs);
    long lRet = SC_NO_ERROR;
    if(m_pReader != NULL)
    {
        m_pReader->SetRawData(pRawData); 
    }
    return lRet == SC_NO_ERROR ? TRUE : FALSE;
}

void CBEIDApp::FillProcessName()
{
    unsigned long ulPid = wxGetProcessId();
    if(ulPid > 0)
    {
#ifdef _WIN32
        CProcessApi papi;
        if (papi.Init(true))
        {
            DWORD pl = papi.ProcessesGetList();
            if (pl)
            {
                CProcessApi::tProcessInfo pi;
                while (papi.ProcessesWalk(pl, &pi))
                {
                    if(pi.pid == ulPid)
                    {
                        wxFileName fName(pi.FileName);
                        strcpy(m_szProcessName, fName.GetFullName().GetData());
                        break;
                    }
                }
            }
            papi.ProcessesFreeList(pl);
        }
#endif
#ifndef _WIN32
        char *pszLine = NULL;
        char *pszToken = NULL;
        char *pszCmd = NULL;
        char szLine[256] = {0};
        char szCommand[64] = {0};
        sprintf(szCommand, "ps -p %ld 2>/dev/null", ulPid);
        FILE *fp = popen(szCommand, "r");
        if(fp != NULL)
        {
            // read the header line
            if (NULL != fgets(szLine, sizeof(szLine), fp))
            {
                for (pszLine = szLine; ; pszLine = NULL)
                {
                    if (NULL == (pszToken = strtok(pszLine, " \t\n")))
                    {
                        pclose(fp);
                        return;
                    }

                    if (0 == strcmp("COMMAND", pszToken) || 0 == strcmp("CMD", pszToken))
                    { 
                        //  We found the COMMAND column 
                        pszCmd = pszToken;
                        break;
                    }
                }
                if (NULL != fgets(szLine, sizeof(szLine), fp))
                {
                   // Grab the "word" underneath the command heading...
                   if (NULL != (pszToken = strtok(pszCmd, " \t\n")))
                   {
                        strcpy(m_szProcessName, pszToken);
                   }
                }
            }
            pclose(fp);
        }
#endif
    }
}

bool CBEIDApp::CheckVersion(BEID_VersionInfo *ptVersionInfo)
{
    bool bAsk = false;
    // Check Applet Interface Version
    if(ptVersionInfo->AppletInterfaceVersion > EIDLIB_CURRENT_APPLET_ITF_VERSION)
    {
        bAsk = true;
    }
    // Check TokenInfo
    if(ptVersionInfo->ElecPersoInterface > EIDLIB_CURRENT_ELECTRO_PERSO_ITF)
    {
        bAsk = true;
    }
    if(bAsk && m_pAutoUpdate != NULL)
    {
        m_pAutoUpdate->RetrieveNewVersion(true);
    }
    return bAsk;
}

int CBEIDApp::AskDownload(wxString & strMessage)
{
    wxCriticalSectionLocker locker(m_cs);
    return (wxMessageBox(strMessage, _("Belgian EID Middleware"), wxYES_NO | wxYES_DEFAULT | wxICON_QUESTION | wxCENTRE));
}

void CBEIDApp::SetDownload(BOOL bDownload)
{
    m_bDownload = bDownload;
}
