/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2006  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "XmlLexerTestSuite.h"
#include "XmlLexer.h"
#include "SplittableBuffer.h"
#include "SBOutStream.h"
#include "BString.h"
#include <boost/test/unit_test.hpp>
#include <sstream>
#include <string>

using boost::unit_test_framework::test_suite;
using boost::unit_test_framework::test_case;

class TestXmlLexer : public XmlLexer
{
public:
	TestXmlLexer() : m_errorLine(0) {}
	
	std::string result() { return m_strm.str(); }
	
	int errorLine() { return m_errorLine; }
private:
	/**
	 * \brief Handle a string like "<?name ... ?>", including "<?xml ... ?>".
	 */
	virtual void processProcInstNode(
		Iterator const& begin, Iterator const& end);
	
	/**
	 * \brief Handle a string like "<!DOCTYPE ... >".
	 */
	virtual void processDocTypeNode(
		Iterator const& begin, Iterator const& end);
	
	virtual void processTextNode(
		Iterator const& begin, Iterator const& end);
	
	/**
	 * \brief Handle a string like "<![CDATA[ ... ]]>".
	 */
	virtual void processCDataNode(
		Iterator const& begin, Iterator const& end);
	
	/**
	 * \brief Handle a string like "<!-- ... -->".
	 */
	virtual void processCommentNode(
		Iterator const& begin, Iterator const& end);
	
	/**
	 * \brief Handle the name of an opening tag.
	 */
	virtual void processOpeningTagName(
		Iterator const& begin, Iterator const& end);
	
	virtual void processAttrName(
		Iterator const& begin, Iterator const& end);
	
	/**
	 * \brief Handle an attribute value <b>with quotes</b>, like "on" or 'on'.
	 */
	virtual void processAttrValueWithQuotes(
		Iterator const& begin, Iterator const& end);
	
	/**
	 * \brief Handle the completion of an opening tag.
	 */
	virtual void processOpeningTag(bool empty_tag);
	
	/**
	 * \brief Handle the name (and completion) of a closing tag.
	 */
	virtual void processClosingTag(
		Iterator const& begin, Iterator const& end);
	
	virtual void processParseError(Position const& pos);
	
	std::ostringstream m_strm;
	int m_errorLine;
};


static void test_1()
{
	SBOutStream input_stream;
	input_stream << "<?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
	"<!DOCTYPE greeting [<!ELEMENT greeting (#PCDATA)>]>\n"
	"<forwarding mode=\"manual\"><![CDATA[<>]]></forwarding>";
	TestXmlLexer lexer;
	lexer.consume(input_stream.data(), /* eof = */ true);
	
	std::ostringstream check_stream;
	check_stream << "PROC_INST: <?xml version=\"1.0\" encoding=\"UTF-8\"?>\n"
	"TEXT: \n\n"
	"DOCTYPE: <!DOCTYPE greeting [<!ELEMENT greeting (#PCDATA)>]>\n"
	"TEXT: \n\n"
	"OP_TAG_NAME: forwarding\n"
	"ATTR_NAME: mode\n"
	"ATTR_VALUE: \"manual\"\n"
	"OP_TAG: 0\n"
	"CDATA: <![CDATA[<>]]>\n"
	"CLO_TAG: forwarding\n";
	
	BOOST_CHECK(lexer.result() == check_stream.str());
}

static void test_err_1()
{
	SBOutStream input_stream;
	input_stream << "line1\nline2\n\n<forwarding\nmode>";
	TestXmlLexer lexer;
	lexer.consume(input_stream.data(), /* eof = */ true);
	
	BOOST_CHECK(lexer.errorLine() == 5);
}


/*============================ XmlLexerTestSuite =========================*/

XmlLexerTestSuite::XmlLexerTestSuite()
{
	add(BOOST_TEST_CASE(&test_1));
	add(BOOST_TEST_CASE(&test_err_1));
}


/*============================== TestXmlLexer ============================*/

void
TestXmlLexer::processProcInstNode(
	Iterator const& begin, Iterator const& end)
{
	m_strm << "PROC_INST: " << tokenAsBString(begin, end) << std::endl;
}

void
TestXmlLexer::processDocTypeNode(
	Iterator const& begin, Iterator const& end)
{
	m_strm << "DOCTYPE: " << tokenAsBString(begin, end) << std::endl;
}

void
TestXmlLexer::processTextNode(
	Iterator const& begin, Iterator const& end)
{
	m_strm << "TEXT: " << tokenAsBString(begin, end) << std::endl;
}

void
TestXmlLexer::processCDataNode(
	Iterator const& begin, Iterator const& end)
{
	m_strm << "CDATA: " << tokenAsBString(begin, end) << std::endl;
}

void
TestXmlLexer::processCommentNode(
	Iterator const& begin, Iterator const& end)
{
	m_strm << "TEXT: " << tokenAsBString(begin, end) << std::endl;
}

void
TestXmlLexer::processOpeningTagName(
	Iterator const& begin, Iterator const& end)
{
	m_strm << "OP_TAG_NAME: " << tokenAsBString(begin, end) << std::endl;
}

void
TestXmlLexer::processAttrName(
	Iterator const& begin, Iterator const& end)
{
	m_strm << "ATTR_NAME: " << tokenAsBString(begin, end) << std::endl;
}

void
TestXmlLexer::processAttrValueWithQuotes(
	Iterator const& begin, Iterator const& end)
{
	m_strm << "ATTR_VALUE: " << tokenAsBString(begin, end) << std::endl;
}

void
TestXmlLexer::processOpeningTag(bool empty_tag)
{
	m_strm << "OP_TAG: " << empty_tag << std::endl;
}

void
TestXmlLexer::processClosingTag(
	Iterator const& begin, Iterator const& end)
{
	m_strm << "CLO_TAG: " << tokenAsBString(begin, end) << std::endl;
}

void
TestXmlLexer::processParseError(Position const& pos)
{
	m_errorLine = pos.line + 1;
	m_strm << "ERROR: " << pos.line << ':' << pos.col << std::endl;
}
