/*
    BFilter - a smart ad-filtering web proxy
    Copyright (C) 2002-2007  Joseph Artsimovich <joseph_a@mail.ru>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#ifndef HTTPCACHE_ITEM_H_
#define HTTPCACHE_ITEM_H_

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "AbstractItem.h"
#include "NonCopyable.h"
#include "FileId.h"
#include "AbstractFileStorage.h"
#include "ETag.h"
#include "BString.h"
#include "IntrusivePtr.h"
#include "SharedPtr.h"
#include "RequestPtr.h"
#include "AtomicCounter.h"
#include "types.h"
#include <ace/config-lite.h>
#include <ace/Synch.h>
#include <string>
#include <memory>
#include <time.h>

class RequestCacheControl;
class HttpRequestMetadata;
class HttpResponseMetadata;
class HttpHeadersCollection;

namespace HttpCache
{

class AbstractResponseReader;
class AbstractResponseWriter;
class AbstractResponseHandler;
class AbstractFileIO;
class RequestResolution;
class ResponseResolution;
class FileInstance;

namespace detail
{

struct FileInfo;

struct CacheInfo
{
	// Member-wise copying is OK.
	
	uint64_t body_length;
	time_t last_modified;
	time_t last_validated;
	unsigned freshness_lifetime;
	ETag etag;
	short status_code;
	bool heuristic_expiration:1;
	bool must_revalidate:1;
	
	CacheInfo();
	
	CacheInfo(HttpResponseMetadata const& metadata, uint64_t body_len);
};


class AccessorHelper
{
public:
	virtual ~AccessorHelper() {}
	
	virtual ResponseResolution handleResponseMT(
		IntrusivePtr<AccessorHelper> const& helper,
		ConstRequestPtr const& request,
		SharedPtr<HttpResponseMetadata const, ACE_MT_SYNCH> const& cached_metadata,
		HttpResponseMetadata const& response_metadata,
		std::auto_ptr<FileInstance> file, uint64_t body_length, uint32_t version) = 0;
	
	virtual void invalidateCacheEntryMT(
		std::auto_ptr<FileInstance> file, uint32_t version) = 0;
	
	virtual void ref() const = 0;
	
	virtual void unref() const = 0;
};


class CacheWriterHelper
{
public:
	virtual ~CacheWriterHelper() {}
	
	virtual void writeCommitMT(
		std::auto_ptr<FileInstance> file,
		SharedPtr<HttpResponseMetadata, ACE_MT_SYNCH> const& metadata,
		CacheInfo const& cache_info, uint32_t version) = 0;
	
	virtual void attachWriterMT() = 0;
	
	virtual void detachWriterMT() = 0;
};

} // namespace detail


/**
 * \brief Represents the exclusive owner of a cached resource.
 * \note This class is completely thread-safe.
 */
class Item :
	public AbstractItem,
	private detail::AccessorHelper,
	private detail::CacheWriterHelper
{
	DECLARE_NON_COPYABLE(Item)
public:
	enum MetadataStatus {
		METADATA_UNMODIFIED,
		METADATA_MODIFIED,
		METADATA_ONLY_DATE_MODIFIED
	};
	
	/**
	 * \brief Creates a new instance.
	 */
	static IntrusivePtr<AbstractItem> create(
		IntrusivePtr<AbstractFileStorage> const& storage,
		FileId const& file_id, bool& file_ok);
	
	virtual ~Item();
	
	virtual void ref() const;
	
	virtual void unref() const;
	
	virtual RequestResolution handleRequest(ConstRequestPtr const& request);
	
	virtual bool prepareForGC(GCFlags flags = GC_DEFAULT);
private:
	Item(IntrusivePtr<AbstractFileStorage> const& storage,
		FileId const& file_id, bool& file_ok);
	
	std::auto_ptr<FileInstance> openCurFileST(AbstractFileStorage::Mode mode);
	
	std::auto_ptr<FileInstance> openFileMT(
		FileInstance const& prototype, FileId const& file_id,
		AbstractFileStorage::Mode mode);
	
	void invalidateCurrentCacheEntryST();
	
	void invalidateCacheEntryST(
		std::auto_ptr<FileInstance> file, uint32_t version);
		
	virtual void invalidateCacheEntryMT(
		std::auto_ptr<FileInstance> file, uint32_t version);
	
	void updateCachedMetadataST();
	
	RequestResolution doHandleRequest(
		ConstRequestPtr const& request,
		RequestCacheControl const& request_cc);
	
	static bool clientAcceptsAgeMT(
		RequestCacheControl const& cc, detail::CacheInfo const& inf);
	
	static ConstRequestPtr addProxyValidationMT(
		HttpRequestMetadata const& request, detail::CacheInfo const& cache_info);
	
	static std::auto_ptr<AbstractResponseReader> createNotModifiedResponseMT(
		detail::CacheInfo const& cache_info);
	
	static std::auto_ptr<AbstractResponseReader> createPreconditionFailedResponseMT();
	
	static std::auto_ptr<AbstractResponseReader> createGatewayTimeoutResponseMT(
		HttpRequestMetadata const& request);
	
	static std::auto_ptr<AbstractResponseReader> createCacheEntryBrokenResponseMT(
		HttpRequestMetadata const& request);
	
	static std::auto_ptr<AbstractResponseReader> createCacheReaderMT(
		IntrusivePtr<detail::AccessorHelper> const& helper,
		detail::CacheInfo const& cache_info,
		SharedPtr<HttpResponseMetadata const, ACE_MT_SYNCH> const& cached_metadata,
		std::auto_ptr<FileInstance> file, uint32_t version);
	
	std::auto_ptr<AbstractResponseReader> createCacheReaderMT(
		IntrusivePtr<detail::AccessorHelper> const& helper,
		std::auto_ptr<FileInstance> file,
		SharedPtr<HttpResponseMetadata const, ACE_MT_SYNCH> const& cached_metadata,
		HttpResponseMetadata const& not_modified_response,
		uint64_t file_length, uint32_t version);
	
	static void prepareMetadataForOutputMT(HttpResponseMetadata& metadata,
		uint64_t body_length, detail::CacheInfo const& cache_info);
	
	std::auto_ptr<AbstractResponseWriter> createCacheWriterMT(
		HttpResponseMetadata const& response_metadata);
	
	static std::auto_ptr<AbstractResponseHandler> createResponseHandlerMT(
		IntrusivePtr<detail::AccessorHelper> const& helper,
		ConstRequestPtr const& request,
		SharedPtr<HttpResponseMetadata const, ACE_MT_SYNCH> const& cached_metadata,
		std::auto_ptr<FileInstance> file, uint64_t body_length, uint32_t version);
	
	static std::auto_ptr<detail::FileInfo>
		loadFileInfoMT(AbstractFileIO& file_io);
	
	static std::auto_ptr<HttpResponseMetadata>
		parseResponseMetadataMT(BString const& data);
	
	static MetadataStatus updateMetadataMT(
		HttpResponseMetadata& metadata,
		HttpResponseMetadata const& not_modified_metadata);
	
	virtual ResponseResolution handleResponseMT(
		IntrusivePtr<detail::AccessorHelper> const& helper,
		ConstRequestPtr const& request,
		SharedPtr<HttpResponseMetadata const, ACE_MT_SYNCH> const& cached_metadata,
		HttpResponseMetadata const& response_metadata,
		std::auto_ptr<FileInstance> file, uint64_t body_length, uint32_t version);
	
	ResponseResolution handle304ResponseMT(
		IntrusivePtr<detail::AccessorHelper> const& helper,
		ConstRequestPtr const& request,
		SharedPtr<HttpResponseMetadata const, ACE_MT_SYNCH> const& cached_metadata,
		HttpResponseMetadata const& response_metadata,
		std::auto_ptr<FileInstance> file, uint64_t body_length, uint32_t version);
	
	static bool responseWeakIdentityMT(
		detail::CacheInfo const& r1, detail::CacheInfo const& r2);
	
	static bool varyOkMT(HttpHeadersCollection const& headers);
	
	static bool mayBeCachedMT(HttpResponseMetadata const& response_metadata);
	
	virtual void writeCommitMT(
		std::auto_ptr<FileInstance> file,
		SharedPtr<HttpResponseMetadata, ACE_MT_SYNCH> const& metadata,
		detail::CacheInfo const& cache_info, uint32_t version);
	
	virtual void attachWriterMT();
	
	virtual void detachWriterMT();
	
	IntrusivePtr<AbstractFileStorage> const m_ptrStorage;
	FileId::Key const m_fileKey;
	
	/**
	 * Number of detail::CacheWriter objects referencing this item.
	 */
	AtomicCounter<ACE_MT_SYNCH> m_numWriters;
	
	/**
	 * The total number of references. 
	 */
	mutable AtomicCounter<ACE_MT_SYNCH> m_numRefs;
	
	/**
	 * When synchronization is needed:
	 * \li To get a consistent snapshot of m_ptrCurFile, m_ptrCachedMetadata,
	 *     m_cachedMetadataStatus, m_cacheInfo and m_curVersion.
	 * \li To modify any of them.
	 * \li To load or update the metadata in a file.
	 */
	ACE_Thread_Mutex m_mutex;
	
	/**
	 * If there is a ready-to-use file on disk, m_ptrCurFile will be set.
	 * This instance never represents an opened file.  It's purpose is to
	 * indicate that the file is there, and to initialize other FileInstances.
	 */
	std::auto_ptr<FileInstance> m_ptrCurFile;
	
	/**
	 * This object is created and updated when necessary.
	 */
	SharedPtr<HttpResponseMetadata const, ACE_MT_SYNCH> m_ptrCachedMetadata;
	
	/**
	 * Indicates if and how m_ptrCachedMetadata differs from the metadata on
	 * disk. If m_ptrCachedMetadata is null, m_cachedMetadataStatus is
	 * METADATA_UNMODIFIED.
	 */
	MetadataStatus m_cachedMetadataStatus;
	
	/**
	 * Some information corresponding to m_ptrCurFile.
	 */
	detail::CacheInfo m_cacheInfo;
	
	/**
	 * File version corresponding to m_ptrCurFile.
	 */
	uint32_t m_curVersion;
	
	AtomicCounter<ACE_MT_SYNCH> m_nextVersion;
	bool m_readyForGC;
};

} // namespace HttpCache

#endif
