/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include <glib.h>
#include "dialogfont.h"
#include "fontutils.h"
#include "listview.h"
#include "scripture.h"
#include "book.h"
#include "utilities.h"
#include "project.h"


FontDialog::FontDialog (Configuration* configuration)
{
  // Save variables.
  myconfiguration = configuration;
  
  fontdialog = gtk_dialog_new ();
  gtk_window_set_title (GTK_WINDOW (fontdialog), "Font");
  gtk_window_set_position (GTK_WINDOW (fontdialog), GTK_WIN_POS_CENTER_ON_PARENT);

  dialog_vbox1 = GTK_DIALOG (fontdialog)->vbox;
  gtk_widget_show (dialog_vbox1);

  ustring label;
  if (configuration->print_high_quality)
    label = "Reorder the fonts in the list: drag the more important fonts higher than the others, and drag the most important font, the one you wish to print in, to the top.";
  else
    label = "Select a font from the list";
  label24 = gtk_label_new (label.c_str());
  gtk_widget_show (label24);
  gtk_box_pack_start (GTK_BOX (dialog_vbox1), label24, FALSE, FALSE, 4);
  gtk_misc_set_alignment (GTK_MISC (label24), 0, 0.5);

  hbox3 = gtk_hbox_new (FALSE, 0);
  gtk_widget_show (hbox3);
  gtk_box_pack_start (GTK_BOX (dialog_vbox1), hbox3, TRUE, TRUE, 0);

  vbox5 = gtk_vbox_new (FALSE, 0);
  gtk_widget_show (vbox5);
  gtk_box_pack_start (GTK_BOX (hbox3), vbox5, TRUE, TRUE, 4);
  
  scrolledwindow2 = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (scrolledwindow2);
  gtk_box_pack_start (GTK_BOX (vbox5), scrolledwindow2, TRUE, TRUE, 0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow2), GTK_POLICY_NEVER, GTK_POLICY_AUTOMATIC);

  treeview1 = gtk_tree_view_new ();
  gtk_widget_show (treeview1);
  gtk_container_add (GTK_CONTAINER (scrolledwindow2), treeview1);
  gtk_widget_set_size_request (treeview1, 200, 500);
  gtk_tree_view_set_headers_visible (GTK_TREE_VIEW (treeview1), FALSE);
  gtk_tree_view_set_reorderable (GTK_TREE_VIEW (treeview1), TRUE);

  // Storage, renderer, column and selection.
  store = gtk_list_store_new (1, G_TYPE_STRING);
  gtk_tree_view_set_model (GTK_TREE_VIEW (treeview1), GTK_TREE_MODEL (store));
  g_object_unref (store);
  GtkCellRenderer *renderer = gtk_cell_renderer_text_new ();
  column = gtk_tree_view_column_new_with_attributes ("", renderer, "text", 0, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (treeview1), column);
  select = gtk_tree_view_get_selection (GTK_TREE_VIEW (treeview1));
  gtk_tree_selection_set_mode (select, GTK_SELECTION_SINGLE);

  scrolledwindow1 = gtk_scrolled_window_new (NULL, NULL);
  gtk_widget_show (scrolledwindow1);
  gtk_box_pack_start (GTK_BOX (hbox3), scrolledwindow1, TRUE, TRUE, 0);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW (scrolledwindow1), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);

  textview1 = gtk_text_view_new ();
  gtk_widget_show (textview1);
  gtk_widget_set_size_request (textview1, 650, -1);
  gtk_container_add (GTK_CONTAINER (scrolledwindow1), textview1);
  gtk_text_view_set_editable (GTK_TEXT_VIEW (textview1), FALSE);
  gtk_text_view_set_accepts_tab (GTK_TEXT_VIEW (textview1), FALSE);
  gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW (textview1), GTK_WRAP_WORD);
  gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW (textview1), FALSE);

  dialog_action_area1 = GTK_DIALOG (fontdialog)->action_area;
  gtk_widget_show (dialog_action_area1);
  gtk_button_box_set_layout (GTK_BUTTON_BOX (dialog_action_area1), GTK_BUTTONBOX_END);

  cancelbutton = gtk_button_new_from_stock ("gtk-cancel");
  gtk_widget_show (cancelbutton);
  gtk_dialog_add_action_widget (GTK_DIALOG (fontdialog), cancelbutton, GTK_RESPONSE_CANCEL);
  GTK_WIDGET_SET_FLAGS (cancelbutton, GTK_CAN_DEFAULT);

  okbutton = gtk_button_new_from_stock ("gtk-ok");
  gtk_widget_show (okbutton);
  gtk_dialog_add_action_widget (GTK_DIALOG (fontdialog), okbutton, GTK_RESPONSE_OK);
  GTK_WIDGET_SET_FLAGS (okbutton, GTK_CAN_DEFAULT);

  g_signal_connect ((gpointer) treeview1, "cursor_changed", G_CALLBACK (on_treeview1_cursor_changed), gpointer(this));
  g_signal_connect ((gpointer) okbutton, "clicked", G_CALLBACK (on_okbutton_clicked), gpointer(this));

  gtk_widget_grab_focus (treeview1);
  gtk_widget_grab_default (okbutton);
  
  // Set the fonts in the view.
  vector<ustring> fonts;
  fonts = font_print_get_families (configuration, true);
  listview_set_strings (treeview1, store, fonts);
  if (!configuration->print_high_quality)
    if (fonts.size() > 0)
      listview_focus_string (treeview1, fonts[0]);  
}


FontDialog::~FontDialog ()
{
  gtk_widget_destroy (fontdialog);
}


int FontDialog::run ()
{
  return gtk_dialog_run (GTK_DIALOG (fontdialog));
}


void FontDialog::on_okbutton_clicked (GtkButton * button, gpointer user_data)
{
  ((FontDialog *) user_data)->on_okbutton ();
}


void FontDialog::on_okbutton ()
{
  // Save the new font data.
  vector <ustring> fonts = listview_get_strings (treeview1);
  // The print quality affects the selection procedure.
  if (!myconfiguration->print_high_quality) {
    ustring font;
    font = listview_get_active_string (treeview1);
    if (!font.empty()) {
      vector<ustring> fonts2;
      fonts2.push_back (font);
      for (unsigned int i = 0; i < fonts.size(); i++) {
        if (fonts[i] != font) {
          fonts2.push_back (fonts[i]);
        }
      }
      fonts.clear();
      fonts.assign (fonts2.begin(), fonts2.end());
    }    
  }
  font_print_set_families (myconfiguration, fonts);  
}


void FontDialog::on_treeview1_cursor_changed (GtkTreeView *treeview, gpointer user_data)
{
  ((FontDialog *) user_data)->on_treeview ();
}


void FontDialog::on_treeview ()
{
  // Set the font in the editor.
  ustring font = listview_get_active_string (treeview1);
  PangoFontDescription *font_desc = pango_font_description_from_string (font.c_str ());
  gtk_widget_modify_font (textview1, font_desc);
  pango_font_description_free (font_desc);
  // Put some text in the textview, ...
  ustring contents;
  if (font == DISABLED_FONTS_BELOW) {
    // ... either information about this line,
    contents = "Fonts dragged below this line will not be used at all.";
  } else {
    // ... or the currently opened chapter,
    if (!myconfiguration->project.empty()) {
      Scripture scripture (myconfiguration->project);
      for (unsigned int i = 0; i < scripture.books.size(); i++) {
        if (scripture.books[i] == myconfiguration->book_opened) {
          Book book (scripture.paths[i], false, "");
          vector<ustring> text;
          book.get_chapter (convert_to_int (myconfiguration->chapter_opened), text);
          for (unsigned int i2 = 0; i2 < text.size(); i2++) {
            contents.append (text[i2]);
            contents.append ("\n");
          }
        }
      }
    }
    // ... or the alphabet.
    if (contents.empty()) {
      contents = ALPHABET + upperCase (ALPHABET);
    }
  }
  gtk_text_buffer_set_text (gtk_text_view_get_buffer (GTK_TEXT_VIEW (textview1)), contents.c_str(), -1);
}
