/*
** Copyright (C) 2003-2006 Teus Benschop.
**  
** This program is free software; you can redistribute it and/or modify
** it under the terms of the GNU General Public License as published by
** the Free Software Foundation; either version 2 of the License, or
** (at your option) any later version.
**  
** This program is distributed in the hope that it will be useful,
** but WITHOUT ANY WARRANTY; without even the implied warranty of
** MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
** GNU General Public License for more details.
**  
** You should have received a copy of the GNU General Public License
** along with this program; if not, write to the Free Software
** Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
**  
*/


#include "libraries.h"
#include "project.h"
#include "utilities.h"
#include "config.h"
#include <glib.h>
#include "directories.h"
#include "stylesheetutils.h"
#include "constants.h"
#include "gwrappers.h"
#include "scripture.h"
#include "bible.h"


Project::Project (Configuration * configuration)
{
  projectname = configuration->project;
  myconfiguration = configuration;
  project_load ();
}


Project::Project (const ustring& project)
/*
Note: This second constructor is only for convenience. It is not as good as 
the first type, neither does it write changes to disc. But it is useful
when one only queries project settings, without editing them.
*/
{
  projectname = project;
  myconfiguration = NULL;
  project_load ();
}


Project::~Project ()
{
  if (myconfiguration) {
    // If the object goes out of scope, it saves the project settings.
    vector <ustring> output;
    output.push_back (VERSION);
    output.push_back ("There is no need for editing this file, because it is maintained by bibledit");
    output.push_back (editor_font);
    output.push_back (mystylesheet);
    output.push_back (versification);
    output.push_back (sword_description);
    output.push_back (sword_about);
    output.push_back (sword_lcsh);
    output.push_back (sword_license);
    output.push_back (sword_version);
    output.push_back (sword_language);
    output.push_back (sword_name);
    try {
      write_lines (filename, output);
    }
    catch (exception & ex) { 
      cerr << ex.what () << endl;
    }
    // Save the printing fonts.
    try {
      write_lines (printing_fonts_filename(), printing_fonts);
    }
    catch (exception & ex) { 
      cerr << ex.what () << endl;
    }
  }
}


void Project::project_load ()
{
  // Read the project file, if there is any. Add the bits that are missing.
  filename = gw_build_filename (directories_get_projects(), projectname, "projectsettings");
  ReadText rc (filename, true);
  for (unsigned int i = 0; i <= 1000; i++) {
    switch (i) {
      case 2:
      {
        if (i < rc.lines.size ()) {
          editor_font = rc.lines[i];
        }
        else {
          if (myconfiguration)
            editor_font = myconfiguration->editor_font_name;
        }
        break;
      }
      case 3:
      {
        if (i < rc.lines.size ()) {
          mystylesheet = rc.lines[i];
        }
        else
        {
          mystylesheet = "Standard";
        }
        break;
      }
      case 4:
      {
        if (i < rc.lines.size ()) {
          versification = rc.lines[i];
        }
        else
        {
          versification = "English";
        }
        break;
      }
      case 5:
      {
        if (i < rc.lines.size ())
          sword_description = rc.lines[i];
        else
          if (myconfiguration)
            sword_description = "Bibledit project " + myconfiguration->project;
        break;
      }
      case 6:
      {
        if (i < rc.lines.size ())
          sword_about = rc.lines[i];
        else
          sword_about = sword_description;
        break;
      }
      case 7:
      {
        if (i < rc.lines.size ())
          sword_lcsh = rc.lines[i];
        else
          sword_lcsh = "Bible--Translation";
        break;
      }
      case 8:
      {
        if (i < rc.lines.size ())
          sword_license = rc.lines[i];
        else
          sword_license = "GNU General Public License";
        break;
      }
      case 9:
      {
        if (i < rc.lines.size ())
          sword_version = rc.lines[i];
        else
          sword_version = "1.0";
        break;
      }
      case 10:
      {
        if (i < rc.lines.size ())
          sword_language = rc.lines[i];
        else
          sword_language = "English";
        break;
      }
      case 11:
      {
        if (i < rc.lines.size ())
          sword_name = rc.lines[i];
        else
          sword_name.clear();
        break;
      }

    }
  }
  // Read the printing fonts.
  ReadText rc2 (printing_fonts_filename(), true);
  printing_fonts.assign (rc2.lines.begin(), rc2.lines.end());
}


ustring Project::get_stylesheet ()
// Returns the stylesheet, checks it.
{
  ustring checked_sheet (mystylesheet);
  // See whether our stylesheet exists.
  vector<ustring> stylesheets;
  stylesheet_get_ones_available (stylesheets);
  set<ustring> sheets (stylesheets.begin(), stylesheets.end());
  // Sheet is there? Fine.
  if (sheets.find (checked_sheet) != sheets.end());
  // Sheets is not there - take Standard, if it's around.
  else if (sheets.find (STANDARDSHEET) != sheets.end())
    checked_sheet = STANDARDSHEET;
  // Else take first sheet in the list.
  else
    checked_sheet = stylesheets[0];
  // Update internal data.
  mystylesheet = checked_sheet;
  return checked_sheet;  
}


void Project::set_stylesheet (const ustring& stylesheet)
{
  mystylesheet = stylesheet;
}


ustring Project::printing_fonts_filename ()
{
  return gw_build_filename (directories_get_projects(), projectname, "printing-fonts");
}


ustring Project::books_reordering_filename ()
{
  return gw_build_filename (directories_get_projects(), projectname, "books-reordering");
}


void Project::reordering_read ()
/*
Reads reordering information from disk.

Updates the reordering information based on the actual books in the object.
This means that, even if books, includes and portions are not in the settings on disk,
they get updated by the actual books in this Scripture. And any excess books,
if they are not anymore in the Scripture, are removed.
Extra books are added at their proper locations.
*/
{
  // Load scripture.
  Scripture scripture (projectname);
  
  // Read the reordering information.
  ReadText rc3 (books_reordering_filename(), true);
  for (unsigned int i = 2; i < rc3.lines.size(); i += 3) {
    reordered_books.push_back (rc3.lines[i - 2]);
    reordered_includes.push_back (convert_to_bool (rc3.lines[i - 1]));
    reordered_portions.push_back (rc3.lines[i]);    
  }
  
  // First pass: remove books no longer in the object.
  {
    set<ustring> bookset (scripture.books.begin(), scripture.books.end());
    vector<ustring> mybooks;
    vector<bool> myincludes;
    vector<ustring> myportions;
    for (unsigned int i = 0; i < reordered_books.size(); i++) {
      if (bookset.find (reordered_books[i]) != bookset.end()) {
        mybooks.push_back (reordered_books[i]);
        myincludes.push_back (reordered_includes[i]);
        myportions.push_back (reordered_portions[i]);
      }
    }
    reordered_books.assign (mybooks.begin(), mybooks.end());
    reordered_includes.assign (myincludes.begin(), myincludes.end());
    reordered_portions.assign (myportions.begin(), myportions.end());
  }

  // Second pass: add books missing in the reordering information.
  set<ustring> reorder_set (reordered_books.begin(), reordered_books.end());
  for (unsigned int i = 0; i < scripture.books.size(); i++) {
    if (reorder_set.find (scripture.books[i]) == reorder_set.end()) {
      // We've found a book which needs to be inserted in the list in the proper place.
      // Measure the "distance" between a book to insert and any of the books 
      // already in the project. Distance can be positive, if the book follows,
      // or negative, if the book goes before.
      vector<ustring> books2;
      vector<int> distances;
      int my_offset = english_id_to_index (scripture.books[i]);
      for (unsigned int i2 = 0; i2 < reordered_books.size(); i2++) {
        int your_offset = english_id_to_index (reordered_books[i2]);
        books2.push_back (reordered_books[i2]);
        distances.push_back (my_offset - your_offset);
      }
      // Find the smallest absolute distance, and the corresponding book.
      int smallest_distance = 10000;
      unsigned int smallest_offset = 0;
      for (unsigned int i2 = 0; i2 < distances.size(); i2++) {
        if (abs (distances[i2]) < smallest_distance) {
          smallest_distance = abs (distances[i2]);
          smallest_offset = i2;
        }          
      }
      // If offset is positive, insert the book after this one, if negative, insert before.
      vector<ustring>::iterator book_iter = reordered_books.begin();
      vector<bool>::iterator includes_iter = reordered_includes.begin();
      vector<ustring>::iterator portions_iter = reordered_portions.begin();
      // Check whether there were any books at all.
      if (!reordered_books.empty()) {
        for (unsigned int i2 = 0; i2 < smallest_offset; i2++) {
          book_iter++;
          includes_iter++;
          portions_iter++;
        }
        if (distances[smallest_offset] > 0) {
          book_iter++;
          includes_iter++;
          portions_iter++;
        }
      }
      reordered_books.insert (book_iter, scripture.books[i]);
      reordered_includes.insert (includes_iter, true);
      reordered_portions.insert (portions_iter, CHAPTER_VERSE_SELECTION_ALL);
    }
  }
}


void Project::reordering_write ()
// Writes reordering information to disk.
{
  try {
    vector<ustring> lines;
    for (unsigned int i = 0; i < reordered_books.size(); i++) {
      lines.push_back (reordered_books[i]);
      lines.push_back (convert_to_string (reordered_includes[i]));
      lines.push_back (reordered_portions[i]);
    }
    write_lines (books_reordering_filename(), lines);
  }
  catch (exception & ex) { 
    cerr << ex.what () << endl;
  }
}


bool Project::reordering_in_order ()
// Indicates whether the reordered books are in the standard order.
{
  int previousoffset = -1;
  for (unsigned int i = 0; i < reordered_books.size(); i++) {
    int currentoffset = english_id_to_index (reordered_books[i]);
    if (currentoffset < previousoffset)
      return false;
    previousoffset = currentoffset;    
  }
  return true;
}


bool Project::reordering_portions_all ()
// Indicates whether all books have their portions set to "all"  and are included.
{
  for (unsigned int i = 0; i < reordered_includes.size(); i++)
    if (!reordered_includes[i])
      return false;
  for (unsigned int i = 0; i < reordered_portions.size(); i++)
    if (reordered_portions[i] != CHAPTER_VERSE_SELECTION_ALL)
      return false;
  return true;
}
